﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkAssert.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/nn_TimeSpan.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/fs.h>
#include <nn/vi.h>
#include <nn/hid.h>

#include <nvnTool/nvnTool_GlslcInterface.h>
#include <nv/nv_MemoryManagement.h>
#include <nvngdSupport/TutorialUtil.h>
#include <nvngdSupport/TutorialBaseClass.h>

extern "C"
{
    PFNNVNGENERICFUNCPTRPROC NVNAPIENTRY nvnBootstrapLoader(const char* name);
}
const int DevtoolsMemorySize = 512 * 1024 * 1024;
static NN_ALIGNAS(4096) uint8_t g_devtoolsMemory[DevtoolsMemorySize];
nn::mem::StandardAllocator g_devtoolsAllocator;

namespace {
    void* Allocate(size_t size)
    {
        return std::malloc(size);
    }

    void Deallocate(void* p, size_t size)
    {
        NN_UNUSED(size);
        std::free(p);
    }

    void* NvAllocateFunction(size_t size, size_t alignment, void* userPtr)
    {
        NN_UNUSED(userPtr);
        // According to specifications of aligned_alloc(), we need to coordinate the size parameter to become the integral multiple of alignment.
        return aligned_alloc(alignment, nn::util::align_up(size, alignment));
    }
    void NvFreeFunction(void* addr, void* userPtr)
    {
        NN_UNUSED(userPtr);
        free(addr);
    }
    void* NvReallocateFunction(void* addr, size_t newSize, void* userPtr)
    {
        NN_UNUSED(userPtr);
        return realloc(addr, newSize);
    }

    void* NvDevtoolsAllocateFunction(size_t size, size_t alignment, void* userPtr)
    {
        // According to specifications of aligned_alloc(), we need to coordinate the size parameter to become the integral multiple of alignment.
        //return aligned_alloc(alignment, nn::util::align_up(size, alignment));
        NN_ASSERT_NOT_NULL(userPtr);
        return reinterpret_cast<nn::mem::StandardAllocator*>(userPtr)->Allocate(size, alignment);

    }
    void NvDevtoolsFreeFunction(void* addr, void* userPtr)
    {
        //NN_UNUSED(userPtr);
        //free(addr);

        NN_ASSERT_NOT_NULL(userPtr);
        reinterpret_cast<nn::mem::StandardAllocator*>(userPtr)->Free(addr);
    }
    void* NvDevtoolsReallocateFunction(void* addr, size_t newSize, void* userPtr)
    {
        //NN_UNUSED(userPtr);
        //return realloc(addr, newSize);

        NN_ASSERT_NOT_NULL(userPtr);
        return reinterpret_cast<nn::mem::StandardAllocator*>(userPtr)->Reallocate(addr, newSize);
    }

    void* GlslcAllocateFunction(size_t size, size_t alignment, void* userPtr)
    {
        NN_UNUSED(userPtr);
        // According to specifications of aligned_alloc(), we need to coordinate the size parameter to become the integral multiple of alignment.
        return aligned_alloc(alignment, nn::util::align_up(size, alignment));
    }
    void GlslcFreeFunction(void* addr, void* userPtr)
    {
        NN_UNUSED(userPtr);
        free(addr);
    }
    void* GlslcReallocateFunction(void* addr, size_t newSize, void* userPtr)
    {
        NN_UNUSED(userPtr);
        return realloc(addr, newSize);
    }
} // namespace

class Tutorial
{
public:
    explicit Tutorial();
    ~Tutorial();

    int Run();
    int TestRun();

private:
    static char const* const m_pWindowClassName;
    static char const* const m_pWindowTitle;

    static int const m_TotalFrameTime = 16; // 60fps.
    // If there is less than this many milliseconds in a frame, just proceed to the next.
    static int const m_PadFrameTime = 2;

    void InitWindow();
    void InitNVN();

    void Draw(unsigned long long millisec);
    bool TestDraw(unsigned long long millisec);

    nn::vi::NativeWindowHandle  m_NativeWindow;
    nn::vi::Display*            m_pDisplay;
    nn::vi::Layer*              m_pLayer;
    int                         m_WindowWidth;
    int                         m_WindowHeight;
};

// static
char const* const Tutorial::m_pWindowClassName = "NVNTutorialWindowClass";
// static
char const* const Tutorial::m_pWindowTitle = "NVN Tutorial";

Tutorial::Tutorial() : m_NativeWindow(NULL), m_pDisplay(NULL), m_pLayer(NULL), m_WindowWidth(1280), m_WindowHeight(720)
{
    InitWindow();
    InitNVN();
}

Tutorial::~Tutorial()
{
    nn::vi::DestroyLayer(m_pLayer);
    nn::vi::CloseDisplay(m_pDisplay);

    nn::vi::Finalize();
}

int Tutorial::Run()
{
    nn::os::Tick frameTime(nn::TimeSpan::FromMilliSeconds(0));
    nn::os::Tick frameTimePrev = nn::os::GetSystemTick();
    int sleepTime = 0;

    bool runLoop = true;
    bool doFrame = true;
    bool doSleep = true;

    t()->Resize(m_WindowWidth, m_WindowHeight);

        /* Initialize touch screen for user exit */
    nn::hid::InitializeTouchScreen();
    nn::hid::TouchScreenState<3> touchScreenState;
    int touchCount = 0;

    NN_LOG("-------------------------------------------------------------------------------------------------\n");
    NN_LOG("Hitting Esc on Windows or touching the Touch Screen with two fingers on NX will exit the program.\n");
    NN_LOG("-------------------------------------------------------------------------------------------------\n");

    while (runLoop)
    {
        if (doFrame)
        {
            frameTime = nn::os::GetSystemTick();

            NN_ASSERT(frameTime >= frameTimePrev, "test");
            Draw(frameTime.ToTimeSpan().GetMilliSeconds() - frameTimePrev.ToTimeSpan().GetMilliSeconds());
            frameTimePrev = frameTime;

            doFrame = false;
        }

        doSleep = runLoop;
        if (doSleep)
        {
            // Figure out how much time to sleep off, if any
            sleepTime = m_TotalFrameTime - static_cast<int>((nn::os::GetSystemTick().ToTimeSpan() - frameTime.ToTimeSpan()).GetMilliSeconds());
            if (sleepTime <= m_PadFrameTime)
            {
                doSleep = false;
                doFrame = true;
            }
        }

            /*
             * If the touch screen has been hit, break out of the run loop
             * and clean up.
             */
        touchCount = nn::hid::GetTouchScreenStates(&touchScreenState, 1);
        if (touchScreenState.count > 1)
        {
            break;
        }
        doFrame = true;
    }

    return 0;
}

int Tutorial::TestRun()
{
    nn::os::Tick frameTime(nn::TimeSpan::FromMilliSeconds(0));
    nn::os::Tick frameTimePrev = nn::os::GetSystemTick();
    int sleepTime = 0;

    bool runLoop = true;
    bool doFrame = true;
    bool doSleep = true;

    t()->Resize(m_WindowWidth, m_WindowHeight);

    /* Initialize touch screen for user exit */
    nn::hid::InitializeTouchScreen();
    //nn::hid::TouchScreenState<3> touchScreenState;
    //int touchCount = 0;

    while (runLoop)
    {
        if (doFrame)
        {
            frameTime = nn::os::GetSystemTick();

            NN_ASSERT(frameTime >= frameTimePrev, "test");
            runLoop = TestDraw(frameTime.ToTimeSpan().GetMilliSeconds() - frameTimePrev.ToTimeSpan().GetMilliSeconds());
            frameTimePrev = frameTime;

            doFrame = false;
        }

        doSleep = runLoop;
        if (doSleep)
        {
            // Figure out how much time to sleep off, if any
            sleepTime = m_TotalFrameTime - static_cast<int>((nn::os::GetSystemTick().ToTimeSpan() - frameTime.ToTimeSpan()).GetMilliSeconds());
            if (sleepTime <= m_PadFrameTime)
            {
                doSleep = false;
                doFrame = true;
            }
        }

        /*
        * If the touch screen has been hit, break out of the run loop
        * and clean up.
        */
        //touchCount = nn::hid::GetTouchScreenStates(&touchScreenState, 1);
        //if (touchScreenState.count > 1)
        //{
        //    break;
        //}
        doFrame = true;
    }

    return 0;
}

void Tutorial::InitWindow()
{
        /* Initialize Visual Interface (VI) system to display
         * to the target's screen
         */
    nn::vi::Initialize();

    nn::Result result = nn::vi::OpenDefaultDisplay(&m_pDisplay);
    NN_ASSERT(result.IsSuccess());

    result = nn::vi::CreateLayer(&m_pLayer, m_pDisplay);
    NN_ASSERT(result.IsSuccess());

    nn::vi::SetLayerScalingMode(m_pLayer, nn::vi::ScalingMode_FitToLayer);

    result = nn::vi::GetNativeWindow(&m_NativeWindow, m_pLayer);
}

void Tutorial::InitNVN()
{
    // The NVN loader is exposed as a obfuscated OpenGL extension in Windows
    t()->Init(nvnBootstrapLoader, m_NativeWindow);
}

void Tutorial::Draw(unsigned long long millisec)
{
    t()->Draw(millisec);
}

bool Tutorial::TestDraw(unsigned long long millisec)
{
    return t()->TestDraw(millisec);
}

bool TutorialBaseClass::TestDraw(uint64_t millisec)
{
    NN_UNUSED(millisec);
    return true;
}

void TutorialRun(bool test)
{
        /* Set functions that the file system uses for allocation/deallocation */
    nn::fs::SetAllocator(Allocate, Deallocate);

        /*
        * Set memory allocator for graphics subsystem.
        * This function must be called before using any graphics API's.
        */
    nv::SetGraphicsAllocator(NvAllocateFunction, NvFreeFunction, NvReallocateFunction, NULL);

        /*
         * Set memory allocator for graphics developer tools and NVN debug layer.
         * This function must be called before using any graphics developer features.
         */
    g_devtoolsAllocator.Initialize(g_devtoolsMemory, DevtoolsMemorySize);
    nv::SetGraphicsDevtoolsAllocator(NvDevtoolsAllocateFunction, NvDevtoolsFreeFunction, NvDevtoolsReallocateFunction, &g_devtoolsAllocator);

        /*
        * Donate memory for graphics driver to work in.
        * This function must be called before using any graphics API's.
        */
    size_t graphicsSystemMemorySize = 8 * 1024 * 1024;
    void* graphicsHeap = malloc(graphicsSystemMemorySize);
    nv::InitializeGraphics(graphicsHeap, graphicsSystemMemorySize);

        /*
         * If using glslc online, it is required that this function is called
         * before you use any GLSLC functions bar glslcInitialize.
         */
    glslcSetAllocator(GlslcAllocateFunction, GlslcFreeFunction, GlslcReallocateFunction, NULL);

    Tutorial tutorial;
    if (test)
    {
        tutorial.TestRun();
    }
    else
    {
        tutorial.Run();
    }
    t()->Shutdown();

    free(graphicsHeap);
}
