﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{UniformBufferManager.h,PageSampleNvnTutorialLibrary}
 *
 * @brief
 *  This file defines a class that manages memory
 *  for uniform buffers by using multiple chunks of
 *  of GPU memory. The active chunk is written to and
 *  submitted to the GPU. At the end of the frame,
 *  it is swapped with the next chunk which then becomes
 *  active and writable for the next frame. This is done to
 *  avoid have to wait too much time for a sync object to
 *  signal that the memory is done being consumed by the
 *  GPU.
 *
 *  Requesting a single chunk for the memory pool will provide
 *  an undivided block of memory for uniform values that don't
 *  necessarily need to be updated at all or just infrequently,
 *  though the user must now handle proper syncing if they
 *  do wish to update the memory.
 */


#pragma once

#include <nvn/nvn.h>
#include <nvn/nvn_FuncPtrInline.h>
#include <vector>
#include <atomic>
#include "ManagedUniformBuffer.h"
#include "TutorialUtil.h"
#include "MemoryPool.h"
#include "FrameBufferManager.h"

class UniformBufferManager : public FrameBufferedMemoryManager
{
    public:
        UniformBufferManager(NVNdevice* pDevice, size_t chunkSize, int numChunks = 2);
        virtual ~UniformBufferManager();
        ManagedUniformBuffer* CreateUniformBuffer(size_t bufferSize);

        virtual void SwapPools();
        virtual int GetNumChunks();

        ptrdiff_t GetMemoryChunk(size_t size);

    private:
        int                                 m_NumChunks;
        MemoryPool                          m_Pool;
        NVNbuffer                           m_Buffer;
        size_t                              m_ChunkSize;
        size_t                              m_PoolSize;

        std::atomic<ptrdiff_t>              m_WriteIndex;
        unsigned                            m_CurrentChunk;
        NVNbufferAddress                    m_BufferAddress;
        char*                               m_pMappedBuffer;

        std::vector<ManagedUniformBuffer*>  m_ManagedBuffers;

        NVNdevice*                          m_pDevice;
        size_t                              m_bufferAlignment;
};
