﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#define NN_PERF_PROFILE_ENABLED

#include <cstdlib>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/os.h>
#include <nn/init.h>
#include <nn/fs.h>
#include <nn/mem.h>
#include <nn/vi.h>
#include <nn/gfx.h>
#include <nnt.h>
#include <nn/gfx/util/gfx_TransientDescriptorAllocatorHolder.h>
#include <nn/gfx/util/gfx_TextureCompressor.h>

#include <nns/gfx/gfx_GraphicsFramework.h>
#include <nns/gfx/gfx_PrimitiveRenderer.h>
#include <nns/gfx/gfx_PrimitiveRendererMeterDrawer.h>

#if NN_GFX_IS_TARGET_NVN
#include <nvn/nvn.h>
#include <nvn/nvn_FuncPtrInline.h>
#endif

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
    #include <nv/nv_MemoryManagement.h>
#endif

#if NN_GFX_IS_TARGET_NVN && defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON )
    #include <nvnTool/nvnTool_GlslcInterface.h>
#endif


#include <nn/perf.h>


#if defined(TEST_GFXUTIL_TEXTURE_COMPRESSION_BENCHMARK_ENABLE_JXR)
#include "testGfxUtil_Jxr.h"
#endif

#include "testGfxUtil_Jpeg.h"
#include "testGfxUtil_BmpReader.h"

#include <cinttypes>

namespace
{

    struct ResShaderType
    {
        ptrdiff_t   binaryOffset;
        size_t      binarySize;
        nn::gfx::ResShaderFile* pResShaderFile;
        nn::gfx::ResShaderContainer* pResShaderContainer;
        nn::gfx::ShaderCodeType codeType;
    };

    ResShaderType g_TextureResShader;
    ResShaderType g_TextureCompressionShader;


    nns::gfx::GraphicsFramework g_GraphicsFramework;

    uint8_t fileBuffer[1024 * 1024 * 4];
    uint8_t sourceDecompressBuffer[1024 * 1024 * 4];
    uint8_t bmpDecompressBuffer[1024 * 1024 * 4];
    uint8_t bc1OnlineResultBuffer[1024 * 1024 * 4];
    uint8_t bc1OfflineResultBuffer[1024 * 1024 * 4];

    size_t LoadBinaryFileToBuffer(const char* path)
    {
        nn::Result result;

        int64_t fileSize = 0;
        {
            nn::fs::FileHandle hFile;
            result = nn::fs::OpenFile(&hFile, path, nn::fs::OpenMode_Read);
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);

            result = nn::fs::GetFileSize(&fileSize, hFile);
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            NN_ASSERT(sizeof(fileBuffer) >= fileSize);

            result = nn::fs::ReadFile(hFile, 0, fileBuffer, static_cast<size_t>(fileSize));
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            nn::fs::CloseFile(hFile);
        }

        return static_cast<size_t>(fileSize);
    }

    void ReadResourceToBuffer(const char* filename, void** ppDestinationBuffer, size_t* pDestinationBufferSizeInBytes)
    {
        nn::Result result;
        nn::fs::FileHandle hFile;
        int64_t fileSize = 0;
        result = nn::fs::OpenFile( &hFile, filename, nn::fs::OpenMode_Read );
        NN_ASSERT( result.IsSuccess() );
        result = nn::fs::GetFileSize( &fileSize, hFile );
        NN_ASSERT( result.IsSuccess() );
        nn::util::BinaryFileHeader fileHeader;
        size_t readSize;
        result = nn::fs::ReadFile( &readSize, hFile, 0, &fileHeader, sizeof( nn::util::BinaryFileHeader ) );
        NN_ASSERT( result.IsSuccess() );
        NN_ASSERT( readSize == sizeof( nn::util::BinaryFileHeader ) );

        size_t alignment = fileHeader.GetAlignment();
        void* pDestinationBuffer = g_GraphicsFramework.AllocateMemory(static_cast<size_t>(fileSize), alignment);

        result = nn::fs::ReadFile( &readSize, hFile, 0, pDestinationBuffer, static_cast<size_t>( fileSize ) );
        NN_ASSERT( result.IsSuccess() );
        NN_ASSERT( readSize == static_cast< size_t >( fileSize ) );
        nn::fs::CloseFile( hFile );

        *ppDestinationBuffer = pDestinationBuffer;
        *pDestinationBufferSizeInBytes = static_cast<size_t>(fileSize);
    }

    const size_t GraphicsSystemMemorySize =  8 * 1024 * 1024;

    nn::gfx::util::TransientDescriptorAllocatorHolder<2> g_TransientTextureDescriptorAllocatorHolder;
    nn::gfx::util::TransientDescriptorAllocator* g_pTransientTextureDescriptorAllocator = NULL;

    nn::gfx::util::TransientDescriptorAllocatorHolder<2> g_TransientSamplerDescriptorAllocatorHolder;
    nn::gfx::util::TransientDescriptorAllocator* g_pTransientSamplerDescriptorAllocator = NULL;

    const int MaxMipLevels = 12;
    const int MaxLayerCount = 12;

    const char* RenderTextureShaderBinaryFilePath = "Contents:/renderTextureShader.bnsh";
    const char* TextureCompressionShaderBinaryFilePath = "Contents:/textureCompressionShader.bnsh";

    const uint32_t ScreenWidth = 1280;
    const uint32_t ScreenHeight = 720;
    const uint32_t MarginX = 4;
    const uint32_t MarginY = 4;

    nn::gfx::util::TextureCompressor            g_TextureCompressor;
    nn::gfx::Sampler                            g_SamplerMinMagMipPointEdgeClamp;
    nn::gfx::DescriptorSlot                     g_SamplerDescriptorSlot;

    void InitializeResShaderFile(ResShaderType* pResShaderType,const char* shaderFilePath)
    {
        NN_LOG("\nLoad ShaderFile: %s\n",shaderFilePath);
        nn::Result result;

        void* pFileBuffer = NULL;
        int64_t fileSize = 0;
        {
            nn::fs::FileHandle hFile;
            result = nn::fs::OpenFile(&hFile, shaderFilePath, nn::fs::OpenMode_Read);
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);

            result = nn::fs::GetFileSize(&fileSize, hFile);
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);

            pFileBuffer = g_GraphicsFramework.AllocateMemory(static_cast<size_t>(fileSize), NN_ALIGNOF(nn::util::BinaryFileHeader));
            NN_ASSERT_NOT_NULL(pFileBuffer);

            result = nn::fs::ReadFile(hFile, 0, pFileBuffer, static_cast<size_t>(fileSize));
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            nn::fs::CloseFile(hFile);
        }

        nn::gfx::Buffer buffer;
        void* pShaderBinary = NULL;
        {
            size_t alignment = reinterpret_cast<nn::gfx::ResShaderFile*>(pFileBuffer)->ToData().fileHeader.GetAlignment();
            pResShaderType->binarySize = static_cast<size_t>(fileSize);

            nn::gfx::Buffer::InfoType info;
            {
                info.SetDefault();
                info.SetGpuAccessFlags(nn::gfx::GpuAccess_Read);
                info.SetSize(pResShaderType->binarySize);
            }

            if(NN_STATIC_CONDITION(nn::gfx::Buffer::IsMemoryPoolRequired))
            {
                pResShaderType->binaryOffset = -1;
                pResShaderType->binaryOffset = g_GraphicsFramework.AllocatePoolMemory(
                    nns::gfx::GraphicsFramework::MemoryPoolType::MemoryPoolType_Shader,
                    pResShaderType->binarySize, alignment);
                NN_ASSERT_NOT_EQUAL(pResShaderType->binaryOffset, -1);


                buffer.Initialize(g_GraphicsFramework.GetDevice(), info,
                    g_GraphicsFramework.GetMemoryPool(nns::gfx::GraphicsFramework::MemoryPoolType::MemoryPoolType_Shader),
                    pResShaderType->binaryOffset, pResShaderType->binarySize);
            }
            else
            {
                buffer.Initialize(g_GraphicsFramework.GetDevice(),info,NULL,0,0);
            }

            pShaderBinary = buffer.Map();
            std::memcpy(pShaderBinary, pFileBuffer, static_cast<size_t>(fileSize));
        }

        {
            g_GraphicsFramework.FreeMemory(pFileBuffer);
            pFileBuffer = NULL;
        }

        pResShaderType->pResShaderFile = nn::gfx::ResShaderFile::ResCast(pShaderBinary);
        pResShaderType->pResShaderContainer = pResShaderType->pResShaderFile->GetShaderContainer();
        NN_ASSERT_NOT_NULL(pResShaderType->pResShaderContainer);
        size_t containerOffset = pResShaderType->binaryOffset + nn::util::BytePtr(pResShaderType->pResShaderFile).Distance(pResShaderType->pResShaderContainer);
        pResShaderType->pResShaderContainer->Initialize(g_GraphicsFramework.GetDevice(), g_GraphicsFramework.GetMemoryPool(nns::gfx::GraphicsFramework::MemoryPoolType_Shader), containerOffset, pResShaderType->binarySize);

        nn::gfx::ResShaderVariation* pVariation = pResShaderType->pResShaderContainer->GetResShaderVariation(0);
        NN_ASSERT_NOT_NULL(pVariation);

        nn::gfx::ShaderInitializeResult shaderResult = nn::gfx::ShaderInitializeResult_Success;
        pResShaderType->codeType = nn::gfx::ShaderCodeType_Binary;
        nn::gfx::ResShaderProgram* pResShaderProgram = pVariation->GetResShaderProgram(pResShaderType->codeType);
        if (pResShaderProgram)
        {
            shaderResult = pResShaderProgram->Initialize(g_GraphicsFramework.GetDevice());
        }

        if (!pResShaderProgram || shaderResult != nn::gfx::ShaderInitializeResult_Success)
        {
            pResShaderType->codeType = nn::gfx::ShaderCodeType_Source;
            pResShaderProgram = pVariation->GetResShaderProgram(pResShaderType->codeType);
            if (pResShaderProgram)
            {
                shaderResult = pResShaderProgram->Initialize(g_GraphicsFramework.GetDevice());
            }
        }
        NN_ASSERT(pResShaderProgram && shaderResult == nn::gfx::ShaderInitializeResult_Success);

        int variationCount = pResShaderType->pResShaderContainer->GetShaderVariationCount();
        for (int idxVariation = 1; idxVariation < variationCount; idxVariation++)
        {
            pVariation = pResShaderType->pResShaderContainer->GetResShaderVariation(idxVariation);
            NN_ASSERT_NOT_NULL(pVariation);
            shaderResult = pVariation->GetResShaderProgram(pResShaderType->codeType)->Initialize(g_GraphicsFramework.GetDevice());
            NN_ASSERT_EQUAL(shaderResult, nn::gfx::ShaderInitializeResult_Success);
        }

        buffer.Unmap();
        buffer.Finalize(g_GraphicsFramework.GetDevice());


        NN_LOG("Load Shader: SUCCESS\n\n");
    }

    void FinalizeResShaderFile(ResShaderType* pResShaderType)
    {
        int variationCount = pResShaderType->pResShaderContainer->GetShaderVariationCount();
        for (int variationIndex = 0; variationIndex < variationCount; ++variationIndex)
        {
            nn::gfx::ResShaderVariation* pVariation = pResShaderType->pResShaderContainer->GetResShaderVariation(variationIndex);
            pVariation->GetResShaderProgram(pResShaderType->codeType)->Finalize(g_GraphicsFramework.GetDevice());
        }

        pResShaderType->pResShaderContainer->Finalize(g_GraphicsFramework.GetDevice());
        memset(pResShaderType, 0, sizeof(ResShaderType));
    }

    nn::gfx::Shader* GetTextureCompressionShader(nn::gfx::ImageFormat imageFormat)
    {
        NN_ASSERT_NOT_NULL(g_TextureCompressionShader.pResShaderContainer);

        int variation = -1;
        switch (imageFormat)
        {
        case nn::gfx::ImageFormat_Bc1_Unorm:
            variation = 10;
            break;

        case nn::gfx::ImageFormat_Bc1_UnormSrgb:
            variation = 11;
            break;

        case nn::gfx::ImageFormat_Bc2_Unorm:
            variation = 12;
            break;

        case nn::gfx::ImageFormat_Bc2_UnormSrgb:
            variation = 13;
            break;

        case nn::gfx::ImageFormat_Bc3_Unorm:
            variation = 14;
            break;

        case nn::gfx::ImageFormat_Bc3_UnormSrgb:
            variation = 15;
            break;

        case nn::gfx::ImageFormat_Bc4_Unorm:
            variation = 16;
            break;

        case nn::gfx::ImageFormat_Bc4_Snorm:
            variation = 17;
            break;

        case nn::gfx::ImageFormat_Bc5_Unorm:
            variation = 18;
            break;

        case nn::gfx::ImageFormat_Bc5_Snorm:
            variation = 19;
            break;

        default:
            NN_UNEXPECTED_DEFAULT;
        }

        return g_TextureCompressionShader.pResShaderContainer->GetResShaderVariation(variation)->GetResShaderProgram(g_TextureCompressionShader.codeType)->GetShader();
    }

    nn::gfx::Texture g_SourceTexture;
    ptrdiff_t g_SourceTextureOffset = -1;
    size_t g_SourceTextureSize = 0;
    size_t g_SourceTextureRowPitch = 0;

    void InitializeSourceTextures(nn::gfx::ImageFormat imageFormat, int width, int height, int mipLevels, int layerCount)
    {
        NN_LOG("Initialize SourceTextures\n");
        nn::gfx::Texture::InfoType info;
        {
            info.SetDefault();
            info.SetGpuAccessFlags(nn::gfx::GpuAccess_Texture);
            info.SetWidth(width);
            info.SetHeight(height);
            info.SetImageFormat(imageFormat);
            info.SetMipCount(mipLevels);
            info.SetArrayLength(layerCount);
            info.SetSwizzle(0);
            info.SetTileMode(nn::gfx::TileMode_Linear);
        }

        if (NN_STATIC_CONDITION(nn::gfx::Texture::IsMemoryPoolRequired))
        {
            size_t size = nn::gfx::Texture::CalculateMipDataSize(g_GraphicsFramework.GetDevice(), info);
            size_t alignment = nn::gfx::Texture::CalculateMipDataAlignment(g_GraphicsFramework.GetDevice(), info);

            nns::gfx::GraphicsFramework::MemoryPoolType memoryPoolType = nns::gfx::GraphicsFramework::MemoryPoolType::MemoryPoolType_Data;
            nn::gfx::MemoryPool* pMemoryPool = g_GraphicsFramework.GetMemoryPool(memoryPoolType);

            ptrdiff_t offset = -1;
            offset = g_GraphicsFramework.AllocatePoolMemory(
                memoryPoolType,
                size, alignment);
            NN_ASSERT_NOT_EQUAL(offset, -1);

            g_SourceTexture.Initialize(g_GraphicsFramework.GetDevice(),info,
                pMemoryPool,
                offset, size);

            g_SourceTextureOffset = offset;
            g_SourceTextureSize = size;
            g_SourceTextureRowPitch = nn::gfx::Texture::GetRowPitch(g_GraphicsFramework.GetDevice(), info);
        }
        else
        {
            g_SourceTexture.Initialize(g_GraphicsFramework.GetDevice(),info,NULL,0,0);
        }
    }

    void FinalizeSourceTextures()
    {
        NN_LOG("Finalize SourceTextures\n");
        g_SourceTexture.Finalize(g_GraphicsFramework.GetDevice());
    }

    void UpdateSourceTextureContent(const void* pSourceBuffer, size_t sourceBufferSizeInBytes, int width, int height, ptrdiff_t sourceBufferStride)
    {
        NN_UNUSED(sourceBufferSizeInBytes);

        nns::gfx::GraphicsFramework::MemoryPoolType memoryPoolType = nns::gfx::GraphicsFramework::MemoryPoolType::MemoryPoolType_Data;
        nn::gfx::MemoryPool* pMemoryPool = g_GraphicsFramework.GetMemoryPool(memoryPoolType);
        void* pMemoryPoolBuffer = pMemoryPool->Map();

        uint8_t* pGpuTextureBuffer = static_cast<uint8_t*>(pMemoryPoolBuffer) + g_SourceTextureOffset;
        const uint8_t* pCpuTextureBuffer = static_cast<const uint8_t*>(pSourceBuffer);

        for (int h = 0; h < height; ++h)
        {
            uint8_t* pGpuTextureLineBuffer = pGpuTextureBuffer;
            const uint8_t* pCpuTextureLineBuffer = pCpuTextureBuffer;

            for (int w = 0; w < width; ++w)
            {
                *(pGpuTextureLineBuffer++)  = *(pCpuTextureLineBuffer++);
                *(pGpuTextureLineBuffer++)  = *(pCpuTextureLineBuffer++);
                *(pGpuTextureLineBuffer++)  = *(pCpuTextureLineBuffer++);
                *(pGpuTextureLineBuffer++)  = 0xFF;
            }

            pCpuTextureBuffer += sourceBufferStride;
            pGpuTextureBuffer += g_SourceTextureRowPitch;
        }


        pMemoryPool->Unmap();
        pMemoryPool->FlushMappedRange(g_SourceTextureOffset, g_SourceTextureSize);
    }

    nn::gfx::TextureView g_SourceTextureView;
    void InitializeSourceTextureView(nn::gfx::ImageFormat imageFormat, int mipLevels, int layerCount)
    {
        NN_LOG("Initialize SourceTextureView\n");
        {
            nn::gfx::TextureView::InfoType info;
            info.SetDefault();
            info.SetTexturePtr(&g_SourceTexture);
            info.SetImageFormat(imageFormat);
            info.SetImageDimension(nn::gfx::ImageDimension_2dArray);
            info.EditSubresourceRange().EditMipRange().SetMipCount(mipLevels);
            info.EditSubresourceRange().EditArrayRange().SetArrayLength(layerCount);
            g_SourceTextureView.Initialize(g_GraphicsFramework.GetDevice(),info);
        }
    }

    void FinalizeSourceTextureView()
    {
        NN_LOG("Finalize SourceTextureView\n");
        g_SourceTextureView.Finalize(g_GraphicsFramework.GetDevice());
    }


    /////////////////

    nn::gfx::Texture g_DestTexture;
    nn::gfx::TextureView g_DestTextureView;
    nn::gfx::util::TextureCompressorTargetInfo g_TextureCompressorTargetTextureInfo;
    void InitializeDestTexture(nn::gfx::ImageFormat imageFormat, int width, int height, int mipLevels, int layerCount)
    {
        {
            nn::gfx::Texture::InfoType info;
            NN_LOG("Initialize DestTexture\n");
            {
                info.SetDefault();
                info.SetGpuAccessFlags(nn::gfx::GpuAccess_Texture);
                info.SetWidth(width);
                info.SetHeight(height);
                info.SetImageFormat(imageFormat);
                info.SetMipCount(mipLevels);
                info.SetArrayLength(layerCount);
            }

            ptrdiff_t offset = 0;
            size_t size = 0;
            if(NN_STATIC_CONDITION(nn::gfx::Texture::IsMemoryPoolRequired))
            {
                size = nn::gfx::Texture::CalculateMipDataSize(g_GraphicsFramework.GetDevice(), info);
                size_t alignment = nn::gfx::Texture::CalculateMipDataAlignment(g_GraphicsFramework.GetDevice(), info);

                offset = -1;
                offset = g_GraphicsFramework.AllocatePoolMemory(
                    nns::gfx::GraphicsFramework::MemoryPoolType::MemoryPoolType_RenderTarget,
                    size, alignment);
                NN_ASSERT_NOT_EQUAL(offset, -1);

                g_DestTexture.Initialize(
                    g_GraphicsFramework.GetDevice(),
                    info,
                    g_GraphicsFramework.GetMemoryPool(nns::gfx::GraphicsFramework::MemoryPoolType::MemoryPoolType_RenderTarget),
                    offset, size);
            }
            else
            {
                g_DestTexture.Initialize(g_GraphicsFramework.GetDevice(),info,NULL,0,0);
            }

            // 圧縮処理クラスの初期化
            NN_LOG("Initialize TextureCompressor\n");
            g_TextureCompressorTargetTextureInfo.SetTexture(info,&g_DestTexture);
            g_TextureCompressor.Initialize(g_GraphicsFramework.GetDevice());

            nn::gfx::Shader* pShader = GetTextureCompressionShader(imageFormat);
            g_TextureCompressor.RegisterCompressionShader(imageFormat,pShader);
        }

        {
            NN_LOG("Initialize DestTextureView\n");
            nn::gfx::TextureView::InfoType info;
            info.SetDefault();
            info.SetTexturePtr(&g_DestTexture);
            info.SetImageFormat(imageFormat);
            info.SetImageDimension(nn::gfx::ImageDimension_2dArray);
            info.EditSubresourceRange().EditMipRange().SetMipCount(mipLevels);
            info.EditSubresourceRange().EditArrayRange().SetArrayLength(layerCount);
            g_DestTextureView.Initialize(g_GraphicsFramework.GetDevice(),info);
        }
    }

    void FinalizeDestTexture()
    {
        NN_LOG("Finalize DestTextureView\n");
        g_DestTextureView.Finalize(g_GraphicsFramework.GetDevice());

        NN_LOG("Finalize TextureCompressor\n");
        g_TextureCompressor.Finalize();

        NN_LOG("Finalize DestTexture\n");
        g_DestTexture.Finalize(g_GraphicsFramework.GetDevice());
    }

    /////////////
    nn::gfx::Texture g_CopyLinearTexture;
    nn::gfx::ColorTargetView g_CopyLinearTextureTargetView;
    nn::gfx::ViewportScissorState g_CopyLinearTextureViewportScissorState;

    nn::gfx::Buffer g_CopyLinearBufferOnlineCompression;
    nn::gfx::Buffer g_CopyLinearBufferOfflineCompression;


    void InitializeCopyLinearBuffers(int width, int height)
    {
        {
            size_t bufferSize = width * height * 4;
            nn::gfx::Buffer::InfoType info;
            {
                info.SetDefault();
                info.SetGpuAccessFlags(nn::gfx::GpuAccess_ColorBuffer); // mimic usage of FrameCapture test app
                info.SetSize(bufferSize);
            }

            if (NN_STATIC_CONDITION(nn::gfx::Buffer::IsMemoryPoolRequired))
            {
                size_t alignment = nn::gfx::Buffer::GetBufferAlignment(g_GraphicsFramework.GetDevice(), info);

                nns::gfx::GraphicsFramework::MemoryPoolType memoryPoolType = nns::gfx::GraphicsFramework::MemoryPoolType::MemoryPoolType_Data;
                nn::gfx::MemoryPool* pMemoryPool = g_GraphicsFramework.GetMemoryPool(memoryPoolType);

                ptrdiff_t offset = -1;
                offset = g_GraphicsFramework.AllocatePoolMemory(
                    memoryPoolType,
                    bufferSize, alignment);
                NN_ASSERT_NOT_EQUAL(offset, -1);

                g_CopyLinearBufferOnlineCompression.Initialize(
                    g_GraphicsFramework.GetDevice(), info,
                    pMemoryPool,
                    offset, bufferSize);

                void* pPtr = static_cast<uint8_t*>(pMemoryPool->Map()) + offset;
                memset(pPtr, 0, bufferSize);
            }
            else
            {
                g_CopyLinearBufferOnlineCompression.Initialize(g_GraphicsFramework.GetDevice(), info, NULL, 0, 0);
            }
        }

        {
            size_t bufferSize = width * height * 4;
            nn::gfx::Buffer::InfoType info;
            {
                info.SetDefault();
                info.SetGpuAccessFlags(nn::gfx::GpuAccess_ColorBuffer); // mimic usage of FrameCapture test app
                info.SetSize(bufferSize);
            }

            if (NN_STATIC_CONDITION(nn::gfx::Buffer::IsMemoryPoolRequired))
            {
                size_t alignment = nn::gfx::Buffer::GetBufferAlignment(g_GraphicsFramework.GetDevice(), info);

                nns::gfx::GraphicsFramework::MemoryPoolType memoryPoolType = nns::gfx::GraphicsFramework::MemoryPoolType::MemoryPoolType_Data;
                nn::gfx::MemoryPool* pMemoryPool = g_GraphicsFramework.GetMemoryPool(memoryPoolType);

                ptrdiff_t offset = -1;
                offset = g_GraphicsFramework.AllocatePoolMemory(
                    memoryPoolType,
                    bufferSize, alignment);
                NN_ASSERT_NOT_EQUAL(offset, -1);

                g_CopyLinearBufferOfflineCompression.Initialize(
                    g_GraphicsFramework.GetDevice(), info,
                    pMemoryPool,
                    offset, bufferSize);

                void* pPtr = static_cast<uint8_t*>(pMemoryPool->Map()) + offset;
                memset(pPtr, 0, bufferSize);
            }
            else
            {
                g_CopyLinearBufferOfflineCompression.Initialize(g_GraphicsFramework.GetDevice(), info, NULL, 0, 0);
            }
        }
    }

    void InitializeCopyLinear(nn::gfx::ImageFormat imageFormat, int width, int height, int mipLevels, int layerCount)
    {
        NN_LOG("Initialize CopyLinearTextures\n");

        InitializeCopyLinearBuffers(width, height);


        {
            nn::gfx::Texture::InfoType info;
            {
                info.SetDefault();
                info.SetGpuAccessFlags(nn::gfx::GpuAccess_Texture | nn::gfx::GpuAccess_ColorBuffer);
                info.SetWidth(width);
                info.SetHeight(height);
                info.SetDepth(1);
                info.SetImageFormat(imageFormat);
                info.SetMipCount(mipLevels);
                info.SetArrayLength(layerCount);
                info.SetImageStorageDimension(nn::gfx::ImageStorageDimension_2d);
                info.SetMipCount(1);
                info.SetTileMode(nn::gfx::TileMode_Linear);
            }

            if (NN_STATIC_CONDITION(nn::gfx::Texture::IsMemoryPoolRequired))
            {
                size_t size = nn::gfx::Texture::CalculateMipDataSize(g_GraphicsFramework.GetDevice(), info);
                size_t alignment = nn::gfx::Texture::CalculateMipDataAlignment(g_GraphicsFramework.GetDevice(), info);

                nns::gfx::GraphicsFramework::MemoryPoolType memoryPoolType = nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget;
                nn::gfx::MemoryPool* pMemoryPool = g_GraphicsFramework.GetMemoryPool(memoryPoolType);

                ptrdiff_t offset = -1;
                offset = g_GraphicsFramework.AllocatePoolMemory(
                    memoryPoolType,
                    size, alignment);
                NN_ASSERT_NOT_EQUAL(offset, -1);

                g_CopyLinearTexture.Initialize(g_GraphicsFramework.GetDevice(),info,
                    pMemoryPool,
                    offset, size);
            }
            else
            {
                g_CopyLinearTexture.Initialize(g_GraphicsFramework.GetDevice(),info,NULL,0,0);
            }
        }
        {
            nn::gfx::ColorTargetView::InfoType info;
            info.SetTexturePtr(&g_CopyLinearTexture);
            info.SetImageFormat(imageFormat);
            info.SetImageDimension(nn::gfx::ImageDimension_2d);
            info.SetMipLevel(0);
            info.EditArrayRange().SetBaseArrayIndex(0);
            info.EditArrayRange().SetArrayLength(1);

            g_CopyLinearTextureTargetView.Initialize(g_GraphicsFramework.GetDevice(), info);
        }
        {
            nn::gfx::ViewportScissorState::InfoType info;
            info.SetDefault();
            info.SetScissorEnabled(false);

            nn::gfx::ViewportStateInfo viewportInfo;
            viewportInfo.SetWidth(static_cast<float>(std::max(width, 1)));
            viewportInfo.SetHeight(static_cast<float>(std::max(height, 1)));

            nn::gfx::ScissorStateInfo scissorInfo;
            scissorInfo.SetDefault();
            scissorInfo.SetWidth(std::max(width, 1));
            scissorInfo.SetHeight(std::max(height, 1));

            info.SetViewportStateInfoArray(&viewportInfo, 1);
            info.SetScissorStateInfoArray(&scissorInfo, 1);

            g_CopyLinearTextureViewportScissorState.Initialize(g_GraphicsFramework.GetDevice(), info);
        }
    }

    void FinalizeCopyLinear()
    {
        NN_LOG("Finalize CopyLinearTextures\n");

        g_CopyLinearTextureViewportScissorState.Finalize(g_GraphicsFramework.GetDevice());
        g_CopyLinearTextureTargetView.Finalize(g_GraphicsFramework.GetDevice());
        g_CopyLinearTexture.Finalize(g_GraphicsFramework.GetDevice());

        g_CopyLinearBufferOnlineCompression.Finalize(g_GraphicsFramework.GetDevice());
        g_CopyLinearBufferOfflineCompression.Finalize(g_GraphicsFramework.GetDevice());
    }


    void CopyLinearBufferContent(nn::gfx::Buffer* pBuffer, void* pDestinationBuffer, size_t destinationBufferSize, int width, int height, ptrdiff_t destinationBufferStride)
    {
        const uint8_t* pGpuTextureBuffer = static_cast<uint8_t*>(pBuffer->Map());
        uint8_t* pCpuTextureBuffer = static_cast<uint8_t*>(pDestinationBuffer);
        NN_ASSERT(static_cast<size_t>(destinationBufferStride * height) <= destinationBufferSize);

        // invalidate CPU cache
        pBuffer->InvalidateMappedRange(0, 4 * width * height);

        for (int h = 0; h  < height; ++h)
        {
            const uint8_t* pGpuTextureLineBuffer = pGpuTextureBuffer;
            uint8_t* pCpuTextureLineBuffer = pCpuTextureBuffer;

            for (int w = 0; w < width; ++w)
            {
                *(pCpuTextureLineBuffer + 0) = *(pGpuTextureLineBuffer + 2);
                *(pCpuTextureLineBuffer + 1) = *(pGpuTextureLineBuffer + 1);
                *(pCpuTextureLineBuffer + 2) = *(pGpuTextureLineBuffer + 0);

                pCpuTextureLineBuffer += 3;
                pGpuTextureLineBuffer += 4;
            }

            pCpuTextureBuffer += destinationBufferStride;
            pGpuTextureBuffer += 4 * width ;
        }

        pBuffer->Unmap();
    }

    //////////////


    nn::gfx::ViewportScissorState g_SourceViewportScissorState[MaxMipLevels * MaxLayerCount];
    nn::gfx::ViewportScissorState g_DestViewportScissorState[MaxMipLevels * MaxLayerCount];
    nn::gfx::ViewportScissorState g_OfflineBc1ViewportScissorState[MaxMipLevels * MaxLayerCount];

    void InitializeRectangleViewportScissorState(nn::gfx::ViewportScissorState* pViewportScissorState, int originX, int originY, int screenWidth, int screenHeight, int mipLevels, int layerCount)
    {
        nn::gfx::ViewportScissorState::InfoType info;
        nn::gfx::ScissorStateInfo scissorInfo;
        nn::gfx::ViewportStateInfo viewportInfo;

        const int imageWidth = ((screenWidth + MarginX) / layerCount) - MarginX;
        const int imageHeight = ((screenHeight + MarginY) / mipLevels) - MarginY;
        NN_ASSERT_GREATER(imageWidth, 0);
        NN_ASSERT_GREATER(imageHeight, 0);

        int p = 0;
        for (int layer = 0; layer < layerCount; layer++)
        {
            int x = originX + layer * (imageWidth + MarginX);
            for (int miplevel = 0; miplevel < mipLevels; miplevel++)
            {
                int y = originY + miplevel * (imageHeight + MarginY);

                //viewport
                {
                    viewportInfo.SetDefault();
                    viewportInfo.SetOriginX(static_cast<float>(x));
                    viewportInfo.SetOriginY(static_cast<float>(y));
                    viewportInfo.SetWidth(static_cast<float>(imageWidth));
                    viewportInfo.SetHeight(static_cast<float>(imageHeight));
                    viewportInfo.SetMinDepth(.0f);
                    viewportInfo.SetMaxDepth(1.0f);
                }

                //scissor
                {
                    scissorInfo.SetDefault();
                    scissorInfo.SetOriginX(x);
                    scissorInfo.SetOriginY(y);
                    scissorInfo.SetWidth(imageWidth);
                    scissorInfo.SetHeight(imageHeight);
                }

                info.SetDefault();
                info.SetScissorEnabled(true);
                info.SetViewportStateInfoArray(&viewportInfo,1);
                info.SetScissorStateInfoArray(&scissorInfo,1);

                pViewportScissorState[p].Initialize(g_GraphicsFramework.GetDevice(),info);
                p++;
            }
        }
    }

    void FinalizeRectangleViewportScissorState(nn::gfx::ViewportScissorState* pViewportScissorState, int mipLevels, int layerCount)
    {
        int p = 0;
        for (int layer = 0; layer < layerCount; layer++)
        {
            for (int miplevel = 0; miplevel < mipLevels; miplevel++)
            {
                pViewportScissorState[p].Finalize(g_GraphicsFramework.GetDevice());
                p++;
            }
        }
    }


    void InitializeRenderRectanglePipeline(int mipLevels, int layerCount)
    {
        // ViewportScissorState
        InitializeRectangleViewportScissorState(
            g_SourceViewportScissorState,
            0,0,
            ScreenWidth / 3,ScreenHeight,
            mipLevels, layerCount
        );
        InitializeRectangleViewportScissorState(
            g_DestViewportScissorState,
            ScreenWidth / 3,0,
            ScreenWidth / 3,ScreenHeight,
            mipLevels, layerCount
        );
        InitializeRectangleViewportScissorState(
            g_OfflineBc1ViewportScissorState,
            (2 * ScreenWidth) / 3,0,
            ScreenWidth / 3,ScreenHeight,
            mipLevels, layerCount
        );
    }

    void FinalizeRenderRectanglePipeLine(int mipLevels, int layerCount)
    {
        FinalizeRectangleViewportScissorState(g_DestViewportScissorState, mipLevels, layerCount);
        FinalizeRectangleViewportScissorState(g_SourceViewportScissorState, mipLevels, layerCount);
        FinalizeRectangleViewportScissorState(g_OfflineBc1ViewportScissorState, mipLevels, layerCount);
    }

    void InitializeRenderTexture(int mipLevels, int layerCount)
    {
        NN_LOG("Initialize RenderTextureRoutine\n");
        InitializeRenderRectanglePipeline(mipLevels, layerCount);
    }

    void FinalizeRenderTexture(int mipLevels, int layerCount)
    {
        NN_LOG("Finalize RenderTextureRoutine\n");
        FinalizeRenderRectanglePipeLine(mipLevels, layerCount);
    }

    void InitializeGfxObjects(nn::gfx::ImageFormat srcFormat,nn::gfx::ImageFormat dstFormat, int width, int height, int mipLevels, int layerCount)
    {

        NN_LOG("Initialize GfxObjects\n");

        InitializeSourceTextures(srcFormat, width, height, mipLevels, layerCount);
        InitializeSourceTextureView(srcFormat, mipLevels, layerCount);

        InitializeRenderTexture(mipLevels, layerCount);

        InitializeDestTexture(dstFormat, width, height, mipLevels, layerCount);

        InitializeCopyLinear(nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm, width, height, mipLevels, layerCount);
    }

    void FinalizeGfxObjects(int width, int height, int mipLevels, int layerCount)
    {
        NN_UNUSED(width);
        NN_UNUSED(height);

        FinalizeCopyLinear();

        FinalizeRenderTexture(mipLevels, layerCount);
        FinalizeDestTexture();

        FinalizeSourceTextureView();
        FinalizeSourceTextures();

        NN_LOG("Finalize GfxObjects\n");
    }



    void* g_pResTextureFileBuffer = NULL;
    nn::gfx::ResTextureFile* g_pResTextureFile = NULL;
    int g_resTextureSlotIndex = -1;
    void InitializeOfflineBc1CompressedTexture(const char* filePath)
    {
        //ptrdiff_t iPoolOffset = 0;
        //size_t iPoolBufferSize = 0;
        //nns::gfx::GraphicsFramework::MemoryPoolType ePoolType = nns::gfx::GraphicsFramework::MemoryPoolType_Data;
        //void* pResTextureFileBuffer = ReadResourceToBuffer(filePath, ePoolType, &iPoolOffset, &iPoolBufferSize);
        //NN_ASSERT(nn::gfx::ResTextureFile::IsValid(pResTextureFileBuffer));
        //pResTextureFile = nn::gfx::ResTextureFile::ResCast(pResTextureFileBuffer);
        //pResTextureFile->Initialize(g_GraphicsFramework.GetDevice(), g_GraphicsFramework.GetMemoryPool(ePoolType), iPoolOffset, iPoolBufferSize);


        size_t resTextureFileBufferSize = 0;
        ReadResourceToBuffer(filePath, &g_pResTextureFileBuffer, &resTextureFileBufferSize);
        g_pResTextureFile = nn::gfx::ResTextureFile::ResCast(g_pResTextureFileBuffer);
        g_pResTextureFile->Initialize(g_GraphicsFramework.GetDevice());

        NN_ASSERT(g_pResTextureFile->GetTextureDic()->GetCount() == 1);
        nn::gfx::ResTexture* pResTexture = g_pResTextureFile->GetResTexture(0);
        pResTexture->Initialize(g_GraphicsFramework.GetDevice());

        // Descriptor slot.
        nn::gfx::TextureInfo* pTextureInfo = pResTexture->GetTextureInfo();
        nn::gfx::ImageFormat imageFormat = pTextureInfo->GetImageFormat();
        NN_ASSERT(imageFormat == nn::gfx::ImageFormat_Bc1_Unorm);

        nn::gfx::TextureView* pTextureView = pResTexture->GetTextureView();
        g_resTextureSlotIndex = g_GraphicsFramework.AllocateDescriptorSlot(nn::gfx::DescriptorPoolType_TextureView, 1);
        g_GraphicsFramework.SetTextureViewToDescriptorPool(g_resTextureSlotIndex, pTextureView);

        nn::gfx::DescriptorSlot descriptorSlot;
        g_GraphicsFramework.GetDescriptorPool(nn::gfx::DescriptorPoolType_TextureView)->GetDescriptorSlot(&descriptorSlot, g_resTextureSlotIndex);
        pResTexture->SetUserDescriptorSlot(descriptorSlot);
    }

    void FinalizeOfflineBc1CompressedTexture()
    {
        nn::gfx::ResTexture* pResTexture = g_pResTextureFile->GetResTexture(0);
        g_GraphicsFramework.FreeDescriptorSlot(nn::gfx::DescriptorPoolType_TextureView, g_resTextureSlotIndex);
        pResTexture->Finalize(g_GraphicsFramework.GetDevice());
        g_pResTextureFile->Finalize(g_GraphicsFramework.GetDevice());

        g_GraphicsFramework.FreeMemory(g_pResTextureFileBuffer);

        g_resTextureSlotIndex = -1;
        g_pResTextureFile = NULL;
        g_pResTextureFileBuffer = NULL;
    }

    void MakeRenderRectangleCommand(nn::gfx::CommandBuffer* pCommandBuffer,nn::gfx::Shader* pShader,nn::gfx::TextureView* pTextureView, int layer, int miplevel)
    {
        nn::gfx::GpuAddress gpuAddress;

        pCommandBuffer->SetBlendState(g_GraphicsFramework.GetBlendState(nns::gfx::GraphicsFramework::BlendStateType_Alpha));
        pCommandBuffer->SetDepthStencilState(g_GraphicsFramework.GetDepthStencilState(nns::gfx::GraphicsFramework::DepthStencilStateType_Disabled));
        pCommandBuffer->SetRasterizerState(g_GraphicsFramework.GetRasterizerState(nns::gfx::GraphicsFramework::RasterizerStateType_FillSolid_CullNone));


        nn::gfx::DescriptorPool* pTextureDescriptorPool = g_pTransientTextureDescriptorAllocator->GetDescriptorPool();
        int textureSlotIndex = g_pTransientTextureDescriptorAllocator->Allocate();
        {
            pTextureDescriptorPool->BeginUpdate();
            pTextureDescriptorPool->SetTextureView(textureSlotIndex,pTextureView);
            pTextureDescriptorPool->EndUpdate();
        }

        nn::gfx::DescriptorPool* pSamplerDescriptorPool = g_pTransientSamplerDescriptorAllocator->GetDescriptorPool();
        int samplerSlotIndex = g_pTransientSamplerDescriptorAllocator->Allocate();
        {
            pSamplerDescriptorPool->BeginUpdate();
            pSamplerDescriptorPool->SetSampler(samplerSlotIndex,&g_SamplerMinMagMipPointEdgeClamp);
            pSamplerDescriptorPool->EndUpdate();
        }

        nn::gfx::DescriptorSlot textureDescriptorSlot;
        nn::gfx::DescriptorSlot samplerDescriptorSlot;
        {
            g_pTransientTextureDescriptorAllocator->GetDescriptorPool()->GetDescriptorSlot(&textureDescriptorSlot,textureSlotIndex);
            g_pTransientSamplerDescriptorAllocator->GetDescriptorPool()->GetDescriptorSlot(&samplerDescriptorSlot,samplerSlotIndex);
        }

        pCommandBuffer->SetTextureAndSampler(0,nn::gfx::ShaderStage_Pixel,textureDescriptorSlot,samplerDescriptorSlot);
        pCommandBuffer->SetShader(pShader, nn::gfx::ShaderStageBit_All);

        // miplevelとレイヤーは15ビットずつあれば十分とする
        NN_ASSERT_RANGE(layer,0,0x8000);
        NN_ASSERT_RANGE(miplevel,0,0x8000);

        int texInfo = layer & 0x7fff;
        texInfo |= ( (miplevel << 15) & 0xffff8000);
        texInfo <<= 2;

        pCommandBuffer->Draw(nn::gfx::PrimitiveTopology_TriangleStrip,4,texInfo,1,0);
    }

    void RenderToDestTexture(nn::gfx::CommandBuffer* pCommandBuffer)
    {
        nn::gfx::DescriptorPool* pTextureDescriptorPool = g_pTransientTextureDescriptorAllocator->GetDescriptorPool();
        int textureSlotIndex = g_pTransientTextureDescriptorAllocator->Allocate();
        {
            pTextureDescriptorPool->BeginUpdate();
            pTextureDescriptorPool->SetTextureView(textureSlotIndex,&g_SourceTextureView);
            pTextureDescriptorPool->EndUpdate();
        }
        nn::gfx::DescriptorSlot textureDescriptorSlot;
        pTextureDescriptorPool->GetDescriptorSlot(&textureDescriptorSlot,textureSlotIndex);

        g_TextureCompressor.MakeCommand(pCommandBuffer,textureDescriptorSlot,g_SamplerDescriptorSlot,g_TextureCompressorTargetTextureInfo);
    }


    void RenderSourceTexture(nn::gfx::CommandBuffer* pCommandBuffer, int mipLevels, int layerCount)
    {
        nn::gfx::Shader* pShader = g_TextureResShader.pResShaderContainer->GetResShaderVariation(0)->GetResShaderProgram(g_TextureResShader.codeType)->GetShader();

        int p = 0;
        for (int layer = 0; layer < layerCount; layer++)
        {
            for (int miplevel = 0; miplevel < mipLevels; miplevel++)
            {
                pCommandBuffer->SetViewportScissorState(&g_SourceViewportScissorState[p]);
                MakeRenderRectangleCommand(pCommandBuffer,pShader,&g_SourceTextureView,layer,miplevel);
                p++;
            }
        }
    }

    void RenderDestTexture(nn::gfx::CommandBuffer* pCommandBuffer, int mipLevels, int layerCount)
    {
        nn::gfx::Shader* pShader = g_TextureResShader.pResShaderContainer->GetResShaderVariation(0)->GetResShaderProgram(g_TextureResShader.codeType)->GetShader();

        int p = 0;
        for (int layer = 0; layer < layerCount; layer++)
        {
            for (int miplevel = 0; miplevel < mipLevels; miplevel++)
            {
                pCommandBuffer->SetViewportScissorState(&g_DestViewportScissorState[p]);
                MakeRenderRectangleCommand(pCommandBuffer, pShader, &g_DestTextureView,layer,miplevel);
                p++;
            }
        }
    }

    void RenderOfflineBc1Texture(nn::gfx::CommandBuffer* pCommandBuffer)
    {
        nn::gfx::Shader* pShader = g_TextureResShader.pResShaderContainer->GetResShaderVariation(0)->GetResShaderProgram(g_TextureResShader.codeType)->GetShader();
        pCommandBuffer->SetViewportScissorState(&g_OfflineBc1ViewportScissorState[0]);
        MakeRenderRectangleCommand(pCommandBuffer, pShader, g_pResTextureFile->GetResTexture(0)->GetTextureView(), 0, 0);
    }

    void CopyToLinearBuffer(nn::gfx::CommandBuffer* pCommandBuffer, int width, int height, int miplevel, int layer, nn::gfx::TextureView* pBc1SourceTextureView)
    {
        nn::gfx::ColorTargetView* pTarget = &g_CopyLinearTextureTargetView;
        nn::gfx::ClearColorValue clearColor;
        memset(&clearColor, 0, sizeof(clearColor));

        nn::gfx::TextureCopyRegion region;
        region.SetDefault();
        region.SetWidth(width);
        region.SetHeight(height);

        nn::gfx::Shader* pShader = g_TextureResShader.pResShaderContainer->GetResShaderVariation(0)->GetResShaderProgram(g_TextureResShader.codeType)->GetShader();

        pCommandBuffer->ClearColorTarget(pTarget, clearColor, NULL);
        pCommandBuffer->SetRenderTargets(1, &pTarget, NULL);
        pCommandBuffer->SetViewportScissorState(&g_CopyLinearTextureViewportScissorState);

        MakeRenderRectangleCommand(pCommandBuffer, pShader, &g_DestTextureView, layer, miplevel);

        pCommandBuffer->SetTextureStateTransition(
                &g_CopyLinearTexture,
                NULL /* pRange */,
                nn::gfx::TextureState_ColorTarget,
                nn::gfx::PipelineStageBit_RenderTarget,
                nn::gfx::TextureState_CopyDestination ,
                nn::gfx::ShaderStageBit_All
            );

        pCommandBuffer->CopyImageToBuffer(&g_CopyLinearBufferOnlineCompression, 0, &g_CopyLinearTexture, region);

        pCommandBuffer->SetTextureStateTransition(
                &g_CopyLinearTexture,
                NULL /* pRange */,
                nn::gfx::TextureState_CopyDestination,
                nn::gfx::ShaderStageBit_All,
                nn::gfx::TextureState_ColorTarget,
                nn::gfx::PipelineStageBit_RenderTarget
            );

        //
        pCommandBuffer->ClearColorTarget(pTarget, clearColor, NULL);
        pCommandBuffer->SetRenderTargets(1, &pTarget, NULL);
        pCommandBuffer->SetViewportScissorState(&g_CopyLinearTextureViewportScissorState);


        MakeRenderRectangleCommand(pCommandBuffer, pShader, pBc1SourceTextureView, layer, miplevel);

        pCommandBuffer->SetTextureStateTransition(
                &g_CopyLinearTexture,
                NULL /* pRange */,
                nn::gfx::TextureState_ColorTarget,
                nn::gfx::PipelineStageBit_RenderTarget,
                nn::gfx::TextureState_CopyDestination ,
                nn::gfx::ShaderStageBit_All
            );

        pCommandBuffer->CopyImageToBuffer(&g_CopyLinearBufferOfflineCompression, 0, &g_CopyLinearTexture, region);

    }


    void InitializeSampler()
    {
        int samplerSlot = g_GraphicsFramework.AllocateDescriptorSlot(nn::gfx::DescriptorPoolType_Sampler,1);
        {
            nn::gfx::Sampler::InfoType info;
            info.SetDefault();
            info.SetFilterMode(nn::gfx::FilterMode_MinPoint_MagPoint_MipPoint);
            info.SetAddressU(nn::gfx::TextureAddressMode_ClampToEdge);
            info.SetAddressV(nn::gfx::TextureAddressMode_ClampToEdge);
            info.SetAddressW(nn::gfx::TextureAddressMode_ClampToEdge);
            g_SamplerMinMagMipPointEdgeClamp.Initialize(g_GraphicsFramework.GetDevice(),info);
        }

        g_GraphicsFramework.SetSamplerToDescriptorPool(samplerSlot,&g_SamplerMinMagMipPointEdgeClamp);
        nn::gfx::DescriptorPool* pSamplerDescriptorPool = g_GraphicsFramework.GetDescriptorPool(nn::gfx::DescriptorPoolType_Sampler);
        pSamplerDescriptorPool->GetDescriptorSlot(&g_SamplerDescriptorSlot,samplerSlot);
    }


    void MountContents()
    {
        static const size_t MountRomCacheBufferSize = 4 * 1024;
        static char s_MountRomCacheBuffer[MountRomCacheBufferSize];
        size_t mountRomCacheUseSize = 0;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::QueryMountRomCacheSize(&mountRomCacheUseSize));
        NN_ASSERT(mountRomCacheUseSize <= MountRomCacheBufferSize);

        NN_LOG("Mount Contents\n");
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountRom("Contents", s_MountRomCacheBuffer, MountRomCacheBufferSize));
        nn::fs::DirectoryEntry entryList[100];
        nn::fs::DirectoryHandle h = {};
        nn::fs::OpenDirectory(&h, "Contents:/", nn::fs::OpenDirectoryMode_All);
        int64_t n = 0;
        nn::fs::ReadDirectory(&n, entryList, h, 100);
        NN_LOG("%d entry found.\n", static_cast<int>(n));
        for(int64_t i = 0; i < n; i++)
        {
            auto& e =entryList[i];
            NN_LOG("  %s%s\n", e.name, (e.directoryEntryType == nn::fs::DirectoryEntryType_Directory ? "/" : ""));
        }
        nn::fs::CloseDirectory(h);
    }


    namespace perf
    {
        void*  g_pLoadMeterMemory = NULL;

        nns::gfx::PrimitiveRenderer::MeterDrawer g_MeterDrawer;


        //---------------------------------------------------------------
        // Initialize the Debug Font
        //---------------------------------------------------------------
        nns::gfx::GraphicsFramework::DebugFontTextWriter g_Writer;
        //nn::util::BytePtr g_DebugFontHeap( NULL );
        void InitializeDebugFont()
        {
            g_GraphicsFramework.InitializeDebugFontTextWriter(&g_Writer, 1024);
        }

        //---------------------------------------------------------------
        // Free the Debug Font
        //---------------------------------------------------------------
        void FinalizeDebugFont()
        {
            g_GraphicsFramework.FinalizeDebugFontTextWriter(&g_Writer);
        }

        //---------------------------------------------------------------
        // Initialize the Load Meter
        //---------------------------------------------------------------
        void InitializeLoadMeter()
        {
            nn::perf::LoadMeterCenterInfo info;
            info.SetCoreCount( 1 );
            info.SetCpuBufferCount( 2 );
            info.SetGpuBufferCount( 2 );
            info.SetCpuSectionCountMax( 64 );
            info.SetGpuSectionCountMax( 64 );
            size_t memorySize = NN_PERF_GET_BUFFER_SIZE(info);
            size_t memoryAlignment = NN_PERF_GET_BUFFER_ALIGNMENT();
            g_pLoadMeterMemory = g_GraphicsFramework.AllocateMemory(memorySize, memoryAlignment);

            size_t memoryPoolSize = NN_PERF_GET_MEMORY_POOL_SIZE(g_GraphicsFramework.GetDevice(), info);
            size_t memoryPoolAlignment = NN_PERF_GET_MEMORY_POOL_ALIGNMENT(g_GraphicsFramework.GetDevice(), info);


            ptrdiff_t memoryPoolOffset = g_GraphicsFramework.AllocatePoolMemory(nns::gfx::GraphicsFramework::MemoryPoolType::MemoryPoolType_Data, memoryPoolSize, memoryPoolAlignment);
            nn::gfx::MemoryPool* pMemoryPool = g_GraphicsFramework.GetMemoryPool(nns::gfx::GraphicsFramework::MemoryPoolType::MemoryPoolType_Data);

            NN_PERF_INITIALIZE_METER(
                g_GraphicsFramework.GetDevice(), info,
                g_pLoadMeterMemory, memorySize,
                pMemoryPool,
                memoryPoolOffset, memoryPoolSize);
        }

        //---------------------------------------------------------------
        // Free the load meter.
        //---------------------------------------------------------------
        void FinalizeLoadMeter()
        {
            NN_PERF_FINALIZE_METER(g_GraphicsFramework.GetDevice());
            g_GraphicsFramework.FreeMemory(g_pLoadMeterMemory);
            g_pLoadMeterMemory = NULL;
        }

        void InitializeGfxObjects()
        {
            InitializeDebugFont();
            InitializeLoadMeter();
        }

        void FinalizeGfxObjects()
        {
            FinalizeLoadMeter();
            FinalizeDebugFont();
        }
    }
}


struct TestResult
{
    nn::TimeSpan        timeLoad;
    nn::TimeSpan        timeDecompression;
    nn::TimeSpan        timeBc1Compression;
    nn::TimeSpan        timeSetTextureStateTransition;

    float               sourceBpp;
    float               sourcePsnr;
    float               bc1OnlinePsnr;
    float               bc1OfflinePsnr;
    size_t              width;
    size_t              height;
};


double ComputePeakSignalToNoiseRatio(const uint8_t* buffer0, const uint8_t* buffer1, int width, int height, ptrdiff_t stride, int bpp)
{
    int lineDataCount = width * bpp / 8;

    uint64_t mse = 0;

    const uint8_t* buffer0LineData    = buffer0;
    const uint8_t* buffer1LineData    = buffer1;

    for (int line = 0; line < width; ++line)
    {

        for (int colorDataIndex = 0; colorDataIndex < lineDataCount; ++colorDataIndex)
        {
            uint8_t value0 = buffer0LineData[colorDataIndex];
            uint8_t value1 = buffer1LineData[colorDataIndex];

            int diff = (int)value0 - (int)value1;
            mse += (diff * diff);

        }

        buffer0LineData += stride;
        buffer1LineData += stride;
    }

    mse /= (width * height);

    double max_i = 255.0f;

    double psnr = 20.0 * log10(max_i) - 10 * log10((double)mse);

    return psnr;

}





enum SourceFileFormat
{
#if defined(TEST_GFXUTIL_TEXTURE_COMPRESSION_BENCHMARK_ENABLE_JXR)
    SourceFileFormat_Jxr,
#endif
    SourceFileFormat_Jpeg,
    SourceFileFormat_Bc1Dds,
};


const char* GetFileFormatExtension(SourceFileFormat format)
{
    switch (format)
    {
#if defined(TEST_GFXUTIL_TEXTURE_COMPRESSION_BENCHMARK_ENABLE_JXR)
        case SourceFileFormat_Jxr:      return "jxr";
#endif
        case SourceFileFormat_Jpeg:     return "jpg";
        case SourceFileFormat_Bc1Dds:   return "dds";
       default:                         NN_ASSERT(0); return NULL;
    }
}

size_t dotest_TextureCompression_Bc1_Unorm_Frame(const char* compressedTextureInputPath, SourceFileFormat format, BmpFileInfo* pBmpInfo, int mipLevels, int layerCount)
{
    size_t compressedFileSizeInBytes = 0;


    NN_PERF_BEGIN_FRAME();

    NN_PERF_BEGIN_MEASURE_NAME("load_source_from_rom");
    compressedFileSizeInBytes = LoadBinaryFileToBuffer(compressedTextureInputPath);
    NN_PERF_END_MEASURE();

    // テクスチャ取得と vsync 待ち
    g_GraphicsFramework.AcquireTexture(0);
    g_GraphicsFramework.WaitDisplaySync(0, nn::TimeSpan::FromSeconds(2));

    nn::gfx::ColorTargetView* pTarget = g_GraphicsFramework.GetColorTargetView();

    g_pTransientTextureDescriptorAllocator->Free();
    g_pTransientSamplerDescriptorAllocator->Free();
    g_pTransientTextureDescriptorAllocator->Begin();
    g_pTransientSamplerDescriptorAllocator->Begin();

    g_GraphicsFramework.BeginFrame(0);
    {
        nn::gfx::CommandBuffer* rootCommandBuffer = g_GraphicsFramework.GetRootCommandBuffer(0);

        NN_PERF_SET_COLOR(nn::util::Color4u8::Yellow());
        NN_PERF_BEGIN_MEASURE_NAME("source_decompression");


        if (format == SourceFileFormat_Jpeg)
        {
            int jpegWidth = 0;
            int jpegHeight = 0;
            ptrdiff_t jpegStride = 0;
            JpegDecodeBitmap(fileBuffer, compressedFileSizeInBytes, sourceDecompressBuffer, sizeof(sourceDecompressBuffer), &jpegWidth, &jpegHeight, &jpegStride);
            NN_ASSERT(jpegWidth == pBmpInfo->width);
            NN_ASSERT(jpegHeight == pBmpInfo->height);
            NN_ASSERT(jpegStride == pBmpInfo->stride);

        }
#if defined(TEST_GFXUTIL_TEXTURE_COMPRESSION_BENCHMARK_ENABLE_JXR)
        else  if (format == SourceFileFormat_Jxr)
        {
            int jxrWidth = 0;
            int jxrHeight = 0;
            ptrdiff_t jxrStride = 0;
            JxrDecodeBitmap(fileBuffer, compressedFileSizeInBytes, sourceDecompressBuffer, sizeof(sourceDecompressBuffer), &jxrWidth, &jxrHeight, &jxrStride);
            NN_ASSERT(jpegWidth == pBmpInfo->width);
            NN_ASSERT(jpegHeight == pBmpInfo->height);
            NN_ASSERT(jpegStride == pBmpInfo->stride);
        }
#endif
        else
        {
            NN_ASSERT(0); // invalid format
        }

        NN_PERF_END_MEASURE();

        NN_PERF_SET_COLOR(nn::util::Color4u8::Blue());
        NN_PERF_BEGIN_MEASURE_NAME("update_source_texture");

        UpdateSourceTextureContent(sourceDecompressBuffer, sizeof(sourceDecompressBuffer), pBmpInfo->width, pBmpInfo->height, pBmpInfo->stride);

        NN_PERF_END_MEASURE();

        // 圧縮先に圧縮元を描画する
        {
            NN_PERF_SET_COLOR_GPU(nn::util::Color4u8::Red());
            NN_PERF_BEGIN_MEASURE_NAME_GPU(rootCommandBuffer, "bc1_compression");

            RenderToDestTexture(rootCommandBuffer);

            NN_PERF_END_MEASURE_GPU(rootCommandBuffer);

            NN_PERF_SET_COLOR_GPU(nn::util::Color4u8::Blue());
            NN_PERF_BEGIN_MEASURE_NAME_GPU(rootCommandBuffer, "call_SetTextureStateTransition");
            rootCommandBuffer->SetTextureStateTransition(
                &g_DestTexture,
                NULL /* pRange */,
                nn::gfx::TextureState_ColorTarget,
                nn::gfx::PipelineStageBit_RenderTarget,
                nn::gfx::TextureState_ShaderRead,
                nn::gfx::PipelineStageBit_PixelShader
            );

            NN_PERF_END_MEASURE_GPU(rootCommandBuffer);
        }

        {
            CopyToLinearBuffer(rootCommandBuffer, pBmpInfo->width, pBmpInfo->height, 0, 0, g_pResTextureFile->GetResTexture(0)->GetTextureView());
        }

        NN_PERF_SET_COLOR_GPU(nn::util::Color4u8::Yellow());
        NN_PERF_BEGIN_MEASURE_NAME_GPU(rootCommandBuffer, "RenderTest");

        //バックバッファをクリア・描画対象に
        rootCommandBuffer->ClearColor(pTarget, .05f, .05f, .05f, 1.0f, NULL);
        rootCommandBuffer->SetRenderTargets(1, &pTarget, NULL);

        RenderSourceTexture(rootCommandBuffer, mipLevels, layerCount);
        RenderDestTexture(rootCommandBuffer, mipLevels, layerCount);
        RenderOfflineBc1Texture(rootCommandBuffer);

        NN_PERF_END_MEASURE_GPU(rootCommandBuffer);
    }


    g_GraphicsFramework.EndFrame(0);

    g_GraphicsFramework.ExecuteCommand(0);
    g_GraphicsFramework.QueuePresentTexture(1);
    g_GraphicsFramework.WaitGpuSync(0, nn::TimeSpan::FromSeconds(2));

    g_pTransientTextureDescriptorAllocator->End();
    g_pTransientSamplerDescriptorAllocator->End();

    NN_PERF_END_FRAME();

    return compressedFileSizeInBytes;
}

void dotest_TextureCompression_Bc1_Unorm(const char* textureFileName, SourceFileFormat format, const char* qfactor, TestResult* pTestResult)
{
    const nn::gfx::ImageFormat srcFormat = nn::gfx::ImageFormat_B8_G8_R8_A8_Unorm;
    const nn::gfx::ImageFormat dstFormat = nn::gfx::ImageFormat_Bc1_Unorm;

    memset(pTestResult, 0, sizeof(TestResult));

    char compressedTextureInputPath[128];
    char originalTextureInputPath[128];
    char offlineBc1CompressedInputPath[128];

    const char* fileExtension = GetFileFormatExtension(format);

    sprintf(compressedTextureInputPath, "Contents:/compressed/%s_%s.%s", textureFileName, qfactor, fileExtension);
    sprintf(originalTextureInputPath, "Contents:/original/%s.bmp", textureFileName);
    sprintf(offlineBc1CompressedInputPath, "Contents:/original/%s.bntx", textureFileName);

    int     mipLevels = 1;
    int     layerCount = 1;

    size_t originalFileSizeInBytes = LoadBinaryFileToBuffer(originalTextureInputPath);

    BmpFileInfo bmpInfo;
    memset(&bmpInfo, 0, sizeof(bmpInfo));

    BmpInputStream bmpIs;
    bmpIs.pBuffer = static_cast<uint8_t*>(fileBuffer);
    bmpIs.bufferSize = originalFileSizeInBytes;
    bmpIs.readOffset = 0;

    bool bFileOk = false;
    if (strstr(originalTextureInputPath, ".bmp") == (originalTextureInputPath + strlen(originalTextureInputPath) - 4))
        bFileOk = ReadBmpFromStream(&bmpIs, &bmpInfo, bmpDecompressBuffer, sizeof(bmpDecompressBuffer));
    else if (strstr(originalTextureInputPath, ".tga") == (originalTextureInputPath + strlen(originalTextureInputPath) - 4))
        bFileOk = ReadTgaFromStream(&bmpIs, &bmpInfo, bmpDecompressBuffer, sizeof(bmpDecompressBuffer));
    NN_ASSERT(bFileOk);


    NN_LOG("Initialize GraphicsFramework\n");
    nns::gfx::GraphicsFramework::FrameworkInfo fwInfo;
    {
        fwInfo.SetDefault();
        fwInfo.SetMemoryPoolSize(nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget, 64 * 1024 * 1024);
        fwInfo.SetMemoryPoolSize(nns::gfx::GraphicsFramework::MemoryPoolType_Data, 32 * 1024 * 1024);
        fwInfo.SetDebugMode(nn::gfx::DebugMode_Full);
    }

    g_GraphicsFramework.Initialize(fwInfo);

    NN_LOG("Initialize Shaders\n");
    memset(&g_TextureCompressionShader, 0, sizeof(g_TextureCompressionShader));
    memset(&g_TextureResShader, 0, sizeof(g_TextureResShader));

    InitializeResShaderFile(&g_TextureCompressionShader, TextureCompressionShaderBinaryFilePath);
    InitializeResShaderFile(&g_TextureResShader,RenderTextureShaderBinaryFilePath);

    InitializeSampler();
    {
        NN_LOG("Initialize DescriptorAllocatorHolder\n");
        g_TransientTextureDescriptorAllocatorHolder.Initialize(
            g_GraphicsFramework.GetDescriptorPool(nn::gfx::DescriptorPoolType_TextureView),
            g_GraphicsFramework.AllocateDescriptorSlot(nn::gfx::DescriptorPoolType_TextureView,1024),
            1024);
        g_pTransientTextureDescriptorAllocator = g_TransientTextureDescriptorAllocatorHolder.Get();

        g_TransientSamplerDescriptorAllocatorHolder.Initialize(
            g_GraphicsFramework.GetDescriptorPool(nn::gfx::DescriptorPoolType_Sampler),
            g_GraphicsFramework.AllocateDescriptorSlot(nn::gfx::DescriptorPoolType_Sampler,1024),
            1024);
        g_pTransientSamplerDescriptorAllocator = g_TransientSamplerDescriptorAllocatorHolder.Get();
    }
    g_pTransientTextureDescriptorAllocator->FillHistory();
    g_pTransientSamplerDescriptorAllocator->FillHistory();


    InitializeGfxObjects(srcFormat, dstFormat, bmpInfo.width, bmpInfo.height, mipLevels, layerCount);
    perf::InitializeGfxObjects();
    InitializeOfflineBc1CompressedTexture(offlineBc1CompressedInputPath);


    size_t compressedFileSizeInBytes = 0;
    compressedFileSizeInBytes = dotest_TextureCompression_Bc1_Unorm_Frame(compressedTextureInputPath, format, &bmpInfo, mipLevels, layerCount);

    NN_PERF_DUMP();

    CopyLinearBufferContent(&g_CopyLinearBufferOnlineCompression, bc1OnlineResultBuffer, sizeof(bc1OnlineResultBuffer), bmpInfo.width, bmpInfo.height, bmpInfo.stride);
    CopyLinearBufferContent(&g_CopyLinearBufferOfflineCompression, bc1OfflineResultBuffer, sizeof(bc1OfflineResultBuffer), bmpInfo.width, bmpInfo.height, bmpInfo.stride);

    double sourcePsnr       = ComputePeakSignalToNoiseRatio(sourceDecompressBuffer, bmpDecompressBuffer, bmpInfo.width, bmpInfo.height, bmpInfo.stride, bmpInfo.bpp);
    double bc1OnlinePsnr    = ComputePeakSignalToNoiseRatio(bc1OnlineResultBuffer, bmpDecompressBuffer, bmpInfo.width, bmpInfo.height, bmpInfo.stride, bmpInfo.bpp);
    double bc1OfflinePsnr   = ComputePeakSignalToNoiseRatio(bc1OfflineResultBuffer, bmpDecompressBuffer, bmpInfo.width, bmpInfo.height, bmpInfo.stride, bmpInfo.bpp);

    pTestResult->timeLoad                       = NN_PERF_GET_MAX_ELAPSED_TIME_CPU(NULL, "load_source_from_rom", 0);
    pTestResult->timeDecompression              = NN_PERF_GET_MAX_ELAPSED_TIME_CPU(NULL, "source_decompression", 0);
    pTestResult->timeBc1Compression             = NN_PERF_GET_MAX_ELAPSED_TIME_GPU("bc1_compression", 0);
    pTestResult->timeSetTextureStateTransition  = NN_PERF_GET_MAX_ELAPSED_TIME_GPU("call_SetTextureStateTransition", 0);

    pTestResult->sourceBpp              = static_cast<float>(compressedFileSizeInBytes) / static_cast<float>(bmpInfo.width * bmpInfo.height);
    pTestResult->sourcePsnr             = static_cast<float>(sourcePsnr);
    pTestResult->bc1OnlinePsnr          = static_cast<float>(bc1OnlinePsnr);
    pTestResult->bc1OfflinePsnr         =static_cast<float>(bc1OfflinePsnr);
    pTestResult->width                  = bmpInfo.width;
    pTestResult->height                 = bmpInfo.height;


    FinalizeOfflineBc1CompressedTexture();
    perf::FinalizeGfxObjects();
    FinalizeGfxObjects(bmpInfo.width, bmpInfo.height, mipLevels, layerCount);

    NN_LOG("Finalize DescriptorAllocatorHolder\n");
    g_TransientTextureDescriptorAllocatorHolder.Finalize();
    g_SamplerMinMagMipPointEdgeClamp.Finalize(g_GraphicsFramework.GetDevice());
    g_TransientSamplerDescriptorAllocatorHolder.Finalize();

    NN_LOG("Finalize Shaders\n");
    FinalizeResShaderFile(&g_TextureResShader);
    FinalizeResShaderFile(&g_TextureCompressionShader);

    NN_LOG("Finalize GraphicsFramework\n");
    g_GraphicsFramework.Finalize();
}

const char* g_textTextureName[] = {
    "Fish_body",
    "Fish_body_spm",
    "MaleA_clothA",
    "MaleA_face",
    "MaleA_face_spm",
    "MaleA_leather",
};

const size_t g_testTextureCount = sizeof(g_textTextureName) / sizeof(g_textTextureName[0]);

#if defined(TEST_GFXUTIL_TEXTURE_COMPRESSION_BENCHMARK_ENABLE_JXR)
const char* g_jxrQfactor[]  = { "1", "2", "3", "4", "5", "6", "10", "15", "20", "30", "40", "50", "60", "80", "100", "150" };
const size_t g_jxrQfactorCount = sizeof(g_jxrQfactor) / sizeof(g_jxrQfactor[0]);

TestResult jxrTestResult[g_testTextureCount][g_jxrQfactorCount];
#endif


const char* g_jpegQuality[] = { "100", "99", "98", "97", "95", "90", "85", "80", "70", "60", "50" };//  "50", "60", "70", "80", "85", "90", "95", "97", "98", "99", "100" };
const size_t g_jpegQualityCount = sizeof(g_jpegQuality) / sizeof(g_jpegQuality[0]);


TestResult jpegTestResult[g_testTextureCount][g_jpegQualityCount];


void PrintResultHeader(SourceFileFormat format, const char* textureName, size_t width, size_t height)
{
    const char* fileExtension = GetFileFormatExtension(format);
    printf("--- \n%s.%s (%zu * %zu)\n", textureName, fileExtension, width, height);
    printf("quality;bpp;loading time (us);decompression time (us);bc1 compression time (us); SetTextureStateTransition time (us);jxr psnr (db);bc1 psnr (db); bc1 offline psnr (db)\n");

}
void PrintResult(const TestResult* pTestResult, const char* quality)
{
    printf("%s;%f;%" PRId64 ";%" PRId64 ";%" PRId64 ";%" PRId64 ";%f;%f;%f\n",
        quality, pTestResult->sourceBpp,
        pTestResult->timeLoad.GetMicroSeconds(), pTestResult->timeDecompression.GetMicroSeconds(),
        pTestResult->timeBc1Compression.GetMicroSeconds(), pTestResult->timeSetTextureStateTransition.GetMicroSeconds(),
        pTestResult->sourcePsnr, pTestResult->bc1OnlinePsnr, pTestResult->bc1OfflinePsnr);
}

extern "C" void nnMain()
{
    MountContents();

    nns::gfx::GraphicsFramework::InitializeGraphicsSystem(GraphicsSystemMemorySize);

    for (size_t textureIndex = 0; textureIndex < g_testTextureCount; ++textureIndex)
    {
#if defined(TEST_GFXUTIL_TEXTURE_COMPRESSION_BENCHMARK_ENABLE_JXR)
        for (size_t qfactorIndex = 0; qfactorIndex < g_jxrQfactorCount; ++qfactorIndex)
        {
            dotest_TextureCompression_Bc1_Unorm(g_textTextureName[textureIndex], SourceFileFormat_Jxr, g_jxrQfactor[qfactorIndex], &jxrTestResult[textureIndex][qfactorIndex]);
        }
#endif

        for (size_t jpegQualityIndex = 0; jpegQualityIndex < g_jpegQualityCount; ++jpegQualityIndex)
        {
            dotest_TextureCompression_Bc1_Unorm(g_textTextureName[textureIndex], SourceFileFormat_Jpeg, g_jpegQuality[jpegQualityIndex], &jpegTestResult[textureIndex][jpegQualityIndex]);
        }
    }


    for (size_t textureIndex = 0; textureIndex < g_testTextureCount; ++textureIndex)
    {
#if defined(TEST_GFXUTIL_TEXTURE_COMPRESSION_BENCHMARK_ENABLE_JXR)
        PrintResultHeader(SourceFileFormat_Jxr, g_textTextureName[textureIndex], jxrTestResult[textureIndex][0].width, jxrTestResult[textureIndex][0].height);
        for (size_t qfactorIndex = 0; qfactorIndex < g_jxrQfactorCount; ++qfactorIndex)
        {
            const TestResult* pTestResult = &jxrTestResult[textureIndex][qfactorIndex];
            PrintResult(pTestResult, g_jxrQfactor[qfactorIndex]);
        }
#endif

        PrintResultHeader(SourceFileFormat_Jpeg, g_textTextureName[textureIndex], jpegTestResult[textureIndex][0].width, jpegTestResult[textureIndex][0].height);
        for (size_t jpegQualityIndex = 0; jpegQualityIndex < g_jpegQualityCount; ++jpegQualityIndex)
        {
            const TestResult* pTestResult = &jpegTestResult[textureIndex][jpegQualityIndex];
            PrintResult(pTestResult, g_jpegQuality[jpegQualityIndex]);
        }
    }

    //コンテンツをアンマウントする
    nn::fs::Unmount("Contents");

}

//-----------------------------------------------------------------------------
//  Startup Functions
//-----------------------------------------------------------------------------

extern "C" void nninitStartup()
{
// Set the total size of the memory heap
    const size_t MemoryHeapSize = 1024 * 1024 * 1024;
    auto result = nn::os::SetMemoryHeapSize( MemoryHeapSize );
    NN_ASSERT(result.IsSuccess(), "Cannot set the memory heap size.");

// Allocated the memory space used by <tt>malloc</tt> from the memory heap
    uintptr_t address;
    result = nn::os::AllocateMemoryBlock( &address, MemoryHeapSize );
    NN_ASSERT( result.IsSuccess(), "Cannot allocate the memory block." );

// Set a memory space for <tt>malloc</tt>
    nn::init::InitializeAllocator( reinterpret_cast<void*>(address), MemoryHeapSize );
}
