﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// シングルスレッド動作の確認テスト

#include <nn/nn_Common.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nnt.h>

#include <nn/gfx/util/detail/gfx_OffsetRingBuffer.h>

// 普通に使えそうなことの確認
TEST(OffsetRingBuffer, FixedUseCase)
{
    const ptrdiff_t Size = 1024 * 1024;
    const ptrdiff_t BaseOffset = 1024;
    const ptrdiff_t InvalidOffset = nn::gfx::util::detail::OffsetRingBuffer::InvalidOffset;

    nn::gfx::util::detail::OffsetRingBuffer ring;

    EXPECT_FALSE(ring.IsInitialized());

    ring.Initialize(BaseOffset, Size);

    EXPECT_TRUE(ring.IsInitialized());
    EXPECT_FALSE(ring.IsRecording());
    EXPECT_EQ(Size, ring.GetSize());
    EXPECT_EQ(BaseOffset, ring.GetBaseOffset());

    ring.Begin();
    EXPECT_TRUE(ring.IsRecording());

    ptrdiff_t o1 = ring.Allocate(100, 8);
    EXPECT_EQ(BaseOffset + 0, o1);

    ptrdiff_t o2 = ring.Allocate(100, 8);
    EXPECT_EQ(BaseOffset + 104, o2);

    ptrdiff_t o3 = ring.Allocate(100, 128);
    EXPECT_EQ(BaseOffset + 256, o3);

    ptrdiff_t o4 = ring.Allocate(900 * 1024, 1024);
    EXPECT_EQ(BaseOffset + 1024, o4);

    nn::gfx::util::detail::OffsetRange range1;
    ring.End(&range1);
    EXPECT_FALSE(ring.IsRecording());
    EXPECT_EQ(0, range1.base);
    EXPECT_EQ(901 * 1024, range1.size);

    ring.Begin();
    EXPECT_TRUE(ring.IsRecording());

    ptrdiff_t o5 = ring.Allocate(200 * 1024, 1024);
    EXPECT_EQ(InvalidOffset, o5);

    ring.ReleaseOffsetRange(&range1);

    ptrdiff_t o6 = ring.Allocate(200 * 1024, 1024);
    EXPECT_EQ(BaseOffset + 0, o6);

    nn::gfx::util::detail::OffsetRange range2;
    ring.End(&range2);
    EXPECT_FALSE(ring.IsRecording());
    EXPECT_EQ(901 * 1024, range2.base);
    EXPECT_EQ(323 * 1024, range2.size);

    nn::gfx::util::detail::OffsetRange range3 = {};

    ring.ReleaseOffsetRange(&range3);
    ring.Begin();
    EXPECT_TRUE(ring.IsRecording());

    ptrdiff_t o7 = ring.Allocate(100 * 1024, 512);
    EXPECT_EQ(BaseOffset + 200 * 1024, o7);

    ptrdiff_t o8 = ring.Allocate(900 * 1024, 1024);
    EXPECT_EQ(InvalidOffset, o8);

    nn::gfx::util::detail::OffsetRange range4;
    ring.End(&range4);
    EXPECT_FALSE(ring.IsRecording());
    EXPECT_EQ(200 * 1024, range4.base);
    EXPECT_EQ(100 * 1024, range4.size);

    ring.Finalize();
    EXPECT_FALSE(ring.IsInitialized());
}// NOLINT(impl/function_size)

// Allocate の境界条件をつつきまわす
TEST(OffsetRingBuffer, AllocateBoundaryCondition)
{
    const ptrdiff_t Size = 1024 * 1024;
    const ptrdiff_t InvalidOffset = nn::gfx::util::detail::OffsetRingBuffer::InvalidOffset;

    // ざっくりした size 検査
    {
        nn::gfx::util::detail::OffsetRingBuffer ring;
        ring.Initialize(0, Size);
        ring.Begin();

        // size == 0 に対しては InvalidOffset を返す
        EXPECT_EQ(InvalidOffset, ring.Allocate(0, 1));
        // 大きすぎる size に対しては InvalidOffset を返す
        EXPECT_EQ(InvalidOffset, ring.Allocate(Size, 1));
    }

    // alignment の調整
    {
        {
            nn::gfx::util::detail::OffsetRingBuffer ring;
            nn::gfx::util::detail::OffsetRange range;
            ring.Initialize(0, Size);
            ring.Begin();

            // 先頭から alignment が大きすぎるアロケート
            EXPECT_EQ(0, ring.Allocate(1, Size * 2));

            ring.Allocate(300, 1);
            ring.End(&range);
            ring.ReleaseOffsetRange(&range);
            ring.Begin();

            // 1 周目の途中から alignment が大きすぎるアロケート
            EXPECT_EQ(0, ring.Allocate(1, Size * 2));

            ring.End(&range);
            ring.ReleaseOffsetRange(&range);
            ring.Begin();

            // 2 周目の途中から alignmnet が大きすぎるアロケート
            ring.Allocate(Size + Size / 2, 1);
            EXPECT_EQ(InvalidOffset, ring.Allocate(1, Size * 2));
        }

        {
            nn::gfx::util::detail::OffsetRingBuffer ring;
            nn::gfx::util::detail::OffsetRange range;

            ring.Initialize(0,300);

            ring.Begin();
            ring.Allocate(260, 1);
            ring.End(&range);
            ring.ReleaseOffsetRange(&range);      // head = 260, tail = 559
            ring.Begin();

            // alignment を考慮して 1 周目から取る
            ptrdiff_t o1 = ring.Allocate(8, 8);   // head = 272, tail = 559
            EXPECT_EQ(264, o1);
            // alignment 丁度で 1 周目から取る
            ptrdiff_t o2 = ring.Allocate(8, 8);
            EXPECT_EQ(272, o2);

            // alignment の都合で 2 周目に突入する
            ptrdiff_t o3 = ring.Allocate(8, 256); // head = 308, tail = 559
            EXPECT_EQ(0, o3);
            // alignment 丁度で 2 周目から取る
            ptrdiff_t o4 = ring.Allocate(8, 8); // head = 316, tail = 559
            EXPECT_EQ(8, o4);
        }

        {
            nn::gfx::util::detail::OffsetRingBuffer ring;
            nn::gfx::util::detail::OffsetRange range;

            ring.Initialize(0, 300);         // head =   0, tail = 299

            ring.Begin();
            ring.Allocate(299, 1);              // head = 299, tail = 299
            ring.End(&range);
            ring.ReleaseOffsetRange(&range);    // head = 299, tail = 598
            ring.Begin();

            ring.Allocate(260, 1);              // head = 560, tail = 598

            // alignment を考慮して 2 周目からとる
            ptrdiff_t o1 = ring.Allocate(1, 8); // head = 565, tail = 598
            EXPECT_EQ(264, o1);

            // alignment を考慮したら 3 周目に突入してしまう
            ptrdiff_t o2 = ring.Allocate(1, 256);
            EXPECT_EQ(InvalidOffset, o2); // どのみち tail に引っかかって Invalid が返ってくるのでテストできてるか疑問。
        }
    }

    // きちっとした size 検査
    {
        {
            nn::gfx::util::detail::OffsetRingBuffer ring;
            nn::gfx::util::detail::OffsetRange r1;
            nn::gfx::util::detail::OffsetRange r2;

            ring.Initialize(0, 100);            // head =   0, tail =  99
            ring.Begin();
            EXPECT_EQ( 0, ring.Allocate(10, 1));// head =  10, tail =  99
            ring.End(&r1);                      // head =  10, tail =  99
            ring.Begin();
            EXPECT_EQ(10, ring.Allocate(89, 1));// head =  99, tail =  99
            ring.End(&r2)   ;                   // head =  99, tail =  99
            ring.ReleaseOffsetRange(&r1);       // head =  99, tail = 109
            ring.Begin();
            EXPECT_EQ( 0, ring.Allocate(5,1));  // head = 105, tail = 109
            ring.End(&r1);                      // head =   5, tail =   9

            ring.Begin();
            ptrdiff_t o1 = ring.Allocate(10,1);
            EXPECT_EQ(InvalidOffset, o1);
        }

    }
}// NOLINT(impl/function_size)

TEST(OffserRingBufferDeathTest, DeathTest)
{
    nn::gfx::util::detail::OffsetRingBuffer ring;
    nn::gfx::util::detail::OffsetRange range;

    // Initialize 前に呼ぶと死ぬことの確認
    EXPECT_DEATH_IF_SUPPORTED(ring.Finalize(), "");
    EXPECT_DEATH_IF_SUPPORTED(ring.IsRecording(), "");
    EXPECT_DEATH_IF_SUPPORTED(ring.GetBaseOffset(), "");
    EXPECT_DEATH_IF_SUPPORTED(ring.GetSize(), "");
    EXPECT_DEATH_IF_SUPPORTED(ring.Begin(), "");
    EXPECT_DEATH_IF_SUPPORTED(ring.End(&range), "");
    EXPECT_DEATH_IF_SUPPORTED(ring.ReleaseOffsetRange(&range), "");
    EXPECT_DEATH_IF_SUPPORTED(ring.Allocate(1,1), "");

    // Initialize のパラメータ検査
    EXPECT_DEATH_IF_SUPPORTED(ring.Initialize(-1, 100), "");
    EXPECT_DEATH_IF_SUPPORTED(ring.Initialize(0, 0), "");
    EXPECT_DEATH_IF_SUPPORTED(ring.Initialize(0, nn::gfx::util::detail::OffsetRingBuffer::SizeMax + 1), "");
    EXPECT_DEATH_IF_SUPPORTED(ring.Initialize(PTRDIFF_MAX - 98, 100), "");

    ring.Initialize(64, 1024);

    // Initialize 後に呼ぶと死ぬことの確認
    EXPECT_DEATH_IF_SUPPORTED(ring.Initialize(0,100), "");

    // Begin 前に呼ぶと死ぬことの確認
    EXPECT_DEATH_IF_SUPPORTED(ring.End(&range), "");
    EXPECT_DEATH_IF_SUPPORTED(ring.Allocate(1,1), "");

    ring.Begin();

    // Begin 後に呼ぶと死ぬことの確認
    EXPECT_DEATH_IF_SUPPORTED(ring.Begin(), "");

    // Allocate のパラメータ検査
    EXPECT_DEATH_IF_SUPPORTED(ring.Allocate(-1,1), "");
    EXPECT_DEATH_IF_SUPPORTED(ring.Allocate(10, 0), "");
    EXPECT_DEATH_IF_SUPPORTED(ring.Allocate(10, 128), "");

    ring.Allocate(200, 1);

    // End のパラメータ検査
    EXPECT_DEATH_IF_SUPPORTED(ring.End(NULL), "");

    ring.End(&range);

    // ReleaseOffsetRange のパラメータ検査
    EXPECT_DEATH_IF_SUPPORTED(ring.ReleaseOffsetRange(NULL), "");
    // 範囲外系
    nn::gfx::util::detail::OffsetRange r0 = { -1, 100 };
    nn::gfx::util::detail::OffsetRange r1 = { 1024, 100 };
    nn::gfx::util::detail::OffsetRange r2 = { 0, -1 };
    nn::gfx::util::detail::OffsetRange r3 = { 0, 1024 };
    EXPECT_DEATH_IF_SUPPORTED(ring.ReleaseOffsetRange(&r0), "");
    EXPECT_DEATH_IF_SUPPORTED(ring.ReleaseOffsetRange(&r1), "");
    EXPECT_DEATH_IF_SUPPORTED(ring.ReleaseOffsetRange(&r2), "");
    EXPECT_DEATH_IF_SUPPORTED(ring.ReleaseOffsetRange(&r3), "");
    // 解放範囲が連続していない系
    nn::gfx::util::detail::OffsetRange r4 = { 0, 100 };
    nn::gfx::util::detail::OffsetRange r5 = { 101, 99}; // 100 が抜けている
    ring.ReleaseOffsetRange(&r4);
    EXPECT_DEATH_IF_SUPPORTED(ring.ReleaseOffsetRange(&r5), "");

    // Finalize しなくても良い。
}
