﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <limits>

#include <nn/nn_Abort.h>

#if defined(NN_BUILD_TARGET_PLATFORM_OS_NN)
#include <nn/pcm/pcm.h>
#include <nn/ts.h>
#endif

#include "testGfxUtil_PlatformMeasurementTracker.h"

PlatformMeasurementTracker::PlatformMeasurementTracker()
: m_MeasuringPointDataArray()
, m_IgnoreUpdateCount(0)
{
}

PlatformMeasurementTracker::~PlatformMeasurementTracker()
{
}

void PlatformMeasurementTracker::Initialize()
{
    InitializeHostPowerMeasuring();

    m_IgnoreUpdateCount = 0;

    for (int i = 0; i < PlatformMeasuringPoint_Count; ++i)
    {
        InitializeData(&m_MeasuringPointDataArray[i]);
    }
}

void PlatformMeasurementTracker::Finalize()
{
    FinalizeHostPowerMeasuring();
}

void PlatformMeasurementTracker::Reset()
{
    for (int i = 0; i < PlatformMeasuringPoint_Count; ++i)
    {
        InitializeData(&m_MeasuringPointDataArray[i]);
    }
}

void PlatformMeasurementTracker::ResetAndIgnoreNextUpdates(int ignoreCount)
{
    Reset();
    m_IgnoreUpdateCount = ignoreCount;
}

void PlatformMeasurementTracker::Update()
{
    if (m_IgnoreUpdateCount > 0)
    {
        m_IgnoreUpdateCount--;
        return;
    }

    for (int i = 0; i < PlatformMeasuringPoint_Count; i++)
    {
        PlatformMeasuringPoint measuringPoint = static_cast<PlatformMeasuringPoint>(i);
        ReadAndUpdateMeasuringPoint(measuringPoint);
    }
}

void PlatformMeasurementTracker::ReadAndUpdateMeasuringPoint(PlatformMeasuringPoint measuringPoint)
{
    int measuredValue = ReadHostCurrentPower(measuringPoint);
    UpdateData(&m_MeasuringPointDataArray[measuringPoint], measuredValue);
}

void PlatformMeasurementTracker::InitializeData(MeasuringPointData* pData)
{
    memset(pData, 0, sizeof(MeasuringPointData));
    pData->hasNewMax = false;
    pData->min = std::numeric_limits<int>::max();
    pData->max = 0;
    pData->average = 0;
    pData->current = 0;
    pData->measureCount = 0;
}

void PlatformMeasurementTracker::UpdateData(MeasuringPointData* pData, int newValue)
{
    pData->hasNewMax = false;

    pData->current = newValue;

    if (pData->min > newValue)
    {
        pData->min = newValue;
    }
    if (pData->max < newValue)
    {
        pData->hasNewMax = true;
        pData->max = newValue;
    }

    int historyIndex = pData->measureCount % MeasuringPointData::MeasureHistoryLength;
    pData->measureHistory[historyIndex] = newValue;
    pData->measureCount++;

    int average = ComputeAverageFromHistory(pData);
    pData->average = average;
}

int PlatformMeasurementTracker::ComputeAverageFromHistory(const MeasuringPointData* pData)
{
    int averageCount = (pData->measureCount < MeasuringPointData::MeasureHistoryLength) ? pData->measureCount : MeasuringPointData::MeasureHistoryLength;
    NN_ASSERT(averageCount > 0);

    int sum = 0;
    for (int i = 0; i < averageCount; ++i)
    {
        sum += pData->measureHistory[i];
    }

    return sum / averageCount;
}


void PlatformMeasurementTracker::InitializeHostPowerMeasuring()
{
#if defined(NN_BUILD_TARGET_PLATFORM_OS_NN)
    nn::pcm::Initialize();
    nn::ts::Initialize();
#endif
}

void PlatformMeasurementTracker::FinalizeHostPowerMeasuring()
{
#if defined(NN_BUILD_TARGET_PLATFORM_OS_NN)
    nn::ts::Finalize();
    nn::pcm::Finalize();
#endif
}

int PlatformMeasurementTracker::ReadHostCurrentPower(PlatformMeasuringPoint measuringPoint)
{
#if defined(NN_BUILD_TARGET_PLATFORM_OS_NN)
    int measureResult = 0;
    switch (measuringPoint)
    {
    case PlatformMeasuringPoint_PowerCpu:
        {
            nn::pcm::MeasuringPoint pcmMeasuringPoint = nn::pcm::MeasuringPoint_Cpu;
            if (nn::pcm::IsSupported(pcmMeasuringPoint))
            {
                measureResult = nn::pcm::ReadCurrentPower(pcmMeasuringPoint);
            }
        }
        break;
    case PlatformMeasuringPoint_PowerGpu:
        {
            nn::pcm::MeasuringPoint pcmMeasuringPoint = nn::pcm::MeasuringPoint_Gpu;
            if (nn::pcm::IsSupported(pcmMeasuringPoint))
            {
                measureResult = nn::pcm::ReadCurrentPower(pcmMeasuringPoint);
            }
        }
        break;
    case PlatformMeasuringPoint_PowerDdr:
        {
            nn::pcm::MeasuringPoint pcmMeasuringPoint = nn::pcm::MeasuringPoint_Ddr;
            if (nn::pcm::IsSupported(pcmMeasuringPoint))
            {
                measureResult = nn::pcm::ReadCurrentPower(pcmMeasuringPoint);
            }
        }
        break;
    case PlatformMeasuringPoint_PowerVsysAp:
        {
            nn::pcm::MeasuringPoint pcmMeasuringPoint = nn::pcm::MeasuringPoint_VsysAp;
            if (nn::pcm::IsSupported(pcmMeasuringPoint))
            {
                measureResult = nn::pcm::ReadCurrentPower(pcmMeasuringPoint);
            }
        }
        break;
    case PlatformMeasuringPoint_ThermalSensorInternal:
        {
            nn::Result result =
                nn::ts::GetTemperature(&measureResult, nn::ts::Location_ThermalSensorInternal);
            if (result.IsFailure())
            {
                measureResult = 0;
            }
        }
        break;
    case PlatformMeasuringPoint_ThermalSensorExternal:
        {
            nn::Result result =
                nn::ts::GetTemperature(&measureResult, nn::ts::Location_ThermalSensorExternal);
            if (result.IsFailure())
            {
                measureResult = 0;
            }
        }
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    return measureResult;

#else
    NN_UNUSED(measuringPoint);
    return 0;
#endif
}
