﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_TimeSpan.h>
#include <nn/gfx/util/gfx_DebugFontTextWriter.h>
#include <nns/gfx/gfx_PrimitiveRenderer.h>

#include "testGfxUtil_PlatformMeasurementTracker.h"

#include "gfxUtilGpuBenchmark_GpuBenchmark.h"
#include "gfxUtilGpuBenchmark_JsonStreamer.h"

struct ApplicationGpuBenchmarkResult
{
    static const int ValueCount = 2;
    union
    {
        uint64_t value[ValueCount];
        struct
        {
            uint64_t cpuTime;
            uint64_t gpuTime;
        } byName;

    };

    inline nn::TimeSpan GetCpuTimeElapsed() const
    {
        return nn::TimeSpan::FromNanoSeconds(byName.cpuTime);
    }

    inline nn::TimeSpan GetGpuTimeElapsed() const
    {
        return nn::TimeSpan::FromNanoSeconds(byName.gpuTime);
    }

};

struct ApplicationGpuBenchmarkValidation
{
    int     cpuMiss;
    int     gpuMiss;
    int     totalMiss;
    int     testCount;

    inline void Clear()
    {
        cpuMiss = 0;
        gpuMiss = 0;
        totalMiss = 0;
        testCount = 0;
    }

    inline double GetCpuMissPercentage() const
    {
        NN_ASSERT(testCount > 0);
        return static_cast<double>(cpuMiss * 100) / static_cast<double>(testCount);
    }

    inline double GetGpuMissPercentage() const
    {
        NN_ASSERT(testCount > 0);
        return static_cast<double>(gpuMiss * 100) / static_cast<double>(testCount);
    }

    inline double GetTotalMissPercentage() const
    {
        NN_ASSERT(testCount > 0);
        return static_cast<double>(totalMiss * 100) / static_cast<double>(testCount);
    }

};


class ApplicationTestData
{
public:
    void*                                   pHostContext;
    int                                     frameCounter;

    nnt::gfx::util::ResourceAllocator*      pResourceAllocator;
    nn::gfx::Device*                        pDevice;


    nn::gfx::Queue::InfoType                benchmarkQueueInfo;
    nn::gfx::Queue                          benchmarkQueue;
    nn::gfx::Semaphore                      benchmarkQueueCompletionSemaphore;

    nnt::gfx::util::RuntimeGfxObjects       runtimeGfxObjects;

    nns::gfx::PrimitiveRenderer::Renderer*  m_pPrimitiveRenderer;


    nn::gfx::util::DebugFontTextWriter      debugFontWriter;
    void*                                   pDebugFontWriterHeap;
    size_t                                  debugFontWriterHeapSize;
    int                                     debugFontWriterTextureSlotIndex;
    int                                     debugFontWriterTextureDescriptor;
    int                                     debugFontWriterSamplerSlotIndex;
    int                                     debugFontWriterSamplerDescriptor;

    static const int                        BenchmarkResultHistoryLength = 256;
    ApplicationGpuBenchmarkResult           benchmarkResultHistory[BenchmarkResultHistoryLength];
    int                                     benchmarkResultHistoryCount;
    int                                     benchmarkResultHistoryIndex;

    PlatformMeasurementTracker              platformMeasurementTracker;

    nn::TimeSpan                            previousRunCpuTimeElapsed;
    nn::TimeSpan                            previousRunGpuTimeElapsed;

#if defined(NN_SDK_BUILD_DEBUG)
    int                                     createdBenchmarkCount;
#endif

public:
    ApplicationTestData()
    : pHostContext()
    , frameCounter(0)
    , pResourceAllocator()
    , pDevice()
    , runtimeGfxObjects()
    , debugFontWriter()
    , pDebugFontWriterHeap()
    , debugFontWriterHeapSize()
    , debugFontWriterTextureSlotIndex()
    , debugFontWriterTextureDescriptor()
    , debugFontWriterSamplerSlotIndex()
    , debugFontWriterSamplerDescriptor()
    , benchmarkResultHistory()
    , benchmarkResultHistoryCount()
    , benchmarkResultHistoryIndex()
    , platformMeasurementTracker()
    , previousRunCpuTimeElapsed()
    , previousRunGpuTimeElapsed()
#if defined(NN_SDK_BUILD_DEBUG)
    , createdBenchmarkCount(0)
#endif
    {
    }
};

enum ApplicationMode
{
    ApplicationMode_SelectMode,
    ApplicationMode_Interactive,
    ApplicationMode_Replay,
    ApplicationMode_FindMaxPower,
    ApplicationMode_UpdateTimings,
#if defined(NNT_GFX_UTIL_ENABLE_LOP)
    ApplicationMode_Profile,
#endif
    ApplicationMode_Exit,
    ApplicationMode_Max,
};


bool IsHostIoRootMounted();
bool TestFilePathMountPointExist(const char* filePath);

void BeginFrame(ApplicationTestData* pTestData);
void EndFrame(ApplicationTestData* pTestData, nnt::gfx::util::GpuBenchmark* pGpuBenchmark);
nn::gfx::CommandBuffer* GetRootCommandBuffer(ApplicationTestData* pTestData);

void DrawFrame(ApplicationTestData* pTestData, nnt::gfx::util::GpuBenchmark* pGpuBenchmark);

void CreateTestCaseData(
    ApplicationTestData* pTestData, nnt::gfx::util::GpuBenchmark* pGpuBenchmark,
    int warmUpCount, int repeatCount,
    nnt::gfx::util::json::Document* pJsonDocument, const char* pTestId);

bool ValidateTestCaseData(
    ApplicationTestData* pTestData,
    nnt::gfx::util::json::TestCaseIterator* pTestCaseIterator);

nnt::gfx::util::GpuBenchmark* InitializeBenchmarkFromTestCase(
    ApplicationTestData* pTestData,
    nnt::gfx::util::json::TestCaseIterator* pTestCaseIterator);
nnt::gfx::util::GpuBenchmark* InitializeBenchmarkFromConfiguration(
    ApplicationTestData* pTestData, const char* testName,
    const int* pPropertyValueArray, int propertyValueArrayCount);
void FinalizeBenchmark(
    ApplicationTestData* pTestData,
    nnt::gfx::util::GpuBenchmark* pBenchmark);

void RecordGpuBenchmarkCommandList(
    ApplicationTestData* pTestData, nnt::gfx::util::GpuBenchmark* pBenchmark,
    int warmUpCount, int runCount);

void RunGpuBenchmarkCommandList(ApplicationTestData* pTestData);

bool RunGpuBenchmarkForDuration(
    nn::TimeSpan* pOutTotalTestDuration, int* pOutTotalRunCount,
    ApplicationTestData* pTestData, nnt::gfx::util::GpuBenchmark* pBenchmark,
    int measureRunCount, int warmUpCount, nn::TimeSpan targetDuration);

void RunGpuBenchmarkForDurationWithRetry(
    nn::TimeSpan* pOutTotalTestDuration, int* pOutTotalRunCount,
    ApplicationTestData* pTestData, nnt::gfx::util::GpuBenchmark* pBenchmark,
    int warmUpCount, nn::TimeSpan targetDuration, int retryCount);

void LogBenchmarkInformation(const nnt::gfx::util::GpuBenchmark* pBenchmark);

void PrintWithSelectionMarker(nn::gfx::util::DebugFontTextWriter* pDebugFontWriter, bool isSelected, const char* message);
static const int g_DisableDrawSelection = -16;
void PrintBenchmarkInformation(
    nn::gfx::util::DebugFontTextWriter* pDebugFontWriter,
    const nnt::gfx::util::GpuBenchmark* pBenchmark, int selectedProperyIndex);


void ResetHistory(ApplicationTestData* pTestData);
void AddResultToHistory(ApplicationTestData* pTestData, nn::TimeSpan cpuTimeElapsed, nn::TimeSpan gpuTimeElapsed);
const ApplicationGpuBenchmarkResult* GetPreviousResult(ApplicationTestData* pTestData);

void ComputeAverageFromHistory(
    ApplicationGpuBenchmarkResult* pOutResult,
    const ApplicationTestData* pTestData);

void ComputeMinMaxFromHistory(
    ApplicationGpuBenchmarkResult* pOutResultMin,
    ApplicationGpuBenchmarkResult* pOutResultMax,
    const ApplicationTestData* pTestData);

void FillDistributionBinFromHistory(
    const ApplicationTestData* pTestData, ApplicationGpuBenchmarkResult* pDistributionBin, int distributionBinCount,
    ApplicationGpuBenchmarkResult* pDistributionCount,
    ApplicationGpuBenchmarkResult* pMin, ApplicationGpuBenchmarkResult* pMax);

void ComputeStandardDeviationFromHistory(
    ApplicationGpuBenchmarkResult* pOutMean,
    ApplicationGpuBenchmarkResult* pOutStandardDeviation,
    const ApplicationTestData* pTestData);

void UpdateResultValidationStandardDeviation(
    ApplicationGpuBenchmarkValidation* pApplicationGpuBenchmarkValidation,
    const ApplicationGpuBenchmarkResult* pTestResult,
    const ApplicationGpuBenchmarkResult* pMean,
    const ApplicationGpuBenchmarkResult* pStandardDeviation,
    int factor);


void DoInteractiveMode(ApplicationTestData* pTestData, nnt::gfx::util::BenchmarkType initialBenchmarkType, const char* outputFilePath);


enum FindMaxPowerUpdateMode
{
    FindMaxPowerUpdateMode_CreateNew,
    FindMaxPowerUpdateMode_UpdateMaxPower,
    FindMaxPowerUpdateMode_Max,
};

void DoFindMaxPowerMode(
    ApplicationTestData* pTestData,
    FindMaxPowerUpdateMode updateMode,
    const char* outputFilePath, int benchmarkMask);

void DoReplayMode(
    ApplicationTestData* pTestData,
    const char* inputFilePath);

void DoUpdateTimingsMode(
    ApplicationTestData* pTestData,
    const char* inputFilePath,
    const char* outputFilePath,
    const char* testCaseFilter);

ApplicationMode DoSelectMode(ApplicationTestData* pTestData);



#if defined(NNT_GFX_UTIL_ENABLE_LOP)
void DoProfileMode(ApplicationTestData* pTestData, nnt::gfx::util::BenchmarkType benchmarkType);
#endif


