﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/gfx/gfx_Types.h>
#include <nn/gfx/gfx_DescriptorSlot.h>
#include <nn/gfx/gfx_MemoryPool.h>
#include <nn/gfx/gfx_DescriptorPool.h>
#include <nn/gfx/util/gfx_DescriptorPoolAllocator.h>
#include <nn/gfx/util/gfx_MemoryPoolAllocator.h>

namespace nnt { namespace gfx { namespace util {
    /*


    const int g_ShaderMemoryPoolFlags =
    nn::gfx::MemoryPoolProperty_CpuCached
    | nn::gfx::MemoryPoolProperty_GpuCached
    | nn::gfx::MemoryPoolProperty_ShaderCode;


    int memoryPoolRenderTargetFlags =
    nn::gfx::MemoryPoolProperty_CpuInvisible
    | nn::gfx::MemoryPoolProperty_GpuCached
    | nn::gfx::MemoryPoolProperty_Compressible;

    int memoryPoolDataFlags = nn::gfx::MemoryPoolProperty_GpuCached | nn::gfx::MemoryPoolProperty_CpuCached;
    */

enum MemoryPoolType
{
    MemoryPoolType_RenderTarget,
    MemoryPoolType_Data,
    MemoryPoolType_CommandBuffer,
    MemoryPoolType_ConstantBuffer,
    MemoryPoolType_Shader,
    MemoryPoolType_QueryBuffer,
    MemoryPoolType_End,
};

typedef void* (*AllocateMemoryCallback)(size_t size, size_t alignment, void* pUserData);
typedef void (*FreeMemoryCallback)(void* pMemory, void* pUserData);

class ResourceAllocatorInfo
{
private:
    AllocateMemoryCallback  m_pAllocateMemoryCallback;
    FreeMemoryCallback      m_pFreeMemoryCallback;
    void*                   m_pMemoryCallbackUserData;

    size_t                  m_MemoryPoolSize[MemoryPoolType_End];
    int                     m_DescriptorPoolCount[nn::gfx::DescriptorPoolType_End];

public:
    void                    SetDefault();

    void                    SetMemoryAllocator(
                                AllocateMemoryCallback pAllocateMemoryCallback,
                                FreeMemoryCallback pFreeMemoryCallback,
                                void* pMemoryCallbackContext);
    AllocateMemoryCallback  GetAllocateMemoryCallback() const;
    FreeMemoryCallback      GetFreeMemoryCallback() const;
    void *                  GetMemoryCallbackUserData() const;

    void                    SetMemoryPoolSize(MemoryPoolType type, size_t sizeInBytes);
    size_t                  GetMemoryPoolSize(MemoryPoolType type) const;

    void                    SetDescriptorPoolCount(nn::gfx::DescriptorPoolType type, int count);
    int                     GetDescriptorPoolCount(nn::gfx::DescriptorPoolType type) const;
};

class ResourceAllocator
{
public:
    typedef ResourceAllocatorInfo InfoType;

private:
    struct MemoryPoolData
    {
        nn::gfx::MemoryPool memoryPool;
        nn::gfx::util::MemoryPoolAllocator allocator;
        void* pMemory;
        size_t memorySize;
    };
    struct DecriptorPoolData
    {
        static const MemoryPoolType MemoryPoolType = MemoryPoolType_ConstantBuffer;

        nn::gfx::DescriptorPool decriptorPool;
        nn::gfx::util::DescriptorPoolAllocator allocator;
        int count;
        ptrdiff_t memoryPoolOffset;
    };

    AllocateMemoryCallback  m_pAllocateMemoryCallback;
    FreeMemoryCallback      m_pFreeMemoryCallback;
    void*                   m_pMemoryCallbackUserData;

    MemoryPoolData          m_MemoryPoolArray[MemoryPoolType_End];
    DecriptorPoolData       m_DescriptorPoolArray[nn::gfx::DescriptorPoolType_End];

#if defined(NN_SDK_BUILD_DEBUG)
    struct MemoryPoolAllocatorStatus
    {
        size_t totalFreeSize;
    };

    struct MemoryPoolAllocatorStatusDump
    {
        int memoryAllocationCount;
        MemoryPoolAllocatorStatus poolState[MemoryPoolType_End];
    };

    struct MemoryPoolAllocatorTrackerData
    {
        static const int DumpMaxCount = 16;

        int statusDumpIndex;
        size_t memoryPoolMaxUsage[MemoryPoolType_End];
        MemoryPoolAllocatorStatusDump statusDump[DumpMaxCount];
    };

    int m_MemoryAllocationCount;
    MemoryPoolAllocatorTrackerData m_MemoryPoolAllocatorTrackerData;
#endif

public:
                                ResourceAllocator();
                                ~ResourceAllocator();

    void                        Initialize(nn::gfx::Device* pDevice, const InfoType& info);
    void                        Finalize(nn::gfx::Device* pDevice);

    void*                       AllocateMemory(size_t size, size_t alignment);
    void                        FreeMemory(void* pMemory);

    template<typename T>
    T*                          NewObject();
    template<typename T>
    void                        DeleteObject(T* pObject);

    nn::gfx::MemoryPool*        GetMemoryPool(MemoryPoolType memoryPoolType);
    ptrdiff_t                   AllocatePoolMemory(MemoryPoolType memoryPoolType, size_t size, size_t alignment);
    void                        FreePoolMemory(MemoryPoolType memoryPoolType, ptrdiff_t offset);
    size_t                      GetMemoryPoolSize(MemoryPoolType memoryPoolType) const;
    size_t                      GetMemoryPoolAllocatorTotalFreeSize(MemoryPoolType memoryPoolType);

    int                         AllocateDescriptorSlots(nn::gfx::DescriptorPoolType descriptorPoolType, int count);
    void                        FreeDescriptorSlots(nn::gfx::DescriptorPoolType descriptorPoolType, int firstSlotIndex);

    nn::gfx::DescriptorPool*    GetDescriptorPool(nn::gfx::DescriptorPoolType descriptorPoolType);

    int                         AllocateAndSetBufferViewToDescriptorPool(
                                    nn::gfx::Buffer* pBuffer, size_t bufferSize, nn::gfx::DescriptorSlot* pDescriptorSlot);
    int                         AllocateAndSetTextureViewToDescriptorPool(
                                    nn::gfx::TextureView* pTextureView, nn::gfx::DescriptorSlot* pDescriptorSlot);
    int                         AllocateAndSetSamplerToDescriptorPool(
                                    nn::gfx::Sampler* pSampler, nn::gfx::DescriptorSlot* pDescriptorSlot);

    void                        SetBufferViewToDescriptorPool(
                                    int slotIndex, int count,
                                    nn::gfx::Buffer* pBufferArray, size_t* pBufferSizeArray,
                                    nn::gfx::DescriptorSlot* pDescriptorSlotArray);
    void                        SetTextureViewToDescriptorPool(
                                    int slotIndex, int count,
                                    nn::gfx::TextureView* pTextureViewArray,
                                    nn::gfx::DescriptorSlot* pDescriptorSlotArray);
    void                        SetSamplerToDescriptorPool(
                                    int slotIndex, int count,
                                    nn::gfx::Sampler* pSamplerArray,
                                    nn::gfx::DescriptorSlot* pDescriptorSlotArray);


private:
    void                    InitializeMemoryPools(const InfoType& info, nn::gfx::Device* pDevice);
    void                    FinalizeMemoryPools(nn::gfx::Device* pDevice);

    void                    InitializeDescriptorPools(const InfoType& info, nn::gfx::Device* pDevice);
    void                    FinalizeDescriptorPools(nn::gfx::Device* pDevice);

    static void*            MemoryPoolAllocatorMalloc(size_t size, void* pUserData);
    static void             MemoryPoolAllocatorFree(void* ptr, void* pUserData);
    static void*            DescriptorPoolAllocatorMalloc(size_t size, void* pUserData);
    static void             DescriptorPoolAllocatorFree(void* ptr, void* pUserData);

#if defined(NN_SDK_BUILD_DEBUG)
private:
    void                    InitializeMemoryPoolAllocatorTrackerData();

public:
    void                    PushMemoryPoolAllocatorStatus();
    void                    PopAndCompareMemoryPoolAllocatorStatus();
    void                    UpdateMemoryPoolAllocatorMaxUsage();
    void                    PrintMemoryPoolAllocatorMaxUsage();
#endif
};

template<typename T>
T* ResourceAllocator::NewObject()
{
    void* pMemory = AllocateMemory(sizeof(T), alignof(T));
    return new (pMemory) T();
}

template<typename T>
void ResourceAllocator::DeleteObject(T* pObject)
{
    pObject->~T();
    FreeMemory(pObject);
}



} } } // namespace nnt { namespace gfx { namespace util {

