﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Assert.h>
#include <nn/fs/fs_File.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/gfx/gfx_MemoryPool.h>
#include <nn/gfx/gfx_Texture.h>

#include "gfxUtilGpuBenchmark_ResInputTexture.h"
#include "gfxUtilGpuBenchmark_ResourceAllocator.h"

namespace nnt { namespace gfx { namespace util {

void LoadResInputTextureToBuffer(
    ResInputTexture* pResInputTexture, const char* textureFilePath,
    ResourceAllocator* pResourceAllocator)
{
    nn::Result result;
    nn::fs::FileHandle hFile;
    int64_t fileSize = 0;
    result = nn::fs::OpenFile(&hFile, textureFilePath, nn::fs::OpenMode_Read);
    NN_ASSERT(result.IsSuccess());
    result = nn::fs::GetFileSize(&fileSize, hFile);
    NN_ASSERT(result.IsSuccess());
    nn::util::BinaryFileHeader fileHeader;
    size_t readSize;
    result = nn::fs::ReadFile(&readSize, hFile, 0, &fileHeader, sizeof(nn::util::BinaryFileHeader));
    NN_ASSERT(result.IsSuccess());
    NN_ASSERT(readSize == sizeof(nn::util::BinaryFileHeader));

    size_t alignment = fileHeader.GetAlignment();
    pResInputTexture->pBuffer           = pResourceAllocator->AllocateMemory(static_cast<size_t>(fileSize), alignment);
    pResInputTexture->bufferSize        = static_cast<size_t>(fileSize);
    pResInputTexture->bufferAlignment   = alignment;

    result = nn::fs::ReadFile(&readSize, hFile, 0, pResInputTexture->pBuffer, static_cast< size_t >(fileSize));
    NN_ASSERT(result.IsSuccess());
    NN_ASSERT(readSize == static_cast< size_t >(fileSize));
    nn::fs::CloseFile(hFile);
}

void CopyResInputTextureBufferContents(
    ResInputTexture* pResInputTextureDst, const ResInputTexture* pResInputTextureSrc,
    ResourceAllocator* pResourceAllocator)
{
    NN_ASSERT(pResInputTextureSrc->pBuffer != nullptr);
    NN_ASSERT(pResInputTextureSrc->bufferSize != 0);
    NN_ASSERT(pResInputTextureSrc->bufferAlignment != 0);

    NN_ASSERT(pResInputTextureDst->pBuffer == nullptr);
    NN_ASSERT(pResInputTextureDst->bufferSize == 0);
    NN_ASSERT(pResInputTextureDst->bufferAlignment == 0);

    pResInputTextureDst->bufferSize = pResInputTextureSrc->bufferSize;
    pResInputTextureDst->bufferAlignment = pResInputTextureSrc->bufferAlignment;
    pResInputTextureDst->pBuffer = pResourceAllocator->AllocateMemory(pResInputTextureDst->bufferSize, pResInputTextureDst->bufferAlignment);

    memcpy(pResInputTextureDst->pBuffer, pResInputTextureSrc->pBuffer, pResInputTextureDst->bufferSize);
}

void InitializeResInputTextureGfxObjects(
    ResInputTexture* pResInputTexture,
    ResourceAllocator* pResourceAllocator, nn::gfx::Device* pDevice)
{
    pResInputTexture->pResTextureFile = nn::gfx::ResTextureFile::ResCast(pResInputTexture->pBuffer);
    pResInputTexture->pResTextureFile->Initialize(pDevice);

    int textureCount = pResInputTexture->pResTextureFile->GetTextureDic()->GetCount();
    NN_ASSERT(textureCount >= 1);


    int baseSlotIndex = pResourceAllocator->AllocateDescriptorSlots(nn::gfx::DescriptorPoolType_TextureView, textureCount);
    pResInputTexture->textureBaseSlotIndex = baseSlotIndex;

    void* ptextureDescriptorSlotArrayMemory = pResourceAllocator->AllocateMemory(sizeof(nn::gfx::DescriptorSlot) * textureCount, NN_ALIGNOF(nn::gfx::DescriptorSlot));
    pResInputTexture->textureDescriptorSlotArray = reinterpret_cast<nn::gfx::DescriptorSlot*>(ptextureDescriptorSlotArrayMemory);

    for (int textureIndex = 0; textureIndex < textureCount; ++textureIndex)
    {
        nn::gfx::ResTexture* pResTexture = pResInputTexture->pResTextureFile->GetResTexture(textureIndex);
        pResTexture->Initialize(pDevice);

        int textureSlotIndex = baseSlotIndex + textureIndex;
        nn::gfx::TextureView* pTextureView = pResTexture->GetTextureView();
        nn::gfx::DescriptorSlot* pDescriptorSlot = &pResInputTexture->textureDescriptorSlotArray[textureIndex];

        pResourceAllocator->SetTextureViewToDescriptorPool(
            textureSlotIndex, 1,
            pTextureView, pDescriptorSlot);

        pResTexture->SetUserDescriptorSlot(*pDescriptorSlot);
    }
}

void InitializeResInputTexture(
    ResInputTexture* pResInputTexture, const char* textureFilePath,
    ResourceAllocator* pResourceAllocator, nn::gfx::Device* pDevice)
{
    LoadResInputTextureToBuffer(pResInputTexture, textureFilePath, pResourceAllocator);
    InitializeResInputTextureGfxObjects(pResInputTexture, pResourceAllocator, pDevice);
}

void FinalizeResInputTexture(
    ResInputTexture* pResInputTexture,
    ResourceAllocator* pResourceAllocator, nn::gfx::Device* pDevice)
{
    int textureCount = pResInputTexture->pResTextureFile->GetTextureDic()->GetCount();
    NN_ASSERT(textureCount >= 1);

    pResourceAllocator->FreeDescriptorSlots(nn::gfx::DescriptorPoolType_TextureView, pResInputTexture->textureBaseSlotIndex);

    for (int textureIndex = 0; textureIndex < textureCount; ++textureIndex)
    {
        nn::gfx::ResTexture* pResTexture = pResInputTexture->pResTextureFile->GetResTexture(textureIndex);
        pResTexture->Finalize(pDevice);
    }

    pResInputTexture->pResTextureFile->Finalize(pDevice);

    pResourceAllocator->FreeMemory(pResInputTexture->textureDescriptorSlotArray);
    pResInputTexture->textureDescriptorSlotArray = nullptr;

    pResourceAllocator->FreeMemory(pResInputTexture->pBuffer);
    pResInputTexture->pBuffer = nullptr;
    pResInputTexture->bufferSize = 0;
    pResInputTexture->bufferAlignment = 0;

    pResInputTexture->pResTextureFile = nullptr;
    pResInputTexture->textureBaseSlotIndex = -1;
}

} } } // namespace nnt { namespace gfx { namespace util {
