﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Assert.h>
#include <nn/nn_Abort.h>
#include <nn/gfx/gfx_Texture.h>
#include <nn/gfx/gfx_Buffer.h>
#include <nn/gfx/gfx_MemoryPool.h>
#include <nn/gfx/gfx_Shader.h>
#include <nn/gfx/gfx_State.h>

#include "gfxUtilGpuBenchmark_ResHelpers.h"


namespace nnt { namespace gfx { namespace util {

namespace
{

const DefaultVertex g_RectangleVertexBufferData[] =
{
    { -1.0f, -1.0f, 0.0f, 0.0f, 0.0f, },
    { -1.0f,  1.0f, 0.0f, 0.0f, 1.0f, },
    { 1.0f, -1.0f, 0.0f, 1.0f, 0.0f, },
    { 1.0f,  1.0f, 0.0f, 1.0f, 1.0f, },
};

uint16_t ConvertFloatToFloat16(float v)
{
    NN_ASSERT(sizeof(uint32_t) == sizeof(v));
    uint32_t floatAsInt = 0;
    memcpy(&floatAsInt, &v, sizeof(v));

    if (floatAsInt == 0)
        return 0;

    int expRaw = (floatAsInt >> 23) & 0xFF;
    NN_ASSERT((expRaw != 0) && (expRaw != 0xFF));
    int expWithoutBias = expRaw - 127;
    NN_ASSERT((expWithoutBias >= -14) && (expWithoutBias <= 15));
    int expScaledWithBias = expWithoutBias + 15;
    int exp = expScaledWithBias & 0x1f;

    int sign = (floatAsInt >> 31) & 0x01;
    int mant = floatAsInt & ((1 << 23) - 1);

    uint16_t result =
        (sign << 15)
        | (exp << 10)
        | (mant >> 13);

    return result;
}

uintptr_t PackOffsetAndFlags(ptrdiff_t memoryPoolOffset, MemoryPoolType memoryPoolType)
{
    NN_ASSERT((memoryPoolOffset & (0xFFull << ((sizeof(ptrdiff_t) * CHAR_BIT) - 8))) == 0);
    NN_STATIC_ASSERT(MemoryPoolType_End <= 0xFFull);

    uintptr_t value = (memoryPoolOffset << 8) | memoryPoolType;
    return value;
}

MemoryPoolType GetMemoryPoolType(uintptr_t packedValue)
{
    return static_cast<MemoryPoolType>(packedValue & 0xFFull);
}

ptrdiff_t GetMemoryPoolOffset(uintptr_t packedValue)
{
    return static_cast<ptrdiff_t>(packedValue >> 8);
}

}

const size_t g_RectangleVertexBufferDataSize = sizeof(g_RectangleVertexBufferData);

void InitializeTexture(
    nn::gfx::Texture* pTexture, const nn::gfx::Texture::InfoType& info,
    ResourceAllocator* pResourceAllocator, MemoryPoolType memoryPoolType,
    nn::gfx::Device* pDevice)
{
    uintptr_t userValue = 0;

    if (NN_STATIC_CONDITION(nn::gfx::Texture::IsMemoryPoolRequired))
    {
        size_t size = nn::gfx::Texture::CalculateMipDataSize(pDevice, info);
        size_t alignment = nn::gfx::Texture::CalculateMipDataAlignment(pDevice, info);

        nn::gfx::MemoryPool* pMemoryPool =
            pResourceAllocator->GetMemoryPool(memoryPoolType);

        ptrdiff_t offset = -1;
        offset = pResourceAllocator->AllocatePoolMemory(
            memoryPoolType, size, alignment);
        NN_ASSERT_NOT_EQUAL(offset, -1);

        pTexture->Initialize(
            pDevice, info,
            pMemoryPool,
            offset, size);

        userValue = PackOffsetAndFlags(offset, memoryPoolType);

        if (memoryPoolType != MemoryPoolType_RenderTarget)
        {
            uint8_t* pBuffer = pMemoryPool->Map<uint8_t>() + offset;
            memset(pBuffer, 0, size);
            pMemoryPool->Unmap();
            pMemoryPool->FlushMappedRange(offset, size);
        }
    }
    else
    {
        pTexture->Initialize(pDevice, info, nullptr, 0, 0);
    }

    pTexture->SetUserPtr(reinterpret_cast<void*>(userValue));
}

void InitializeTextureDisableFrameBufferCompression(
    nn::gfx::Texture* pTexture, const nn::gfx::Texture::InfoType& info,
    ResourceAllocator* pResourceAllocator, MemoryPoolType memoryPoolType,
    nn::gfx::Device* pDevice)
{
    NN_ASSERT(memoryPoolType != MemoryPoolType_RenderTarget);
    NN_ASSERT((info.GetGpuAccessFlags() & nn::gfx::GpuAccess_ColorBuffer) != 0);

    uintptr_t userValue = 0;

    if (NN_STATIC_CONDITION(nn::gfx::Texture::IsMemoryPoolRequired))
    {
        size_t size = nn::gfx::Texture::CalculateMipDataSize(pDevice, info);
        size_t alignment = nn::gfx::Texture::CalculateMipDataAlignment(pDevice, info);

        nn::gfx::MemoryPool* pMemoryPool = pResourceAllocator->GetMemoryPool(memoryPoolType);

        ptrdiff_t offset = -1;
        offset = pResourceAllocator->AllocatePoolMemory(
            memoryPoolType,
            size, alignment);
        NN_ASSERT_NOT_EQUAL(offset, -1);

        nn::gfx::Texture::InfoType initTextureInfo = info;
        initTextureInfo.SetGpuAccessFlags(info.GetGpuAccessFlags() & ~nn::gfx::GpuAccess_ColorBuffer);

        pTexture->Initialize(pDevice, initTextureInfo,
            pMemoryPool,
            offset, size);

        userValue = PackOffsetAndFlags(offset, memoryPoolType);

        uint8_t* pBuffer = pMemoryPool->Map<uint8_t>() + offset;
        memset(pBuffer, 0, size);
        pMemoryPool->Unmap();
        pMemoryPool->FlushMappedRange(offset, size);
    }
    else
    {
        pTexture->Initialize(pDevice, info, nullptr, 0, 0);
    }

    pTexture->SetUserPtr(reinterpret_cast<void*>(userValue));
}

void FinalizeTexture(
    nn::gfx::Texture* pTexture,
    ResourceAllocator* pResourceAllocator, nn::gfx::Device* pDevice)
{
    uintptr_t userValue = reinterpret_cast<uintptr_t>(pTexture->GetUserPtr());

    if (NN_STATIC_CONDITION(nn::gfx::Texture::IsMemoryPoolRequired))
    {
        ptrdiff_t poolOffset = GetMemoryPoolOffset(userValue);
        MemoryPoolType memoryPoolType = GetMemoryPoolType(userValue);

        pResourceAllocator->FreePoolMemory(memoryPoolType, poolOffset);
    }
    else
    {
        NN_ASSERT(userValue == 0);
    }

    pTexture->Finalize(pDevice);
}

void FillTextureWithColorPattern(
    nn::gfx::Texture* pTexture, const nn::gfx::Texture::InfoType& info,
    int randomSeed, ResourceAllocator* pResourceAllocator, nn::gfx::Device* pDevice)
{
    NN_ASSERT((info.GetImageFormat() == nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm)
        || (info.GetImageFormat() == nn::gfx::ImageFormat_R16_G16_B16_A16_Float))

    uintptr_t userValue = reinterpret_cast<uintptr_t>(pTexture->GetUserPtr());
    if (NN_STATIC_CONDITION(nn::gfx::Texture::IsMemoryPoolRequired))
    {
        ptrdiff_t poolOffset = GetMemoryPoolOffset(userValue);
        MemoryPoolType memoryPoolType = GetMemoryPoolType(userValue);

        size_t bufferSize = nn::gfx::Texture::CalculateMipDataSize(pDevice, info);

        uint8_t* pBuffer = pResourceAllocator->GetMemoryPool(memoryPoolType)->Map<uint8_t>() + poolOffset;
        uint8_t* pBufferEnd = pBuffer + bufferSize;

        int red = (randomSeed >> 0) & 0xFF;
        int green = (randomSeed >> 8) & 0xFF;
        int blue = (randomSeed >> 16) & 0xFF;
        int alpha = 0xFF;

        while (pBuffer < pBufferEnd)
        {
            if (info.GetImageFormat() == nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm)
            {
                int mask = 0xFF;
                uint8_t* pPixelBuffer = pBuffer;
                pPixelBuffer[0] = red & mask;
                pPixelBuffer[1] = green & mask;
                pPixelBuffer[2] = blue & mask;
                pPixelBuffer[3] = alpha & mask;
                pBuffer += 4;
            }
            else if (info.GetImageFormat() == nn::gfx::ImageFormat_R16_G16_B16_A16_Float)
            {
                uint16_t* pPixelBuffer = reinterpret_cast<uint16_t*>(pBuffer);

                int mask = 0xFF;
                float redAsFloat = static_cast<float>(red & mask) / 255.0f;
                float greenAsFloat = static_cast<float>(green & mask) / 255.0f;
                float blueAsFloat = static_cast<float>(blue & mask) / 255.0f;
                float alphaAsFloat = static_cast<float>(alpha & mask) / 255.0f;

                pPixelBuffer[0] = ConvertFloatToFloat16(redAsFloat);
                pPixelBuffer[1] = ConvertFloatToFloat16(greenAsFloat);
                pPixelBuffer[2] = ConvertFloatToFloat16(blueAsFloat);
                pPixelBuffer[3] = ConvertFloatToFloat16(alphaAsFloat);

                pBuffer += 8;
            }
            else
            {
                NN_ABORT(); // 非対応
            }

            red += 0xA;
            green += 0x7;
            blue += 0x13;
            alpha = 0xFF;
        }

        pResourceAllocator->GetMemoryPool(memoryPoolType)->Unmap();
        pResourceAllocator->GetMemoryPool(memoryPoolType)->FlushMappedRange(poolOffset, bufferSize);
    }
}

void InitializeBuffer(
    nn::gfx::Buffer* pBuffer, const nn::gfx::Buffer::InfoType& info,
    ResourceAllocator* pResourceAllocator, MemoryPoolType memoryPoolType,
    nn::gfx::Device* pDevice)
{
    uintptr_t userValue = 0;

    if (NN_STATIC_CONDITION(nn::gfx::Texture::IsMemoryPoolRequired))
    {
        size_t size = info.GetSize();
        size_t alignment = nn::gfx::Buffer::GetBufferAlignment(pDevice, info);

        nn::gfx::MemoryPool* pMemoryPool = pResourceAllocator->GetMemoryPool(memoryPoolType);

        ptrdiff_t offset = -1;
        offset = pResourceAllocator->AllocatePoolMemory(
            memoryPoolType,
            size, alignment);
        NN_ASSERT_NOT_EQUAL(offset, -1);

        pBuffer->Initialize(pDevice, info,
            pMemoryPool,
            offset, size);

        userValue = PackOffsetAndFlags(offset, memoryPoolType);
    }
    else
    {
        pBuffer->Initialize(pDevice, info, nullptr, 0, 0);
    }

    pBuffer->SetUserPtr(reinterpret_cast<void*>(userValue));
}

void FinalizeBuffer(
    nn::gfx::Buffer* pBuffer,
    ResourceAllocator* pResourceAllocator, nn::gfx::Device* pDevice)
{
    uintptr_t userValue = reinterpret_cast<uintptr_t>(pBuffer->GetUserPtr());

    if (NN_STATIC_CONDITION(nn::gfx::Texture::IsMemoryPoolRequired))
    {
        ptrdiff_t poolOffset = GetMemoryPoolOffset(userValue);
        MemoryPoolType memoryPoolType = GetMemoryPoolType(userValue);

        pResourceAllocator->FreePoolMemory(memoryPoolType, poolOffset);
    }
    else
    {
        NN_ASSERT(userValue == 0);
    }

    pBuffer->Finalize(pDevice);
}

void ClearBufferContent(nn::gfx::Buffer* pBuffer, size_t bufferSize)
{
    void* pMemory = pBuffer->Map();
    memset(pMemory, 0, bufferSize);
    pBuffer->Unmap();
    pBuffer->FlushMappedRange(0, bufferSize);
}

void InitializeFullScreenQuadVertexBuffer(
    nn::gfx::Buffer* pBuffer,
    ResourceAllocator* pResourceAllocator, nn::gfx::Device* pDevice)
{
    const size_t bufferSize = sizeof(g_RectangleVertexBufferData);

    nn::gfx::Buffer::InfoType rectangleVertexBufferInfo;
    {
        rectangleVertexBufferInfo.SetDefault();
        rectangleVertexBufferInfo.SetGpuAccessFlags(nn::gfx::GpuAccess_VertexBuffer);
        rectangleVertexBufferInfo.SetSize(bufferSize);
    }

    InitializeBuffer(
        pBuffer, rectangleVertexBufferInfo,
        pResourceAllocator, MemoryPoolType_Data, pDevice);
    CopyContentToBuffer(pBuffer, g_RectangleVertexBufferData, bufferSize);
}

void InitializeVertexBuffer(
    nn::gfx::Buffer* pBuffer, const void* pData, size_t dataSize,
    ResourceAllocator* pResourceAllocator, nn::gfx::Device* pDevice)
{
    nn::gfx::Buffer::InfoType rectangleVertexBufferInfo;
    {
        rectangleVertexBufferInfo.SetDefault();
        rectangleVertexBufferInfo.SetGpuAccessFlags(nn::gfx::GpuAccess_VertexBuffer);
        rectangleVertexBufferInfo.SetSize(dataSize);
    }

    InitializeBuffer(
        pBuffer, rectangleVertexBufferInfo,
        pResourceAllocator, MemoryPoolType_Data, pDevice);
    CopyContentToBuffer(pBuffer, pData, dataSize);
}

void CopyContentToBuffer(nn::gfx::Buffer* pBuffer, const void* pDataToCopy, size_t dataToCopySizeInBytes)
{
    void* pBufferContent = pBuffer->Map();

    memcpy(pBufferContent, pDataToCopy, dataToCopySizeInBytes);

    pBuffer->Unmap();
    pBuffer->FlushMappedRange(0, dataToCopySizeInBytes);
}

void InitializeVertexState(
    nn::gfx::VertexState* pVertexState, const nn::gfx::VertexStateInfo& info,
    ResourceAllocator* pResourceAllocator, nn::gfx::Device* pDevice)
{
    size_t size = nn::gfx::VertexState::GetRequiredMemorySize(info);

    if (size > 0)
    {
        void* pMemory = pResourceAllocator->AllocateMemory(size, nn::gfx::VertexState::RequiredMemoryInfo_Alignment);
        pVertexState->SetMemory(pMemory, size);
    }

    pVertexState->Initialize(pDevice, info, nullptr);
}

void FinalizeVertexState(
    nn::gfx::VertexState* pVertexState,
    ResourceAllocator* pResourceAllocator, nn::gfx::Device* pDevice)
{
    void* pMemory = pVertexState->GetMemory();
    pVertexState->Finalize(pDevice);

    if (pMemory != nullptr)
    {
        pResourceAllocator->FreeMemory(pMemory);
    }
}


size_t InitializeColorRenderTarget(
    nn::gfx::Texture* pTexture, nn::gfx::Buffer* pOutputCopyBuffer,
    nn::gfx::ColorTargetView* pColorTargetView, nn::gfx::ViewportScissorState* pViewportScissorState,
    int renderWidth, int renderHeight, nn::gfx::ImageFormat renderFormat, nn::gfx::TileMode tileMode,
    ResourceAllocator* pResourceAllocator, nn::gfx::Device* pDevice)
{
    nn::gfx::Texture::InfoType renderTextureInfo;
    {
        renderTextureInfo.SetDefault();
        renderTextureInfo.SetGpuAccessFlags(nn::gfx::GpuAccess_Texture | nn::gfx::GpuAccess_ColorBuffer);
        renderTextureInfo.SetWidth(renderWidth);
        renderTextureInfo.SetHeight(renderHeight);
        renderTextureInfo.SetDepth(1);
        renderTextureInfo.SetImageFormat(renderFormat);
        renderTextureInfo.SetMipCount(1);
        renderTextureInfo.SetArrayLength(1);
        renderTextureInfo.SetImageStorageDimension(nn::gfx::ImageStorageDimension_2d);
        renderTextureInfo.SetTileMode(tileMode);
    }
    InitializeTexture(
        pTexture, renderTextureInfo,
        pResourceAllocator, MemoryPoolType_RenderTarget,
        pDevice);

    size_t textureStorageSize = nn::gfx::Texture::CalculateMipDataSize(pDevice, renderTextureInfo);

    if (pOutputCopyBuffer != nullptr)
    {
        nn::gfx::Buffer::InfoType outputCopyBufferInfo;
        outputCopyBufferInfo.SetDefault();
        outputCopyBufferInfo.SetGpuAccessFlags(
            nn::gfx::GpuAccess_Image | nn::gfx::GpuAccess_Write);
        outputCopyBufferInfo.SetSize(textureStorageSize);
        InitializeBuffer(
            pOutputCopyBuffer, outputCopyBufferInfo,
            pResourceAllocator, MemoryPoolType_Data, pDevice);
        ClearBufferContent(pOutputCopyBuffer, textureStorageSize);
    }

    nn::gfx::ColorTargetView::InfoType renderTextureColorTargetViewInfo;
    {
        renderTextureColorTargetViewInfo.SetDefault();
        renderTextureColorTargetViewInfo.SetTexturePtr(pTexture);
        renderTextureColorTargetViewInfo.SetImageFormat(renderFormat);
        renderTextureColorTargetViewInfo.SetImageDimension(nn::gfx::ImageDimension_2d);
        renderTextureColorTargetViewInfo.SetMipLevel(0);
        renderTextureColorTargetViewInfo.EditArrayRange().SetBaseArrayIndex(0);
        renderTextureColorTargetViewInfo.EditArrayRange().SetArrayLength(1);
    }

    pColorTargetView->Initialize(pDevice, renderTextureColorTargetViewInfo);

    if (pViewportScissorState != nullptr)
    {
        nn::gfx::ViewportScissorState::InfoType info;
        info.SetDefault();
        info.SetScissorEnabled(false);

        nn::gfx::ViewportStateInfo viewportInfo;
        viewportInfo.SetDefault();
        viewportInfo.SetWidth(static_cast<float>(renderWidth));
        viewportInfo.SetHeight(static_cast<float>(renderHeight));

        nn::gfx::ScissorStateInfo scissorInfo;
        scissorInfo.SetDefault();
        scissorInfo.SetWidth(renderWidth);
        scissorInfo.SetHeight(renderHeight);

        info.SetViewportStateInfoArray(&viewportInfo, 1);
        info.SetScissorStateInfoArray(&scissorInfo, 1);

        pViewportScissorState->Initialize(pDevice, info);
    }

    return textureStorageSize;
}

size_t InitializeColorRenderTargetDisableFrameBufferCompression(
    nn::gfx::Texture* pTexture, nn::gfx::Buffer* pOutputCopyBuffer,
    nn::gfx::ColorTargetView* pColorTargetView, nn::gfx::ViewportScissorState* pViewportScissorState,
    int renderWidth, int renderHeight, nn::gfx::ImageFormat renderFormat, nn::gfx::TileMode tileMode,
    ResourceAllocator* pResourceAllocator, nn::gfx::Device* pDevice)
{
    nn::gfx::Texture::InfoType renderTextureInfo;
    {
        renderTextureInfo.SetDefault();
        renderTextureInfo.SetGpuAccessFlags(nn::gfx::GpuAccess_Texture | nn::gfx::GpuAccess_ColorBuffer);
        renderTextureInfo.SetWidth(renderWidth);
        renderTextureInfo.SetHeight(renderHeight);
        renderTextureInfo.SetDepth(1);
        renderTextureInfo.SetImageFormat(renderFormat);
        renderTextureInfo.SetMipCount(1);
        renderTextureInfo.SetArrayLength(1);
        renderTextureInfo.SetImageStorageDimension(nn::gfx::ImageStorageDimension_2d);
        renderTextureInfo.SetTileMode(tileMode);
    }
    InitializeTextureDisableFrameBufferCompression(
        pTexture, renderTextureInfo,
        pResourceAllocator, MemoryPoolType_Data, pDevice);

    size_t textureStorageSize = nn::gfx::Texture::CalculateMipDataSize(pDevice, renderTextureInfo);

    if (pOutputCopyBuffer != nullptr)
    {
        nn::gfx::Buffer::InfoType outputCopyBufferInfo;
        outputCopyBufferInfo.SetDefault();
        outputCopyBufferInfo.SetGpuAccessFlags(
            nn::gfx::GpuAccess_Image | nn::gfx::GpuAccess_Write);
        outputCopyBufferInfo.SetSize(textureStorageSize);
        InitializeBuffer(
            pOutputCopyBuffer, outputCopyBufferInfo,
            pResourceAllocator, MemoryPoolType_Data, pDevice);
        ClearBufferContent(pOutputCopyBuffer, textureStorageSize);
    }

    nn::gfx::ColorTargetView::InfoType renderTextureColorTargetViewInfo;
    {
        renderTextureColorTargetViewInfo.SetDefault();
        renderTextureColorTargetViewInfo.SetTexturePtr(pTexture);
        renderTextureColorTargetViewInfo.SetImageFormat(renderFormat);
        renderTextureColorTargetViewInfo.SetImageDimension(nn::gfx::ImageDimension_2d);
        renderTextureColorTargetViewInfo.SetMipLevel(0);
        renderTextureColorTargetViewInfo.EditArrayRange().SetBaseArrayIndex(0);
        renderTextureColorTargetViewInfo.EditArrayRange().SetArrayLength(1);
    }
    pColorTargetView->Initialize(pDevice, renderTextureColorTargetViewInfo);

    if (pViewportScissorState != nullptr)
    {
        nn::gfx::ViewportScissorState::InfoType info;
        info.SetDefault();
        info.SetScissorEnabled(false);

        nn::gfx::ViewportStateInfo viewportInfo;
        viewportInfo.SetDefault();
        viewportInfo.SetWidth(static_cast<float>(renderWidth));
        viewportInfo.SetHeight(static_cast<float>(renderHeight));

        nn::gfx::ScissorStateInfo scissorInfo;
        scissorInfo.SetDefault();
        scissorInfo.SetWidth(renderWidth);
        scissorInfo.SetHeight(renderHeight);

        info.SetViewportStateInfoArray(&viewportInfo, 1);
        info.SetScissorStateInfoArray(&scissorInfo, 1);

        pViewportScissorState->Initialize(pDevice, info);
    }

    return textureStorageSize;
}


void FinalizeColorRenderTarget(
    nn::gfx::Texture* pTexture, nn::gfx::Buffer* pOutputCopyBuffer,
    nn::gfx::ColorTargetView* pColorTargetView, nn::gfx::ViewportScissorState* pViewportScissorState,
    ResourceAllocator* pResourceAllocator, nn::gfx::Device* pDevice)
{
    if (pViewportScissorState != nullptr)
    {
        pViewportScissorState->Finalize(pDevice);
    }

    pColorTargetView->Finalize(pDevice);

    if (pOutputCopyBuffer != nullptr)
    {
        FinalizeBuffer(pOutputCopyBuffer, pResourceAllocator, pDevice);
    }

    FinalizeTexture(pTexture, pResourceAllocator, pDevice);
}


void InitializeBlendState(
    nn::gfx::BlendState* pBlendState, nn::gfx::BlendStateInfo& info,
    ResourceAllocator* pResourceAllocator, nn::gfx::Device* pDevice)
{
    size_t size = nn::gfx::BlendState::GetRequiredMemorySize(info);
    if (size > 0) {
        void* pMemory = pResourceAllocator->AllocateMemory(size, nn::gfx::BlendState::RequiredMemoryInfo_Alignment);
        pBlendState->SetMemory(pMemory, size);
    }

    pBlendState->Initialize(pDevice, info);
}

void FinalizeBlendState(
    nn::gfx::BlendState* pBlendState,
    ResourceAllocator* pResourceAllocator, nn::gfx::Device* pDevice)
{
    void* pMemory = pBlendState->GetMemory();
    pBlendState->Finalize(pDevice);

    if (pMemory != nullptr) {
        pResourceAllocator->FreeMemory(pMemory);
    }
}

} } } // namespace nnt { namespace gfx { namespace util {
