﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once


#include "gfxUtilGpuBenchmark_GpuBenchmark.h"

#include "gfxUtilGpuBenchmark_Property.h"
#include "gfxUtilGpuBenchmark_PropertyArray.h"
#include "gfxUtilGpuBenchmark_ResShader.h"

#include <nn/gfx/gfx_Texture.h>
#include <nn/gfx/gfx_State.h>
#include <nn/gfx/gfx_DescriptorSlot.h>

//#define NN_GFXUTIL_GPUBENCHMARK_COMPOSITE_TEST_DEBUG


namespace nnt { namespace gfx { namespace util {

class GpuBenchmarkCompositeTest : public GpuBenchmark
{
public:
    static const char*  ClassName;

    enum RenderTargetMode
    {
        RenderTargetMode_8I,
        RenderTargetMode_16F,
    };

    enum DepthMode
    {
        DepthMode_Off,
        DepthMode_On,
    };

    enum ShaderLoadMode
    {
        ShaderLoadMode_None,
        ShaderLoadMode_1xMatrixMul,
        ShaderLoadMode_2xMatrixMul,
        ShaderLoadMode_4xMatrixMul,
        ShaderLoadMode_8xMatrixMul,
    };

    enum TextureMode
    {
        TextureMode_None,
        TextureMode_1xBC1,
        TextureMode_2xBC1,
        TextureMode_4xBC1,
    };

    enum BlendingMode
    {
        BlendingMode_Off,
        BlendingMode_AlphaAdd,
        BlendingMode_AlphaSub,
    };

    enum VertexAttributeMode
    {
        VertexAttributeMode_None,
        VertexAttributeMode_1,
        VertexAttributeMode_2,
        VertexAttributeMode_4,
        VertexAttributeMode_8,
    };


private:
    static const nn::gfx::TileMode      m_TileMode                      = nn::gfx::TileMode_Optimal;
    static const int                    m_MaxTextureCount               = 4;
    static const int                    m_MaxAttributeCount             = 8;
    static const int                    m_ConstantBufferMatrixCount     = 64;
    const size_t                        m_TransformConstantBufferSize   = sizeof(float) * 16 * m_ConstantBufferMatrixCount;

    int                                 m_RenderSize;
    RenderTargetMode                    m_RenderTargetMode;
    DepthMode                           m_DepthMode;
    ShaderLoadMode                      m_ShaderLoadMode;
    TextureMode                         m_TextureMode;
    BlendingMode                        m_BlendingMode;
    VertexAttributeMode                 m_VertexAttributeMode;
    int                                 m_VertexGridSize;

    nn::gfx::Texture                    m_RenderTexture;
    nn::gfx::ColorTargetView            m_RenderTextureColorTargetView;
    nn::gfx::Texture                    m_DepthStencilTexture;
    nn::gfx::DepthStencilView           m_DepthStencilView;
    nn::gfx::ViewportScissorState       m_ViewportScissorState;

    nn::gfx::DepthStencilState          m_DepthStencilState;

    ResShader                           m_ResShader;


    struct TextureData
    {
        nn::gfx::Texture            texture;
        nn::gfx::TextureView        textureView;
        int                         slotIndex;
        nn::gfx::DescriptorSlot     descriptorSlot;
    };

    TextureData                         m_TextureDataArray[m_MaxTextureCount];
    nn::gfx::Sampler                    m_TextureSampler;
    int                                 m_TextureSamplerSlotIndex;
    nn::gfx::DescriptorSlot             m_TextureSamplerDescriptorSlot;


    nn::gfx::Buffer                     m_VertexBuffer;
    int                                 m_VertexStride;

    nn::gfx::Buffer                     m_ConstantBuffer;
    int                                 m_ConstantBufferDescriptorSlotIndex;
    nn::gfx::DescriptorSlot             m_ConstantBufferDescriptorSlot;
    nn::gfx::VertexState                m_VertexState;


    nn::gfx::Buffer                     m_TransformConstantBuffer;
    int                                 m_TransformConstantBufferDescriptorSlotIndex;
    nn::gfx::DescriptorSlot             m_TransformConstantBufferDescriptorSlot;

    nn::gfx::BlendState                 m_BlendState;

    nn::gfx::Buffer                     m_OutputCopyBuffer;
    size_t                              m_OutputCopyBufferSize;

    enum Property
    {
        Property_RenderSize,
        Property_RenderTargetMode,
        Property_DepthMode,
        Property_ShaderLoadMode,
        Property_TextureMode,
        Property_BlendingMode,
        Property_VertexAttributeMode,
        Property_VertexGridSize,
        Property_Count,

    };

    PropertyArray<Property_Count>       m_PropertyArray;

public:
                                                GpuBenchmarkCompositeTest();
    virtual                                     ~GpuBenchmarkCompositeTest();

    virtual void                                Initialize(ResourceAllocator* pResourceAllocator) NN_OVERRIDE;
    virtual void                                Finalize(ResourceAllocator* pResourceAllocator) NN_OVERRIDE;

    virtual void                                InitializeGfxObjects(ResourceAllocator* pResourceAllocator, nn::gfx::Device* pDevice) NN_OVERRIDE;
    virtual void                                FinalizeGfxObjects(ResourceAllocator* pResourceAllocator, nn::gfx::Device* pDevice) NN_OVERRIDE;

    int                                         GetRenderSize() const;
    void                                        SetRenderSize(int value);

    RenderTargetMode                            GetRenderTargetMode() const;
    void                                        SetRenderTargetMode(RenderTargetMode value);

    DepthMode                                   GetDepthMode() const;
    void                                        SetDepthMode(DepthMode value);

    ShaderLoadMode                              GetShaderLoadMode() const;
    void                                        SetShaderLoadMode(ShaderLoadMode value);

    TextureMode                                 GetTextureMode() const;
    void                                        SetTextureMode(TextureMode value);

    BlendingMode                                GetBlendingMode() const;
    void                                        SetBlendingMode(BlendingMode value);

    VertexAttributeMode                         GetVertexAttributeMode() const;
    void                                        SetVertexAttributeMode(VertexAttributeMode value);

    int                                         GetVertexGridSize() const;
    void                                        SetVertexGridSize(int value);

    virtual void                                PreBenchmark(nn::gfx::CommandBuffer* pTestCommandBuffer) NN_OVERRIDE;
    virtual void                                DoBenchmark(nn::gfx::CommandBuffer* pTestCommandBuffer, int runCount) NN_OVERRIDE;

    virtual void                                PrintResults(nn::TimeSpan cpuTimeElapsed, nn::TimeSpan gpuTimeElapsed, int runCount, nn::gfx::util::DebugFontTextWriter* pDebugFontTextWriter) NN_OVERRIDE;

    virtual void                                CopyResultToBuffer(nn::gfx::CommandBuffer* pCommandBuffer) NN_OVERRIDE;
    virtual void                                MapResultBuffer(void** pOutBuffer, size_t* pOutBufferSize) NN_OVERRIDE;
    virtual void                                UnmapResultBuffer() NN_OVERRIDE;

    virtual const char*                         GetName() const NN_OVERRIDE;
    virtual BenchmarkType                       GetType() const NN_OVERRIDE;
    virtual int                                 GetPropertyCount() const NN_OVERRIDE;
    virtual int                                 FillPropertyList(const GpuBenchmarkPropertyHolder** ppDestinationArray, int destinationArrayMaxSize) const NN_OVERRIDE;
    virtual int                                 FillPropertyList(GpuBenchmarkPropertyHolder** ppDestinationArray, int destinationArrayMaxSize) NN_OVERRIDE;
    virtual GpuBenchmarkPropertyHolder*         FindPropertyByName(const char* propertyName) NN_OVERRIDE;
    virtual GpuBenchmarkPropertyHolder*         GetPropertyByIndex(int index) NN_OVERRIDE;

private:
    nn::gfx::ImageFormat                        GetRenderTargetFormat() const;
    nn::gfx::ImageFormat                        GetDepthStencilFormat() const;
    int                                         GetTextureCount() const;
    int                                         GetVertexAttributeCount() const;
    int                                         GetVertexCount() const;
    bool                                        GetBlendEnabled() const;
    nn::gfx::BlendFunction                      GetBlendFunction() const;

    void                                        InitializeGfxObjectsRenderTarget(ResourceAllocator* pResourceAllocator, nn::gfx::Device* pDevice);
    void                                        FinalizeGfxObjectsRenderTarget(ResourceAllocator* pResourceAllocator, nn::gfx::Device* pDevice);

    void                                        InitializeGfxObjectsTexture(ResourceAllocator* pResourceAllocator, nn::gfx::Device* pDevice);
    void                                        FinalizeGfxObjectsTexture(ResourceAllocator* pResourceAllocator, nn::gfx::Device* pDevice);

    void                                        InitializeGfxObjectsVertexBuffer(ResourceAllocator* pResourceAllocator, nn::gfx::Device* pDevice);
    void                                        FinalizeGfxObjectsVertexBuffer(ResourceAllocator* pResourceAllocator, nn::gfx::Device* pDevice);

    void                                        InitializeGfxObjectsTranform(ResourceAllocator* pResourceAllocator, nn::gfx::Device* pDevice);
    void                                        FinalizeGfxObjectsTranform(ResourceAllocator* pResourceAllocator, nn::gfx::Device* pDevice);

    void                                        InitializeGfxObjectsBlendState(ResourceAllocator* pResourceAllocator, nn::gfx::Device* pDevice);
    void                                        FinalizeGfxObjectsBlendState(ResourceAllocator* pResourceAllocator, nn::gfx::Device* pDevice);

};


inline int GpuBenchmarkCompositeTest::GetRenderSize() const
{
    return m_RenderSize;
}

inline void GpuBenchmarkCompositeTest::SetRenderSize(int value)
{
    m_RenderSize = value;
}

inline GpuBenchmarkCompositeTest::RenderTargetMode GpuBenchmarkCompositeTest::GetRenderTargetMode() const
{
    return m_RenderTargetMode;
}

inline void GpuBenchmarkCompositeTest::SetRenderTargetMode(RenderTargetMode value)
{
    m_RenderTargetMode = value;
}

inline GpuBenchmarkCompositeTest::DepthMode GpuBenchmarkCompositeTest::GetDepthMode() const
{
    return m_DepthMode;
}

inline void GpuBenchmarkCompositeTest::SetDepthMode(DepthMode value)
{
    m_DepthMode = value;
}


inline GpuBenchmarkCompositeTest::ShaderLoadMode GpuBenchmarkCompositeTest::GetShaderLoadMode() const
{
    return m_ShaderLoadMode;
}

inline void GpuBenchmarkCompositeTest::SetShaderLoadMode(ShaderLoadMode value)
{
    m_ShaderLoadMode = value;
}

inline GpuBenchmarkCompositeTest::TextureMode GpuBenchmarkCompositeTest::GetTextureMode() const
{
    return m_TextureMode;
}

inline void GpuBenchmarkCompositeTest::SetTextureMode(TextureMode value)
{
    m_TextureMode = value;
}

inline GpuBenchmarkCompositeTest::BlendingMode GpuBenchmarkCompositeTest::GetBlendingMode() const
{
    return m_BlendingMode;
}

inline void GpuBenchmarkCompositeTest::SetBlendingMode(BlendingMode value)
{
    m_BlendingMode= value;
}

inline GpuBenchmarkCompositeTest::VertexAttributeMode GpuBenchmarkCompositeTest::GetVertexAttributeMode() const
{
    return m_VertexAttributeMode;
}

inline void GpuBenchmarkCompositeTest::SetVertexAttributeMode(VertexAttributeMode value)
{
    m_VertexAttributeMode = value;
}

inline int GpuBenchmarkCompositeTest::GetVertexGridSize() const
{
    return m_VertexGridSize;
}

inline void GpuBenchmarkCompositeTest::SetVertexGridSize(int value)
{
    m_VertexGridSize = value;
}



} } } // namespace nnt { namespace gfx { namespace util {
