﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include <nn/gfx.h>
#include <nn/gfx/util/gfx_DebugFontTextWriter.h>

#if NN_GFX_IS_TARGET_NVN
#include <nvn/nvn.h>
#include <nvn/nvn_FuncPtrInline.h>
#endif

#include "gfxUtilGpuBenchmark_GpuBenchmarkBlitTexture.h"

#include "gfxUtilGpuBenchmark_ResHelpers.h"
#include "gfxUtilGpuBenchmark_PropertyMacros.h"
#include "gfxUtilGpuBenchmark_ComputeRenderQuadShaderVariationIndex.h"
#include "gfxUtilGpuBenchmark_ResourceAllocator.h"

#include "gfxUtilGpuBenchmark_BuiltinRenderQuadShader.h"


namespace nnt { namespace gfx { namespace util {

const char* GpuBenchmarkBlitTexture::ClassName = "BlitTexture";

GpuBenchmarkBlitTexture::GpuBenchmarkBlitTexture()
{
}

GpuBenchmarkBlitTexture::~GpuBenchmarkBlitTexture()
{
}

void GpuBenchmarkBlitTexture::Initialize(ResourceAllocator* pResourceAllocator)
{
    // RenderSizeConfiguration
    GpuBenchmarkPropertyHolder* pPropertyRenderSize = m_PropertyArray.Get(Property_RenderSizeConfiguration);
    BENCHMARK_PROPERTY_ENUM_DEFINITION(
        pPropertyRenderSize, "RenderSize",
        RenderSizeConfiguration, pResourceAllocator,
        GpuBenchmarkBlitTexture::GetRenderSizeConfiguration,
        GpuBenchmarkBlitTexture::SetRenderSizeConfiguration,
        "1280x720", RenderSizeConfiguration_1280x720,
        "1920x1080", RenderSizeConfiguration_1920x1080);

    // RenderFormat
    GpuBenchmarkPropertyHolder* pPropertyRenderFormat = m_PropertyArray.Get(Property_RenderFormat);
    BENCHMARK_PROPERTY_ENUM_DEFINITION(
        pPropertyRenderFormat, "RenderFormat",
        nn::gfx::ImageFormat, pResourceAllocator,
        GpuBenchmarkBlitTexture::GetRenderFormat,
        GpuBenchmarkBlitTexture::SetRenderFormat,
        "ImageFormat_R8_G8_B8_A8_Unorm", nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm,
        "ImageFormat_R16_G16_B16_A16_Float", nn::gfx::ImageFormat_R16_G16_B16_A16_Float);

#if NN_GFX_IS_TARGET_NVN
    // RenderNvnCompression
    GpuBenchmarkPropertyHolder* pPropertyRenderNvnCompression = m_PropertyArray.Get(Property_RenderNvnCompression);
    BENCHMARK_PROPERTY_ENUM_DEFINITION(
        pPropertyRenderNvnCompression, "RenderNvnCompression",
        RenderNvnCompression, pResourceAllocator,
        GpuBenchmarkBlitTexture::GetRenderNvnCompression,
        GpuBenchmarkBlitTexture::SetRenderNvnCompression,
        "Enabled", RenderNvnCompression_Enabled,
        "ForceDisabled", RenderNvnCompression_ForceDisable);
#endif

    // BlendingMode
    GpuBenchmarkPropertyHolder* pPropertyBlendingMode = m_PropertyArray.Get(Property_BlendingMode);
    BENCHMARK_PROPERTY_ENUM_DEFINITION(
        pPropertyBlendingMode, "BlendingMode",
        BlendingMode, pResourceAllocator,
        GpuBenchmarkBlitTexture::GetBlendingMode,
        GpuBenchmarkBlitTexture::SetBlendingMode,
        "None", BlendingMode_Off,
        "AlphaAdd", BlendingMode_AlphaAdd);

    // TextureSize
    GpuBenchmarkPropertyHolder* pPropertyTextureSize = m_PropertyArray.Get(Property_TextureSize);
    BENCHMARK_PROPERTY_ENUM_DEFINITION(
        pPropertyTextureSize, "TextureSize",
        TextureSize, pResourceAllocator,
        GpuBenchmarkBlitTexture::GetTextureSize,
        GpuBenchmarkBlitTexture::SetTextureSize,
        "1280x720", TextureSize_1280x720,
        "1920x1080", TextureSize_1920x1080,
        "256x256", TextureSize_256x256,
        "512x512", TextureSize_512x512,
        "1024x1024", TextureSize_1024x1024,
        "2048x2048", TextureSize_2048x2048);

    // TextureFilterMode
    GpuBenchmarkPropertyHolder* pPropertyTextureFilterMode = m_PropertyArray.Get(Property_TextureFilterMode);
    BENCHMARK_PROPERTY_ENUM_DEFINITION(
        pPropertyTextureFilterMode, "TextureFilterMode",
        nn::gfx::FilterMode, pResourceAllocator,
        GpuBenchmarkBlitTexture::GetTextureFilterMode,
        GpuBenchmarkBlitTexture::SetTextureFilterMode,
        "MinPoint_MagPoint_MipPoint", nn::gfx::FilterMode_MinPoint_MagPoint_MipPoint,
        "MinLinear_MagLinear_MipPoint", nn::gfx::FilterMode_MinLinear_MagLinear_MipPoint,
        "MinLinear_MagLinear_MipLinear", nn::gfx::FilterMode_MinLinear_MagLinear_MipLinear,
        "Anisotropic", nn::gfx::FilterMode_Anisotropic);

    // TextureFormat
    GpuBenchmarkPropertyHolder* pPropertyTextureFormat = m_PropertyArray.Get(Property_TextureFormat);
    BENCHMARK_PROPERTY_ENUM_DEFINITION(
        pPropertyTextureFormat, "TextureFormat",
        nn::gfx::ImageFormat, pResourceAllocator,
        GpuBenchmarkBlitTexture::GetTextureFormat,
        GpuBenchmarkBlitTexture::SetTextureFormat,
        "R8_G8_B8_A8_Unorm", nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm,
        "R16_G16_B16_A16_Float", nn::gfx::ImageFormat_R16_G16_B16_A16_Float);

    GpuBenchmarkPropertyHolder* pPropertyTextureFillMode = m_PropertyArray.Get(Property_TextureFillMode);
    BENCHMARK_PROPERTY_ENUM_DEFINITION(
        pPropertyTextureFillMode, "TextureFillMode",
        TextureFillMode, pResourceAllocator,
        GpuBenchmarkBlitTexture::GetTextureFillMode,
        GpuBenchmarkBlitTexture::SetTextureFillMode,
        "SingleColor", TextureFillMode_SingleColor,
        "ColorPattern", TextureFillMode_ColorPattern);
}

void GpuBenchmarkBlitTexture::Finalize(ResourceAllocator* pResourceAllocator)
{
    NN_UNUSED(pResourceAllocator);

    for (int i = 0; i < m_PropertyArray.GetCount(); ++i)
        m_PropertyArray.Get(i)->Finalize();
}

void GpuBenchmarkBlitTexture::InitializeGfxObjects(ResourceAllocator* pResourceAllocator, nn::gfx::Device* pDevice)
{
    GpuBenchmark::InitializeGfxObjects(pResourceAllocator, pDevice);

    InitializeRenderTexture(pResourceAllocator, pDevice);

    int textureWidth = GetTextureWidth();
    int textureHeight = GetTextureHeight();
    int mipCount = 1;

    nn::gfx::Texture::InfoType textureInfo;
    {
        textureInfo.SetDefault();
        textureInfo.SetGpuAccessFlags(nn::gfx::GpuAccess_Texture);
        textureInfo.SetWidth(textureWidth);
        textureInfo.SetHeight(textureHeight);
        textureInfo.SetImageFormat(m_TextureFormat);
        textureInfo.SetTileMode(nn::gfx::TileMode_Optimal);
        textureInfo.SetMipCount(mipCount);
    }

    InitializeTexture(&m_Texture, textureInfo, pResourceAllocator, MemoryPoolType_Data, pDevice);

    if (m_TextureFillMode == TextureFillMode_ColorPattern)
    {
        FillTextureWithColorPattern(&m_Texture, textureInfo, 0, pResourceAllocator, pDevice);
    }

    nn::gfx::TextureView::InfoType textureViewInfo;
    {
        textureViewInfo.SetDefault();
        textureViewInfo.SetTexturePtr(&m_Texture);
        textureViewInfo.SetImageDimension(nn::gfx::ImageDimension_2d);
        textureViewInfo.SetImageFormat(m_TextureFormat);
        textureViewInfo.EditSubresourceRange().EditMipRange().SetMipCount(mipCount);
    }
    m_TextureView.Initialize(pDevice, textureViewInfo);

    m_TextureSlotIndex = pResourceAllocator->AllocateAndSetTextureViewToDescriptorPool(
        &m_TextureView, &m_TextureDescriptorSlot);

    nn::gfx::Sampler::InfoType textureSamplerInfo;
    {
        textureSamplerInfo.SetDefault();
        textureSamplerInfo.SetFilterMode(m_TextureFilterMode);
        textureSamplerInfo.SetAddressU(nn::gfx::TextureAddressMode_ClampToEdge);
        textureSamplerInfo.SetAddressV(nn::gfx::TextureAddressMode_ClampToEdge);
        textureSamplerInfo.SetAddressW(nn::gfx::TextureAddressMode_ClampToEdge);
    }
    m_TextureSampler.Initialize(pDevice, textureSamplerInfo);

    m_TextureSamplerSlotIndex = pResourceAllocator->AllocateAndSetSamplerToDescriptorPool(
        &m_TextureSampler, &m_TextureSamplerDescriptorSlot);

    bool                    blendEnabled;
    nn::gfx::BlendFunction  blendFunction;
    nn::gfx::BlendFactor    blendDestinationFactor;
    nn::gfx::BlendFactor    blendSourceFactor;

    GetBlendConfiguration(&blendEnabled, &blendFunction, &blendDestinationFactor, &blendSourceFactor);

    nn::gfx::BlendTargetStateInfo targetInfo;
    targetInfo.SetDefault();
    targetInfo.SetChannelMask(nn::gfx::ChannelMask_Red | nn::gfx::ChannelMask_Green | nn::gfx::ChannelMask_Blue | nn::gfx::ChannelMask_Alpha);
    targetInfo.SetBlendEnabled(blendEnabled);
    targetInfo.SetColorBlendFunction(blendFunction);
    targetInfo.SetDestinationColorBlendFactor(blendDestinationFactor);
    targetInfo.SetSourceColorBlendFactor(blendSourceFactor);

    nn::gfx::BlendState::InfoType blendInfo;
    blendInfo.SetDefault();
    blendInfo.SetBlendTargetStateInfoArray(&targetInfo, 1);
    blendInfo.SetBlendConstant(0.5f, 0.5f, 0.5f, 0.5f);
    InitializeBlendState(&m_BlendState, blendInfo, pResourceAllocator, pDevice);


    InitializeResShader(&m_ResShader, g_RenderQuadShaderData, sizeof(g_RenderQuadShaderData), pResourceAllocator, pDevice);
}

void GpuBenchmarkBlitTexture::FinalizeGfxObjects(ResourceAllocator* pResourceAllocator, nn::gfx::Device* pDevice)
{
    FinalizeResShader(&m_ResShader, pResourceAllocator, pDevice);

    FinalizeBlendState(&m_BlendState, pResourceAllocator, pDevice);

    pResourceAllocator->FreeDescriptorSlots(nn::gfx::DescriptorPoolType_Sampler, m_TextureSamplerSlotIndex);
    m_TextureSamplerSlotIndex = -1;
    m_TextureSampler.Finalize(pDevice);

    pResourceAllocator->FreeDescriptorSlots(nn::gfx::DescriptorPoolType_TextureView, m_TextureSlotIndex);
    m_TextureSlotIndex = -1;
    m_TextureView.Finalize(pDevice);
    FinalizeTexture(&m_Texture, pResourceAllocator, pDevice);

    FinalizeColorRenderTarget(
        &m_RenderTexture, &m_OutputCopyBuffer,
        &m_RenderTextureColorTargetView, &m_ViewportScissorState,
        pResourceAllocator, pDevice);

    GpuBenchmark::FinalizeGfxObjects(pResourceAllocator, pDevice);
}

void GpuBenchmarkBlitTexture::PreBenchmark(nn::gfx::CommandBuffer* pTestCommandBuffer)
{
    pTestCommandBuffer->SetBlendState(&m_BlendState);

    const int shaderVariationIndex = ComputeRenderQuadShaderVariationIndex(1, 1);
    nn::gfx::ResShaderVariation* pResShaderVariation = m_ResShader.pResShaderContainer->GetResShaderVariation(shaderVariationIndex);
    nn::gfx::ResShaderProgram* pResShaderProgram = pResShaderVariation->GetResShaderProgram(m_ResShader.codeType);

#if defined(NN_SDK_BUILD_DEBUG)
    nn::util::BinTPtr<nn::gfx::ResShaderReflectionData> pPtrResShaderReflectionData = pResShaderProgram->ToData().pShaderReflection;
    if (pPtrResShaderReflectionData.Get() != nullptr)
    {
        nn::util::BinTPtr<nn::gfx::ResShaderReflectionStageData > pPtrResShaderReflectionStageData = pPtrResShaderReflectionData.Get()->pPixelReflection;
        nn::util::BinTPtr<nn::util::ResDic> pPtrSamplerDic = pPtrResShaderReflectionStageData.Get()->pSamplerDic;
        NN_ASSERT(pPtrSamplerDic.Get()->GetCount() == 1);
    }
#endif

    nn::gfx::Shader* pShader = pResShaderProgram->GetShader();
    pTestCommandBuffer->SetShader(pShader, nn::gfx::ShaderStageBit_All);

    pTestCommandBuffer->SetTextureAndSampler(
        0, nn::gfx::ShaderStage_Pixel, m_TextureDescriptorSlot, m_TextureSamplerDescriptorSlot);

#if 0 //NN_GFX_IS_TARGET_NVN
    const uint32_t clearColorui[] = { 127, 127, 127, 127 };
    nn::gfx::Device::DataType& dataType = nn::gfx::AccessorToData(pDevice);
    NVNboolean result = nvnDeviceRegisterFastClearColorui(dataType.pNvnDevice, clearColorui, NVN_FORMAT_RGBA8UI);
    if (result != NVN_TRUE)
    {
        NN_LOG("Cannot register clear color\n");
    }
#endif
    nn::gfx::ColorTargetView* pTestTarget = &m_RenderTextureColorTargetView;
    pTestCommandBuffer->ClearColor(pTestTarget, 0.5f, 0.5f, 0.5f, 0.5f, nullptr);
    pTestCommandBuffer->SetRenderTargets(1, &pTestTarget, nullptr);
    pTestCommandBuffer->SetViewportScissorState(&m_ViewportScissorState);
}

void GpuBenchmarkBlitTexture::DoBenchmark(nn::gfx::CommandBuffer* pTestCommandBuffer, int runCount)
{
    const int scale = 1;
    const int vertexOffset = (scale << 8);
    for (int runIndex = 0; runIndex < runCount; ++runIndex)
    {
        pTestCommandBuffer->Draw(nn::gfx::PrimitiveTopology_TriangleStrip, 4, vertexOffset);
    }
}

void GpuBenchmarkBlitTexture::RenderDebug(nn::gfx::CommandBuffer* pTestCommandBuffer)
{
#if defined(NN_GFXUTIL_GPUBENCHMARK_BLITTEXTURE_DEBUG)
    const int shaderVariationIndex = ComputeRenderQuadShaderVariationIndex(1, 1);
    nn::gfx::ResShaderVariation* pResShaderVariation = m_ResShader.pResShaderContainer->GetResShaderVariation(shaderVariationIndex);
    nn::gfx::ResShaderProgram* pResShaderProgram = pResShaderVariation->GetResShaderProgram(m_ResShader.codeType);

    nn::gfx::Shader* pShader = pResShaderProgram->GetShader();
    pTestCommandBuffer->SetShader(pShader, nn::gfx::ShaderStageBit_All);

    pTestCommandBuffer->SetTextureAndSampler(
        0, nn::gfx::ShaderStage_Pixel, m_TextureDescriptorSlot, m_TextureSamplerDescriptorSlot);

    const int scale = 2;
    const int offsetX = 1;
    const int offsetY = 1;
    const int vertexOffset = (scale << 8) | (offsetX << 16) | (offsetY << 24);
    pTestCommandBuffer->Draw(nn::gfx::PrimitiveTopology_TriangleStrip, 4, vertexOffset);
#else
    NN_UNUSED(pTestCommandBuffer);
#endif
}

void GpuBenchmarkBlitTexture::PrintResults(nn::TimeSpan cpuTimeElapsed, nn::TimeSpan gpuTimeElapsed, int runCount, nn::gfx::util::DebugFontTextWriter* pDebugFontTextWriter)
{
    NN_ASSERT(runCount > 0);

    uint64_t gpuTimeElapsedValueInNs = static_cast<uint64_t>(gpuTimeElapsed.GetNanoSeconds());
    uint64_t cpuTimeElapsedValueInNs = static_cast<uint64_t>(cpuTimeElapsed.GetNanoSeconds());

    uint64_t gpuTimeElapsedAvgValueInNs = gpuTimeElapsedValueInNs / static_cast<uint64_t>(runCount);
    uint64_t cpuTimeElapsedAvgValueInNs = cpuTimeElapsedValueInNs / static_cast<uint64_t>(runCount);

    pDebugFontTextWriter->Print("gpu time (ns): %8lu\n", gpuTimeElapsedAvgValueInNs);
    pDebugFontTextWriter->Print("cpu time (ns): %8lu\n", cpuTimeElapsedAvgValueInNs);
    pDebugFontTextWriter->Print("total gpu time (ns): %12lu\n", gpuTimeElapsedValueInNs);
    pDebugFontTextWriter->Print("total cpu time (ns): %12lu\n", cpuTimeElapsedValueInNs);
}

void GpuBenchmarkBlitTexture::CopyResultToBuffer(nn::gfx::CommandBuffer* pCommandBuffer)
{
    int renderWidth = GetRenderWidth();
    int renderHeight = GetRenderHeight();

    nn::gfx::BufferTextureCopyRegion bufferTextureCopyRegion;
    bufferTextureCopyRegion.SetDefault();
    bufferTextureCopyRegion.SetBufferImageWidth(renderWidth);
    bufferTextureCopyRegion.SetBufferImageHeight(renderHeight);
    bufferTextureCopyRegion.EditTextureCopyRegion().SetDefault();
    bufferTextureCopyRegion.EditTextureCopyRegion().SetWidth(renderWidth);
    bufferTextureCopyRegion.EditTextureCopyRegion().SetHeight(renderHeight);
    bufferTextureCopyRegion.EditTextureCopyRegion().EditSubresource().SetDefault();

    pCommandBuffer->InvalidateMemory(nn::gfx::GpuAccess_Texture);
    pCommandBuffer->CopyImageToBuffer(&m_OutputCopyBuffer, &m_RenderTexture, bufferTextureCopyRegion);
    pCommandBuffer->InvalidateMemory(nn::gfx::GpuAccess_Write);
}

void GpuBenchmarkBlitTexture::MapResultBuffer(void** pOutBuffer, size_t* pOutBufferSize)
{
    *pOutBuffer = m_OutputCopyBuffer.Map();
    *pOutBufferSize = m_OutputCopyBufferSize;
}

void GpuBenchmarkBlitTexture::UnmapResultBuffer()
{
    m_OutputCopyBuffer.Unmap();
}

const char* GpuBenchmarkBlitTexture::GetName() const
{
    return ClassName;
}

BenchmarkType GpuBenchmarkBlitTexture::GetType() const
{
    return BenchmarkType_BlitTexture;
}

int GpuBenchmarkBlitTexture::GetPropertyCount() const
{
    return m_PropertyArray.GetCount();
}

int GpuBenchmarkBlitTexture::FillPropertyList(const GpuBenchmarkPropertyHolder** ppDestinationArray, int destinationArrayMaxSize) const
{
    return m_PropertyArray.FillPropertyList(ppDestinationArray, destinationArrayMaxSize);
}

int GpuBenchmarkBlitTexture::FillPropertyList(GpuBenchmarkPropertyHolder** ppDestinationArray, int destinationArrayMaxSize)
{
    return m_PropertyArray.FillPropertyList(ppDestinationArray, destinationArrayMaxSize);
}

GpuBenchmarkPropertyHolder* GpuBenchmarkBlitTexture::FindPropertyByName(const char* propertyName)
{
    return m_PropertyArray.FindPropertyByName(propertyName);
}

GpuBenchmarkPropertyHolder* GpuBenchmarkBlitTexture::GetPropertyByIndex(int index)
{
    return m_PropertyArray.Get(index);
}

int GpuBenchmarkBlitTexture::GetRenderWidth() const
{
    switch (m_RenderSizeConfiguration)
    {
    case RenderSizeConfiguration_1280x720:
        return 1280;
    case RenderSizeConfiguration_1920x1080:
        return 1920;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

int GpuBenchmarkBlitTexture::GetRenderHeight() const
{
    switch (m_RenderSizeConfiguration)
    {
    case RenderSizeConfiguration_1280x720:
        return 720;
    case RenderSizeConfiguration_1920x1080:
        return 1080;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

int GpuBenchmarkBlitTexture::GetTextureWidth() const
{
    switch (m_TextureSize)
    {
    case TextureSize_1280x720:
        return 1280;
    case TextureSize_1920x1080:
        return 1920;
    case TextureSize_256x256:
        return 256;
    case TextureSize_512x512:
        return 512;
    case TextureSize_1024x1024:
        return 1024;
    case TextureSize_2048x2048:
        return 2048;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

int GpuBenchmarkBlitTexture::GetTextureHeight() const
{
    switch (m_TextureSize)
    {
    case TextureSize_1280x720:
        return 720;
    case TextureSize_1920x1080:
        return 1080;
    case TextureSize_256x256:
        return 256;
    case TextureSize_512x512:
        return 512;
    case TextureSize_1024x1024:
        return 1024;
    case TextureSize_2048x2048:
        return 2048;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

}

void GpuBenchmarkBlitTexture::GetBlendConfiguration(
    bool* pOutBlendEnabled,
    nn::gfx::BlendFunction* pOutBlendFunction,
    nn::gfx::BlendFactor* pOutBlendDestinationFactor,
    nn::gfx::BlendFactor* pOutBlendSourceFactor)
{
    switch (m_BlendingMode)
    {
    case BlendingMode_Off:
        *pOutBlendEnabled = false;
        *pOutBlendFunction = nn::gfx::BlendFunction_Add;
        *pOutBlendSourceFactor = nn::gfx::BlendFactor_SourceAlpha;
        *pOutBlendDestinationFactor = nn::gfx::BlendFactor_OneMinusSourceAlpha;
        break;

    case BlendingMode_AlphaAdd:
        *pOutBlendEnabled = true;
        *pOutBlendFunction = nn::gfx::BlendFunction_Add;
        *pOutBlendSourceFactor = nn::gfx::BlendFactor_SourceAlpha;
        *pOutBlendDestinationFactor = nn::gfx::BlendFactor_OneMinusSourceAlpha;
        break;

    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

void GpuBenchmarkBlitTexture::InitializeRenderTexture(ResourceAllocator* pResourceAllocator, nn::gfx::Device* pDevice)
{
    static const nn::gfx::TileMode tileMode = nn::gfx::TileMode_Optimal;

    int renderWidth = GetRenderWidth();
    int renderHeight = GetRenderHeight();
    bool disableFrameBufferCompression = false;

#if NN_GFX_IS_TARGET_NVN
    if (m_RenderNvnCompression == RenderNvnCompression_ForceDisable)
    {
        disableFrameBufferCompression = true;
    }
#endif

    if (disableFrameBufferCompression)
    {
        m_OutputCopyBufferSize = InitializeColorRenderTargetDisableFrameBufferCompression(
            &m_RenderTexture, &m_OutputCopyBuffer,
            &m_RenderTextureColorTargetView, &m_ViewportScissorState,
            renderWidth, renderHeight, m_RenderFormat, tileMode, pResourceAllocator, pDevice);
    }
    else
    {
        m_OutputCopyBufferSize = InitializeColorRenderTarget(
            &m_RenderTexture, &m_OutputCopyBuffer,
            &m_RenderTextureColorTargetView, &m_ViewportScissorState,
            renderWidth, renderHeight, m_RenderFormat, tileMode, pResourceAllocator, pDevice);
    }
}

void GpuBenchmarkBlitTexture::FinalizeRenderTexture(ResourceAllocator* pResourceAllocator, nn::gfx::Device* pDevice)
{
    FinalizeColorRenderTarget(
        &m_Texture, &m_OutputCopyBuffer,
        &m_RenderTextureColorTargetView,
        &m_ViewportScissorState, pResourceAllocator, pDevice);
}

} } } // namespace nnt { namespace gfx { namespace util {
