﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/////////////////////////////////////////////////////////////////////////////
//
// Draws a rotating octahedron with each tessellation types.
//
//////////////////////////////////////////////////////////////////////////////

#include <cstdio>
#include <cstring>

#include <gfx/demo.h>

#include <nn/nn_Log.h>
#include <nnt.h>
#include <nnt/nnt_Argument.h>

////////////////////////////////////////////////////
//
// Assets data, types and interface for demos
//
////////////////////////////////////////////////////

// A .gsh file is a GX2 specific file that contains vertex and pixel
// shader data.
static const char* ShaderList[] =
{
    "shaders/tessellateAdaptive",
    "shaders/tessellateAdaptiveSO",
    "shaders/tessellateAdaptiveHlslcc",
};

#if NN_GFX_IS_TARGET_GX
// Fetch Shader
static GX2FetchShader g_tessFetchShader;

#else
static int s_staticUniformLoc = -1;
static int s_dynamicUniformLoc = -1;
#endif

// Extra state objects to render the final objects
static nn::gfx::BlendState s_blendState;
static void* s_pBlendStateData;
static nn::gfx::RasterizerState s_rasterizerState;

// Struct for uniform blocks
// Need to match uniform blocks in shader code
typedef struct _UniformBlockDynamic
{
    Mtx44   modelMtx44;            // 0
    float   time;                  // 64
} UniformBlockDynamic;

typedef struct _UniformBlockStatic
{
    Mtx44   viewMtx44;    // 0
    Mtx44   projMtx44;    // 64
    f32     lightPos[4];  // 128
} UniformBlockStatic;

// Struct for uniforms
typedef struct _Uniforms
{
    // Uniform Block for dynamic updating data
    // Double-buffered (GPU reads one while the CPU writes the other)
    DEMOGfxBuffer dynamicUB[ 2 ];

    // Uniform Block for static data
    DEMOGfxBuffer staticUB;
} Uniforms;

static DEMO_F32x3F32x3 OCT_VERTEX_DATA[] = {

    { { { { { 1.0f, 0.0f, 0.0f } } },    // 0:0 PP0
        { { { 1.0f, 0.0f, 0.0f } } } } },
    { { { { { 0.0f, 1.0f, 0.0f } } },    // 0:1 0PP
        { { { 0.0f, 1.0f, 0.0f } } } } },
    { { { { { 0.0f, 0.0f, 1.0f } } },    // 0:2 P0P
        { { { 0.0f, 0.0f, 1.0f } } } } },

    { { { { { 0.0f, 1.0f, 0.0f } } },    // 1:3 NP0
        { { { 0.0f, 1.0f, 0.0f } } } } },
    { { { { { -1.0f, 0.0f, 0.0f } } },   // 1:4 N0P
        { { { -1.0f, 0.0f, 0.0f } } } } },
    { { { { { 0.0f, 0.0f, 1.0f } } },    // 1:5 0PP
        { { { 0.0f, 0.0f, 1.0f } } } } },

    { { { { { -1.0f, 0.0f, 0.0f } } },   // 2:6 NN0
        { { { -1.0f, 0.0f, 0.0f } } } } },
    { { { { { 0.0f, -1.0f, 0.0f } } },   // 2:7 0NP
        { { { 0.0f, -1.0f, 0.0f } } } } },
    { { { { { 0.0f, 0.0f, 1.0f } } },    // 2:8 N0P
        { { { 0.0f, 0.0f, 1.0f } } } } },

    { { { { { 0.0f, -1.0f, 0.0f } } },   // 3:9 PN0
        { { { 0.0f, -1.0f, 0.0f } } } } },
    { { { { { 1.0f, 0.0f, 0.0f } } },    // 3:10 P0P
        { { { 1.0f, 0.0f, 0.0f } } } } },
    { { { { { 0.0f, 0.0f, 1.0f } } },    // 3:11 0NP
        { { { 0.0f, 0.0f, 1.0f } } } } },

    { { { { { 0.0f, 1.0f, 0.0f } } },    // 4:12 PP0
        { { { 0.0f, 1.0f, 0.0f } } } } },
    { { { { { 1.0f, 0.0f, 0.0f } } },    // 4:13 P0N
        { { { 1.0f, 0.0f, 0.0f } } } } },
    { { { { { 0.0f, 0.0f, -1.0f } } },   // 4:14 0PN
        { { { 0.0f, 0.0f, -1.0f } } } } },

    { { { { { -1.0f, 0.0f, 0.0f } } },   // 5:15 NP0
        { { { -1.0f, 0.0f, 0.0f } } } } },
    { { { { { 0.0f, 1.0f, 0.0f } } },    // 5:16 0PN
        { { { 0.0f, 1.0f, 0.0f } } } } },
    { { { { { 0.0f, 0.0f, -1.0f } } },   // 5:17 N0N
        { { { 0.0f, 0.0f, -1.0f } } } } },

    { { { { { 0.0f, -1.0f, 0.0f } } },   // 6:18 NN0
        { { { 0.0f, -1.0f, 0.0f } } } } },
    { { { { { -1.0f, 0.0f, 0.0f } } },   // 6:19 N0N
        { { { -1.0f, 0.0f, 0.0f } } } } },
    { { { { { 0.0f, 0.0f, -1.0f } } },   // 6:20 0NN
        { { { 0.0f, 0.0f, -1.0f } } } } },

    { { { { { 1.0f, 0.0f, 0.0f } } },    // 7:21 PN0
        { { { 1.0f, 0.0f, 0.0f } } } } },
    { { { { { 0.0f, -1.0f, 0.0f } } },   // 7:22 0NN
        { { { 0.0f, -1.0f, 0.0f } } } } },
    { { { { { 0.0f, 0.0f, -1.0f } } },   // 7:23 P0N
        { { { 0.0f, 0.0f, -1.0f } } } } },
};

// edge midpoints calculated from vertex data at load time
static DEMO_F32x3 OCT_MIDPOINT_DATA[] = {

    { { {0.0f, 0.0f, 0.0f} } },
    { { {0.0f, 0.0f, 0.0f} } },
    { { {0.0f, 0.0f, 0.0f} } },
    { { {0.0f, 0.0f, 0.0f} } },
    { { {0.0f, 0.0f, 0.0f} } },
    { { {0.0f, 0.0f, 0.0f} } },
    { { {0.0f, 0.0f, 0.0f} } },
    { { {0.0f, 0.0f, 0.0f} } },
    { { {0.0f, 0.0f, 0.0f} } },
    { { {0.0f, 0.0f, 0.0f} } },
    { { {0.0f, 0.0f, 0.0f} } },
    { { {0.0f, 0.0f, 0.0f} } },
    { { {0.0f, 0.0f, 0.0f} } },
    { { {0.0f, 0.0f, 0.0f} } },
    { { {0.0f, 0.0f, 0.0f} } },
    { { {0.0f, 0.0f, 0.0f} } },
    { { {0.0f, 0.0f, 0.0f} } },
    { { {0.0f, 0.0f, 0.0f} } },
    { { {0.0f, 0.0f, 0.0f} } },
    { { {0.0f, 0.0f, 0.0f} } },
    { { {0.0f, 0.0f, 0.0f} } },
    { { {0.0f, 0.0f, 0.0f} } },
    { { {0.0f, 0.0f, 0.0f} } },
    { { {0.0f, 0.0f, 0.0f} } }
};

static const u32 OCT_NUM_VERTICES = 24;

// Struct for attribute buffers
typedef struct _AttribBuffer
{
    u32 vertexSize;
    u32 vertexStride;
    u32 midpointSize;
    u32 midpointStride;

    DEMOGfxBuffer vertexBuffer;
    DEMOGfxBuffer midpointBuffer;
} AttribBuffer;

// Viewport Information
static const int VIEWPORT_MAX_W = 16;
static const int VIEWPORT_MAX_H = 9;

// Animation data
static u32 s_frameCount;
static f32 s_time;
static f32 s_yrad;
static f32 s_zpos;
static f32 s_zvel;
static bool s_drawOutline;

////////////////////////////////////////////////////
//
// Prototypes
//
////////////////////////////////////////////////////
#if NN_GFX_IS_TARGET_GX
static void InitStreamOutBuffers(GX2StreamOutBuffer *pStreamOutBuf, DEMOGfxShader* pShader);
static void InitShaderSO( DEMOGfxPipeline *pPipeline, AttribBuffer *pAttribData, Uniforms *pUniforms, GX2StreamOutBuffer *pStreamOutBuf, const char *fileName );
static void FreeShaderSO(DEMOGfxPipeline *pPipeline, GX2StreamOutBuffer *pStreamOutBuf);
static void DrawModel(DEMOGfxPipeline *pPipeline, AttribBuffer *pAttribData, Uniforms *pUniforms, GX2StreamOutBuffer *pStreamOutBuffer);
#else
static void DrawModel( DEMOGfxPipeline *pPipeline, AttribBuffer *pAttribData, Uniforms *pUniforms );
#endif

static void InitShaderUB( DEMOGfxPipeline *pPipeline, AttribBuffer *pAttribData, Uniforms *pUniforms, const char* fileNames );
static void FreeShaderUB(DEMOGfxPipeline *pPipeline, Uniforms *pUniforms);
static void InitAttribData(AttribBuffer *pAttribData, u32 vertexDataSize, u32 vertexDataStride, void *vertexData, u32 midpointDataSize, u32 midpointDataStride, void *midpointDat);
static void FreeAttribData(AttribBuffer *pAttribData);
static void InitUniforms(Uniforms *pUniforms);
static void UpdateUniforms(Uniforms *pUniforms);

////////////////////////////////////////////////////
//
// Functions
//
////////////////////////////////////////////////////

#if NN_GFX_IS_TARGET_GX
static void InitStreamOutBuffers(GX2StreamOutBuffer *pStreamOutBuf, DEMOGfxShader* pShader)
{
    u32 memSize;

    // Calculate the size of the stream out buffer
    pStreamOutBuf->size = (OCT_NUM_VERTICES * sizeof(f32));

    // Use a single allocation for the stream out buffer and context memory
    memSize = pStreamOutBuf->size + sizeof(GX2StreamOutContext);

    // allocate the stream out buffer and the memory for the stream out context
    pStreamOutBuf->dataPtr = DEMOGfxAllocMEM2(memSize, GX2_STREAMOUT_BUFFER_ALIGNMENT);

    pStreamOutBuf->ctxPtr = (GX2StreamOutContext*)((u8*)pStreamOutBuf->dataPtr + pStreamOutBuf->size);

    // Invalidate the stream out buffer
    GX2Invalidate(GX2_INVALIDATE_CPU, pStreamOutBuf->dataPtr, memSize);

    // Set the vertex stride for the stream out buffers
    pStreamOutBuf->vertexStride =
        reinterpret_cast< GX2VertexShader* >( pShader->vertexShaderData )->streamOutVertexStride[0];

    // Set the stream out buffers
    GX2SetStreamOutBuffer(0, pStreamOutBuf);
}

// The initialization function for the streamout portions of this sample.
// It is responsible for allocating the streamout shader and buffers
// as well as ensuring that data is flushed from the CPU to GPU memory
// for UB Shader
static void InitShaderSO(DEMOGfxPipeline *pPipeline, AttribBuffer *pAttribData, Uniforms *pUniforms, GX2StreamOutBuffer *pStreamOutBuf, const char *fileName)
{
    DEMOGfxShader *pShader = &pPipeline->shaders;

    // Load shader binary to memory allocated by DEMOGfxLoadAssetFile. This
    // memory must be freed with a call to DEMOFree after shaders are loaded.
    DEMOGfxLoadShadersFromFile( pShader, 0, fileName );

    // Init attribute to shader
    DEMOGfxInitShaderAttribute(pShader,
                               "a_position",
                               0,
                               0,
                               nn::gfx::AttributeFormat_32_32_32_Float);

    // Get vs uniform block location
    DEMOGfxGetVertexShaderUniformBlockLocation(pShader, "ub_staticBlock");
    DEMOGfxGetVertexShaderUniformBlockLocation(pShader, "ub_dynamicBlock");

    // Set vs static uniform block
    DEMOCommandBuffer.SetConstantBuffer( pShader->uniformBlocksVS.location[ 0 ], nn::gfx::ShaderStage_Vertex,
        pUniforms->staticUB.gpuAddress, pUniforms->staticUB.size );

    // Initialize the stream out buffers
    // This needs to be done after the shaders are initialized
    InitStreamOutBuffers(pStreamOutBuf, pShader);

    // Setup the pipeline

    // This is just for stream out so disable rasterization. Unfortunately we need
    // to setup the blend/color targets even though we're not using them.
    pPipeline->blendTargetStateCount = 1;
    pPipeline->colorTargetStateCount = 1;
    pPipeline->blendTargetStateInfoArray[ 0 ].SetDefault();
    pPipeline->colorTargetStateInfoArray[ 0 ].SetDefault();
    pPipeline->rasterizerStateInfo.SetRasterEnabled( false );
    pPipeline->rasterizerStateInfo.SetDepthClipEnabled( true );

    DEMOGfxInitShaderVertexBuffer( &pPipeline->shaders, 0, pAttribData->midpointStride, 0 );

    pPipeline->Initialize( &DEMODevice );
}

static BOOL GfxInitFetchShaderEx(DEMOGfxShader *pShader)
{
    GX2AttribStream attribs[ 16 ];

    if(!pShader)
    {
        OSReport("Error : Null pointer.\n");
        return FALSE;
    }

    memset(&g_tessFetchShader, 0, sizeof(g_tessFetchShader));

    g_tessFetchShader.shaderSize =
        GX2CalcFetchShaderSizeEx(pShader->attribCount + 2,
                                 GX2_FETCH_SHADER_TESSELATION_TRIANGLES,
                                 GX2_TESSELLATION_MODE_ADAPTIVE);


    u32 offset = 0;
    int index = 0;

    // Get hardware location for the system generated values
    GX2InitAttribStream( &attribs[ index++ ],
        GX2GetVertexAttribVarLocation(reinterpret_cast< GX2VertexShader* >( pShader->vertexShaderData ), "gl_VertexTriangleIndex"),
        GX2_SYSTEM_GENERATED_ATTRIBUTE,
        GX2_SYSGEN_ATTRIB_TESSELLATION_INDEX,
        GX2_ATTRIB_FORMAT_32_32_32_UINT );

    GX2InitAttribStream( &attribs[ index++ ],
        GX2GetVertexAttribVarLocation(reinterpret_cast< GX2VertexShader* >( pShader->vertexShaderData ), "gl_BarycentricCoord" ),
        GX2_SYSTEM_GENERATED_ATTRIBUTE,
        GX2_SYSGEN_ATTRIB_TESSELLATION_BARYC,
        GX2_ATTRIB_FORMAT_32_32_32_FLOAT );

    while ( index - 2 < pShader->attribCount )
    {
        GX2InitAttribStream( &attribs[ index ],
            pShader->vertexAttributes[ index - 2 ].GetShaderSlot(),
            0,
            offset,
            pShader->vertexAttributes[ index - 2 ].GetFormat() ==
            nn::gfx::AttributeFormat_32_32_32_Float ?  GX2_ATTRIB_FORMAT_32_32_32_FLOAT : GX2_ATTRIB_FORMAT_32_32_32_UINT );

        offset += 12;
        index++;
    }


    // Allocate memory for the fetch shaders based on the number of attributes
    // for the vertex shader.
    g_tessFetchShader.shaderPtr = DEMOGfxAllocMEM2( g_tessFetchShader.shaderSize, GX2_SHADER_ALIGNMENT );

    // Initialize the fetch shader. This will construct the fetch shader to
    // use later in a call to GX2SetShaders.
    GX2InitFetchShaderEx( &g_tessFetchShader,
        g_tessFetchShader.shaderPtr,
        index,
        attribs,
        GX2_FETCH_SHADER_TESSELATION_TRIANGLES,
        GX2_TESSELLATION_MODE_ADAPTIVE );

    return TRUE;
}
#endif

// The initialization function for the rendering portions of this sample.
// It is responsible for allocating the three types of shaders and buffers
// as well as ensuring that data is flushed from the CPU to GPU memory
// for UB Shader
static void InitShaderUB( DEMOGfxPipeline *pPipeline, AttribBuffer *pAttribData, Uniforms *pUniforms, const char* fileName )
{
    DEMOGfxShader* pShader = &pPipeline->shaders;

    DEMOGfxLoadShadersFromFile(pShader, 0, fileName);

    // Init attribute to shader
    DEMOGfxInitShaderAttribute( pShader,
        "a_position",
        0,
        0,
        nn::gfx::AttributeFormat_32_32_32_Float );

    DEMOGfxInitShaderAttribute( pShader,
        "a_normal",
        0,
        sizeof( DEMO_F32x3 ),
        nn::gfx::AttributeFormat_32_32_32_Float );

#if NN_GFX_IS_TARGET_GX
    // Get vs uniform block location
    DEMOGfxGetVertexShaderUniformBlockLocation(pShader, "ub_staticBlock");
    DEMOGfxGetVertexShaderUniformBlockLocation(pShader, "ub_dynamicBlock");
#endif

#if NN_GFX_IS_TARGET_GX
    // Set fetch Shader buffer
    GfxInitFetchShaderEx(pShader);
#endif

    // Initialize the static uniform's buffer/view
    pUniforms->staticUB.Initialize( sizeof( UniformBlockStatic ), NULL,
        nn::gfx::GpuAccess_ConstantBuffer, 0 );

    // Initialize camera position
    InitUniforms(pUniforms);

    // Initialize the dynamic uniform's buffer/view
    pUniforms->dynamicUB[ 0 ].Initialize( sizeof( UniformBlockDynamic ), NULL,
        nn::gfx::GpuAccess_ConstantBuffer, 0 );
    pUniforms->dynamicUB[ 1 ].Initialize( sizeof( UniformBlockDynamic ), NULL,
        nn::gfx::GpuAccess_ConstantBuffer, 0 );

    pPipeline->SetDefaults();

    pPipeline->blendTargetStateCount = 1;
    pPipeline->colorTargetStateCount = 1;

    pPipeline->colorTargetStateInfoArray[ 0 ].SetDefault();
    pPipeline->blendTargetStateInfoArray[ 0 ].SetDefault();
    pPipeline->blendTargetStateInfoArray[ 0 ].SetBlendEnabled( true );
    pPipeline->blendTargetStateInfoArray[ 0 ].SetSourceColorBlendFactor( nn::gfx::BlendFactor_SourceAlpha );
    pPipeline->blendTargetStateInfoArray[ 0 ].SetDestinationColorBlendFactor( nn::gfx::BlendFactor_OneMinusSourceAlpha );
    pPipeline->blendTargetStateInfoArray[ 0 ].SetSourceAlphaBlendFactor( nn::gfx::BlendFactor_SourceAlpha );
    pPipeline->blendTargetStateInfoArray[ 0 ].SetDestinationAlphaBlendFactor( nn::gfx::BlendFactor_OneMinusSourceAlpha );
    pPipeline->blendTargetStateInfoArray[ 0 ].SetColorBlendFunction( nn::gfx::BlendFunction_Add );
    pPipeline->blendTargetStateInfoArray[ 0 ].SetAlphaBlendFunction( nn::gfx::BlendFunction_Add );

    pPipeline->depthStencilStateInfo.SetDepthComparisonFunction( nn::gfx::ComparisonFunction_Less );

    // Setup auxiliary states to switch from wire frame to solid mode
    pPipeline->blendStateInfo.SetBlendTargetStateInfoArray(
        pPipeline->blendTargetStateInfoArray, pPipeline->blendTargetStateCount );

    size_t blendDataSize = nn::gfx::BlendState::GetRequiredMemorySize( pPipeline->blendStateInfo );
    s_pBlendStateData = DEMOGfxAllocMEM2( blendDataSize, nn::gfx::BlendState::RequiredMemoryInfo_Alignment );
    s_blendState.SetMemory( s_pBlendStateData, blendDataSize );
    s_blendState.Initialize( &DEMODevice, pPipeline->blendStateInfo );

    DEMOTestIsUseHlslccGlsl()
        ? pPipeline->rasterizerStateInfo.SetCullMode( nn::gfx::CullMode_Front )
        : pPipeline->rasterizerStateInfo.SetCullMode( nn::gfx::CullMode_Back );

    s_rasterizerState.Initialize( &DEMODevice, pPipeline->rasterizerStateInfo );

    // Set the main pipeline to draw an outline
    pPipeline->blendStateInfo.SetLogicOperationEnabled( true );
    pPipeline->blendStateInfo.SetLogicOperation( nn::gfx::LogicOperation_Clear );

    pPipeline->rasterizerStateInfo.SetCullMode( nn::gfx::CullMode_None );
    pPipeline->rasterizerStateInfo.SetFillMode( nn::gfx::FillMode_Wireframe );
    pPipeline->rasterizerStateInfo.SetDepthBias( -5 );
    pPipeline->rasterizerStateInfo.SetDepthBiasClamp( 0.0f );
    pPipeline->rasterizerStateInfo.SetSlopeScaledDepthBias( 0.0f );

    // Setup tessellation control information
    pPipeline->tessellationStateInfo.SetPatchControlPointCount( 3 );

    // Setup the buffers
    DEMOGfxInitShaderVertexBuffer( &pPipeline->shaders, 0, pAttribData->vertexStride, 0 );

#if !NN_GFX_IS_TARGET_GX
    s_staticUniformLoc = pPipeline->shaders.GetInterfaceSlot( nn::gfx::ShaderStage_Domain, nn::gfx::ShaderInterfaceType_ConstantBuffer, "ub_staticBlock" );
    s_dynamicUniformLoc = pPipeline->shaders.GetInterfaceSlot( nn::gfx::ShaderStage_Domain, nn::gfx::ShaderInterfaceType_ConstantBuffer, "ub_dynamicBlock" );
#endif

    pPipeline->Initialize( &DEMODevice );
}

// Free shader
static void FreeShaderUB( DEMOGfxPipeline *pPipeline, Uniforms *pUniforms )
{
#if NN_GFX_IS_TARGET_GX
    if(g_tessFetchShader.shaderPtr)
        DEMOGfxFreeMEM2(g_tessFetchShader.shaderPtr);
#endif

    pPipeline->Finalize( &DEMODevice );
    pUniforms->dynamicUB[ 0 ].Finalize();
    pUniforms->dynamicUB[ 1 ].Finalize();
    pUniforms->staticUB.Finalize();

    s_rasterizerState.Finalize( &DEMODevice );
    s_blendState.Finalize( &DEMODevice );
    DEMOGfxFreeMEM2( s_pBlendStateData );
}

#if NN_GFX_IS_TARGET_GX
// Free streamout shader
static void FreeShaderSO(DEMOGfxPipeline *pPipeline, GX2StreamOutBuffer *pStreamOutBuf)
{
    pPipeline->Finalize( &DEMODevice );
    DEMOGfxFreeMEM2(pStreamOutBuf->dataPtr);
}
#endif

// Initialize attribute buffer and data
static void InitAttribData(AttribBuffer *pAttribData, u32 vertexDataSize, u32 vertexDataStride, void *vertexData, u32 midpointDataSize, u32 midpointDataStride, void *midpointData)
{
    u32 i, j, k;

    // Set values for vertex
    pAttribData->vertexSize   = vertexDataSize;
    pAttribData->vertexStride = vertexDataStride;

    // Set values for midpoint
    pAttribData->midpointSize   = midpointDataSize;
    pAttribData->midpointStride = midpointDataStride;

    // Calculate edge midpoints from vertices
    for(i = 0, j = 0; i < midpointDataSize; i += midpointDataStride, j += vertexDataStride)
    {
        ASSERT(j < vertexDataSize && "Midpoint out of range");

        // find the midpoint of each edge between the 3 vertices
        s32 connectingVertOffset = (s32)vertexDataStride;
        if(j / vertexDataStride % 3 == 2) connectingVertOffset = -2 * (s32)vertexDataStride;

        for(k = 0; k < 3; ++k)
        {
            *((f32*)((u8*)midpointData + i) + k) = *((f32*)((u8*)vertexData + j) + k) * 0.5f +
                                                   *((f32*)((u8*)vertexData + j + connectingVertOffset) + k) * 0.5f;
        }
    }

    // Allocate buffers for attributes
    pAttribData->vertexBuffer.Initialize( vertexDataSize, vertexData,
        nn::gfx::GpuAccess_VertexBuffer, 0 );
    pAttribData->midpointBuffer.Initialize( midpointDataSize, midpointData,
        nn::gfx::GpuAccess_VertexBuffer, 0 );
}

// Free attibute buffers
static void FreeAttribData(AttribBuffer *pAttribData)
{
    // Free buffers for attributes
    pAttribData->vertexBuffer.Finalize();
    pAttribData->midpointBuffer.Finalize();
}

// Init function for uniforms
static void InitUniforms(Uniforms *pUniforms)
{
    Mtx   lookAtMtx34;
    Vec     up = {0.0f, 1.0f, 0.0f};
    Vec  objPt = {0.0f, 0.0f, 0.0f};
    Vec camLoc = {0.0f, 1.0f,-3.0f};
    Vec lightLoc = {-14.364607f, 20.0f,-8.0f};
    f32   pers  = 50.0f;
    f32 aspect  = (f32)VIEWPORT_MAX_W / (f32)VIEWPORT_MAX_H;
    f32  znear  = 1.0f;
    f32   zfar  = 2000.0f;

    UniformBlockStatic* pStaticUB = pUniforms->staticUB.Map< UniformBlockStatic >();

    // Compute perspective matrix
    MTXPerspective(pStaticUB->projMtx44, pers, aspect, znear, zfar);

    // Compute lookAt matrix
    MTXLookAt(lookAtMtx34, &camLoc, &up, &objPt);
    MTX34To44(lookAtMtx34, pStaticUB->viewMtx44);

    // Set Light Position
    pStaticUB->lightPos[0] = lightLoc.x;
    pStaticUB->lightPos[1] = lightLoc.y;
    pStaticUB->lightPos[2] = lightLoc.z;
    pStaticUB->lightPos[3] = 0.0f;

#if NN_GFX_IS_TARGET_GX
    // Uniform buffers must be in little-endian mode
    GX2EndianSwap(pStaticUB, sizeof(UniformBlockStatic));
#endif

    pUniforms->staticUB.Unmap();

}

// Update uniforms
static void UpdateUniforms(Uniforms *pUniforms)
{
    UniformBlockDynamic* pDynamicUB = pUniforms->dynamicUB[ s_frameCount % 2 ].Map< UniformBlockDynamic >();

    // row major matrix
    Mtx  transMtx34;
    Mtx  rotYMtx34;
    Mtx  modelMtx34;

    // Compute rotation matrix
    MTXRotRad(rotYMtx34, 'y', s_yrad);
    MTXTrans(transMtx34, 0, 0, s_zpos);
    s_zvel += 0.01f;
    s_zpos += s_zvel;
    if(s_zpos > 200.0f)
    {
        s_zvel = -2.005f;
        s_zpos = 200.0f;

        s_drawOutline = !s_drawOutline;
    }

    // Compute model matrix
    MTXConcat(transMtx34, rotYMtx34, modelMtx34);
    MTX34To44(modelMtx34, pDynamicUB->modelMtx44);

    s_yrad += 0.02f;

    // Get time
    pDynamicUB->time = s_time;
    s_time += 0.001f;

    if ( s_time > 1.0f )
    {
        s_time = 0.0f;
    }

#if NN_GFX_IS_TARGET_GX
    // Uniform buffers must be little-endian
    GX2EndianSwap(pDynamicUB, sizeof(UniformBlockDynamic));
#endif

    pUniforms->dynamicUB[ s_frameCount % 2 ].Unmap();
}

// The draw function for the rendering portions of this app
#if NN_GFX_IS_TARGET_GX
static void DrawModel(DEMOGfxPipeline *pPipeline, AttribBuffer *pAttribData, Uniforms *pUniforms, GX2StreamOutBuffer *pStreamOutBuffer)
#else
static void DrawModel(DEMOGfxPipeline *pPipeline, AttribBuffer *pAttribData, Uniforms *pUniforms)
#endif
{
    // When using the DEMO library, it is necessary to call
    // DEMOGfxBeforeRender and DEMOGfxDoneRender before and after drawing.
    DEMOGfxBeforeRender();

    nn::gfx::ColorTargetView* pCurrentScanBuffer = DEMOGetColorBufferView();
    DEMOCommandBuffer.ClearColor( pCurrentScanBuffer, 0.45f, 0.5f, 0.72f, 1.0f, NULL );
    DEMOCommandBuffer.ClearDepthStencil( &DEMODepthBufferView, 1.0f, 0, nn::gfx::DepthStencilClearMode_DepthStencil, NULL );

    DEMOGfxSetDefaultRenderTarget();
    DEMOGfxSetDefaultViewportScissor();

#if NN_GFX_IS_TARGET_GX

    GX2SetShaderMode( GX2_SHADER_MODE_UNIFORM_BLOCK );

    ////////////////////////////////////////////////////////////////////////////////////////
    //// Pass 1 Disable the rasterizer and store the midpoint data in a stream out buffer //
    ////////////////////////////////////////////////////////////////////////////////////////

    DEMOCommandBuffer.InvalidateMemory( nn::gfx::GpuAccess_ConstantBuffer );

    // Set vs static uniform block
    DEMOCommandBuffer.SetConstantBuffer( pPipeline[ 1 ].shaders.uniformBlocksVS.location[ 0 ],
        nn::gfx::ShaderStage_Vertex, pUniforms->staticUB.gpuAddress, pUniforms->staticUB.size );

    // Set Dynamic (per-frame) uniform blocks

    // Set vs dynamic uniform block (same location for streamout and tesselate shaders)
    DEMOCommandBuffer.SetConstantBuffer( pPipeline[ 1 ].shaders.uniformBlocksVS.location[ 1 ],
        nn::gfx::ShaderStage_Vertex, pUniforms->dynamicUB[ s_frameCount % 2 ].gpuAddress, pUniforms->dynamicUB[ s_frameCount % 2 ].size );

    // Set Attrib buffer
    DEMOCommandBuffer.SetVertexBuffer( 0, pAttribData->midpointBuffer.gpuAddress, pAttribData->midpointStride, pAttribData->midpointBuffer.size );

    // This call with set all shaders.
    DEMOCommandBuffer.SetPipeline( &pPipeline[ 1 ].pipeline );

    // Enable Stream out
    GX2SetStreamOutEnable(GX2_ENABLE);

    // This needs to be called before the first draw that uses the buffer (after a context switch)
    GX2SetStreamOutContext(0, pStreamOutBuffer, GX2_TRUE);

    // Draw Triangle and capture the vertex data in the stream out buffer
    GX2Draw(GX2_PRIMITIVE_TRIANGLES, OCT_NUM_VERTICES);

    // This needs to be called before GX2DrawStreamOut() is used or before a context switch
    GX2SaveStreamOutContext(0, pStreamOutBuffer);

    // Disable Stream out
    GX2SetStreamOutEnable(GX2_DISABLE);

    // Flush the stream out data so it is visible
    GX2Invalidate( GX2_INVALIDATE_STREAMOUT_BUFFER, 0, ~0 );
#endif

    ///////////////////////////////////////////////////////////////////////////
    //// Pass 2 Tessellate based on edge factors calculated using Stream Out //
    ///////////////////////////////////////////////////////////////////////////

#if NN_GFX_IS_TARGET_GX

    // Set tessellation mode
    GX2SetTessellation(GX2_TESSELLATION_MODE_ADAPTIVE,
                       GX2_PRIMITIVE_TESSELLATE_TRIANGLES,
                       GX2_INDEX_FORMAT_U32_LE);  // Must be set to 32 for non-indexed triangles


    // Set tessellation Factor
    GX2SetMinTessellationLevel(1.0f);
    GX2SetMaxTessellationLevel(15.0);

    // Set vs static uniform block
    DEMOCommandBuffer.SetConstantBuffer( pPipeline[ 0 ].shaders.uniformBlocksVS.location[ 0 ],
        nn::gfx::ShaderStage_Vertex, pUniforms->staticUB.gpuAddress, pUniforms->staticUB.size );

    // Set Dynamic (per-frame) uniform blocks

    // Set vs dynamic uniform block (same location for streamout and tesselate shaders)
    DEMOCommandBuffer.SetConstantBuffer( pPipeline[ 0 ].shaders.uniformBlocksVS.location[ 1 ],
        nn::gfx::ShaderStage_Vertex, pUniforms->dynamicUB[ s_frameCount % 2 ].gpuAddress, pUniforms->dynamicUB[ s_frameCount % 2 ].size );
#else
    // Set vs static uniform block
    DEMOCommandBuffer.SetConstantBuffer( s_staticUniformLoc,
            nn::gfx::ShaderStage_Domain, pUniforms->staticUB.gpuAddress, pUniforms->staticUB.size );
    DEMOCommandBuffer.SetConstantBuffer( s_staticUniformLoc,
            nn::gfx::ShaderStage_Hull, pUniforms->staticUB.gpuAddress, pUniforms->staticUB.size );

    // Set Dynamic (per-frame) uniform blocks

    // Set vs dynamic uniform block (same location for streamout and tesselate shaders)
    DEMOCommandBuffer.SetConstantBuffer( s_dynamicUniformLoc,
        nn::gfx::ShaderStage_Domain, pUniforms->dynamicUB[ s_frameCount % 2 ].gpuAddress, pUniforms->dynamicUB[ s_frameCount % 2 ].size );
    DEMOCommandBuffer.SetConstantBuffer( s_dynamicUniformLoc,
        nn::gfx::ShaderStage_Hull, pUniforms->dynamicUB[ s_frameCount % 2 ].gpuAddress, pUniforms->dynamicUB[ s_frameCount % 2 ].size );
#endif

    // Set Attrib buffer
    DEMOCommandBuffer.SetVertexBuffer( 0,
        pAttribData->vertexBuffer.gpuAddress, pAttribData->vertexStride, pAttribData->vertexBuffer.size );

    // Setup the main state
    DEMOCommandBuffer.SetPipeline( &pPipeline[ 0 ].pipeline );

#if NN_GFX_IS_TARGET_GX
    // GX2 needs special treatment for tessellation shaders
    GX2SetFetchShader( &g_tessFetchShader );
#endif

    // Since we need the setup above, simply skip the outline rendering
    if ( s_drawOutline )
    {
#if NN_GFX_IS_TARGET_GX
        GX2DrawAdaptive(GX2_PRIMITIVE_TESSELLATE_TRIANGLES, OCT_NUM_VERTICES, pStreamOutBuffer->dataPtr, 0);
#else
        DEMOCommandBuffer.Draw( nn::gfx::PrimitiveTopology_PatchList, OCT_NUM_VERTICES, 0 );
#endif

    }

    // Reset the state for fill mode
    DEMOCommandBuffer.SetBlendState( &s_blendState );
    DEMOCommandBuffer.SetRasterizerState( &s_rasterizerState );

    // Draw with triangle
#if NN_GFX_IS_TARGET_GX
    GX2DrawAdaptive(GX2_PRIMITIVE_TESSELLATE_TRIANGLES, OCT_NUM_VERTICES, pStreamOutBuffer->dataPtr, 0);
#else
    DEMOCommandBuffer.Draw( nn::gfx::PrimitiveTopology_PatchList, OCT_NUM_VERTICES, 0 );
#endif

    DEMOFontSetSpacing( 0 );
    DEMOFontSetGridSize( 100, 33 );
    DEMOFontPrintf( 4, 2, "<Adaptive Tessellation>" );

    // This function will handle presenting the rendered buffer to the screen
    // by swapping the color buffer, setting the new context state,
    // resetting the color buffer, and flushing the pipeline.
    DEMOGfxDoneRender();

}

//extern "C" void nnMain()
TEST(GfxTessellateAdaptive, Run)
{
    int argc = nnt::GetHostArgc();
    char **argv = nnt::GetHostArgv();

#if NN_GFX_IS_TARGET_GX
    DEMOGfxPipeline demoPipeline[ 2 ];
#else
    DEMOGfxPipeline demoPipeline[ 1 ];
#endif

    // Uniforms for shader with Uniform Blocks
    Uniforms demoUniformUB;

    // Attrib buffers
    AttribBuffer demoOctaAttribData;

#if NN_GFX_IS_TARGET_GX
    // Streamout buffer
    GX2StreamOutBuffer demoStreamOutBuf = {0};
#endif

    // Initialize the DEMO library, DEMO test system, and create the
    // primary display.
    DEMOInit();
    DEMOTestInit(argc, argv);

    // This function will create the appropriate size color and depth
    // buffers, setup the scan buffer, and initialize the viewport and
    // scissor rectangles to be the entire screen.
    DEMOGfxInit(argc, argv);

    DEMOFontInit();

    // After the DEMO library is initialize the sample's initialization
    // function can be called.

    // Initialize attributes and buffers
    InitAttribData(&demoOctaAttribData,
                   sizeof(OCT_VERTEX_DATA), sizeof(DEMO_F32x3F32x3), OCT_VERTEX_DATA,
                   sizeof(OCT_MIDPOINT_DATA), sizeof(DEMO_F32x3), OCT_MIDPOINT_DATA);

    // Initialize shaders and uniform blocks
    if( DEMOTestIsUseHlslccGlsl() )
    {
        InitShaderUB(&demoPipeline[0], &demoOctaAttribData, &demoUniformUB, ShaderList[2]);
    }
    else
    {
        InitShaderUB(&demoPipeline[0], &demoOctaAttribData, &demoUniformUB, ShaderList[0]);
    }

#if NN_GFX_IS_TARGET_GX
    // Initialize streamout shader and buffer
    InitShaderSO(&demoPipeline[1], &demoOctaAttribData, &demoUniformUB, &demoStreamOutBuf, ShaderList[1]);
#endif

    // Set defaults
    s_frameCount=0;
    s_time = 0.0f;
    s_yrad = 0.0f;
    s_zvel = -2.005f;
    s_zpos = 200.0f;
    s_drawOutline = true;

    while (DEMOIsRunning())
    {
        // In this example, Double buffer for uniform buffer is necessary
        // because CPU always proceeds one frame that GPU processes.
        // CPU writes N+1 th frame while GPU reads N th frame data

        // Update Model Mtx
        // 1) CPU writes uniform buffer of N th frame
        UpdateUniforms(&demoUniformUB);

        // Draw
        // 2) CPU waits VI sync (DEMOGfxBeforeRender)
        // 3) GPU starts working for N th frame
        // 4) Copys framebuffer into scanbuffer and flip (DEMOGfxDoneRender)
        // 5) N += 1;
#if NN_GFX_IS_TARGET_GX
        DrawModel(demoPipeline, &demoOctaAttribData, &demoUniformUB, &demoStreamOutBuf);
#else
        DrawModel( demoPipeline, &demoOctaAttribData, &demoUniformUB );
#endif

        // Update frame counter to double buffer dynamic Uniforms
        s_frameCount++;
    }

    // Free shaders and uniform blocks and buffers
    FreeShaderUB(&demoPipeline[0], &demoUniformUB);
#if NN_GFX_IS_TARGET_GX
    FreeShaderSO(&demoPipeline[1], &demoStreamOutBuf);
#endif

    // Free attribute buffers
    FreeAttribData(&demoOctaAttribData);

    DEMOFontShutdown();
    DEMOTestShutdown();
    DEMOGfxShutdown();
    DEMOShutdown();

    SUCCEED();
}
