﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{GfxSimple.cpp,PageSampleGfxSimple}
 *
 * @brief
 *  シンプルなグラフィックス処理のサンプルプログラム
 */

/**
 * @page PageSampleGfxSimple GfxSimple
 * @tableofcontents
 *
 * @brief
 *  シンプルなグラフィックス処理のサンプルプログラムの解説です。
 *
 * @section PageSampleGfxSimple_SectionBrief 概要
 *  gfx を使用して、シェーダやテクスチャを使ったグラフィックス処理を行うシンプルなサンプルです。
 *
 * @section PageSampleGfxSimple_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/GfxSimple
 *  Samples/Sources/Applications/GfxSimple @endlink 以下にあります。
 *
 * @section PageSampleGfxSimple_SectionNecessaryEnvironment 必要な環境
 *  画面表示が利用可能である必要があります。
 *
 * @section PageSampleGfxSimple_SectionHowToOperate 操作方法
 *  特にありません。
 *
 * @section PageSampleGfxSimple_SectionPrecaution 注意事項
 *  特にありません。
 *
 * @section PageSampleGfxSimple_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleGfxSimple_SectionDetail 解説
 *  このサンプルプログラムは、シェーダリソースとテクスチャリソースを読み込み、
 *  単純なポリゴンをシェーダバリエーションによってアニメーションさせながらレンダリングします。
 *  利用するリソースはビルド時にビルド前イベントにからコンバートされています。
 *  リソースのコンバートにシェーダコンバータとテクスチャコンバータを利用しています。
 *
 * このサンプルプログラムの処理の流れは以下の通りです。
 *
 * - ディスプレイを取得・レイヤを初期化
 * - デバイスを初期化
 * - メモリプールを初期化
 * - スワップチェーンを初期化
 * - キューを初期化
 * - コマンドバッファを初期化
 * - ビューポートを初期化
 * - ラスタライザステートを初期化
 * - ブレンドステートを初期化
 * - 深度ステンシルステートを初期化
 * - 頂点ステートを初期化
 * - 頂点バッファを初期化
 * - インデクスバッファを初期化
 * - 定数バッファを初期化
 * - サンプラを初期化
 * - デスクリプタプールを初期化
 * - テクスチャリソースを初期化
 * - シェーダリソースを初期化
 * - ループ開始
 * - シェーダバリエーションを選択
 * - コマンドリストを作成
 * - コマンドリストを実行
 * - ディスプレイへプレゼンテーション
 * - ループ開始に戻る
 * - 各種オブジェクトを破棄
 */

#include <cstdlib>
#include <cstring>

#include <nn/nn_Assert.h>

#include <nn/os.h>
#include <nn/fs.h>
#include <nn/init.h>
#include <nn/vi.h>
#include <nn/gfx.h>
#include <nn/util/util_ScopeExit.h>

#if NN_GFX_IS_TARGET_NVN
    #include <nvn/nvn.h>
    #include <nvn/nvn_FuncPtrInline.h>
#endif

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
    #include <nv/nv_MemoryManagement.h>
#endif

/* For Capture */
#include <nnt.h>
#include <nnt/graphics/testGraphics_Path.h>
#include <nnt/graphics/testGraphics_GetHostExecutableFilepath.h>
#include <nnt/graphics/testGraphics_Png.h>
#include <nnt/graphics/testGraphics_CreateDirectories.h>
#include <nnt/graphics/testGraphics_InitializeAllocatorFunctionForStandardAllocator.h>

#include "testGfx_ColorTargetViewCapture.h"
/* For Capture */

namespace {

const float Vertices[] =
{
    0.0f, 0.5f, 0.0f, 0.0f, 0.0f,
    -0.5f, 0.0f, 0.0f, 0.0f, 1.0f,
    0.5f, 0.0f, 0.0f, 1.0f, 0.0f,
    0.0f, -0.5f, 0.0f, 1.0f, 1.0f
};

const int Indices[] =
{
    0, 1, 2, 3
};

int g_BufferDescriptorBaseIndex = 0;
int g_TextureDescriptorBaseIndex = 0;
int g_SamplerDescriptorBaseIndex = 0;

//-----------------------------------------------------------------------------
// メモリ

const size_t GraphicsSystemMemorySize = 8 * 1024 * 1024;
const int RenderWidth = 1280;
const int RenderHeight = 720;

nn::util::BytePtr g_pMemoryHeap( NULL );
nn::util::BytePtr g_pMemory( NULL );

enum MemoryPoolType
{
    MemoryPoolType_CpuUncached_GpuCached,
    MemoryPoolType_CpuInvisible_GpuCached_Compressible,

    MemoryPoolType_End
};

const size_t MemoryPoolSize[ MemoryPoolType_End ] =
{
    16 * 1024 * 1024,
    20 * 1024 * 1024
};

void* g_pPoolMemory[ MemoryPoolType_End ] = {};
ptrdiff_t g_MemoryPoolOffset[ MemoryPoolType_End ] = {};

size_t g_MaxScratchMemory = 0;
ptrdiff_t g_ScratchMemoryOffset = 0;

void* ReadResource( const char* filename )
{
    nn::Result result;
    nn::fs::FileHandle hFile;

    int64_t fileSize = 0;
    result = nn::fs::OpenFile( &hFile, filename, nn::fs::OpenMode_Read );
    NN_ASSERT( result.IsSuccess() );

    result = nn::fs::GetFileSize( &fileSize, hFile );
    NN_ASSERT( result.IsSuccess() );

    nn::util::BinaryFileHeader fileHeader;
    size_t readSize;
    result = nn::fs::ReadFile( &readSize, hFile, 0, &fileHeader, sizeof( nn::util::BinaryFileHeader ) );
    NN_ASSERT( result.IsSuccess() );
    NN_ASSERT( readSize == sizeof( nn::util::BinaryFileHeader ) );
    size_t alignment = fileHeader.GetAlignment();

    g_pMemory.AlignUp( alignment );
    void* pBuffer = g_pMemory.Get();
    result = nn::fs::ReadFile( &readSize, hFile, 0, pBuffer, static_cast< size_t >( fileSize ) );
    NN_ASSERT( result.IsSuccess() );
    NN_ASSERT( readSize == static_cast< size_t >( fileSize ) );
    g_pMemory.Advance( static_cast< ptrdiff_t >( fileSize ) );

    nn::fs::CloseFile( hFile );

    return pBuffer;
}

// レイヤを初期化
nn::vi::Display* g_pDisplay;
nn::vi::Layer* g_pLayer;
void InitializeLayer()
{
    nn::Result result = nn::vi::OpenDefaultDisplay( &g_pDisplay );
    NN_ASSERT( result.IsSuccess() );
    NN_UNUSED( result );

    result = nn::vi::CreateLayer( &g_pLayer, g_pDisplay );
    NN_ASSERT( result.IsSuccess() );

    result = nn::vi::SetLayerScalingMode( g_pLayer, nn::vi::ScalingMode_FitToLayer );
    NN_ASSERT( result.IsSuccess() );
}

// デバイスを初期化
nn::gfx::Device g_Device;
void InitializeDevice()
{
    nn::gfx::Device::InfoType info;
    info.SetDefault();
    info.SetApiVersion( nn::gfx::ApiMajorVersion, nn::gfx::ApiMinorVersion );
    g_Device.Initialize( info );
}

nn::gfx::MemoryPool g_MemoryPool[ MemoryPoolType_End ];
void InitializeMemoryPool()
{
    const int MemoryPoolProperty[ MemoryPoolType_End ] =
    {
        nn::gfx::MemoryPoolProperty_CpuUncached | nn::gfx::MemoryPoolProperty_GpuCached,
        nn::gfx::MemoryPoolProperty_CpuInvisible | nn::gfx::MemoryPoolProperty_GpuCached
            | nn::gfx::MemoryPoolProperty_Compressible
    };

    nn::gfx::MemoryPool::InfoType info;
    for( int idxMemoryPool = 0; idxMemoryPool < MemoryPoolType_End; ++idxMemoryPool )
    {
        info.SetDefault();
        info.SetMemoryPoolProperty( MemoryPoolProperty[ idxMemoryPool ] );
        size_t alignment = nn::gfx::MemoryPool::GetPoolMemoryAlignment( &g_Device, info );
        size_t granularity = nn::gfx::MemoryPool::GetPoolMemorySizeGranularity( &g_Device, info );
        g_pPoolMemory[ idxMemoryPool ] = malloc( MemoryPoolSize[ idxMemoryPool ] );
        void* pPoolMemoryAligned = nn::util::BytePtr(
            g_pPoolMemory[ idxMemoryPool ] ).AlignUp( alignment ).Get();
        size_t memoryPoolSizeAligned = nn::util::align_down( MemoryPoolSize[ idxMemoryPool ], granularity );
        info.SetPoolMemory( pPoolMemoryAligned, memoryPoolSizeAligned );
        g_MemoryPool[ idxMemoryPool ].Initialize( &g_Device, info );
        g_MemoryPoolOffset[ idxMemoryPool ] = 0;
    }
}

// スワップチェーンを初期化
nn::gfx::SwapChain g_SwapChain;
void InitializeSwapChain()
{
    nn::gfx::SwapChain::InfoType info;

    info.SetDefault();
    info.SetLayer( g_pLayer );
    info.SetWidth( RenderWidth );
    info.SetHeight( RenderHeight );
    info.SetFormat( nn::gfx::ImageFormat_R8_G8_B8_A8_UnormSrgb );
    info.SetBufferCount( 2 );
    if( NN_STATIC_CONDITION( nn::gfx::SwapChain::IsMemoryPoolRequired ) )
    {
        ptrdiff_t& offset = g_MemoryPoolOffset[ MemoryPoolType_CpuInvisible_GpuCached_Compressible ];
        size_t size = g_SwapChain.CalculateScanBufferSize( &g_Device, info );
        offset = nn::util::align_up( offset, nn::gfx::SwapChain::GetScanBufferAlignment( &g_Device, info ) );
        g_SwapChain.Initialize( &g_Device, info,
            &g_MemoryPool[ MemoryPoolType_CpuInvisible_GpuCached_Compressible ], offset, size );
        offset += size;
    }
    else
    {
        g_SwapChain.Initialize( &g_Device, info, NULL, 0, 0 );
    }
}

// キューを初期化
nn::gfx::Queue g_Queue;
void InitializeQueue()
{
    nn::gfx::Queue::InfoType info;
    info.SetDefault();
    info.SetCapability( nn::gfx::QueueCapability_Graphics );
    g_Queue.Initialize( &g_Device, info );
}

// コマンドバッファを初期化
nn::gfx::CommandBuffer g_CommandBuffer;
void InitializeCommandBuffer()
{
    nn::gfx::CommandBuffer::InfoType info;
    info.SetDefault();
    info.SetQueueCapability( nn::gfx::QueueCapability_Graphics );
    info.SetCommandBufferType( nn::gfx::CommandBufferType_Direct );
    g_CommandBuffer.Initialize( &g_Device, info );
}

// ビューポートシザーを初期化
nn::gfx::ViewportScissorState g_ViewportScissor;
void InitializeViewport()
{
    nn::gfx::ViewportScissorState::InfoType info;
    info.SetDefault();
    info.SetScissorEnabled( true );
    nn::gfx::ViewportStateInfo viewportInfo;
    {
        viewportInfo.SetDefault();
        viewportInfo.SetWidth( static_cast< float >( RenderWidth ) );
        viewportInfo.SetHeight( static_cast< float >( RenderHeight ) );
    }
    nn::gfx::ScissorStateInfo scissorInfo;
    {
        scissorInfo.SetDefault();
        scissorInfo.SetWidth( RenderWidth );
        scissorInfo.SetHeight( RenderHeight );
    }
    info.SetViewportStateInfoArray( &viewportInfo, 1 );
    info.SetScissorStateInfoArray( &scissorInfo, 1 );
    g_ViewportScissor.Initialize( &g_Device, info );
}

// Initialize the rasterizer state
nn::gfx::RasterizerState g_RasterizerState;
static void InitializeRasterizerState()
{
    nn::gfx::RasterizerState::InfoType info;
    info.SetDefault();
    info.SetCullMode( nn::gfx::CullMode_None );
    info.SetPrimitiveTopologyType( nn::gfx::PrimitiveTopologyType_Triangle );
    info.SetScissorEnabled( true );
    info.SetDepthClipEnabled( false );
    g_RasterizerState.Initialize( &g_Device, info );
}

// Initialize the blend state
nn::gfx::BlendState g_BlendState;
static void InitializeBlendState()
{
    nn::gfx::BlendState::InfoType info;
    info.SetDefault();
    nn::gfx::BlendTargetStateInfo targetInfo;
    {
        targetInfo.SetDefault();
    };
    info.SetBlendTargetStateInfoArray( &targetInfo, 1 );
    size_t size = nn::gfx::BlendState::GetRequiredMemorySize( info );
    g_pMemory.AlignUp( nn::gfx::BlendState::RequiredMemoryInfo_Alignment );
    g_BlendState.SetMemory( g_pMemory.Get(), size );
    g_pMemory.Advance( size );
    g_BlendState.Initialize( &g_Device, info );
}

// Initialize the depth stencil state
nn::gfx::DepthStencilState g_DepthStencilState;
static void InitializeDepthStencilState()
{
    nn::gfx::DepthStencilState::InfoType info;
    info.SetDefault();
    info.SetDepthTestEnabled( false );
    info.SetDepthWriteEnabled( false );
    g_DepthStencilState.Initialize( &g_Device, info );
}

// Initialize the vertex state
nn::gfx::VertexState g_VertexState;
static void InitializeVertexState()
{
    nn::gfx::VertexState::InfoType info;
    info.SetDefault();
    uint16_t stride = sizeof( float ) * 5;
    nn::gfx::VertexAttributeStateInfo attribs[ 2 ];
    {
        attribs[ 0 ].SetDefault();
        attribs[ 0 ].SetBufferIndex( 0 );
        attribs[ 0 ].SetFormat( nn::gfx::AttributeFormat_32_32_32_Float );
        attribs[ 0 ].SetOffset( 0 );
        attribs[ 0 ].SetShaderSlot( 0 );
    }
    {
        attribs[ 1 ].SetDefault();
        attribs[ 1 ].SetBufferIndex( 0 );
        attribs[ 1 ].SetFormat( nn::gfx::AttributeFormat_32_32_Float );
        attribs[ 1 ].SetOffset( sizeof( float ) * 3 );
        attribs[ 1 ].SetShaderSlot( 1 );
    }
    nn::gfx::VertexBufferStateInfo buffer;
    {
        buffer.SetDefault();
        buffer.SetStride( stride );
    }
    info.SetVertexAttributeStateInfoArray( attribs, 2 );
    info.SetVertexBufferStateInfoArray( &buffer, 1 );
    size_t size = nn::gfx::VertexState::GetRequiredMemorySize( info );
    g_pMemory.AlignUp( nn::gfx::VertexState::RequiredMemoryInfo_Alignment );
    g_VertexState.SetMemory( g_pMemory.Get(), size );
    g_pMemory.Advance( size );
    g_VertexState.Initialize( &g_Device, info, NULL );
}

// 頂点用のバッファを初期化
nn::gfx::Buffer g_VertexBuffer;
void InitializeVertexBuffer()
{
    nn::gfx::Buffer::InfoType info;
    info.SetDefault();
    info.SetSize( sizeof( Vertices ) );
    info.SetGpuAccessFlags( nn::gfx::GpuAccess_VertexBuffer );
    if( NN_STATIC_CONDITION( nn::gfx::Buffer::IsMemoryPoolRequired ) )
    {
        ptrdiff_t& offset = g_MemoryPoolOffset[ MemoryPoolType_CpuUncached_GpuCached ];
        offset = nn::util::align_up( offset, nn::gfx::Buffer::GetBufferAlignment( &g_Device, info ) );
        g_VertexBuffer.Initialize( &g_Device, info,
            &g_MemoryPool[ MemoryPoolType_CpuUncached_GpuCached ], offset, info.GetSize() );
        offset += info.GetSize();
    }
    else
    {
        g_VertexBuffer.Initialize( &g_Device, info, NULL, 0, 0 );
    }

    void* pMapped = g_VertexBuffer.Map();
    memcpy( pMapped, Vertices, info.GetSize() );
    g_VertexBuffer.Unmap();
}

// インデクス用のバッファを初期化
nn::gfx::Buffer g_IndexBuffer;
void InitializeIndexBuffer()
{
    nn::gfx::Buffer::InfoType info;
    info.SetDefault();
    info.SetSize( sizeof( Indices ) );
    info.SetGpuAccessFlags( nn::gfx::GpuAccess_IndexBuffer );
    if( NN_STATIC_CONDITION( nn::gfx::Buffer::IsMemoryPoolRequired ) )
    {
        ptrdiff_t& offset = g_MemoryPoolOffset[ MemoryPoolType_CpuUncached_GpuCached ];
        offset = nn::util::align_up( offset, nn::gfx::Buffer::GetBufferAlignment( &g_Device, info ) );
        g_IndexBuffer.Initialize( &g_Device, info,
            &g_MemoryPool[ MemoryPoolType_CpuUncached_GpuCached ], offset, info.GetSize() );
        offset += info.GetSize();
    }
    else
    {
        g_IndexBuffer.Initialize( &g_Device, info, NULL, 0, 0 );
    }

    void* pMapped = g_IndexBuffer.Map();
    memcpy( pMapped, Indices, info.GetSize() );
    g_IndexBuffer.Unmap();
}

// 定数バッファ用のバッファを初期化
nn::gfx::Buffer g_ConstantBuffer;
void InitializeConstantBuffer()
{
    nn::gfx::Buffer::InfoType info;
    info.SetDefault();
    info.SetSize( sizeof( float ) * 4 );
    info.SetGpuAccessFlags( nn::gfx::GpuAccess_ConstantBuffer );
    if( NN_STATIC_CONDITION( nn::gfx::Buffer::IsMemoryPoolRequired ) )
    {
        ptrdiff_t& offset = g_MemoryPoolOffset[ MemoryPoolType_CpuUncached_GpuCached ];
        offset = nn::util::align_up( offset, nn::gfx::Buffer::GetBufferAlignment( &g_Device, info ) );
        g_ConstantBuffer.Initialize( &g_Device, info,
            &g_MemoryPool[ MemoryPoolType_CpuUncached_GpuCached ], offset, info.GetSize() );
        offset += info.GetSize();
    }
    else
    {
        g_ConstantBuffer.Initialize( &g_Device, info, NULL, 0, 0 );
    }
}

// サンプラを初期化
nn::gfx::Sampler g_Sampler;
void InitializeSampler()
{
    nn::gfx::Sampler::InfoType info;
    info.SetDefault();
    info.SetFilterMode( nn::gfx::FilterMode_MinLinear_MagLinear_MipPoint );
    info.SetAddressU( nn::gfx::TextureAddressMode_Mirror );
    info.SetAddressV( nn::gfx::TextureAddressMode_Mirror );
    info.SetAddressW( nn::gfx::TextureAddressMode_Mirror );
    g_Sampler.Initialize( &g_Device, info );
}

nn::gfx::DescriptorPool g_BufferDescriptorPool;
void InitializeBufferDescriptorPool()
{
    nn::gfx::DescriptorPool::InfoType info;
    info.SetDefault();
    info.SetDescriptorPoolType( nn::gfx::DescriptorPoolType_BufferView );
    info.SetSlotCount( g_BufferDescriptorBaseIndex + 1 );
    size_t size = nn::gfx::DescriptorPool::CalculateDescriptorPoolSize( &g_Device, info );
    ptrdiff_t& offset = g_MemoryPoolOffset[ MemoryPoolType_CpuUncached_GpuCached ];
    offset = nn::util::align_up( offset, nn::gfx::DescriptorPool::GetDescriptorPoolAlignment( &g_Device, info ) );
    g_BufferDescriptorPool.Initialize( &g_Device, info,
        &g_MemoryPool[ MemoryPoolType_CpuUncached_GpuCached ], offset, size );
    offset += size;
}

nn::gfx::DescriptorPool g_SamplerDescriptorPool;
void InitializeSamplerDescriptorPool()
{
    nn::gfx::DescriptorPool::InfoType info;
    info.SetDefault();
    info.SetDescriptorPoolType( nn::gfx::DescriptorPoolType_Sampler );
    info.SetSlotCount( g_SamplerDescriptorBaseIndex + 1 );
    size_t size = nn::gfx::DescriptorPool::CalculateDescriptorPoolSize( &g_Device, info );
    ptrdiff_t& offset = g_MemoryPoolOffset[ MemoryPoolType_CpuUncached_GpuCached ];
    offset = nn::util::align_up( offset, nn::gfx::DescriptorPool::GetDescriptorPoolAlignment( &g_Device, info ) );
    g_SamplerDescriptorPool.Initialize( &g_Device, info,
        &g_MemoryPool[ MemoryPoolType_CpuUncached_GpuCached ], offset, size );
    offset += size;
}

nn::gfx::DescriptorPool g_TextureDescriptorPool;
void InitializeTextureDescriptorPool()
{
    nn::gfx::DescriptorPool::InfoType info;
    info.SetDefault();
    info.SetDescriptorPoolType( nn::gfx::DescriptorPoolType_TextureView );
    info.SetSlotCount( g_TextureDescriptorBaseIndex + 1 );
    size_t size = nn::gfx::DescriptorPool::CalculateDescriptorPoolSize( &g_Device, info );
    ptrdiff_t& offset = g_MemoryPoolOffset[ MemoryPoolType_CpuUncached_GpuCached ];
    offset = nn::util::align_up( offset, nn::gfx::DescriptorPool::GetDescriptorPoolAlignment( &g_Device, info ) );
    g_TextureDescriptorPool.Initialize( &g_Device, info,
        &g_MemoryPool[ MemoryPoolType_CpuUncached_GpuCached ], offset, size );
    offset += size;
}

nn::gfx::ResTextureFile* g_pResTextureFile;
void InitializeResTextureFile()
{
    void *pResource = ReadResource( "Contents:/SampleTexture.bntx" );
    g_pResTextureFile = nn::gfx::ResTextureFile::ResCast( pResource );
    g_pResTextureFile->Initialize( &g_Device );
    for( int idxTexture = 0, textureCount = g_pResTextureFile->GetTextureDic()->GetCount();
        idxTexture < textureCount; ++idxTexture )
    {
        g_pResTextureFile->GetResTexture( idxTexture )->Initialize( &g_Device );
    }
}

void FinalizeResTextureFile()
{
    for( int idxTexture = 0, textureCount = g_pResTextureFile->GetTextureDic()->GetCount();
        idxTexture < textureCount; ++idxTexture )
    {
        g_pResTextureFile->GetResTexture( idxTexture )->Finalize( &g_Device );
    }
    g_pResTextureFile->Finalize( &g_Device );
}

nn::gfx::ResShaderFile* g_pResShaderFile;
void InitializeResShaderFile()
{
    void* pResource = ReadResource( "Contents:/SampleShader.bnsh" );
    g_pResShaderFile = nn::gfx::ResShaderFile::ResCast( pResource );
    nn::gfx::ResShaderContainer* pContainer = g_pResShaderFile->GetShaderContainer();
    NN_ASSERT_NOT_NULL( pContainer );
    pContainer->Initialize( &g_Device );

    for( int idxVariation = 0, variationCount = pContainer->GetShaderVariationCount();
        idxVariation < variationCount; ++idxVariation )
    {
        nn::gfx::ResShaderProgram* pProgram = pContainer->GetResShaderProgram( idxVariation );
        nn::gfx::ShaderInitializeResult result = pProgram->Initialize( &g_Device );
        NN_ASSERT_EQUAL( result, nn::gfx::ShaderInitializeResult_Success );
        NN_UNUSED( result );
    }

#if NN_GFX_IS_TARGET_NVN
    g_MaxScratchMemory = nn::gfx::NvnGetMaxRecommendedScratchMemorySize( &g_Device, &g_pResShaderFile, 1 );
    int scratchMemoryAlignment;
    nvnDeviceGetInteger( g_Device.ToData()->pNvnDevice,
        NVN_DEVICE_INFO_SHADER_SCRATCH_MEMORY_ALIGNMENT, &scratchMemoryAlignment );
    ptrdiff_t& offset = g_MemoryPoolOffset[ MemoryPoolType_CpuInvisible_GpuCached_Compressible ];
    offset = nn::util::align_up( offset, scratchMemoryAlignment );
    g_ScratchMemoryOffset = offset;
    offset += g_MaxScratchMemory;
#endif
    NN_UNUSED( g_MaxScratchMemory );
    NN_UNUSED( g_ScratchMemoryOffset );
}

void FinalizeResShaderFile()
{
    auto pContainer = g_pResShaderFile->GetShaderContainer();
    NN_ASSERT_NOT_NULL( pContainer );

    for( int idxVariation = 0, variationCount = pContainer->GetShaderVariationCount();
        idxVariation < variationCount; ++idxVariation )
    {
        nn::gfx::ResShaderProgram* pProgram = pContainer->GetResShaderProgram( idxVariation );
        pProgram->Finalize( &g_Device );
    }

    pContainer->Finalize( &g_Device );
}

/* For Capture */
nn::mem::StandardAllocator g_TestGraphicsStandardAllocator;
nnt::graphics::AllocatorFunction g_TestGraphicsAllocatorFunction;
nnt::gfx::ColorTargetViewCapture g_ColorTargetViewCapture;
void InitializeFrameCapture()
{
    nn::Result result;
    NN_UNUSED( result );

    // FS の初期化
    result = nn::fs::MountHostRoot();
    NN_ASSERT( result.IsSuccess() );

    // 出力ディレクトリ
    nnt::graphics::Path outputsPath(nnt::graphics::GetHostExecutableFilepath());
    outputsPath.CombineAssign("../Outputs");
    outputsPath.NormalizeAssign();

    result = nnt::graphics::CreateDirectories(outputsPath);
    NN_ASSERT(result.IsSuccess());
    result = nn::fs::MountHost("Outputs", outputsPath.GetString());
    NN_ASSERT(result.IsSuccess());

    // キャプチャ機能の初期化
    static const size_t HeapSize = 16 * 1024 * 1024;
    g_TestGraphicsStandardAllocator.Initialize( malloc(HeapSize), HeapSize );
    nnt::graphics::InitializeAllocatorFunctionForStandardAllocator::Initialize(
        &g_TestGraphicsAllocatorFunction,
        &g_TestGraphicsStandardAllocator);

    ptrdiff_t newMemoryPoolOffset = 0;
    ptrdiff_t newInvisibleMemoryPoolOffset = 0;
    g_ColorTargetViewCapture.Initialize(
        &newMemoryPoolOffset,
        &newInvisibleMemoryPoolOffset,
        &g_Device,
        &g_MemoryPool[ MemoryPoolType_CpuUncached_GpuCached ],
        &g_MemoryPool[ MemoryPoolType_CpuInvisible_GpuCached_Compressible ],
        g_MemoryPoolOffset[ MemoryPoolType_CpuUncached_GpuCached ],
        g_MemoryPoolOffset[ MemoryPoolType_CpuInvisible_GpuCached_Compressible ],
        &g_TestGraphicsAllocatorFunction,
        RenderWidth,
        RenderHeight);
    g_MemoryPoolOffset[ MemoryPoolType_CpuUncached_GpuCached ] = newMemoryPoolOffset;
    g_MemoryPoolOffset[ MemoryPoolType_CpuInvisible_GpuCached_Compressible ] = newInvisibleMemoryPoolOffset;
}

void FinalizeFrameCapture()
{
    g_ColorTargetViewCapture.Finalize();
    g_TestGraphicsStandardAllocator.Finalize();

    nn::fs::Unmount("Outputs");
    nn::fs::UnmountHostRoot();
}
/* For Capture */

void InitializeGfxObjects()
{
    g_pMemoryHeap.Reset( malloc( 1024 * 1024 ) );
    g_pMemory = g_pMemoryHeap;

    InitializeDevice();

#if NN_GFX_IS_TARGET_NVN
    nn::gfx::Device::DataType& deviceData = nn::gfx::AccessorToData( g_Device );
    nvnDeviceGetInteger( deviceData.pNvnDevice,
        NVN_DEVICE_INFO_RESERVED_TEXTURE_DESCRIPTORS, &g_TextureDescriptorBaseIndex );
    nvnDeviceGetInteger( deviceData.pNvnDevice,
        NVN_DEVICE_INFO_RESERVED_SAMPLER_DESCRIPTORS, &g_SamplerDescriptorBaseIndex );
#endif

    InitializeMemoryPool();

    InitializeSwapChain();
    InitializeQueue();

    InitializeCommandBuffer();
    InitializeViewport();

    InitializeRasterizerState();
    InitializeBlendState();
    InitializeDepthStencilState();
    InitializeVertexState();

    InitializeVertexBuffer();
    InitializeIndexBuffer();
    InitializeConstantBuffer();
    InitializeSampler();

    InitializeBufferDescriptorPool();
    InitializeSamplerDescriptorPool();
    InitializeTextureDescriptorPool();

    InitializeResTextureFile();
    InitializeResShaderFile();

    NN_ASSERT( g_pMemoryHeap.Distance( g_pMemory.Get() ) < 1024 * 1024 );
    for( int idxMemoryPool = 0; idxMemoryPool < MemoryPoolType_End; ++idxMemoryPool )
    {
        NN_ASSERT( static_cast< size_t >( g_MemoryPoolOffset[ idxMemoryPool ] )
            < MemoryPoolSize[ idxMemoryPool ] );
    }
}

void MakeCommand( int frame )
{
    nn::gfx::GpuAddress gpuAddress;

    g_CommandBuffer.Reset();
    ptrdiff_t& offset = g_MemoryPoolOffset[ MemoryPoolType_CpuUncached_GpuCached ];
    offset = nn::util::align_up( offset, nn::gfx::CommandBuffer::GetCommandMemoryAlignment( &g_Device ) );
    // ワンタイムのコマンドバッファには nn::gfx::MemoryPoolProperty_CpuUncached | nn::gfx::MemoryPoolProperty_GpuUncached のメモリプールを使うことがより適しています
    g_CommandBuffer.AddCommandMemory(
        &g_MemoryPool[ MemoryPoolType_CpuUncached_GpuCached ], offset, 1024 * 1024 );
    g_pMemory.AlignUp( 256 );
    g_CommandBuffer.AddControlMemory( g_pMemory.Get(), 256 );
    g_CommandBuffer.Begin();
    {
        nn::gfx::ColorTargetView* pTarget = g_SwapChain.AcquireNextScanBufferView();

        g_VertexBuffer.GetGpuAddress( &gpuAddress );
        g_CommandBuffer.SetVertexBuffer( 0, gpuAddress, sizeof( float ) * 5, sizeof( Vertices ) );

        g_CommandBuffer.InvalidateMemory( nn::gfx::GpuAccess_ShaderCode | nn::gfx::GpuAccess_Descriptor );

        g_CommandBuffer.SetDescriptorPool( &g_BufferDescriptorPool );
        g_CommandBuffer.SetDescriptorPool( &g_TextureDescriptorPool );
        g_CommandBuffer.SetDescriptorPool( &g_SamplerDescriptorPool );

        g_CommandBuffer.ClearColor( pTarget, 0.3f, 0.1f, 0.1f, 1.0f, NULL );

#if NN_GFX_IS_TARGET_NVN
        nvnCommandBufferSetShaderScratchMemory( g_CommandBuffer.ToData()->pNvnCommandBuffer,
            g_MemoryPool[ MemoryPoolType_CpuInvisible_GpuCached_Compressible ].ToData()->pNvnMemoryPool,
            g_ScratchMemoryOffset, g_MaxScratchMemory );
#endif

        nn::gfx::ResShaderContainer* pContainer = g_pResShaderFile->GetShaderContainer();
        int variation = frame / 32 % pContainer->GetShaderVariationCount();
        nn::gfx::ResShaderProgram* pProgram = pContainer->GetResShaderProgram( variation );
        nn::gfx::Shader* pShader = pProgram->GetShader();
        g_CommandBuffer.SetShader( pShader, nn::gfx::ShaderStageBit_All );

        g_CommandBuffer.SetRenderTargets( 1, &pTarget, NULL );

        g_CommandBuffer.SetViewportScissorState( &g_ViewportScissor );

        g_CommandBuffer.SetRasterizerState( &g_RasterizerState );
        g_CommandBuffer.SetBlendState( &g_BlendState );
        g_CommandBuffer.SetDepthStencilState( &g_DepthStencilState );
        g_CommandBuffer.SetVertexState( &g_VertexState );

        g_CommandBuffer.InvalidateMemory( nn::gfx::GpuAccess_Texture | nn::gfx::GpuAccess_IndexBuffer
            | nn::gfx::GpuAccess_ConstantBuffer | nn::gfx::GpuAccess_VertexBuffer );

        nn::gfx::DescriptorSlot constantBufferDescriptor;
        nn::gfx::DescriptorSlot textureDescriptor;
        nn::gfx::DescriptorSlot samplerDescriptor;
        g_BufferDescriptorPool.GetDescriptorSlot( &constantBufferDescriptor, g_BufferDescriptorBaseIndex );
        g_TextureDescriptorPool.GetDescriptorSlot( &textureDescriptor, g_TextureDescriptorBaseIndex );
        g_SamplerDescriptorPool.GetDescriptorSlot( &samplerDescriptor, g_SamplerDescriptorBaseIndex );

        int slotMat = pShader->GetInterfaceSlot( nn::gfx::ShaderStage_Pixel,
            nn::gfx::ShaderInterfaceType_ConstantBuffer, "Mat" );
        int slotTex = pShader->GetInterfaceSlot( nn::gfx::ShaderStage_Pixel,
            nn::gfx::ShaderInterfaceType_Sampler, "tex" );
        if( slotMat >= 0 )
        {
            g_CommandBuffer.SetConstantBuffer( slotMat,
                nn::gfx::ShaderStage_Pixel, constantBufferDescriptor );
        }
        if( slotTex >= 0 )
        {
            g_CommandBuffer.SetTextureAndSampler( slotTex,
                nn::gfx::ShaderStage_Pixel, textureDescriptor, samplerDescriptor );
        }

        g_IndexBuffer.GetGpuAddress( &gpuAddress );
        g_CommandBuffer.DrawIndexed( nn::gfx::PrimitiveTopology_TriangleStrip,
            nn::gfx::IndexFormat_Uint32, gpuAddress, 4, 0 );

        /* For Capture */
        g_ColorTargetViewCapture.PushCaptureCommand(&g_CommandBuffer, pTarget);
        /* For Capture */
    }
    g_CommandBuffer.End();
}

void FinalizeGfxObjects()
{
    // 各オブジェクトを破棄
    FinalizeResShaderFile();
    FinalizeResTextureFile();

    g_Sampler.Finalize( &g_Device );
    g_ConstantBuffer.Finalize( &g_Device );
    g_IndexBuffer.Finalize( &g_Device );
    g_VertexBuffer.Finalize( &g_Device );

    g_RasterizerState.Finalize( &g_Device );
    g_BlendState.Finalize( &g_Device );
    g_DepthStencilState.Finalize( &g_Device );
    g_VertexState.Finalize( &g_Device );

    g_BufferDescriptorPool.Finalize( &g_Device );
    g_TextureDescriptorPool.Finalize( &g_Device );
    g_SamplerDescriptorPool.Finalize( &g_Device );

    g_ViewportScissor.Finalize( &g_Device );
    g_CommandBuffer.Finalize( &g_Device );
    g_SwapChain.Finalize( &g_Device );
    g_Queue.Finalize( &g_Device );
    for( int idxMemoryPool = 0; idxMemoryPool < MemoryPoolType_End; ++idxMemoryPool )
    {
        g_MemoryPool[ idxMemoryPool ].Finalize( &g_Device );
        free( g_pPoolMemory[ idxMemoryPool ] );
    }
    g_Device.Finalize();

    free( g_pMemoryHeap.Get() );
}

void* Allocate( size_t size )
{
    return malloc( size );
}

void Deallocate( void* ptr, size_t )
{
    free( ptr );
}

#if  defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
static void* NvAllocateFunction(size_t size, size_t alignment, void* userPtr)
{
    NN_UNUSED(userPtr);
    return aligned_alloc(alignment, size);
}

static void NvFreeFunction(void* addr, void* userPtr)
{
    NN_UNUSED(userPtr);
    free(addr);
}

static void* NvReallocateFunction(void* addr, size_t newSize, void* userPtr)
{
    NN_UNUSED(userPtr);
    return realloc(addr, newSize);
}
#endif
}

//
//  Main Function
//  メイン関数です。
//

//extern "C" void nnMain()
TEST(GfxSimple, Run)
{
    nn::Result result;

    nn::fs::SetAllocator( Allocate, Deallocate );

    size_t mountRomCacheBufferSize = 0;
    result = nn::fs::QueryMountRomCacheSize( &mountRomCacheBufferSize );
    void* mountRomCacheBuffer = Allocate( mountRomCacheBufferSize );
    NN_ASSERT_NOT_NULL( mountRomCacheBuffer );
    NN_UTIL_SCOPE_EXIT
    {
        Deallocate( mountRomCacheBuffer, mountRomCacheBufferSize );
    };
    result = nn::fs::MountRom( "Contents", mountRomCacheBuffer, mountRomCacheBufferSize );
    NN_ASSERT( result.IsSuccess() );

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
    // グラフィックスシステムのためのメモリ周りの初期化を行います。
    {
        nv::SetGraphicsAllocator(NvAllocateFunction, NvFreeFunction, NvReallocateFunction, NULL);
        nv::SetGraphicsDevtoolsAllocator(NvAllocateFunction, NvFreeFunction, NvReallocateFunction, NULL);
        nv::InitializeGraphics(malloc(GraphicsSystemMemorySize), GraphicsSystemMemorySize);
    }
#endif

    nn::vi::Initialize();
    InitializeLayer();

    nn::gfx::Initialize();
    InitializeGfxObjects();
    InitializeFrameCapture();

    g_BufferDescriptorPool.BeginUpdate();
    {
        nn::gfx::GpuAddress gpuAddress;
        g_ConstantBuffer.GetGpuAddress( &gpuAddress );
        g_BufferDescriptorPool.SetBufferView( g_BufferDescriptorBaseIndex, gpuAddress, sizeof( float ) * 4 );
    }
    g_BufferDescriptorPool.EndUpdate();

    g_TextureDescriptorPool.BeginUpdate();
    {
        g_TextureDescriptorPool.SetTextureView( g_TextureDescriptorBaseIndex,
            g_pResTextureFile->GetResTexture( 0 )->GetTextureView() );
    }
    g_TextureDescriptorPool.EndUpdate();

    g_SamplerDescriptorPool.BeginUpdate();
    {
        g_SamplerDescriptorPool.SetSampler( g_SamplerDescriptorBaseIndex, &g_Sampler );
    }
    g_SamplerDescriptorPool.EndUpdate();

    /* For Capture */
    // 32 frame ごとにリソースが切り替わるため、32 frame 単位でインクリメントするよう変更
    /* For Capture */
    // 毎フレームのレンダリング
    for( int frame = 0; frame < 60 * 10; frame += 32 )
    {
        MakeCommand( frame );

        // 定数バッファを更新
        float* pConstantBuffer = g_ConstantBuffer.Map< float >();
        {
            pConstantBuffer[ 0 ] = ( frame & 0xFF ) * ( 1.0f / 256.0f );
            pConstantBuffer[ 1 ] = 0.4f;
            pConstantBuffer[ 2 ] = 0.4f;
            pConstantBuffer[ 3 ] = 1.0f;
        }
#ifdef CAFE
        GX2EndianSwap( pConstantBuffer, sizeof( float ) * 4 );
#endif
        g_ConstantBuffer.Unmap();

        // コマンドの実行
        g_Queue.ExecuteCommand( &g_CommandBuffer, NULL );

        // 結果の表示
        g_Queue.Present( &g_SwapChain, 1 );
        g_Queue.Sync();

        /* For Capture */
        // キャプチャの保存
        if(g_ColorTargetViewCapture.IsCapturing())
        {
            g_ColorTargetViewCapture.FetchCaptureResult();
            const uint8_t* imgData = g_ColorTargetViewCapture.GetCapturedImageData();
            size_t imgDataSize = g_ColorTargetViewCapture.GetCapturedImageDataSize();
            nnt::graphics::PngIhdr ihdr;
            ihdr.SetDefault();
            ihdr.bitDepth = 8;
            ihdr.channels = 4;
            ihdr.colorType = nnt::graphics::PngColorType_RgbAlpha;
            ihdr.height = RenderHeight;
            ihdr.width = RenderWidth;
            nnt::graphics::PngIO::Write(
                nnt::graphics::Path("Outputs:/Frame_%d.png", frame).GetString(),
                imgData,
                imgDataSize,
                &ihdr,
                &g_TestGraphicsAllocatorFunction);
        }
        /* For Capture */
    }

    FinalizeFrameCapture();
    FinalizeGfxObjects();
    nn::gfx::Finalize();

    nn::vi::DestroyLayer( g_pLayer );
    nn::vi::CloseDisplay( g_pDisplay );
    nn::vi::Finalize();

    SUCCEED();
}
