﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <cstring>
#include <cmath>

#include <gfx/demo.h>

#if NN_GFX_IS_TARGET_GX
#include <cafe/pads/wpad/wpad.h>
#include <cafe/gx2ut.h> // Need for GX2UTExpandDepth
#endif

#include <nnt.h>
#include <nnt/nnt_Argument.h>

#define PI 3.14159f

// ----- Surface Information

#define RENDERTARGET_WIDTH (DEMOColorBufferInfo.GetWidth())
#define RENDERTARGET_HEIGHT (DEMOColorBufferInfo.GetHeight())

#define SURFACE_WIDTH RENDERTARGET_WIDTH
#define SURFACE_HEIGHT RENDERTARGET_HEIGHT

static const int SHADOWMAP_WIDTH = 1024;
static const int SHADOWMAP_HEIGHT = 1024;

//---------------------------------------------------------------------------
//  Texture Data
//---------------------------------------------------------------------------
static const char * const TEXTURE_FILE_PENCIL = "textures/pencilRendering/pencil";

static const int NUM_PENCIL_TEXTURES = 8;

static struct
{
    nn::gfx::Texture texture;
    nn::gfx::TextureInfo pencilTextureInfo;
    nn::gfx::TextureView view;
    nn::gfx::ColorTargetView colorView;
    nn::gfx::DescriptorSlot slot;
    DEMOGfxMemPool* pPool;
} pencilTexture[NUM_PENCIL_TEXTURES];

static const int NUM_PENCIL_VERTICES = 12;
static const int NUM_PENCIL_TEXCOORDS = 8;
static const int NUM_PENCIL_INDICES = 6;
static const int PENCIL_TEXTURE_WIDTH = 512;
static const int PENCIL_TEXTURE_SUBHEIGHT = 512;
static const int PENCIL_TEXTURE_HEIGHT = 8 * PENCIL_TEXTURE_SUBHEIGHT;
static const int PENCIL_TEXTURE_DIVISIONS = (PENCIL_TEXTURE_HEIGHT / PENCIL_TEXTURE_SUBHEIGHT);

//---------------------------------------------------------------------------
//  Model Data
//---------------------------------------------------------------------------
static const int NUM_MODEL = 4;
static const char* MODEL_FILES[NUM_MODEL][3]
    = {{"geometries/elephantPosition.dat",
       "geometries/elephantNormal.dat",
       "geometries/elephantIdx.dat"},
       {"geometries/teapotPosition.dat",
       "geometries/teapotNormal.dat",
       "geometries/teapotIdx.dat"},
       {"geometries/torusPosition.dat",
       "geometries/torusNormal.dat",
       "geometries/torusIdx.dat"},
       {"geometries/spherePosition.dat",
       "geometries/sphereNormal.dat",
       "geometries/sphereIdx.dat"}};

static u32 modelLen[NUM_MODEL][3];

static u32 MODEL_VTX_COUNT[NUM_MODEL] = {41142, 119400, 8640, 2880};

static const int MODEL_VTX_STRIDE = sizeof(f32) * 3;
static const int CONTOUR_SEGMENTS = 64;
static const int CONTOUR_VERTICES = (CONTOUR_SEGMENTS + 1);

// ----- Shader information

static const char * const GSH_SHADER_FILE_PENCIL_SHADOW[] =
{
    "shaders/pencilRenderingShadow",
    "shaders/pencilRenderingShadowHlslcc"
};
static const char * const GSH_SHADER_FILE_PENCIL_EXP[] =
{
    "shaders/pencilRenderingExp",
    "shaders/pencilRenderingExpHlslcc",
};
static const char * const GSH_SHADER_FILE_PENCIL_BLUR_H[] =
{
    "shaders/pencilRenderingBlurH",
    "shaders/pencilRenderingBlurHHlslcc",
};
static const char * const GSH_SHADER_FILE_PENCIL_BLUR_V[] =
{
    "shaders/pencilRenderingBlurV",
    "shaders/pencilRenderingBlurVHlslcc",
};
static const char * const GSH_SHADER_FILE_PENCIL_CREATION[] =
{
    "shaders/pencilRenderingCreation",
    "shaders/pencilRenderingCreationHlslcc",
};
static const char * const GSH_SHADER_FILE_PENCIL_GEOMETRY[] =
{
    "shaders/pencilRenderingSetup",
    "shaders/pencilRenderingSetupHlslcc",
};
static const char * const GSH_SHADER_FILE_PENCIL_OUTLINE[] =
{
    "shaders/pencilRenderingOutline",
    "shaders/pencilRenderingOutlineHlslcc",
};
static const char * const GSH_SHADER_FILE_PENCIL_LIGHTING[] =
{
    "shaders/pencilRenderingLighting",
    "shaders/pencilRenderingLightingHlslcc",
};
static const char * const GSH_SHADER_FILE_PENCIL_INTERIOR[] =
{
    "shaders/pencilRenderingInterior",
    "shaders/pencilRenderingInteriorHlslcc",
};
static const char * const GSH_SHADER_FILE_PENCIL_CONTOURS[] =
{
    "shaders/pencilRenderingContours",
    "shaders/pencilRenderingContoursHlslcc",
};
static const char * const GSH_SHADER_FILE_TEXTURE2D[] =
{
    "shaders/texture2D",
    "shaders/texture2DHlslcc",
};
static int g_ShaderFileIdx = 0;

struct ShadowUniforms
{
    Mtx44 modelMtx44;
    Mtx44 viewprojMtx44;
};

static struct _ShadowPass
{
    DEMOGfxShader demoShader;

    // -- Vertex Fetch Info -- //
    nn::gfx::Pipeline pipeline;
    void* pPipelineData;

    // -- Viewport Info -- //
    nn::gfx::ViewportScissorState viewportScissor;
    void* pViewportScissorData;

    // -- Shader Location -- //
    int posLoc;

    // Constant Buffers
    DEMOGfxBuffer uniforms;

    int uniformLoc;

    // -- Texture Setting -- //
    nn::gfx::Texture myDepthBuffer;
    nn::gfx::DepthStencilView myDepthBufferView;
    DEMOGfxMemPool* myDepthBufferData;
} ShadowPass;

static struct _ExpShadowPass
{
    DEMOGfxShader demoShader;

    // -- Vertex Fetch Info -- //
    nn::gfx::Pipeline pipeline;
    void* pPipelineData;

    // -- Viewport Info -- //
    nn::gfx::ViewportScissorState viewportScissor;
    void* pViewportScissorData;

    // -- Shader Location -- //
    int posLoc;
    int texcoordLoc;
    int textureLoc;

    int surfaceSizeLoc;

    // -- Initial Data -- //
    int numAttrib;
    int numIndices;
    f32 posInitData[12];
    f32 texInitData[8];
    u32 idxInitData[6];

    // -- Attribute Buffers -- //
    DEMOGfxBuffer positionBuffer;
    DEMOGfxBuffer textureCoordinateBuffer;
    DEMOGfxBuffer indexBuffer;

    // -- Texture Setting -- //
    nn::gfx::Texture myColorBuffer;
    nn::gfx::ColorTargetView myColorBufferView;
    DEMOGfxMemPool* myColorBufferData;

    nn::gfx::TextureView myDepthBufferView;
    nn::gfx::DescriptorSlot myDepthBufferSlot;

    nn::gfx::Sampler mySampler;
    nn::gfx::DescriptorSlot mySamplerSlot;
} ExpShadowPass;

static struct _BlurHoriShadowPass
{
    DEMOGfxShader demoShader;

    // -- Vertex Fetch Info -- //
    nn::gfx::Pipeline pipeline;
    void* pPipelineData;

    // -- Viewport Info -- //
    nn::gfx::ViewportScissorState viewportScissor;
    void* pViewportScissorData;

    // -- Shader Location -- //
    int posLoc;
    int texcoordLoc;
    int textureLoc;

    int surfaceSizeLoc;

    // -- Initial Data -- //
    int numAttrib;
    int numIndices;
    f32 posInitData[12];
    f32 texInitData[8];
    u32 idxInitData[6];

    // -- Attribute Buffers -- //
    DEMOGfxBuffer positionBuffer;
    DEMOGfxBuffer textureCoordinateBuffer;
    DEMOGfxBuffer indexBuffer;

    // -- Texture Setting -- //
    nn::gfx::Texture myColorBuffer;
    nn::gfx::ColorTargetView myColorBufferView;
    DEMOGfxMemPool* myColorBufferData;

    nn::gfx::TextureView myTextureView;
    nn::gfx::DescriptorSlot myTextureBufferSlot;
    nn::gfx::Sampler mySampler;
    nn::gfx::DescriptorSlot mySamplerSlot;
} BlurHoriShadowPass;

static struct _BlurVertShadowPass
{
    DEMOGfxShader demoShader;

    // -- Vertex Fetch Info -- //
    nn::gfx::Pipeline pipeline;
    void* pPipelineData;

    // -- Viewport Info -- //
    nn::gfx::ViewportScissorState viewportScissor;
    void* pViewportScissorData;

    // -- Shader Location -- //
    int posLoc;
    int texcoordLoc;
    int textureLoc;

    int surfaceSizeLoc;

    // -- Initial Data -- //
    int numAttrib;
    int numIndices;
    f32 posInitData[12];
    f32 texInitData[8];
    u32 idxInitData[6];

    // -- Attribute Buffers -- //
    DEMOGfxBuffer positionBuffer;
    DEMOGfxBuffer textureCoordinateBuffer;
    DEMOGfxBuffer indexBuffer;

    // -- Texture Setting -- //
    nn::gfx::ColorTargetView myColorBufferView;
    nn::gfx::TextureView myTextureView;
    nn::gfx::DescriptorSlot myTextureBufferSlot;
    nn::gfx::Sampler mySampler;
    nn::gfx::DescriptorSlot mySamplerSlot;
} BlurVertShadowPass;

struct GeometryPassVertexUniforms
{
    Mtx44 modelMtx44;
    Mtx44 viewprojMtx44;
};

struct GeometryPassPixelUniforms
{
    Qtrn color;
};

static struct _GeometryPass
{
    DEMOGfxShader demoShader;

    // -- Vertex Fetch Info -- //
    nn::gfx::Pipeline pipeline;
    void* pPipelineData;

    // -- Viewport Info -- //
    nn::gfx::ViewportScissorState viewportScissor;
    void* pViewportScissorData;

    // -- Shader Location -- //
    int posLoc;
    int nrmLoc;

    // Constant Buffers
    DEMOGfxBuffer vertexUniformBuffer;
    int vertexUniformBufferLoc;

    DEMOGfxBuffer pixelUniformBuffer;
    int pixelUniformBufferLoc;

    // -- Attribute Buffers -- //
    DEMOGfxBuffer positionBuffers[NUM_MODEL];
    DEMOGfxBuffer normalBuffers[NUM_MODEL];
    DEMOGfxBuffer indexBuffers[NUM_MODEL];
    f32* posBuf[NUM_MODEL];
    f32* nrmBuf[NUM_MODEL];
    u32* idxBuf[NUM_MODEL];

    // -- Texture Setting -- //
    nn::gfx::Texture myDepthBuffer;
    nn::gfx::DepthStencilView myDepthBufferView;
    nn::gfx::Texture myColorBuffer[2];
    nn::gfx::ColorTargetView myColorBufferView[2];
    void * myDepthBufferData;
    DEMOGfxMemPool* myColorBufferData[2];
} GeometryPass;

struct OutlineUniforms
{
    Qtrn surfaceSize;
};

static struct _OutlinePass
{
    DEMOGfxShader demoShader;

    // -- Vertex Fetch Info -- //
    nn::gfx::Pipeline pipeline;
    void* pPipelineData;

    // -- Viewport Info -- //
    nn::gfx::ViewportScissorState viewportScissor;
    void* pViewportScissorData;

    // -- Shader Location -- //
    int posLoc;
    int texcoordLoc;
    int textureLoc[2];

    int surfaceSizeLoc;

    DEMOGfxBuffer uniformBuffer;

    // -- Initial Data -- //
    int numAttrib;
    int numIndices;
    f32 posInitData[12];
    f32 texInitData[8];
    u32 idxInitData[6];

    // -- Attribute Buffers -- //
    DEMOGfxBuffer positionBuffer;
    DEMOGfxBuffer textureCoordinateBuffer;
    DEMOGfxBuffer indexBuffer;

    // -- Texture Setting -- //
    nn::gfx::Texture myColorBuffer;
    nn::gfx::ColorTargetView myColorBufferView;
    DEMOGfxMemPool* myColorBufferData;
    nn::gfx::Texture myTextureBuffer[2];
    nn::gfx::TextureView myTextureBufferView[2];
    nn::gfx::DescriptorSlot myTextureBufferSlot[2];
    nn::gfx::Sampler mySampler;
    nn::gfx::DescriptorSlot mySamplerSlot;
} OutlinePass;

struct LightingUniforms
{
    Qtrn ambientColor;
    Qtrn lightColor;
    Qtrn lightPosition;
    Qtrn cameraPosition;
    Qtrn frustum;
    Mtx44 viewInverse;
    Mtx44 shadowMtx;
};

static struct _LightingPass
{
    DEMOGfxShader demoShader;

    // -- Vertex Fetch Info -- //
    nn::gfx::Pipeline pipeline;
    void* pPipelineData;

    // -- Viewport Info -- //
    nn::gfx::ViewportScissorState viewportScissor;
    void* pViewportScissorData;

    // -- Shader Location -- //
    int posLoc;
    int texcoordLoc;
    int textureLoc[5];

    // Constant Buffers
    DEMOGfxBuffer uniformBuffer;

    int uniformBlockLoc;

    // -- Initial Data -- //
    int numAttrib;
    int numIndices;
    f32 posInitData[12];
    f32 texInitData[8];
    u32 idxInitData[6];

    // -- Attribute Buffers -- //
    DEMOGfxBuffer positionBuffer;
    DEMOGfxBuffer textureCoordinateBuffer;
    DEMOGfxBuffer indexBuffer;

    // -- Texture Setting -- //
    nn::gfx::Texture myColorBuffer;
    nn::gfx::ColorTargetView myColorBufferView;
    DEMOGfxMemPool* myColorBufferData;

    nn::gfx::TextureView myTextureBuffer[5];
    nn::gfx::DescriptorSlot myTextureBufferSlot[5];
    nn::gfx::Sampler mySampler;
    nn::gfx::DescriptorSlot mySamplerSlot;
    nn::gfx::Sampler myDepthSampler;
    nn::gfx::DescriptorSlot myDepthSamplerSlot;
} LightingPass;

struct InteriorUniforms
{
    Qtrn divisions;
};

static struct _InteriorPass
{
    DEMOGfxShader demoShader;

    // -- Vertex Fetch Info -- //
    nn::gfx::Pipeline pipeline;
    void* pPipelineData;

    // -- Viewport Info -- //
    nn::gfx::ViewportScissorState viewportScissor;
    void* pViewportScissorData;

    // -- Shader Location -- //
    int posLoc;
    int texcoordLoc;
    int textureLoc[2];

    int divisionsLoc;
    DEMOGfxBuffer uniformBuffer;

    // -- Initial Data -- //
    int numAttrib;
    int numIndices;
    f32 posInitData[12];
    f32 texInitData[8];
    u32 idxInitData[6];

    // -- Attribute Buffers -- //
    DEMOGfxBuffer positionBuffer;
    DEMOGfxBuffer textureCoordinateBuffer;
    DEMOGfxBuffer indexBuffer;

    // -- Texture Setting -- //
    nn::gfx::TextureView myTextureBufferView;
    nn::gfx::DescriptorSlot myTextureBufferSlot;
    nn::gfx::Sampler mySampler;
    nn::gfx::DescriptorSlot mySamplerSlot;
} InteriorPass;

struct ContourUniforms
{
    Qtrn wobble0;
    Qtrn wobble1;
    Qtrn wobble2;
    Qtrn wobble3;
};

static struct _ContoursPass
{
    DEMOGfxShader demoShader;

    // -- Vertex Fetch Info -- //
    nn::gfx::Pipeline pipeline;
    void* pPipelineData;

    // -- Viewport Info -- //
    nn::gfx::ViewportScissorState viewportScissor;
    void* pViewportScissorData;

    // -- Shader Location -- //
    int posLoc;
    int texcoordLoc;
    int textureLoc;

    int uniformLoc;
    DEMOGfxBuffer uniformBuffer;

    // -- Initial Data -- //
    int numAttrib;
    int numIndices;
    f32 posInitData[3 * CONTOUR_VERTICES * CONTOUR_VERTICES];
    f32 texInitData[2 * CONTOUR_VERTICES * CONTOUR_VERTICES];
    u32 idxInitData[6 * CONTOUR_SEGMENTS * CONTOUR_SEGMENTS];

    // -- Attribute Buffers -- //
    DEMOGfxBuffer positionBuffer;
    DEMOGfxBuffer textureCoordinateBuffer;
    DEMOGfxBuffer indexBuffer;

    // -- Texture Setting -- //
    nn::gfx::TextureView myTextureView;
    nn::gfx::DescriptorSlot myTextureBufferSlot;

    nn::gfx::Sampler mySampler;
    nn::gfx::DescriptorSlot mySamplerSlot;
} ContoursPass;

struct BuffersUniforms
{
    Qtrn offset;
};

static struct _BuffersPass
{
    DEMOGfxShader demoShader;

    // -- Vertex Fetch Info -- //
    nn::gfx::Pipeline pipeline[2];
    void* pPipelineData[2];

    // -- Viewport Info -- //
    nn::gfx::ViewportScissorState viewportScissor;
    void* pViewportScissorData;

    // -- Shader Location -- //
    int posLoc;
    int texcoordLoc;
    int textureLoc;

    int offsetLoc;
    DEMOGfxBuffer uniformBuffer[7];

    // -- Initial Data -- //
    int numAttrib;
    int numIndices;
    f32 posInitData[12];
    f32 texInitData[8];
    u32 idxInitData[6];

    // -- Attribute Buffers -- //
    DEMOGfxBuffer positionBuffer;
    DEMOGfxBuffer textureCoordinateBuffer;
    DEMOGfxBuffer indexBuffer;

    // -- Texture Setting -- //
    nn::gfx::TextureView myTextureBuffer[7];
    nn::gfx::DescriptorSlot myTextureBufferSlot[7];
    nn::gfx::Sampler mySampler;
    nn::gfx::DescriptorSlot mySamplerSlot;
} BuffersPass;

// Globals
static float s_rotateY;
static BOOL s_drawBuffers;
static u32 s_modelIdx;
static u32 s_ticks;

// Prototype
static void CameraInit( Mtx44* resultViewProjMtx44, Mtx44* resultInvViewMtx44, Qtrn& cameraPosition, Qtrn& frustum );
static void LightInit( Mtx44* resultViewProjMtx44, Qtrn& lightingPosition );
static void SceneInit();
static void CreatePencilTextures();
static void SceneUpdate( Mtx44* modelMtx );
static void ShadowPassInit();
static void ExpShadowPassInit();
static void BlurHoriShadowPassInit();
static void BlurVertShadowPassInit();
static void GeometryPassInit();
static void OutlinePassInit();
static void LightingPassInit();
static void InteriorPassInit();
static void ContoursPassInit();
static void BuffersPassInit();
static void ShadowPassFinalize();
static void ExpShadowPassFinalize();
static void BlurHoriShadowPassFinalize();
static void BlurVertShadowPassFinalize();
static void GeometryPassFinalize();
static void OutlinePassFinalize();
static void LightingPassFinalize();
static void InteriorPassFinalize();
static void ContoursPassFinalize();
static void BuffersPassFinalize();
static void SceneFree();
static void ShadowPassDraw();
static void ExpShadowPassDraw();
static void BlurHoriShadowPassDraw();
static void BlurVertShadowPassDraw();
static void GeometryPassDraw();
static void OutlinePassDraw();
static void LightingPassDraw();
static void InteriorPassDraw();
static void ContoursPassDraw();
static void BuffersPassDraw();
static void SceneDraw();

static void SetDefaultPipelineStateInfo(nn::gfx::Pipeline::InfoType& info,
                                        nn::gfx::BlendTargetStateInfo* blendTargetStateInfoArray,
                                        int blendTargetStateCount,
                                        nn::gfx::ColorTargetStateInfo* colorTargetStateInfoArray,
                                        int colorTargetStateCount,
                                        nn::gfx::VertexAttributeStateInfo* vertexAttributeStateInfoArray,
                                        int vertexAttributeStateCount,
                                        nn::gfx::VertexBufferStateInfo* vertexBufferStateInfoArray,
                                        int vertexBufferStateCount,
                                        nn::gfx::DepthStencilStateInfo& depthStencilStateInfo,
                                        nn::gfx::RasterizerStateInfo& rasterizerStateInfo,
                                        nn::gfx::BlendStateInfo& blendStateInfo,
                                        nn::gfx::RenderTargetStateInfo& renderTargetStateInfo,
                                        nn::gfx::VertexStateInfo& vertexStateInfo)
{
    // Setup the default state like in Cafe

    info.SetDefault();

    depthStencilStateInfo.SetDefault();
    depthStencilStateInfo.SetDepthComparisonFunction( nn::gfx::ComparisonFunction_Less );
    depthStencilStateInfo.SetDepthTestEnabled( true );
    depthStencilStateInfo.SetDepthWriteEnabled( true );
    depthStencilStateInfo.SetStencilTestEnabled( false );
    depthStencilStateInfo.SetStencilReadMask( 0xFF );
    depthStencilStateInfo.SetStencilWriteMask( 0xFF );
    depthStencilStateInfo.EditBackStencilStateInfo().SetComparisonFunction( nn::gfx::ComparisonFunction_Always );
    depthStencilStateInfo.EditBackStencilStateInfo().SetDepthPassOperation( nn::gfx::StencilOperation_Replace );
    depthStencilStateInfo.EditBackStencilStateInfo().SetDepthFailOperation( nn::gfx::StencilOperation_Replace );
    depthStencilStateInfo.EditBackStencilStateInfo().SetStencilFailOperation( nn::gfx::StencilOperation_Replace );
    depthStencilStateInfo.EditBackStencilStateInfo().SetStencilRef( 0x1 );
    depthStencilStateInfo.EditFrontStencilStateInfo().SetComparisonFunction( nn::gfx::ComparisonFunction_Always );
    depthStencilStateInfo.EditFrontStencilStateInfo().SetDepthPassOperation( nn::gfx::StencilOperation_Replace );
    depthStencilStateInfo.EditFrontStencilStateInfo().SetDepthFailOperation( nn::gfx::StencilOperation_Replace );
    depthStencilStateInfo.EditFrontStencilStateInfo().SetStencilFailOperation( nn::gfx::StencilOperation_Replace );
    depthStencilStateInfo.EditFrontStencilStateInfo().SetStencilRef( 0x1 );

    rasterizerStateInfo.SetDefault();
    rasterizerStateInfo.SetFrontFace( nn::gfx::FrontFace_Ccw );
    rasterizerStateInfo.SetCullMode( nn::gfx::CullMode_None );
    rasterizerStateInfo.SetFillMode( nn::gfx::FillMode_Solid );
    rasterizerStateInfo.SetDepthBias( 0 );
    rasterizerStateInfo.SetMultisampleEnabled( false );
    rasterizerStateInfo.SetRasterEnabled( true );
    rasterizerStateInfo.SetScissorEnabled( true );
    rasterizerStateInfo.SetSlopeScaledDepthBias( 0.0f );
    rasterizerStateInfo.SetDepthClipEnabled( false );

    blendStateInfo.SetDefault();
    blendStateInfo.SetBlendConstant( 0.0f, 0.0f, 0.0f, 0.0f );
    blendStateInfo.SetBlendTargetStateInfoArray( blendTargetStateInfoArray, blendTargetStateCount );

    renderTargetStateInfo.SetDefault();
    renderTargetStateInfo.SetDepthStencilFormat( DEMODepthBufferInfo.GetImageFormat() );
    renderTargetStateInfo.SetColorTargetStateInfoArray( colorTargetStateInfoArray, colorTargetStateCount );

    vertexStateInfo.SetDefault();
    vertexStateInfo.SetVertexAttributeStateInfoArray( vertexAttributeStateInfoArray, vertexAttributeStateCount );
    vertexStateInfo.SetVertexBufferStateInfoArray( vertexBufferStateInfoArray, vertexBufferStateCount );

}

// Init function for setting camera projection matrix
static void CameraInit( Mtx44* resultViewProjMtx44, Mtx44* resultInvViewMtx44, Qtrn& cameraPosition, Qtrn& frustum )
{
    // row major matricies
    Mtx lookAtMtx34;
    Mtx invLookAtMtx34;
    Mtx44 projMtx44;
    Mtx44 viewMtx44;

    Vec     up = { 0.0f, 1.0f, 0.0f };
    Vec  objPt = { 0.0f, 0.0f, 0.0f };
    Vec camLoc = { 0.0f, 0.0f, 300.0f };

    f32   pers = 30.0f; // degrees
    f32 aspect = (f32)SURFACE_WIDTH / (f32)SURFACE_HEIGHT;
    f32  znear = 200.0f;
    f32   zfar = 400.0f;

    cameraPosition.x = camLoc.x;
    cameraPosition.y = camLoc.y;
    cameraPosition.z = camLoc.z;
    cameraPosition.w = 1.0f;

    // Set camera frustum
    frustum.y = tan(0.5f * pers * PI / 180.0f);
    frustum.x = frustum.y * aspect;
    frustum.z = znear;
    frustum.w = zfar - znear;

    // Compute perspective matrix (w/ linear z)
    MTXPerspective( projMtx44, pers, aspect, znear, zfar );
    projMtx44[2][2] /= 0.5f * (zfar + znear);
    projMtx44[2][3] /= (zfar);
    projMtx44[2][3] -= 1.0f;

    // Compute lookAt matrix
    MTXLookAt( lookAtMtx34, &camLoc, &up, &objPt );
    MTX34To44( lookAtMtx34, viewMtx44 );
    MTXInverse( lookAtMtx34, invLookAtMtx34 );

    // return view+proj and invView
    MTX44Concat( projMtx44, viewMtx44, *resultViewProjMtx44 );
    MTX34To44( invLookAtMtx34, *resultInvViewMtx44 );
}

// Init function for setting shadow projection matrix
static void LightInit( Mtx44* resultViewProjMtx44, Qtrn& lightPosition )
{
    // row major matricies
    Mtx lookAtMtx34;
    Mtx44 projMtx44;
    Mtx44 viewMtx44;

    Vec     up = { 0.0f, 1.0f, 0.0f };
    Vec  objPt = { 0.0f, 0.0f, 0.0f };
    Vec lightLoc = { 80.0f, 60.0f, 80.0f };

    f32   pers = 90.0f; // degrees
    f32 aspect = (f32)SHADOWMAP_WIDTH / (f32)SHADOWMAP_HEIGHT;
    f32  znear = 10.0f;
    f32   zfar = 200.0f;

    // Set light position
    lightPosition.x = lightLoc.x;
    lightPosition.y = lightLoc.y;
    lightPosition.z = lightLoc.z;
    lightPosition.w = 1.0f;

    // Compute perspective matrix (w/ linear z)
    MTXPerspective( projMtx44, pers, aspect, znear, zfar );
    projMtx44[ 2 ][ 2 ] /= 0.5f * ( zfar + znear );
    projMtx44[ 2 ][ 3 ] /= ( zfar );
    projMtx44[ 2 ][ 3 ] -= 1.0f;

    // Compute lookAt matrix
    MTXLookAt( lookAtMtx34, &lightLoc, &up, &objPt );
    MTX34To44( lookAtMtx34, viewMtx44 );

    // return view+proj and invView
    MTX44Concat( projMtx44, viewMtx44, *resultViewProjMtx44 );
}

// The init function for the rendering portions of this app
static void SceneInit()
{
    // Initialize globals
    s_rotateY = 0.0f;
    s_drawBuffers = TRUE;
    s_modelIdx = 0;
    s_ticks = 0;

    // Generate pencil texture used for interior shading
    CreatePencilTextures();

    // Initialize passes
    ShadowPassInit();
    ExpShadowPassInit();
    BlurHoriShadowPassInit();
    BlurVertShadowPassInit();
    GeometryPassInit();
    OutlinePassInit();
    LightingPassInit();
    InteriorPassInit();
    ContoursPassInit();
    BuffersPassInit();

    // Initialze a vew things that couldn't be until after creation:
    ShadowUniforms* pShadowUniforms = ShadowPass.uniforms.Map< ShadowUniforms >( );
    GeometryPassVertexUniforms* pGeometryUniforms = GeometryPass.vertexUniformBuffer.Map< GeometryPassVertexUniforms >();
    LightingUniforms* pLightingUniforms = LightingPass.uniformBuffer.Map< LightingUniforms >();
#if NN_GFX_IS_TARGET_GX
    GX2EndianSwap( pShadowUniforms, sizeof( ShadowUniforms ) );
    GX2EndianSwap( pGeometryUniforms, sizeof( GeometryPassVertexUniforms ) );
    GX2EndianSwap( pLightingUniforms, sizeof( LightingUniforms ) );
#endif

    CameraInit( &pGeometryUniforms->viewprojMtx44, &pLightingUniforms->viewInverse, pLightingUniforms->cameraPosition,
        pLightingUniforms->frustum );
    SceneUpdate(&pGeometryUniforms->modelMtx44);
    LightInit(&pShadowUniforms->viewprojMtx44, pLightingUniforms->lightPosition);
    memcpy(&pLightingUniforms->shadowMtx, &pShadowUniforms->viewprojMtx44, sizeof( pLightingUniforms->shadowMtx ) );
    memcpy( &pShadowUniforms->modelMtx44, &pGeometryUniforms->modelMtx44, sizeof( pGeometryUniforms->modelMtx44 ) );

#if NN_GFX_IS_TARGET_GX
    GX2EndianSwap( pShadowUniforms, sizeof( ShadowUniforms ) );
    GX2EndianSwap( pGeometryUniforms, sizeof( GeometryPassVertexUniforms ) );
    GX2EndianSwap( pLightingUniforms, sizeof( LightingUniforms ) );
#endif
    LightingPass.uniformBuffer.Unmap();
    GeometryPass.vertexUniformBuffer.Unmap();
    ShadowPass.uniforms.Unmap();
}

static void CreatePencilGeometry( f32* posBuf, f32* texcoordBuf, u32* idxBuf )
{
    // geometry setup
    posBuf[0] = -0.25f; posBuf[1] = -0.015625f; posBuf[2] =  0.0f;
    posBuf[3] =  0.25f; posBuf[4] = -0.015625f; posBuf[5] =  0.0f;
    posBuf[6] =  0.25f; posBuf[7] =  0.015625f; posBuf[8] =  0.0f;
    posBuf[9] = -0.25f; posBuf[10] = 0.015625f; posBuf[11] = 0.0f;

    texcoordBuf[ 0 ] = 0.0f; texcoordBuf[ 1 ] = 1.0f;
    texcoordBuf[ 2 ] = 1.0f; texcoordBuf[ 3 ] = 1.0f;
    texcoordBuf[ 4 ] = 1.0f; texcoordBuf[ 5 ] = 0.0f;
    texcoordBuf[ 6 ] = 0.0f; texcoordBuf[ 7 ] = 0.0f;

    idxBuf[0] = 0;
    idxBuf[1] = 1;
    idxBuf[2] = 2;
    idxBuf[3] = 2;
    idxBuf[4] = 3;
    idxBuf[5] = 0;
}

static void CreatePencilTexturesInitPipeline( nn::gfx::Pipeline& pipeline, void** ppData,
                                             nn::gfx::Shader* pShader )
{
    const int numAttribs = 3;
    const int numBuffers = 3;
    nn::gfx::Pipeline::InfoType pipelineInfo;
    nn::gfx::VertexBufferStateInfo vertexBufferStates[ 3 ];
    nn::gfx::VertexAttributeStateInfo vertexAttributeStates[ 3 ];
    nn::gfx::BlendTargetStateInfo blendTargetStates[ 1 ];
    nn::gfx::ColorTargetStateInfo colorTargetStates[ 1 ];
    int posLoc = pShader->GetInterfaceSlot( nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_Input, "a_position" );
    int texcoordLoc = pShader->GetInterfaceSlot( nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_Input, "a_texCoord" );
    int modifyLoc = pShader->GetInterfaceSlot( nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_Input, "a_modify" );

    pipelineInfo.SetDefault();

    // Setup vertex attributes
    vertexAttributeStates[0].SetDefault();
    vertexAttributeStates[0].SetBufferIndex( posLoc );
    vertexAttributeStates[0].SetShaderSlot( posLoc );
    vertexAttributeStates[0].SetFormat( nn::gfx::AttributeFormat_32_32_32_Float );
    vertexAttributeStates[1].SetDefault();
    vertexAttributeStates[1].SetBufferIndex( texcoordLoc );
    vertexAttributeStates[1].SetShaderSlot( texcoordLoc );
    vertexAttributeStates[1].SetFormat( nn::gfx::AttributeFormat_32_32_Float );
    vertexAttributeStates[2].SetDefault();
    vertexAttributeStates[2].SetBufferIndex( modifyLoc );
    vertexAttributeStates[2].SetShaderSlot( modifyLoc );
    vertexAttributeStates[2].SetFormat( nn::gfx::AttributeFormat_32_32_32_32_Float );

    // Setup vertex buffers
    ASSERT( posLoc < 3 );
    ASSERT( texcoordLoc < 3 );
    ASSERT( modifyLoc < 3 );
    vertexBufferStates[posLoc].SetDefault();
    vertexBufferStates[posLoc].SetStride( 3 * sizeof(float) );
    vertexBufferStates[texcoordLoc].SetDefault();
    vertexBufferStates[texcoordLoc].SetStride( 2 * sizeof(float) );
    vertexBufferStates[modifyLoc].SetDefault();
    vertexBufferStates[modifyLoc].SetStride( 4 * sizeof(float) );
    vertexBufferStates[modifyLoc].SetDivisor( 1 );

    // Setup BlendControl State
    blendTargetStates[0].SetDefault();
    blendTargetStates[0].SetBlendEnabled( true );
    blendTargetStates[0].SetSourceColorBlendFactor( nn::gfx::BlendFactor_SourceAlpha );
    blendTargetStates[0].SetDestinationColorBlendFactor( nn::gfx::BlendFactor_OneMinusSourceAlpha );
    blendTargetStates[0].SetColorBlendFunction( nn::gfx::BlendFunction_Add );
    blendTargetStates[0].SetSourceAlphaBlendFactor( nn::gfx::BlendFactor_SourceAlpha );
    blendTargetStates[0].SetDestinationAlphaBlendFactor( nn::gfx::BlendFactor_OneMinusSourceAlpha );
    blendTargetStates[0].SetAlphaBlendFunction( nn::gfx::BlendFunction_Add );

    // Setup the render target state.
    colorTargetStates[ 0 ].SetDefault();
    colorTargetStates[ 0 ].SetFormat( pencilTexture[ 0 ].pencilTextureInfo.GetImageFormat() );

    nn::gfx::DepthStencilStateInfo depthStencilStateInfo;
    nn::gfx::RasterizerStateInfo rasterizerStateInfo;
    nn::gfx::BlendStateInfo blendStateInfo;
    nn::gfx::RenderTargetStateInfo renderTargetStateInfo;
    nn::gfx::VertexStateInfo vertexStateInfo;

    SetDefaultPipelineStateInfo( pipelineInfo, blendTargetStates, 1, colorTargetStates, 1,
        vertexAttributeStates, numAttribs, vertexBufferStates, numBuffers, depthStencilStateInfo, rasterizerStateInfo,
        blendStateInfo, renderTargetStateInfo, vertexStateInfo);

    // Override the default Depth/Stencil state
    depthStencilStateInfo.SetDepthTestEnabled( false );
    depthStencilStateInfo.SetDepthWriteEnabled( false );
    depthStencilStateInfo.SetDepthComparisonFunction( nn::gfx::ComparisonFunction_Never );

    // Setup shaders and state info
    pipelineInfo.SetShaderPtr( pShader );
    pipelineInfo.SetBlendStateInfo(&blendStateInfo);
    pipelineInfo.SetDepthStencilStateInfo(&depthStencilStateInfo);
    pipelineInfo.SetRasterizerStateInfo(&rasterizerStateInfo);
    pipelineInfo.SetRenderTargetStateInfo(&renderTargetStateInfo);
    pipelineInfo.SetVertexStateInfo(&vertexStateInfo);

    // Initialize the pipeline
    size_t pipelineSize = nn::gfx::Pipeline::GetRequiredMemorySize( pipelineInfo );
    void* pipelineData = DEMOGfxAllocMEM2( pipelineSize, nn::gfx::Pipeline::RequiredMemoryInfo_Alignment );
    pipeline.SetMemory( pipelineData, pipelineSize );
    pipeline.Initialize( &DEMODevice, pipelineInfo );

    // Return the pointer so the app can free it.
    *ppData = pipelineData;
}

static void RenderPencilTextures( nn::gfx::Pipeline* pipeline,
                                DEMOGfxBuffer* modifyBuffer,
                                DEMOGfxBuffer* positionBuffer,
                                DEMOGfxBuffer* textureCoordinateBuffer,
                                DEMOGfxBuffer* indexBuffer,
                                nn::gfx::DescriptorSlot& basePencilTextureSlot,
                                nn::gfx::DescriptorSlot& basePencilSamplerSlot,
                                int posLoc,
                                int texcoordLoc,
                                int modifyLoc,
                                int textureLoc,
                                int numIndices
                                )
{
    for( int index = 0; index < NUM_PENCIL_TEXTURES; index++ )
    {
        const nn::gfx::ColorTargetView* const colorTargets[1] = { &pencilTexture[index].colorView };

        // Start by clearing the pencil texture
        DEMOCommandBuffer.Begin();
        DEMOCommandBuffer.ClearColor( &pencilTexture[index].colorView, 1.0f, 1.0f, 1.0f, 1.0f, NULL );
        DEMOCommandBuffer.End();

        DEMOQueue.ExecuteCommand( &DEMOCommandBuffer, NULL );
        DEMOQueue.Sync();

        // Render repeated strokes, the first shade does nothing
        for(int shade = 1; shade < PENCIL_TEXTURE_DIVISIONS; ++shade)
        {
            nn::gfx::ViewportScissorState viewportScissor;
            void* pViewportScissorData = NULL;

            // Setup the viewport/scissor
            DEMOGfxSetViewportScissorState( &viewportScissor, &pViewportScissorData,
                0.0f, static_cast< float >(shade * PENCIL_TEXTURE_SUBHEIGHT),
                static_cast< float >(PENCIL_TEXTURE_WIDTH), static_cast< float >(PENCIL_TEXTURE_SUBHEIGHT),
                0.0f, 1.0f, static_cast< float >( PENCIL_TEXTURE_HEIGHT ), true );

            // Setup the uniforms for this loop
            float* mOffset = modifyBuffer->Map< float >( );
            for(int stroke = 0; stroke < shade * 4096; ++stroke)
            {
                int offset = stroke * 4;

                // Setup the uniform for the current draw call.
                mOffset[offset + 0] = -1.0f + float(DEMORand() % 1024) / 1024.0f * 2.0f; // stroke x offset
                mOffset[offset + 1] = -1.0f + float(DEMORand() % 1024) / 1024.0f * 2.0f; // stroke y offset
                mOffset[offset + 2] = -0.3f + 0.2f * shade; // stroke rotation
                mOffset[offset + 3] = float(shade) / float(PENCIL_TEXTURE_DIVISIONS); // stroke darkness
            }
            modifyBuffer->Unmap();

            // Starting CommandBuffer
            DEMOCommandBuffer.Begin();
#if NN_GFX_IS_TARGET_GX
    GX2SetShaderMode( GX2_SHADER_MODE_UNIFORM_BLOCK );
#endif
            DEMOCommandBuffer.SetRenderTargets( 1, colorTargets, NULL );
            DEMOCommandBuffer.SetPipeline( pipeline );

            DEMOCommandBuffer.SetDescriptorPool( &DEMOTextureDescriptorPool );
            DEMOCommandBuffer.SetDescriptorPool( &DEMOSamplerDescriptorPool );
            DEMOCommandBuffer.SetDescriptorPool( &DEMOBufferViewDescriptorPool );

            DEMOCommandBuffer.SetVertexBuffer( posLoc, positionBuffer->gpuAddress, sizeof(float) * 3, positionBuffer->size );
            DEMOCommandBuffer.SetVertexBuffer( texcoordLoc, textureCoordinateBuffer->gpuAddress, sizeof(float) * 2, textureCoordinateBuffer->size );
            DEMOCommandBuffer.SetVertexBuffer( modifyLoc, modifyBuffer->gpuAddress, sizeof(float) * 4, modifyBuffer->size );
            DEMOCommandBuffer.SetTextureAndSampler( textureLoc, nn::gfx::ShaderStage_Pixel, basePencilTextureSlot, basePencilSamplerSlot );

            // Setup the viewport/scissor
            DEMOCommandBuffer.SetViewportScissorState( &viewportScissor );

            DEMOCommandBuffer.DrawIndexed( nn::gfx::PrimitiveTopology_TriangleList, nn::gfx::IndexFormat_Uint32, indexBuffer->gpuAddress,
                numIndices, 0, shade * 4096, 0 );

            DEMOCommandBuffer.FlushMemory( nn::gfx::GpuAccess_ColorBuffer );
            DEMOCommandBuffer.InvalidateMemory( nn::gfx::GpuAccess_Texture );

            DEMOCommandBuffer.End();

            // This lets use dispatch the buffers more frequently to avoid timeouts
            DEMOQueue.ExecuteCommand( &DEMOCommandBuffer, NULL );
            DEMOQueue.Sync();

            viewportScissor.Finalize( &DEMODevice );
            if ( pViewportScissorData )
            {
                DEMOGfxFreeMEM2( pViewportScissorData );
            }
        }
    }
}

// Creates the pencil textures for the interior shading
static void CreatePencilTextures()
{
    DEMOGfxTexture basePencilTexture;
    nn::gfx::Sampler basePencilSampler;
    nn::gfx::DescriptorSlot basePencilSamplerSlot;

    basePencilTexture.Initialize( TEXTURE_FILE_PENCIL );

    // Initialize pencil render color buffer
    for(int i = 0; i < NUM_PENCIL_TEXTURES; ++i)
    {
        pencilTexture[ i ].pencilTextureInfo.SetImageFormat( nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm );
        DEMOGfxSetupTextureBuffer( &pencilTexture[i].texture, &pencilTexture[i].view, &pencilTexture[i].slot, &pencilTexture[i].colorView, NULL, &pencilTexture[i].pPool,
            PENCIL_TEXTURE_WIDTH, PENCIL_TEXTURE_HEIGHT, 1, 1, nn::gfx::ImageDimension_2d, nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm,
            nn::gfx::DepthStencilFetchMode_DepthComponent, 0 );
    }

    DEMOGfxInitSampler( &basePencilSampler, &basePencilSamplerSlot, nn::gfx::TextureAddressMode_ClampToEdge,
        nn::gfx::FilterMode_MinPoint_MagPoint_MipPoint, nn::gfx::ComparisonFunction_Always );

    // Get Shader pointers from Shader file
    DEMOGfxShader pencilShader;
    DEMOGfxLoadShadersFromFile(  &pencilShader , 0, GSH_SHADER_FILE_PENCIL_CREATION[g_ShaderFileIdx] );
    // Attribute Location Lookup
    int posLoc = pencilShader.GetInterfaceSlot( nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_Input, "a_position" );
    int texcoordLoc = pencilShader.GetInterfaceSlot( nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_Input, "a_texCoord" );

    // Sampler Location Lookup
    int textureLoc = pencilShader.GetInterfaceSlot( nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_Sampler, "s_texture" );

    // Lookup the instanced attribute
    int modifyLoc = pencilShader.GetInterfaceSlot( nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_Input, "a_modify" );

    //
    // Setup the input vertex/index buffers
    //
    nn::gfx::Buffer::InfoType bufferInfo;
    bufferInfo.SetDefault();

    bufferInfo.SetGpuAccessFlags( nn::gfx::GpuAccess_VertexBuffer );
    f32* posBuf = ( f32 * ) DEMOGfxAllocMEM2( sizeof( float ) * NUM_PENCIL_VERTICES, nn::gfx::Buffer::GetBufferAlignment( &DEMODevice, bufferInfo ) );
    f32* texcoordBuf = (f32 *)DEMOGfxAllocMEM2( sizeof(float) * NUM_PENCIL_TEXCOORDS, nn::gfx::Buffer::GetBufferAlignment( &DEMODevice, bufferInfo ) );
    u32 modifySize = sizeof(float) * 4 * PENCIL_TEXTURE_DIVISIONS * 4096;
    f32* modifyBuf = (f32 *)DEMOGfxAllocMEM2( modifySize, nn::gfx::Buffer::GetBufferAlignment( &DEMODevice, bufferInfo ) );

    bufferInfo.SetGpuAccessFlags( nn::gfx::GpuAccess_IndexBuffer );
    u32* idxBuf = (u32 *)DEMOGfxAllocMEM2( sizeof(float) * NUM_PENCIL_INDICES, nn::gfx::Buffer::GetBufferAlignment( &DEMODevice, bufferInfo ) );

    CreatePencilGeometry( posBuf, texcoordBuf, idxBuf );

    // Create Buffers/Views
    DEMOGfxBuffer positionBuffer;
    DEMOGfxBuffer textureCoordinateBuffer;
    DEMOGfxBuffer modifyBuffer;
    DEMOGfxBuffer indexBuffer;

    positionBuffer.Initialize( sizeof( float ) * NUM_PENCIL_VERTICES, posBuf, nn::gfx::GpuAccess_VertexBuffer, 0 );
    textureCoordinateBuffer.Initialize( sizeof( float ) * NUM_PENCIL_TEXCOORDS, texcoordBuf, nn::gfx::GpuAccess_VertexBuffer, 0 );
    modifyBuffer.Initialize( modifySize, modifyBuf, nn::gfx::GpuAccess_VertexBuffer, 0 );
    indexBuffer.Initialize( sizeof(float) * NUM_PENCIL_INDICES, idxBuf, nn::gfx::GpuAccess_IndexBuffer, 0 );

    //
    // Setup the Pipeline
    //
    nn::gfx::Pipeline pipeline;
    void* pipelineData = NULL;
    CreatePencilTexturesInitPipeline( pipeline, &pipelineData, pencilShader.GetShader() );

    // Start with a clean sheet
    DEMOSRand(0);

    //
    // Generate the Pencil Textures
    //
    RenderPencilTextures( &pipeline, &modifyBuffer, &positionBuffer,
                        &textureCoordinateBuffer, &indexBuffer,
                        basePencilTexture.GetDescriptorSlot( 0 ), basePencilSamplerSlot, posLoc,
                        texcoordLoc, modifyLoc, textureLoc, NUM_PENCIL_INDICES );

    // Free resources
    pipeline.Finalize( &DEMODevice );
    DEMOGfxFreeMEM2(pipelineData );

    basePencilSampler.Finalize( &DEMODevice );

    positionBuffer.Finalize( &DEMODevice );
    DEMOGfxFreeMEM2(posBuf);

    textureCoordinateBuffer.Finalize( &DEMODevice );
    DEMOGfxFreeMEM2(texcoordBuf);

    modifyBuffer.Finalize( &DEMODevice );
    DEMOGfxFreeMEM2(modifyBuf);

    indexBuffer.Finalize( &DEMODevice );
    DEMOGfxFreeMEM2(idxBuf);

    // Free the pencil texture
    for(int i = 0; i < NUM_PENCIL_TEXTURES; i++)
    {
        pencilTexture[i].colorView.Finalize( &DEMODevice );
    }
    basePencilTexture.Finalize();

    // Free shaders
    DEMOGfxFreeShaders( &pencilShader );

} //NOLINT(readability/fn_size)

// rotate the model
static void SceneUpdate( Mtx44* modelMtx )
{
    Mtx modelWorldMtx;

    MTXIdentity( modelWorldMtx );
    MTXRotRad( modelWorldMtx, 'y', s_rotateY / 4.0f );
    MTX34To44( modelWorldMtx, *modelMtx );

    s_rotateY += 0.06f;
    if(s_rotateY > PI * 8)
    {
        s_rotateY -= PI * 8;
        s_drawBuffers = !s_drawBuffers;
        if(s_drawBuffers)
        {
            s_modelIdx = (s_modelIdx + 1) % NUM_MODEL;
        }
    }
    ++s_ticks;
}

// Initialized the Shadow pass
static void ShadowPassInit()
{
    // initialize uniforms
    ShadowPass.uniforms.Initialize( sizeof( struct ShadowUniforms ), NULL, nn::gfx::GpuAccess_ConstantBuffer | nn::gfx::GpuAccess_Read, 0 );

    // Load shader binary to memory
    DEMOGfxLoadShadersFromFile(  &ShadowPass.demoShader , 0, GSH_SHADER_FILE_PENCIL_SHADOW[g_ShaderFileIdx] );

    // Attribute Location Lookup
    ShadowPass.posLoc = ShadowPass.demoShader.GetInterfaceSlot( nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_Input, "a_position" );

    // Uniform Location Lookup
    ShadowPass.uniformLoc = ShadowPass.demoShader.GetInterfaceSlot( nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_ConstantBuffer, "u_uniforms" );

    //
    // Setup the Pipeline
    //
    nn::gfx::Pipeline::InfoType pipelineInfo;
    nn::gfx::VertexBufferStateInfo vertexBufferStates[ 1 ];
    nn::gfx::VertexAttributeStateInfo vertexAttributeStates[ 1 ];
    nn::gfx::BlendTargetStateInfo blendTargetStates[ 1 ];

    pipelineInfo.SetDefault();

    // Setup vertex attributes
    vertexAttributeStates[0].SetDefault();
    vertexAttributeStates[0].SetBufferIndex( ShadowPass.posLoc );
    vertexAttributeStates[0].SetShaderSlot( ShadowPass.posLoc );
    vertexAttributeStates[0].SetFormat( nn::gfx::AttributeFormat_32_32_32_Float );

    // Setup vertex buffers
    ASSERT( ShadowPass.posLoc == 0 );
    vertexBufferStates[ShadowPass.posLoc].SetDefault();
    vertexBufferStates[ShadowPass.posLoc].SetStride( 3 * sizeof(float) );

    // Disable the color buffer
    blendTargetStates[0].SetDefault();
    blendTargetStates[0].SetBlendEnabled( false );
    blendTargetStates[0].SetChannelMask( 0 );

    nn::gfx::DepthStencilStateInfo depthStencilStateInfo;
    nn::gfx::RasterizerStateInfo rasterizerStateInfo;
    nn::gfx::BlendStateInfo blendStateInfo;
    nn::gfx::RenderTargetStateInfo renderTargetStateInfo;
    nn::gfx::VertexStateInfo vertexStateInfo;

    SetDefaultPipelineStateInfo(pipelineInfo, blendTargetStates, 1, NULL, 0,
        vertexAttributeStates, 1, vertexBufferStates, 1, depthStencilStateInfo, rasterizerStateInfo,
        blendStateInfo, renderTargetStateInfo, vertexStateInfo);

    // Setup shaders and state info
    pipelineInfo.SetShaderPtr(ShadowPass.demoShader.GetShader());
    pipelineInfo.SetBlendStateInfo(&blendStateInfo);
    pipelineInfo.SetDepthStencilStateInfo(&depthStencilStateInfo);
    pipelineInfo.SetRasterizerStateInfo(&rasterizerStateInfo);
    pipelineInfo.SetRenderTargetStateInfo(&renderTargetStateInfo);
    pipelineInfo.SetVertexStateInfo(&vertexStateInfo);

    // Initialize the pipeline
    size_t pipelineSize = nn::gfx::Pipeline::GetRequiredMemorySize( pipelineInfo );
    ShadowPass.pPipelineData = DEMOGfxAllocMEM2( pipelineSize,
        nn::gfx::Pipeline::RequiredMemoryInfo_Alignment );
    ShadowPass.pipeline.SetMemory( ShadowPass.pPipelineData, pipelineSize );
    ShadowPass.pipeline.Initialize( &DEMODevice, pipelineInfo );

    // Initialize render depth buffer
    DEMOGfxSetupTextureBuffer(&ShadowPass.myDepthBuffer, NULL, NULL, NULL,
        &ShadowPass.myDepthBufferView, &ShadowPass.myDepthBufferData,
        SHADOWMAP_WIDTH, SHADOWMAP_HEIGHT, 1, 1, nn::gfx::ImageDimension_2d,
        nn::gfx::ImageFormat_D32_Float, nn::gfx::DepthStencilFetchMode_DepthComponent, 0 );

#if NN_GFX_IS_TARGET_GX
    // This optimization is only for Cafe
    GX2InitDepthBufferRangeBase( reinterpret_cast< GX2DepthBuffer* >( &GeometryPass.myDepthBufferView.ToData()->gx2DepthBuffer ), GX2_ZRANGE_BASE_ZMAX);
#endif

    // Setup the viewport/scissor
    DEMOGfxSetViewportScissorState( &ShadowPass.viewportScissor, &ShadowPass.pViewportScissorData,
        0.0f, 0.0f, static_cast< float >( SHADOWMAP_WIDTH ), static_cast< float >( SHADOWMAP_HEIGHT), 0.0f, 1.0f, static_cast< float >( SHADOWMAP_HEIGHT ), true );
}

static void ExpShadowPassInit()
{
    // Initialize render color buffers
    DEMOGfxSetupTextureBuffer( &ExpShadowPass.myColorBuffer, NULL, NULL, &ExpShadowPass.myColorBufferView, NULL,
        &ExpShadowPass.myColorBufferData, SHADOWMAP_WIDTH, SHADOWMAP_HEIGHT, 1, 1,
        nn::gfx::ImageDimension_2d, nn::gfx::ImageFormat_R32_Float,
        nn::gfx::DepthStencilFetchMode_DepthComponent, 0 );

    DEMOGfxSetupTextureView( &ShadowPass.myDepthBuffer, &ExpShadowPass.myDepthBufferView, &ExpShadowPass.myDepthBufferSlot, nn::gfx::ImageDimension_2d,
        nn::gfx::ImageFormat_D32_Float, nn::gfx::DepthStencilFetchMode_DepthComponent );

    DEMOGfxInitSampler( &ExpShadowPass.mySampler, &ExpShadowPass.mySamplerSlot, nn::gfx::TextureAddressMode_ClampToEdge,
        nn::gfx::FilterMode_MinPoint_MagPoint_MipPoint, nn::gfx::ComparisonFunction_Always );

    // geometry setup
    ExpShadowPass.posInitData[0] = -1.0f; ExpShadowPass.posInitData[1] = -1.0f;
    ExpShadowPass.posInitData[2] =  0.0f; ExpShadowPass.posInitData[3] =  1.0f;
    ExpShadowPass.posInitData[4] = -1.0f; ExpShadowPass.posInitData[5] =  0.0f;
    ExpShadowPass.posInitData[6] =  1.0f; ExpShadowPass.posInitData[7] =  1.0f;
    ExpShadowPass.posInitData[8] =  0.0f; ExpShadowPass.posInitData[9] = -1.0f;
    ExpShadowPass.posInitData[10] = 1.0f; ExpShadowPass.posInitData[11] = 0.0f;

    ExpShadowPass.texInitData[0] =  0.0f; ExpShadowPass.texInitData[1] =  1.0f;
    ExpShadowPass.texInitData[2] =  1.0f; ExpShadowPass.texInitData[3] =  1.0f;
    ExpShadowPass.texInitData[4] =  1.0f; ExpShadowPass.texInitData[5] =  0.0f;
    ExpShadowPass.texInitData[6] =  0.0f; ExpShadowPass.texInitData[7] =  0.0f;

    ExpShadowPass.idxInitData[0] = 0;
    ExpShadowPass.idxInitData[1] = 1;
    ExpShadowPass.idxInitData[2] = 2;
    ExpShadowPass.idxInitData[3] = 2;
    ExpShadowPass.idxInitData[4] = 3;
    ExpShadowPass.idxInitData[5] = 0;

    ExpShadowPass.numAttrib = 2;
    ExpShadowPass.numIndices = 6;

    // Load shaders
    DEMOGfxLoadShadersFromFile(  &ExpShadowPass.demoShader , 0,  GSH_SHADER_FILE_PENCIL_EXP[g_ShaderFileIdx] );

    // Attribute Location Lookup
    ExpShadowPass.posLoc = ExpShadowPass.demoShader.GetInterfaceSlot( nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_Input, "a_position" );
    ExpShadowPass.texcoordLoc = ExpShadowPass.demoShader.GetInterfaceSlot( nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_Input, "a_texCoord" );

    // Sampler Location Lookup
    ExpShadowPass.textureLoc = ExpShadowPass.demoShader.GetInterfaceSlot( nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_Sampler, "s_source" );

    // Vertex position buffer
     ExpShadowPass.positionBuffer.Initialize(  sizeof(ExpShadowPass.posInitData),  ExpShadowPass.posInitData,  nn::gfx::GpuAccess_VertexBuffer | nn::gfx::GpuAccess_Read,  0  );

    // Vertex texcoord buffer
     ExpShadowPass.textureCoordinateBuffer.Initialize(  sizeof(ExpShadowPass.texInitData),  ExpShadowPass.texInitData,  nn::gfx::GpuAccess_VertexBuffer | nn::gfx::GpuAccess_Read,  0  );

    // Index buffer
     ExpShadowPass.indexBuffer.Initialize(  sizeof(ExpShadowPass.idxInitData),  ExpShadowPass.idxInitData,  nn::gfx::GpuAccess_IndexBuffer | nn::gfx::GpuAccess_Read,  0  );

    //
    // Pipeline Setup
    //
    nn::gfx::Pipeline pipeline;
    nn::gfx::Pipeline::InfoType pipelineInfo;
    nn::gfx::VertexBufferStateInfo vertexBufferStates[ 2 ];
    nn::gfx::VertexAttributeStateInfo vertexAttributeStates[ 2 ];
    nn::gfx::BlendTargetStateInfo blendTargetStates[ 1 ];
    nn::gfx::ColorTargetStateInfo colorTargetStates[ 1 ];

    pipelineInfo.SetDefault();

    // Setup vertex attributes
    vertexAttributeStates[0].SetDefault();
    vertexAttributeStates[0].SetBufferIndex( ExpShadowPass.posLoc );
    vertexAttributeStates[0].SetShaderSlot( ExpShadowPass.posLoc );
    vertexAttributeStates[0].SetFormat( nn::gfx::AttributeFormat_32_32_32_Float );
    vertexAttributeStates[1].SetDefault();
    vertexAttributeStates[1].SetBufferIndex( ExpShadowPass.texcoordLoc );
    vertexAttributeStates[1].SetShaderSlot( ExpShadowPass.texcoordLoc );
    vertexAttributeStates[1].SetFormat( nn::gfx::AttributeFormat_32_32_Float );

    // Setup vertex buffers
    ASSERT( ExpShadowPass.posLoc < ExpShadowPass.numAttrib );
    ASSERT( ExpShadowPass.texcoordLoc < ExpShadowPass.numAttrib );
    vertexBufferStates[ExpShadowPass.posLoc].SetDefault();
    vertexBufferStates[ExpShadowPass.posLoc].SetStride( 3 * sizeof(float) );
    vertexBufferStates[ExpShadowPass.texcoordLoc].SetDefault();
    vertexBufferStates[ExpShadowPass.texcoordLoc].SetStride( 2 * sizeof(float) );

    // Setup BlendControl State
    blendTargetStates[0].SetDefault();
    blendTargetStates[0].SetBlendEnabled( true );
    blendTargetStates[0].SetSourceColorBlendFactor( nn::gfx::BlendFactor_SourceAlpha );
    blendTargetStates[0].SetDestinationColorBlendFactor( nn::gfx::BlendFactor_OneMinusSourceAlpha );
    blendTargetStates[0].SetColorBlendFunction( nn::gfx::BlendFunction_Add );
    blendTargetStates[0].SetSourceAlphaBlendFactor( nn::gfx::BlendFactor_SourceAlpha );
    blendTargetStates[0].SetDestinationAlphaBlendFactor( nn::gfx::BlendFactor_OneMinusSourceAlpha );
    blendTargetStates[0].SetAlphaBlendFunction( nn::gfx::BlendFunction_Add );

    // Setup the render target state.
    colorTargetStates[ 0 ].SetDefault();
    colorTargetStates[ 0 ].SetFormat( pencilTexture[ 0 ].pencilTextureInfo.GetImageFormat() );

    nn::gfx::DepthStencilStateInfo depthStencilStateInfo;
    nn::gfx::RasterizerStateInfo rasterizerStateInfo;
    nn::gfx::BlendStateInfo blendStateInfo;
    nn::gfx::RenderTargetStateInfo renderTargetStateInfo;
    nn::gfx::VertexStateInfo vertexStateInfo;

    SetDefaultPipelineStateInfo(pipelineInfo, blendTargetStates, 1, colorTargetStates, 1,
        vertexAttributeStates, 2, vertexBufferStates, 2, depthStencilStateInfo, rasterizerStateInfo,
        blendStateInfo, renderTargetStateInfo, vertexStateInfo);


    // Override the default Depth/Stencil state
    depthStencilStateInfo.SetDepthTestEnabled( false );
    depthStencilStateInfo.SetDepthWriteEnabled( false );
    depthStencilStateInfo.SetDepthComparisonFunction( nn::gfx::ComparisonFunction_Never );

    // Setup shaders and state info
    pipelineInfo.SetShaderPtr(ExpShadowPass.demoShader.GetShader());
    pipelineInfo.SetBlendStateInfo(&blendStateInfo);
    pipelineInfo.SetDepthStencilStateInfo(&depthStencilStateInfo);
    pipelineInfo.SetRasterizerStateInfo(&rasterizerStateInfo);
    pipelineInfo.SetRenderTargetStateInfo(&renderTargetStateInfo);
    pipelineInfo.SetVertexStateInfo(&vertexStateInfo);
    // Initialize the pipeline
    size_t pipelineSize = nn::gfx::Pipeline::GetRequiredMemorySize( pipelineInfo );
    ExpShadowPass.pPipelineData = DEMOGfxAllocMEM2( pipelineSize,
        nn::gfx::Pipeline::RequiredMemoryInfo_Alignment );
    ExpShadowPass.pipeline.SetMemory( ExpShadowPass.pPipelineData, pipelineSize );
    ExpShadowPass.pipeline.Initialize( &DEMODevice, pipelineInfo );

    // Setup the viewport/scissor
    DEMOGfxSetViewportScissorState( &ExpShadowPass.viewportScissor, &ExpShadowPass.pViewportScissorData,
        0.0f, 0.0f, static_cast< float >( SHADOWMAP_WIDTH ), static_cast< float >( SHADOWMAP_HEIGHT ), 0.0f, 1.0f, static_cast< float >( SHADOWMAP_HEIGHT ), true );
}

// Initializes the Horizontal Blur Shadow pass
static void BlurHoriShadowPassInit()
{
    DEMOGfxSetupTextureBuffer(&BlurHoriShadowPass.myColorBuffer, NULL, NULL, &BlurHoriShadowPass.myColorBufferView,
        NULL, &BlurHoriShadowPass.myColorBufferData, SHADOWMAP_WIDTH, SHADOWMAP_HEIGHT, 1, 1,
        nn::gfx::ImageDimension_2d, nn::gfx::ImageFormat_R32_Float, nn::gfx::DepthStencilFetchMode_DepthComponent, 0 );

    // Connects the ExpShadowPass to the BlurHoriShadowPass
    DEMOGfxSetupTextureView( &ExpShadowPass.myColorBuffer, &BlurHoriShadowPass.myTextureView, &BlurHoriShadowPass.myTextureBufferSlot,
        nn::gfx::ImageDimension_2d, nn::gfx::ImageFormat_R32_Float, nn::gfx::DepthStencilFetchMode_DepthComponent );

    // Set up the sampler object
    DEMOGfxInitSampler( &BlurHoriShadowPass.mySampler, &BlurHoriShadowPass.mySamplerSlot, nn::gfx::TextureAddressMode_ClampToEdge,
        nn::gfx::FilterMode_MinPoint_MagPoint_MipPoint, nn::gfx::ComparisonFunction_Always );

    // geometry setup
    BlurHoriShadowPass.posInitData[0] = -1.0f; BlurHoriShadowPass.posInitData[1] = -1.0f;
    BlurHoriShadowPass.posInitData[2] =  0.0f; BlurHoriShadowPass.posInitData[3] =  1.0f;
    BlurHoriShadowPass.posInitData[4] = -1.0f; BlurHoriShadowPass.posInitData[5] =  0.0f;
    BlurHoriShadowPass.posInitData[6] =  1.0f; BlurHoriShadowPass.posInitData[7] =  1.0f;
    BlurHoriShadowPass.posInitData[8] =  0.0f; BlurHoriShadowPass.posInitData[9] = -1.0f;
    BlurHoriShadowPass.posInitData[10] = 1.0f; BlurHoriShadowPass.posInitData[11] = 0.0f;

    BlurHoriShadowPass.texInitData[0] =  0.0f; BlurHoriShadowPass.texInitData[1] =  1.0f;
    BlurHoriShadowPass.texInitData[2] =  1.0f; BlurHoriShadowPass.texInitData[3] =  1.0f;
    BlurHoriShadowPass.texInitData[4] =  1.0f; BlurHoriShadowPass.texInitData[5] =  0.0f;
    BlurHoriShadowPass.texInitData[6] =  0.0f; BlurHoriShadowPass.texInitData[7] =  0.0f;

    BlurHoriShadowPass.idxInitData[0] = 0;
    BlurHoriShadowPass.idxInitData[1] = 1;
    BlurHoriShadowPass.idxInitData[2] = 2;
    BlurHoriShadowPass.idxInitData[3] = 2;
    BlurHoriShadowPass.idxInitData[4] = 3;
    BlurHoriShadowPass.idxInitData[5] = 0;
    BlurHoriShadowPass.numAttrib = 2;
    BlurHoriShadowPass.numIndices = 6;

    // Load shaders
    DEMOGfxLoadShadersFromFile(  &BlurHoriShadowPass.demoShader , 0,  GSH_SHADER_FILE_PENCIL_BLUR_H[g_ShaderFileIdx] );
    // Attribute Location Lookup
    BlurHoriShadowPass.posLoc      = BlurHoriShadowPass.demoShader.GetInterfaceSlot( nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_Input, "a_position");
    BlurHoriShadowPass.texcoordLoc = BlurHoriShadowPass.demoShader.GetInterfaceSlot( nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_Input, "a_texCoord");

    // Sampler Location Lookup
    BlurHoriShadowPass.textureLoc = BlurHoriShadowPass.demoShader.GetInterfaceSlot( nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_Sampler, "s_source" );

    // Attribute Format Setup
    nn::gfx::Pipeline::InfoType pipelineInfo;
    nn::gfx::VertexBufferStateInfo vertexBufferStates[ 2 ];
    nn::gfx::VertexAttributeStateInfo vertexAttributeStates[ 2 ];
    nn::gfx::BlendTargetStateInfo blendTargetStates[ 1 ];
    nn::gfx::ColorTargetStateInfo colorTargetStates[ 1 ];

    pipelineInfo.SetDefault();

    // Setup vertex attributes
    vertexAttributeStates[0].SetDefault();
    vertexAttributeStates[0].SetBufferIndex( BlurHoriShadowPass.posLoc );
    vertexAttributeStates[0].SetShaderSlot( BlurHoriShadowPass.posLoc );
    vertexAttributeStates[0].SetFormat( nn::gfx::AttributeFormat_32_32_32_Float );
    vertexAttributeStates[1].SetDefault();
    vertexAttributeStates[1].SetBufferIndex( BlurHoriShadowPass.texcoordLoc );
    vertexAttributeStates[1].SetShaderSlot( BlurHoriShadowPass.texcoordLoc );
    vertexAttributeStates[1].SetFormat( nn::gfx::AttributeFormat_32_32_Float );

    // Setup vertex buffers
    ASSERT( BlurHoriShadowPass.posLoc < 2 );
    ASSERT( BlurHoriShadowPass.texcoordLoc < 2 );
    vertexBufferStates[BlurHoriShadowPass.posLoc].SetDefault();
    vertexBufferStates[BlurHoriShadowPass.posLoc].SetStride( 3 * sizeof(float) );
    vertexBufferStates[BlurHoriShadowPass.texcoordLoc].SetDefault();
    vertexBufferStates[BlurHoriShadowPass.texcoordLoc].SetStride( 2 * sizeof(float) );

    // Setup BlendControl State
    blendTargetStates[0].SetDefault();
    blendTargetStates[0].SetBlendEnabled( false );
    blendTargetStates[0].SetSourceColorBlendFactor( nn::gfx::BlendFactor_SourceAlpha );
    blendTargetStates[0].SetDestinationColorBlendFactor( nn::gfx::BlendFactor_OneMinusSourceAlpha );
    blendTargetStates[0].SetColorBlendFunction( nn::gfx::BlendFunction_Add );
    blendTargetStates[0].SetSourceAlphaBlendFactor( nn::gfx::BlendFactor_SourceAlpha );
    blendTargetStates[0].SetDestinationAlphaBlendFactor( nn::gfx::BlendFactor_OneMinusSourceAlpha );
    blendTargetStates[0].SetAlphaBlendFunction( nn::gfx::BlendFunction_Add );

    // Setup the render target state.
    colorTargetStates[ 0 ].SetDefault();
    colorTargetStates[ 0 ].SetFormat( pencilTexture[ 0 ].pencilTextureInfo.GetImageFormat() );

    nn::gfx::DepthStencilStateInfo depthStencilStateInfo;
    nn::gfx::RasterizerStateInfo rasterizerStateInfo;
    nn::gfx::BlendStateInfo blendStateInfo;
    nn::gfx::RenderTargetStateInfo renderTargetStateInfo;
    nn::gfx::VertexStateInfo vertexStateInfo;

    SetDefaultPipelineStateInfo(pipelineInfo, blendTargetStates, 1, colorTargetStates, 1,
        vertexAttributeStates, 2, vertexBufferStates, 2, depthStencilStateInfo, rasterizerStateInfo,
        blendStateInfo, renderTargetStateInfo, vertexStateInfo);

    // Disable the depth buffer
    depthStencilStateInfo.SetDepthTestEnabled( false );
    depthStencilStateInfo.SetDepthWriteEnabled( false );
    depthStencilStateInfo.SetDepthComparisonFunction( nn::gfx::ComparisonFunction_Never );

    // Setup shaders and state info
    pipelineInfo.SetShaderPtr(BlurHoriShadowPass.demoShader.GetShader());
    pipelineInfo.SetBlendStateInfo(&blendStateInfo);
    pipelineInfo.SetDepthStencilStateInfo(&depthStencilStateInfo);
    pipelineInfo.SetRasterizerStateInfo(&rasterizerStateInfo);
    pipelineInfo.SetRenderTargetStateInfo(&renderTargetStateInfo);
    pipelineInfo.SetVertexStateInfo(&vertexStateInfo);

    size_t pipelineSize = nn::gfx::Pipeline::GetRequiredMemorySize( pipelineInfo );
    BlurHoriShadowPass.pPipelineData = DEMOGfxAllocMEM2( pipelineSize, nn::gfx::Pipeline::RequiredMemoryInfo_Alignment );
    BlurHoriShadowPass.pipeline.SetMemory( BlurHoriShadowPass.pPipelineData, pipelineSize );
    BlurHoriShadowPass.pipeline.Initialize( &DEMODevice, pipelineInfo );

    // Vertex position buffer
     BlurHoriShadowPass.positionBuffer.Initialize(  sizeof(BlurHoriShadowPass.posInitData),  BlurHoriShadowPass.posInitData,  nn::gfx::GpuAccess_VertexBuffer | nn::gfx::GpuAccess_Read,  0  );

    // Vertex texcoord buffer
     BlurHoriShadowPass.textureCoordinateBuffer.Initialize(  sizeof(BlurHoriShadowPass.texInitData),  BlurHoriShadowPass.texInitData,  nn::gfx::GpuAccess_VertexBuffer | nn::gfx::GpuAccess_Read,  0 );

    // Index buffer
     BlurHoriShadowPass.indexBuffer.Initialize(  sizeof(BlurHoriShadowPass.idxInitData),  BlurHoriShadowPass.idxInitData,  nn::gfx::GpuAccess_IndexBuffer | nn::gfx::GpuAccess_Read,  0 );

    // Render to entire surface
    DEMOGfxSetViewportScissorState( &BlurHoriShadowPass.viewportScissor, &BlurHoriShadowPass.pViewportScissorData,
        0.0f, 0.0f, static_cast< float >( SHADOWMAP_WIDTH ), static_cast< float >( SHADOWMAP_HEIGHT ), 0.0f, 1.0f, static_cast< float >( SHADOWMAP_HEIGHT ), true );
}

// Initializes the Vertical Blur Shadow pass
static void BlurVertShadowPassInit()
{

    // Initialize render color buffers
    DEMOGfxSetupColorView( &ExpShadowPass.myColorBuffer, &BlurVertShadowPass.myColorBufferView,
        nn::gfx::ImageDimension_2d, nn::gfx::ImageFormat_R32_Float );
    DEMOGfxSetupTextureView( &BlurHoriShadowPass.myColorBuffer, &BlurVertShadowPass.myTextureView, &BlurVertShadowPass.myTextureBufferSlot,
        nn::gfx::ImageDimension_2d, nn::gfx::ImageFormat_R32_Float, nn::gfx::DepthStencilFetchMode_DepthComponent );

    DEMOGfxInitSampler(&BlurVertShadowPass.mySampler, &BlurVertShadowPass.mySamplerSlot, nn::gfx::TextureAddressMode_ClampToEdge,
        nn::gfx::FilterMode_MinPoint_MagPoint_MipPoint, nn::gfx::ComparisonFunction_Always );

    // geometry setup
    BlurVertShadowPass.posInitData[0] = -1.0f; BlurVertShadowPass.posInitData[1] = -1.0f;
    BlurVertShadowPass.posInitData[2] =  0.0f; BlurVertShadowPass.posInitData[3] =  1.0f;
    BlurVertShadowPass.posInitData[4] = -1.0f; BlurVertShadowPass.posInitData[5] =  0.0f;
    BlurVertShadowPass.posInitData[6] =  1.0f; BlurVertShadowPass.posInitData[7] =  1.0f;
    BlurVertShadowPass.posInitData[8] =  0.0f; BlurVertShadowPass.posInitData[9] = -1.0f;
    BlurVertShadowPass.posInitData[10] = 1.0f; BlurVertShadowPass.posInitData[11] = 0.0f;

    BlurVertShadowPass.texInitData[0] =  0.0f; BlurVertShadowPass.texInitData[1] =  1.0f;
    BlurVertShadowPass.texInitData[2] =  1.0f; BlurVertShadowPass.texInitData[3] =  1.0f;
    BlurVertShadowPass.texInitData[4] =  1.0f; BlurVertShadowPass.texInitData[5] =  0.0f;
    BlurVertShadowPass.texInitData[6] =  0.0f; BlurVertShadowPass.texInitData[7] =  0.0f;

    BlurVertShadowPass.idxInitData[0] = 0;
    BlurVertShadowPass.idxInitData[1] = 1;
    BlurVertShadowPass.idxInitData[2] = 2;
    BlurVertShadowPass.idxInitData[3] = 2;
    BlurVertShadowPass.idxInitData[4] = 3;
    BlurVertShadowPass.idxInitData[5] = 0;
    BlurVertShadowPass.numAttrib = 2;
    BlurVertShadowPass.numIndices = 6;

    // Load shaders
    DEMOGfxLoadShadersFromFile(  &BlurVertShadowPass.demoShader , 0,  GSH_SHADER_FILE_PENCIL_BLUR_V[g_ShaderFileIdx] );

    // Attribute Location Lookup
    BlurVertShadowPass.posLoc      = BlurVertShadowPass.demoShader.GetInterfaceSlot( nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_Input, "a_position");
    BlurVertShadowPass.texcoordLoc = BlurVertShadowPass.demoShader.GetInterfaceSlot( nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_Input, "a_texCoord");

    // Sampler Location Lookup
    BlurVertShadowPass.textureLoc = BlurVertShadowPass.demoShader.GetInterfaceSlot( nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_Sampler, "s_source" );

    // Attribute Format Setup
    nn::gfx::Pipeline::InfoType pipelineInfo;
    nn::gfx::VertexBufferStateInfo vertexBufferStates[ 2 ];
    nn::gfx::VertexAttributeStateInfo vertexAttributeStates[ 2 ];
    nn::gfx::BlendTargetStateInfo blendTargetStates[ 1 ];
    nn::gfx::ColorTargetStateInfo colorTargetStates[ 1 ];

    pipelineInfo.SetDefault();

    // Setup vertex attributes
    vertexAttributeStates[0].SetDefault();
    vertexAttributeStates[0].SetBufferIndex( BlurVertShadowPass.posLoc );
    vertexAttributeStates[0].SetShaderSlot( BlurVertShadowPass.posLoc );
    vertexAttributeStates[0].SetFormat( nn::gfx::AttributeFormat_32_32_32_Float );
    vertexAttributeStates[1].SetDefault();
    vertexAttributeStates[1].SetBufferIndex( BlurVertShadowPass.texcoordLoc );
    vertexAttributeStates[1].SetShaderSlot( BlurVertShadowPass.texcoordLoc );
    vertexAttributeStates[1].SetFormat( nn::gfx::AttributeFormat_32_32_Float );

    // Setup vertex buffers
    ASSERT( BlurVertShadowPass.posLoc < 2 );
    ASSERT( BlurVertShadowPass.texcoordLoc < 2 );
    vertexBufferStates[BlurVertShadowPass.posLoc].SetDefault();
    vertexBufferStates[BlurVertShadowPass.posLoc].SetStride( 3 * sizeof(float) );
    vertexBufferStates[BlurVertShadowPass.texcoordLoc].SetDefault();
    vertexBufferStates[BlurVertShadowPass.texcoordLoc].SetStride( 2 * sizeof(float) );

    // Setup BlendControl State
    blendTargetStates[0].SetDefault();
    blendTargetStates[0].SetBlendEnabled( false );
    blendTargetStates[0].SetSourceColorBlendFactor( nn::gfx::BlendFactor_SourceAlpha );
    blendTargetStates[0].SetDestinationColorBlendFactor( nn::gfx::BlendFactor_OneMinusSourceAlpha );
    blendTargetStates[0].SetColorBlendFunction( nn::gfx::BlendFunction_Add );
    blendTargetStates[0].SetSourceAlphaBlendFactor( nn::gfx::BlendFactor_SourceAlpha );
    blendTargetStates[0].SetDestinationAlphaBlendFactor( nn::gfx::BlendFactor_OneMinusSourceAlpha );
    blendTargetStates[0].SetAlphaBlendFunction( nn::gfx::BlendFunction_Add );

    // Setup the render target state.
    colorTargetStates[ 0 ].SetDefault();
    colorTargetStates[ 0 ].SetFormat( pencilTexture[ 0 ].pencilTextureInfo.GetImageFormat() );

    nn::gfx::DepthStencilStateInfo depthStencilStateInfo;
    nn::gfx::RasterizerStateInfo rasterizerStateInfo;
    nn::gfx::BlendStateInfo blendStateInfo;
    nn::gfx::RenderTargetStateInfo renderTargetStateInfo;
    nn::gfx::VertexStateInfo vertexStateInfo;

    SetDefaultPipelineStateInfo(pipelineInfo, blendTargetStates, 1, colorTargetStates, 1,
        vertexAttributeStates, 2, vertexBufferStates, 2, depthStencilStateInfo, rasterizerStateInfo,
        blendStateInfo, renderTargetStateInfo, vertexStateInfo);

    // Disable the depth buffer
    depthStencilStateInfo.SetDepthTestEnabled( false );
    depthStencilStateInfo.SetDepthWriteEnabled( false );
    depthStencilStateInfo.SetDepthComparisonFunction( nn::gfx::ComparisonFunction_Never );

    // Setup shaders and state info
    pipelineInfo.SetShaderPtr(BlurVertShadowPass.demoShader.GetShader());
    pipelineInfo.SetBlendStateInfo(&blendStateInfo);
    pipelineInfo.SetDepthStencilStateInfo(&depthStencilStateInfo);
    pipelineInfo.SetRasterizerStateInfo(&rasterizerStateInfo);
    pipelineInfo.SetRenderTargetStateInfo(&renderTargetStateInfo);
    pipelineInfo.SetVertexStateInfo(&vertexStateInfo);

    size_t pipelineSize = nn::gfx::Pipeline::GetRequiredMemorySize( pipelineInfo );
    BlurVertShadowPass.pPipelineData = DEMOGfxAllocMEM2( pipelineSize,
        nn::gfx::Pipeline::RequiredMemoryInfo_Alignment );
    BlurVertShadowPass.pipeline.SetMemory( BlurVertShadowPass.pPipelineData, pipelineSize );
    BlurVertShadowPass.pipeline.Initialize( &DEMODevice, pipelineInfo );

    // Vertex position buffer
     BlurVertShadowPass.positionBuffer.Initialize(  sizeof(BlurVertShadowPass.posInitData),  BlurVertShadowPass.posInitData,  nn::gfx::GpuAccess_VertexBuffer | nn::gfx::GpuAccess_Read,  0  );

    // Vertex texcoord buffer
     BlurVertShadowPass.textureCoordinateBuffer.Initialize(  sizeof(BlurVertShadowPass.texInitData),  BlurVertShadowPass.texInitData,  nn::gfx::GpuAccess_VertexBuffer | nn::gfx::GpuAccess_Read,  0  );

    // Index buffer
     BlurVertShadowPass.indexBuffer.Initialize(  sizeof(BlurVertShadowPass.idxInitData),  BlurVertShadowPass.idxInitData,  nn::gfx::GpuAccess_IndexBuffer | nn::gfx::GpuAccess_Read,  0  );

    // Render to entire surface
    DEMOGfxSetViewportScissorState( &BlurVertShadowPass.viewportScissor, &BlurVertShadowPass.pViewportScissorData,
        0.0f, 0.0f, static_cast< float >( SHADOWMAP_WIDTH ), static_cast< float >( SHADOWMAP_HEIGHT ), 0.0f, 1.0f, static_cast< float >( SHADOWMAP_HEIGHT ), true );
}

// Initialized the Geometry pass
static void GeometryPassInit()
{
    u32 i;

    // Initialize vertex uniforms
    // Initialize this later
     GeometryPass.vertexUniformBuffer.Initialize(  sizeof( struct GeometryPassVertexUniforms ),  NULL,  nn::gfx::GpuAccess_Read | nn::gfx::GpuAccess_ConstantBuffer,  0  );

    GeometryPass.pixelUniformBuffer.Initialize(  sizeof(struct GeometryPassPixelUniforms),  NULL,  nn::gfx::GpuAccess_Read | nn::gfx::GpuAccess_ConstantBuffer,  0 );

    GeometryPassPixelUniforms*pPixelUniformData = GeometryPass.pixelUniformBuffer.Map< GeometryPassPixelUniforms >();
    pPixelUniformData->color.x = 0.9f;
    pPixelUniformData->color.y = 0.9f;
    pPixelUniformData->color.z = 0.9f;
    pPixelUniformData->color.w = 80.0f / 255.0f;
#if NN_GFX_IS_TARGET_GX
    GX2EndianSwap( pPixelUniformData, sizeof( *pPixelUniformData ) );
#endif
    GeometryPass.pixelUniformBuffer.Unmap();

    // load models
    for(i = 0; i < NUM_MODEL; ++i)
    {
        // Position buffer
        GeometryPass.posBuf[i] = (f32 *)DEMOGfxLoadModelFile(MODEL_FILES[i][0], &modelLen[i][0]);
         GeometryPass.positionBuffers[i].Initialize(  modelLen[i][0],  GeometryPass.posBuf[i],  nn::gfx::GpuAccess_Read | nn::gfx::GpuAccess_VertexBuffer,  0  );

        // Color(Normal) buffer
        GeometryPass.nrmBuf[i] = (f32 *)DEMOGfxLoadModelFile(MODEL_FILES[i][1], &modelLen[i][1]);
         GeometryPass.normalBuffers[i].Initialize(  modelLen[i][1],  GeometryPass.nrmBuf[i],  nn::gfx::GpuAccess_Read | nn::gfx::GpuAccess_VertexBuffer,  0  );

        // Index buffer
        GeometryPass.idxBuf[i] = (u32 *)DEMOGfxLoadModelFile(MODEL_FILES[i][2], &modelLen[i][2]);
         GeometryPass.indexBuffers[i].Initialize(  modelLen[i][2],  GeometryPass.idxBuf[i],  nn::gfx::GpuAccess_Read | nn::gfx::GpuAccess_IndexBuffer,  0  );
    };

    // Load shader binary to memory
    DEMOGfxLoadShadersFromFile(  &GeometryPass.demoShader , 0,  GSH_SHADER_FILE_PENCIL_GEOMETRY[g_ShaderFileIdx] );

    // Attribute Location Lookup
    GeometryPass.posLoc = GeometryPass.demoShader.GetInterfaceSlot( nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_Input, "a_position" );
    GeometryPass.nrmLoc = GeometryPass.demoShader.GetInterfaceSlot( nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_Input, "a_normal" );

    // Uniform Location Lookup
    GeometryPass.vertexUniformBufferLoc = GeometryPass.demoShader.GetInterfaceSlot( nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_ConstantBuffer, "u_uniforms" );
    GeometryPass.pixelUniformBufferLoc = GeometryPass.demoShader.GetInterfaceSlot( nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_ConstantBuffer, "u_pixelUniforms" );

    // Attribute Format Setup
    nn::gfx::Pipeline::InfoType pipelineInfo;
    nn::gfx::VertexBufferStateInfo vertexBufferStates[ 2 ];
    nn::gfx::VertexAttributeStateInfo vertexAttributeStates[ 2 ];
    nn::gfx::BlendTargetStateInfo blendTargetStates[ 2 ];
    nn::gfx::ColorTargetStateInfo colorTargetStates[ 2 ];

    pipelineInfo.SetDefault();

    // Setup vertex attributes
    vertexAttributeStates[0].SetDefault();
    vertexAttributeStates[0].SetBufferIndex( GeometryPass.posLoc );
    vertexAttributeStates[0].SetShaderSlot( GeometryPass.posLoc );
    vertexAttributeStates[0].SetFormat( nn::gfx::AttributeFormat_32_32_32_Float );
    vertexAttributeStates[1].SetDefault();
    vertexAttributeStates[1].SetBufferIndex( GeometryPass.nrmLoc );
    vertexAttributeStates[1].SetShaderSlot( GeometryPass.nrmLoc );
    vertexAttributeStates[1].SetFormat( nn::gfx::AttributeFormat_32_32_32_Float );

    // Setup vertex buffers
    ASSERT( GeometryPass.posLoc < 2 );
    ASSERT( GeometryPass.nrmLoc < 2 );
    vertexBufferStates[GeometryPass.posLoc].SetDefault();
    vertexBufferStates[GeometryPass.posLoc].SetStride( 3 * sizeof(float) );
    vertexBufferStates[GeometryPass.nrmLoc].SetDefault();
    vertexBufferStates[GeometryPass.nrmLoc].SetStride( 3 * sizeof(float) );

    // Setup BlendControl State
    blendTargetStates[0].SetDefault();
    blendTargetStates[0].SetBlendEnabled( false );
    blendTargetStates[0].SetSourceColorBlendFactor( nn::gfx::BlendFactor_SourceAlpha );
    blendTargetStates[0].SetDestinationColorBlendFactor( nn::gfx::BlendFactor_OneMinusSourceAlpha );
    blendTargetStates[0].SetColorBlendFunction( nn::gfx::BlendFunction_Add );
    blendTargetStates[0].SetSourceAlphaBlendFactor( nn::gfx::BlendFactor_SourceAlpha );
    blendTargetStates[0].SetDestinationAlphaBlendFactor( nn::gfx::BlendFactor_OneMinusSourceAlpha );
    blendTargetStates[0].SetAlphaBlendFunction( nn::gfx::BlendFunction_Add );

    blendTargetStates[1].SetDefault();
    blendTargetStates[1].SetBlendEnabled( false );
    blendTargetStates[1].SetSourceColorBlendFactor( nn::gfx::BlendFactor_SourceAlpha );
    blendTargetStates[1].SetDestinationColorBlendFactor( nn::gfx::BlendFactor_OneMinusSourceAlpha );
    blendTargetStates[1].SetColorBlendFunction( nn::gfx::BlendFunction_Add );
    blendTargetStates[1].SetSourceAlphaBlendFactor( nn::gfx::BlendFactor_SourceAlpha );
    blendTargetStates[1].SetDestinationAlphaBlendFactor( nn::gfx::BlendFactor_OneMinusSourceAlpha );
    blendTargetStates[1].SetAlphaBlendFunction( nn::gfx::BlendFunction_Add );


    // Setup the render target state.
    colorTargetStates[ 0 ].SetDefault();
    colorTargetStates[ 0 ].SetFormat( nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm );
    colorTargetStates[ 1 ].SetDefault();
    colorTargetStates[ 1 ].SetFormat( nn::gfx::ImageFormat_R8_G8_B8_A8_Snorm ); // NOTE: This is less precise than the original but nn::gfx::ImageFormat_R10_G10_B10_A2_Snorm doesn't exist

    nn::gfx::DepthStencilStateInfo depthStencilStateInfo;
    nn::gfx::RasterizerStateInfo rasterizerStateInfo;
    nn::gfx::BlendStateInfo blendStateInfo;
    nn::gfx::RenderTargetStateInfo renderTargetStateInfo;
    nn::gfx::VertexStateInfo vertexStateInfo;

    SetDefaultPipelineStateInfo(pipelineInfo, blendTargetStates, 2, colorTargetStates, 2,
        vertexAttributeStates, 2, vertexBufferStates, 2, depthStencilStateInfo, rasterizerStateInfo,
        blendStateInfo, renderTargetStateInfo, vertexStateInfo);

    // Setup shaders and state info
    pipelineInfo.SetShaderPtr(GeometryPass.demoShader.GetShader());
    pipelineInfo.SetBlendStateInfo(&blendStateInfo);
    pipelineInfo.SetDepthStencilStateInfo(&depthStencilStateInfo);
    pipelineInfo.SetRasterizerStateInfo(&rasterizerStateInfo);
    pipelineInfo.SetRenderTargetStateInfo(&renderTargetStateInfo);
    pipelineInfo.SetVertexStateInfo(&vertexStateInfo);

    size_t pipelineSize = nn::gfx::Pipeline::GetRequiredMemorySize( pipelineInfo );
    GeometryPass.pPipelineData = DEMOGfxAllocMEM2( pipelineSize, nn::gfx::Pipeline::RequiredMemoryInfo_Alignment );
    GeometryPass.pipeline.SetMemory( GeometryPass.pPipelineData, pipelineSize );
    GeometryPass.pipeline.Initialize( &DEMODevice, pipelineInfo );

    // Initialize render color buffers
    DEMOGfxSetupTextureBuffer(&GeometryPass.myColorBuffer[0], NULL, NULL, &GeometryPass.myColorBufferView[0],
        NULL, &GeometryPass.myColorBufferData[0], RENDERTARGET_WIDTH, RENDERTARGET_HEIGHT, 1, 1,
        nn::gfx::ImageDimension_2d, nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm, nn::gfx::DepthStencilFetchMode_DepthComponent, 1 );

    DEMOGfxSetupTextureBuffer(&GeometryPass.myColorBuffer[1], NULL, NULL, &GeometryPass.myColorBufferView[1],
        NULL, &GeometryPass.myColorBufferData[1], RENDERTARGET_WIDTH, RENDERTARGET_HEIGHT, 1, 1,
        nn::gfx::ImageDimension_2d, nn::gfx::ImageFormat_R8_G8_B8_A8_Snorm, // NOTE: This is less precise than the original but nn::gfx::ImageFormat_R10_G10_B10_A2_Snorm doesn't exist
        nn::gfx::DepthStencilFetchMode_DepthComponent, 2 );

    // Initialize render depth buffer
    DEMOGfxSetupDepthView(&DEMODepthBuffer, &GeometryPass.myDepthBufferView, nn::gfx::ImageDimension_2d);

#if NN_GFX_IS_TARGET_GX
    // This optimization is only for Cafe
    GX2InitDepthBufferRangeBase( reinterpret_cast< GX2DepthBuffer* >( &GeometryPass.myDepthBufferView.ToData()->gx2DepthBuffer ), GX2_ZRANGE_BASE_ZMAX);
#endif

    DEMOGfxSetViewportScissorState( &GeometryPass.viewportScissor, &GeometryPass.pViewportScissorData,
        0.0f, 0.0f, static_cast< float >( RENDERTARGET_WIDTH ), static_cast< float >( RENDERTARGET_HEIGHT ), 0.0f, 1.0f, static_cast< float >( RENDERTARGET_HEIGHT ), true );
}

// Initializes the Outline pass
static void OutlinePassInit()
{
    // Initialize render color buffers
    DEMOGfxSetupTextureBuffer( &OutlinePass.myColorBuffer, NULL, NULL, &OutlinePass.myColorBufferView,
        NULL, &OutlinePass.myColorBufferData, RENDERTARGET_WIDTH, RENDERTARGET_HEIGHT, 1, 1,
        nn::gfx::ImageDimension_2d, nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm, nn::gfx::DepthStencilFetchMode_DepthComponent, 0 );

    // Initialize texture buffer

    // DEMODepthBuffer is used by GeometryPass.myDepthBufferView
    DEMOGfxSetupTextureView(&DEMODepthBuffer, &OutlinePass.myTextureBufferView[0], &OutlinePass.myTextureBufferSlot[0],
        nn::gfx::ImageDimension_2d, nn::gfx::ImageFormat_D32_Float, nn::gfx::DepthStencilFetchMode_DepthComponent );

    DEMOGfxSetupTextureView(&GeometryPass.myColorBuffer[1], &OutlinePass.myTextureBufferView[1], &OutlinePass.myTextureBufferSlot[1],
        nn::gfx::ImageDimension_2d, nn::gfx::ImageFormat_R8_G8_B8_A8_Snorm, nn::gfx::DepthStencilFetchMode_DepthComponent ); // NOTE: This is less precise than the original but nn::gfx::ImageFormat_R10_G10_B10_A2_Snorm doesn't exist

    // Set up the sampler object
    DEMOGfxInitSampler(&OutlinePass.mySampler, &OutlinePass.mySamplerSlot, nn::gfx::TextureAddressMode_ClampToEdge,
        nn::gfx::FilterMode_MinPoint_MagPoint_MipPoint, nn::gfx::ComparisonFunction_Always );

    // geometry setup
    OutlinePass.posInitData[0] = -1.0f; OutlinePass.posInitData[1] = -1.0f;
    OutlinePass.posInitData[2] =  0.0f; OutlinePass.posInitData[3] =  1.0f;
    OutlinePass.posInitData[4] = -1.0f; OutlinePass.posInitData[5] =  0.0f;
    OutlinePass.posInitData[6] =  1.0f; OutlinePass.posInitData[7] =  1.0f;
    OutlinePass.posInitData[8] =  0.0f; OutlinePass.posInitData[9] = -1.0f;
    OutlinePass.posInitData[10] = 1.0f; OutlinePass.posInitData[11] = 0.0f;

    OutlinePass.texInitData[0] =  0.0f; OutlinePass.texInitData[1] =  1.0f;
    OutlinePass.texInitData[2] =  1.0f; OutlinePass.texInitData[3] =  1.0f;
    OutlinePass.texInitData[4] =  1.0f; OutlinePass.texInitData[5] =  0.0f;
    OutlinePass.texInitData[6] =  0.0f; OutlinePass.texInitData[7] =  0.0f;

    OutlinePass.idxInitData[0] = 0;
    OutlinePass.idxInitData[1] = 1;
    OutlinePass.idxInitData[2] = 2;
    OutlinePass.idxInitData[3] = 2;
    OutlinePass.idxInitData[4] = 3;
    OutlinePass.idxInitData[5] = 0;
    OutlinePass.numAttrib = 2;
    OutlinePass.numIndices = 6;

    // Load shaders
    DEMOGfxLoadShadersFromFile(  &OutlinePass.demoShader , 0,  GSH_SHADER_FILE_PENCIL_OUTLINE[g_ShaderFileIdx] );

    // Attribute Location Lookup
    OutlinePass.posLoc = OutlinePass.demoShader.GetInterfaceSlot( nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_Input, "a_position" );
    OutlinePass.texcoordLoc = OutlinePass.demoShader.GetInterfaceSlot( nn::gfx::ShaderStage_Vertex,  nn::gfx::ShaderInterfaceType_Input, "a_texCoord" );

    // Sampler Location Lookup
    OutlinePass.textureLoc[ 0 ] = OutlinePass.demoShader.GetInterfaceSlot( nn::gfx::ShaderStage_Pixel,  nn::gfx::ShaderInterfaceType_Sampler, "s_depth" );
    OutlinePass.textureLoc[ 1 ] = OutlinePass.demoShader.GetInterfaceSlot( nn::gfx::ShaderStage_Pixel,  nn::gfx::ShaderInterfaceType_Sampler, "s_normal" );

    // Uniform Location lookup
    OutlinePass.surfaceSizeLoc = OutlinePass.demoShader.GetInterfaceSlot( nn::gfx::ShaderStage_Pixel,  nn::gfx::ShaderInterfaceType_ConstantBuffer, "u_uniforms" );

    // Attribute Format Setup
    nn::gfx::Pipeline::InfoType pipelineInfo;
    nn::gfx::VertexBufferStateInfo vertexBufferStates[ 2 ];
    nn::gfx::VertexAttributeStateInfo vertexAttributeStates[ 2 ];
    nn::gfx::BlendTargetStateInfo blendTargetStates[ 1 ];
    nn::gfx::ColorTargetStateInfo colorTargetStates[ 1 ];

    pipelineInfo.SetDefault();

    // Setup vertex attributes
    vertexAttributeStates[0].SetDefault();
    vertexAttributeStates[0].SetBufferIndex( OutlinePass.posLoc );
    vertexAttributeStates[0].SetShaderSlot( OutlinePass.posLoc );
    vertexAttributeStates[0].SetFormat( nn::gfx::AttributeFormat_32_32_32_Float );
    vertexAttributeStates[1].SetDefault();
    vertexAttributeStates[1].SetBufferIndex( OutlinePass.texcoordLoc );
    vertexAttributeStates[1].SetShaderSlot( OutlinePass.texcoordLoc );
    vertexAttributeStates[1].SetFormat( nn::gfx::AttributeFormat_32_32_Float );

    // Setup vertex buffers
    ASSERT( OutlinePass.posLoc < 2 );
    ASSERT( OutlinePass.texcoordLoc < 2 );
    vertexBufferStates[OutlinePass.posLoc].SetDefault();
    vertexBufferStates[OutlinePass.posLoc].SetStride( 3 * sizeof(float) );
    vertexBufferStates[OutlinePass.texcoordLoc].SetDefault();
    vertexBufferStates[OutlinePass.texcoordLoc].SetStride( 2 * sizeof(float) );

    // Setup BlendControl State
    blendTargetStates[0].SetDefault();
    blendTargetStates[0].SetBlendEnabled( false );
    blendTargetStates[0].SetSourceColorBlendFactor( nn::gfx::BlendFactor_SourceAlpha );
    blendTargetStates[0].SetDestinationColorBlendFactor( nn::gfx::BlendFactor_OneMinusSourceAlpha );
    blendTargetStates[0].SetColorBlendFunction( nn::gfx::BlendFunction_Add );
    blendTargetStates[0].SetSourceAlphaBlendFactor( nn::gfx::BlendFactor_SourceAlpha );
    blendTargetStates[0].SetDestinationAlphaBlendFactor( nn::gfx::BlendFactor_OneMinusSourceAlpha );
    blendTargetStates[0].SetAlphaBlendFunction( nn::gfx::BlendFunction_Add );

    // Setup the render target state.
    colorTargetStates[ 0 ].SetDefault();
    colorTargetStates[ 0 ].SetFormat( pencilTexture[ 0 ].pencilTextureInfo.GetImageFormat() );

    nn::gfx::DepthStencilStateInfo depthStencilStateInfo;
    nn::gfx::RasterizerStateInfo rasterizerStateInfo;
    nn::gfx::BlendStateInfo blendStateInfo;
    nn::gfx::RenderTargetStateInfo renderTargetStateInfo;
    nn::gfx::VertexStateInfo vertexStateInfo;

    SetDefaultPipelineStateInfo(pipelineInfo, blendTargetStates, 1, colorTargetStates, 1,
        vertexAttributeStates, 2, vertexBufferStates, 2, depthStencilStateInfo, rasterizerStateInfo,
        blendStateInfo, renderTargetStateInfo, vertexStateInfo);

    // Disable the depth buffer
    depthStencilStateInfo.SetDepthTestEnabled( false );
    depthStencilStateInfo.SetDepthWriteEnabled( false );
    depthStencilStateInfo.SetDepthComparisonFunction( nn::gfx::ComparisonFunction_Never );

    // Setup shaders and state info
    pipelineInfo.SetShaderPtr(OutlinePass.demoShader.GetShader());
    pipelineInfo.SetBlendStateInfo(&blendStateInfo);
    pipelineInfo.SetDepthStencilStateInfo(&depthStencilStateInfo);
    pipelineInfo.SetRasterizerStateInfo(&rasterizerStateInfo);
    pipelineInfo.SetRenderTargetStateInfo(&renderTargetStateInfo);
    pipelineInfo.SetVertexStateInfo(&vertexStateInfo);

    size_t pipelineSize = nn::gfx::Pipeline::GetRequiredMemorySize( pipelineInfo );
    OutlinePass.pPipelineData = DEMOGfxAllocMEM2( pipelineSize, nn::gfx::Pipeline::RequiredMemoryInfo_Alignment );
    OutlinePass.pipeline.SetMemory( OutlinePass.pPipelineData, pipelineSize );
    OutlinePass.pipeline.Initialize( &DEMODevice, pipelineInfo );

    // Vertex position buffer
     OutlinePass.positionBuffer.Initialize(  sizeof(OutlinePass.posInitData),  OutlinePass.posInitData,  nn::gfx::GpuAccess_Read | nn::gfx::GpuAccess_VertexBuffer,  0  );

    // Vertex texcoord buffer
     OutlinePass.textureCoordinateBuffer.Initialize(  sizeof(OutlinePass.texInitData),  OutlinePass.texInitData,  nn::gfx::GpuAccess_Read | nn::gfx::GpuAccess_VertexBuffer,  0  );

    // Index buffer
     OutlinePass.indexBuffer.Initialize(  sizeof(OutlinePass.idxInitData),  OutlinePass.idxInitData,  nn::gfx::GpuAccess_Read | nn::gfx::GpuAccess_IndexBuffer,  0  );

    // Render to entire surface
    DEMOGfxSetViewportScissorState( &OutlinePass.viewportScissor, &OutlinePass.pViewportScissorData,
        0.0f, 0.0f, static_cast< float >( SURFACE_WIDTH ), static_cast< float >( SURFACE_HEIGHT ), 0.0f, 1.0f, static_cast< float >( SURFACE_HEIGHT ), true );

    // Set uniforms
    struct OutlineUniforms temp;
    memset(&temp, 0, sizeof(temp));
    temp.surfaceSize.x = f32(SURFACE_WIDTH);
    temp.surfaceSize.y = f32(SURFACE_HEIGHT);
    temp.surfaceSize.z = 0.0f;
    temp.surfaceSize.w = 0.0f;
#if NN_GFX_IS_TARGET_GX
    GX2EndianSwap( &temp, sizeof(struct OutlineUniforms) );
#endif
     OutlinePass.uniformBuffer.Initialize(  sizeof(temp),  &temp,  nn::gfx::GpuAccess_ConstantBuffer | nn::gfx::GpuAccess_Read,  0  );
} //NOLINT(readability/fn_size)

static void InitQuadGeometry(float* posInitData, float* texInitData, uint32_t* idxInitData, int* pNumAttrib, int* pNumIndices)
{
    // geometry setup
    posInitData[0] = -1.0f; posInitData[1] = -1.0f;
    posInitData[2] =  0.0f; posInitData[3] =  1.0f;
    posInitData[4] = -1.0f; posInitData[5] =  0.0f;
    posInitData[6] =  1.0f; posInitData[7] =  1.0f;
    posInitData[8] =  0.0f; posInitData[9] = -1.0f;
    posInitData[10] = 1.0f; posInitData[11] = 0.0f;

    texInitData[0] =  0.0f; texInitData[1] =  1.0f;
    texInitData[2] =  1.0f; texInitData[3] =  1.0f;
    texInitData[4] =  1.0f; texInitData[5] =  0.0f;
    texInitData[6] =  0.0f; texInitData[7] =  0.0f;

    idxInitData[0] = 0;
    idxInitData[1] = 1;
    idxInitData[2] = 2;
    idxInitData[3] = 2;
    idxInitData[4] = 3;
    idxInitData[5] = 0;
    *pNumAttrib = 2;
    *pNumIndices = 6;
}

// Initializes the Lighting pass
static void LightingPassInit()
{
    // Initialize render color buffers
    DEMOGfxSetupTextureBuffer( &LightingPass.myColorBuffer, NULL, NULL,
        &LightingPass.myColorBufferView, NULL, &LightingPass.myColorBufferData,
        RENDERTARGET_WIDTH, RENDERTARGET_HEIGHT, 1, 1, nn::gfx::ImageDimension_2d,
        nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm, nn::gfx::DepthStencilFetchMode_DepthComponent, 0 );

    // initialize uniforms
    LightingPass.uniformBuffer.Initialize( sizeof( struct LightingUniforms ), NULL, nn::gfx::GpuAccess_ConstantBuffer | nn::gfx::GpuAccess_Read, 0 );

    struct LightingUniforms* pLightingUniforms = LightingPass.uniformBuffer.Map< struct LightingUniforms >( );
    pLightingUniforms->ambientColor.x = 0.0f;
    pLightingUniforms->ambientColor.y = 0.0f;
    pLightingUniforms->ambientColor.z = 0.0f;
    pLightingUniforms->ambientColor.w = 1.0f;

    pLightingUniforms->lightColor.x = 1.0f;
    pLightingUniforms->lightColor.y = 1.0f;
    pLightingUniforms->lightColor.z = 1.0f;
    pLightingUniforms->lightColor.w = 80.0f / 255.0f; // attenuation / 255.0f

#if NN_GFX_IS_TARGET_GX
    // Since this is the first time writing this it is safe to swap the whole thing
    GX2EndianSwap( pLightingUniforms, sizeof( *pLightingUniforms ) );
#endif
    LightingPass.uniformBuffer.Unmap();

    // Initialize texture buffers
    DEMOGfxSetupTextureView( &GeometryPass.myColorBuffer[0], &LightingPass.myTextureBuffer[0], &LightingPass.myTextureBufferSlot[0],
        nn::gfx::ImageDimension_2d, nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm, nn::gfx::DepthStencilFetchMode_DepthComponent );
    DEMOGfxSetupTextureView( &GeometryPass.myColorBuffer[1], &LightingPass.myTextureBuffer[1], &LightingPass.myTextureBufferSlot[1],
        nn::gfx::ImageDimension_2d, nn::gfx::ImageFormat_R8_G8_B8_A8_Snorm, nn::gfx::DepthStencilFetchMode_DepthComponent ); // NOTE: This is less precise than the original but nn::gfx::ImageFormat_R10_G10_B10_A2_Snorm doesn't exist
    DEMOGfxSetupTextureView( &DEMODepthBuffer, &LightingPass.myTextureBuffer[2], &LightingPass.myTextureBufferSlot[2],
        nn::gfx::ImageDimension_2d, nn::gfx::ImageFormat_D32_Float, nn::gfx::DepthStencilFetchMode_DepthComponent );
    DEMOGfxSetupTextureView( &ShadowPass.myDepthBuffer, &LightingPass.myTextureBuffer[3], &LightingPass.myTextureBufferSlot[3],
        nn::gfx::ImageDimension_2d, nn::gfx::ImageFormat_D32_Float, nn::gfx::DepthStencilFetchMode_DepthComponent );

    // BlurVertShadowPass.myColorBufferView maps to ExpShadowPass.myColorBuffer
    DEMOGfxSetupTextureView( &ExpShadowPass.myColorBuffer, &LightingPass.myTextureBuffer[4], &LightingPass.myTextureBufferSlot[4],
        nn::gfx::ImageDimension_2d, nn::gfx::ImageFormat_R32_Float, nn::gfx::DepthStencilFetchMode_DepthComponent );

    // Set up the sampler objects
    DEMOGfxInitSampler(&LightingPass.mySampler, &LightingPass.mySamplerSlot, nn::gfx::TextureAddressMode_ClampToEdge,
        nn::gfx::FilterMode_MinPoint_MagPoint_MipPoint, nn::gfx::ComparisonFunction_Always );

    // Depth sampler needs special setup.
    DEMOGfxInitSampler(&LightingPass.myDepthSampler, &LightingPass.myDepthSamplerSlot, nn::gfx::TextureAddressMode_ClampToEdge,
        nn::gfx::FilterMode_MinLinear_MagLinear_MipPoint, nn::gfx::ComparisonFunction_Less );

    // geometry setup
    InitQuadGeometry(LightingPass.posInitData, LightingPass.texInitData, LightingPass.idxInitData, &LightingPass.numAttrib, &LightingPass.numIndices );

    // Load shaders
    DEMOGfxLoadShadersFromFile(  &LightingPass.demoShader , 0,  GSH_SHADER_FILE_PENCIL_LIGHTING[g_ShaderFileIdx] );

    // Attribute Location Lookup
    LightingPass.posLoc = LightingPass.demoShader.GetInterfaceSlot( nn::gfx::ShaderStage_Vertex,  nn::gfx::ShaderInterfaceType_Input, "a_position" );
    LightingPass.texcoordLoc = LightingPass.demoShader.GetInterfaceSlot( nn::gfx::ShaderStage_Vertex,  nn::gfx::ShaderInterfaceType_Input, "a_texCoord" );

    // Sampler Location Lookup
    LightingPass.textureLoc[ 0 ] = LightingPass.demoShader.GetInterfaceSlot( nn::gfx::ShaderStage_Pixel,  nn::gfx::ShaderInterfaceType_Sampler, "s_diffuse" );
    LightingPass.textureLoc[ 1 ] = LightingPass.demoShader.GetInterfaceSlot( nn::gfx::ShaderStage_Pixel,  nn::gfx::ShaderInterfaceType_Sampler, "s_normal" );
    LightingPass.textureLoc[ 2 ] = LightingPass.demoShader.GetInterfaceSlot( nn::gfx::ShaderStage_Pixel,  nn::gfx::ShaderInterfaceType_Sampler, "s_depth" );
    LightingPass.textureLoc[ 3 ] = LightingPass.demoShader.GetInterfaceSlot( nn::gfx::ShaderStage_Pixel,  nn::gfx::ShaderInterfaceType_Sampler, "s_shadow" );
    LightingPass.textureLoc[ 4 ] = LightingPass.demoShader.GetInterfaceSlot( nn::gfx::ShaderStage_Pixel,  nn::gfx::ShaderInterfaceType_Sampler, "s_expshadow" );

    // Uniform Location Lookup
    LightingPass.uniformBlockLoc = LightingPass.demoShader.GetInterfaceSlot( nn::gfx::ShaderStage_Pixel,  nn::gfx::ShaderInterfaceType_ConstantBuffer, "u_uniforms" );

    // Attribute Format Setup
    nn::gfx::Pipeline::InfoType pipelineInfo;
    nn::gfx::VertexBufferStateInfo vertexBufferStates[ 2 ];
    nn::gfx::VertexAttributeStateInfo vertexAttributeStates[ 2 ];
    nn::gfx::BlendTargetStateInfo blendTargetStates[ 1 ];
    nn::gfx::ColorTargetStateInfo colorTargetStates[ 1 ];

    pipelineInfo.SetDefault();

    // Setup vertex attributes
    vertexAttributeStates[0].SetDefault();
    vertexAttributeStates[0].SetBufferIndex( LightingPass.posLoc );
    vertexAttributeStates[0].SetShaderSlot( LightingPass.posLoc );
    vertexAttributeStates[0].SetFormat( nn::gfx::AttributeFormat_32_32_32_Float );
    vertexAttributeStates[1].SetDefault();
    vertexAttributeStates[1].SetBufferIndex( LightingPass.texcoordLoc );
    vertexAttributeStates[1].SetShaderSlot( LightingPass.texcoordLoc );
    vertexAttributeStates[1].SetFormat( nn::gfx::AttributeFormat_32_32_Float );

    // Setup vertex buffers
    ASSERT( LightingPass.posLoc < 2 );
    ASSERT( LightingPass.texcoordLoc < 2 );
    vertexBufferStates[LightingPass.posLoc].SetDefault();
    vertexBufferStates[LightingPass.posLoc].SetStride( 3 * sizeof(float) );
    vertexBufferStates[LightingPass.texcoordLoc].SetDefault();
    vertexBufferStates[LightingPass.texcoordLoc].SetStride( 2 * sizeof(float) );

    // Setup BlendControl State
    blendTargetStates[0].SetDefault();
    blendTargetStates[0].SetBlendEnabled( false );
    blendTargetStates[0].SetSourceColorBlendFactor( nn::gfx::BlendFactor_SourceAlpha );
    blendTargetStates[0].SetDestinationColorBlendFactor( nn::gfx::BlendFactor_OneMinusSourceAlpha );
    blendTargetStates[0].SetColorBlendFunction( nn::gfx::BlendFunction_Add );
    blendTargetStates[0].SetSourceAlphaBlendFactor( nn::gfx::BlendFactor_SourceAlpha );
    blendTargetStates[0].SetDestinationAlphaBlendFactor( nn::gfx::BlendFactor_OneMinusSourceAlpha );
    blendTargetStates[0].SetAlphaBlendFunction( nn::gfx::BlendFunction_Add );

    // Setup the render target state.
    colorTargetStates[ 0 ].SetDefault();
    colorTargetStates[ 0 ].SetFormat( pencilTexture[ 0 ].pencilTextureInfo.GetImageFormat() );

    nn::gfx::DepthStencilStateInfo depthStencilStateInfo;
    nn::gfx::RasterizerStateInfo rasterizerStateInfo;
    nn::gfx::BlendStateInfo blendStateInfo;
    nn::gfx::RenderTargetStateInfo renderTargetStateInfo;
    nn::gfx::VertexStateInfo vertexStateInfo;

    SetDefaultPipelineStateInfo(pipelineInfo, blendTargetStates, 1, colorTargetStates, 1,
        vertexAttributeStates, 2, vertexBufferStates, 2, depthStencilStateInfo, rasterizerStateInfo,
        blendStateInfo, renderTargetStateInfo, vertexStateInfo);

    // Disable the depth buffer
    depthStencilStateInfo.SetDepthTestEnabled( false );
    depthStencilStateInfo.SetDepthWriteEnabled( false );
    depthStencilStateInfo.SetDepthComparisonFunction( nn::gfx::ComparisonFunction_Never );

    // Setup shaders and state info
    pipelineInfo.SetShaderPtr(LightingPass.demoShader.GetShader());
    pipelineInfo.SetBlendStateInfo(&blendStateInfo);
    pipelineInfo.SetDepthStencilStateInfo(&depthStencilStateInfo);
    pipelineInfo.SetRasterizerStateInfo(&rasterizerStateInfo);
    pipelineInfo.SetRenderTargetStateInfo(&renderTargetStateInfo);
    pipelineInfo.SetVertexStateInfo(&vertexStateInfo);

    size_t pipelineSize = nn::gfx::Pipeline::GetRequiredMemorySize( pipelineInfo );
    LightingPass.pPipelineData = DEMOGfxAllocMEM2( pipelineSize, nn::gfx::Pipeline::RequiredMemoryInfo_Alignment );
    LightingPass.pipeline.SetMemory( LightingPass.pPipelineData, pipelineSize );
    LightingPass.pipeline.Initialize( &DEMODevice, pipelineInfo );

    // Vertex position buffer
     LightingPass.positionBuffer.Initialize(  sizeof(LightingPass.posInitData),  LightingPass.posInitData,  nn::gfx::GpuAccess_Read | nn::gfx::GpuAccess_VertexBuffer,  0  );

    // Vertex texcoord buffer
     LightingPass.textureCoordinateBuffer.Initialize(  sizeof(LightingPass.texInitData),  LightingPass.texInitData,  nn::gfx::GpuAccess_Read | nn::gfx::GpuAccess_VertexBuffer,  0  );

    // Index buffer
     LightingPass.indexBuffer.Initialize(  sizeof(LightingPass.idxInitData),  LightingPass.idxInitData,  nn::gfx::GpuAccess_Read | nn::gfx::GpuAccess_IndexBuffer,  0  );

    // Render lit scene to entire screen
    DEMOGfxSetViewportScissorState( &LightingPass.viewportScissor, &LightingPass.pViewportScissorData,
        0.0f, 0.0f, static_cast< float >( SURFACE_WIDTH ), static_cast< float >( SURFACE_HEIGHT ), 0.0f, 1.0f, static_cast< float >( SURFACE_HEIGHT ), true );
} //NOLINT(readability/fn_size)

// Initializes the Blur pass
static void InteriorPassInit()
{
    // Initialize texture buffers
    DEMOGfxSetupTextureView( &LightingPass.myColorBuffer, &InteriorPass.myTextureBufferView, &InteriorPass.myTextureBufferSlot,
        nn::gfx::ImageDimension_2d, nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm, nn::gfx::DepthStencilFetchMode_DepthComponent);

    // Set up the sampler object
    DEMOGfxInitSampler(&InteriorPass.mySampler, &InteriorPass.mySamplerSlot, nn::gfx::TextureAddressMode_ClampToEdge,
        nn::gfx::FilterMode_MinPoint_MagPoint_MipPoint, nn::gfx::ComparisonFunction_Always );

    // geometry setup
    InteriorPass.posInitData[0] = -1.0f; InteriorPass.posInitData[1] = -1.0f;
    InteriorPass.posInitData[2] =  0.0f; InteriorPass.posInitData[3] =  1.0f;
    InteriorPass.posInitData[4] = -1.0f; InteriorPass.posInitData[5] =  0.0f;
    InteriorPass.posInitData[6] =  1.0f; InteriorPass.posInitData[7] =  1.0f;
    InteriorPass.posInitData[8] =  0.0f; InteriorPass.posInitData[9] = -1.0f;
    InteriorPass.posInitData[10] = 1.0f; InteriorPass.posInitData[11] = 0.0f;

    InteriorPass.texInitData[0] =  0.0f; InteriorPass.texInitData[1] =  1.0f;
    InteriorPass.texInitData[2] =  1.0f; InteriorPass.texInitData[3] =  1.0f;
    InteriorPass.texInitData[4] =  1.0f; InteriorPass.texInitData[5] =  0.0f;
    InteriorPass.texInitData[6] =  0.0f; InteriorPass.texInitData[7] =  0.0f;

    InteriorPass.idxInitData[0] = 0;
    InteriorPass.idxInitData[1] = 1;
    InteriorPass.idxInitData[2] = 2;
    InteriorPass.idxInitData[3] = 2;
    InteriorPass.idxInitData[4] = 3;
    InteriorPass.idxInitData[5] = 0;
    InteriorPass.numAttrib = 2;
    InteriorPass.numIndices = 6;

    // Load shaders
    DEMOGfxLoadShadersFromFile(  &InteriorPass.demoShader , 0,  GSH_SHADER_FILE_PENCIL_INTERIOR[g_ShaderFileIdx] );

    // Attribute Location Lookup
    InteriorPass.posLoc = InteriorPass.demoShader.GetInterfaceSlot( nn::gfx::ShaderStage_Vertex,  nn::gfx::ShaderInterfaceType_Input, "a_position" );
    InteriorPass.texcoordLoc = InteriorPass.demoShader.GetInterfaceSlot( nn::gfx::ShaderStage_Vertex,  nn::gfx::ShaderInterfaceType_Input, "a_texCoord" );

    // Sampler Location Lookup
    InteriorPass.textureLoc[ 0 ] = InteriorPass.demoShader.GetInterfaceSlot( nn::gfx::ShaderStage_Pixel,  nn::gfx::ShaderInterfaceType_Sampler, "s_image" );
    InteriorPass.textureLoc[ 1 ] = InteriorPass.demoShader.GetInterfaceSlot( nn::gfx::ShaderStage_Pixel,  nn::gfx::ShaderInterfaceType_Sampler, "s_pencil" );

    // Uniform Location Lookup
    InteriorPass.divisionsLoc = InteriorPass.demoShader.GetInterfaceSlot( nn::gfx::ShaderStage_Pixel,  nn::gfx::ShaderInterfaceType_ConstantBuffer, "u_uniforms" );

    // Attribute Format Setup
    nn::gfx::Pipeline::InfoType pipelineInfo;
    nn::gfx::VertexBufferStateInfo vertexBufferStates[ 2 ];
    nn::gfx::VertexAttributeStateInfo vertexAttributeStates[ 2 ];
    nn::gfx::BlendTargetStateInfo blendTargetStates[ 1 ];
    nn::gfx::ColorTargetStateInfo colorTargetStates[ 1 ];

    pipelineInfo.SetDefault();

    // Setup vertex attributes
    vertexAttributeStates[0].SetDefault();
    vertexAttributeStates[0].SetBufferIndex( InteriorPass.posLoc );
    vertexAttributeStates[0].SetShaderSlot( InteriorPass.posLoc );
    vertexAttributeStates[0].SetFormat( nn::gfx::AttributeFormat_32_32_32_Float );
    vertexAttributeStates[1].SetDefault();
    vertexAttributeStates[1].SetBufferIndex( InteriorPass.texcoordLoc );
    vertexAttributeStates[1].SetShaderSlot( InteriorPass.texcoordLoc );
    vertexAttributeStates[1].SetFormat( nn::gfx::AttributeFormat_32_32_Float );

    // Setup vertex buffers
    ASSERT( InteriorPass.posLoc < 2 );
    ASSERT( InteriorPass.texcoordLoc < 2 );
    vertexBufferStates[InteriorPass.posLoc].SetDefault();
    vertexBufferStates[InteriorPass.posLoc].SetStride( 3 * sizeof(float) );
    vertexBufferStates[InteriorPass.texcoordLoc].SetDefault();
    vertexBufferStates[InteriorPass.texcoordLoc].SetStride( 2 * sizeof(float) );

    // Setup BlendControl State
    blendTargetStates[0].SetDefault();
    blendTargetStates[0].SetBlendEnabled( false );
    blendTargetStates[0].SetSourceColorBlendFactor( nn::gfx::BlendFactor_SourceAlpha );
    blendTargetStates[0].SetDestinationColorBlendFactor( nn::gfx::BlendFactor_OneMinusSourceAlpha );
    blendTargetStates[0].SetColorBlendFunction( nn::gfx::BlendFunction_Add );
    blendTargetStates[0].SetSourceAlphaBlendFactor( nn::gfx::BlendFactor_SourceAlpha );
    blendTargetStates[0].SetDestinationAlphaBlendFactor( nn::gfx::BlendFactor_OneMinusSourceAlpha );
    blendTargetStates[0].SetAlphaBlendFunction( nn::gfx::BlendFunction_Add );

    // Setup the render target state.
    colorTargetStates[ 0 ].SetDefault();
    colorTargetStates[ 0 ].SetFormat( pencilTexture[ 0 ].pencilTextureInfo.GetImageFormat() );

    nn::gfx::DepthStencilStateInfo depthStencilStateInfo;
    nn::gfx::RasterizerStateInfo rasterizerStateInfo;
    nn::gfx::BlendStateInfo blendStateInfo;
    nn::gfx::RenderTargetStateInfo renderTargetStateInfo;
    nn::gfx::VertexStateInfo vertexStateInfo;

    SetDefaultPipelineStateInfo(pipelineInfo, blendTargetStates, 1, colorTargetStates, 1,
        vertexAttributeStates, 2, vertexBufferStates, 2, depthStencilStateInfo, rasterizerStateInfo,
        blendStateInfo, renderTargetStateInfo, vertexStateInfo);

    // Disable the depth buffer
    depthStencilStateInfo.SetDepthTestEnabled( false );
    depthStencilStateInfo.SetDepthWriteEnabled( false );
    depthStencilStateInfo.SetDepthComparisonFunction( nn::gfx::ComparisonFunction_Never );

    // Setup shaders and state info
    pipelineInfo.SetShaderPtr(InteriorPass.demoShader.GetShader());
    pipelineInfo.SetBlendStateInfo(&blendStateInfo);
    pipelineInfo.SetDepthStencilStateInfo(&depthStencilStateInfo);
    pipelineInfo.SetRasterizerStateInfo(&rasterizerStateInfo);
    pipelineInfo.SetRenderTargetStateInfo(&renderTargetStateInfo);
    pipelineInfo.SetVertexStateInfo(&vertexStateInfo);

    size_t pipelineSize = nn::gfx::Pipeline::GetRequiredMemorySize( pipelineInfo );
    InteriorPass.pPipelineData = DEMOGfxAllocMEM2( pipelineSize,
        nn::gfx::Pipeline::RequiredMemoryInfo_Alignment );
    InteriorPass.pipeline.SetMemory( InteriorPass.pPipelineData, pipelineSize );
    InteriorPass.pipeline.Initialize( &DEMODevice, pipelineInfo );

    // Vertex position buffer
     InteriorPass.positionBuffer.Initialize(  sizeof(InteriorPass.posInitData),  InteriorPass.posInitData,  nn::gfx::GpuAccess_Read | nn::gfx::GpuAccess_VertexBuffer,  0  );

    // Vertex texcoord buffer
     InteriorPass.textureCoordinateBuffer.Initialize(  sizeof(InteriorPass.texInitData),  InteriorPass.texInitData,  nn::gfx::GpuAccess_Read | nn::gfx::GpuAccess_VertexBuffer,  0  );

    // Index buffer
     InteriorPass.indexBuffer.Initialize(  sizeof(InteriorPass.idxInitData),  InteriorPass.idxInitData,  nn::gfx::GpuAccess_Read | nn::gfx::GpuAccess_IndexBuffer,  0  );

    // Render lit scene to entire screen
    DEMOGfxSetViewportScissorState( &InteriorPass.viewportScissor, &InteriorPass.pViewportScissorData,
        0.0f, 0.0f, static_cast< float >( SURFACE_WIDTH ), static_cast< float >( SURFACE_HEIGHT ), 0.0f, 1.0f, static_cast< float >(SURFACE_HEIGHT), false );

    // Uniform Location Lookup
    struct InteriorUniforms temp;
    struct InteriorUniforms* pUniformData = &temp;
    pUniformData->divisions.x = f32(PENCIL_TEXTURE_DIVISIONS);
    pUniformData->divisions.y = 0.0f;
    pUniformData->divisions.z = 0.0f;
    pUniformData->divisions.w = 0.0f;
#if NN_GFX_IS_TARGET_GX
    GX2EndianSwap( pUniformData, sizeof(struct InteriorUniforms) );
#endif
     InteriorPass.uniformBuffer.Initialize(  sizeof(*pUniformData),  pUniformData,  nn::gfx::GpuAccess_ConstantBuffer | nn::gfx::GpuAccess_Read,  0  );
}

// Initializes the Multiple Contours pass
static void ContoursPassInit()
{
    // Initialize texture buffers
    DEMOGfxSetupTextureView( &OutlinePass.myColorBuffer, &ContoursPass.myTextureView, &ContoursPass.myTextureBufferSlot,
        nn::gfx::ImageDimension_2d, nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm, nn::gfx::DepthStencilFetchMode_DepthComponent );

    // Set up the sampler object
    DEMOGfxInitSampler(&ContoursPass.mySampler, &ContoursPass.mySamplerSlot, nn::gfx::TextureAddressMode_ClampToEdge,
        nn::gfx::FilterMode_MinPoint_MagPoint_MipPoint, nn::gfx::ComparisonFunction_Always );

    // geometry setup

    // create texCoord data
    for(int i = 0; i < CONTOUR_VERTICES; ++i)
    {
        for(int j = 0; j < CONTOUR_VERTICES; ++j)
        {
            float x = (float)i;
            float y = (float)j;

            ContoursPass.texInitData[2 * (CONTOUR_VERTICES * j + i) + 0] = x / float(CONTOUR_VERTICES - 1); // u
            ContoursPass.texInitData[2 * (CONTOUR_VERTICES * j + i) + 1] = y / float(CONTOUR_VERTICES - 1); // v
            ContoursPass.posInitData[3 * (CONTOUR_VERTICES * j + i) + 0] = -1.0f + 2.0f * (x / float(CONTOUR_VERTICES - 1)); // x
            ContoursPass.posInitData[3 * (CONTOUR_VERTICES * j + i) + 1] = 1.0f - 2.0f * (y / float(CONTOUR_VERTICES - 1)); // y
            ContoursPass.posInitData[3 * (CONTOUR_VERTICES * j + i) + 2] = 0.0f; // z
        }
    }

    // create index data
    for(int i = 0; i < CONTOUR_SEGMENTS; ++i)
    {
        for(int j = 0; j < CONTOUR_SEGMENTS; ++j)
        {
            ContoursPass.idxInitData[6 * (CONTOUR_SEGMENTS * j + i) + 0] = CONTOUR_VERTICES * (j + 0) + ( i + 0);
            ContoursPass.idxInitData[6 * (CONTOUR_SEGMENTS * j + i) + 1] = CONTOUR_VERTICES * (j + 0) + ( i + 1);
            ContoursPass.idxInitData[6 * (CONTOUR_SEGMENTS * j + i) + 2] = CONTOUR_VERTICES * (j + 1) + ( i + 1);
            ContoursPass.idxInitData[6 * (CONTOUR_SEGMENTS * j + i) + 3] = CONTOUR_VERTICES * (j + 1) + ( i + 1);
            ContoursPass.idxInitData[6 * (CONTOUR_SEGMENTS * j + i) + 4] = CONTOUR_VERTICES * (j + 1) + ( i + 0);
            ContoursPass.idxInitData[6 * (CONTOUR_SEGMENTS * j + i) + 5] = CONTOUR_VERTICES * (j + 0) + ( i + 0);
        }
    }

    ContoursPass.numAttrib = 2;
    ContoursPass.numIndices = 6 * CONTOUR_SEGMENTS * CONTOUR_SEGMENTS;

    // Load shader binary to memory
    DEMOGfxLoadShadersFromFile(  &ContoursPass.demoShader , 0,  GSH_SHADER_FILE_PENCIL_CONTOURS[g_ShaderFileIdx] );

    // Attribute Location Lookup
    ContoursPass.posLoc = ContoursPass.demoShader.GetInterfaceSlot( nn::gfx::ShaderStage_Vertex,  nn::gfx::ShaderInterfaceType_Input, "a_position" );
    ContoursPass.texcoordLoc = ContoursPass.demoShader.GetInterfaceSlot( nn::gfx::ShaderStage_Vertex,  nn::gfx::ShaderInterfaceType_Input, "a_texCoord" );

    // Sampler Location Lookup
    ContoursPass.textureLoc = ContoursPass.demoShader.GetInterfaceSlot( nn::gfx::ShaderStage_Pixel,  nn::gfx::ShaderInterfaceType_Sampler, "s_outline" );

    // Uniform Location Lookup
    ContoursPass.uniformLoc = ContoursPass.demoShader.GetInterfaceSlot( nn::gfx::ShaderStage_Vertex,  nn::gfx::ShaderInterfaceType_ConstantBuffer, "u_uniforms" );

    // Attribute Format Setup
    nn::gfx::Pipeline::InfoType pipelineInfo;
    nn::gfx::VertexBufferStateInfo vertexBufferStates[ 2 ];
    nn::gfx::VertexAttributeStateInfo vertexAttributeStates[ 2 ];
    nn::gfx::BlendTargetStateInfo blendTargetStates[ 1 ];
    nn::gfx::ColorTargetStateInfo colorTargetStates[ 1 ];

    pipelineInfo.SetDefault();

    // Setup vertex attributes
    vertexAttributeStates[0].SetDefault();
    vertexAttributeStates[0].SetBufferIndex( ContoursPass.posLoc );
    vertexAttributeStates[0].SetShaderSlot( ContoursPass.posLoc );
    vertexAttributeStates[0].SetFormat( nn::gfx::AttributeFormat_32_32_32_Float );
    vertexAttributeStates[1].SetDefault();
    vertexAttributeStates[1].SetBufferIndex( ContoursPass.texcoordLoc );
    vertexAttributeStates[1].SetShaderSlot( ContoursPass.texcoordLoc );
    vertexAttributeStates[1].SetFormat( nn::gfx::AttributeFormat_32_32_Float );

    // Setup vertex buffers
    ASSERT( ContoursPass.posLoc < 2 );
    ASSERT( ContoursPass.texcoordLoc < 2 );
    vertexBufferStates[ContoursPass.posLoc].SetDefault();
    vertexBufferStates[ContoursPass.posLoc].SetStride( 3 * sizeof(float) );
    vertexBufferStates[ContoursPass.texcoordLoc].SetDefault();
    vertexBufferStates[ContoursPass.texcoordLoc].SetStride( 2 * sizeof(float) );

    // Setup BlendControl State
    blendTargetStates[0].SetDefault();
    blendTargetStates[0].SetBlendEnabled( true );
    blendTargetStates[0].SetSourceColorBlendFactor( nn::gfx::BlendFactor_SourceAlpha );
    blendTargetStates[0].SetDestinationColorBlendFactor( nn::gfx::BlendFactor_OneMinusSourceAlpha );
    blendTargetStates[0].SetColorBlendFunction( nn::gfx::BlendFunction_Add );
    blendTargetStates[0].SetSourceAlphaBlendFactor( nn::gfx::BlendFactor_SourceAlpha );
    blendTargetStates[0].SetDestinationAlphaBlendFactor( nn::gfx::BlendFactor_OneMinusSourceAlpha );
    blendTargetStates[0].SetAlphaBlendFunction( nn::gfx::BlendFunction_Add );

    // Setup the render target state.
    colorTargetStates[ 0 ].SetDefault();
    colorTargetStates[ 0 ].SetFormat( pencilTexture[ 0 ].pencilTextureInfo.GetImageFormat() );

    nn::gfx::DepthStencilStateInfo depthStencilStateInfo;
    nn::gfx::RasterizerStateInfo rasterizerStateInfo;
    nn::gfx::BlendStateInfo blendStateInfo;
    nn::gfx::RenderTargetStateInfo renderTargetStateInfo;
    nn::gfx::VertexStateInfo vertexStateInfo;

    SetDefaultPipelineStateInfo(pipelineInfo, blendTargetStates, 1, colorTargetStates, 1,
        vertexAttributeStates, 2, vertexBufferStates, 2, depthStencilStateInfo, rasterizerStateInfo,
        blendStateInfo, renderTargetStateInfo, vertexStateInfo);

    // Disable the depth buffer
    depthStencilStateInfo.SetDepthTestEnabled( false );
    depthStencilStateInfo.SetDepthWriteEnabled( false );
    depthStencilStateInfo.SetDepthComparisonFunction( nn::gfx::ComparisonFunction_Never );

    // Setup shaders and state info
    pipelineInfo.SetShaderPtr(ContoursPass.demoShader.GetShader());
    pipelineInfo.SetBlendStateInfo(&blendStateInfo);
    pipelineInfo.SetDepthStencilStateInfo(&depthStencilStateInfo);
    pipelineInfo.SetRasterizerStateInfo(&rasterizerStateInfo);
    pipelineInfo.SetRenderTargetStateInfo(&renderTargetStateInfo);
    pipelineInfo.SetVertexStateInfo(&vertexStateInfo);

    size_t pipelineSize = nn::gfx::Pipeline::GetRequiredMemorySize( pipelineInfo );
    ContoursPass.pPipelineData = DEMOGfxAllocMEM2( pipelineSize, nn::gfx::Pipeline::RequiredMemoryInfo_Alignment );
    ContoursPass.pipeline.SetMemory( ContoursPass.pPipelineData, pipelineSize );
    ContoursPass.pipeline.Initialize( &DEMODevice, pipelineInfo );

    // Vertex position buffer
     ContoursPass.positionBuffer.Initialize(  sizeof(ContoursPass.posInitData),  ContoursPass.posInitData,  nn::gfx::GpuAccess_Read | nn::gfx::GpuAccess_VertexBuffer,  0  );

    // Vertex texcoord buffer
     ContoursPass.textureCoordinateBuffer.Initialize(  sizeof(ContoursPass.texInitData),  ContoursPass.texInitData,  nn::gfx::GpuAccess_Read | nn::gfx::GpuAccess_VertexBuffer,  0  );

    // Index buffer
     ContoursPass.indexBuffer.Initialize(  sizeof(ContoursPass.idxInitData),  ContoursPass.idxInitData,  nn::gfx::GpuAccess_Read | nn::gfx::GpuAccess_IndexBuffer,  0  );

    // Setup Uniform
     ContoursPass.uniformBuffer.Initialize(  sizeof(struct ContourUniforms),  NULL,  nn::gfx::GpuAccess_Read | nn::gfx::GpuAccess_ConstantBuffer,  0  );

    // Render to entire screen
    DEMOGfxSetViewportScissorState( &ContoursPass.viewportScissor, &ContoursPass.pViewportScissorData,
        0.0f, 0.0f, static_cast< float >( SURFACE_WIDTH ), static_cast< float >( SURFACE_HEIGHT ), 0.0f, 1.0f, static_cast< float >( SURFACE_HEIGHT ), false );
}

static void BufferPassTextureInit()
{
    DEMOGfxSetupTextureView( &GeometryPass.myColorBuffer[0], &BuffersPass.myTextureBuffer[0], &BuffersPass.myTextureBufferSlot[0],
        nn::gfx::ImageDimension_2d, nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm, nn::gfx::DepthStencilFetchMode_DepthComponent );
    DEMOGfxSetupTextureView( &GeometryPass.myColorBuffer[1], &BuffersPass.myTextureBuffer[1], &BuffersPass.myTextureBufferSlot[1],
        nn::gfx::ImageDimension_2d, nn::gfx::ImageFormat_R8_G8_B8_A8_Snorm, nn::gfx::DepthStencilFetchMode_DepthComponent); // NOTE: This is less precise than the original but nn::gfx::ImageFormat_R10_G10_B10_A2_Snorm doesn't exist
    DEMOGfxSetupTextureView( &DEMODepthBuffer, &BuffersPass.myTextureBuffer[2], &BuffersPass.myTextureBufferSlot[2],
        nn::gfx::ImageDimension_2d, nn::gfx::ImageFormat_D32_Float, nn::gfx::DepthStencilFetchMode_DepthComponent);
    DEMOGfxSetupTextureView( &OutlinePass.myColorBuffer, &BuffersPass.myTextureBuffer[3], &BuffersPass.myTextureBufferSlot[3],
        nn::gfx::ImageDimension_2d, nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm, nn::gfx::DepthStencilFetchMode_DepthComponent);
    DEMOGfxSetupTextureView( &LightingPass.myColorBuffer, &BuffersPass.myTextureBuffer[4], &BuffersPass.myTextureBufferSlot[4],
        nn::gfx::ImageDimension_2d, nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm, nn::gfx::DepthStencilFetchMode_DepthComponent);
    DEMOGfxSetupTextureView( &pencilTexture[0].texture, &BuffersPass.myTextureBuffer[5], &BuffersPass.myTextureBufferSlot[5],
        nn::gfx::ImageDimension_2d, nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm, nn::gfx::DepthStencilFetchMode_DepthComponent);
    DEMOGfxSetupTextureView( &ShadowPass.myDepthBuffer, &BuffersPass.myTextureBuffer[6], &BuffersPass.myTextureBufferSlot[6],
        nn::gfx::ImageDimension_2d, nn::gfx::ImageFormat_D32_Float, nn::gfx::DepthStencilFetchMode_DepthComponent);
}

// Initializes the Buffers pass
static void BuffersPassInit()
{
    // Initialize texture buffers
    BufferPassTextureInit();

    // geometry setup
    BuffersPass.posInitData[0] = -0.125f; BuffersPass.posInitData[1] = -0.5f; BuffersPass.posInitData[2] =  0.0f;
    BuffersPass.posInitData[3] =  0.125f; BuffersPass.posInitData[4] = -0.5f; BuffersPass.posInitData[5] =  0.0f;
    BuffersPass.posInitData[6] =  0.125f; BuffersPass.posInitData[7] =  0.5f; BuffersPass.posInitData[8] =  0.0f;
    BuffersPass.posInitData[9] = -0.125f; BuffersPass.posInitData[10] = 0.5f; BuffersPass.posInitData[11] = 0.0f;

    BuffersPass.texInitData[0] =  0.0f; BuffersPass.texInitData[1] =  1.0f;
    BuffersPass.texInitData[2] =  1.0f; BuffersPass.texInitData[3] =  1.0f;
    BuffersPass.texInitData[4] =  1.0f; BuffersPass.texInitData[5] =  0.0f;
    BuffersPass.texInitData[6] =  0.0f; BuffersPass.texInitData[7] =  0.0f;

    BuffersPass.idxInitData[0] = 0;
    BuffersPass.idxInitData[1] = 1;
    BuffersPass.idxInitData[2] = 2;
    BuffersPass.idxInitData[3] = 2;
    BuffersPass.idxInitData[4] = 3;
    BuffersPass.idxInitData[5] = 0;
    BuffersPass.numAttrib = 2;
    BuffersPass.numIndices = 6;

    // Load shader binary to memory
    DEMOGfxLoadShadersFromFile(  &BuffersPass.demoShader , 0,  GSH_SHADER_FILE_TEXTURE2D[g_ShaderFileIdx] );

    // Attribute Location Lookup
    BuffersPass.posLoc = BuffersPass.demoShader.GetInterfaceSlot( nn::gfx::ShaderStage_Vertex,  nn::gfx::ShaderInterfaceType_Input, "a_position" );
    BuffersPass.texcoordLoc = BuffersPass.demoShader.GetInterfaceSlot( nn::gfx::ShaderStage_Vertex,  nn::gfx::ShaderInterfaceType_Input, "a_texCoord" );

    // Sampler Location Lookup
    BuffersPass.textureLoc = BuffersPass.demoShader.GetInterfaceSlot( nn::gfx::ShaderStage_Pixel,  nn::gfx::ShaderInterfaceType_Sampler, "s_texture" );

    // Uniform Location lookup
    BuffersPass.offsetLoc = BuffersPass.demoShader.GetInterfaceSlot( nn::gfx::ShaderStage_Vertex,  nn::gfx::ShaderInterfaceType_ConstantBuffer, "u_uniforms" );

    // Attribute Format Setup
    nn::gfx::Pipeline::InfoType pipelineInfo;
    nn::gfx::VertexBufferStateInfo vertexBufferStates[ 2 ];
    nn::gfx::VertexAttributeStateInfo vertexAttributeStates[ 2 ];
    nn::gfx::BlendTargetStateInfo blendTargetStates[ 1 ];
    nn::gfx::ColorTargetStateInfo colorTargetStates[ 1 ];

    pipelineInfo.SetDefault();

    // Setup vertex attributes
    vertexAttributeStates[0].SetDefault();
    vertexAttributeStates[0].SetBufferIndex( BuffersPass.posLoc );
    vertexAttributeStates[0].SetShaderSlot( BuffersPass.posLoc );
    vertexAttributeStates[0].SetFormat( nn::gfx::AttributeFormat_32_32_32_Float );
    vertexAttributeStates[1].SetDefault();
    vertexAttributeStates[1].SetBufferIndex( BuffersPass.texcoordLoc );
    vertexAttributeStates[1].SetShaderSlot( BuffersPass.texcoordLoc );
    vertexAttributeStates[1].SetFormat( nn::gfx::AttributeFormat_32_32_Float );

    // Setup vertex buffers
    ASSERT( BuffersPass.posLoc < 2 );
    ASSERT( BuffersPass.texcoordLoc < 2 );
    vertexBufferStates[BuffersPass.posLoc].SetDefault();
    vertexBufferStates[BuffersPass.posLoc].SetStride( 3 * sizeof(float) );
    vertexBufferStates[BuffersPass.texcoordLoc].SetDefault();
    vertexBufferStates[BuffersPass.texcoordLoc].SetStride( 2 * sizeof(float) );

    // Setup BlendControl State
    blendTargetStates[0].SetDefault();
    blendTargetStates[0].SetBlendEnabled( false );
    blendTargetStates[0].SetSourceColorBlendFactor( nn::gfx::BlendFactor_SourceAlpha );
    blendTargetStates[0].SetDestinationColorBlendFactor( nn::gfx::BlendFactor_OneMinusSourceAlpha );
    blendTargetStates[0].SetColorBlendFunction( nn::gfx::BlendFunction_Add );
    blendTargetStates[0].SetSourceAlphaBlendFactor( nn::gfx::BlendFactor_SourceAlpha );
    blendTargetStates[0].SetDestinationAlphaBlendFactor( nn::gfx::BlendFactor_OneMinusSourceAlpha );
    blendTargetStates[0].SetAlphaBlendFunction( nn::gfx::BlendFunction_Add );

    // Setup the render target state.
    colorTargetStates[ 0 ].SetDefault();
    colorTargetStates[ 0 ].SetFormat( pencilTexture[ 0 ].pencilTextureInfo.GetImageFormat() );

    nn::gfx::DepthStencilStateInfo depthStencilStateInfo;
    nn::gfx::RasterizerStateInfo rasterizerStateInfo;
    nn::gfx::BlendStateInfo blendStateInfo;
    nn::gfx::RenderTargetStateInfo renderTargetStateInfo;
    nn::gfx::VertexStateInfo vertexStateInfo;

    SetDefaultPipelineStateInfo(pipelineInfo, blendTargetStates, 1, colorTargetStates, 1,
        vertexAttributeStates, 2, vertexBufferStates, 2, depthStencilStateInfo, rasterizerStateInfo,
        blendStateInfo, renderTargetStateInfo, vertexStateInfo);

    // Disable the depth buffer
    depthStencilStateInfo.SetDefault();
    depthStencilStateInfo.SetDepthTestEnabled( false );
    depthStencilStateInfo.SetDepthWriteEnabled( false );
    depthStencilStateInfo.SetDepthComparisonFunction( nn::gfx::ComparisonFunction_Never );
    pipelineInfo.SetDepthStencilStateInfo(&depthStencilStateInfo);
    pipelineInfo.SetRasterizerStateInfo(&rasterizerStateInfo);
    pipelineInfo.SetBlendStateInfo(&blendStateInfo);
    pipelineInfo.SetRenderTargetStateInfo(&renderTargetStateInfo);
    pipelineInfo.SetVertexStateInfo(&vertexStateInfo);

    // Setup VS/PS
    pipelineInfo.SetShaderPtr( BuffersPass.demoShader.GetShader() );

    {
        size_t pipelineSize = nn::gfx::Pipeline::GetRequiredMemorySize( pipelineInfo );
        BuffersPass.pPipelineData[0] = DEMOGfxAllocMEM2( pipelineSize, nn::gfx::Pipeline::RequiredMemoryInfo_Alignment );
        BuffersPass.pipeline[0].SetMemory( BuffersPass.pPipelineData[0], pipelineSize );
        BuffersPass.pipeline[0].Initialize( &DEMODevice, pipelineInfo );
    }

    // When i == 3, we enable blending
    blendTargetStates[0].SetBlendEnabled( true );

    {
        size_t pipelineSize = nn::gfx::Pipeline::GetRequiredMemorySize( pipelineInfo );
        BuffersPass.pPipelineData[1] = DEMOGfxAllocMEM2( pipelineSize, nn::gfx::Pipeline::RequiredMemoryInfo_Alignment );
        BuffersPass.pipeline[1].SetMemory( BuffersPass.pPipelineData[1], pipelineSize );
        BuffersPass.pipeline[1].Initialize( &DEMODevice, pipelineInfo );
    }

    // Vertex position buffer
    BuffersPass.positionBuffer.Initialize( sizeof(BuffersPass.posInitData), BuffersPass.posInitData, nn::gfx::GpuAccess_Read | nn::gfx::GpuAccess_VertexBuffer, 0 );

    // Vertex texcoord buffer
    BuffersPass.textureCoordinateBuffer.Initialize( sizeof(BuffersPass.texInitData), BuffersPass.texInitData, nn::gfx::GpuAccess_Read | nn::gfx::GpuAccess_VertexBuffer, 0 );

    // Index buffer
    BuffersPass.indexBuffer.Initialize( sizeof(BuffersPass.idxInitData), BuffersPass.idxInitData, nn::gfx::GpuAccess_Read | nn::gfx::GpuAccess_IndexBuffer, 0 );

    DEMOGfxInitSampler(&BuffersPass.mySampler, &BuffersPass.mySamplerSlot, nn::gfx::TextureAddressMode_Repeat,
        nn::gfx::FilterMode_MinPoint_MagPoint_MipPoint, nn::gfx::ComparisonFunction_Always );

#if NN_GFX_IS_TARGET_D3D
    DEMOGfxSetViewportScissorState( &BuffersPass.viewportScissor, &BuffersPass.pViewportScissorData,
        0.0f, static_cast< float >( SURFACE_HEIGHT ) / 4 * 3, static_cast< float >( SURFACE_WIDTH ), static_cast< float >( SURFACE_HEIGHT ) / 4, 0.0f, 1.0f, static_cast< float >( SURFACE_HEIGHT ), false );
#else
    DEMOGfxSetViewportScissorState( &BuffersPass.viewportScissor, &BuffersPass.pViewportScissorData,
        0.0f, 0.0f, static_cast< float >( SURFACE_WIDTH ), static_cast< float >( SURFACE_HEIGHT ) / 4, 0.0f, 1.0f, static_cast< float >( SURFACE_HEIGHT ), false );
#endif

    // Setup uniform blocks
    for( int idx = 0; idx < sizeof(BuffersPass.uniformBuffer) / sizeof(BuffersPass.uniformBuffer[0]); idx++ )
    {
        struct BuffersUniforms temp;
        memset(&temp, 0, sizeof(struct BuffersUniforms));
         BuffersPass.uniformBuffer[idx].Initialize(  sizeof(struct BuffersUniforms),  &temp,  nn::gfx::GpuAccess_ConstantBuffer | nn::gfx::GpuAccess_Read,  0  );
    }
} //NOLINT(readability/fn_size)

static void GeometryPassFinalize()
{
    // Geometry pass free
    for(int i = 0; i < NUM_MODEL; ++i)
    {
        GeometryPass.positionBuffers[i].Finalize( &DEMODevice );
        GeometryPass.normalBuffers[i].Finalize( &DEMODevice );
        GeometryPass.indexBuffers[i].Finalize( &DEMODevice );

        DEMOGfxFreeMEM2(GeometryPass.posBuf[i]);
        DEMOGfxFreeMEM2(GeometryPass.nrmBuf[i]);
        DEMOGfxFreeMEM2(GeometryPass.idxBuf[i]);
    }

    DEMOGfxFreeShaders( &GeometryPass.demoShader );
    GeometryPass.viewportScissor.Finalize( &DEMODevice );
    GeometryPass.vertexUniformBuffer.Finalize( &DEMODevice );
    GeometryPass.pixelUniformBuffer.Finalize( &DEMODevice );
    GeometryPass.pipeline.Finalize( &DEMODevice );
    GeometryPass.myDepthBufferView.Finalize( &DEMODevice );
    GeometryPass.myColorBufferView[0].Finalize( &DEMODevice );
    GeometryPass.myColorBufferView[1].Finalize( &DEMODevice );
    GeometryPass.myColorBuffer[0].Finalize( &DEMODevice );
    GeometryPass.myColorBuffer[1].Finalize( &DEMODevice );
    GeometryPass.myColorBufferData[ 0 ]->Finalize();
    GeometryPass.myColorBufferData[ 1 ]->Finalize();
    delete GeometryPass.myColorBufferData[ 0 ];
    delete GeometryPass.myColorBufferData[ 1 ];
    DEMOGfxFreeMEM2(GeometryPass.pPipelineData);
    DEMOGfxFreeMEM2(GeometryPass.pViewportScissorData);
}

static void BuffersPassFinalize()
{
    // Free BufferPass which differs from the rest
    DEMOGfxFreeShaders( &BuffersPass.demoShader );
    BuffersPass.pipeline[0].Finalize( &DEMODevice );
    BuffersPass.pipeline[1].Finalize( &DEMODevice );
    BuffersPass.viewportScissor.Finalize( &DEMODevice );
    BuffersPass.positionBuffer.Finalize( &DEMODevice );
    BuffersPass.indexBuffer.Finalize( &DEMODevice );
    BuffersPass.textureCoordinateBuffer.Finalize( &DEMODevice );
    BuffersPass.mySampler.Finalize( &DEMODevice );
    DEMOGfxFreeMEM2(BuffersPass.pViewportScissorData);
    DEMOGfxFreeMEM2(BuffersPass.pPipelineData[0]);
    DEMOGfxFreeMEM2(BuffersPass.pPipelineData[1]);

    for( int i = 0; i < sizeof(BuffersPass.uniformBuffer) / sizeof(BuffersPass.uniformBuffer[0]); i++ )
    {
        BuffersPass.uniformBuffer[i].Finalize( &DEMODevice );
    }

    for( int i = 0; i < sizeof(BuffersPass.myTextureBuffer) / sizeof(BuffersPass.myTextureBuffer[0]); i++)
    {
        BuffersPass.myTextureBuffer[i].Finalize( &DEMODevice );
    }
}

static void ContoursPassFinalize()
{
    DEMOGfxFreeShaders( &ContoursPass.demoShader );
    ContoursPass.pipeline.Finalize( &DEMODevice );
    ContoursPass.positionBuffer.Finalize( &DEMODevice );
    ContoursPass.indexBuffer.Finalize( &DEMODevice );
    ContoursPass.textureCoordinateBuffer.Finalize( &DEMODevice );
    ContoursPass.viewportScissor.Finalize( &DEMODevice );
    ContoursPass.myTextureView.Finalize( &DEMODevice );
    ContoursPass.mySampler.Finalize( &DEMODevice );
    ContoursPass.uniformBuffer.Finalize( &DEMODevice );
    DEMOGfxFreeMEM2(ContoursPass.pViewportScissorData);
    DEMOGfxFreeMEM2(ContoursPass.pPipelineData );
}

static void InteriorPassFinalize()
{
    DEMOGfxFreeShaders( &InteriorPass.demoShader );
    InteriorPass.pipeline.Finalize( &DEMODevice );
    InteriorPass.positionBuffer.Finalize( &DEMODevice );
    InteriorPass.indexBuffer.Finalize( &DEMODevice );
    InteriorPass.textureCoordinateBuffer.Finalize( &DEMODevice );
    InteriorPass.viewportScissor.Finalize( &DEMODevice );
    InteriorPass.myTextureBufferView.Finalize( &DEMODevice );
    InteriorPass.mySampler.Finalize( &DEMODevice );
    InteriorPass.uniformBuffer.Finalize( &DEMODevice );
    DEMOGfxFreeMEM2(InteriorPass.pViewportScissorData);
    DEMOGfxFreeMEM2(InteriorPass.pPipelineData );
}

static void LightingPassFinalize()
{
    DEMOGfxFreeShaders( &LightingPass.demoShader );
    LightingPass.pipeline.Finalize( &DEMODevice );
    LightingPass.positionBuffer.Finalize( &DEMODevice );
    LightingPass.indexBuffer.Finalize( &DEMODevice );
    LightingPass.textureCoordinateBuffer.Finalize( &DEMODevice );
    LightingPass.viewportScissor.Finalize( &DEMODevice );
    LightingPass.myColorBufferView.Finalize( &DEMODevice );
    LightingPass.myColorBuffer.Finalize( &DEMODevice );
    LightingPass.myColorBufferData->Finalize();
    delete LightingPass.myColorBufferData;
    LightingPass.mySampler.Finalize( &DEMODevice );
    LightingPass.myDepthSampler.Finalize( &DEMODevice );
    LightingPass.uniformBuffer.Finalize( &DEMODevice );
    DEMOGfxFreeMEM2(LightingPass.pViewportScissorData);
    DEMOGfxFreeMEM2(LightingPass.pPipelineData );

    for( int idx = 0 ; idx < sizeof(LightingPass.myTextureBuffer) / sizeof(LightingPass.myTextureBuffer[0]); idx++ )
    {
        LightingPass.myTextureBuffer[idx].Finalize( &DEMODevice );
    }
}

static void OutlinePassFinalize()
{
    DEMOGfxFreeShaders( &OutlinePass.demoShader );
    OutlinePass.pipeline.Finalize( &DEMODevice );
    OutlinePass.positionBuffer.Finalize( &DEMODevice );
    OutlinePass.indexBuffer.Finalize( &DEMODevice );
    OutlinePass.textureCoordinateBuffer.Finalize( &DEMODevice );
    OutlinePass.viewportScissor.Finalize( &DEMODevice );
    OutlinePass.mySampler.Finalize( &DEMODevice );
    OutlinePass.uniformBuffer.Finalize( &DEMODevice );
    OutlinePass.myColorBufferView.Finalize( &DEMODevice );
    OutlinePass.myColorBuffer.Finalize( &DEMODevice );
    OutlinePass.myColorBufferData->Finalize();
    delete OutlinePass.myColorBufferData;
    OutlinePass.myTextureBufferView[0].Finalize( &DEMODevice );
    OutlinePass.myTextureBufferView[1].Finalize( &DEMODevice );
    DEMOGfxFreeMEM2(OutlinePass.pViewportScissorData);
    DEMOGfxFreeMEM2(OutlinePass.pPipelineData );
}

static void BlurVertShadowPassFinalize()
{
    DEMOGfxFreeShaders( &BlurVertShadowPass.demoShader );
    BlurVertShadowPass.pipeline.Finalize( &DEMODevice );
    BlurVertShadowPass.positionBuffer.Finalize( &DEMODevice );
    BlurVertShadowPass.indexBuffer.Finalize( &DEMODevice );
    BlurVertShadowPass.textureCoordinateBuffer.Finalize( &DEMODevice );
    BlurVertShadowPass.viewportScissor.Finalize( &DEMODevice );
    BlurVertShadowPass.myTextureView.Finalize( &DEMODevice );
    BlurVertShadowPass.myColorBufferView.Finalize(&DEMODevice);
    BlurVertShadowPass.mySampler.Finalize( &DEMODevice );
    DEMOGfxFreeMEM2(BlurVertShadowPass.pViewportScissorData);
    DEMOGfxFreeMEM2(BlurVertShadowPass.pPipelineData );
}

static void BlurHoriShadowPassFinalize()
{
    DEMOGfxFreeShaders( &BlurHoriShadowPass.demoShader );
    BlurHoriShadowPass.pipeline.Finalize( &DEMODevice );
    BlurHoriShadowPass.positionBuffer.Finalize( &DEMODevice );
    BlurHoriShadowPass.indexBuffer.Finalize( &DEMODevice );
    BlurHoriShadowPass.textureCoordinateBuffer.Finalize( &DEMODevice );
    BlurHoriShadowPass.viewportScissor.Finalize( &DEMODevice );
    BlurHoriShadowPass.myColorBufferView.Finalize( &DEMODevice );
    BlurHoriShadowPass.myColorBuffer.Finalize( &DEMODevice );
    BlurHoriShadowPass.myColorBufferData->Finalize();
    delete BlurHoriShadowPass.myColorBufferData;
    BlurHoriShadowPass.myTextureView.Finalize( &DEMODevice );
    BlurHoriShadowPass.mySampler.Finalize( &DEMODevice );
    DEMOGfxFreeMEM2(BlurHoriShadowPass.pViewportScissorData);
    DEMOGfxFreeMEM2(BlurHoriShadowPass.pPipelineData );
}

static void ExpShadowPassFinalize()
{
    DEMOGfxFreeShaders( &ExpShadowPass.demoShader );
    ExpShadowPass.pipeline.Finalize( &DEMODevice );
    ExpShadowPass.positionBuffer.Finalize( &DEMODevice );
    ExpShadowPass.indexBuffer.Finalize( &DEMODevice );
    ExpShadowPass.textureCoordinateBuffer.Finalize( &DEMODevice );
    ExpShadowPass.viewportScissor.Finalize( &DEMODevice );
    ExpShadowPass.myColorBufferView.Finalize( &DEMODevice );
    ExpShadowPass.myColorBuffer.Finalize( &DEMODevice );
    ExpShadowPass.myColorBufferData->Finalize();
    delete ExpShadowPass.myColorBufferData;
    ExpShadowPass.myDepthBufferView.Finalize( &DEMODevice );
    ExpShadowPass.mySampler.Finalize( &DEMODevice );
    DEMOGfxFreeMEM2(ExpShadowPass.pViewportScissorData);
    DEMOGfxFreeMEM2(ExpShadowPass.pPipelineData );
}

static void ShadowPassFinalize()
{
    DEMOGfxFreeShaders( &ShadowPass.demoShader );
    ShadowPass.pipeline.Finalize( &DEMODevice );
    ShadowPass.viewportScissor.Finalize( &DEMODevice );
    ShadowPass.uniforms.Finalize( &DEMODevice );
    ShadowPass.myDepthBufferView.Finalize( &DEMODevice );
    ShadowPass.myDepthBuffer.Finalize( &DEMODevice );
    ShadowPass.myDepthBufferData->Finalize();
    delete ShadowPass.myDepthBufferData;
    DEMOGfxFreeMEM2(ShadowPass.pViewportScissorData);
    DEMOGfxFreeMEM2(ShadowPass.pPipelineData);
}

// Deallocates all resources
static void SceneFree()
{
    // Free Passes
    BuffersPassFinalize();
    ContoursPassFinalize();
    InteriorPassFinalize();
    LightingPassFinalize();
    OutlinePassFinalize();
    GeometryPassFinalize();
    BlurVertShadowPassFinalize();
    BlurHoriShadowPassFinalize();
    ExpShadowPassFinalize();
    ShadowPassFinalize();

    // Free pencil texture
    for(int i = 0; i < NUM_PENCIL_TEXTURES; ++i)
    {
        pencilTexture[i].view.Finalize( &DEMODevice );
        pencilTexture[i].texture.Finalize( &DEMODevice );
        pencilTexture[i].pPool->Finalize();
    }
}

// The draw function for the rendering portions of this app
static void SceneDraw()
{
    DEMOGfxBeforeRender();

    // Render scene using multiple passes
    ShadowPassDraw();
    ExpShadowPassDraw();
    BlurHoriShadowPassDraw();
    BlurVertShadowPassDraw();
    GeometryPassDraw();
    OutlinePassDraw();
    LightingPassDraw();
    InteriorPassDraw();
    ContoursPassDraw();

    // DRAW G-BUFFERS
    BuffersPassDraw();

    DEMOGfxDoneRender();
}

// Render shadow buffer
static void ShadowPassDraw()
{
    DEMOCommandBuffer.ClearDepthStencil( &ShadowPass.myDepthBufferView,
                     1.0, 0, nn::gfx::DepthStencilClearMode_DepthStencil, NULL );

#if NN_GFX_IS_TARGET_GX
    GX2SetShaderMode( GX2_SHADER_MODE_UNIFORM_BLOCK );
#endif
    DEMOCommandBuffer.SetRenderTargets( 0, NULL, &ShadowPass.myDepthBufferView );
    DEMOCommandBuffer.SetViewportScissorState( &ShadowPass.viewportScissor );


    DEMOCommandBuffer.SetPipeline( &ShadowPass.pipeline );

    // Bind vertex & normal buffer
    DEMOCommandBuffer.SetVertexBuffer( ShadowPass.posLoc, GeometryPass.positionBuffers[s_modelIdx].gpuAddress, MODEL_VTX_STRIDE, GeometryPass.positionBuffers[s_modelIdx].size );

    // Uniform Location Lookup
    DEMOCommandBuffer.SetConstantBuffer( ShadowPass.uniformLoc, nn::gfx::ShaderStage_Vertex, ShadowPass.uniforms.gpuAddress, ShadowPass.uniforms.size );

    // Draw Model
    DEMOCommandBuffer.DrawIndexed( nn::gfx::PrimitiveTopology_TriangleList, nn::gfx::IndexFormat_Uint32, GeometryPass.indexBuffers[s_modelIdx].gpuAddress,
        MODEL_VTX_COUNT[s_modelIdx], 0 );

#if NN_GFX_IS_TARGET_GX
    // Due to HiZ which is implicitly allocated we must expand during render to depth texture
    GX2UTSetExpandDepthState(GX2_ENABLE);
    GX2UTExpandDepthBufferOp( reinterpret_cast< GX2DepthBuffer* >( &ShadowPass.myDepthBufferView.ToData()->gx2DepthBuffer) );
    GX2UTSetExpandDepthState(GX2_DISABLE);
#endif

    DEMOCommandBuffer.FlushMemory( nn::gfx::GpuAccess_DepthStencil );
    DEMOCommandBuffer.InvalidateMemory( nn::gfx::GpuAccess_Texture );
}

// Make the shadow map into an exponential shadow map
static void ExpShadowPassDraw()
{
    const nn::gfx::ColorTargetView* colorBuffers[] = { &ExpShadowPass.myColorBufferView };

    DEMOCommandBuffer.SetViewportScissorState( &ExpShadowPass.viewportScissor );
    DEMOCommandBuffer.SetRenderTargets( 1, colorBuffers, NULL );
    DEMOCommandBuffer.SetPipeline( &ExpShadowPass.pipeline );

    // Bind position buffer
    DEMOCommandBuffer.SetVertexBuffer( ExpShadowPass.posLoc, ExpShadowPass.positionBuffer.gpuAddress, sizeof(float) * 3, ExpShadowPass.positionBuffer.size );

    // Bind texcoord buffer
    DEMOCommandBuffer.SetVertexBuffer( ExpShadowPass.texcoordLoc, ExpShadowPass.textureCoordinateBuffer.gpuAddress, sizeof(float) * 2, ExpShadowPass.textureCoordinateBuffer.size );

    // ------------------------------------------------------------
    DEMOCommandBuffer.SetTextureAndSampler( ExpShadowPass.textureLoc, nn::gfx::ShaderStage_Pixel, ExpShadowPass.myDepthBufferSlot, ExpShadowPass.mySamplerSlot );

    // -----------------------
    // draw quad
    DEMOCommandBuffer.DrawIndexed( nn::gfx::PrimitiveTopology_TriangleList, nn::gfx::IndexFormat_Uint32, ExpShadowPass.indexBuffer.gpuAddress,
        ExpShadowPass.numIndices, 0, 1, 0 );

    DEMOCommandBuffer.FlushMemory( nn::gfx::GpuAccess_ColorBuffer );
    DEMOCommandBuffer.InvalidateMemory( nn::gfx::GpuAccess_Texture );
}

// Blur the exponential shadow map horizontally
static void BlurHoriShadowPassDraw()
{
    nn::gfx::ColorTargetView* colorTargets[ 1 ];
    colorTargets[ 0 ] = &BlurHoriShadowPass.myColorBufferView;

    DEMOCommandBuffer.SetRenderTargets( 1, colorTargets, NULL );
    DEMOCommandBuffer.SetPipeline( &BlurHoriShadowPass.pipeline );
    DEMOCommandBuffer.SetViewportScissorState( &BlurHoriShadowPass.viewportScissor );

    // Bind vertex buffers
    DEMOCommandBuffer.SetVertexBuffer( BlurHoriShadowPass.posLoc, BlurHoriShadowPass.positionBuffer.gpuAddress, sizeof(float) * 3, BlurHoriShadowPass.positionBuffer.size );
    DEMOCommandBuffer.SetVertexBuffer( BlurHoriShadowPass.texcoordLoc, BlurHoriShadowPass.textureCoordinateBuffer.gpuAddress, sizeof(float) * 2, BlurHoriShadowPass.textureCoordinateBuffer.size );

    // ------------------------------------------------------------

    DEMOCommandBuffer.SetTextureAndSampler( BlurHoriShadowPass.textureLoc, nn::gfx::ShaderStage_Pixel, BlurHoriShadowPass.myTextureBufferSlot, BlurHoriShadowPass.mySamplerSlot );

    // -----------------------
    // draw quad
    DEMOCommandBuffer.DrawIndexed( nn::gfx::PrimitiveTopology_TriangleList, nn::gfx::IndexFormat_Uint32, BlurHoriShadowPass.indexBuffer.gpuAddress,
        BlurHoriShadowPass.numIndices, 0, 1, 0 );

    DEMOCommandBuffer.FlushMemory( nn::gfx::GpuAccess_ColorBuffer );
    DEMOCommandBuffer.InvalidateMemory( nn::gfx::GpuAccess_Texture );
}

// Blur the exponential shadow map vertically
static void BlurVertShadowPassDraw()
{
    nn::gfx::ColorTargetView* colorTargets[ 1 ];
    colorTargets[ 0 ] = &BlurVertShadowPass.myColorBufferView;

    DEMOCommandBuffer.SetRenderTargets( 1, colorTargets, NULL );
    DEMOCommandBuffer.SetPipeline( &BlurVertShadowPass.pipeline );
    DEMOCommandBuffer.SetViewportScissorState( &BlurVertShadowPass.viewportScissor );

    // Bind vertex buffers
    DEMOCommandBuffer.SetVertexBuffer( BlurVertShadowPass.posLoc, BlurVertShadowPass.positionBuffer.gpuAddress, sizeof(float) * 3, BlurVertShadowPass.positionBuffer.size );
    DEMOCommandBuffer.SetVertexBuffer( BlurVertShadowPass.texcoordLoc, BlurVertShadowPass.textureCoordinateBuffer.gpuAddress, sizeof(float) * 2, BlurVertShadowPass.textureCoordinateBuffer.size );

    // ------------------------------------------------------------

    DEMOCommandBuffer.SetTextureAndSampler( BlurVertShadowPass.textureLoc, nn::gfx::ShaderStage_Pixel, BlurVertShadowPass.myTextureBufferSlot, BlurVertShadowPass.mySamplerSlot );

    // -----------------------
    // draw quad
    DEMOCommandBuffer.DrawIndexed( nn::gfx::PrimitiveTopology_TriangleList, nn::gfx::IndexFormat_Uint32, BlurVertShadowPass.indexBuffer.gpuAddress,
        BlurVertShadowPass.numIndices, 0, 1, 0 );

    DEMOCommandBuffer.FlushMemory( nn::gfx::GpuAccess_ColorBuffer );
    DEMOCommandBuffer.InvalidateMemory( nn::gfx::GpuAccess_Texture );

}

// Render Geometry Info to G-Buffers
static void GeometryPassDraw()
{
    nn::gfx::ColorTargetView* colorBuffers[2] = { &GeometryPass.myColorBufferView[0], &GeometryPass.myColorBufferView[1] };

    DEMOCommandBuffer.ClearColor( colorBuffers[0], 0.4f, 0.4f, 0.4f, 0.0f, NULL );
    DEMOCommandBuffer.ClearColor( colorBuffers[1], 0.0f, 0.0f, 0.0f, 0.0f, NULL );
    DEMOCommandBuffer.ClearDepthStencil( &GeometryPass.myDepthBufferView,
        1.0f, 0,
        nn::gfx::DepthStencilClearMode_DepthStencil, NULL );

#if NN_GFX_IS_TARGET_GX
    GX2SetShaderMode( GX2_SHADER_MODE_UNIFORM_BLOCK );
#endif
    DEMOCommandBuffer.SetRenderTargets( 2, colorBuffers, &GeometryPass.myDepthBufferView );

    DEMOCommandBuffer.SetViewportScissorState( &GeometryPass.viewportScissor );
    DEMOCommandBuffer.SetPipeline( &GeometryPass.pipeline );

    // Bind vertex & normal buffer
    DEMOCommandBuffer.SetVertexBuffer( GeometryPass.posLoc, GeometryPass.positionBuffers[s_modelIdx].gpuAddress, MODEL_VTX_STRIDE, GeometryPass.positionBuffers[s_modelIdx].size );
    DEMOCommandBuffer.SetVertexBuffer( GeometryPass.nrmLoc, GeometryPass.normalBuffers[s_modelIdx].gpuAddress, MODEL_VTX_STRIDE, GeometryPass.normalBuffers[s_modelIdx].size );

    DEMOCommandBuffer.SetConstantBuffer( GeometryPass.vertexUniformBufferLoc, nn::gfx::ShaderStage_Vertex, GeometryPass.vertexUniformBuffer.gpuAddress, GeometryPass.vertexUniformBuffer.size );
    DEMOCommandBuffer.SetConstantBuffer( GeometryPass.pixelUniformBufferLoc, nn::gfx::ShaderStage_Pixel, GeometryPass.pixelUniformBuffer.gpuAddress, GeometryPass.pixelUniformBuffer.size );

    DEMOCommandBuffer.DrawIndexed( nn::gfx::PrimitiveTopology_TriangleList, nn::gfx::IndexFormat_Uint32, GeometryPass.indexBuffers[s_modelIdx].gpuAddress,
        MODEL_VTX_COUNT[s_modelIdx], 0, 1, 0 );

#if NN_GFX_IS_TARGET_GX
    // Due to HiZ which is implicitly allocated we must expand during render to depth texture
    GX2UTSetExpandDepthState(GX2_ENABLE);
    GX2UTExpandDepthBufferOp( reinterpret_cast< GX2DepthBuffer* >( &GeometryPass.myDepthBufferView.ToData()->gx2DepthBuffer) );
    GX2UTSetExpandDepthState(GX2_DISABLE);
#endif

    DEMOCommandBuffer.FlushMemory( nn::gfx::GpuAccess_ColorBuffer | nn::gfx::GpuAccess_DepthStencil );
    DEMOCommandBuffer.InvalidateMemory( nn::gfx::GpuAccess_Texture );
}

// Detect outlines using depth and normal buffers
static void OutlinePassDraw()
{
    nn::gfx::ColorTargetView* colorTargets[1] = { &OutlinePass.myColorBufferView };

    DEMOCommandBuffer.ClearColor( colorTargets[0], 0.0f, 0.0f, 0.0f, 0.0f, NULL );

#if NN_GFX_IS_TARGET_GX
    GX2SetShaderMode( GX2_SHADER_MODE_UNIFORM_BLOCK );
#endif
    DEMOCommandBuffer.SetViewportScissorState( &OutlinePass.viewportScissor );
    DEMOCommandBuffer.SetRenderTargets( 1, colorTargets, NULL );
    DEMOCommandBuffer.SetPipeline( &OutlinePass.pipeline );

    // Bind vertex & normal buffer
    DEMOCommandBuffer.SetVertexBuffer( OutlinePass.posLoc, OutlinePass.positionBuffer.gpuAddress, sizeof(float) * 3, OutlinePass.positionBuffer.size );
    DEMOCommandBuffer.SetVertexBuffer( OutlinePass.texcoordLoc, OutlinePass.textureCoordinateBuffer.gpuAddress, sizeof(float) * 2, OutlinePass.textureCoordinateBuffer.size );

    DEMOCommandBuffer.SetConstantBuffer( OutlinePass.surfaceSizeLoc, nn::gfx::ShaderStage_Pixel, OutlinePass.uniformBuffer.gpuAddress, OutlinePass.uniformBuffer.size );

    // ------------------------------------------------------------

    DEMOCommandBuffer.SetTextureAndSampler( OutlinePass.textureLoc[0], nn::gfx::ShaderStage_Pixel, OutlinePass.myTextureBufferSlot[0], OutlinePass.mySamplerSlot );
    DEMOCommandBuffer.SetTextureAndSampler( OutlinePass.textureLoc[1], nn::gfx::ShaderStage_Pixel, OutlinePass.myTextureBufferSlot[1], OutlinePass.mySamplerSlot );

    // -----------------------
    // draw quad
    DEMOCommandBuffer.DrawIndexed( nn::gfx::PrimitiveTopology_TriangleList, nn::gfx::IndexFormat_Uint32, OutlinePass.indexBuffer.gpuAddress,
        OutlinePass.numIndices, 0, 1, 0 );

    DEMOCommandBuffer.FlushMemory( nn::gfx::GpuAccess_ColorBuffer );
    DEMOCommandBuffer.InvalidateMemory( nn::gfx::GpuAccess_Texture );
}

// Draw Lit Models using G-Buffers
static void LightingPassDraw()
{
    nn::gfx::ColorTargetView* colorTargets[] = { &LightingPass.myColorBufferView };

    DEMOCommandBuffer.ClearColor( colorTargets[0], 1.0f, 1.0f, 1.0f, 0.0f, NULL );

#if NN_GFX_IS_TARGET_GX
    GX2SetShaderMode( GX2_SHADER_MODE_UNIFORM_BLOCK );
#endif
    DEMOCommandBuffer.SetRenderTargets( 1, colorTargets, NULL );
    DEMOCommandBuffer.SetViewportScissorState( &LightingPass.viewportScissor );
    DEMOCommandBuffer.SetPipeline( &LightingPass.pipeline );

    DEMOCommandBuffer.SetVertexBuffer( LightingPass.posLoc, LightingPass.positionBuffer.gpuAddress, sizeof(float) * 3, LightingPass.positionBuffer.size );
    DEMOCommandBuffer.SetVertexBuffer( LightingPass.texcoordLoc, LightingPass.textureCoordinateBuffer.gpuAddress, sizeof(float) * 2, LightingPass.textureCoordinateBuffer.size );

    // ------------------------------------------------------------
    DEMOCommandBuffer.SetTextureAndSampler( LightingPass.textureLoc[0], nn::gfx::ShaderStage_Pixel, LightingPass.myTextureBufferSlot[0], LightingPass.mySamplerSlot );
    DEMOCommandBuffer.SetTextureAndSampler( LightingPass.textureLoc[1], nn::gfx::ShaderStage_Pixel, LightingPass.myTextureBufferSlot[1], LightingPass.mySamplerSlot );
    DEMOCommandBuffer.SetTextureAndSampler( LightingPass.textureLoc[2], nn::gfx::ShaderStage_Pixel, LightingPass.myTextureBufferSlot[2], LightingPass.mySamplerSlot );
    DEMOCommandBuffer.SetTextureAndSampler( LightingPass.textureLoc[3], nn::gfx::ShaderStage_Pixel, LightingPass.myTextureBufferSlot[3], LightingPass.myDepthSamplerSlot );
    DEMOCommandBuffer.SetTextureAndSampler( LightingPass.textureLoc[4], nn::gfx::ShaderStage_Pixel, LightingPass.myTextureBufferSlot[4], LightingPass.myDepthSamplerSlot );

    // Uniform Location Lookup
    DEMOCommandBuffer.SetConstantBuffer( LightingPass.uniformBlockLoc, nn::gfx::ShaderStage_Pixel, LightingPass.uniformBuffer.gpuAddress, LightingPass.uniformBuffer.size );

    // -----------------------
    // draw quad
    DEMOCommandBuffer.DrawIndexed( nn::gfx::PrimitiveTopology_TriangleList, nn::gfx::IndexFormat_Uint32, LightingPass.indexBuffer.gpuAddress,
        LightingPass.numIndices, 0, 1, 0 );

    DEMOCommandBuffer.FlushMemory( nn::gfx::GpuAccess_ColorBuffer );
    DEMOCommandBuffer.InvalidateMemory( nn::gfx::GpuAccess_Texture );
}

// Draw Sketchy Interior
static void InteriorPassDraw()
{
    nn::gfx::ColorTargetView* pCurrentScanBuffer = DEMOGetColorBufferView();
    nn::gfx::ColorTargetView* colorTargets[] = { pCurrentScanBuffer };

    DEMOCommandBuffer.ClearColor( colorTargets[0], 1.0f, 1.0f, 1.0f, 0.0f, NULL );

#if NN_GFX_IS_TARGET_GX
    GX2SetShaderMode( GX2_SHADER_MODE_UNIFORM_BLOCK );
#endif
    DEMOCommandBuffer.SetRenderTargets( 1, colorTargets, NULL );
    DEMOCommandBuffer.SetViewportScissorState( &InteriorPass.viewportScissor );
    DEMOCommandBuffer.SetPipeline( &InteriorPass.pipeline );

    DEMOCommandBuffer.SetVertexBuffer( InteriorPass.posLoc, InteriorPass.positionBuffer.gpuAddress, sizeof(float) * 3, InteriorPass.positionBuffer.size );
    DEMOCommandBuffer.SetVertexBuffer( InteriorPass.texcoordLoc, InteriorPass.textureCoordinateBuffer.gpuAddress, sizeof(float) * 2, InteriorPass.textureCoordinateBuffer.size );

    // ------------------------------------------------------------

    static int pencilIndex = 0;
    static int changeTimer = 0;
    if(++changeTimer % 10 == 0)
    {
        ++pencilIndex %= NUM_PENCIL_TEXTURES;
    }

    DEMOCommandBuffer.SetTextureAndSampler( InteriorPass.textureLoc[0], nn::gfx::ShaderStage_Pixel, InteriorPass.myTextureBufferSlot, InteriorPass.mySamplerSlot );
    DEMOCommandBuffer.SetTextureAndSampler( InteriorPass.textureLoc[1], nn::gfx::ShaderStage_Pixel, pencilTexture[pencilIndex].slot, InteriorPass.mySamplerSlot );

    DEMOCommandBuffer.SetConstantBuffer( InteriorPass.divisionsLoc, nn::gfx::ShaderStage_Pixel, InteriorPass.uniformBuffer.gpuAddress, InteriorPass.uniformBuffer.size );

    // -----------------------
    // draw quad
    DEMOCommandBuffer.DrawIndexed( nn::gfx::PrimitiveTopology_TriangleList, nn::gfx::IndexFormat_Uint32, InteriorPass.indexBuffer.gpuAddress,
        InteriorPass.numIndices, 0, 1, 0 );
}

// Draw Multiple Contours using Outline Texture
static void ContoursPassDraw()
{
    nn::gfx::ColorTargetView* pCurrentScanBuffer = DEMOGetColorBufferView();
    nn::gfx::ColorTargetView* colorTargets[] = { pCurrentScanBuffer };

    DEMOCommandBuffer.SetRenderTargets( 1, colorTargets, NULL );
    DEMOCommandBuffer.SetViewportScissorState( &ContoursPass.viewportScissor );
    DEMOCommandBuffer.SetPipeline( &ContoursPass.pipeline );

    DEMOCommandBuffer.SetVertexBuffer( ContoursPass.posLoc, ContoursPass.positionBuffer.gpuAddress, sizeof(float) * 3, ContoursPass.positionBuffer.size );
    DEMOCommandBuffer.SetVertexBuffer( ContoursPass.texcoordLoc, ContoursPass.textureCoordinateBuffer.gpuAddress, sizeof(float) * 2, ContoursPass.textureCoordinateBuffer.size );

    // ------------------------------------------------------------

    DEMOCommandBuffer.SetTextureAndSampler( ContoursPass.textureLoc, nn::gfx::ShaderStage_Pixel, ContoursPass.myTextureBufferSlot, ContoursPass.mySamplerSlot );

    static float wobble[4][4] = { {0.003f, 40.0f, 2.0f, 4.0f},
                                  {0.002f, 60.0f, 1.0f, 1.0f},
                                  {0.002f, 80.0f, 0.5f, 5.5f},
                                  {0.001f, 55.0f, 5.0f, 6.0f} };

    static u32 changeTimer = 0;
    if(++changeTimer % 10 == 0)
    {
        for(u8 i = 0; i < 4; ++i)
        {
            wobble[i][2] = float(DEMORand() % 1024) / 1024.0f * 6.28f;
            wobble[i][3] = float(DEMORand() % 1024) / 1024.0f * 6.28f;
        }
    }

    ContourUniforms* pUniforms = ContoursPass.uniformBuffer.Map< ContourUniforms >( );
    memcpy( pUniforms, wobble, sizeof( ContourUniforms ) );
#if NN_GFX_IS_TARGET_GX
    GX2EndianSwap( pUniforms, sizeof( ContourUniforms ) );
#endif
    ContoursPass.uniformBuffer.Unmap();

    DEMOCommandBuffer.SetConstantBuffer( ContoursPass.uniformLoc, nn::gfx::ShaderStage_Vertex, ContoursPass.uniformBuffer.gpuAddress, ContoursPass.uniformBuffer.size );

    // -----------------------
    // draw quad
    DEMOCommandBuffer.DrawIndexed( nn::gfx::PrimitiveTopology_TriangleList, nn::gfx::IndexFormat_Uint32, ContoursPass.indexBuffer.gpuAddress,
        ContoursPass.numIndices, 0, 1, 0 );
}

// Draw G-Buffers used for Lighting
static void BuffersPassDraw()
{
    if(s_drawBuffers == FALSE)
    {
        return;
    }

    nn::gfx::ColorTargetView* pCurrentScanBuffer = DEMOGetColorBufferView();
    nn::gfx::ColorTargetView* colorTargets[ 1 ];
    colorTargets[ 0 ] = pCurrentScanBuffer;

    DEMOCommandBuffer.SetRenderTargets( 1, colorTargets, NULL );
    DEMOCommandBuffer.SetViewportScissorState( &BuffersPass.viewportScissor );

    DEMOCommandBuffer.SetVertexBuffer( BuffersPass.posLoc, BuffersPass.positionBuffer.gpuAddress, sizeof(float) * 3, BuffersPass.positionBuffer.size );
    DEMOCommandBuffer.SetVertexBuffer( BuffersPass.texcoordLoc, BuffersPass.textureCoordinateBuffer.gpuAddress, sizeof(float) * 2, BuffersPass.positionBuffer.size );

    // ------------------------------------------------------------


    // Draw Buffers
    for(u32 i = 0; i < sizeof(BuffersPass.myTextureBuffer) / sizeof(BuffersPass.myTextureBuffer[0]); ++i)
    {
        float* mOffset = BuffersPass.uniformBuffer[i].Map< float >( );
        mOffset[0] = -0.875f + float(i % 8) * 0.25f;
        mOffset[1] = 0.5f - float(i / 8);
        mOffset[2] = 0.0f;
        mOffset[3] = 0.0f;
#if NN_GFX_IS_TARGET_GX
        GX2EndianSwap( mOffset, sizeof( float ) * 4 );
#endif
        BuffersPass.uniformBuffer[i].Unmap();

        DEMOCommandBuffer.SetConstantBuffer( BuffersPass.offsetLoc, nn::gfx::ShaderStage_Vertex, BuffersPass.uniformBuffer[i].gpuAddress, BuffersPass.uniformBuffer[i].size );

        DEMOCommandBuffer.SetTextureAndSampler( BuffersPass.textureLoc, nn::gfx::ShaderStage_Pixel, BuffersPass.myTextureBufferSlot[i], BuffersPass.mySamplerSlot );


        // The 3rd buffer has blending enabled.
        DEMOCommandBuffer.SetPipeline( &BuffersPass.pipeline[ i == 3 ? 1 : 0 ] );

        DEMOCommandBuffer.DrawIndexed( nn::gfx::PrimitiveTopology_TriangleList, nn::gfx::IndexFormat_Uint32, BuffersPass.indexBuffer.gpuAddress,
            BuffersPass.numIndices, 0, 1, 0 );
    }
}

//extern "C" void nnMain()
TEST(GfxPencilRendering, Run)
{
    int argc = nnt::GetHostArgc();
    char** argv = nnt::GetHostArgv();

    // always use 32F depth buffer without AA
    int myargc = argc + 1;
    char **myargv = new char*[myargc];
    for(int i = 0; i < argc; ++i)
    {
        myargv[i] = argv[i];
    }
    const char* extraConfig = "DEMO_DB_FORMAT=32F DEMO_AA_MODE=0";
    myargv[ argc ] = new char[ strlen( extraConfig ) + 1 ];
    memcpy( myargv[ argc ], extraConfig, strlen( extraConfig ) + 1 );

    DEMOInit();
    DEMOTestInit(myargc, myargv);
    DEMOGfxInit(myargc, myargv);
    DEMOFontInit();
    DEMOTestIsUseHlslccGlsl() ? g_ShaderFileIdx = 1 : g_ShaderFileIdx = 0;

    delete myargv[ argc ];
    delete [] myargv;

    SceneInit();
    while(DEMOIsRunning())
    {
        SceneDraw();

        // Share some uniform variables among the different passes and
        // update scene variables
        GeometryPassVertexUniforms* pGeometryUniforms = GeometryPass.vertexUniformBuffer.Map< GeometryPassVertexUniforms >();
        ShadowUniforms* pShadowUniforms = ShadowPass.uniforms.Map< ShadowUniforms >();
        SceneUpdate(&pGeometryUniforms->modelMtx44);
        memcpy(&pShadowUniforms->modelMtx44, &pGeometryUniforms->modelMtx44, sizeof(pGeometryUniforms->modelMtx44));

#if NN_GFX_IS_TARGET_GX
        GX2EndianSwap( &pGeometryUniforms->modelMtx44, sizeof(pGeometryUniforms->modelMtx44) );
        GX2EndianSwap( &pShadowUniforms->modelMtx44, sizeof(pShadowUniforms->modelMtx44) );
#endif
        GeometryPass.vertexUniformBuffer.Unmap();
        ShadowPass.uniforms.Unmap();
    }

    SceneFree();

    DEMOFontShutdown();
    DEMOTestShutdown();
    DEMOGfxShutdown();
    DEMOShutdown();

    SUCCEED();
}
