﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <cmath>

#include <gfx/demo.h>
#include <nn/gfx.h>
#include <nnt.h>
#include <nnt/nnt_Argument.h>

#define PI 3.14159265358979323846F

// ----- Surface Information

#define SURFACE_WIDTH  (DEMOColorBufferInfo.GetWidth() )
#define SURFACE_HEIGHT  (DEMOColorBufferInfo.GetHeight() )

// ----- Shader information
static int g_ShaderFileIdx = 0;
static const char * const GSH_SHADER_FILE[] =
{
    "shaders/manyLights",
    "shaders/manyLightsHlslcc",
};

//---------------------------------------------------------------------------*
//  Model Data
//---------------------------------------------------------------------------*/
static const int NUM_MODEL = 4;
static const char* MODEL_FILES[NUM_MODEL][3]
    = {{"geometries/manyLights/elephantPosition.dat",
       "geometries/manyLights/elephantNormal.dat",
       "geometries/manyLights/elephantIdx.dat"},
       {"geometries/manyLights/teapotPosition.dat",
       "geometries/manyLights/teapotNormal.dat",
       "geometries/manyLights/teapotIdx.dat"},
       {"geometries/manyLights/torusPosition.dat",
       "geometries/manyLights/torusNormal.dat",
       "geometries/manyLights/torusIdx.dat"},
       {"geometries/manyLights/spherePosition.dat",
       "geometries/manyLights/sphereNormal.dat",
       "geometries/manyLights/sphereIdx.dat"}};

static u32    modelLen[NUM_MODEL][3];

static u32 MODEL_VTX_COUNT[NUM_MODEL] = {41142, 119400, 8640, 2880};

static const int MODEL_POS_IDX = 0;
static const int MODEL_POS_STRIDE = sizeof( f32 ) * 3;
static const int MODEL_POS_OFFSET = 0;
static const int MODEL_NORMAL_IDX = 1;
static const int MODEL_NORMAL_STRIDE = sizeof( f32 ) * 3;
static const int MODEL_NORMAL_OFFSET = 0;

// -----

// Pointers to GPU buffers
DEMOGfxBuffer positionBuffers[ NUM_MODEL ];
DEMOGfxBuffer normalBuffers[ NUM_MODEL ];
DEMOGfxBuffer indexBuffers[ NUM_MODEL ];
static f32 * posBuf[ NUM_MODEL ];
static f32 * normalBuf[NUM_MODEL];
static u32 * idxBuf[NUM_MODEL];

static u32 s_ticks = 0;
static u32 s_modelIdx = 0;

// Matrices
struct Matrices
{
    Mtx44   modelMtx44;
    Mtx44   viewMtx44;
    Mtx44   projMtx44;
};

// Light Info
static const int MAX_LIGHT = 100;

typedef struct _LightInfo{
    f32 theta, phi;
    f32 vtheta, vphi;
    Vec power;
}LightInfo;

static LightInfo s_lights[MAX_LIGHT];
static float s_Radius = 150.0;

struct Lights
{
    int numLight[4];
    Quaternion s_lightPosLE[ MAX_LIGHT ];
    Quaternion s_lightPowerLE[ MAX_LIGHT ];
};

static u32 s_numLight = 10;

static int myMatrixBlock;
static DEMOGfxBuffer matrixBuffer;
static int myLightsBlock;
static DEMOGfxBuffer lightBuffer;

static bool s_lightAnim = TRUE;

DEMOGfxPipeline myPipeline;
nn::gfx::ViewportScissorState myViewport;
static void* myViewportMem;

// Proto Type
static void CameraInit(Mtx44 resultProjMtx44, Mtx44 resultViewMtx44);
static int SceneInit();
static void PrintInfo();
static void UpdateLight();
static int SceneDraw();
static void PadTick();

// Setting Up Pad Information
static void PadTick()
{
    u16 buttonDown = DEMOPadGetButtonDown(0);

    if ( DEMO_PAD_BUTTON_A & buttonDown )
    {
        if ( s_modelIdx < NUM_MODEL - 1 )
        {
            s_modelIdx++;
        }
        else
        {
            s_modelIdx = 0;
        }
        return;
    }

    if ( DEMO_PAD_BUTTON_B & buttonDown )
    {
        if ( s_modelIdx >= 1 )
        {
            s_modelIdx--;
        }
        else
        {
            s_modelIdx = NUM_MODEL - 1;
        }
        return;
    }

    if (DEMO_PAD_BUTTON_Y & buttonDown)
    {
        if ( FALSE == s_lightAnim )
        {
            s_lightAnim = TRUE;
        }
        else
        {
            s_lightAnim = FALSE;
        }
        return;
    }

    if ( ( DEMO_PAD_BUTTON_RIGHT & buttonDown ) || ( DEMO_PAD_BUTTON_LEFT & buttonDown ) )
    {
        u32 i;
        if ( DEMO_PAD_BUTTON_RIGHT & buttonDown )
        {
            s_numLight++;
            if ( s_numLight >= MAX_LIGHT )
            {
                s_numLight = MAX_LIGHT - 1;
            }
        }
        else
        {
            s_numLight--;
            if ( s_numLight < 1 )
            {
                s_numLight = 1;
            }
        }

        struct Lights* pLights = lightBuffer.Map< struct Lights >();
#if NN_GFX_IS_TARGET_GX
        GX2EndianSwap( pLights, sizeof( *pLights ) );
#endif
        pLights->numLight[0] = s_numLight;
        for ( i = 0; i < s_numLight; i++ )
        {
            pLights->s_lightPowerLE[ i ].x = s_lights[ i ].power.x / ( f32 )s_numLight;
            pLights->s_lightPowerLE[ i ].y = s_lights[ i ].power.y / ( f32 )s_numLight;
            pLights->s_lightPowerLE[ i ].z = s_lights[ i ].power.z / ( f32 )s_numLight;
        }
#if NN_GFX_IS_TARGET_GX
        GX2EndianSwap( pLights, sizeof( *pLights ) );
#endif
        lightBuffer.Unmap();

        return;
    }
}

// Init function for setting projection matrix
static void CameraInit( Mtx44 resultProjMtx44, Mtx44 resultViewMtx44 )
{
    // row major matricies
    Mtx   lookAtMtx34;

    Vec     up = {0.0f,  1.0f, 0.0f};
    Vec  objPt = {0.0f, 0.0f, 0.0f};
    Vec camLoc = {0.0f, 0.0f, 300.0f};

    f32   pers = 30.0f;
    f32 aspect = (f32)SURFACE_WIDTH / (f32)SURFACE_HEIGHT;
    f32  znear = 50.0f;
    f32   zfar = 2000.0f;

    // Compute perspective matrix
    MTXPerspective( resultProjMtx44, pers, aspect, znear, zfar );

    // Compute lookAt matrix
    MTXLookAt( lookAtMtx34, &camLoc, &up, &objPt );
    MTX34To44( lookAtMtx34, resultViewMtx44 );
}

// The init function for the rendering portions of this app
static int SceneInit()
{
    u32 i;

    DEMOGfxSetViewportScissorState( &myViewport, &myViewportMem, 0.0f, 0.0f, ( float )SURFACE_WIDTH,
        ( float )SURFACE_HEIGHT, 0.0f, 1.0f, ( float )SURFACE_HEIGHT, false);

    DEMOGfxLoadShadersFromFile( &myPipeline.shaders, 0, GSH_SHADER_FILE[g_ShaderFileIdx] );

    DEMOGfxInitShaderAttribute( &myPipeline.shaders, "a_position", MODEL_POS_IDX, MODEL_POS_OFFSET, nn::gfx::AttributeFormat_32_32_32_Float );
    DEMOGfxInitShaderVertexBuffer( &myPipeline.shaders, 0, MODEL_POS_STRIDE, 0 );
    DEMOGfxInitShaderAttribute( &myPipeline.shaders, "a_normal", MODEL_NORMAL_IDX, MODEL_NORMAL_OFFSET, nn::gfx::AttributeFormat_32_32_32_Float );
    DEMOGfxInitShaderVertexBuffer( &myPipeline.shaders, 1, MODEL_NORMAL_STRIDE, 0 );

    // Initialize random number seed to get consistent values back
    DEMOSRand( 0 );

    for ( i = 0; i < NUM_MODEL; i++ )
    {
        // Position buffer
        posBuf[i] = (f32 *)DEMOGfxLoadModelFile(MODEL_FILES[i][0], &modelLen[i][0]);
        positionBuffers[ i ].Initialize( modelLen[ i ][ 0 ], posBuf[ i ], nn::gfx::GpuAccess_VertexBuffer, 0 );

        // Color(Normal) buffer
        normalBuf[ i ] = ( f32 * )DEMOGfxLoadModelFile( MODEL_FILES[ i ][ 1 ], &modelLen[ i ][ 1 ] );
        normalBuffers[ i ].Initialize( modelLen[ i ][ 1 ], normalBuf[ i ], nn::gfx::GpuAccess_VertexBuffer, 0 );

        // Index buffer
        idxBuf[ i ] = ( u32 * )DEMOGfxLoadModelFile( MODEL_FILES[ i ][ 2 ], &modelLen[ i ][ 2 ] );
        indexBuffers[ i ].Initialize( modelLen[ i ][ 2 ], idxBuf[ i ], nn::gfx::GpuAccess_IndexBuffer, 0 );
    };

    // Uniform Location Lookup
    myLightsBlock = myPipeline.shaders.GetInterfaceSlot( nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_ConstantBuffer, "myLights" );
    lightBuffer.Initialize( sizeof( Lights ), NULL, nn::gfx::GpuAccess_ConstantBuffer, 0 );

    struct Lights* myLightsBuf = lightBuffer.Map< struct Lights >();

    for ( i = 0; i < MAX_LIGHT; i++ )
    {
        myLightsBuf->numLight[0] = s_numLight;

        //float radius = 150.0;
        float power = 45000.0;

        s_lights[i].theta = DEMOFRand() * 179.99f * PI / 180.0f;
        s_lights[i].phi = DEMOFRand() * 359.99f * PI / 180.0f;
        s_lights[i].vtheta = (f32)(DEMOFRand() * 2.0 - 1.0) * PI / 180.0f;
        s_lights[i].vphi = (f32)(DEMOFRand() * 2.0 - 1.0) * PI / 180.0f;

        myLightsBuf->s_lightPosLE[i].x = cosf(s_lights[i].phi) *
                           sinf(s_lights[i].theta) * s_Radius;
        myLightsBuf->s_lightPosLE[ i ].y = sinf( s_lights[ i ].phi ) * s_Radius;
        myLightsBuf->s_lightPosLE[ i ].z = cosf( s_lights[ i ].phi ) *
                           cosf(s_lights[i].theta) * s_Radius;
        myLightsBuf->s_lightPosLE[ i ].w = 1.0f;

        s_lights[i].power.x = DEMOFRand() * power;
        s_lights[i].power.y = DEMOFRand() * power;
        s_lights[i].power.z = DEMOFRand() * power;

        myLightsBuf->s_lightPowerLE[ i ].x = s_lights[ i ].power.x / ( f32 )s_numLight;
        myLightsBuf->s_lightPowerLE[ i ].y = s_lights[ i ].power.y / ( f32 )s_numLight;
        myLightsBuf->s_lightPowerLE[ i ].z = s_lights[ i ].power.z / ( f32 )s_numLight;
        myLightsBuf->s_lightPowerLE[ i ].w = 0.0f;
    }
#if NN_GFX_IS_TARGET_GX
    GX2EndianSwap( myLightsBuf, sizeof( *myLightsBuf ) );
#endif
    lightBuffer.Unmap();

    myMatrixBlock = myPipeline.shaders.GetInterfaceSlot( nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_ConstantBuffer, "myMatrix" );
    matrixBuffer.Initialize( sizeof( Matrices ), NULL, nn::gfx::GpuAccess_ConstantBuffer, 0 );
    struct Matrices* myMatrixBuf = matrixBuffer.Map< struct Matrices >();

    CameraInit( myMatrixBuf->projMtx44, myMatrixBuf->viewMtx44 );
    MTX44Identity( myMatrixBuf->modelMtx44 );
#if NN_GFX_IS_TARGET_GX
    GX2EndianSwap( myMatrixBuf, sizeof( *myMatrixBuf ) );
#endif
    matrixBuffer.Unmap();

    myPipeline.colorTargetStateCount = 1;
    myPipeline.colorTargetStateInfoArray[ 0 ].SetDefault();
    myPipeline.colorTargetStateInfoArray[ 0 ].SetFormat( nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm );
    myPipeline.blendTargetStateCount = 1;
    myPipeline.blendTargetStateInfoArray[ 0 ].SetDefault();

    myPipeline.Initialize( &DEMODevice );

    return 1;
}

static void PrintInfo()
{
    // Set Demo Font state
    // Set Demo Font String
    DEMOFontPrintf(2, 1, "<Simple Manylights>");
    DEMOFontPrintf(3, 2, "- Lights:%d", s_numLight);
    if (FALSE == s_lightAnim)
    {
        DEMOFontPrintf(3, 3, "- Lights Animation: OFF");
    }
    else
    {
        DEMOFontPrintf(3, 3, "- Lights Animation: ON");
    }
    DEMOFontPrintf(3, 5, "- Key L/R change the number of lights");
    DEMOFontPrintf(3, 6, "- Press Y to change light animation");
    DEMOFontPrintf(3, 7, "- Press A/B to change model");
}

// Update Light
static void UpdateLight()
{
    u32 i;
    struct Lights* pLights = lightBuffer.Map< struct Lights >();
#if NN_GFX_IS_TARGET_GX
    GX2EndianSwap( pLights, sizeof( *pLights ) );
#endif
    pLights->numLight[0] = s_numLight;
    for ( i = 0; i < s_numLight; i++ )
    {
        s_lights[i].theta += s_lights[i].vtheta;
        s_lights[i].phi += s_lights[i].vphi;

        pLights->s_lightPosLE[ i ].x = cosf( s_lights[ i ].phi ) *
            sinf( s_lights[ i ].theta ) * s_Radius;
        pLights->s_lightPosLE[ i ].y = sinf( s_lights[ i ].phi ) * s_Radius;
        pLights->s_lightPosLE[ i ].z = cosf( s_lights[ i ].phi ) * cosf(s_lights[i].theta) * s_Radius;
    }
#if NN_GFX_IS_TARGET_GX
    GX2EndianSwap( pLights, sizeof( *pLights ) );
#endif
    lightBuffer.Unmap();
}

// The draw function for the rendering portions of this app
static int SceneDraw()
{
    DEMOGfxBeforeRender();

    // Clear buffers
    nn::gfx::ColorTargetView* pCurrentScanBuffer = DEMOGetColorBufferView();
    nn::gfx::ColorTargetView* colorTargets[ ] = { pCurrentScanBuffer };

    DEMOCommandBuffer.ClearColor( colorTargets[ 0 ], 0.1f, 0.1f, 0.1f, 1.0f, NULL );
    DEMOCommandBuffer.ClearDepthStencil( &DEMODepthBufferView, 1.0f, 0, nn::gfx::DepthStencilClearMode_DepthStencil, NULL );

#if NN_GFX_IS_TARGET_GX
    GX2SetShaderMode( GX2_SHADER_MODE_UNIFORM_BLOCK );
#endif
    DEMOCommandBuffer.SetRenderTargets( 1, colorTargets, &DEMODepthBufferView );
    DEMOCommandBuffer.SetViewportScissorState( &myViewport );
    DEMOCommandBuffer.SetPipeline( &myPipeline.pipeline );

    // Bind vertex & normal buffers
    DEMOCommandBuffer.SetVertexBuffer( MODEL_POS_IDX, positionBuffers[ s_modelIdx ].gpuAddress, MODEL_POS_STRIDE, positionBuffers[ s_modelIdx ].size );
    DEMOCommandBuffer.SetVertexBuffer( MODEL_NORMAL_IDX, normalBuffers[ s_modelIdx ].gpuAddress, MODEL_NORMAL_STRIDE, normalBuffers[ s_modelIdx ].size );

    //Uniforms can be overwritten because DEMOGfxDoneRender() does a Sync
    // Camera Update
    struct Matrices* pMtx = matrixBuffer.Map< struct Matrices >();
#if NN_GFX_IS_TARGET_GX
    GX2EndianSwap( pMtx, sizeof( *pMtx ) );
#endif

    CameraInit( pMtx->projMtx44, pMtx->viewMtx44 );
    MTX44Identity( pMtx->modelMtx44 );

#if NN_GFX_IS_TARGET_GX
    GX2EndianSwap( pMtx, sizeof( *pMtx ) );
#endif
    matrixBuffer.Unmap();

    // Light Update
    if ( s_lightAnim == true )
    {
        UpdateLight();
    }
    DEMOCommandBuffer.SetConstantBuffer( myMatrixBlock, nn::gfx::ShaderStage_Vertex, matrixBuffer.gpuAddress, matrixBuffer.size );
    DEMOCommandBuffer.SetConstantBuffer( myLightsBlock, nn::gfx::ShaderStage_Pixel, lightBuffer.gpuAddress, lightBuffer.size );

    // Draw Model
    DEMOCommandBuffer.DrawIndexed( nn::gfx::PrimitiveTopology_TriangleList, nn::gfx::IndexFormat_Uint32,
        indexBuffers[ s_modelIdx ].gpuAddress, MODEL_VTX_COUNT[ s_modelIdx ], 0 );
    // Draw Infomation
    PrintInfo();

    DEMOGfxDoneRender();

    s_ticks++;

    return 1; // 0 makes it exit
}

//extern "C" void nnMain()
TEST(GfxManyLights, Run)
{
    int argc = nnt::GetHostArgc();
    char** argv = nnt::GetHostArgv();

    // Init DEMO lib
    DEMOInit();
    DEMOTestInit(argc, argv);
    DEMOGfxInit(argc, argv);
    DEMOFontInit();
    DEMOTestIsUseHlslccGlsl() ? g_ShaderFileIdx = 1 : g_ShaderFileIdx = 0;
    SceneInit();

    while (DEMOIsRunning())
    {
        DEMOPadRead();
        PadTick();
        SceneDraw();
    }

    // Free various resources
    matrixBuffer.Finalize();
    lightBuffer.Finalize();

    for ( u32 i = 0; i < NUM_MODEL; i++ )
    {
        positionBuffers[ i ].Finalize();
        DEMOGfxFreeMEM2(posBuf[i]);
        normalBuffers[ i ].Finalize();
        DEMOGfxFreeMEM2( normalBuf[ i ] );
        indexBuffers[ i ].Finalize();
        DEMOGfxFreeMEM2( idxBuf[ i ] );
    }

    myPipeline.Finalize( &DEMODevice );
    myViewport.Finalize( &DEMODevice );

    DEMOFontShutdown();
    DEMOTestShutdown();
    DEMOGfxShutdown();
    DEMOShutdown();

    SUCCEED();
}
