﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <string>
#include <nn/vi/vi_Layer.h>

namespace Graphics {

class CheckGpuResource
{
public:

    //! @brief        自動計測用クラスを初期化します。
    //! @param[in]    presentInterval presentinterval を指定します。
    void Initialize(int presentInterval) NN_NOEXCEPT
    {
        m_GpuLoadSum = 0;
        m_GpuLoadAverage = 0;
        m_GpuDelaySum = 0;
        m_GpuDelayAverage = 0;
        m_StableFrameCount = 0;
        m_PresentInterval = presentInterval;
        if (m_PresentInterval == 1)
        {
            m_FpsThreshold = 40;
        }
        else
        {
            m_FpsThreshold = 22;
        }
        m_ObjectCountMax = 500;
        isDefineLargeObjectCount = false;
    }

    //! @brief 自動で最大負荷を計測します。毎フレーム呼んでください。
    bool MeasureMaxGpuLoad(int* objectCount, int* liteObjectCount) NN_NOEXCEPT;
    bool MeasureMaxGpuLoad(int* liteObjectCount) NN_NOEXCEPT;
    bool MeasureMaxGpuLoadImpl(int* objectCount, bool enableShowResult) NN_NOEXCEPT;

    //! @brief 自動で GPU 負荷を最大化します。毎フレーム呼んでください。
    void MaximizeGpuLoad(int* objectCount) NN_NOEXCEPT;

    int GetGpuLoadAverage() NN_NOEXCEPT
    {
        return m_GpuLoadAverage;
    }

    int GetGpuDelayAverage() NN_NOEXCEPT
    {
        return m_GpuDelayAverage;
    }

private:
    int m_GpuLoadSum;
    int m_GpuLoadAverage;
    int m_GpuDelaySum;
    int m_GpuDelayAverage;
    int m_StableFrameCount;
    int m_FpsThreshold;
    int m_PresentInterval;
    int m_ObjectCountMax;
    bool isDefineLargeObjectCount;
};

struct Rgba
{
    float r;
    float g;
    float b;
    float a;
};

// フレームワークモード
enum FrameworkMode
{
    // QueueFinish を使用する従来のフレームワーク。(AppletOption1 + QueueFinish())
    // 非推奨。
    FrameworkMode_QueueFinish,

    // QueueAcquireTexture を使用するフレームワーク。
    // 非推奨ではないが、WindowAcquireTexture を使用するフレームワークをより推奨。
    FrameworkMode_AppletOption1,    // NX と Generic で挙動が異なる

    // WindowAcquireTexture を使用するフレームワーク。
    // vsync の前にコマンドの提出を行い、提出後 vsync を待ってから GPU が処理を始める。( AppletOption3 )
    // GPU を vsync 直後から使用できる。
    FrameworkMode_DeferredExecution,    // NX と Generic で挙動が異なる

    // WindowAcquireTexture を使用するフレームワーク。
    // vsync 後にコマンドの提出を行い、提出後から GPU が処理を始める。( EPD Mode )
    // sead や cafe の実装と互換がある。
    FrameworkMode_DeferredSubmission,

    // 表示の遅延が最少となるフレームワーク。
    FrameworkMode_Immediate,

    // Ocean の OA で使用されるフレームワーク。
    // 30fps で動作する。
    FrameworkMode_Ocean,
};

// テストモード
enum TestMode
{
    // アプリケーションの自動計測テスト。
    TestMode_ApplicationTest,

    // GPU 負荷が高いダミーアプリ。OverlayTest での計測時に使用。
    TestMode_ApplicationHighLoad,

    // GPU 負荷が低いダミーアプリ。OA の計測時に使用。
    TestMode_ApplicationLowLoad,

    // オーバーレイの自動計測テスト。
    TestMode_OverlayTest,

    // 常に描画を行うダミーオーバーレイ。ApplicationTest での計測時に使用。
    TestMode_OverlayVisible,

    // 常に描画を行わないダミーオーバーレイ(初めの10フレームだけは描画する)。ApplicationTest での計測時に使用。
    TestMode_OverlayInvisible,
};

void InitializeGraphics( Rgba clearColor, std::string title, FrameworkMode frameworkMode, TestMode testMode, int presentInterval);

void SetMeasureParameters(int presentInterval);
int GetGpuLoadAverage();
int GetGpuDelayAverage();

void GraphicsThreadFunction(void *arg);

void FinalizeGraphics();

} // Graphics
