﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// -----------------------------------------------------------------------------
//  demoGfd.h
//
// -----------------------------------------------------------------------------

#pragma once

#include <gfx/demo.h>

#ifdef CAFE
#include <cafe/os.h>
#include <cafe/gx2.h>
#include <cafe/gfd.h>
#endif

/// @addtogroup demoGfd
/// @{

/// \brief Get a GX2VertexShader with actual shader data from a block of memory loaded from GX2 shader file (.gsh)
///
/// Given raw data loaded from .gsh shader file, and an initialized GX2VertexShader structure,
/// this fills the GX2VertexShader structure with data from the n'th vertex shader in the file.
///
/// \param ppShader             Pointer-pointer to GX2VertexShader structure
/// \param index                Index of vertex shader to extract.
/// \param pData                Input data, mapped from GX2 Shader file.
/// \retval                     True on success, or false if data is invalid or index is out of range.
///
BOOL DEMOGFDReadVertexShader(void *ppShader, u32 index, const void *pData);

/// \brief Get a GX2PixelShader with actual shader data from a block of memory loaded from GX2 shader file (.gsh)
///
/// Given raw data loaded from .gsh shader file, and an initialized GX2PixelShader structure,
/// this fills the GX2PixelShader structure with data from the n'th pixel shader in the file.
///
/// \param ppShader             Pointer-pointer to GX2PixelShader structure
/// \param index                Index of pixel shader to extract.
/// \param pData                Input data, mapped from GX2 Shader file.
/// \retval                     True on success, or false if data is invalid or index is out of range.
///
BOOL DEMOGFDReadPixelShader(void *ppShader, u32 index, const void *pData);

/// \brief Get a GX2GeometryShader with actual shader data from a block of memory loaded from GX2 shader file (.gsh)
///
/// Given raw data loaded from .gsh shader file, and an initialized GX2GeometryShader structure,
/// this fills the GX2GeometryShader structure with data from the n'th geometry shader in the file.
///
/// \param ppShader             Pointer-pointer to GX2GeometryShader structure
/// \param index                Index of geometry shader to extract.
/// \param pData                Input data, mapped from GX2 Shader file.
/// \retval                     True on success, or false if data is invalid or index is out of range.
///
BOOL DEMOGFDReadGeometryShader(void *ppShader, u32 index, const void *pData);

/// \brief Get a GX2ComputeShader with actual shader data from a block of memory loaded from GX2 shader file (.gsh)
///
/// Given raw data loaded from .gsh shader file, and an initialized GX2ComputeShader structure,
/// this fills the GX2ComputeShader structure with data from the n'th compute shader in the file.
///
/// \param ppShader             Pointer-pointer to GX2ComputeShader structure
/// \param index                Index of vertex shader to extract.
/// \param pData                Input data, mapped from GX2 Shader file.
/// \retval                     True on success, or false if data is invalid or index is out of range.
///
BOOL DEMOGFDReadComputeShader(void *ppShader, u32 index, const void *pData);

/// \brief Get a GX2Texture with actual texture data from a block of memory loaded from a GX2 texture file (.gtx)
///
/// Given raw data loaded from .gtx texture file, and an initialized GX2Texture structure,
/// this fills the GX2Texture structure with data for the n'th texture in the file.
///
/// \param pTexture         Pointer to nn::gfx::Texture
/// \param pInfo            Pointer to nn::gfx::TextureInfo
/// \param pView            Pointer to nn::gfx::TextureView
/// \param pPool            Pointer to DEMOGfxMemPool*
/// \param index            Which texture to extract.
/// \param pData            Input data, mapped from GX2 texture file (.gtx).
/// \retval                 True on success, or false if data is invalid or index is out of range.
///
BOOL DEMOGFDReadTexture(nn::gfx::Texture* pTexture, nn::gfx::TextureInfo* pInfo, nn::gfx::TextureView* pView, DEMOGfxMemPool** pPool, u32 index, const void *pData);

/// \brief Free a GX2VertexShader data
///
/// \param pShader         Pointer to GX2VertexShader Structure
///
void DEMOGFDFreeVertexShader(void* pShader);

/// \brief Free a GX2PixelShader data
///
/// \param pShader         Pointer to GX2PixelShader Structure
///
void DEMOGFDFreePixelShader(void* pShader);

/// \brief Free a GX2GeometryShader data
///
/// \param pShader         Pointer to GX2GeometryShader Structure
///
void DEMOGFDFreeGeometryShader(void* pShader);


/// \brief Free a GX2ComputeShader data
///
/// \param pShader         Pointer to GX2ComputeShader Structure
///
void DEMOGFDFreeComputeShader(void* pShader);

/// \brief Free a GX2Texture data
///
/// \param pTexture         Pointer to nn::gfx::Texture allocated by DEMOGFD
/// \param pTextureView     Pointer to nn::gfx::TextureView setup by DEMOGFD. May be NULL;
///
void DEMOGFDFreeTexture(nn::gfx::Texture *pTexture, nn::gfx::TextureView* pTextureView, DEMOGfxMemPool* pMemPool);

#ifdef CAFE
/// \brief Get GX2Texture pointer with actual texture data from a block of memory loaded from an aligned GX2 texture file (.gtx)
///
/// This buffer is aligned with alignSize.
/// File is automatically closed after it's read from.
/// Call \ref DEMOGFDFreeAlignedTexture() to free this buffer.
///
/// \param pFileBuf Pointer-pointer to file buffer
/// \param index Which texture to extract
/// \param align Alignment size for buffer
/// \param fileName File string name to read
/// \retval GX2Texture pointer
///
GX2Texture* DEMOGFDReadAlignedTexture(void **pFileBuf, u32 index, u32 align, const char *fileName);

/// \brief Free an aligned GX2Texture File buffer
///
/// \param pFileBuf         Pointer to GX2Texture File buffer
///
void DEMOGFDFreeAlignedTexture(void *pFileBuf);
#endif // CAFE

/// @}  // demoGfd
