﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// -----------------------------------------------------------------------------
//  demoDRC.h
//
// -----------------------------------------------------------------------------

#pragma once

#include <gfx/demo.h>

/// @addtogroup demoDRC
/// @{

#define DEMOMOVColorBuffer                   DEMODRCColorBuffer
#define DEMOMOVDepthBuffer                   DEMODRCDepthBuffer
#define DEMOMOVContextState                  DEMODRCContextState
#define DEMOMOVInit                          DEMODRCInit
#define DEMOMOVShutdown                      DEMODRCShutdown
#define DEMOMOVIsRunning                     DEMODRCIsRunning
#define DEMOMOVBeforeRender                  DEMODRCBeforeRender
#define DEMOMOVDoneRender                    DEMODRCDoneRender
#define DEMOMOVSetDefaultRenderTarget        DEMODRCSetDefaultRenderTarget

/// \brief Default color buffer for DEMODRC (global)
extern GX2ColorBuffer DEMODRCColorBuffer;

/// \brief Default depth buffer for DEMODRC (global)
extern GX2DepthBuffer DEMODRCDepthBuffer;

/// \brief Default context state ptr for DEMODRC (global)
extern GX2ContextState *DEMODRCContextState;

/// \brief Structure used to define an instance of DEMODRC
typedef struct
{
    /// \brief Context state ptr for DEMODRC (instance)
    GX2ContextState *contextState;

} DEMODRCInstance;

/// \brief Set up DRC render/depth/scan buffers.
///
/// This function should be called after calling \ref DEMOGfxInit and  before
/// calling any other DRC functions. \n
/// The following arguments can be specified:\n
///
///  -DEMO_DRC_WIDTH n       (set DRC RT width to n) \n
///  -DEMO_DRC_HEIGHT n      (set DRC RT height to n) \n
///  -DEMO_DRC_CB_FORMAT str (set DRC color buffer format to 8_8_8_8 or 10_10_10_2) \n
///  -DEMO_DRC_DB_FORMAT str (set DRC depth buffer format to 16, 32F, 8_24, X24_8_32F) \n
///  -DEMO_DRC_AA_MODE n     (set DRC AA mode to 0/1/2/3) \n
///
/// Set up defaults for all options are (854/480/8_8_8_8/32F/no AA).\n
/// Unknown args are ignored, and the arg list is left unchanged.\n
///
/// \param argc number of arguments
/// \param argv argument values
///
/// \retval A pointer to the instance created by this function
//
DEMODRCInstance* DEMODRCInit(int argc, char *argv[]);

/// \brief Release allocated buffers.
///
void DEMODRCShutdown(void);

/// \brief Release allocated buffers.
///
void DEMODRCReleaseForeground(void);

/// \brief Reacquire allocated buffers.
///
void DEMODRCAcquiredForeground(void);

/// \brief Get demo DRC running state
///
/// \retval TRUE if \ref DEMODRCInit() has been called; false otherwise.
BOOL DEMODRCIsRunning(void);

/// Set demo DRC context to current.
/// Need to call this API before Drawing with demo DRC context created in DEMODRCInit.
void DEMODRCSetContextState(void);

/// \brief Change state for DRC
/// Functionality may change in the future.
///
void DEMODRCBeforeRender(void);

/// \brief Copy render buffer to scan buffers.
/// Note that it does not swap and wait for the new frame to appear.
///
void DEMODRCDoneRender(void);

/// \brief Obtain DRC connect/disconnect status.
///
GX2DRCMode DEMODRCGetStatus(void);

/// \brief A helper function to reset render targets to DEMO DRC buffers.
///
inline void DEMODRCSetDefaultRenderTarget()
{
    GX2SetColorBuffer(&DEMODRCColorBuffer, GX2_RENDER_TARGET_0);
    GX2SetDepthBuffer(&DEMODRCDepthBuffer);
}

/// \brief Creates a new instance of DEMODRC
///
/// This will create a new instance, in addition to setting the instance.
/// This is called by DEMODRCInit().
DEMODRCInstance* DEMODRCAddInstance(void);

/// \brief Deletes an instance of DEMODRC
///
/// This is called by DEMODRCShutdown().
void DEMODRCDeleteInstance(DEMODRCInstance *instance);

/// \brief Sets the instance of DEMODRC
void DEMODRCSetInstance(DEMODRCInstance *instance);

/// \brief Gets the instance of DEMODRC
DEMODRCInstance* DEMODRCGetInstance(void);

/// @}  // demoDRC
