﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "testG3d_ScopedMemoryLeakDetector.h"
#include "testG3d_ViewerTestAllocator.h"
#include "testG3d_FileIoUtility.h"
#include "testG3d_ViewerTestUtility.h"

#include "nnt/g3d/testG3d_TestUtility.h"

#include <model/g3d_EditSkeletonObj.h>
#include <model/g3d_EditModelObj.h>
#include <nn/g3d/viewer/g3d_ViewerServer.h>
#include <nn/g3d/viewer/g3d_ViewerCallbackUtility.h>

#include <nn/g3d/g3d_ModelObj.h>
#include <nn/g3d/g3d_ResFile.h>

#include <nn/gfx/gfx_ResTexture.h>

#include <nn/htcs.h>
#include <nn/gfx/gfx_Device.h>
#include <nn/fs.h>

#include <nnt/nntest.h>
#include <nn/os.h>

using namespace nn::g3d;
using namespace nn::g3d::viewer;
using namespace nn::g3d::viewer::detail;
using namespace G3dTest;

namespace {
    struct ThreadContext
    {
        volatile bool isLiving; //!< スレッドの生存状態
        volatile int testCounter; //!< テスト用に自由に使うことができるカウンター

        ThreadContext()
            : isLiving(false)
            , testCounter(0)
        {
        }
    };

    // nn::os::ThreadStackAlignment でアラインされた値である必要があります。
    static const size_t PollThreadStackSize = 8 * 1024;

    // 通信スレッドのスタック領域です。
    NN_OS_ALIGNAS_THREAD_STACK char g_PollThreadStack[PollThreadStackSize];

    void PollThreadEntryFunction(void *pArg)
    {
        ThreadContext* pContext = reinterpret_cast<ThreadContext*>(pArg);
        const nn::TimeSpan PollInterval = nn::TimeSpan::FromMilliSeconds(1);
        while (pContext->isLiving)
        {
            nn::g3d::viewer::ViewerServer::GetInstance().Poll();
            nn::os::SleepThread(PollInterval);
        }
    }
}

//! @brief ViewerServer の初期化破棄のテストです。
TEST(G3dViewerServerTest, ViewerServerInitializeFinalize)
{
    TestFuncPrePostProcessHandler prePostProcessHandler(__FUNCTION__);
    G3dTest::ScopedMemoryLeakDetector memoryLeakDetector;
    const int TestCount = 3;
    for (int tryCounter = 0; tryCounter < TestCount; ++tryCounter)
    {
        G3dTest::ViewerTestAllocator allocator;
        ViewerTestCallback callback;
        ViewerResult result;

        EXPECT_FALSE(ViewerServer::IsInitialized());

        ViewerServer::InitializeArg initArg(
            GetDevice(),
            G3dTest::ViewerTestAllocator::AllocateWithUserData,
            G3dTest::ViewerTestAllocator::FreeWithUserData,
            &allocator);
        initArg.SetCallback(&callback);
        result = ViewerServer::Initialize(initArg);
        EXPECT_TRUE(result == ViewerResult_Success);
        EXPECT_TRUE(ViewerServer::IsInitialized());

        // 通信スレッド作成
        nn::os::ThreadType pollThread;
        ThreadContext pollThreadContext;
        nn::Result createThreadResult = nn::os::CreateThread(
            &pollThread, PollThreadEntryFunction,
            &pollThreadContext, g_PollThreadStack, PollThreadStackSize, nn::os::LowestThreadPriority);
        EXPECT_TRUE(result == ViewerResult_Success);
        pollThreadContext.isLiving = true;
        nn::os::StartThread(&pollThread);

        ViewerServer& instance = ViewerServer::GetInstance();
        result = instance.Open();
        EXPECT_TRUE(result == ViewerResult_Success);

        // 3DEditor がいない場合は IsOpened も false になる
        EXPECT_FALSE(instance.IsOpened());
        EXPECT_FALSE(instance.IsConnected());

        // 本当はテストドライバーを作って、強制的に接続状態にして内部挙動をテストしたい
        const int ExecuteCount = 100;
        for (int executeCounter = 0; executeCounter < ExecuteCount; ++executeCounter)
        {
            instance.ExecuteCommands();
        }

        instance.Close();

        pollThreadContext.isLiving = false;
        nn::os::WaitThread(&pollThread);
        nn::os::DestroyThread(&pollThread);

        ViewerServer::Finalize();
    }
}
