﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "SampleSources/g3ddemo_DemoUtility.h"
#include <nn/util/util_BitArray.h>
#if defined(NN_BUILD_APISET_GENERIC)
#include "SampleSources/g3ddemo_GenericUtility.h"
#elif defined(NN_BUILD_APISET_NX)
#include "SampleSources/g3ddemo_NXUtility.h"
#endif
#include <nn/vi.h>
#include <nn/font.h>
#include <nn/gfx/util/gfx_DebugFontTextWriter.h>

#include <nn/perf.h>
#include <nns/gfx/gfx_PrimitiveRenderer.h>
#include <nns/gfx/gfx_PrimitiveRendererMeterDrawer.h>
#include <nns/gfx/gfx_GraphicsFramework.h>
#include <nns/gfx/gfx_ColorBuffer.h>
#include <nns/gfx/gfx_DepthStencilBuffer.h>
#include <nns/gfx/gfx_FrameBuffer.h>
#include <nns/g3d.h>

namespace nn { namespace g3d {

class ResFile;
class ResShaderProgram;

}} // namespace nn::g3d

namespace nn { namespace g3d { namespace demo {

enum Max
{
    Max_Width = 8192,
    Max_Height = 8192,
    Max_ScreenTextLength = 256,
};

enum ScanBufferIndex
{
    ScanBufferIndex_Current,
    ScanBufferIndex_Next
};

enum BlendMode
{
    BlendMode_Opaque,
    BlendMode_Translucent,
    BlendMode_TranslucentAdd,
    BlendMode_Costom1,
    BlendMode_Costom2,
    BlendMode_Max
};

enum CullMode
{
    CullMode_Back,
    CullMode_Front,
    CullMode_None,
    CullMode_Max
};

enum FillMode
{
    FillMode_Wireframe,
    FillMode_Solid,
    FillMode_Max
};

enum DepthMode
{
    DepthMode_WriteOn,
    DepthMode_WriteOff,
    DepthMode_Max
};
//--------------------------------------------------------------------------------------------------

void InitializeGfx() NN_NOEXCEPT;
void ShutdownGfx() NN_NOEXCEPT;
nn::vi::NativeWindowHandle GetWindowHandle() NN_NOEXCEPT;

nn::gfx::DescriptorSlot RegisterTextureToDescriptorPool(nn::gfx::TextureView* pTextureView) NN_NOEXCEPT;
void UnregisterTextureFromDescriptorPool(nn::gfx::TextureView* pTextureView) NN_NOEXCEPT;
void RegisterTextureFileToDescriptorPool(nn::gfx::ResTextureFile* pTextureFile) NN_NOEXCEPT;
void UnregisterTextureFileFromDescriptorPool(nn::gfx::ResTextureFile* pTextureFile) NN_NOEXCEPT;
nn::gfx::DescriptorSlot RegisterSamplerToDescriptorPool(nn::gfx::Sampler* pSampler) NN_NOEXCEPT;
void UnregisterSamplerFromDescriptorPool(nn::gfx::Sampler* pSampler) NN_NOEXCEPT;
void RegisterSamplerToDescriptorPool(nn::g3d::ResFile* pResFile) NN_NOEXCEPT;
void UnregisterSamplerFromDescriptorPool(nn::g3d::ResFile* pResFile) NN_NOEXCEPT;

//! @brief モデルの持つすべてのサンプラーをディスクリプタプールへ再登録します。
void ReregisterSamplerToDescriptorPool(nn::g3d::ModelObj* pModelObj) NN_NOEXCEPT;

int  MakeGfxResShaderFileList(const nn::gfx::ResShaderFile*** pGfxShaderFileList, const Vector<nn::g3d::ResShaderFile*>& shaderFileList) NN_NOEXCEPT;
void SetShaderScratchMemory(const Vector<nn::g3d::ResShaderFile*>& shaderFileList) NN_NOEXCEPT;
void SetShaderScratchMemory(const Vector<nn::g3d::ResShaderArchive*>& shaderArchiveList) NN_NOEXCEPT;

void CreateDummyTextureAndSampler(nn::gfx::Device* pDevice) NN_NOEXCEPT;
void DeleteDummyTextureAndSampler(nn::gfx::Device* pDevice) NN_NOEXCEPT;
nn::gfx::DescriptorSlot GetDummySamplerDescriptorSlot() NN_NOEXCEPT;
nn::gfx::ResTexture* GetDummyTexture() NN_NOEXCEPT;

const nn::gfx::BlendState* GetBlendState(int type) NN_NOEXCEPT;
const nn::gfx::RasterizerState* GetRasterizerState(int callMode, int fillMode = FillMode_Solid) NN_NOEXCEPT;
const nn::gfx::DepthStencilState* GetDepthStencilState(int type) NN_NOEXCEPT;

//--------------------------------------------------------------------------------------------------

class ScreenInfo
{
public:

    ScreenInfo() NN_NOEXCEPT
        : m_Width(100)
        , m_Height(100)
        , m_MarginWidth(4)
        , m_MarginHeight(4)
        , m_Red(0)
        , m_Green(0)
        , m_Blue(0)
        , m_Alpha(128)
        , m_QuadCount(0)
        , m_BufferIndex(0)
        , m_pPrimitiveRenderer(nullptr)
        , m_pDebugWriter(nullptr)
    {
    }

    struct QuadParameter
    {
        float x;
        float y;
        float width;
        float height;
        int layer;
        nn::gfx::DescriptorSlot textureDescriptorSlot;
        nn::gfx::DescriptorSlot samplerDescriptorSlot;
        nn::util::Color4u8 color;
    };

    void Initialize(nn::gfx::Device* pDevice) NN_NOEXCEPT;
    void Cleanup(nn::gfx::Device* pDevice) NN_NOEXCEPT;

    void StartPrint() NN_NOEXCEPT;
    void EndPrint() NN_NOEXCEPT {};
    void Print(const char* format, ...) NN_NOEXCEPT;
    void Print(float x, float y, const char* format, ...) NN_NOEXCEPT;
    void Draw(nn::gfx::CommandBuffer* pCommandBuffer) NN_NOEXCEPT;

    void AddQuad(float x, float y, float width, float height, const nn::util::Color4u8& color) NN_NOEXCEPT;
    void AddQuadTexture(float x, float y, float width, float height, const nn::util::Color4u8& color, const nn::gfx::DescriptorSlot& textureDescriptorSlot, const nn::gfx::DescriptorSlot& samplerDescriptorSlot) NN_NOEXCEPT;
    void AddQuadTexture(float x, float y, float width, float height, int layer, const nn::util::Color4u8& color, const nn::gfx::DescriptorSlot& textureDescriptorSlot, const nn::gfx::DescriptorSlot& samplerDescriptorSlot) NN_NOEXCEPT;
    void SetTextColor(uint8_t red, uint8_t green , uint8_t blue, uint8_t alpha) NN_NOEXCEPT;
    void SetFontSize(float width) NN_NOEXCEPT;

    void SetWindowColor(uint8_t red, uint8_t green, uint8_t blue, uint8_t alpha) NN_NOEXCEPT
    {
        m_Red = red;
        m_Green = green;
        m_Blue = blue;
        m_Alpha = alpha;
    }

    void SetWindowPosition(float x, float y) NN_NOEXCEPT
    {
        m_PosX = x;
        m_PosY = y;
    }

    void SetWindowSize(float width, float height) NN_NOEXCEPT
    {
        m_Width = width;
        m_Height = height;
    }

    void SetMarginSize(float width, float height) NN_NOEXCEPT
    {
        m_MarginWidth = width;
        m_MarginHeight = height;
    }

    void AdjustWindowSize() NN_NOEXCEPT
    {
        m_Width = m_AdjustWidth + m_MarginWidth * 2; // 上下左右の余白を追加する。
        m_Height = m_AdjustHeight + m_MarginHeight * 2;
    }

    float GetWindowWidth() const NN_NOEXCEPT
    {
        return m_Width;
    }

    float GetWindowHeight() const NN_NOEXCEPT
    {
        return m_Height;
    }

    float GetMarginWidth() const NN_NOEXCEPT
    {
        return m_MarginWidth;
    }

    float GetMarginHeight() const NN_NOEXCEPT
    {
        return m_MarginHeight;
    }

    float GetWindowPositionX() const NN_NOEXCEPT
    {
        return m_PosX;
    }

    float GetWindowPositionY() const NN_NOEXCEPT
    {
        return m_PosY;
    }

    float GetFontHeight() const NN_NOEXCEPT;

    float GetTextWidth(const char* text) const NN_NOEXCEPT;

private:
    float m_PosX;
    float m_PosY;
    float m_Width;
    float m_Height;
    float m_AdjustWidth;
    float m_AdjustHeight;
    float m_MarginWidth;
    float m_MarginHeight;
    uint8_t m_Red;
    uint8_t m_Green;
    uint8_t m_Blue;
    uint8_t m_Alpha;
    nn::util::Matrix4x4fType m_ProjectionMtx;
    nn::util::Matrix4x3fType m_ViewMtx;
    QuadParameter m_QuadArray[32];
    int m_QuadCount;
    int m_BufferIndex; // プリミティブレンダラのバッファーを交互に切り替える

    // プリミティブレンダラ用
    nns::gfx::PrimitiveRenderer::Renderer* m_pPrimitiveRenderer;
    nn::gfx::BlendState      m_BlendState;
    nn::gfx::RasterizerState m_RasterizerState;
    ptrdiff_t m_MemoryOffset;

    // フォント用
    nns::gfx::GraphicsFramework::DebugFontTextWriter* m_pDebugWriter;
};

//--------------------------------------------------------------------------------------------------

class LoadMeter
{
public:
    LoadMeter() NN_NOEXCEPT
    {}

    void Initialize(nn::gfx::Device* pDevice, nn::perf::LoadMeterCenterInfo info) NN_NOEXCEPT;
    void Cleanup(nn::gfx::Device* pDevice) NN_NOEXCEPT;
    void Draw(nn::gfx::CommandBuffer* pCommandBuffer) NN_NOEXCEPT;

    void SetScale(int scale) NN_NOEXCEPT
    {
        m_pMeterDrawer->SetScale(scale);
    }

private:
    nns::gfx::PrimitiveRenderer::MeterDrawer* m_pMeterDrawer;
    nns::gfx::PrimitiveRenderer::Renderer* m_pPrimitiveRenderer;
    nns::gfx::GraphicsFramework::DebugFontTextWriter* m_pDebugWriter;
    ptrdiff_t m_MemoryOffset;
#if defined( NN_PERF_PROFILE_ENABLED )
    void* m_pLoadMeterMemory;
    ptrdiff_t m_MeterMemoryPoolOffset;
#endif
};

//--------------------------------------------------------------------------------------------------

void LoadUniformBlock(
    nn::gfx::CommandBuffer* pCommandBuffer,
    const nn::gfx::Buffer* pBuffer,
    const nn::g3d::ResShaderProgram* pShaderProgram,
    int blockIndex, size_t size
) NN_NOEXCEPT;

void LoadShaderStorageBlock(
    nn::gfx::CommandBuffer* pCommandBuffer,
    const nn::gfx::Buffer* pBuffer,
    const nn::g3d::ResShaderProgram* pShaderProgram,
    int blockIndex, size_t size
) NN_NOEXCEPT;

void LoadTextureSampler(
    nn::gfx::CommandBuffer* pCommandBuffer,
    const nn::gfx::DescriptorSlot& textureDescriptor,
    const nn::gfx::DescriptorSlot& samplerDescriptor,
    const nn::g3d::ResShaderProgram* pShaderProgram,
    int samplerIndex
) NN_NOEXCEPT;

void InitializePrimitiveRenderer(nns::gfx::PrimitiveRenderer::Renderer** ppPrimitiveRenderer, ptrdiff_t* pMemoryOffset) NN_NOEXCEPT;

void FinalizePrimitiveRenderer(nns::gfx::PrimitiveRenderer::Renderer** ppPrimitiveRenderer, ptrdiff_t* pMemoryOffset) NN_NOEXCEPT;

void InitializeFrameBuffer(nns::gfx::FrameBuffer* pFrameBuffer, const nns::gfx::FrameBuffer::InfoType* info) NN_NOEXCEPT;

void FinalizeFrameBuffer(nns::gfx::FrameBuffer* pFrameBuffer) NN_NOEXCEPT;

void ClearFrameBuffer(nn::gfx::CommandBuffer* pCommandBuffer, nns::gfx::FrameBuffer* pFrameBuffer) NN_NOEXCEPT;
void SetFrameBufferToRenderTarget(nn::gfx::CommandBuffer* pCommandBuffer, nns::gfx::FrameBuffer* pFrameBuffer) NN_NOEXCEPT;

nns::gfx::GraphicsFramework* GetGfxFramework() NN_NOEXCEPT;

}}} // namespace nn::g3d::demo

