﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "SampleSources/g3ddemo_GfxUtility.h"

namespace nn { namespace g3d { namespace demo {

namespace {

enum
{
    Max_Descriptor = 1024 * 2
};

const size_t g_ConstantBufferMemoryPoolSize = 32 * 1024 * 1024;
const size_t g_OthersMemoryPoolSize = 1024 * 1024;
const size_t g_RenderTargetMemoryPoolSize = 160 * 1024 * 1024;
const size_t g_CommandBufferMemoryPoolSize = 64 * 1024 * 1024;

const size_t g_NnsG3dMemorySize = 16 * 1024 * 1024;
const size_t g_NnsG3dMemoryPoolSize = 16 * 1024 * 1024;

int g_DescriptorSlotIndexArray[Max_Descriptor];
nn::vi::NativeWindowHandle g_WndHandle = nullptr;

// Gfxフレームワーク
nns::gfx::GraphicsFramework g_GfxFramework;

// Gfxフレームワーク用アロケータ
const size_t g_FrameworkHeapSize = 320 * 1024 * 1024;
void* g_pFrameworkMemory = nullptr;
nn::mem::StandardAllocator g_FrameworkAllocator;

// ダミーテクスチャー用
nn::gfx::ResTextureFile* g_pSharedTextureFile = nullptr;
nn::gfx::ResTexture* g_pDummyTexture = nullptr;
nn::gfx::Sampler g_DummySampler;
nn::gfx::DescriptorSlot g_DummySamplerDescriptorSlot;

// プリミティブレンダラ共用リソース
nns::gfx::PrimitiveRenderer::GraphicsResource g_PrimitiveRendererResrouce;
void* g_pPrimitiveRendererResrouceMemory = nullptr;
ptrdiff_t g_pPrimitiveRendererResrouceMemoryPoolOffset = -1;

// ブレンドステート
nn::gfx::BlendState g_BlendState[BlendMode_Max];

// ラスタライザーステート
nn::gfx::RasterizerState g_RasterizerState[CullMode_Max][FillMode_Max];

// デプスステンシルステート
nn::gfx::DepthStencilState g_DepthStencilState[DepthMode_Max];

const int InvalidLayerIndex = -1;

int FindCharCount(const char* string, int maxLength, char targetChar)
{
    int count = 0;
    for (int charIndex = 0, length = static_cast<int>(strnlen(string, maxLength));
        charIndex < length; ++charIndex)
    {
        if (string[charIndex] == targetChar)
        {
            ++count;
        }
    }
    return count;
}

void InitializeBlendState()
{
    nn::gfx::Device* pDevice = g_GfxFramework.GetDevice();
    {
        nn::gfx::BlendStateInfo blendStateInfo;
        blendStateInfo.SetDefault();
        nn::gfx::BlendTargetStateInfo blendTargetStateInfo;
        blendTargetStateInfo.SetDefault();
        blendStateInfo.SetBlendTargetStateInfoArray(&blendTargetStateInfo, 1);
        size_t memorySize = nn::gfx::BlendState::GetRequiredMemorySize(blendStateInfo);

        void* pData = AllocateMemory(memorySize, nn::gfx::BlendState::RequiredMemoryInfo_Alignment);
        g_BlendState[BlendMode_Opaque].SetMemory(pData, memorySize);
        g_BlendState[BlendMode_Opaque].Initialize(pDevice, blendStateInfo);
    }

    {
        nn::gfx::BlendState::InfoType blendStateInfo;
        blendStateInfo.SetDefault();
        nn::gfx::BlendTargetStateInfo blendTargetStateInfo;
        blendTargetStateInfo.SetDefault();
        blendTargetStateInfo.SetBlendEnabled(true);
        blendTargetStateInfo.SetColorBlendFunction(nn::gfx::BlendFunction_Add);
        blendTargetStateInfo.SetSourceColorBlendFactor(nn::gfx::BlendFactor_SourceAlpha);
        blendTargetStateInfo.SetDestinationColorBlendFactor(nn::gfx::BlendFactor_OneMinusSourceAlpha);
        blendTargetStateInfo.SetAlphaBlendFunction(nn::gfx::BlendFunction_Add);
        blendTargetStateInfo.SetSourceAlphaBlendFactor(nn::gfx::BlendFactor_One);
        blendTargetStateInfo.SetDestinationAlphaBlendFactor(nn::gfx::BlendFactor_Zero);
        blendStateInfo.SetBlendTargetStateInfoArray(&blendTargetStateInfo, 1);
        size_t memorySize = nn::gfx::BlendState::GetRequiredMemorySize(blendStateInfo);
        void* pData = AllocateMemory(memorySize, nn::gfx::BlendState::RequiredMemoryInfo_Alignment);
        NN_ASSERT_NOT_NULL(pData);
        g_BlendState[BlendMode_Translucent].SetMemory(pData, memorySize);
        g_BlendState[BlendMode_Translucent].Initialize(pDevice, blendStateInfo);
    }
    {
        nn::gfx::BlendState::InfoType blendStateInfo;
        blendStateInfo.SetDefault();
        nn::gfx::BlendTargetStateInfo blendTargetStateInfo;
        blendTargetStateInfo.SetDefault();
        blendTargetStateInfo.SetBlendEnabled(true);
        blendTargetStateInfo.SetColorBlendFunction(nn::gfx::BlendFunction_Add);
        blendTargetStateInfo.SetSourceColorBlendFactor(nn::gfx::BlendFactor_SourceAlpha);
        blendTargetStateInfo.SetDestinationColorBlendFactor(nn::gfx::BlendFactor_One);
        blendTargetStateInfo.SetAlphaBlendFunction(nn::gfx::BlendFunction_Add);
        blendTargetStateInfo.SetSourceAlphaBlendFactor(nn::gfx::BlendFactor_One);
        blendTargetStateInfo.SetDestinationAlphaBlendFactor(nn::gfx::BlendFactor_Zero);
        blendStateInfo.SetBlendTargetStateInfoArray(&blendTargetStateInfo, 1);
        size_t memorySize = nn::gfx::BlendState::GetRequiredMemorySize(blendStateInfo);
        void* pData = AllocateMemory(memorySize, nn::gfx::BlendState::RequiredMemoryInfo_Alignment);
        NN_ASSERT_NOT_NULL(pData);
        g_BlendState[BlendMode_TranslucentAdd].SetMemory(pData, memorySize);
        g_BlendState[BlendMode_TranslucentAdd].Initialize(pDevice, blendStateInfo);
    }
    {
        nn::gfx::BlendState::InfoType blendStateInfo;
        blendStateInfo.SetDefault();
        nn::gfx::BlendTargetStateInfo blendTargetStateInfo;
        blendTargetStateInfo.SetDefault();
        blendTargetStateInfo.SetBlendEnabled(true);
        blendTargetStateInfo.SetColorBlendFunction(nn::gfx::BlendFunction_Add);
        blendTargetStateInfo.SetSourceColorBlendFactor(nn::gfx::BlendFactor_One);
        blendTargetStateInfo.SetDestinationColorBlendFactor(nn::gfx::BlendFactor_One);
        blendTargetStateInfo.SetAlphaBlendFunction(nn::gfx::BlendFunction_Add);
        blendTargetStateInfo.SetSourceAlphaBlendFactor(nn::gfx::BlendFactor_One);
        blendTargetStateInfo.SetDestinationAlphaBlendFactor(nn::gfx::BlendFactor_One);
        blendStateInfo.SetBlendTargetStateInfoArray(&blendTargetStateInfo, 1);
        size_t memorySize = nn::gfx::BlendState::GetRequiredMemorySize(blendStateInfo);
        void* pData = AllocateMemory(memorySize, nn::gfx::BlendState::RequiredMemoryInfo_Alignment);
        NN_ASSERT_NOT_NULL(pData);
        g_BlendState[BlendMode_Costom1].SetMemory(pData, memorySize);
        g_BlendState[BlendMode_Costom1].Initialize(pDevice, blendStateInfo);
    }
    {
        nn::gfx::BlendState::InfoType blendStateInfo;
        blendStateInfo.SetDefault();
        nn::gfx::BlendTargetStateInfo blendTargetStateInfo;
        blendTargetStateInfo.SetDefault();
        blendTargetStateInfo.SetBlendEnabled(true);
        blendTargetStateInfo.SetColorBlendFunction(nn::gfx::BlendFunction_Add);
        blendTargetStateInfo.SetSourceColorBlendFactor(nn::gfx::BlendFactor_One);
        blendTargetStateInfo.SetDestinationColorBlendFactor(nn::gfx::BlendFactor_Zero);
        blendTargetStateInfo.SetAlphaBlendFunction(nn::gfx::BlendFunction_Add);
        blendTargetStateInfo.SetSourceAlphaBlendFactor(nn::gfx::BlendFactor_One);
        blendTargetStateInfo.SetDestinationAlphaBlendFactor(nn::gfx::BlendFactor_Zero);
        blendStateInfo.SetBlendTargetStateInfoArray(&blendTargetStateInfo, 1);
        size_t memorySize = nn::gfx::BlendState::GetRequiredMemorySize(blendStateInfo);
        void* pData = AllocateMemory(memorySize, nn::gfx::BlendState::RequiredMemoryInfo_Alignment);
        NN_ASSERT_NOT_NULL(pData);
        g_BlendState[BlendMode_Costom2].SetMemory(pData, memorySize);
        g_BlendState[BlendMode_Costom2].Initialize(pDevice, blendStateInfo);
    }
}

void FinalizeBlendState()
{
    nn::gfx::Device* pDevice = g_GfxFramework.GetDevice();
    for (int index = 0; index < BlendMode_Max; ++index)
    {
        void* ptr = g_BlendState[index].GetMemory();
        g_BlendState[index].Finalize(pDevice);
        if (ptr)
        {
            FreeMemory(ptr);
        }
    }
}

void InitializeRasterizerState()
{
    nn::gfx::Device* pDevice = g_GfxFramework.GetDevice();
    {
        nn::gfx::RasterizerStateInfo rasterizerStateInfo;
        rasterizerStateInfo.SetDefault();
        rasterizerStateInfo.SetCullMode(nn::gfx::CullMode_Back);
        rasterizerStateInfo.SetFillMode(nn::gfx::FillMode_Solid);
        rasterizerStateInfo.SetPrimitiveTopologyType(nn::gfx::PrimitiveTopologyType_Triangle);
        rasterizerStateInfo.SetScissorEnabled(true);
        rasterizerStateInfo.SetDepthClipEnabled(false);
        g_RasterizerState[CullMode_Back][FillMode_Solid].Initialize(pDevice, rasterizerStateInfo);

        rasterizerStateInfo.SetFillMode(nn::gfx::FillMode_Wireframe);
        g_RasterizerState[CullMode_Back][FillMode_Wireframe].Initialize(pDevice, rasterizerStateInfo);
    }
    {
        nn::gfx::RasterizerStateInfo rasterizerStateInfo;
        rasterizerStateInfo.SetDefault();
        rasterizerStateInfo.SetCullMode(nn::gfx::CullMode_Front);
        rasterizerStateInfo.SetFillMode(nn::gfx::FillMode_Solid);
        rasterizerStateInfo.SetPrimitiveTopologyType(nn::gfx::PrimitiveTopologyType_Triangle);
        rasterizerStateInfo.SetScissorEnabled(true);
        rasterizerStateInfo.SetDepthClipEnabled(false);
        g_RasterizerState[CullMode_Front][FillMode_Solid].Initialize(pDevice, rasterizerStateInfo);

        rasterizerStateInfo.SetFillMode(nn::gfx::FillMode_Wireframe);
        g_RasterizerState[CullMode_Front][FillMode_Wireframe].Initialize(pDevice, rasterizerStateInfo);
    }
    {
        nn::gfx::RasterizerStateInfo rasterizerStateInfo;
        rasterizerStateInfo.SetDefault();
        rasterizerStateInfo.SetCullMode(nn::gfx::CullMode_None);
        rasterizerStateInfo.SetFillMode(nn::gfx::FillMode_Solid);
        rasterizerStateInfo.SetPrimitiveTopologyType(nn::gfx::PrimitiveTopologyType_Triangle);
        rasterizerStateInfo.SetScissorEnabled(true);
        rasterizerStateInfo.SetDepthClipEnabled(false);
        g_RasterizerState[CullMode_None][FillMode_Solid].Initialize(pDevice, rasterizerStateInfo);

        rasterizerStateInfo.SetFillMode(nn::gfx::FillMode_Wireframe);
        g_RasterizerState[CullMode_None][FillMode_Wireframe].Initialize(pDevice, rasterizerStateInfo);
    }
}

void FinalizeRasterizerState()
{
    nn::gfx::Device* pDevice = g_GfxFramework.GetDevice();
    for (int cullModeIndex = 0; cullModeIndex < CullMode_Max; ++cullModeIndex)
    {
        for (int fillModeIndex = 0; fillModeIndex < FillMode_Max; ++fillModeIndex)
        {
            g_RasterizerState[cullModeIndex][fillModeIndex].Finalize(pDevice);
        }
    }
}

void InitializeDepthStencilState()
{
    nn::gfx::Device* pDevice = g_GfxFramework.GetDevice();
    {
        nn::gfx::DepthStencilStateInfo depthStencilStateInfo;
        depthStencilStateInfo.SetDefault();
        depthStencilStateInfo.SetDepthTestEnabled(true);
        depthStencilStateInfo.SetDepthWriteEnabled(true);
        g_DepthStencilState[DepthMode_WriteOn].Initialize(pDevice, depthStencilStateInfo);
    }
    {
        nn::gfx::DepthStencilStateInfo depthStencilStateInfo;
        depthStencilStateInfo.SetDefault();
        depthStencilStateInfo.SetDepthTestEnabled(true);
        depthStencilStateInfo.SetDepthWriteEnabled(false);
        g_DepthStencilState[DepthMode_WriteOff].Initialize(pDevice, depthStencilStateInfo);
    }
}

void FinalizeDepthStencilState()
{
    nn::gfx::Device* pDevice = g_GfxFramework.GetDevice();
    for (int index = 0; index < DepthMode_Max; ++index)
    {
        g_DepthStencilState[index].Finalize(pDevice);
    }
}

}

const nn::gfx::BlendState* GetBlendState(int type) NN_NOEXCEPT
{
    NN_ASSERT_RANGE(type, 0, BlendMode_Max);
    return &g_BlendState[type];
}

const nn::gfx::RasterizerState* GetRasterizerState(int cullMode, int fillMode) NN_NOEXCEPT
{
    NN_ASSERT_RANGE(cullMode, 0, CullMode_Max);
    NN_ASSERT_RANGE(fillMode, 0, FillMode_Max);
    return &g_RasterizerState[cullMode][fillMode];
}

const nn::gfx::DepthStencilState* GetDepthStencilState(int type) NN_NOEXCEPT
{
    NN_ASSERT_RANGE(type, 0, DepthMode_Max);
    return &g_DepthStencilState[type];
}

//--------------------------------------------------------------------------------------------------
// ディスクリプタープール
nn::gfx::DescriptorSlot RegisterTextureToDescriptorPool(nn::gfx::TextureView* pTextureView) NN_NOEXCEPT
{
    nn::gfx::DescriptorPool* pDescriptorPool = GetGfxFramework()->GetDescriptorPool(nn::gfx::DescriptorPoolType_TextureView);
    pDescriptorPool->BeginUpdate();

    int slotIndex = GetGfxFramework()->AllocateDescriptorSlot(nn::gfx::DescriptorPoolType_TextureView, 1);
    NN_ASSERT(slotIndex != -1);

    pTextureView->SetUserPtr(&g_DescriptorSlotIndexArray[slotIndex]);
    pDescriptorPool->SetTextureView(slotIndex, pTextureView);
    nn::gfx::DescriptorSlot descriptorSlot;
    pDescriptorPool->GetDescriptorSlot(&descriptorSlot, slotIndex);

    pDescriptorPool->EndUpdate();

    return descriptorSlot;
}

void UnregisterTextureFromDescriptorPool(nn::gfx::TextureView* pTextureView) NN_NOEXCEPT
{
    int* pIndex = reinterpret_cast<int*>(pTextureView->GetUserPtr());
    if (pIndex)
    {
        int slotIndex = *pIndex;
        GetGfxFramework()->FreeDescriptorSlot(nn::gfx::DescriptorPoolType_TextureView, slotIndex);
        pTextureView->SetUserPtr(nullptr);
    }
}

void RegisterTextureFileToDescriptorPool(nn::gfx::ResTextureFile* pTextureFile) NN_NOEXCEPT
{
    nn::gfx::DescriptorPool* pDescriptorPool = GetGfxFramework()->GetDescriptorPool(nn::gfx::DescriptorPoolType_TextureView);
    pDescriptorPool->BeginUpdate();
    for (int textureIndex = 0, textureCount = pTextureFile->GetTextureDic()->GetCount(); textureIndex < textureCount; ++textureIndex)
    {
        nn::gfx::ResTexture* pResTexture = pTextureFile->GetResTexture(textureIndex);
        int slotIndex = GetGfxFramework()->AllocateDescriptorSlot(nn::gfx::DescriptorPoolType_TextureView, 1);
        NN_ASSERT(slotIndex != -1);

        nn::gfx::TextureView* pTextureView = pResTexture->GetTextureView();
        pTextureView->SetUserPtr(&g_DescriptorSlotIndexArray[slotIndex]);

        pDescriptorPool->SetTextureView(slotIndex, pTextureView);
        nn::gfx::DescriptorSlot registeredDescriptorSlot;
        pDescriptorPool->GetDescriptorSlot(&registeredDescriptorSlot, slotIndex);
        nn::gfx::DescriptorSlot checkDescriptorSlot;
        pResTexture->GetUserDescriptorSlot(&checkDescriptorSlot);
        NN_ASSERT(checkDescriptorSlot.IsValid());
        pResTexture->SetUserDescriptorSlot(registeredDescriptorSlot);
    }
    pDescriptorPool->EndUpdate();
}

void UnregisterTextureFileFromDescriptorPool(nn::gfx::ResTextureFile* pTextureFile) NN_NOEXCEPT
{
    for (int textureIndex = 0, textureCount = pTextureFile->GetTextureDic()->GetCount(); textureIndex < textureCount; ++textureIndex)
    {
        nn::gfx::TextureView* pTextureView = pTextureFile->GetResTexture(textureIndex)->GetTextureView();
        int* pIndex = reinterpret_cast<int*>(pTextureView->GetUserPtr());
        if (pIndex)
        {
            int slotIndex = *pIndex;
            GetGfxFramework()->FreeDescriptorSlot(nn::gfx::DescriptorPoolType_TextureView, slotIndex);
            pTextureView->SetUserPtr(nullptr);
        }
    }
}

nn::gfx::DescriptorSlot RegisterSamplerToDescriptorPool(nn::gfx::Sampler* pSampler) NN_NOEXCEPT
{
    nn::gfx::DescriptorPool* pDescriptorPool = GetGfxFramework()->GetDescriptorPool(nn::gfx::DescriptorPoolType_Sampler);

    int slotIndex = GetGfxFramework()->AllocateDescriptorSlot(nn::gfx::DescriptorPoolType_Sampler, 1);
    NN_ASSERT(slotIndex != -1);

    pDescriptorPool->BeginUpdate();
    pSampler->SetUserPtr(&g_DescriptorSlotIndexArray[slotIndex]);
    pDescriptorPool->SetSampler(slotIndex, pSampler);
    nn::gfx::DescriptorSlot descriptorSlot;
    pDescriptorPool->GetDescriptorSlot(&descriptorSlot, slotIndex);
    pDescriptorPool->EndUpdate();

    return descriptorSlot;
}

void UnregisterSamplerFromDescriptorPool(nn::gfx::Sampler* pSampler) NN_NOEXCEPT
{
    int* pIndex = reinterpret_cast<int*>(pSampler->GetUserPtr());
    if (pIndex)
    {
        int slotIndex = *(reinterpret_cast<int*>(pSampler->GetUserPtr()));
        GetGfxFramework()->FreeDescriptorSlot(nn::gfx::DescriptorPoolType_Sampler, slotIndex);
        pSampler->SetUserPtr(nullptr);
        return;
    }
    NN_ASSERT_NOT_NULL(pIndex);
}

void RegisterSamplerToDescriptorPool(nn::g3d::ResFile* pResFile) NN_NOEXCEPT
{
    for (int modelIndex = 0; modelIndex < pResFile->GetModelCount(); ++modelIndex)
    {
        nn::g3d::ResModel* pResModel = pResFile->GetModel(modelIndex);
        for (int materialIndex = 0; materialIndex < pResModel->GetMaterialCount(); ++materialIndex)
        {
            nn::g3d::ResMaterial* pResMaterial = pResModel->GetMaterial(materialIndex);
            for (int samplerIndex = 0; samplerIndex < pResMaterial->GetSamplerCount(); ++samplerIndex)
            {
                nn::gfx::Sampler* pSampler = pResMaterial->GetSampler(samplerIndex);
                nn::gfx::DescriptorSlot descriptorSlot = RegisterSamplerToDescriptorPool(pSampler);
                pResMaterial->SetSamplerDescriptorSlot(samplerIndex, descriptorSlot);
            }
        }
    }
}

void UnregisterSamplerFromDescriptorPool(nn::g3d::ResFile* pResFile) NN_NOEXCEPT
{
    for (int modelIndex = 0; modelIndex < pResFile->GetModelCount(); ++modelIndex)
    {
        nn::g3d::ResModel* pResModel = pResFile->GetModel(modelIndex);
        for (int materialIndex = 0; materialIndex < pResModel->GetMaterialCount(); ++materialIndex)
        {
            nn::g3d::ResMaterial* pResMaterial = pResModel->GetMaterial(materialIndex);
            for (int samplerIndex = 0; samplerIndex < pResMaterial->GetSamplerCount(); ++samplerIndex)
            {
                nn::gfx::Sampler* pSampler = pResMaterial->GetSampler(samplerIndex);
                UnregisterSamplerFromDescriptorPool(pSampler);
            }
        }
    }
}

void ReregisterSamplerToDescriptorPool(nn::g3d::ModelObj* pModelObj) NN_NOEXCEPT{
    for (int materialIndex = 0; materialIndex < pModelObj->GetMaterialCount(); ++materialIndex)
    {
        nn::g3d::MaterialObj* pMaterialObj = pModelObj->GetMaterial(materialIndex);
        nn::g3d::ResMaterial* pResMaterial = const_cast<nn::g3d::ResMaterial*>(pMaterialObj->GetResource());
        for (int samplerIndex = 0; samplerIndex < pMaterialObj->GetSamplerCount(); ++samplerIndex)
        {
            nn::g3d::SamplerRef samplerRef = pMaterialObj->GetSampler(samplerIndex);
            nn::gfx::Sampler* pSampler = const_cast<nn::gfx::Sampler*>(samplerRef.GetSampler());
            UnregisterSamplerFromDescriptorPool(pSampler);
            nn::gfx::DescriptorSlot descriptorSlot = RegisterSamplerToDescriptorPool(pSampler);
            pResMaterial->SetSamplerDescriptorSlot(samplerIndex, descriptorSlot);
        }
    }
}

//------------------------------------------------------------------------------
//  プリミティブレンダラを初期化する
//------------------------------------------------------------------------------
void InitializePrimitiveRenderer(nns::gfx::PrimitiveRenderer::Renderer** ppPrimitiveRenderer, ptrdiff_t* pMemoryOffset) NN_NOEXCEPT
{
    // ダブルバッファーで運用する場合は、SetMultiBufferQuantity で 2 を指定し、
    // プリミティブレンダラ内部のバッファーをダブルにしたうえで、
    // g_PrimitiveRenderer.Update(); で利用するバッファーを選択( 0 -> 1 -> 0 -> 1 )する
    nns::gfx::PrimitiveRenderer::RendererInfo info;
    info.SetDefault();
    info.SetMultiBufferQuantity(2);

    nn::gfx::Device* pDevice = GetGfxFramework()->GetDevice();
    size_t memoryPoolSize = nns::gfx::PrimitiveRenderer::Renderer::GetRequiredMemoryPoolSize(pDevice, info);
    *pMemoryOffset = GetGfxFramework()->AllocatePoolMemory(nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer,
                                          memoryPoolSize,
                                          nns::gfx::PrimitiveRenderer::Renderer::GetMemoryPoolAlignment(pDevice, info));

    void* ptr = AllocateMemory(sizeof(nns::gfx::PrimitiveRenderer::Renderer), 16);
    *ppPrimitiveRenderer = new (ptr)nns::gfx::PrimitiveRenderer::Renderer();
    (*ppPrimitiveRenderer)->Initialize(GetGfxFramework()->GetDevice(), info, &g_PrimitiveRendererResrouce, GetGfxFramework()->GetMemoryPool(nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer), *pMemoryOffset, memoryPoolSize);
    (*ppPrimitiveRenderer)->SetScreenWidth(GetGfxFramework()->GetDisplayWidth());
    (*ppPrimitiveRenderer)->SetScreenHeight(GetGfxFramework()->GetDisplayHeight());
}

//------------------------------------------------------------------------------
//  プリミティブレンダラを解放する
//------------------------------------------------------------------------------
void FinalizePrimitiveRenderer(nns::gfx::PrimitiveRenderer::Renderer** ppPrimitiveRenderer, ptrdiff_t* pMemoryOffset) NN_NOEXCEPT
{
    // メモリープールから確保したオフセットを解放
    GetGfxFramework()->FreePoolMemory(nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer, *pMemoryOffset);
    *pMemoryOffset = -1;

    (*ppPrimitiveRenderer)->Finalize(GetGfxFramework()->GetDevice());
    FreeMemory(*ppPrimitiveRenderer);
    *ppPrimitiveRenderer = nullptr;
}

//------------------------------------------------------------------------------
//  フレームワークを取得する
//------------------------------------------------------------------------------
nns::gfx::GraphicsFramework* GetGfxFramework() NN_NOEXCEPT
{
    return &g_GfxFramework;
}

//------------------------------------------------------------------------------
void* UserAllocatorMalloc(size_t size, size_t alignment, void* pUserData)
{
    NN_UNUSED(pUserData);
    return g_FrameworkAllocator.Allocate(size, alignment);
}

void UserAllocatorFree(void* ptr, void* pUserData)
{
    NN_UNUSED(pUserData);
    g_FrameworkAllocator.Free(ptr);
}

void InitializeGfx() NN_NOEXCEPT
{
    // フレームワークを初期化する
    if (!g_GfxFramework.IsInitialized())
    {
        // フレームワーク専用のアロケータを作成
        g_pFrameworkMemory = malloc(g_FrameworkHeapSize);
        NN_ASSERT_NOT_NULL(g_pFrameworkMemory);
        g_FrameworkAllocator.Initialize(g_pFrameworkMemory, g_FrameworkHeapSize);

        nns::gfx::GraphicsFramework::FrameworkInfo fwInfo;
        fwInfo.SetDefault();
        fwInfo.SetMemoryPoolSize(nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer, g_ConstantBufferMemoryPoolSize);
        fwInfo.SetMemoryPoolSize(nns::gfx::GraphicsFramework::MemoryPoolType_Others, g_OthersMemoryPoolSize);
        fwInfo.SetMemoryPoolSize(nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget, g_RenderTargetMemoryPoolSize);
        fwInfo.SetMemoryPoolSize(nns::gfx::GraphicsFramework::MemoryPoolType_CommandBuffer, g_CommandBufferMemoryPoolSize);
        fwInfo.SetDescriptorPoolSlotCount(nn::gfx::DescriptorPoolType_TextureView, Max_Descriptor);
        fwInfo.SetDescriptorPoolSlotCount(nn::gfx::DescriptorPoolType_Sampler, Max_Descriptor);
        fwInfo.SetColorBufferFormat(nn::gfx::ImageFormat_R11_G11_B10_Float);
        fwInfo.SetBufferCount(2);
        fwInfo.SetRootCommandBufferCommandMemorySize(16 * 1024 * 1024);
        fwInfo.SetRootCommandBufferControlMemorySize(2 * 1024 * 1024);

        g_GfxFramework.Initialize(
                    fwInfo,
                    UserAllocatorMalloc,
                    UserAllocatorFree,
                    nullptr);
    }

    // プリミティブレンダラリソースを初期化する
    nn::gfx::Device* pDevice = GetGfxFramework()->GetDevice();
    {
        size_t memoryPoolSize = nns::gfx::PrimitiveRenderer::GraphicsResource::GetRequiredMemoryPoolSize(pDevice);

        g_pPrimitiveRendererResrouceMemoryPoolOffset = g_GfxFramework.AllocatePoolMemory(
                                                nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer,
                                                memoryPoolSize,
                                                nns::gfx::PrimitiveRenderer::GraphicsResource::GetMemoryPoolAlignment(pDevice));

        size_t memorySize = nns::gfx::PrimitiveRenderer::GraphicsResource::GetRequiredMemorySize(pDevice);
        g_pPrimitiveRendererResrouceMemory = AllocateMemory(memorySize,
                                                            nns::gfx::PrimitiveRenderer::GraphicsResource::GetRequiredMemoryAlignment(pDevice));

        nn::gfx::MemoryPool* pMemoryPool = g_GfxFramework.GetMemoryPool(nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer);
        g_PrimitiveRendererResrouce.Initialize(pDevice,
                                               g_pPrimitiveRendererResrouceMemory,
                                               memorySize,
                                               pMemoryPool,
                                               g_pPrimitiveRendererResrouceMemoryPoolOffset,
                                               memoryPoolSize);
    }

    // nns::g3dを初期化
    {
        void* pMemory = AllocateMemory(g_NnsG3dMemorySize);
        NN_ASSERT_NOT_NULL(pMemory);
        void* pMemoryPoolMemory = AllocateMemory(g_NnsG3dMemoryPoolSize, nns::g3d::GetMemoryPoolAlignment(pDevice));
        NN_ASSERT_NOT_NULL(pMemoryPoolMemory);
        nns::g3d::Initialize(pDevice, pMemory, g_NnsG3dMemorySize, pMemoryPoolMemory, g_NnsG3dMemoryPoolSize);
    }

    for (int index = 0; index < Max_Descriptor; ++index)
    {
        g_DescriptorSlotIndexArray[index] = index;
    }

    // ウィンドウハンドルを取得する
    {
        nn::Result result = nn::vi::GetNativeWindow(&g_WndHandle, g_GfxFramework.GetLayer());
        NN_ASSERT(result.IsSuccess());
    }

    InitializeBlendState();
    InitializeRasterizerState();
    InitializeDepthStencilState();

#if defined(NN_BUILD_TARGET_PLATFORM_OS_WIN)
    SetForegroundWindow(static_cast<HWND>(GetWindowHandle()));
#endif
} // NOLINT

void ShutdownGfx() NN_NOEXCEPT
{
    FinalizeDepthStencilState();
    FinalizeRasterizerState();
    FinalizeBlendState();

    // nns::g3d終了処理
    {
        nns::g3d::Finalize(GetGfxFramework()->GetDevice());
    }

    // プリミティブレンダラリソースを解放する
    {
        // メモリープールから確保したオフセットを解放
        GetGfxFramework()->FreePoolMemory(nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer, g_pPrimitiveRendererResrouceMemoryPoolOffset);
        g_pPrimitiveRendererResrouceMemoryPoolOffset = -1;

        g_PrimitiveRendererResrouce.Finalize(GetGfxFramework()->GetDevice());
        FreeMemory(g_pPrimitiveRendererResrouceMemory);
    }

    // Gfxフレームワークを解放する
    if (g_GfxFramework.IsInitialized())
    {
        g_GfxFramework.Finalize();
    }

    // Gfxフレームワーク用アロケータを解放する
    {
        g_FrameworkAllocator.Finalize();
        free(g_pFrameworkMemory);
        g_pFrameworkMemory = nullptr;
    }
}

//------------------------------------------------------------------------------
//  nns::gfx::FrameBuffer を初期化する
//------------------------------------------------------------------------------
void InitializeFrameBuffer(nns::gfx::FrameBuffer* pFrameBuffer, const nns::gfx::FrameBuffer::InfoType* pInfo) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pFrameBuffer);
    NN_ASSERT_NOT_NULL(pInfo);

    nn::gfx::Device* pDevice = GetGfxFramework()->GetDevice();

    size_t requiredMemoryPoolSize = nns::gfx::FrameBuffer::GetRequiredMemoryPoolSize(pDevice, *pInfo);
    size_t memoryPoolAlignment = nns::gfx::FrameBuffer::GetMemoryPoolAlignment(pDevice, *pInfo);

    ptrdiff_t offset = GetGfxFramework()->AllocatePoolMemory(nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget, requiredMemoryPoolSize, memoryPoolAlignment);
    nn::gfx::MemoryPool* pMemoryPool = GetGfxFramework()->GetMemoryPool(nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget);

    pFrameBuffer->Initialize(pDevice, pInfo, pMemoryPool, offset, UserAllocatorMalloc, UserAllocatorFree, nullptr);
}

//------------------------------------------------------------------------------
//  nns::gfx::FrameBuffer の終了処理を実行する
//------------------------------------------------------------------------------
void FinalizeFrameBuffer(nns::gfx::FrameBuffer* pFrameBuffer) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pFrameBuffer);

    nn::gfx::Device* pDevice = GetGfxFramework()->GetDevice();

    ptrdiff_t offset = pFrameBuffer->GetMemoryPoolOffset();

    pFrameBuffer->Finalize(pDevice, UserAllocatorMalloc, UserAllocatorFree, nullptr);

    GetGfxFramework()->FreePoolMemory(nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget, offset);
}

void ClearFrameBuffer(nn::gfx::CommandBuffer * pCommandBuffer, nns::gfx::FrameBuffer * pFrameBuffer) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pCommandBuffer);
    NN_ASSERT_NOT_NULL(pFrameBuffer);

    nns::gfx::ColorBuffer* pColorBuffer = pFrameBuffer->GetColorBuffer();

    if (pColorBuffer != nullptr)
    {
        nn::gfx::ColorTargetView* pColorTargetView = pFrameBuffer->GetColorBuffer()->GetColorTargetView();
        if (pColorTargetView != nullptr)
        {
            pCommandBuffer->ClearColor(pColorTargetView,
                pFrameBuffer->GetClearColor(nns::gfx::FrameBuffer::ClearColorComponent_Red),
                pFrameBuffer->GetClearColor(nns::gfx::FrameBuffer::ClearColorComponent_Green),
                pFrameBuffer->GetClearColor(nns::gfx::FrameBuffer::ClearColorComponent_Blue),
                pFrameBuffer->GetClearColor(nns::gfx::FrameBuffer::ClearColorComponent_Alpha), nullptr);
        }
    }

    nns::gfx::DepthStencilBuffer *pDepthStencilBuffer = pFrameBuffer->GetDepthBuffer();

    if (pDepthStencilBuffer != nullptr)
    {
        nn::gfx::DepthStencilView*  pDepthStencilView = pDepthStencilBuffer->GetDepthStencilView();
        if (pDepthStencilView)
        {
            pCommandBuffer->ClearDepthStencil(pDepthStencilView,
                1.0f,
                0,
                nn::gfx::DepthStencilClearMode_DepthStencil,
                nullptr);
        }
    }
}

void SetFrameBufferToRenderTarget(nn::gfx::CommandBuffer* pCommandBuffer, nns::gfx::FrameBuffer* pFrameBuffer) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pCommandBuffer);
    NN_ASSERT_NOT_NULL(pFrameBuffer);

    nn::gfx::ColorTargetView* pColorTargetView = pFrameBuffer->GetColorBuffer()->GetColorTargetView();
    nn::gfx::DepthStencilView* pDepthStencilView = nullptr;

    nns::gfx::DepthStencilBuffer*   pDepthStencilBuffer = pFrameBuffer->GetDepthBuffer();
    if (pDepthStencilBuffer != nullptr)
    {
        pDepthStencilView = pDepthStencilBuffer->GetDepthStencilView();
    }

    pCommandBuffer->SetRenderTargets(
        1,
        &pColorTargetView,
        pDepthStencilView);
}

nn::vi::NativeWindowHandle GetWindowHandle() NN_NOEXCEPT
{
    return g_WndHandle;
}

int MakeGfxResShaderFileList(const nn::gfx::ResShaderFile*** pGfxShaderFileList, const Vector<nn::g3d::ResShaderFile*>& shaderFileList) NN_NOEXCEPT
{
    int shadingModelCount = 0;
    for (int fileIndex = 0; fileIndex < shaderFileList.GetCount(); ++fileIndex)
    {
        shadingModelCount += shaderFileList[fileIndex]->GetResShaderArchive()->GetShadingModelCount();
    }

    const nn::gfx::ResShaderFile** pShaderFileList = AllocateMemory<const nn::gfx::ResShaderFile*>(sizeof(nn::gfx::ResShaderFile*) * shadingModelCount);
    NN_ASSERT_NOT_NULL(pShaderFileList);

    int shaderFileIndex = 0;
    for (int fileIndex = 0; fileIndex < shaderFileList.GetCount(); ++fileIndex)
    {
        nn::g3d::ResShaderArchive* pResShaderArchive = shaderFileList[fileIndex]->GetResShaderArchive();
        for (int shadingModelIndex = 0; shadingModelIndex < pResShaderArchive->GetShadingModelCount(); ++shadingModelIndex)
        {
            pShaderFileList[shaderFileIndex] = pResShaderArchive->GetShadingModel(shadingModelIndex)->GetGfxResShaderFile();
            ++shaderFileIndex;
        }
    }

    *pGfxShaderFileList = pShaderFileList;

    return shadingModelCount;
}

int MakeGfxResShaderFileList(const nn::gfx::ResShaderFile*** pGfxShaderFileList, const Vector<nn::g3d::ResShaderArchive*>& shaderArchiveList)
{
    int shadingModelCount = 0;
    for (int fileIndex = 0; fileIndex < shaderArchiveList.GetCount(); ++fileIndex)
    {
        shadingModelCount += shaderArchiveList[fileIndex]->GetShadingModelCount();
    }

    const nn::gfx::ResShaderFile** pShaderFileList = AllocateMemory<const nn::gfx::ResShaderFile*>(sizeof(nn::gfx::ResShaderFile*) * shadingModelCount);
    NN_ASSERT_NOT_NULL(pShaderFileList);

    int shaderFileIndex = 0;
    for (int fileIndex = 0; fileIndex < shaderArchiveList.GetCount(); ++fileIndex)
    {
        nn::g3d::ResShaderArchive* pResShaderArchive = shaderArchiveList[fileIndex];
        for (int shadingModelIndex = 0; shadingModelIndex < pResShaderArchive->GetShadingModelCount(); ++shadingModelIndex)
        {
            pShaderFileList[shaderFileIndex] = pResShaderArchive->GetShadingModel(shadingModelIndex)->GetGfxResShaderFile();
            ++shaderFileIndex;
        }
    }

    *pGfxShaderFileList = pShaderFileList;

    return shadingModelCount;
}

void SetShaderScratchMemory(const Vector<nn::g3d::ResShaderFile*>& shaderFileList) NN_NOEXCEPT
{
    const nn::gfx::ResShaderFile** resShaderFileList;
    int shaderFileCount = MakeGfxResShaderFileList(&resShaderFileList, shaderFileList);
    size_t shaderScratchMemorySize = GetShaderMemorySize(g_GfxFramework.GetDevice(), resShaderFileList, shaderFileCount);
    g_GfxFramework.SetShaderScratchMemorySize(shaderScratchMemorySize);
    FreeMemory(resShaderFileList);
}

void SetShaderScratchMemory(const Vector<nn::g3d::ResShaderArchive*>& shaderArchiveList) NN_NOEXCEPT
{
    const nn::gfx::ResShaderFile** resShaderFileList;
    int shaderFileCount = MakeGfxResShaderFileList(&resShaderFileList, shaderArchiveList);
    size_t shaderScratchMemorySize = GetShaderMemorySize(g_GfxFramework.GetDevice(), resShaderFileList, shaderFileCount);
    g_GfxFramework.SetShaderScratchMemorySize(shaderScratchMemorySize);
    FreeMemory(resShaderFileList);
}

nn::gfx::DescriptorSlot GetDummySamplerDescriptorSlot() NN_NOEXCEPT
{
    return g_DummySamplerDescriptorSlot;
}

nn::gfx::ResTexture* GetDummyTexture() NN_NOEXCEPT
{
    return g_pDummyTexture;
}

void CreateDummyTextureAndSampler(nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    const char* SharedTexturePath = "Resource:/Shared.bntx";
    const char* DummyTextureName = "Dummy";

    // ダミーテクスチャーファイルの読み込み
    {
        size_t size = 0;
        size_t alignment = GetFileAlignment(SharedTexturePath);
        void* pFile = LoadFile(SharedTexturePath, &size, alignment);

        // 読み込んだファイルをResTextureFileに変換します
        g_pSharedTextureFile = nn::gfx::ResTextureFile::ResCast(pFile);
        g_pSharedTextureFile->Initialize(pDevice);
        for (int textureIndex = 0, texCount = g_pSharedTextureFile->GetTextureDic()->GetCount(); textureIndex < texCount; ++textureIndex)
        {
            nn::gfx::ResTexture* pResTexture = g_pSharedTextureFile->GetResTexture(textureIndex);
            pResTexture->Initialize(pDevice);
        }

        RegisterTextureFileToDescriptorPool(g_pSharedTextureFile);

        const nn::util::ResDic* pSharedTextureDic = g_pSharedTextureFile->GetTextureDic();
        int dummyTextureIndex = pSharedTextureDic->FindIndex(DummyTextureName);
        NN_ASSERT(dummyTextureIndex != nn::util::ResDic::Npos);

        g_pDummyTexture = g_pSharedTextureFile->GetResTexture(dummyTextureIndex);
    }

    // ダミーサンプラーの作成
    {
        nn::gfx::SamplerInfo info;
        info.SetDefault();

        g_DummySampler.Initialize(pDevice, info);
        g_DummySamplerDescriptorSlot = RegisterSamplerToDescriptorPool(&g_DummySampler);
    }
}

void DeleteDummyTextureAndSampler(nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    // ダミーテクスチャーの破棄
    for (int textureIndex = 0; textureIndex < g_pSharedTextureFile->GetTextureDic()->GetCount(); ++textureIndex)
    {
        nn::gfx::ResTexture* pResTexture = g_pSharedTextureFile->GetResTexture(textureIndex);
        pResTexture->Finalize(pDevice);
    }
    g_pSharedTextureFile->Finalize(pDevice);
    FreeMemory(g_pSharedTextureFile);

    UnregisterTextureFileFromDescriptorPool(g_pSharedTextureFile);

    // ダミーサンプラーの破棄
    g_DummySampler.Finalize(pDevice);
    UnregisterSamplerFromDescriptorPool(&g_DummySampler);

    g_pDummyTexture = nullptr;
}


//------------------------------------------------------------------------------
// テキストスクリーンを初期化する
//------------------------------------------------------------------------------
void ScreenInfo::Initialize(nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pDevice);

    // プリミティブレンダラの初期化
    {
        InitializePrimitiveRenderer(&m_pPrimitiveRenderer, &m_MemoryOffset);
    }

    // ラスタライザーステートを初期化
    {
        nn::gfx::RasterizerState::InfoType info;
        info.SetDefault();
        info.SetCullMode(nn::gfx::CullMode_None);
        info.SetPrimitiveTopologyType(nn::gfx::PrimitiveTopologyType_Triangle);
        info.SetScissorEnabled(true);
        info.SetDepthClipEnabled(true);
        m_RasterizerState.Initialize(pDevice, info);
    }

    // ブレンドステートを初期化
    {
        nn::gfx::BlendState::InfoType info;
        info.SetDefault();

        nn::gfx::BlendTargetStateInfo targetInfo;
        {
            targetInfo.SetDefault();
            targetInfo.SetChannelMask(nn::gfx::ChannelMask_Red | nn::gfx::ChannelMask_Green | nn::gfx::ChannelMask_Blue);
            targetInfo.SetBlendEnabled(true);
            targetInfo.SetColorBlendFunction(nn::gfx::BlendFunction_Add);
            targetInfo.SetDestinationColorBlendFactor(nn::gfx::BlendFactor_OneMinusSourceAlpha);
            targetInfo.SetSourceColorBlendFactor(nn::gfx::BlendFactor_SourceAlpha);
        };

        info.SetBlendTargetStateInfoArray(&targetInfo, 1);
        size_t memorySize = nn::gfx::BlendState::GetRequiredMemorySize(info);
        m_BlendState.SetMemory(AllocateMemory(memorySize, nn::gfx::BlendState::RequiredMemoryInfo_Alignment), memorySize);
        m_BlendState.Initialize(GetGfxFramework()->GetDevice(), info);
    }

    // 正射影行列初期化
    {
        const float displayWidth = static_cast<float>(GetGfxFramework()->GetDisplayWidth());
        const float displayHeight = static_cast<float>(GetGfxFramework()->GetDisplayHeight());

        nn::util::MatrixOrthographicOffCenterRightHanded(&m_ProjectionMtx,
                                                         0 /*left*/,
                                                         displayWidth /*right*/,
                                                         displayHeight /*bottom*/,
                                                         0 /*top*/,
                                                         0.0f /*near*/,
                                                         300.0f /*far*/ );
    }

    // ビュー行列
    {
        nn::util::Vector3fType camPos = { 0.0f, 0.0f, 50.0f };
        nn::util::Vector3fType comUp = { 0.0f, 1.0f, 0.0f };
        nn::util::Vector3fType target = { 0.0f, 0.0f, 0.0f };
        nn::util::MatrixLookAtRightHanded(
                            &m_ViewMtx,
                            camPos,
                            target,
                            comUp);
    }

    // デバッグフォントを初期化処理
    {
        void* ptr = AllocateMemory(sizeof(nns::gfx::GraphicsFramework::DebugFontTextWriter), 16);
        m_pDebugWriter = new (ptr)nns::gfx::GraphicsFramework::DebugFontTextWriter();
        g_GfxFramework.InitializeDebugFontTextWriter(m_pDebugWriter, 1024);
    }
}

//------------------------------------------------------------------------------
// テキストスクリーンを破棄する
//------------------------------------------------------------------------------
void ScreenInfo::Cleanup(nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pDevice);

    // ブレンドステートを破棄
    {
        void* pData = m_BlendState.GetMemory();
        m_BlendState.Finalize(pDevice);
        if (pData)
        {
            FreeMemory(pData);
        }
    }

    // ラスタライザーステートを破棄
    {
        m_RasterizerState.Finalize(pDevice);
    }

    // プリミティブレンダラを破棄
    {
        FinalizePrimitiveRenderer(&m_pPrimitiveRenderer, &m_MemoryOffset);
    }

    // デバッグフォントを破棄
    {
        g_GfxFramework.FinalizeDebugFontTextWriter(m_pDebugWriter);
        FreeMemory(m_pDebugWriter);
    }
}

//------------------------------------------------------------------------------
// テキスト描画コマンドを実行する
//------------------------------------------------------------------------------
void ScreenInfo::Draw(nn::gfx::CommandBuffer* pCommandBuffer) NN_NOEXCEPT
{
    // プリミティブレンダラの更新処理
    m_pPrimitiveRenderer->Update(m_BufferIndex);
    m_BufferIndex = 1 - m_BufferIndex;

    // 描画ステートを設定
    pCommandBuffer->SetBlendState(&m_BlendState);
    pCommandBuffer->SetRasterizerState(&m_RasterizerState);

    // スクリーンを描画する
    {
        // サイズとカラーを決定する
        nn::util::Vector3fType center = { m_PosX, m_PosY, 0.0f };
        nn::util::Vector3fType size   = { m_Width, m_Height, 1.0f };
        nn::util::Uint8x4 color = {{ m_Red, m_Green, m_Blue, m_Alpha }};

        nn::util::Vector3fType translate;
        nn::util::VectorSet(&translate, 0.0f, 0.0f, 0.0f);
        nn::util::Matrix4x3fType modelMatrix;
        nn::util::MatrixIdentity(&modelMatrix);
        nn::util::MatrixSetAxisW(&modelMatrix, translate);
        m_pPrimitiveRenderer->SetDefaultParameters();
        m_pPrimitiveRenderer->SetModelMatrix(&modelMatrix);
        m_pPrimitiveRenderer->SetViewMatrix(&m_ViewMtx);
        m_pPrimitiveRenderer->SetProjectionMatrix(&m_ProjectionMtx);
        m_pPrimitiveRenderer->SetColor(color);
        m_pPrimitiveRenderer->DrawQuad(pCommandBuffer, center, size);
    }

    // Quad を描画する
    {
        float marginPosX = m_PosX + m_MarginWidth;
        float marginPosY = m_PosY + m_MarginHeight;
        for(int i = 0; i < m_QuadCount; ++i)
        {
            m_pPrimitiveRenderer->SetDefaultParameters();
            m_pPrimitiveRenderer->SetColor(m_QuadArray[i].color);
            if (!m_QuadArray[i].textureDescriptorSlot.IsValid() || !m_QuadArray[i].samplerDescriptorSlot.IsValid())
            {
                m_pPrimitiveRenderer->Draw2DRect(pCommandBuffer, marginPosX + m_QuadArray[i].x, marginPosY + m_QuadArray[i].y, m_QuadArray[i].width, m_QuadArray[i].height);
                continue;
            }

            nn::util::Vector3fType center = { m_QuadArray[i].x, m_QuadArray[i].y, 0.0f };
            nn::util::Vector3fType size = { m_QuadArray[i].width, m_QuadArray[i].height, 1.0f };

            m_pPrimitiveRenderer->SetViewMatrix(&m_ViewMtx);
            m_pPrimitiveRenderer->SetProjectionMatrix(&m_ProjectionMtx);
            m_pPrimitiveRenderer->SetBlendState(pCommandBuffer, nns::gfx::PrimitiveRenderer::BlendType_Opacity);

            if (m_QuadArray[i].layer != InvalidLayerIndex)
            {
                m_pPrimitiveRenderer->DrawQuadTextureArray(pCommandBuffer, center, size, m_QuadArray[i].textureDescriptorSlot, m_QuadArray[i].samplerDescriptorSlot, static_cast<float>(m_QuadArray[i].layer));
            }
            else
            {
                m_pPrimitiveRenderer->DrawQuad(pCommandBuffer, center, size, m_QuadArray[i].textureDescriptorSlot, m_QuadArray[i].samplerDescriptorSlot);
            }

            m_QuadArray[i].textureDescriptorSlot.Invalidate();
            m_QuadArray[i].samplerDescriptorSlot.Invalidate();
        }
        m_QuadCount = 0;
    }

    // 文字列を描画する
    {
        m_pDebugWriter->object.Draw(pCommandBuffer);
    }
}

//------------------------------------------------------------------------------
// テキスト作成の開始する
//------------------------------------------------------------------------------
void ScreenInfo::StartPrint() NN_NOEXCEPT
{
    m_pDebugWriter->object.SetCursor(m_PosX + m_MarginWidth, m_PosY + m_MarginHeight);
    m_AdjustWidth = 0.0f;
    m_AdjustHeight = 0.0f;
}

//------------------------------------------------------------------------------
// テキストを追加作成する
//------------------------------------------------------------------------------
void ScreenInfo::Print(const char* format, ...) NN_NOEXCEPT
{
    char stringBuffer[Max_ScreenTextLength];
    va_list arglist;
    va_start(arglist, format);
    vsnprintf(stringBuffer, Max_ScreenTextLength, format, arglist);
    m_pDebugWriter->object.Print(stringBuffer);
    va_end(arglist);

    // 追加されたテキストに合わせてウインドウサイズを更新する
    // 高さ
    int returnCodeCount = FindCharCount(stringBuffer, Max_ScreenTextLength, '\n');
    float stringHeight = m_pDebugWriter->object.GetFixedWidth();
    m_AdjustHeight += stringHeight * returnCodeCount;

    // 幅
    float stringWidth = m_pDebugWriter->object.CalculateStringWidth(stringBuffer);
    if (stringWidth > m_AdjustWidth)
    {
        m_AdjustWidth = stringWidth;
    }
}

//------------------------------------------------------------------------------
// 描画座標を指定してテキストを追加作成する
//------------------------------------------------------------------------------
void ScreenInfo::Print(float x, float y, const char* format, ...) NN_NOEXCEPT
{
    float saveX = m_pDebugWriter->object.GetCursorX();
    float saveY = m_pDebugWriter->object.GetCursorY();
    m_pDebugWriter->object.SetCursor(m_PosX + m_MarginWidth + x, m_PosY + m_MarginHeight + y);

    char stringBuffer[Max_ScreenTextLength];
    va_list arglist;
    va_start(arglist, format);
    vsnprintf(stringBuffer, Max_ScreenTextLength, format, arglist);
    m_pDebugWriter->object.Print(stringBuffer);
    va_end(arglist);

    // 追加されたテキストに合わせてウインドウサイズを更新する
    // 高さ
    if( y > m_AdjustHeight)
    {
        m_AdjustHeight = y;
    }

    // 幅
    float stringWidth = m_pDebugWriter->object.CalculateStringWidth(stringBuffer) + x;
    if (stringWidth > m_AdjustWidth)
    {
        m_AdjustWidth = stringWidth;
    }
    m_pDebugWriter->object.SetCursor(saveX, saveY);
}

//------------------------------------------------------------------------------
// Quad を追加作成する
//------------------------------------------------------------------------------
void ScreenInfo::AddQuad(float x, float y, float width, float height, const nn::util::Color4u8& color) NN_NOEXCEPT
{
    AddQuadTexture(x, y, width, height, color, m_QuadArray[m_QuadCount].textureDescriptorSlot, m_QuadArray[m_QuadCount].samplerDescriptorSlot);
}

void ScreenInfo::AddQuadTexture(float x, float y, float width, float height, const nn::util::Color4u8& color, const nn::gfx::DescriptorSlot& textureDescriptorSlot, const nn::gfx::DescriptorSlot& samplerDescriptorSlot) NN_NOEXCEPT
{
    AddQuadTexture(x, y, width, height, InvalidLayerIndex, color, textureDescriptorSlot, samplerDescriptorSlot);
}

void ScreenInfo::AddQuadTexture(float x, float y, float width, float height, int layer, const nn::util::Color4u8& color, const nn::gfx::DescriptorSlot& textureDescriptorSlot, const nn::gfx::DescriptorSlot& samplerDescriptorSlot) NN_NOEXCEPT
{
    NN_ASSERT(m_QuadCount < 32, "Over MaxRect");
    m_QuadArray[m_QuadCount].x = x;
    m_QuadArray[m_QuadCount].y = y;
    m_QuadArray[m_QuadCount].width = width;
    m_QuadArray[m_QuadCount].height = height;
    m_QuadArray[m_QuadCount].layer = layer;
    m_QuadArray[m_QuadCount].color = color;
    m_QuadArray[m_QuadCount].textureDescriptorSlot = textureDescriptorSlot;
    m_QuadArray[m_QuadCount].samplerDescriptorSlot = samplerDescriptorSlot;
    m_QuadCount++;
}

//------------------------------------------------------------------------------
// テキストカラーを設定する
//------------------------------------------------------------------------------
void ScreenInfo::SetTextColor(uint8_t red, uint8_t green , uint8_t blue, uint8_t alpha) NN_NOEXCEPT
{
    nn::util::Color4u8Type color;
    color.v[0] = red;
    color.v[1] = green;
    color.v[2] = blue;
    color.v[3] = alpha;
    m_pDebugWriter->object.SetTextColor(color);
}

//------------------------------------------------------------------------------
// フォントサイズを設定する
//------------------------------------------------------------------------------
void ScreenInfo::SetFontSize(float width) NN_NOEXCEPT
{
    m_pDebugWriter->object.SetFixedWidth(width);
}

//------------------------------------------------------------------------------
// フォントサイズを取得する
//------------------------------------------------------------------------------
float ScreenInfo::GetFontHeight() const NN_NOEXCEPT
{
    return m_pDebugWriter->object.GetFixedWidth();
}

//------------------------------------------------------------------------------
// 文字列の幅を取得する
//------------------------------------------------------------------------------
float ScreenInfo::GetTextWidth(const char* text) const NN_NOEXCEPT
{
    return m_pDebugWriter->object.CalculateStringWidth(text);
}

//--------------------------------------------------------------------------------------------------

//------------------------------------------------------------------------------
// 計測メーターを初期化する
//------------------------------------------------------------------------------
void LoadMeter::Initialize(nn::gfx::Device* pDevice,  nn::perf::LoadMeterCenterInfo info) NN_NOEXCEPT
{
    // プリミティブレンダラの初期化
    InitializePrimitiveRenderer(&m_pPrimitiveRenderer, &m_MemoryOffset);

    // デバッグフォントの初期化
    void* ptr = AllocateMemory(sizeof(nns::gfx::GraphicsFramework::DebugFontTextWriter), 16);
    m_pDebugWriter = new (ptr)nns::gfx::GraphicsFramework::DebugFontTextWriter();
    g_GfxFramework.InitializeDebugFontTextWriter(m_pDebugWriter, 1024);

    // LoadMeterCenter の初期化
#if defined( NN_PERF_PROFILE_ENABLED )
    size_t memorySize = NN_PERF_GET_BUFFER_SIZE(info);
    size_t memoryAlignment = NN_PERF_GET_BUFFER_ALIGNMENT();
    m_pLoadMeterMemory = AllocateMemory(memorySize, memoryAlignment);

    size_t size = NN_PERF_GET_MEMORY_POOL_SIZE( pDevice, info);
    size_t alignment = NN_PERF_GET_MEMORY_POOL_ALIGNMENT( pDevice, info);
    m_MeterMemoryPoolOffset = GetGfxFramework()->AllocatePoolMemory(nns::gfx::GraphicsFramework::MemoryPoolType_Others, size, alignment);

    NN_PERF_INITIALIZE_METER(pDevice, info, m_pLoadMeterMemory, memorySize, GetGfxFramework()->GetMemoryPool(nns::gfx::GraphicsFramework::MemoryPoolType_Others), m_MeterMemoryPoolOffset, size);
#endif

    void* pData = AllocateMemory(sizeof(nns::gfx::PrimitiveRenderer::MeterDrawer));
    NN_ASSERT_NOT_NULL(pData);
    m_pMeterDrawer = new(pData) nns::gfx::PrimitiveRenderer::MeterDrawer;

    nn::util::Float2 pos = NN_UTIL_FLOAT_2_INITIALIZER(GetGfxFramework()->GetDisplayWidth() * 0.05f, 680.f);
    m_pMeterDrawer->SetPosition(pos);
    m_pMeterDrawer->SetWidth(GetGfxFramework()->GetDisplayWidth() * 0.9f);
    m_pMeterDrawer->SetDebugFontTextWriter(&m_pDebugWriter->object);

    // NN_PERF_ENABLED が未定義の場合の警告抑制
    NN_UNUSED(info);
    NN_UNUSED(pDevice);
}

//------------------------------------------------------------------------------
// 計測メーターを破棄する
//------------------------------------------------------------------------------
void LoadMeter::Cleanup(nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    // デバッグフォントを解放する
    g_GfxFramework.FinalizeDebugFontTextWriter(m_pDebugWriter);
    FreeMemory(m_pDebugWriter);

    void* pData = m_pMeterDrawer;
    FreeMemory(pData);
    m_pMeterDrawer = nullptr;
#if defined( NN_PERF_PROFILE_ENABLED )
    NN_PERF_FINALIZE_METER(pDevice);
    GetGfxFramework()->FreePoolMemory(nns::gfx::GraphicsFramework::MemoryPoolType_Others, m_MeterMemoryPoolOffset);
    FreeMemory(m_pLoadMeterMemory);
#endif

    // プリミティブレンダラを解放する
    FinalizePrimitiveRenderer(&m_pPrimitiveRenderer, &m_MemoryOffset);

    // NN_PERF_ENABLED が未定義の場合の警告抑制
    NN_UNUSED(pDevice);
}

//------------------------------------------------------------------------------
// 計測メーターを描画する
//------------------------------------------------------------------------------
void LoadMeter::Draw(nn::gfx::CommandBuffer* pCommandBuffer) NN_NOEXCEPT
{
    // プリミティブレンダラの更新処理
    static int bufferIndex = 0;
    m_pPrimitiveRenderer->Update(bufferIndex);
    bufferIndex = 1 - bufferIndex;

    if(NN_STATIC_CONDITION( NN_PERF_IS_ENABLED() ))
    {
        // 描画位置を自動調整
        nn::util::Float2 pos = NN_UTIL_FLOAT_2_INITIALIZER( GetGfxFramework()->GetDisplayWidth() * 0.05f,
            GetGfxFramework()->GetDisplayHeight() - m_pMeterDrawer->GetHeight(NN_PERF_GET_FRAME_METER()) - m_pMeterDrawer->GetBarHeight() );
        m_pMeterDrawer->SetPosition(pos);
        m_pMeterDrawer->Draw(pCommandBuffer, m_pPrimitiveRenderer, NN_PERF_GET_FRAME_METER());
        m_pDebugWriter->object.Draw(pCommandBuffer);
    }
}

//--------------------------------------------------------------------------------------------------

void LoadUniformBlock(
    nn::gfx::CommandBuffer* pCommandBuffer,
    const nn::gfx::Buffer* pBuffer,
    const nn::g3d::ResShaderProgram* pShaderProgram,
    int blockIndex,
    size_t size
) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pBuffer);
    NN_ASSERT_NOT_NULL(pShaderProgram);

    nn::gfx::GpuAddress gpuAddress;
    pBuffer->GetGpuAddress(&gpuAddress);
    int locationVS = pShaderProgram->GetUniformBlockLocation(blockIndex, nn::g3d::Stage_Vertex);
    if (locationVS != nn::g3d::ShaderLocationNone)
    {
        pCommandBuffer->SetConstantBuffer(locationVS, nn::gfx::ShaderStage_Vertex, gpuAddress, size);
    }
    int locationGS = pShaderProgram->GetUniformBlockLocation(blockIndex, nn::g3d::Stage_Geometry);
    if (locationGS != nn::g3d::ShaderLocationNone)
    {
        pCommandBuffer->SetConstantBuffer(locationGS, nn::gfx::ShaderStage_Geometry, gpuAddress, size);
    }
    int locationFS = pShaderProgram->GetUniformBlockLocation(blockIndex, nn::g3d::Stage_Pixel);
    if (locationFS != nn::g3d::ShaderLocationNone)
    {
        pCommandBuffer->SetConstantBuffer(locationFS, nn::gfx::ShaderStage_Pixel, gpuAddress, size);
    }
    int locationCS = pShaderProgram->GetUniformBlockLocation(blockIndex, nn::g3d::Stage_Compute);
    if (locationCS != nn::g3d::ShaderLocationNone)
    {
        pCommandBuffer->SetConstantBuffer(locationCS, nn::gfx::ShaderStage_Compute, gpuAddress, size);
    }
}

void LoadShaderStorageBlock(
    nn::gfx::CommandBuffer* pCommandBuffer,
    const nn::gfx::Buffer* pBuffer,
    const nn::g3d::ResShaderProgram* pShaderProgram,
    int blockIndex,
    size_t size
) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pBuffer);
    NN_ASSERT_NOT_NULL(pShaderProgram);

    nn::gfx::GpuAddress gpuAddress;
    pBuffer->GetGpuAddress(&gpuAddress);
    int locationVS = pShaderProgram->GetShaderStorageBlockLocation(blockIndex, nn::g3d::Stage_Vertex);
    if (locationVS != nn::g3d::ShaderLocationNone)
    {
        pCommandBuffer->SetUnorderedAccessBuffer(locationVS, nn::gfx::ShaderStage_Vertex, gpuAddress, size);
    }
    int locationGS = pShaderProgram->GetShaderStorageBlockLocation(blockIndex, nn::g3d::Stage_Geometry);
    if (locationGS != nn::g3d::ShaderLocationNone)
    {
        pCommandBuffer->SetUnorderedAccessBuffer(locationGS, nn::gfx::ShaderStage_Geometry, gpuAddress, size);
    }
    int locationFS = pShaderProgram->GetShaderStorageBlockLocation(blockIndex, nn::g3d::Stage_Pixel);
    if (locationFS != nn::g3d::ShaderLocationNone)
    {
        pCommandBuffer->SetUnorderedAccessBuffer(locationFS, nn::gfx::ShaderStage_Pixel, gpuAddress, size);
    }
    int locationCS = pShaderProgram->GetShaderStorageBlockLocation(blockIndex, nn::g3d::Stage_Compute);
    if (locationCS != nn::g3d::ShaderLocationNone)
    {
        pCommandBuffer->SetUnorderedAccessBuffer(locationCS, nn::gfx::ShaderStage_Compute, gpuAddress, size);
    }
}

void LoadTextureSampler(
    nn::gfx::CommandBuffer* pCommandBuffer,
    const nn::gfx::DescriptorSlot& textureDescriptor,
    const nn::gfx::DescriptorSlot& samplerDescriptor,
    const nn::g3d::ResShaderProgram* pShaderProgram,
    int samplerIndex
) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pShaderProgram);

    int locationVS = pShaderProgram->GetSamplerLocation(samplerIndex, nn::g3d::Stage_Vertex);
    if (locationVS != nn::g3d::ShaderLocationNone)
    {
        pCommandBuffer->SetTextureAndSampler(locationVS, nn::gfx::ShaderStage_Vertex, textureDescriptor, samplerDescriptor);
    }
    int locationGS = pShaderProgram->GetSamplerLocation(samplerIndex, nn::g3d::Stage_Geometry);
    if (locationGS != nn::g3d::ShaderLocationNone)
    {
        pCommandBuffer->SetTextureAndSampler(locationGS, nn::gfx::ShaderStage_Geometry, textureDescriptor, samplerDescriptor);
    }
    int locationFS = pShaderProgram->GetSamplerLocation(samplerIndex, nn::g3d::Stage_Pixel);
    if (locationFS != nn::g3d::ShaderLocationNone)
    {
        pCommandBuffer->SetTextureAndSampler(locationFS, nn::gfx::ShaderStage_Pixel, textureDescriptor, samplerDescriptor);
    }
    int locationCS = pShaderProgram->GetSamplerLocation(samplerIndex, nn::g3d::Stage_Compute);
    if (locationCS != nn::g3d::ShaderLocationNone)
    {
        pCommandBuffer->SetTextureAndSampler(locationCS, nn::gfx::ShaderStage_Compute, textureDescriptor, samplerDescriptor);
    }
}

}}} // namespace nn::g3d::demo
