﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "SampleSources/g3ddemo_ViewerUtility.h"
#include "SampleSources/Town.h"
#include <nn/g3d/g3d_Viewer.h>

namespace g3ddemo = nn::g3d::demo;

extern nn::g3d::ResShaderArchive*   g_pTownShaderArchive;
extern nn::g3d::ResShaderArchive*   g_pTownDemoShaderArchive;
extern nn::g3d::ResShadingModel*    g_pResShadingModelPtr[ShadingModelType_Count];

extern nns::g3d::RenderView         g_RenderView;
extern nn::util::Vector3fType       g_CameraPosition;
extern nn::util::Vector3fType       g_CameraTarget;

extern int          g_SelectedModelIndex;
extern int          g_SelectedShapeIndex;
extern int          g_SelectedSubmeshIndex;
extern MenuFlagSet  g_MenuFlag;

namespace
{
    nn::g3d::ResShaderArchive** g_pViewerResShaderArchivePtrArray[] = {
        &g_pTownShaderArchive,
        &g_pTownDemoShaderArchive
    };

    // ビューアーへの登録。
    // ビューアーライブラリーのために専用のクラスなどを作成する必要はありません。
    void QueueAttachModelCommand(nn::g3d::ModelObj* pModelObj)
    {
        nn::g3d::viewer::ViewerResult result = nn::g3d::viewer::ViewerServer::GetInstance().QueueAttachModelCommand(pModelObj);
        NN_ASSERT(result == nn::g3d::viewer::ViewerResult_Success);
    }

    // pTargetModelObj と一致する ModelObj を保持する ModelAnimObj をリソースホルダーから探し出し、返します。
    nns::g3d::ModelAnimObj* FindModelAnimObjFromResourceHolder(g3ddemo::ResourceHolder* pResourceHolder, const nn::g3d::ModelObj* pTargetModelObj)
    {
        for (nns::g3d::ModelAnimObj* pModelAnimObj : pResourceHolder->modelAnimObjs)
        {
            if (pModelAnimObj->GetModelObj() != pTargetModelObj)
            {
                continue;
            }

            return pModelAnimObj;
        }
        return nullptr;
    }

    // ResShaderAssign と一致するシェーディングモデルを保持するシェーダーアーカイブを探索し、取得します。
    nn::g3d::ResShaderArchive* FindShaderArchiveFromViewerShader(nn::g3d::ResShaderAssign* pResShaderAssign)
    {
        nn::g3d::ResShadingModel* pResShadingModel = nullptr;
        for (int shaderIndex = 0; shaderIndex < NN_ARRAY_SIZE(g_pViewerResShaderArchivePtrArray); ++shaderIndex)
        {
            pResShadingModel = (*(g_pViewerResShaderArchivePtrArray[shaderIndex]))->FindShadingModel(pResShaderAssign->GetShadingModelName());
            if (!pResShadingModel)
            {
                continue;
            }

            return *g_pViewerResShaderArchivePtrArray[shaderIndex];
        }

        return nullptr;
    }

    // シャドウサンプラーを構築します。
    void RebuildShadowSampler(nn::gfx::Device* pDevice, nn::gfx::Sampler* pSampler) NN_NOEXCEPT
    {
        pSampler->Finalize(pDevice);

        nn::gfx::Sampler::InfoType samplerInfo;
        samplerInfo.SetDefault();
        samplerInfo.SetAddressU(nn::gfx::TextureAddressMode_ClampToBorder);
        samplerInfo.SetAddressV(nn::gfx::TextureAddressMode_ClampToBorder);
        samplerInfo.SetAddressW(nn::gfx::TextureAddressMode_ClampToEdge);
        samplerInfo.SetBorderColorType(nn::gfx::TextureBorderColorType_White);
        samplerInfo.SetFilterMode(nn::gfx::FilterMode_Comparison_MinLinear_MagLinear_MipLinear);
        samplerInfo.SetComparisonFunction(nn::gfx::ComparisonFunction_LessEqual);

        pSampler->Initialize(pDevice, samplerInfo);
    }

    // サンプラを割り当てます。
    void BindSampler(g3ddemo::ResourceHolder* pHolder, const nn::gfx::DescriptorSlot& samplerSlot) NN_NOEXCEPT
    {
        for (nns::g3d::RenderModelObj* pRenderModelObj : pHolder->renderModelObjs)
        {
            int shapeCount = pRenderModelObj->GetModelObj()->GetShapeCount();
            for (int shapeIndex = 0; shapeIndex < shapeCount; ++shapeIndex)
            {
                nns::g3d::RenderUnitObj* pRenderUnitObj = pRenderModelObj->GetRenderUnitObj(shapeIndex);
                nns::g3d::Sampler* pSampler = pRenderUnitObj->FindSampler(DrawPassType_Model, "_d0");

                if (!pSampler)
                {
                    continue;
                }

                pSampler->SetSamplerDescriptorSlot(samplerSlot);
            }
        }
    }
}

void TownViewer::Update() NN_NOEXCEPT
{
    // ロードされたモデルを遅延初期化します。
    CreateModelsFromLoadedResFiles();

    // アンロードされたモデルを遅延破棄します。
    DestroyUnloadedResFiles();

    // ビューアーライブラリーに登録したコマンド、3DEditor から受信したコマンドを処理します。
    nn::g3d::viewer::ViewerServer::GetInstance().ExecuteCommands();
}

nn::g3d::ResShaderArchive* TownViewer::FindShaderArchive(const char* pShaderName) NN_NOEXCEPT
{
    for (nn::g3d::ResShaderFile* pResShaderFile : GetResourceHolder()->shaderFiles)
    {
        if (strcmp(pResShaderFile->GetResShaderArchive()->GetName(), pShaderName) != 0)
        {
            continue;
        }
        return pResShaderFile->GetResShaderArchive();
    }

    return nullptr;
}

void TownViewer::AttachShader(const char* pShaderName) NN_NOEXCEPT
{
    nn::g3d::ResShaderArchive* pShader = FindShaderArchive(pShaderName);
    if (pShader)
    {
        if (!nn::g3d::viewer::ViewerServer::GetInstance().IsShaderArchiveAttached(pShader) &&
            !nn::g3d::viewer::ViewerServer::GetInstance().IsShaderArchiveAttaching(pShader))
        {
            nn::g3d::viewer::ViewerServer::GetInstance().QueueAttachShaderArchiveCommand(pShader);
        }
        else
        {
            nn::g3d::viewer::ViewerServer::GetInstance().QueueDetachShaderArchiveCommand(pShader);
        }
    }
}

// ビューアのテクスチャーバインドコールバックはカスタマイズすることができます。
nn::g3d::TextureRef TownViewer::ViewerTextureBindCallback(
    const char* name,
    const nn::gfx::ResTextureFile* pResTextureFile,
    void* pUserData
) NN_NOEXCEPT
{
    NN_UNUSED(pUserData);
    return g3ddemo::TextureBindCallback(name, (void*)pResTextureFile);
}

// 関数コールバックのサンプルです。
void TownViewer::TownViewerCallback(
    void* pOutArg,
    const void* pInArg,
    nn::g3d::viewer::CallbackType type,
    void* pUserData
) NN_NOEXCEPT
{
    TownViewer* pTownViewer = reinterpret_cast<TownViewer*>(pUserData);
    switch (type)
    {
    case nn::g3d::viewer::CallbackType_ModelFileLoaded:
        {
            NN_LOG("ModelFileLoaded\n");
            const nn::g3d::viewer::ModelFileLoadedArg* pCastedArg = reinterpret_cast<const nn::g3d::viewer::ModelFileLoadedArg*>(pInArg);
            nn::g3d::viewer::ModelFileLoadedOutArg* pCastedOutArg = reinterpret_cast<nn::g3d::viewer::ModelFileLoadedOutArg*>(pOutArg);

            // 3DEditor が開いたモデルがリソースとして送られてきます。
            // リソースファイルはアプリケーションが管理する必要あるため、コピーします。
            nn::g3d::ResFile* pResFile = g3ddemo::CreateCopiedResFile(pCastedArg->pResFile, pCastedArg->fileSize, pCastedArg->alignment);
            if (pResFile == nullptr)
            {
                nn::g3d::viewer::ViewerServer::SendUserMessageArg sendMessageArg(
                    "メモリー確保に失敗しました",
                    nn::g3d::viewer::ViewerServer::SendUserMessageArg::MessageType_Error,
                    nn::g3d::viewer::ViewerServer::SendUserMessageArg::MessageDestination_Log);
                nn::g3d::viewer::ViewerResult result = nn::g3d::viewer::ViewerServer::GetInstance().QueueSendUserMessageCommand(sendMessageArg);
                NN_UNUSED(result);
                return;
            }

            // コピーした ResFile は nn::g3d::viewer に渡す必要があります。
            pCastedOutArg->pResFile = pResFile;

            pTownViewer->GetResourceHolder()->files.PushBack(pResFile);

            // このコールバックの外で遅延初期化するために一時保存します。
            pTownViewer->PushLoadedModelFile(pResFile);
        }
        return;
    case nn::g3d::viewer::CallbackType_ModelFileUnloaded:
        {
            NN_LOG("ModelFileUnloaded\n");
            const nn::g3d::viewer::ModelFileUnloadedArg* pCastedArg = reinterpret_cast<const nn::g3d::viewer::ModelFileUnloadedArg*>(pInArg);
            g3ddemo::ResourceHolder* pHolder = pTownViewer->GetResourceHolder();

            // このコールバックの外でモデルの遅延破棄するために一時保存します。
            pTownViewer->PushUnloadedModelFile(pCastedArg->pResFile);

            // 破棄予定のモデルを描画対象から外します。
            pTownViewer->RemoveModelsFromResourceHolder(pCastedArg->pResFile);

            if (g_SelectedModelIndex >= pHolder->modelAnimObjs.GetCount())
            {
                g_SelectedModelIndex = pHolder->modelAnimObjs.GetCount() - 1;
                g_SelectedShapeIndex = 0;
                g_SelectedSubmeshIndex = 0;
            }
        }
        return;
    case nn::g3d::viewer::CallbackType_TextureFileLoaded:
        {
            NN_LOG("TextureFileLoaded\n");
            const nn::g3d::viewer::TextureFileLoadedArg* pCastedArg = reinterpret_cast<const nn::g3d::viewer::TextureFileLoadedArg*>(pInArg);
            g3ddemo::SetupTexture(g3ddemo::GetGfxFramework()->GetDevice(), pCastedArg->pResTextureFile);
        }
        return;
    case nn::g3d::viewer::CallbackType_TextureFileUnloaded:
        {
            NN_LOG("TextureFileUnloaded\n");
            const nn::g3d::viewer::TextureFileUnloadedArg* pCastedArg = reinterpret_cast<const nn::g3d::viewer::TextureFileUnloadedArg*>(pInArg);
            g3ddemo::CleanupTexture(g3ddemo::GetGfxFramework()->GetDevice(), pCastedArg->pResTextureFile);
        }
        return;
    case nn::g3d::viewer::CallbackType_ShaderAssignUpdated:
        {
            NN_LOG("ShaderAssignUpdated\n");

            const nn::g3d::viewer::ShaderAssignUpdatedArg* pCastedArg = reinterpret_cast<const nn::g3d::viewer::ShaderAssignUpdatedArg*>(pInArg);
            nn::gfx::Device* pDevice = g3ddemo::GetGfxFramework()->GetDevice();
            g3ddemo::ResourceHolder* pHolder = pTownViewer->GetResourceHolder();


            // モデルリソースの構造変更を伴う編集が 3DEditor で行われたため、
            // 古いモデルリソースの破棄と新しいモデルリソースの構築を行います。
            // モデルインスタンスが新しいリソースを参照するように、
            // ビューアーライブラリーが参照関係を変更します。

            int materialCount = pCastedArg->pNewResModel->GetMaterialCount();
            nn::g3d::ResShaderArchive** pResShaderArchivesPtr = nullptr;
            pResShaderArchivesPtr = g3ddemo::AllocateMemory<nn::g3d::ResShaderArchive*>(sizeof(nn::g3d::ResShaderArchive*) * materialCount);

            for (int materialIndex = 0; materialIndex < materialCount; ++materialIndex)
            {
                // サンプラーの再構築
                {
                    nn::g3d::ResMaterial* pOldResMaterial = pCastedArg->pOldResModel->GetMaterial(materialIndex);
                    nn::g3d::ResMaterial* pNewResMaterial = pCastedArg->pNewResModel->GetMaterial(materialIndex);
                    for (int samplerIndex = 0, samplerCount = pOldResMaterial->GetSamplerCount(); samplerIndex < samplerCount; ++samplerIndex)
                    {
                        g3ddemo::UnregisterSamplerFromDescriptorPool(pOldResMaterial->GetSampler(samplerIndex));
                    }

                    for (int samplerIndex = 0, samplerCount = pNewResMaterial->GetSamplerCount(); samplerIndex < samplerCount; ++samplerIndex)
                    {
                        const char* samplerName = pNewResMaterial->GetSamplerName(samplerIndex);
                        nn::gfx::Sampler* pSampler = pNewResMaterial->GetSampler(samplerIndex);
                        nn::gfx::DescriptorSlot descriptorSlot = g3ddemo::RegisterSamplerToDescriptorPool(pSampler);

                        // env の _d0 サンプラーはシャドウサンプラとして再構築します。
                        if (strcmp("_d0", samplerName) == 0 &&
                            strcmp("town_env", pNewResMaterial->GetShaderAssign()->GetShadingModelName()) == 0)
                        {
                            g3ddemo::UnregisterSamplerFromDescriptorPool(pSampler);
                            RebuildShadowSampler(pDevice, pSampler);
                            descriptorSlot = g3ddemo::RegisterSamplerToDescriptorPool(pSampler);
                            BindSampler(pHolder, descriptorSlot);
                        }
                        pNewResMaterial->SetSamplerDescriptorSlot(samplerIndex, descriptorSlot);
                    }
                }

                // シェーダーアーカイブを設定
                {
                    if (pCastedArg->pMaterialShaderInfos)
                    {
                        nn::g3d::viewer::ShaderAssignUpdatedArg::MaterialShaderInfo& materialShaderInfo = pCastedArg->pMaterialShaderInfos[materialIndex];
                        pResShaderArchivesPtr[materialIndex] = materialShaderInfo.pResShaderArchive;
                        if (pResShaderArchivesPtr[materialIndex])
                        {
                            continue;
                        }

                        if (materialShaderInfo.isOptimized && !materialShaderInfo.isOptimizationSkipped)
                        {
                            // 以下の条件を満たす場合にこの分岐に入ります。
                            //
                            // - 3DEditor チーム設定の<CreateShaderVariationCommand>の AllowNoOutput を true に設定している
                            // - 対象マテリアルについてシェーダバリエーション生成コマンドで fsv を出力しなかった
                            //   (3DEditor がシェーダコンバートをスキップした)
                            //
                            // このサンプルでは特に何もしていませんが、ここでアプリケーションが保有する最適化済みのシェーダを
                            // 3DEditor が作成するシェーダバーリエーションの代わりに割り当てることができます。
                            NN_LOG("3DEditor skipped creating shader variation for material \"%s\"\n",
                                            pCastedArg->pNewResModel->GetMaterialName(materialIndex));
                        }
                    }
                    nn::g3d::ResShaderAssign* pShaderAssign = pCastedArg->pNewResModel->GetMaterial(materialIndex)->GetShaderAssign();
                    pResShaderArchivesPtr[materialIndex] = FindShaderArchiveFromViewerShader(pShaderAssign);
                }
            }

            // モデルリソースの構造の変化に伴い RenderModel の再構築
            for (nns::g3d::RenderModelObj* pRenderModelObj : pHolder->renderModelObjs)
            {
                if (pCastedArg->pModelObj != pRenderModelObj->GetModelObj())
                {
                    continue;
                }

                nns::g3d::RenderModel* pRenderModel = pRenderModelObj->GetRenderModel();

                // RenderModel の再構築
                pRenderModel->Finalize(pDevice);
                pRenderModel->Initialize(pDevice, pCastedArg->pNewResModel, DrawPassType_Count);

                bool isShaderAssigned = true;

                // 描画パスの設定
                isShaderAssigned &= pRenderModel->CreateDrawPass(pDevice, DrawPassType_Model, pResShaderArchivesPtr, materialCount);
                isShaderAssigned &= pRenderModel->CreateDrawPass(pDevice, DrawPassType_Water, pResShaderArchivesPtr, materialCount);
                isShaderAssigned &= pRenderModel->CreateDrawPass(pDevice, DrawPassType_Shadow, g_pResShadingModelPtr[ShadingModelType_Shadow]);
                isShaderAssigned &= pRenderModel->CreateDrawPass(pDevice, DrawPassType_Geometry, g_pResShadingModelPtr[ShadingModelType_Geometry]);

                // アプリケーションがサポートしないシェーダーを割り当てたモデルが 3DEditor に読み込まれていたり、
                // 読み込み済みのモデルにサポート外のシェーダーが割り当てられた場合に 3DEditor にエラーログを出力します。
                if (!isShaderAssigned)
                {
                    nn::g3d::viewer::ViewerServer::SendUserMessageArg userMessageArg(
                        "シェーダーのバインドに失敗しました。 town シェーダーを割り当てます。",
                        nn::g3d::viewer::ViewerServer::SendUserMessageArg::MessageType_Error,
                        nn::g3d::viewer::ViewerServer::SendUserMessageArg::MessageDestination_Log);
                    nn::g3d::viewer::ViewerServer::GetInstance().QueueSendUserMessageCommand(userMessageArg);

                    nn::g3d::ResShadingModel* pTownShadingModel = g_pTownShaderArchive->FindShadingModel("town");
                    NN_ASSERT_NOT_NULL(pTownShadingModel);
                    nn::g3d::ResShadingModel* pResShadingModels[DrawPassType_Count] =
                    {
                        pTownShadingModel,
                        pTownShadingModel,
                        g_pResShadingModelPtr[ShadingModelType_Shadow],
                        g_pResShadingModelPtr[ShadingModelType_Geometry]
                    };

                    for (int passIndex = 0; passIndex < DrawPassType_Count; ++passIndex)
                    {
                        // RenderMaterial の初期化
                        for (int materialIndex = 0; materialIndex < materialCount; ++materialIndex)
                        {
                            nns::g3d::RenderMaterial* pRenderMaterial = pRenderModel->GetRenderMaterial(materialIndex);
                            if (pRenderMaterial->IsInitialized())
                            {
                                continue;
                            }

                            nn::g3d::ResMaterial* pResMaterial = pCastedArg->pNewResModel->GetMaterial(materialIndex);
                            bool isInitialized = pRenderMaterial->Initialize(pDevice, pResMaterial, pResShadingModels[passIndex]);
                            NN_ASSERT(isInitialized);
                            NN_UNUSED(isInitialized);
                        }

                        // RenderShape の初期化
                        for (int shapeIndex = 0, shapeCount = pRenderModel->GetResModel()->GetShapeCount(); shapeIndex < shapeCount; ++shapeIndex)
                        {
                            nns::g3d::RenderShape* pRenderShape = pRenderModel->GetRenderShape(passIndex, shapeIndex);
                            if (pRenderShape->IsInitialized())
                            {
                                continue;
                            }

                            nn::g3d::ResShape* pResShape = pCastedArg->pNewResModel->GetShape(shapeIndex);
                            int materialIndex = pResShape->GetMaterialIndex();
                            nns::g3d::RenderMaterial* pRenderMaterial = pRenderModel->GetRenderMaterial(passIndex, materialIndex);
                            NN_ASSERT(pRenderMaterial->IsInitialized());

                            bool isInitialized = pRenderShape->Initialize(pDevice, pResShape, pRenderMaterial);
                            NN_ASSERT(isInitialized);
                            NN_UNUSED(isInitialized);
                        }
                    }
                }

                SetupRenderState(pRenderModel);
            }

            // ShaderScratchMemory のサイズを再計算します。
            {
                g3ddemo::Vector<nn::g3d::ResShaderArchive*> shaderArchives;
                g3ddemo::CreateVector<nn::g3d::ResShaderArchive*>(&shaderArchives, materialCount + pHolder->shaderFiles.GetCount());

                for (int materialIndex = 0; materialIndex < materialCount; ++materialIndex)
                {
                    shaderArchives.PushBack(pResShaderArchivesPtr[materialIndex]);
                }

                for (nn::g3d::ResShaderFile* pResShaderFile : pHolder->shaderFiles)
                {
                    shaderArchives.PushBack(pResShaderFile->GetResShaderArchive());
                }
                g3ddemo::SetShaderScratchMemory(shaderArchives);
                g3ddemo::DestroyVector<nn::g3d::ResShaderArchive*>(&shaderArchives);
            }

            g3ddemo::FreeMemory(pResShaderArchivesPtr);
        }
        return;
    case nn::g3d::viewer::CallbackType_MaterialUpdated:
        {
            NN_LOG("MaterialUpdated\n");
            const nn::g3d::viewer::MaterialUpdatedArg* pCastedArg = reinterpret_cast<const nn::g3d::viewer::MaterialUpdatedArg*>(pInArg);
            g3ddemo::ResourceHolder* pHolder = pTownViewer->GetResourceHolder();

            // ShaderAssignUpdated() で行われた変更に伴って ModelObj に変更が必要な場合は
            // このコールバックで処理を行います。
            // 編集のために使用するヒープを区別したい場合は arg.state で状態を判定します。
            // ShaderAssignUpdated() で初期化したリソースに従って MaterialObj や ShapeObj が
            // ビューアーライブラリーによって作り直されているので、必要な初期化を行います。

            nn::g3d::ModelObj* pModelObj = pCastedArg->pModelObj;

            // RenderModel の変更に追従するため RenderModelObj を再構築します。
            for (nns::g3d::RenderModelObj* pRenderModelObj : pHolder->renderModelObjs)
            {
                if (pRenderModelObj->GetModelObj() != pModelObj)
                {
                    continue;
                }

                nns::g3d::RenderModel* pRenderModel = pRenderModelObj->GetRenderModel();
                pRenderModelObj->Finalize();
                pRenderModelObj->Initialize(pModelObj, pRenderModel);

                // LOD 判定やユニフォームブロックに関する再設定を行います。
                SetupRenderModelObj(pRenderModelObj);
            }
        }
        return;
    case nn::g3d::viewer::CallbackType_ShapeUpdated:
        {
            NN_LOG("ShapeUpdated\n");
            const nn::g3d::viewer::ShapeUpdatedArg* pCastedArg = reinterpret_cast<const nn::g3d::viewer::ShapeUpdatedArg*>(pInArg);

            nn::g3d::ModelObj* pUpdatedModelObj = pCastedArg->pModelObj;
            nns::g3d::ModelAnimObj* pTargetModelAnimObj = FindModelAnimObjFromResourceHolder(pTownViewer->GetResourceHolder(), pUpdatedModelObj);
            pTargetModelAnimObj->Calculate();
        }
    return;
    case nn::g3d::viewer::CallbackType_BoneBindUpdated:
        {
            NN_LOG("BoneBindUpdated\n");
            const nn::g3d::viewer::BoneBindUpdatedArg* pCastedArg = reinterpret_cast<const nn::g3d::viewer::BoneBindUpdatedArg*>(pInArg);

            // 3DEditor でプレビュー配置のバインド設定が行われた際にモデルに通知を行います。
            // 配置自体はここで行わずにアプリケーションのバインド機能によってバインドします。
            nns::g3d::ModelAnimObj* pTargetModelAnimObj = FindModelAnimObjFromResourceHolder(pTownViewer->GetResourceHolder(), pCastedArg->pModelObj);
            pTargetModelAnimObj->BindTo(pCastedArg->pParentModelObj, pCastedArg->parentBoneIndex);
        }
        return;
    case nn::g3d::viewer::CallbackType_ModelLayoutUpdated:
        {
            NN_LOG("ModelLayoutUpdated\n");
            const nn::g3d::viewer::ModelLayoutUpdatedArg* pCastedArg = reinterpret_cast<const nn::g3d::viewer::ModelLayoutUpdatedArg*>(pInArg);

            pTownViewer->SetModelLayout(*pCastedArg);

            nns::g3d::ModelAnimObj* pModelAnimObj = FindModelAnimObjFromResourceHolder(pTownViewer->GetResourceHolder(), pCastedArg->pModelObj);
            if (g3ddemo::IsInitialTranslate(*pCastedArg) && !pModelAnimObj->IsBound())
            {
                g3ddemo::MoveModelToFrontOfCamera(pModelAnimObj, g_CameraPosition, g_CameraTarget);

                nn::g3d::viewer::ViewerServer::LayoutModelArg layoutModelArg(
                    pCastedArg->pModelObj,
                    pModelAnimObj->GetScale(),
                    pModelAnimObj->GetRotate(),
                    pModelAnimObj->GetTranslate());
                nn::g3d::viewer::ViewerServer::GetInstance().QueueLayoutModelCommand(layoutModelArg);
            }
        }
        return;
    case nn::g3d::viewer::CallbackType_SendModelLayoutRequested:
        {
            NN_LOG("SendModelLayoutRequested\n");
            nn::g3d::viewer::SendModelLayoutRequestedOutArg* pCastedOutArg = reinterpret_cast<nn::g3d::viewer::SendModelLayoutRequestedOutArg*>(pOutArg);
            const nn::g3d::viewer::SendModelLayoutRequestedArg* pCastedArg = reinterpret_cast<const nn::g3d::viewer::SendModelLayoutRequestedArg*>(pInArg);

            nns::g3d::ModelAnimObj* pModelAnimObj = FindModelAnimObjFromResourceHolder(pTownViewer->GetResourceHolder(), pCastedArg->pModelObj);
            pCastedOutArg->scale = pModelAnimObj->GetScale();
            pCastedOutArg->rotate = pModelAnimObj->GetRotate();
            pCastedOutArg->translate = pModelAnimObj->GetTranslate();
        }
        return;
    case nn::g3d::viewer::CallbackType_RenderInfoUpdated:
        {
            NN_LOG("RenderInfoUpdated\n");
            const nn::g3d::viewer::RenderInfoUpdatedArg* pCastedArg = reinterpret_cast<const nn::g3d::viewer::RenderInfoUpdatedArg*>(pInArg);

            pTownViewer->UpdateRenderInfo(pCastedArg->pModelObj, pCastedArg->materialIndex, pCastedArg->name);
        }
        return;
    case nn::g3d::viewer::CallbackType_TargetSelected:
        {
            NN_LOG("TargetSelected\n");
            const nn::g3d::viewer::TargetSelectedArg* pCastedArg = reinterpret_cast<const nn::g3d::viewer::TargetSelectedArg*>(pInArg);
            NN_UNUSED(pCastedArg);
            if (pCastedArg->targetKind == nn::g3d::viewer::TargetSelectedArg::TargetKind_Model)
            {
                // 選択されたモデルにフォーカスします。
                for (int modelIndex = 0, modelCount = pTownViewer->GetResourceHolder()->modelAnimObjs.GetCount(); modelIndex < modelCount; ++modelIndex)
                {
                    nns::g3d::ModelAnimObj* pModelAnimObj = pTownViewer->GetResourceHolder()->modelAnimObjs[modelIndex];
                    if (pModelAnimObj->GetModelObj() != pCastedArg->pModelObj)
                    {
                        continue;
                    }

                    g_SelectedModelIndex = modelIndex;
                    g_SelectedShapeIndex = 0;
                    g_SelectedSubmeshIndex = 0;
                    break;
                }
            }
        }
        return;
    case nn::g3d::viewer::CallbackType_SamplerParamUpdated:
        {
            NN_LOG("SamplerParamUpdated\n");
            const nn::g3d::viewer::SamplerParamUpdatedArg* pCastedArg = reinterpret_cast<const nn::g3d::viewer::SamplerParamUpdatedArg*>(pInArg);

            // サンプラーが更新されたのでディスクリプタプールへ再登録します。
            g3ddemo::ReregisterSamplerToDescriptorPool(pCastedArg->pModelObj);
        }
        return;
    case nn::g3d::viewer::CallbackType_ShaderProgramUpdated:
        {
            NN_LOG("ShaderProgramUpdated\n");
            const nn::g3d::viewer::ShaderProgramUpdatedArg* pCastedArg = reinterpret_cast<const nn::g3d::viewer::ShaderProgramUpdatedArg*>(pInArg);
            NN_UNUSED(pCastedArg);
            pTownViewer->SetShaderMemory();
        }
        return;
    case nn::g3d::viewer::CallbackType_ShaderAttached:
        {
            NN_LOG("ShaderAttached\n");
        }
        return;
    case nn::g3d::viewer::CallbackType_ShaderDetached:
        {
            NN_LOG("ShaderDetached\n");
        }
        return;
    case nn::g3d::viewer::CallbackType_ModelUserScriptExecuted:
        {
            NN_LOG("ModelUserScriptExecuted\n");
            const nn::g3d::viewer::ModelUserScriptExecutedArg* pCastedArg = reinterpret_cast<const nn::g3d::viewer::ModelUserScriptExecutedArg*>(pInArg);

            NN_LOG("ScriptText: \n%s\n", pCastedArg->scriptText);

            const nn::g3d::ResModel* pResModel = pCastedArg->pTargetModelObj->GetResource();
            NN_LOG("Target Model Name: %s\n", pResModel->GetName());

            NN_LOG("selectedBoneCount = %d\n", pCastedArg->selectedBoneCount);
            for (int i = 0; i < pCastedArg->selectedBoneCount; ++i)
            {
                int boneIndex = pCastedArg->pSelectedBoneIndices[i];
                NN_LOG("pSelectedBoneIndices[%d] = %d\n", i, boneIndex);
            }

            NN_LOG("selectedShapeCount = %d\n", pCastedArg->selectedShapeCount);
            for (int i = 0; i < pCastedArg->selectedShapeCount; ++i)
            {
                int shapeIndex = pCastedArg->pSelectedShapeIndices[i];
                NN_LOG("pSelectedShapeIndices[%d] = %d\n", i, shapeIndex);
            }

            NN_LOG("selectedMaterialCount = %d\n", pCastedArg->selectedMaterialCount);
            for (int i = 0; i < pCastedArg->selectedMaterialCount; ++i)
            {
                int materialIndex = pCastedArg->pSelectedMaterialIndices[i];
                NN_LOG("pSelectedMaterialIndices[%d] = %d\n", i, materialIndex);
            }
        }
        return;
    default:
        {
            NN_LOG("TownViewerCallback: %d\n", type);
        }
        return;
    }
} // NOLINT

void TownViewer::FinalizeInternal() NN_NOEXCEPT
{
    DestroyUnloadedResFiles();
}

void TownViewer::PushLoadedModelFile(nn::g3d::ResFile* pResFile) NN_NOEXCEPT
{
    NN_ASSERT(m_LoadedResFileCount < LoadModelFileCountMax);
    m_LoadedResFiles[m_LoadedResFileCount] = pResFile;
    ++m_LoadedResFileCount;
}

void TownViewer::PushUnloadedModelFile(nn::g3d::ResFile* pResFile) NN_NOEXCEPT
{
    NN_ASSERT(m_UnloadedResFileCount < LoadModelFileCountMax);
    m_UnloadedResFiles[m_UnloadedResFileCount] = pResFile;
    ++m_UnloadedResFileCount;
}

void TownViewer::CreateModelsFromLoadedResFiles() NN_NOEXCEPT
{
    nn::gfx::Device* pDevice = g3ddemo::GetGfxFramework()->GetDevice();
    g3ddemo::ResourceHolder* pHolder = GetResourceHolder();

    // ひとつの ResFile から作成した任意のインスタンス数分のモデルを g3d::viewer に登録することが可能です。
    for (int fileIndex = 0; fileIndex < m_LoadedResFileCount; ++fileIndex)
    {
        nn::g3d::ResFile* pResFile = m_LoadedResFiles[fileIndex];

        // pResFile 内に入っている ResMaterial の シェーダーアサインを確認し、
        // 適切なシェーダーを pResFile の UserPtr に設定します。
        // 一つの ResFile で 異なるシェーダーがアサインされている場合には、
        // ShaderAssignUpdated にて適切なシェーダーをアサインします。
        const nn::g3d::ResShaderAssign* pResShaderAssign = pResFile->GetModel(0)->GetMaterial(0)->GetShaderAssign();
        nn::g3d::ResShadingModel* pResShadingModel = nullptr;
        nn::g3d::ResShaderArchive* pModelShaderArchive = nullptr;
        for (int shaderIndex = 0; shaderIndex < NN_ARRAY_SIZE(g_pViewerResShaderArchivePtrArray); ++shaderIndex)
        {
            nn::g3d::ResShaderArchive* pResShaderArchive = (*(g_pViewerResShaderArchivePtrArray[shaderIndex]));
            pResShadingModel = pResShaderArchive->FindShadingModel(pResShaderAssign->GetShadingModelName());
            if (pResShadingModel != nullptr)
            {
                pModelShaderArchive = pResShaderArchive;
                break;
            }
        }

        if (!pModelShaderArchive)
        {
            nn::g3d::viewer::ViewerServer::SendUserMessageArg userMessageArg(
                "シェーダーアサインに登録されているシェーディングモデルが見つかりませんでした。",
                nn::g3d::viewer::ViewerServer::SendUserMessageArg::MessageType_Error,
                nn::g3d::viewer::ViewerServer::SendUserMessageArg::MessageDestination_Log);
            nn::g3d::viewer::ViewerServer::GetInstance().QueueSendUserMessageCommand(userMessageArg);
        }

        pResFile->SetUserPtr(pModelShaderArchive);

        // リソース内のサンプラーをデスクリプタプールに登録します。
        g3ddemo::RegisterSamplerToDescriptorPool(pResFile);

        // モデルインスタンス構築処理は、アプリケーションで使用している構築処理を
        // そのまま使用することができます。

        CreateRenderModel(pHolder, pDevice, pResFile);

        // モデルインスタンスの生成を行い、QueueAttachModelCommand() 関数を用いてビューアーに登録します。
        CreateModelAnimObj(pHolder, pDevice, pResFile, QueueAttachModelCommand);

        int modelCount = pResFile->GetModelCount();
        const int modelAnimObjCount = pHolder->modelAnimObjs.GetCount();
        const int renderModelCount = pHolder->renderModels.GetCount();
        while (modelCount > 0)
        {
            CreateRenderModelObj(
                pHolder,
                pHolder->modelAnimObjs[modelAnimObjCount - modelCount],
                pHolder->renderModels[renderModelCount - modelCount]
            );

            --modelCount;
        }
    }

    m_LoadedResFileCount = 0;
}

void TownViewer::DestroyUnloadedResFiles() NN_NOEXCEPT
{
    if (m_UnloadedResFileCount == 0)
    {
        return;
    }

    g3ddemo::ResourceHolder* pResourceHolder = GetResourceHolder();
    nn::gfx::Device* pDevice = g3ddemo::GetGfxFramework()->GetDevice();
    for (int unloadedResFileIndex = 0; unloadedResFileIndex < m_UnloadedResFileCount; ++unloadedResFileIndex)
    {
        nn::g3d::ResFile* pResFile = m_UnloadedResFiles[unloadedResFileIndex];
        for (int fileIndex = 0, fileCount = pResourceHolder->files.GetCount(); fileIndex < fileCount; ++fileIndex)
        {
            if (pResourceHolder->files[fileIndex] != pResFile)
            {
                continue;
            }

            g3ddemo::UnregisterSamplerFromDescriptorPool(pResFile);
            pResFile->Cleanup(pDevice);
            g3ddemo::FreeMemory(pResFile);
            pResourceHolder->files.Erase(fileIndex);
            break;
        }
    }

    if (g_SelectedModelIndex >= GetResourceHolder()->modelAnimObjs.GetCount())
    {
        g_SelectedModelIndex = GetResourceHolder()->modelAnimObjs.GetCount() - 1;
        g_SelectedShapeIndex = 0;
        g_SelectedSubmeshIndex = 0;
    }

    m_UnloadedResFileCount = 0;
}

void TownViewer::SetShaderMemory() NN_NOEXCEPT
{
    g3ddemo::SetShaderScratchMemory(GetResourceHolder()->shaderFiles);
}

void TownViewer::RemoveModelsFromResourceHolder(nn::g3d::ResFile* pResFile) NN_NOEXCEPT
{
    nn::gfx::Device* pDevice = g3ddemo::GetGfxFramework()->GetDevice();
    g3ddemo::ResourceHolder* pHolder = GetResourceHolder();

    for (int modelIndex = 0, modelCount = pResFile->GetModelCount(); modelIndex < modelCount; ++modelIndex)
    {
        nn::g3d::ResModel* pTargetResModel = pResFile->GetModel(modelIndex);

        // RenderModelObj の破棄
        int renderModelObjIndex = 0;
        while (renderModelObjIndex < pHolder->renderModelObjs.GetCount())
        {
            nns::g3d::RenderModelObj* pRenderModelObj = pHolder->renderModelObjs[renderModelObjIndex];
            const nn::g3d::ResModel* pResModel = pRenderModelObj->GetModelObj()->GetResource();
            if (pTargetResModel != pResModel)
            {
                ++renderModelObjIndex;
                continue;
            }

            nns::g3d::DestroyRenderModelObj(pRenderModelObj);
            pHolder->renderModelObjs.Erase(renderModelObjIndex);
        }

        // ModelAnimObj の破棄
        int modelAnimIndex = 0;
        while (modelAnimIndex < pHolder->modelAnimObjs.GetCount())
        {
            nns::g3d::ModelAnimObj* pModelAnimObj = pHolder->modelAnimObjs[modelAnimIndex];
            const nn::g3d::ResModel* pResModel = pModelAnimObj->GetModelObj()->GetResource();
            if (pTargetResModel != pResModel)
            {
                ++modelAnimIndex;
                continue;
            }

            nns::g3d::DestroyModelAnimObj(pDevice, pModelAnimObj);
            pHolder->modelAnimObjs.Erase(modelAnimIndex);
        }

        // RenderModel の破棄
        int renderModelIndex = 0;
        while (renderModelIndex < pHolder->renderModels.GetCount())
        {
            nns::g3d::RenderModel* pRenderModel = pHolder->renderModels[renderModelIndex];
            const nn::g3d::ResModel* pResModel = pRenderModel->GetResModel();
            if (pTargetResModel != pResModel)
            {
                ++renderModelIndex;
                continue;
            }

            nns::g3d::DestroyRenderModel(pDevice, pRenderModel);
            pHolder->renderModels.Erase(renderModelIndex);
        }
    }
}

void TownViewer::UpdateRenderInfo(const nn::g3d::ModelObj* pModelObj, int materialIndex, const char* renderInfoName) NN_NOEXCEPT
{
    const nn::g3d::ResRenderInfo* pRenderInfo = pModelObj->GetMaterial(materialIndex)->GetResource()->FindRenderInfo(renderInfoName);
    if (!pRenderInfo)
    {
        return;
    }

    g3ddemo::ResourceHolder* pHolder = GetResourceHolder();
    for (nns::g3d::RenderModelObj* pRenderModelObj : pHolder->renderModelObjs)
    {
        if (pRenderModelObj->GetModelObj() != pModelObj)
        {
            continue;
        }

        if (strcmp("culling", renderInfoName) == 0)
        {
            SetWireframe(pRenderModelObj, g_MenuFlag.Test<MenuFlag::Wireframe>() ? g3ddemo::FillMode_Wireframe : g3ddemo::FillMode_Solid, materialIndex);
        }
        else if (strcmp("blend", renderInfoName) == 0)
        {
            int blendType = *pRenderInfo->GetInt();
            NN_ASSERT_RANGE(blendType, g3ddemo::BlendMode_Opaque, g3ddemo::BlendMode_Max);
            const nn::gfx::BlendState* pBlendState = g3ddemo::GetBlendState(blendType);
            pRenderModelObj->GetRenderModel()->GetRenderMaterial(materialIndex)->SetBlendState(pBlendState);
        }
        else if (strcmp("ice", renderInfoName) == 0)
        {
            bool isIceOptionEnabled = pRenderInfo && (*pRenderInfo->GetInt() != 0);
            nns::g3d::RenderMaterial* pRenderMaterial = pRenderModelObj->GetRenderModel()->GetRenderMaterial(materialIndex);

            for (int shapeIndex = 0, shapeCount = pRenderModelObj->GetModelObj()->GetShapeCount(); shapeIndex < shapeCount; ++shapeIndex)
            {
                nns::g3d::RenderUnitObj* pRenderUnitObj = pRenderModelObj->GetRenderUnitObj(shapeIndex);
                if (pRenderUnitObj->GetRenderShape()->GetRenderMaterial() != pRenderMaterial)
                {
                    continue;
                }

                nn::g3d::ShaderSelector* pShaderSelector = pRenderUnitObj->GetShaderSelector();
                int optionIndex = pShaderSelector->FindDynamicOptionIndex("ice_world");
                if (optionIndex == nn::util::ResDic::Npos)
                {
                    continue;
                }

                const nn::g3d::ResShaderOption* pResShaderOption = pShaderSelector->GetDynamicOption(optionIndex);
                int choiceIndex = pResShaderOption->FindChoiceIndex(isIceOptionEnabled ? "1" : "0");
                pShaderSelector->WriteDynamicKey(optionIndex, choiceIndex);
            }
        }
    }
}
