﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "testG3d_CaptureSetting.h"

#include <nn/fs.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>

#include <string>

namespace {
    const char ReturnCodes[] = { '\n', '\r' };
    const char ReturnCodeCount = sizeof(ReturnCodes) / sizeof(char);

    void ReadFileToStringBuffer(
        char* pOutReadBuffer, size_t* pOutReadSize, size_t readBufferSize, const char* filePath)
    {
        nn::fs::FileHandle handle;
        nn::Result openResult = nn::fs::OpenFile(&handle, filePath, nn::fs::OpenMode_Read);
        NN_ASSERT(openResult.IsSuccess());

        nn::Result readResult = nn::fs::ReadFile(pOutReadSize, handle, 0, pOutReadBuffer, readBufferSize);
        NN_ASSERT(readResult.IsSuccess());

        // このバッファを文字列としてそのまま扱えるようにするため、末尾を NULL 文字で埋めるので +1 する
        *pOutReadSize += 1;
        NN_ASSERT(*pOutReadSize < readBufferSize, "Tool large file size");
        pOutReadBuffer[*pOutReadSize - 1] = '\0';

        nn::fs::CloseFile(handle);
    }

    bool IsMatched(char target, const char* pMatchCharList, int matchCharListCount)
    {
        for (int charIndex = 0; charIndex < matchCharListCount; ++charIndex)
        {
            if (target == pMatchCharList[charIndex])
            {
                return true;
            }
        }

        return false;
    }

    // 後ろの空白を削除する
    void TrimTrailingWhitespace(char* pInOutTarget)
    {
        for (char* pChar = pInOutTarget + strlen(pInOutTarget) - 1, *pEnd = pInOutTarget - 1;
            pChar != pEnd; --pChar)
        {
            if (*pChar == ' ')
            {
                *pChar = '\0';
            }
            else
            {
                return;
            }
        }
    }

    // 文字列バッファを文字区切りで文字列リストに変換します。
    // # で始まる行はスキップします。
    void ConvertStringBufferToStringList(
        char** pOutStringList, int* pOutCount, size_t maxStringListCount,
        char* pSourceBuffer, size_t bufferSize, const char* delimiters, int delimiterCount)
    {
        NN_UNUSED(maxStringListCount);
        const char CommentOutMark = '#';
        int elemCount = 0;
        char* currentString = pSourceBuffer;
        for (char* pReadData = pSourceBuffer, *pEnd = pSourceBuffer + bufferSize; pReadData != pEnd; ++pReadData)
        {
            if (IsMatched(*pReadData, delimiters, delimiterCount) || ((pReadData + 1) == pEnd))
            {
                // ファイルリード時に pEnd のひとつ前のアドレスは \0 に書き換えているので
                // (pReadData + 1) == pEnd のときに pReadData を \0 に書き換えても問題ない
                *pReadData = '\0';

                // 先頭の空白文字を飛ばす
                while (*currentString == ' ')
                {
                    ++currentString;
                }

                // 先頭が # はコメントとみなす
                if ((*currentString != CommentOutMark) && (*currentString != '\0'))
                {
                    NN_ASSERT(elemCount < static_cast<int>(maxStringListCount), "Too many elements");
                    TrimTrailingWhitespace(currentString);
                    pOutStringList[elemCount] = currentString;
                    ++elemCount;
                }

                currentString = pReadData + 1;
            }
        }

        *pOutCount = elemCount;
    }

    bool IsMatched(const char* str1, const char* str2)
    {
        return strcmp(str1, str2) == 0;
    }
}

nnt::g3d::CaptureSetting::CaptureSetting() NN_NOEXCEPT
    : m_LodLevel(-1)
    , m_ResourceCount(0)
    , m_ModelCount(0)
    , m_CameraCount(0)
    , m_pCurrentReadStringBuffer(&m_ReadStringBuffer[0])
    , m_CurrentReadStringBufferSize(ReadBufferSizeMax)
{
}

void nnt::g3d::CaptureSetting::ParseStringToModelSetting(char* pSourceString) NN_NOEXCEPT
{
    char* valueStringList[8];
    int valueCount;
    {
        const char ParamDelims[] = { ',' };
        const int ParamDelimCount = sizeof(ParamDelims) / sizeof(char);
        ConvertStringBufferToStringList(
            valueStringList, &valueCount, sizeof(valueStringList) / sizeof(char*), pSourceString, strlen(pSourceString) + 1,
            ParamDelims, ParamDelimCount);
        NN_ASSERT(4 <= valueCount && valueCount <= 8, "Invalid model setting element count");
    }

    Model* pModelSetting = &m_Models[m_ModelCount];
    pModelSetting->name = valueStringList[0];
    pModelSetting->posX = std::stof(valueStringList[1]);
    pModelSetting->posY = std::stof(valueStringList[2]);
    pModelSetting->posZ = std::stof(valueStringList[3]);
    if (valueCount > 4)
    {
        pModelSetting->isAutoAnimBindEnabled = strcmp("true", valueStringList[4]) == 0 ? true : false;
        if (pModelSetting->isAutoAnimBindEnabled && valueCount > 5)
        {
            pModelSetting->autoBindAnimName = valueStringList[5];
            if (valueCount > 6)
            {
                pModelSetting->isMirroringEnabled = strcmp("true", valueStringList[6]) == 0 ? true : false;
                if (valueCount > 7)
                {
                    pModelSetting->retargetingHostName = valueStringList[7];
                }
            }
        }
    }

    NN_LOG("Models[%d]: %s, (%f, %f, %f), %s(animName:%s, mirroring:%s, retargetHost:%s)\n",
        m_ModelCount,
        pModelSetting->name,
        pModelSetting->posX,
        pModelSetting->posY,
        pModelSetting->posZ,
        pModelSetting->isAutoAnimBindEnabled ? "true" : "false",
        pModelSetting->autoBindAnimName ? pModelSetting->autoBindAnimName : pModelSetting->name,
        pModelSetting->isMirroringEnabled ? "true" : "false",
        pModelSetting->retargetingHostName ? pModelSetting->retargetingHostName : "false");

    ++m_ModelCount;
}

void nnt::g3d::CaptureSetting::ParseStringToCameraSetting(char* pSourceString) NN_NOEXCEPT
{
    char* valueStringList[9];
    int valueCount;
    {
        const char ParamDelims[] = { ',' };
        const int ParamDelimCount = sizeof(ParamDelims) / sizeof(char);
        ConvertStringBufferToStringList(
            valueStringList, &valueCount, sizeof(valueStringList) / sizeof(char*), pSourceString, strlen(pSourceString) + 1, ParamDelims, ParamDelimCount);
        NN_ASSERT(6 <= valueCount && valueCount <= 9, "Invalid camera setting element count");
    }

    Camera* pCameraSetting = &m_Cameras[m_CameraCount];
    pCameraSetting->posX = std::stof(valueStringList[0]);
    pCameraSetting->posY = std::stof(valueStringList[1]);
    pCameraSetting->posZ = std::stof(valueStringList[2]);
    pCameraSetting->targetX = std::stof(valueStringList[3]);
    pCameraSetting->targetY = std::stof(valueStringList[4]);
    pCameraSetting->targetZ = std::stof(valueStringList[5]);
    if (valueCount >= 7)
    {
        pCameraSetting->frame = std::stof(valueStringList[6]);
    }
    if (valueCount >= 8)
    {
        pCameraSetting->shapeIndex = std::stoi(valueStringList[7]);
    }
    if (valueCount >= 9)
    {
        pCameraSetting->sumeshIndex = std::stoi(valueStringList[8]);
    }

    NN_LOG("Cameras[%d]: pos(%f, %f, %f), target(%f, %f, %f), frame = %f\n",
        m_CameraCount,
        pCameraSetting->posX,
        pCameraSetting->posY,
        pCameraSetting->posZ,
        pCameraSetting->targetX,
        pCameraSetting->targetY,
        pCameraSetting->targetZ,
        pCameraSetting->frame);

    ++m_CameraCount;
}

void nnt::g3d::CaptureSetting::ParseStringToSettingNameAndValue(char** pOutName, char** pOutValue, char* pSourceString) NN_NOEXCEPT
{
    char* pEqualSplitedStrings[2];
    int splitedCount;
    {
        const char ParamDelims[] = { '=' };
        const int ParamDelimCount = sizeof(ParamDelims) / sizeof(char);
        ConvertStringBufferToStringList(
            pEqualSplitedStrings, &splitedCount, sizeof(pEqualSplitedStrings) / sizeof(char*), pSourceString, strlen(pSourceString) + 1,
            ParamDelims, ParamDelimCount);
        NN_ASSERT(splitedCount == 2, "Invalid \"=\" count");
    }

    *pOutName = pEqualSplitedStrings[0];
    *pOutValue = pEqualSplitedStrings[1];
}

void nnt::g3d::CaptureSetting::LoadSetting(const char* filePath) NN_NOEXCEPT
{
    size_t readBytes;
    ReadFileToStringBuffer(m_pCurrentReadStringBuffer, &readBytes, m_CurrentReadStringBufferSize, filePath);
    char* pReadStringList[ResourceCountMax];
    int readStringCount;
    ConvertStringBufferToStringList(
        pReadStringList, &readStringCount, ResourceCountMax, m_pCurrentReadStringBuffer, readBytes,
        ReturnCodes, ReturnCodeCount);

    // 次回読むときのためにバッファを末尾にずらす
    m_pCurrentReadStringBuffer = m_pCurrentReadStringBuffer + readBytes;
    m_CurrentReadStringBufferSize = m_CurrentReadStringBufferSize - readBytes;

    // 各行の解析
    for (int readStringIndex = 0; readStringIndex < readStringCount; ++readStringIndex)
    {
        char* pParamName, *pParamValue;
        ParseStringToSettingNameAndValue(&pParamName, &pParamValue, pReadStringList[readStringIndex]);

        if (IsMatched(pParamName, "Resource"))
        {
            // パス指定
            m_Resources[m_ResourceCount].path = pParamValue;
            NN_LOG("Resources[%d]: %s\n",
                m_ResourceCount, m_Resources[m_ResourceCount].path);
            ++m_ResourceCount;
        }
        else if (IsMatched(pParamName, "Model"))
        {
            // モデルの座標指定
            ParseStringToModelSetting(pParamValue);
        }
        else if (IsMatched(pParamName, "Camera"))
        {
            // カメラ設定
            ParseStringToCameraSetting(pParamValue);
        }
        else if (IsMatched(pParamName, "LodLevel"))
        {
            // LOD レベル指定
            m_LodLevel = std::stoi(pParamValue);
            NN_LOG("Lod Level: %d\n", m_LodLevel);
        }
        else if (IsMatched(pParamName, "OutputImageName"))
        {
            m_OutputImageName = pParamValue;
            NN_LOG("OutputImageName: %s\n", m_OutputImageName);
        }
        else
        {
            NN_ABORT("Invalid parameter %s", pParamName);
        }
    }
}
