﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <cstdint>
#include <nn/fs.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nnt/nnt_Argument.h>
#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/teamcity/testTeamcity_Logger.h>

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
#include <nv/nv_MemoryManagement.h>
#endif

#include "testEft_ProgramOptions.h"

//------------------------------------------------------------------------------
// メモリ割り当て・破棄関数
//------------------------------------------------------------------------------
static void* Allocate(size_t s)
{
    return malloc(s);
}

static void Deallocate(void* p, size_t)
{
    free(p);
}

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
//------------------------------------------------------------------------------
// グラフィックスシステム用メモリ割り当て・破棄関数
//------------------------------------------------------------------------------
static void* NvAllocateFunction(size_t size, size_t alignment, void* userPtr)
{
    NN_UNUSED(userPtr);
    return aligned_alloc(alignment, size);
}
static void NvFreeFunction(void* addr, void* userPtr)
{
    NN_UNUSED(userPtr);
    free(addr);
}
static void* NvReallocateFunction(void* addr, size_t newSize, void* userPtr)
{
    NN_UNUSED(userPtr);
    return realloc(addr, newSize);
}
#endif

nnt::eft::ProgramOptions g_ProgramOptions = {};

//---------------------------------------------------------------------------
//  Main 関数
//---------------------------------------------------------------------------
extern "C" void nnMain()
{
    nn::fs::SetAllocator(Allocate, Deallocate);

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
    // グラフィックスシステムのためのメモリ周りの初期化を行います。
    {
        const size_t GraphicsSystemMemorySize = 8 * 1024 * 1024;
        nv::SetGraphicsAllocator(NvAllocateFunction, NvFreeFunction, NvReallocateFunction, NULL);
        nv::SetGraphicsDevtoolsAllocator(NvAllocateFunction, NvFreeFunction, NvReallocateFunction, NULL);
        nv::InitializeGraphics( malloc( GraphicsSystemMemorySize ), GraphicsSystemMemorySize );
    }
#endif

    int argc = ::nnt::GetHostArgc();
    char** argv = ::nnt::GetHostArgv();

    NN_LOG("Running nnMain() from testEft_TestMain.cpp\n");
    ::testing::InitGoogleTest(&argc, argv);

    ::testing::TestEventListeners& listeners =
        ::testing::UnitTest::GetInstance()->listeners();
    ::testing::TestEventListener* defaultResultPrinter =
        listeners.Release(listeners.default_result_printer());
#if defined(NN_BUILD_CONFIG_HARDWARE_BDSLIMX6) || defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK1) || defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2) || defined(NN_BUILD_CONFIG_HARDWARE_NX)
    listeners.Append(new ::nnt::teamcity::ServiceMessageLogger());
#endif
    listeners.Append(defaultResultPrinter);

    nnt::eft::ProgramOptions::Analyze(&g_ProgramOptions, argc, argv);
    NN_LOG("ptcl   : \"%s\"\n", g_ProgramOptions.ptclPath.GetString());
    NN_LOG("outDir : \"%s\"\n", g_ProgramOptions.outDir.GetString());
    NN_LOG("width  : %d\n", g_ProgramOptions.screenWidth);
    NN_LOG("height : %d\n", g_ProgramOptions.screenHeight);
    NN_LOG("frame  : ");
    for( int i = 0; i < g_ProgramOptions.numberOfCapturingFrames; i++ )
    {
        NN_LOG("%d%c", (int)g_ProgramOptions.capturingFrames[i], i == g_ProgramOptions.numberOfCapturingFrames - 1 ? '\n' : ',');
    }
    NN_ASSERT(!g_ProgramOptions.outDir.IsEmpty());
    NN_ASSERT(!g_ProgramOptions.ptclPath.IsEmpty());
    NN_ASSERT(g_ProgramOptions.screenWidth != 0);
    NN_ASSERT(g_ProgramOptions.screenHeight != 0);

    int result = RUN_ALL_TESTS();
    ::nnt::Exit(result);
}
