﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/vfx/viewer/vfx_FileSystem.h>

#include "testEft_EffectSystem.h"

namespace nnt{
namespace eft{

//------------------------------------------------------------------------------
// コンストラクタ
//------------------------------------------------------------------------------
EffectSystem::EffectSystem()
    : m_pHeap(NULL)
    , m_pSystem(NULL)
    , m_pPtclResource(NULL)
    , m_ResourceId(0)
{
}

//------------------------------------------------------------------------------
// デストラクタ
//------------------------------------------------------------------------------
EffectSystem::~EffectSystem()
{
}



//------------------------------------------------------------------------------
// 初期化
//------------------------------------------------------------------------------
bool EffectSystem::Initialize( nn::vfx::Heap* pHeap, EftRenderSystem* pRenderSystem )
{
    m_pHeap = pHeap;
    m_pRenderSystem = pRenderSystem;

    // エフェクトの初期化
    nn::vfx::Config  config;
    config.SetEffectHeap( pHeap );
    config.SetEffectDynamicHeap( pHeap );
    config.SetEmitterCount( 1024 );
    config.SetStripeCount( 128 );
    config.SetEmitterSetCount( 256 );
    config.SetDoubleBufferSize( 1024 * 128 );
    config.SetParticleSortBufferCount( 8192 );
    config.SetGpuBufferSize( 1024 * 1024 * 10 );
    config.SetGfxDevice( pRenderSystem->GetDevice() );
    m_pSystem = new nn::vfx::System( config );
    CustomActionTest::Initialize( m_pSystem );

    return true;
}

//------------------------------------------------------------------------------
// 解放処理
//------------------------------------------------------------------------------
void EffectSystem::Finalize()
{
    if ( m_pSystem )
    {
        delete m_pSystem;
    }
    m_pSystem = nullptr;

    if( m_pPtclResource )
    {
        m_pHeap->Free( m_pPtclResource );
        m_pPtclResource = nullptr;
    }
}

//------------------------------------------------------------------------------
// エフェクトリソースの登録
//------------------------------------------------------------------------------
uint32_t EffectSystem::EntryResource( const char* filePath )
{
    size_t fileSize = 0;

    bool resLoad = nn::vfx::viewer::detail::FileSystem::Load( &m_pPtclResource, &fileSize, filePath, m_pHeap );
    NN_SDK_ASSERT( resLoad == true );
    NN_UNUSED( resLoad );

    m_pSystem->EntryResource( m_pHeap, m_pPtclResource, m_ResourceId, true );

    return m_ResourceId;
}

//------------------------------------------------------------------------------
// エフェクトリソースの破棄
//------------------------------------------------------------------------------
void EffectSystem::ClearResource()
{
    m_pSystem->ClearResource( m_pHeap, 0 );
    m_pHeap->Free( m_pPtclResource );
    m_pPtclResource = nullptr;
}

//------------------------------------------------------------------------------
// 指定リソースIDのエミッタセット数の取得
//------------------------------------------------------------------------------
uint32_t EffectSystem::GetEmitterSetNum( uint32_t resId )
{
    if (m_pSystem == NULL)
    {
        return 0;
    }

    return m_pSystem->GetResource( resId )->GetEmitterSetCount();
}

//------------------------------------------------------------------------------
// 指定IDのエミッタセットを生成
//------------------------------------------------------------------------------
void EffectSystem::CreateEmitterSet( uint32_t emitterSetId, uint32_t resourceId, uint32_t groupId )
{
    m_EsetHandle.Invalidate();
    m_pSystem->CreateEmitterSetId( &m_EsetHandle, emitterSetId, resourceId, groupId );
    nn::vfx::detail::OutputLog( "EmitterSet Name : %s\n", m_EsetHandle.GetEmitterSet()->GetName() );
}

//------------------------------------------------------------------------------
// エミッタセットの破棄
//------------------------------------------------------------------------------
void EffectSystem::DestroyEmitterSet()
{
    if (m_EsetHandle.IsValid() == true)
    {
        m_pSystem->KillEmitterSet( m_EsetHandle.GetEmitterSet() );
    }
}

//------------------------------------------------------------------------------
// リソースIDの取得
//------------------------------------------------------------------------------
uint32_t EffectSystem::GetResourceId()
{
    return m_ResourceId;
}

//---------------------------------------------------------------------------
//  定期処理
//---------------------------------------------------------------------------
void EffectSystem::Calc()
{
    // エフェクトランタイムの計算処理
    m_pSystem->BeginFrame();
    m_pSystem->Calculate( 0, 1.0f, true );
}

//---------------------------------------------------------------------------
//  描画処理
//---------------------------------------------------------------------------
void EffectSystem::DrawEffectSystem( const nn::util::Matrix4x3fType& viewMatrix, const nn::util::Matrix4x4fType& projMatrix, const nn::util::Vector3fType& eyePos, float screenNear, float screenFar )
{
    int processingIndex = 0;
    nn::vfx::ViewParam view;
    view.Set( viewMatrix, projMatrix, eyePos, screenNear, screenFar );
    m_pSystem->SwapBuffer();
    m_pSystem->SetViewParam(processingIndex, &view);

    m_pSystem->SetFrameBufferTexture(processingIndex, *m_pRenderSystem->GetColorCopyBufferDescSlot());
    m_pSystem->SetDepthBufferTexture(processingIndex, *m_pRenderSystem->GetDepthStencilCopyBufferDescSlot());

    // 不透明
    m_pSystem->Draw(0, m_pRenderSystem->GetCommandBuffer(), 0, nn::vfx::DrawPathFlag_1, true, true, NULL);

    // カラーバッファをコピー
    m_pRenderSystem->GetCommandBuffer()->FlushMemory( nn::gfx::GpuAccess_ColorBuffer );
    m_pRenderSystem->CopyTextureBuffer( m_pRenderSystem->GetColorCopyBuffer(), m_pRenderSystem->GetColorBuffer() );

    // 深度ステンシルバッファをコピー
    m_pRenderSystem->GetCommandBuffer()->FlushMemory( nn::gfx::GpuAccess_DepthStencil );
    m_pRenderSystem->CopyTextureBuffer( m_pRenderSystem->GetDepthStencilCopyBuffer(), m_pRenderSystem->GetDepthStencilBuffer() );

    // デフォルト
    m_pSystem->Draw(0, m_pRenderSystem->GetCommandBuffer(), 0, nn::vfx::DrawPathFlag_0, true, true, NULL);
}

}
}
