﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/eft/eft2_System.h>
#include <nw/eft/eftvw2_FileSystem.h>
#include <nnt.h>
#include <filesystem>

#include "testEft_FileManager.h"
#include "testEft_PngFile.h"
#include "ScreenCapture.h"
#include "util/PathUtility.h"
#include "nwDemo/System.h"


#if EFT_IS_CAFE
enum{
    MaxPathLength = 256,
};

    #define sprintf_s sprintf
#else
enum{
    MaxPathLength = _MAX_PATH,
};
#endif

//---------------------------
//  定数
//---------------------------

static const int ScreenWidth  = 640;
static const int ScreenHeight = 480;

#if EFT_IS_WIN
    static const char* PLATFORM_NAME = "Win32";   // プラットフォーム名
#endif
#if EFT_IS_CAFE
    static const char* PLATFORM_NAME = "Cafe";  // プラットフォーム名
#endif


//---------------------------
//  変数
//---------------------------
static const int                        MaxCountOfCapturingFrames = 10;             // 描画するフレーム配列の最大値
static char                             g_OutDir[MaxPathLength]      = "";              // 出力先ディレクトリ
static char                             g_PtclPath[MaxPathLength]    = "";              // エフェクトリソースパス
static float                            g_DrawFrame[MaxCountOfCapturingFrames];     // 描画するフレーム配列
static int                              g_DrawFrameCount = 0;                       // 描画するフレーム数
static nw::ut::MemoryAllocator          g_NwAllocator;                              // Allocator
static nw::eftdemo::System*             g_DemoSystem = NULL;                        // Demo System
static nw::dev::PrimitiveRenderer*      g_PrimitiveRenderer = NULL;                 // PrimitiveRenderer
static nw::eft2::System*                g_EffectSystem = NULL;                      // Effect System
static TestHeap                         g_TestHeap;                                 // テスト用ヒープ
static nw::eftdemo::FrameBuffer         g_FrameBuffer64;                            // デフォルトフレームバッファ
static ScreenCapture                    g_ScreenCapture;                            // スクリーンキャプチャ
static nw::math::VEC3                   g_CameraPosition;                           // カメラ位置
static nw::math::VEC3                   g_CameraLookAt;                             // カメラ視点
static nw::math::Matrix34               g_View;                                     // モデルビュー
static nw::math::MTX44                  g_ProjctionMatrix;                          // プロジェクション
static nw::eft2::Handle                 g_EsetHandle;                               // エフェクト操作ハンドル
static void*                            g_PtclResource = NULL;                      // PTCLリソース
static u32                              g_EsetNum = 0;                              // エミッタセット数

//---------------------------------------------------
//  カスタムアクション
//---------------------------------------------------
class CustomActionTest
{
public:
    //------------------------------------------------------------------------------
    //  初期化
    //------------------------------------------------------------------------------
    static void Initialize( nw::eft2::System* system )
    {
        // エミッタビルボード アクション
        nw::eft2::CallbackSet action1CbSet;
        action1CbSet.emitterMtxSet = _EmitterMatrixSetCallback;
        system->SetCallback( nw::eft2::EFT_CUSTOM_ACTION_CALLBACK_ID_1, action1CbSet );

        // パーティクル位置操作 アクション
        nw::eft2::CallbackSet action2CbSet;
        action2CbSet.particleEmit = _ParticleEmitCallback;
        system->SetCallback( nw::eft2::EFT_CUSTOM_ACTION_CALLBACK_ID_2, action2CbSet );
    }

    //------------------------------------------------------------------------------
    //  描画設定後コールバック
    //------------------------------------------------------------------------------
    static void _EmitterMatrixSetCallback( nw::eft2::EmitterMatrixSetArg& arg )
    {
        arg.SetEmitterBillboardMatrix( g_View );
    }

    //------------------------------------------------------------------------------
    //  パーティクル生成後コールバック
    //------------------------------------------------------------------------------
    static bool _ParticleEmitCallback( nw::eft2::ParticleEmitArg& arg )
    {
        nw::eft2::ResCustomActionData* actionParam = reinterpret_cast<nw::eft2::ResCustomActionData*>(arg.emitter->GetCustomActionParameter());
        arg.SetWorldPos(
            static_cast<float>(actionParam->iValue[0]),
            static_cast<float>(actionParam->iValue[1]),
            static_cast<float>(actionParam->iValue[2]) );
        return true;
    }
};

//---------------------------------------------------------------------------
// Usage
//---------------------------------------------------------------------------
static void PrintUsage()
{
    NW_LOG("Usage: testEft_ScreenCapture.exe [options]\n");
    NW_LOG("   -outDir   output directory\n");
    NW_LOG("   -ptcl     effect resource filepath\n");
    NW_LOG("   -frame    draw frames from comma-separated values \n");
}

//---------------------------------------------------------------------------
//
//---------------------------------------------------------------------------
static void AnalyzeCommandLineOption(int argc, char **argv)
{
    // 引数解析
    for (int i = 1; i < argc; i++)
    {
        if (strcmp(argv[i], "-outDir") == 0)
        {
            if (i + 1 < argc)
            {
                sprintf(g_OutDir, "%s", PathUtility::GetAbsolutePathFromCurrentDirectory(argv[i + 1]).c_str());
                i++;
            }
            continue;
        }
        // エフェクトリソース
        if (strcmp(argv[i], "-ptcl") == 0)
        {
            if (i + 1 < argc)
            {
                sprintf(g_PtclPath, "%s", PathUtility::GetFilePathFromCurrentDirectory(argv[i + 1]).c_str());
                i++;
            }
            continue;
        }
        // 描画するフレーム指定
        if (strcmp(argv[i], "-frame") == 0)
        {
            if (i + 1 < argc)
            {
                char* frame;
                char* pFrameStr = const_cast<char*>(argv[i + 1]);
                frame = strtok(pFrameStr, ",");
                while (frame != NULL)
                {
                    if (g_DrawFrameCount >= MaxCountOfCapturingFrames)
                    {
                        break;
                    }
                    g_DrawFrame[g_DrawFrameCount] = static_cast<float>(atoi(frame));
                    g_DrawFrameCount++;
                    frame = strtok(NULL, ",");
                }
                i++;
            }
            continue;
        }
    }
}

//------------------------------------------------------------------------------
//  初期化処理
//------------------------------------------------------------------------------
void Initialize()
{
    // ファイルシステムの初期化
    nw::dev::FileDeviceManager* fileSystem = nw::dev::FileDeviceManager::GetInstance();
    nw::dev::FileDeviceManager::CreateArg fileDeviceArg;
    fileDeviceArg.allocator = &g_NwAllocator;
    fileSystem->Initialize( fileDeviceArg );

    // デモシステムの作成
    nw::eftdemo::System::CreateArg    arg;

    arg.allocator   = &g_NwAllocator;
    arg.width       = ScreenWidth;
    arg.height      = ScreenHeight;
    arg.drawMeter   = true;
    arg.fontPath    = "common/fonts/nintendo_NTLG-DB_002.bffnt";
    arg.clearColor  = 0x333333FF;
    arg.waitVBlank  = 0;

#if defined(NW_PLATFORM_WIN32)
    arg.waitVBlank  = 0;
#endif

#if defined(NW_PLATFORM_CAFE)
    arg.waitVBlank  = 1;
    arg.fontShaderPath = "common/shaders/font_BuildinShader.gsh";
#endif

    nw::eftdemo::System::LoadConfigFile(&g_NwAllocator, &arg);

    g_DemoSystem = new nw::eftdemo::System( arg );

    // デモシステムの初期化
    g_DemoSystem->Initialize();

    // 入力インターフェースの初期化
    g_DemoSystem->InitializeInputInterface();

    // グラフィックシステムの初期化
    g_DemoSystem->InitializeGraphicsSystem();

    nw::gfnd::Graphics::GetInstance()->LockDrawContext();
    {
        // PrimitiveRenderer の初期化
        g_PrimitiveRenderer = nw::dev::PrimitiveRenderer::GetInstance();

#if defined(NW_PLATFORM_WIN32)
        g_PrimitiveRenderer->Initialize( &g_NwAllocator );
#endif
#if defined(NW_PLATFORM_CAFE)
        // PrimitiveRenderer で用いるシェーダーバイナリへのパスを指定する。
        nw::dev::FileDeviceManager* fileSystem = nw::dev::FileDeviceManager::GetInstance();
        nw::dev::FileDevice::LoadArg loadArg;
        loadArg.path = "common/shaders/dev_PrimitiveRenderer.gsh";
        loadArg.allocator = &gNwAllocator;
        u8* binary = fileSystem->Load( loadArg );
        gPrimitiveRenderer->InitializeFromBinary( &gNwAllocator, binary, loadArg.readSize );
        fileSystem->Unload( loadArg, binary );
#endif
    }
    nw::gfnd::Graphics::GetInstance()->UnlockDrawContext();
}

//---------------------------------------------------------------------------
// エフェクトの初期化
//---------------------------------------------------------------------------
static void InitializeEffectSystem( nw::eft2::Heap* heap )
{
    nw::eft2::Config  config;
    config.SetEffectHeap( heap );
    config.SetEffectDynamicHeap( heap );
    g_EffectSystem = new nw::eft2::System( config );
    CustomActionTest::Initialize( g_EffectSystem );
}

//---------------------------------------------------------------------------
// エフェクトリソースの登録
//---------------------------------------------------------------------------
static u32 EntryResource( nw::eft2::Heap* heap )
{
    u32 resId = 0;
    u32 fileSize = 0;

    bool resLoad = nw::eftvw2::FileSystem::Load( heap, g_PtclPath, &g_PtclResource, &fileSize );
    EFT_ASSERT( resLoad == true );

    g_EffectSystem->EntryResource( heap, g_PtclResource, resId );

    return g_EffectSystem->GetResource( resId )->GetEmitterSetNum();
}

//---------------------------------------------------------------------------
//  エフェクトリソースの破棄。
//---------------------------------------------------------------------------
void ClearResrouce( nw::eft2::Heap* heap )
{
    g_EffectSystem->ClearResource( heap, 0 );
    heap->Free( g_PtclResource );
}

//---------------------------------------------------------------------------
//  エフェクトリソースの登録。
//---------------------------------------------------------------------------
void CreateEmitterSet( nw::eft2::Handle* handle, u32 res, u32 idx )
{
    handle->Invalidate();
    g_EffectSystem->CreateEmitterSetID( handle, idx, res, 0 );
}

//---------------------------------------------------------------------------
//  エフェクトリソースの破棄。
//---------------------------------------------------------------------------
void DestroyEmitterSet( nw::eft2::Handle* handle )
{
    g_EffectSystem->KillEmitterSet( handle->GetEmitterSet() );
}

//---------------------------------------------------------------------------
//  エフェクトの定期処理。
//---------------------------------------------------------------------------
void CalcEffectSystem()
{
    // エフェクトランタイムの計算処理
    g_EffectSystem->BeginFrame();
    g_EffectSystem->Calc( 0, 1.0f, true );
}

//---------------------------------------------------------------------------
//  エフェクトの描画処理。
//---------------------------------------------------------------------------
void DrawEffectSystem( nw::math::Matrix34* viewMatrix,
                      nw::math::Matrix44* projMatrix,
                      nw::math::VEC3*     eyePos,
                      f32                 screenNear,
                      f32                 screenFar )
{
    #define DRAW_PATH_OPAQU_BUFFER              ( nw::eft2::EFT_DRAW_PATH_FLAG_3 )
    #define DRAW_PATH_PRE_COPY_COLOR_BUFFER     ( nw::eft2::EFT_DRAW_PATH_FLAG_2 )
    #define DRAW_PATH_DEFAULT                   ( nw::eft2::EFT_DRAW_PATH_FLAG_0 )
    #define DRAW_PATH_REDUCTION_BUFFER          ( nw::eft2::EFT_DRAW_PATH_FLAG_1 )

    nw::eft2::ViewParam view;
    view.Set( viewMatrix, projMatrix, eyePos, screenNear, screenFar );
    g_EffectSystem->SwapBuffer();
    g_EffectSystem->SetViewParam( &view );
    g_EffectSystem->Draw( 0, DRAW_PATH_OPAQU_BUFFER );
    g_EffectSystem->Draw( 0, DRAW_PATH_PRE_COPY_COLOR_BUFFER );
    g_EffectSystem->Draw( 0, DRAW_PATH_DEFAULT );
    g_EffectSystem->Draw( 0, DRAW_PATH_REDUCTION_BUFFER );
}

//---------------------------------------------------------------------------
//  スクリーンキャプチャ。
//---------------------------------------------------------------------------
void CaptureScreen( u32 resID, u32 emitterSetID, u32 frame )
{
    // キャプチャ画像のファイル名を設定
    nw::eft2::Resource* resource = g_EffectSystem->GetResource( resID );

    char fileName[MaxPathLength];
    sprintf_s( fileName, "%s\\%s_%03d.png", g_OutDir, resource->GetEmitterSetName( emitterSetID ), frame );

    // キャプチャ実行
    g_ScreenCapture.Capture();

    // キャプチャ画像を取得
    int width         = g_ScreenCapture.GetWidth();
    int height        = g_ScreenCapture.GetHeight();
    const u8* pPixels = g_ScreenCapture.GetPixels();

    // キャプチャ画像を保存
    WritePng( width, height, pPixels, fileName );
}

//---------------------------------------------------------------------------
// 主処理
//---------------------------------------------------------------------------
static void MainLoopFunction()
{
    u32 counter = 0;
    u32 esetIndex = 0;
    nw::gfnd::Graphics* graphics = nw::gfnd::Graphics::GetInstance();
    bool bLoop = true;

    while ( !g_DemoSystem->IsExiting() && bLoop )
    {
        graphics->LockDrawContext();
        nw::eftdemo::FrameBuffer* currentFrameBuffer = &g_FrameBuffer64;
        {
            // 書き込み先フレームバッファの選定
            currentFrameBuffer->Bind();
            currentFrameBuffer->SetClearColor( nw::math::VEC4( 0.0f, 0.0f, 0.0f, 0.0f) );
            currentFrameBuffer->Clear( true, true );

            // エフェクトの計算/描画処理
            CalcEffectSystem();

        #if EFT_IS_CAFE
            GX2SetShaderMode( GX2_SHADER_MODE_UNIFORM_BLOCK );
            GX2Invalidate( static_cast<GX2InvalidateType>( GX2_INVALIDATE_SHADER ), NULL, 0xFFFFFFFF );
        #endif

            DrawEffectSystem( &g_View, &g_ProjctionMatrix, &g_CameraPosition, 0.1f, 10000.f );

        #if EFT_IS_CAFE
            GX2SetShaderMode( GX2_SHADER_MODE_UNIFORM_REGISTER );
            GX2Invalidate( static_cast<GX2InvalidateType>( GX2_INVALIDATE_SHADER ), NULL, 0xFFFFFFFF );
        #endif

        #if EFT_IS_WIN
            //glEnable(  GL_BLEND );
            //glBlendFunc( GL_SRC_ALPHA , GL_ONE );
            //glBlendEquationEXT(GL_FUNC_ADD_EXT);
            //glBlendFunc( GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA );
        #endif

            //if ( gDemoSystem->GetTextWriterInitialized() )
            //{ DebugDraw( gDemoSystem->GetTextWriter() ); }

            // キャプチャ
            if ( counter == 10 ) { CaptureScreen( 0, esetIndex, 10 );  }
            if ( counter == 30 ) { CaptureScreen( 0, esetIndex, 30 ); }
            if ( counter == 60 )
            {
                CaptureScreen( 0, esetIndex, 60 );
                DestroyEmitterSet( &g_EsetHandle );
                esetIndex++;
                if ( esetIndex == g_EsetNum ) bLoop = false;
                CreateEmitterSet( &g_EsetHandle, 0, esetIndex );
                counter = 0;
            }

            // ForceCalc の動作確認を兼ねたキャプチャもよいかも。
        }
        graphics->UnlockDrawContext();

        // バッファのスワップ
        g_DemoSystem->SwapBuffer( currentFrameBuffer );

        //  VBlank 待ち
        g_DemoSystem->WaitForFlip();

        counter++;
    }
}

//------------------------------------------------------------------------------
//  終了処理
//------------------------------------------------------------------------------
void Finalize()
{
    // PrimitiveRenderer終了処理
    g_PrimitiveRenderer->Finalize( &g_NwAllocator );

    // デモシステム終了処理
    g_DemoSystem->FinalizeGraphicsSystem();
    g_DemoSystem->FinalizeInputInterface();
    g_DemoSystem->Finalize();

    // ファイルシステム終了処理
    nw::dev::FileDeviceManager* fileSystem = nw::dev::FileDeviceManager::GetInstance();
    fileSystem->Finalize();
}

//---------------------------------------------------------------------------
// メイン
//---------------------------------------------------------------------------
#if defined(NW_PLATFORM_WIN32)
int PCSDKMain(int argc, char **argv)
#elif defined(NW_PLATFORM_CAFE)
int main(int argc, char **argv)
#endif
{
    if (argc == 1)
    {
        PrintUsage();
        return 0;
    }

    // 引数解析
    ::testing::InitGoogleTest(&argc, argv);
    AnalyzeCommandLineOption(argc, argv);

    std::tr2::sys::create_directories(std::tr2::sys::path(g_OutDir));

    ::testing::TestEventListeners& listeners =
        ::testing::UnitTest::GetInstance()->listeners();
    ::testing::TestEventListener* defaultResultPrinter =
        listeners.Release(listeners.default_result_printer());
#if defined(NN_BUILD_CONFIG_HARDWARE_BDSLIMX6) || defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK1) || defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2) || defined(NN_BUILD_CONFIG_HARDWARE_NX)
    listeners.Append(new ::nnt::teamcity::ServiceMessageLogger());
#endif
    listeners.Append(defaultResultPrinter);

#if ( EFT_IS_WIN && defined(_DEBUG) && _DEBUG )
    _CrtSetDbgFlag(_CRTDBG_LEAK_CHECK_DF | _CRTDBG_ALLOC_MEM_DF);
#endif

    // アロケーターの初期化
    const size_t memSize = 512 * 1024 * 1024;
#if defined(NW_PLATFORM_WIN32)
    void* addr = malloc( memSize );
#else
    void* addr = MEMAllocFromDefaultHeap( memSize );
#endif
    g_NwAllocator.Initialize( addr, memSize );


    // テスト実行
    int result = RUN_ALL_TESTS();


    // アロケータ終了処理
    g_NwAllocator.Finalize();
    free( addr );

    ::nnt::Exit(result);
    return result;
}


TEST(ScreenCaptureDemo, All)
{
    //----------------------------------------
    // フレームワーク初期化処理
    //----------------------------------------
    Initialize();

    g_TestHeap.SetNwAllocator( &g_NwAllocator );

    FileManager::GetInstance().SetHeap( &g_TestHeap );

    //----------------------------------------
    // フレームワーク初期化処理
    //----------------------------------------
    const f32 width  = ScreenWidth;
    const f32 height = ScreenHeight;

    nw::gfnd::Graphics::GetInstance()->LockDrawContext();
    {
        nw::ut::MemoryAllocator* allocator = &g_NwAllocator;

        // 標準フレームバッファ Float
        g_FrameBuffer64.Initialize( allocator, (s32)width, (s32)height, nw::eftdemo::FrameBuffer::FRAMEBUFFER_TYPE_FLOAT16 );

        // スクリーンキャプチャを初期化
        g_ScreenCapture.Initialize( &g_TestHeap, &g_FrameBuffer64 );
    }
    nw::gfnd::Graphics::GetInstance()->UnlockDrawContext();

    //----------------------------------------
    // カメラを初期化
    //----------------------------------------
    g_CameraPosition.Set( 0.0f, 30.0f, 50.0f );
    g_CameraLookAt.Set  ( 0.0f, 00.0f, 0.0f );

    nw::math::VEC3 comUp ( 0.0f, 1.0f, 0.0f );
    nw::math::VEC3 target( 0.0f, 0.0f, 0.0f);

    g_View.Identity();
    g_View.SetLookAt( g_CameraPosition, comUp, target );

    //----------------------------------------
    // プロジェクションの初期化
    //----------------------------------------
    nw::math::MTX44Perspective( &g_ProjctionMatrix, nw::math::F_PI / 3, width / (f32)height, 0.1f, 1000.f);

    //----------------------------------------
    // リソース設定
    //----------------------------------------
    nw::gfnd::Graphics::GetInstance()->LockDrawContext();
    {
        g_TestHeap.SetNwAllocator( &g_NwAllocator );

        // エフェクトシステムの初期化とリソース読込
        InitializeEffectSystem( &g_TestHeap );

        // 乱数を固定する
        nw::eft2::Random::SetGlobalSeed( TRUE, 32 );

        // リソース読み込み
        g_EsetNum = EntryResource( &g_TestHeap );

        // 0番目のエミッタセットを生成
        CreateEmitterSet( &g_EsetHandle, 0, 0 );
    }
    nw::gfnd::Graphics::GetInstance()->UnlockDrawContext();

    MainLoopFunction();

    // スクリーンキャプチャを破棄
    g_ScreenCapture.Release();

    // リソース破棄
    ClearResrouce( &g_TestHeap );

    // フレームワークの終了処理
    Finalize();

    SUCCEED();
}
