﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/vfx/vfx_System.h>

class HandlingTestSet;

//------------------------------------------------------------------------------
//  テスト実装状況
//  o: 実装済
//  +: コマンド追加済み、テスト未実装
//  /: 似たコマンドがあるため実装を省略
//  x: 未実装 (保留)
//------------------------------------------------------------------------------
// o SetAddVel(const nn::util::Vector3fType& velAdd)
// o SetAllDirectionalVel(float velScale)
// o SetAlpha(float alpha)
// o SetCalcEnable(bool flag)
// - SetColor(float r, float g, float b)
// - SetColor(const nn::util::Vector4fType& color)
// o SetColor(const nw::ut::Color4f& color)
// o SetDirectional(const nn::util::Vector3fType& dir)
// o SetDirectionalVel(float velScale)
// o SetDrawPriority(unsigned char priority)
// o SetEmissionIntervalScale(float ratio)
// o SetEmissionParticleScale(nn::util::Float2& ratio)
// - SetEmissionParticleScale(float scale)
// - SetEmissionParticleScale(const nn::util::Float2& scale)
// o SetEmissionRatioScale(float ratio)
// - SetEmissionScale(float scale) = SetEmissionParticleScale
// o SetEmitterColor(const nn::util::Vector4fType& color0,
// - SetEmitterColor0(const nn::util::Vector4fType& color1)
// - SetEmitterColor1(const nn::util::Vector4fType& color0)
// o SetEmitterScale(const nn::util::Vector3fType& scale)
// o SetEmitterVisible(const char* emitterName, bool flag)
// - SetEmitterVolumeScale(const nn::util::Vector3fType& scale) = SetEmitterScale
// x SetFinalizeCallback(EmitterSetFinalizeCallback callback)
// o SetInitRotate(const nn::util::Vector3fType& rot)
// - SetInitRotate(const Vector3i& rot)
// - SetManualParticleEmission()
// o SetManualParticleEmission(bool emissionEnable)
// o SetManualParticleEmissionWithParticleCount(uint32_t particle
// - SetMatrix(const nn::util::Matrix4x3fType& matrixSRT) = SetMtx
// o SetMatrixFast(const nn::util::Matrix4x3fType& matrixSRT)
// - SetMatrixFast(const nn::util::Matrix4x3fType& matrixRT, float
// - SetMatrixFast(const nn::util::Matrix4x3fType& matrixRT,
// o SetMtx(const nn::util::Matrix4x3fType& matrixSRT)
// - SetMtxFast(const nn::util::Matrix4x3fType& matrixSRT)
// - SetMtxFast(const nn::util::Matrix4x3fType& matrixRT, float scale)
// - SetMtxFast(const nn::util::Matrix4x3fType& matrixRT, const nw::
// o SetParticleLifeScale(float ratio)
// o SetParticleScale(float scale)
// - SetParticleScale(float scaleX, float scaleY)
// - SetParticleScale(const nn::util::Float2& scale)
// - SetParticleScale(nn::util::Float2& ratio)
// - SetPos(const nn::util::Vector3fType& pos) = SetMatrix
// - SetPos(const nn::util::Vector3fType& pos, float scale) = SetMatrix
// - SetPos(const nn::util::Vector3fType& pos, const nn::util::Vector3fType = SetMatrix
// o SetRandomVel(float velScale)
// o SetRuntimeUserData(uint32_t data)
// o SetStartFrame(int frame)
// - SetStopCalc(bool isStopCalc) = SetCalcEnable
// - SetStopCalcAndDraw(bool isStopCalcAndDraw) = SetCalcEnable(), SetVisible()
// - SetStopDraw(bool isStopDraw) = SetVisible
// x SetSystem(System* sys)
// x SetViewFlag(DrawViewFlag flag)
// o SetVisible(bool flag)


//------------------------------------------------------------------------------
//! @brief  操作コマンド
//------------------------------------------------------------------------------
class HandlingCommand
{
public:
    //------------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //!
    //! @param[in] type     コマンドタイプ
    //! @param[in] exeFrame 実行フレーム番号
    //------------------------------------------------------------------------------
    NN_IMPLICIT HandlingCommand( int exeFrame );

    //------------------------------------------------------------------------------
    //! @brief  デストラクタです。
    //------------------------------------------------------------------------------
    virtual ~HandlingCommand() {};

    //------------------------------------------------------------------------------
    //! @brief  コマンド開始フレーム番号を取得します。
    //!
    //! @return コマンド開始フレーム番号を返します。
    //------------------------------------------------------------------------------
    virtual int GetBeginFrame() const;

    //------------------------------------------------------------------------------
    //! @brief  コマンド終了フレーム番号を取得します。
    //!
    //! @return コマンド終了フレーム番号を返します。
    //------------------------------------------------------------------------------
    virtual int GetEndFrame() const;

    //------------------------------------------------------------------------------
    //! @brief  コマンドを実行するかどうか取得します。
    //!
    //! @param[in] numFrame フレーム番号
    //! @return コマンドを実行するときtrueを返します。
    //------------------------------------------------------------------------------
    virtual bool DoExecute( int numFrame ) const;

    //------------------------------------------------------------------------------
    //! @brief  コマンドを実行します。
    //!
    //! @param[in] pTest テストセット
    //------------------------------------------------------------------------------
    virtual void Execute( HandlingTestSet* pTest ) = 0;

private:
    int                 m_exeFrame;  //!< 実行フレーム番号
};


//------------------------------------------------------------------------------
//! @brief  エミッタセット作成コマンド
//------------------------------------------------------------------------------
class CreateEsetCommand : public HandlingCommand
{
public:
    //------------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //!
    //! @param[in] exeFrame 実行フレーム番号
    //! @param[in] esetId   エミッタセットID
    //------------------------------------------------------------------------------
    CreateEsetCommand( int exeFrame, uint32_t esetId );

    //------------------------------------------------------------------------------
    //! @brief  コマンドを実行します。
    //!
    //! @param[in] pTest テストセット
    //------------------------------------------------------------------------------
    virtual void Execute( HandlingTestSet* pTest );

private:
    uint32_t m_esetId;  //!< エミッタセットID
};


//------------------------------------------------------------------------------
//! @brief  エミッタセット削除コマンド
//------------------------------------------------------------------------------
class KillEsetCommand : public HandlingCommand
{
public:
    //------------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //!
    //! @param[in] exeFrame    実行フレーム番号
    //! @param[in] pEmitterSet エミッタセット
    //------------------------------------------------------------------------------
    KillEsetCommand( int exeFrame, nn::vfx::EmitterSet* pEmitterSet );

    //------------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //!
    //! @param[in] exeFrame        実行フレーム番号
    //! @param[in] pEmitterSetName エミッタセット名
    //! @param[in] resID           リソースID
    //------------------------------------------------------------------------------
    KillEsetCommand( int exeFrame, const char* pEmitterSetName, uint32_t resId );

    //------------------------------------------------------------------------------
    //! @brief  コマンドを実行します。
    //!
    //! @param[in] pTest テストセット
    //------------------------------------------------------------------------------
    virtual void Execute( HandlingTestSet* pTest );

private:
    nn::vfx::EmitterSet*  m_pEmitterSet;      //!< エミッタセット
    const char*           m_pEmitterSetName;  //!< エミッタセット名
    uint32_t                   m_resId;            //!< リソースID
};


//------------------------------------------------------------------------------
//! @brief  エミッタセット全削除コマンド
//------------------------------------------------------------------------------
class KillAllEsetCommand : public HandlingCommand
{
public:
    //------------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //!
    //! @param[in] exeFrame 実行フレーム番号
    //------------------------------------------------------------------------------
    NN_IMPLICIT KillAllEsetCommand( int exeFrame );

    //------------------------------------------------------------------------------
    //! @brief  コマンドを実行します。
    //!
    //! @param[in] pTest テストセット
    //------------------------------------------------------------------------------
    virtual void Execute( HandlingTestSet* pTest );
};


//------------------------------------------------------------------------------
//! @brief  フェードコマンド
//------------------------------------------------------------------------------
class FadeCommand : public HandlingCommand
{
public:
    //------------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //!
    //! @param[in] exeFrame 実行フレーム番号
    //------------------------------------------------------------------------------
    NN_IMPLICIT FadeCommand( int exeFrame );

    //------------------------------------------------------------------------------
    //! @brief  コマンドを実行します。
    //!
    //! @param[in] pTest テストセット
    //------------------------------------------------------------------------------
    virtual void Execute( HandlingTestSet* pTest );
};


//------------------------------------------------------------------------------
//! @brief  速度への加算値設定コマンド
//------------------------------------------------------------------------------
class SetAddVelCommand : public HandlingCommand
{
public:
    //------------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //!
    //! @param[in] exeFrame 実行フレーム番号
    //! @param[in] velocity 速度への加算値
    //------------------------------------------------------------------------------
    SetAddVelCommand( int exeFrame, const nn::util::Vector3fType& velocity );

    //------------------------------------------------------------------------------
    //! @brief  コマンドを実行します。
    //!
    //! @param[in] pTest テストセット
    //------------------------------------------------------------------------------
    virtual void Execute( HandlingTestSet* pTest );

private:
    nn::util::Vector3fType m_velocity;  //!< 速度への加算値
};


//------------------------------------------------------------------------------
//! @brief  全方向速度スケール設定コマンド
//------------------------------------------------------------------------------
class SetAllDirectionalVelCommand : public HandlingCommand
{
public:
    //------------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //!
    //! @param[in] exeFrame 実行フレーム番号
    //! @param[in] velocity 全方向速度スケール
    //------------------------------------------------------------------------------
    SetAllDirectionalVelCommand( int exeFrame, float scale );

    //------------------------------------------------------------------------------
    //! @brief  コマンドを実行します。
    //!
    //! @param[in] pTest テストセット
    //------------------------------------------------------------------------------
    virtual void Execute( HandlingTestSet* pTest );

private:
    float m_scale;  //!< 全方向速度スケール
};


//------------------------------------------------------------------------------
//! @brief  描画アルファ設定コマンド
//------------------------------------------------------------------------------
class SetAlphaCommand : public HandlingCommand
{
public:
    //------------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //!
    //! @param[in] exeFrame 実行フレーム番号
    //! @param[in] alpha    描画アルファ
    //------------------------------------------------------------------------------
    SetAlphaCommand( int exeFrame, float alpha );

    //------------------------------------------------------------------------------
    //! @brief  コマンドを実行します。
    //!
    //! @param[in] pTest テストセット
    //------------------------------------------------------------------------------
    virtual void Execute( HandlingTestSet* pTest );

private:
    float m_alpha;  //!< 描画アルファ
};


//------------------------------------------------------------------------------
//! @brief  計算処理の有効/無効設定コマンド
//------------------------------------------------------------------------------
class SetCalcEnableCommand : public HandlingCommand
{
public:
    //------------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //!
    //! @param[in] exeFrame 実行フレーム番号
    //! @param[in] enabled  計算処理の有効/無効
    //------------------------------------------------------------------------------
    SetCalcEnableCommand( int exeFrame, bool enabled );

    //------------------------------------------------------------------------------
    //! @brief  コマンドを実行します。
    //!
    //! @param[in] pTest テストセット
    //------------------------------------------------------------------------------
    virtual void Execute( HandlingTestSet* pTest );

private:
    bool m_enabled;  //!< 計算処理の有効/無効
};


//------------------------------------------------------------------------------
//! @brief  カラー設定コマンド
//------------------------------------------------------------------------------
class SetColorCommand : public HandlingCommand
{
public:
    //------------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //!
    //! @param[in] exeFrame 実行フレーム番号
    //! @param[in] color    カラー
    //------------------------------------------------------------------------------
    SetColorCommand( int exeFrame, const nn::util::Vector4fType& color );

    //------------------------------------------------------------------------------
    //! @brief  コマンドを実行します。
    //!
    //! @param[in] pTest テストセット
    //------------------------------------------------------------------------------
    virtual void Execute( HandlingTestSet* pTest );

private:
    nn::util::Vector4fType m_color;  //!< カラー
};


//------------------------------------------------------------------------------
//! @brief  指定方向設定コマンド
//------------------------------------------------------------------------------
class SetDirectionalCommand : public HandlingCommand
{
public:
    //------------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //!
    //! @param[in] exeFrame  実行フレーム番号
    //! @param[in] direction 指定方向
    //------------------------------------------------------------------------------
    SetDirectionalCommand( int exeFrame, const nn::util::Vector3fType& direction );

    //------------------------------------------------------------------------------
    //! @brief  コマンドを実行します。
    //!
    //! @param[in] pTest テストセット
    //------------------------------------------------------------------------------
    virtual void Execute( HandlingTestSet* pTest );

private:
    nn::util::Vector3fType m_direction;  //!< 指定方向
};


//------------------------------------------------------------------------------
//! @brief  指定方向速度スケール設定コマンド
//------------------------------------------------------------------------------
class SetDirectionalVelCommand : public HandlingCommand
{
public:
    //------------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //!
    //! @param[in] exeFrame 実行フレーム番号
    //! @param[in] scale    指定方向速度スケール
    //------------------------------------------------------------------------------
    SetDirectionalVelCommand( int exeFrame, float scale );

    //------------------------------------------------------------------------------
    //! @brief  コマンドを実行します。
    //!
    //! @param[in] pTest テストセット
    //------------------------------------------------------------------------------
    virtual void Execute( HandlingTestSet* pTest );

private:
    float m_scale;  //!< 指定方向速度スケール
};


//------------------------------------------------------------------------------
//! @brief  描画優先度設定コマンド
//------------------------------------------------------------------------------
class SetDrawPriorityCommand : public HandlingCommand
{
public:
    //------------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //!
    //! @param[in] exeFrame 実行フレーム番号
    //! @param[in] priority 描画優先度
    //------------------------------------------------------------------------------
    SetDrawPriorityCommand( int exeFrame, unsigned char priority );

    //------------------------------------------------------------------------------
    //! @brief  コマンドを実行します。
    //!
    //! @param[in] pTest テストセット
    //------------------------------------------------------------------------------
    virtual void Execute( HandlingTestSet* pTest );

private:
    unsigned char m_priority;  //!< 描画優先度
};


//------------------------------------------------------------------------------
//! @brief  放出間隔スケール設定コマンド
//------------------------------------------------------------------------------
class SetEmissionIntervalScaleCommand : public HandlingCommand
{
public:
    //------------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //!
    //! @param[in] exeFrame 実行フレーム番号
    //! @param[in] scale    寿命スケール
    //------------------------------------------------------------------------------
    SetEmissionIntervalScaleCommand( int exeFrame, float scale );

    //------------------------------------------------------------------------------
    //! @brief  コマンドを実行します。
    //!
    //! @param[in] pTest テストセット
    //------------------------------------------------------------------------------
    virtual void Execute( HandlingTestSet* pTest );

private:
    float m_scale;  //!< スケール
};


//------------------------------------------------------------------------------
//! @brief  放出スケール設定コマンド
//------------------------------------------------------------------------------
class SetEmissionParticleScaleCommand : public HandlingCommand
{
public:
    //------------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //!
    //! @param[in] exeFrame 実行フレーム番号
    //! @param[in] scale    放出スケール
    //------------------------------------------------------------------------------
    SetEmissionParticleScaleCommand( int exeFrame, const nn::util::Float2& scale );

    //------------------------------------------------------------------------------
    //! @brief  コマンドを実行します。
    //!
    //! @param[in] pTest テストセット
    //------------------------------------------------------------------------------
    virtual void Execute( HandlingTestSet* pTest );

private:
    nn::util::Float2 m_scale;  //!< 放出スケール
};


//------------------------------------------------------------------------------
//! @brief  放出レートスケール設定コマンド
//------------------------------------------------------------------------------
class SetEmissionRatioScaleCommand : public HandlingCommand
{
public:
    //------------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //!
    //! @param[in] exeFrame 実行フレーム番号
    //! @param[in] scale    寿命スケール
    //------------------------------------------------------------------------------
    SetEmissionRatioScaleCommand( int exeFrame, float scale );

    //------------------------------------------------------------------------------
    //! @brief  コマンドを実行します。
    //!
    //! @param[in] pTest テストセット
    //------------------------------------------------------------------------------
    virtual void Execute( HandlingTestSet* pTest );

private:
    float m_scale;  //!< スケール
};


//------------------------------------------------------------------------------
//! @brief  エミッタ乗算カラー設定コマンド
//------------------------------------------------------------------------------
class SetEmitterColorCommand : public HandlingCommand
{
public:
    //------------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //!
    //! @param[in] exeFrame 実行フレーム番号
    //! @param[in] color0   エミッタ乗算カラー0
    //! @param[in] color1   エミッタ乗算カラー1
    //------------------------------------------------------------------------------
    SetEmitterColorCommand( int exeFrame, const nn::util::Vector4fType& color0, const nn::util::Vector4fType& color1 );

    //------------------------------------------------------------------------------
    //! @brief  コマンドを実行します。
    //!
    //! @param[in] pTest テストセット
    //------------------------------------------------------------------------------
    virtual void Execute( HandlingTestSet* pTest );

private:
    nn::util::Vector4fType m_color0;  //!< エミッタ乗算カラー0
    nn::util::Vector4fType m_color1;  //!< エミッタ乗算カラー1
};


//------------------------------------------------------------------------------
//! @brief  エミッタスケール設定コマンド
//------------------------------------------------------------------------------
class SetEmitterScaleCommand : public HandlingCommand
{
public:
    //------------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //!
    //! @param[in] exeFrame 実行フレーム番号
    //! @param[in] scale    エミッタスケール
    //------------------------------------------------------------------------------
    SetEmitterScaleCommand( int exeFrame, const nn::util::Vector3fType& scale );

    //------------------------------------------------------------------------------
    //! @brief  コマンドを実行します。
    //!
    //! @param[in] pTest テストセット
    //------------------------------------------------------------------------------
    virtual void Execute( HandlingTestSet* pTest );

private:
    nn::util::Vector3fType m_scale;  //!< エミッタスケール
};


//------------------------------------------------------------------------------
//! @brief  エミッタ描画処理の有効/無効設定コマンド
//------------------------------------------------------------------------------
class SetEmitterVisibleCommand : public HandlingCommand
{
public:
    //------------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //!
    //! @param[in] exeFrame 実行フレーム番号
    //! @param[in] visible  エミッタ描画処理の有効/無効
    //------------------------------------------------------------------------------
    SetEmitterVisibleCommand( int exeFrame, uint32_t emitterIndex, bool visible );

    //------------------------------------------------------------------------------
    //! @brief  コマンドを実行します。
    //!
    //! @param[in] pTest テストセット
    //------------------------------------------------------------------------------
    virtual void Execute( HandlingTestSet* pTest );

private:
    uint32_t  m_emitterIndex;  //!< エミッタインデックス
    bool m_visible;       //!< エミッタ描画処理の有効/無効
};


//------------------------------------------------------------------------------
//! @brief  初期角度設定コマンド
//------------------------------------------------------------------------------
class SetInitRotateCommand : public HandlingCommand
{
public:
    //------------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //!
    //! @param[in] exeFrame 実行フレーム番号
    //! @param[in] rotate   初期角度
    //------------------------------------------------------------------------------
    SetInitRotateCommand( int exeFrame, const nn::util::Vector3fType& rotate );

    //------------------------------------------------------------------------------
    //! @brief  コマンドを実行します。
    //!
    //! @param[in] pTest テストセット
    //------------------------------------------------------------------------------
    virtual void Execute( HandlingTestSet* pTest );

private:
    nn::util::Vector3fType m_rotate;  //!< 初期角度
};


//------------------------------------------------------------------------------
//! @brief  行列設定コマンド
//------------------------------------------------------------------------------
class SetMatrixCommand : public HandlingCommand
{
public:
    //------------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //!
    //! @param[in] exeFrame 実行フレーム番号
    //! @param[in] matrix   行列
    //------------------------------------------------------------------------------
    SetMatrixCommand( int exeFrame, const nn::util::Matrix4x3fType& matrix );

    //------------------------------------------------------------------------------
    //! @brief  コマンドを実行します。
    //!
    //! @param[in] pTest テストセット
    //------------------------------------------------------------------------------
    virtual void Execute( HandlingTestSet* pTest );

private:
    nn::util::Matrix4x3fType m_matrix;  //!< 行列
};


//------------------------------------------------------------------------------
//! @brief  スケール抜き行列設定コマンド
//------------------------------------------------------------------------------
class SetMatrixFastCommand : public HandlingCommand
{
public:
    //------------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //!
    //! @param[in] exeFrame 実行フレーム番号
    //! @param[in] matrix   行列
    //! @param[in] scale    スケール
    //------------------------------------------------------------------------------
    SetMatrixFastCommand( int exeFrame, const nn::util::Matrix4x3fType& matrix, const nn::util::Vector3fType& scale );

    //------------------------------------------------------------------------------
    //! @brief  コマンドを実行します。
    //!
    //! @param[in] pTest テストセット
    //------------------------------------------------------------------------------
    virtual void Execute( HandlingTestSet* pTest );

private:
    nn::util::Matrix4x3fType m_matrix;  //!< 行列
    nn::util::Vector3fType  m_scale;   //!< スケール
};


//------------------------------------------------------------------------------
//! @brief  寿命スケール設定コマンド
//------------------------------------------------------------------------------
class SetParticleLifeScaleCommand : public HandlingCommand
{
public:
    //------------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //!
    //! @param[in] exeFrame 実行フレーム番号
    //! @param[in] scale    寿命スケール
    //------------------------------------------------------------------------------
    SetParticleLifeScaleCommand( int exeFrame, float scale );

    //------------------------------------------------------------------------------
    //! @brief  コマンドを実行します。
    //!
    //! @param[in] pTest テストセット
    //------------------------------------------------------------------------------
    virtual void Execute( HandlingTestSet* pTest );

private:
    float m_scale;  //!< 寿命スケール
};


//------------------------------------------------------------------------------
//! @brief  パーティクルスケール設定コマンド
//------------------------------------------------------------------------------
class SetParticleScaleCommand : public HandlingCommand
{
public:
    //------------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //!
    //! @param[in] exeFrame 実行フレーム番号
    //! @param[in] scale    パーティクルスケール
    //------------------------------------------------------------------------------
    SetParticleScaleCommand( int exeFrame, const nn::util::Float2& scale );

    //------------------------------------------------------------------------------
    //! @brief  コマンドを実行します。
    //!
    //! @param[in] pTest テストセット
    //------------------------------------------------------------------------------
    virtual void Execute( HandlingTestSet* pTest );

private:
    const nn::util::Float2& m_scale;  //!< パーティクルスケール
};


//------------------------------------------------------------------------------
//! @brief  初期速度ランダムスケール設定コマンド
//------------------------------------------------------------------------------
class SetRandomVelCommand : public HandlingCommand
{
public:
    //------------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //!
    //! @param[in] exeFrame 実行フレーム番号
    //! @param[in] scale    初期速度ランダムスケール
    //------------------------------------------------------------------------------
    SetRandomVelCommand( int exeFrame, float scale );

    //------------------------------------------------------------------------------
    //! @brief  コマンドを実行します。
    //!
    //! @param[in] pTest テストセット
    //------------------------------------------------------------------------------
    virtual void Execute( HandlingTestSet* pTest );

private:
    float m_scale;  //!< 初期速度ランダムスケール
};


//------------------------------------------------------------------------------
//! @brief  ユーザーデータ設定コマンド
//------------------------------------------------------------------------------
class SetRuntimeUserDataCommand : public HandlingCommand
{
public:
    //------------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //!
    //! @param[in] exeFrame 実行フレーム番号
    //! @param[in] data     ユーザーデータ
    //------------------------------------------------------------------------------
    SetRuntimeUserDataCommand( int exeFrame, void* data );

    //------------------------------------------------------------------------------
    //! @brief  コマンドを実行します。
    //!
    //! @param[in] pTest テストセット
    //------------------------------------------------------------------------------
    virtual void Execute( HandlingTestSet* pTest );

private:
    void*   m_data;  //!< ユーザーデータ
};


//------------------------------------------------------------------------------
//! @brief  開始フレーム設定コマンド
//------------------------------------------------------------------------------
class SetStartFrameCommand : public HandlingCommand
{
public:
    //------------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //!
    //! @param[in] exeFrame 実行フレーム番号
    //! @param[in] frame    開始フレーム
    //------------------------------------------------------------------------------
    SetStartFrameCommand( int exeFrame, int frame );

    //------------------------------------------------------------------------------
    //! @brief  コマンドを実行します。
    //!
    //! @param[in] pTest テストセット
    //------------------------------------------------------------------------------
    virtual void Execute( HandlingTestSet* pTest );

private:
    int m_frame;  //!< 開始フレーム
};


//------------------------------------------------------------------------------
//! @brief  表示/非表示設定コマンド
//------------------------------------------------------------------------------
class SetVisibleCommand : public HandlingCommand
{
public:
    //------------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //!
    //! @param[in] exeFrame 実行フレーム番号
    //! @param[in] visible  表示/非表示
    //------------------------------------------------------------------------------
    SetVisibleCommand( int exeFrame, bool visible );

    //------------------------------------------------------------------------------
    //! @brief  コマンドを実行します。
    //!
    //! @param[in] pTest テストセット
    //------------------------------------------------------------------------------
    virtual void Execute( HandlingTestSet* pTest );

private:
    bool m_visible;  //!< 表示/非表示
};
