﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

// testEft_EmitterSetHandling.cppからincludeするための書き方になっています
#include <gtest/gtest.h>
#include "testEft_HandlingTestSet.h"
#include "testEft_HandlingCommand.h"
#include "EmitterSetHandlingGfxPtcl.h"

//---------------------------------------------------------------------------
//! @brief  基本テストを行います。
//---------------------------------------------------------------------------
TEST( HandlingTest, SimpleTest )
{
    {
        const char* pTestName = "SimpleTest";

        HandlingTestSet testSet;

        testSet.SetEffectSystem( g_EffectSystem.GetSystem() );
        testSet.SetEsetHandle( g_EffectSystem.GetHandle() );

        int captureFrames[] = { 10, 30, 60, -1 };
        testSet.SetCaptureFrames( captureFrames );

        CreateEsetCommand createEsetCommand( 0, 0 );
        testSet.AddCommand( &createEsetCommand );

        RunLoop(&testSet, pTestName);
    }
}

//---------------------------------------------------------------------------
//! @brief  エミッタセット全削除テストを行います。
//---------------------------------------------------------------------------
TEST( HandlingTest, KillAllEsetTest )
{
    {
        const char* pTestName = "KillAllEsetTest";

        HandlingTestSet testSet;

        testSet.SetEffectSystem( g_EffectSystem.GetSystem() );
        testSet.SetEsetHandle( g_EffectSystem.GetHandle() );

        int captureFrames[] = { 10, 30, 60, -1 };
        testSet.SetCaptureFrames( captureFrames );

        CreateEsetCommand createEsetCommand( 0, 0 );
        testSet.AddCommand( &createEsetCommand );
        KillAllEsetCommand killAllEsetCommand( 40 );
        testSet.AddCommand( &killAllEsetCommand );

        RunLoop(&testSet, pTestName);
    }
}

//---------------------------------------------------------------------------
//! @brief  フェードテストを行います。
//---------------------------------------------------------------------------
TEST( HandlingTest, FadeTest )
{
    {
        const char* pTestName = "FadeTest";

        HandlingTestSet testSet;

        testSet.SetEffectSystem( g_EffectSystem.GetSystem() );
        testSet.SetEsetHandle( g_EffectSystem.GetHandle()  );

        int captureFrames[] = { 10, 30, 60, -1 };
        testSet.SetCaptureFrames( captureFrames );

        CreateEsetCommand createEsetCommand( 0, FadeTest );
        testSet.AddCommand( &createEsetCommand );
        FadeCommand fadeCommand( 25 );
        testSet.AddCommand( &fadeCommand );

        RunLoop(&testSet, pTestName);
    }
}

//---------------------------------------------------------------------------
//! @brief  速度への加算値設定テストを行います。
//---------------------------------------------------------------------------
TEST( HandlingTest, SetAddVelTest )
{
    {
        const char* pTestName = "SetAddVelTest";

        HandlingTestSet testSet;

        testSet.SetEffectSystem( g_EffectSystem.GetSystem() );
        testSet.SetEsetHandle( g_EffectSystem.GetHandle()  );

        int captureFrames[] = { 10, 30, 60, -1 };
        testSet.SetCaptureFrames( captureFrames );

        CreateEsetCommand createEsetCommand( 0, 0 );
        testSet.AddCommand( &createEsetCommand );

        nn::util::Vector3fType vec;
        nn::util::VectorSet( &vec, 1.0f, 2.0f, 3.0f );
        SetAddVelCommand setAddVelCommand( 30, vec );
        testSet.AddCommand( &setAddVelCommand );

        RunLoop(&testSet, pTestName);
    }
}

//---------------------------------------------------------------------------
//! @brief  全方向速度スケール設定テストを行います。
//---------------------------------------------------------------------------
TEST( HandlingTest, SetAllDirectionalVelTest )
{
    {
        const char* pTestName = "SetAllDirectionalVelTest";

        HandlingTestSet testSet;

        testSet.SetEffectSystem( g_EffectSystem.GetSystem() );
        testSet.SetEsetHandle( g_EffectSystem.GetHandle()  );

        int captureFrames[] = { 10, 30, 60, -1 };
        testSet.SetCaptureFrames( captureFrames );

        CreateEsetCommand createEsetCommand( 0, 0 );
        testSet.AddCommand( &createEsetCommand );
        SetAllDirectionalVelCommand setAllDirectionalValCommand( 30,  4.0f );
        testSet.AddCommand( &setAllDirectionalValCommand );

        RunLoop(&testSet, pTestName);
    }
}

//---------------------------------------------------------------------------
//! @brief  描画アルファ設定テストを行います。
//---------------------------------------------------------------------------
TEST( HandlingTest, SetAlphaTest )
{
    {
        const char* pTestName = "SetAlphaTest";

        HandlingTestSet testSet;

        testSet.SetEffectSystem( g_EffectSystem.GetSystem() );
        testSet.SetEsetHandle( g_EffectSystem.GetHandle()  );

        int captureFrames[] = { 10, 30, 60, -1 };
        testSet.SetCaptureFrames( captureFrames );

        CreateEsetCommand createEsetCommand( 0, 0 );
        testSet.AddCommand( &createEsetCommand );
        SetAlphaCommand setAlphaCommand( 30, 0.3f );
        testSet.AddCommand( &setAlphaCommand );

        RunLoop(&testSet, pTestName);
    }
}

//---------------------------------------------------------------------------
//! @brief  計算処理の有効/無効設定テストを行います。
//---------------------------------------------------------------------------
TEST( HandlingTest, SetCalcEnableTest )
{
    {
        const char* pTestName = "SetCalcEnableTest";

        HandlingTestSet testSet;

        testSet.SetEffectSystem( g_EffectSystem.GetSystem() );
        testSet.SetEsetHandle( g_EffectSystem.GetHandle()  );

        int captureFrames[] = { 10, 30, 60, -1 };
        testSet.SetCaptureFrames( captureFrames );

        CreateEsetCommand createEsetCommand( 0, 0 );
        testSet.AddCommand( &createEsetCommand );
        SetCalcEnableCommand setCalcEnableCommand( 30, false );
        testSet.AddCommand( &setCalcEnableCommand );

        RunLoop(&testSet, pTestName);
    }
}

//---------------------------------------------------------------------------
//! @brief  描画カラー設定テストを行います。
//---------------------------------------------------------------------------
TEST( HandlingTest, SetColorTest )
{
    {
        const char* pTestName = "SetColorTest";

        HandlingTestSet testSet;

        testSet.SetEffectSystem( g_EffectSystem.GetSystem() );
        testSet.SetEsetHandle( g_EffectSystem.GetHandle()  );

        int captureFrames[] = { 10, 30, 60, -1 };
        testSet.SetCaptureFrames( captureFrames );

        CreateEsetCommand createEsetCommand( 0, 0 );
        testSet.AddCommand( &createEsetCommand );

        nn::util::Vector4fType vec;
        nn::util::VectorSet( &vec, 0.8f, 0.6f, 0.4f, 1.0f );
        SetColorCommand setColorCommand( 30, vec );
        testSet.AddCommand( &setColorCommand );

        RunLoop(&testSet, pTestName);
    }
}

//---------------------------------------------------------------------------
//! @brief  指定方向設定テストを行います。
//---------------------------------------------------------------------------
TEST( HandlingTest, SetDirectionalTest )
{
    {
        const char* pTestName = "SetDirectionalTest";

        HandlingTestSet testSet;

        testSet.SetEffectSystem( g_EffectSystem.GetSystem() );
        testSet.SetEsetHandle( g_EffectSystem.GetHandle()  );

        int captureFrames[] = { 10, 30, 60, -1 };
        testSet.SetCaptureFrames( captureFrames );

        CreateEsetCommand createEsetCommand( 0, 0 );
        testSet.AddCommand( &createEsetCommand );

        nn::util::Vector3fType vec;
        nn::util::VectorSet( &vec, 2.0f, 1.5f, 0.7f );
        SetDirectionalCommand setDirectionalCommand( 30, vec );
        testSet.AddCommand( &setDirectionalCommand );

        RunLoop(&testSet, pTestName);
    }
}

//---------------------------------------------------------------------------
//! @brief  指定方向速度スケール設定テストを行います。
//---------------------------------------------------------------------------
TEST( HandlingTest, SetDirectionalVelTest )
{
    {
        const char* pTestName = "SetDirectionalVelTest";

        HandlingTestSet testSet;

        testSet.SetEffectSystem( g_EffectSystem.GetSystem() );
        testSet.SetEsetHandle( g_EffectSystem.GetHandle()  );

        int captureFrames[] = { 10, 30, 60, -1 };
        testSet.SetCaptureFrames( captureFrames );

        CreateEsetCommand createEsetCommand( 0, 0 );
        testSet.AddCommand( &createEsetCommand );

        nn::util::Vector3fType vec;
        nn::util::VectorSet( &vec, 2.0f, 1.5f, 0.7f );
        SetDirectionalCommand setDirectionalCommand( 30, vec );
        testSet.AddCommand( &setDirectionalCommand );

        SetDirectionalVelCommand setDirectionalVelCommand( 30, 6.5f );
        testSet.AddCommand( &setDirectionalVelCommand );

        RunLoop(&testSet, pTestName);
    }
}

//---------------------------------------------------------------------------
//! @brief  描画プライオリティ設定テストを行います。
//---------------------------------------------------------------------------
TEST( HandlingTest, SetDrawPriorityTest )
{
    {
        const char* pTestName = "SetDrawPriorityTest";

        HandlingTestSet testSet;

        testSet.SetEffectSystem( g_EffectSystem.GetSystem() );
        testSet.SetEsetHandle( g_EffectSystem.GetHandle()  );

        int captureFrames[] = { 10, 30, 60, -1 };
        testSet.SetCaptureFrames( captureFrames );

        CreateEsetCommand createEsetCommand( 0, 0 );
        testSet.AddCommand( &createEsetCommand );

        SetDrawPriorityCommand setDrawPriorityCommand( 30, 0 );
        testSet.AddCommand( &setDrawPriorityCommand );

        RunLoop(&testSet, pTestName);
    }
}

//---------------------------------------------------------------------------
//! @brief  放出スケール設定テストを行います。
//---------------------------------------------------------------------------
TEST( HandlingTest, SetEmissionScaleTest )
{
    {
        const char* pTestName = "SetEmissionScaleTest";

        HandlingTestSet testSet;

        testSet.SetEffectSystem( g_EffectSystem.GetSystem() );
        testSet.SetEsetHandle( g_EffectSystem.GetHandle()  );

        int captureFrames[] = { 10, 30, 60, -1 };
        testSet.SetCaptureFrames( captureFrames );

        CreateEsetCommand createEsetCommand( 0, 0 );
        testSet.AddCommand( &createEsetCommand );

        nn::util::Float2 f2;
        f2.v[0] = 2.0f;
        f2.v[1] = 1.5f;
        SetEmissionParticleScaleCommand setEmissionParticleScaleCommand ( 30, f2 );
        testSet.AddCommand( &setEmissionParticleScaleCommand );

        RunLoop(&testSet, pTestName);
    }
}

//---------------------------------------------------------------------------
//! @brief  放出間隔スケール設定テストを行います。
//---------------------------------------------------------------------------
TEST( HandlingTest, SetEmissionIntervalScaleTest )
{
    {
        const char* pTestName = "SetEmissionIntervalScaleTest";

        HandlingTestSet testSet;

        testSet.SetEffectSystem( g_EffectSystem.GetSystem() );
        testSet.SetEsetHandle( g_EffectSystem.GetHandle()  );

        int captureFrames[] = { 10, 30, 60, -1 };
        testSet.SetCaptureFrames( captureFrames );

        CreateEsetCommand createEsetCommand( 0, 0 );
        testSet.AddCommand( &createEsetCommand );

        SetEmissionIntervalScaleCommand setEmissionIntervalScaleCommand( 30, 0.6f );
        testSet.AddCommand( &setEmissionIntervalScaleCommand );

        RunLoop(&testSet, pTestName);
    }
}

//---------------------------------------------------------------------------
//! @brief  放出レートスケール設定テストを行います。
//---------------------------------------------------------------------------
TEST( HandlingTest, SetEmissionRatioScaleTest )
{
    {
        const char* pTestName = "SetEmissionRatioScaleTest";

        HandlingTestSet testSet;

        testSet.SetEffectSystem( g_EffectSystem.GetSystem() );
        testSet.SetEsetHandle( g_EffectSystem.GetHandle()  );

        int captureFrames[] = { 10, 30, 60, -1 };
        testSet.SetCaptureFrames( captureFrames );

        CreateEsetCommand createEsetCommand( 0, 0 );
        testSet.AddCommand( &createEsetCommand );

        SetEmissionRatioScaleCommand setEmissionRatioScaleCommand( 30, 0.6f );
        testSet.AddCommand( &setEmissionRatioScaleCommand );

        RunLoop(&testSet, pTestName);
    }
}

//---------------------------------------------------------------------------
//! @brief  エミッタカラー設定テストを行います。
//---------------------------------------------------------------------------
TEST( HandlingTest, SetEmitterColorTest )
{
    {
        const char* pTestName = "SetEmitterColorTest";

        HandlingTestSet testSet;

        testSet.SetEffectSystem( g_EffectSystem.GetSystem() );
        testSet.SetEsetHandle( g_EffectSystem.GetHandle()  );

        int captureFrames[] = { 10, 30, 60, -1 };
        testSet.SetCaptureFrames( captureFrames );

        CreateEsetCommand createEsetCommand( 0, 0 );
        testSet.AddCommand( &createEsetCommand );

        nn::util::Vector4fType vec, vec2;
        nn::util::VectorSet( &vec, 0.2f, 0.4f, 0.8f, 0.5f );
        nn::util::VectorSet( &vec2, 0.9f, 0.5f, 0.1f, 0.3f );
        SetEmitterColorCommand setEmitterColorCommand( 30, vec, vec2 );
        testSet.AddCommand( &setEmitterColorCommand );

        RunLoop(&testSet, pTestName);
    }
}

//---------------------------------------------------------------------------
//! @brief  エミッタスケール設定テストを行います。
//---------------------------------------------------------------------------
TEST( HandlingTest, SetEmitterScaleTest )
{
    {
        const char* pTestName = "SetEmitterScaleTest";

        HandlingTestSet testSet;

        testSet.SetEffectSystem( g_EffectSystem.GetSystem() );
        testSet.SetEsetHandle( g_EffectSystem.GetHandle()  );

        int captureFrames[] = { 10, 30, 60, -1 };
        testSet.SetCaptureFrames( captureFrames );

        CreateEsetCommand createEsetCommand( 0, 0 );
        testSet.AddCommand( &createEsetCommand );

        nn::util::Vector3fType vec;
        nn::util::VectorSet( &vec, 2.0f, 1.5f, 0.8f );
        SetEmitterScaleCommand setEmitterScaleCommand( 30, vec );
        testSet.AddCommand( &setEmitterScaleCommand );

        RunLoop(&testSet, pTestName);
    }
}

//---------------------------------------------------------------------------
//! @brief  エミッタ描画処理の有効/無効設定テストを行います。
//---------------------------------------------------------------------------
TEST( HandlingTest, SetEmitterVisibleTest )
{
    {
        const char* pTestName = "SetEmitterVisibleTest";

        HandlingTestSet testSet;

        testSet.SetEffectSystem( g_EffectSystem.GetSystem() );
        testSet.SetEsetHandle( g_EffectSystem.GetHandle()  );

        int captureFrames[] = { 10, 30, 60, -1 };
        testSet.SetCaptureFrames( captureFrames );

        CreateEsetCommand createEsetCommand( 0, 0 );
        testSet.AddCommand( &createEsetCommand );
        SetEmitterVisibleCommand setEmitterVisibleCommand( 30, 0, false );
        testSet.AddCommand( &setEmitterVisibleCommand );

        RunLoop(&testSet, pTestName);
    }
}

//---------------------------------------------------------------------------
//! @brief  初期角度設定テストを行います。
//---------------------------------------------------------------------------
TEST( HandlingTest, SetInitRotateTest )
{
    {
        const char* pTestName = "SetInitRotateTest";

        HandlingTestSet testSet;

        testSet.SetEffectSystem( g_EffectSystem.GetSystem() );
        testSet.SetEsetHandle( g_EffectSystem.GetHandle()  );

        int captureFrames[] = { 10, 30, 60, -1 };
        testSet.SetCaptureFrames( captureFrames );

        CreateEsetCommand createEsetCommand( 0, 0 );
        testSet.AddCommand( &createEsetCommand );

        nn::util::Vector3fType vec;
        nn::util::VectorSet( &vec, 30.0f, 60.0f, 270.0f );
        SetInitRotateCommand setInitRotateCommand( 30, vec );
        testSet.AddCommand( &setInitRotateCommand );

        RunLoop(&testSet, pTestName);
    }
}


//---------------------------------------------------------------------------
//! @brief  行列設定テストを行います。
//---------------------------------------------------------------------------
TEST( HandlingTest, SetMatrixTest )
{
    {
        const char* pTestName = "SetMatrixTest";

        HandlingTestSet testSet;

        testSet.SetEffectSystem( g_EffectSystem.GetSystem() );
        testSet.SetEsetHandle( g_EffectSystem.GetHandle()  );

        int captureFrames[] = { 10, 30, 60, -1 };
        testSet.SetCaptureFrames( captureFrames );

        CreateEsetCommand createEsetCommand( 0, TrackingTest );
        testSet.AddCommand( &createEsetCommand );

        // NOTE:SetMatrixTest と SetMatrixFastTest は計算結果が同じになる事
        nn::util::Matrix4x3fType matrix;
        nn::util::Vector3fType scale;
        nn::util::Vector3fType rotate;
        nn::util::Vector3fType translate;

        nn::util::MatrixIdentity( &matrix );
        nn::util::VectorSet( &scale, 0.5f, 1.0f, 1.0f );
        nn::util::VectorSet(&rotate,
            nn::util::DegreeToRadian( 0.0f ),
            nn::util::DegreeToRadian( 0.0f ),
            nn::util::DegreeToRadian( 0.0f ));
        nn::util::VectorSet( &translate, 0.0f, 0.0f, 0.0f );
        nn::util::MatrixSetScaleRotateXyz( &matrix, scale, rotate );
        nn::util::MatrixSetAxisW( &matrix, translate );
        SetMatrixCommand setMatrixCommand( 0, matrix );
        testSet.AddCommand( &setMatrixCommand );

        nn::util::MatrixIdentity( &matrix );
        nn::util::VectorSet( &scale, 1.0f, 0.5f, 1.0f );
        nn::util::VectorSet(&rotate,
            nn::util::DegreeToRadian( 45.0f ),
            nn::util::DegreeToRadian( 0.0f ),
            nn::util::DegreeToRadian( 0.0f ));
        nn::util::VectorSet( &translate, 0.0f, 10.0f, 0.0f );
        nn::util::MatrixSetScaleRotateXyz( &matrix, scale, rotate );
        nn::util::MatrixSetAxisW( &matrix, translate );
        SetMatrixCommand setMatrixCommand2( 5, matrix );
        testSet.AddCommand( &setMatrixCommand2 );

        nn::util::MatrixIdentity( &matrix );
        nn::util::VectorSet( &scale, 0.75f, 0.75f, 0.0f );
        nn::util::VectorSet(&rotate,
            nn::util::DegreeToRadian( 0.0f ),
            nn::util::DegreeToRadian( 0.0f ),
            nn::util::DegreeToRadian( 45.0f ));
        nn::util::VectorSet( &translate, 0.0f, -10.0f, 5.0f );
        nn::util::MatrixSetScaleRotateXyz( &matrix, scale, rotate );
        nn::util::MatrixSetAxisW( &matrix, translate );
        SetMatrixCommand setMatrixCommand3( 20, matrix );
        testSet.AddCommand( &setMatrixCommand3 );

        nn::util::MatrixIdentity( &matrix );
        nn::util::VectorSet( &scale, 1.25f, 1.25f, 1.25f );
        nn::util::VectorSet(&rotate,
            nn::util::DegreeToRadian( 0.0f ),
            nn::util::DegreeToRadian( 0.0f ),
            nn::util::DegreeToRadian( 135.0f ));
        nn::util::VectorSet( &translate, 0.0f, 5.0f, 0.0f );
        nn::util::MatrixSetScaleRotateXyz( &matrix, scale, rotate );
        nn::util::MatrixSetAxisW( &matrix, translate );
        SetMatrixCommand setMatrixCommand4( 45, matrix );
        testSet.AddCommand( &setMatrixCommand4 );

        RunLoop(&testSet, pTestName);
    }
}

//---------------------------------------------------------------------------
//! @brief  スケール抜き行列設定テストを行います。
//!
//! 注意）正解画像は SetMatrixTest で生成したものを利用する事
//---------------------------------------------------------------------------
TEST( HandlingTest, SetMatrixFastTest )
{
    {
        const char* pTestName = "SetMatrixFastTest";

        HandlingTestSet testSet;

        testSet.SetEffectSystem( g_EffectSystem.GetSystem() );
        testSet.SetEsetHandle( g_EffectSystem.GetHandle()  );

        int captureFrames[] = { 10, 30, 60, -1 };
        testSet.SetCaptureFrames( captureFrames );

        CreateEsetCommand createEsetCommand( 0, TrackingTest );
        testSet.AddCommand( &createEsetCommand );

        // NOTE:SetMatrixTest と SetMatrixFastTest は計算結果が同じになる事
        nn::util::Matrix4x3fType matrix;
        nn::util::Vector3fType scale;
        nn::util::Vector3fType rotate;
        nn::util::Vector3fType translate;

        nn::util::MatrixIdentity( &matrix );
        nn::util::VectorSet( &scale, 0.5f, 1.0f, 1.0f );
        nn::util::VectorSet(&rotate,
            nn::util::DegreeToRadian( 0.0f ),
            nn::util::DegreeToRadian( 0.0f ),
            nn::util::DegreeToRadian( 0.0f ));
        nn::util::VectorSet( &translate, 0.0f, 0.0f, 0.0f );
        nn::util::MatrixSetRotateXyz( &matrix, rotate );
        nn::util::MatrixSetAxisW( &matrix, translate );
        SetMatrixFastCommand setMatrixFastCommand( 0, matrix, scale );
        testSet.AddCommand( &setMatrixFastCommand );

        nn::util::MatrixIdentity( &matrix );
        nn::util::VectorSet( &scale, 1.0f, 0.5f, 1.0f );
        nn::util::VectorSet(&rotate,
            nn::util::DegreeToRadian( 45.0f ),
            nn::util::DegreeToRadian( 0.0f ),
            nn::util::DegreeToRadian( 0.0f ));
        nn::util::VectorSet( &translate, 0.0f, 10.0f, 0.0f );
        nn::util::MatrixSetRotateXyz( &matrix, rotate );
        nn::util::MatrixSetAxisW( &matrix, translate );
        SetMatrixFastCommand setMatrixFastCommand2( 5, matrix, scale );
        testSet.AddCommand( &setMatrixFastCommand2 );

        nn::util::MatrixIdentity( &matrix );
        nn::util::VectorSet( &scale, 0.75f, 0.75f, 0.0f );
        nn::util::VectorSet(&rotate,
            nn::util::DegreeToRadian( 0.0f ),
            nn::util::DegreeToRadian( 0.0f ),
            nn::util::DegreeToRadian( 45.0f ));
        nn::util::VectorSet( &translate, 0.0f, -10.0f, 5.0f );
        nn::util::MatrixSetRotateXyz( &matrix, rotate );
        nn::util::MatrixSetAxisW( &matrix, translate );
        SetMatrixFastCommand setMatrixFastCommand3( 20, matrix, scale );
        testSet.AddCommand( &setMatrixFastCommand3 );

        nn::util::MatrixIdentity( &matrix );
        nn::util::VectorSet( &scale, 1.25f, 1.25f, 1.25f );
        nn::util::VectorSet(&rotate,
            nn::util::DegreeToRadian( 0.0f ),
            nn::util::DegreeToRadian( 0.0f ),
            nn::util::DegreeToRadian( 135.0f ));
        nn::util::VectorSet( &translate, 0.0f, 5.0f, 0.0f );
        nn::util::MatrixSetRotateXyz( &matrix, rotate );
        nn::util::MatrixSetAxisW( &matrix, translate );
        SetMatrixFastCommand setMatrixFastCommand4( 45, matrix, scale );
        testSet.AddCommand( &setMatrixFastCommand4 );

        RunLoop(&testSet, pTestName);
    }
}

//---------------------------------------------------------------------------
//! @brief  寿命スケール設定テストを行います。
//---------------------------------------------------------------------------
TEST( HandlingTest, SetParticleLifeScaleTest )
{
    {
        const char* pTestName = "SetParticleLifeScaleTest";

        HandlingTestSet testSet;

        testSet.SetEffectSystem( g_EffectSystem.GetSystem() );
        testSet.SetEsetHandle( g_EffectSystem.GetHandle()  );

        int captureFrames[] = { 10, 30, 60, -1 };
        testSet.SetCaptureFrames( captureFrames );

        CreateEsetCommand createEsetCommand( 0, 0 );
        testSet.AddCommand( &createEsetCommand );

        SetParticleLifeScaleCommand setParticleLifeScaleCommand( 30, 0.6f );
        testSet.AddCommand( &setParticleLifeScaleCommand );

        RunLoop(&testSet, pTestName);
    }
}

//---------------------------------------------------------------------------
//! @brief  パーティクルスケール設定テストを行います。
//---------------------------------------------------------------------------
TEST( HandlingTest, SetParticleScaleTest )
{
    {
        const char* pTestName = "SetParticleScaleTest";

        HandlingTestSet testSet;

        testSet.SetEffectSystem( g_EffectSystem.GetSystem() );
        testSet.SetEsetHandle( g_EffectSystem.GetHandle()  );

        int captureFrames[] = { 10, 30, 60, -1 };
        testSet.SetCaptureFrames( captureFrames );

        CreateEsetCommand createEsetCommand( 0, 0 );
        testSet.AddCommand( &createEsetCommand );

        nn::util::Float2 scale;
        scale.v[0] = 2.0f;
        scale.v[1] = 1.5f;
        SetParticleScaleCommand setParticleScaleCommand( 30, scale );
        testSet.AddCommand( &setParticleScaleCommand );

        RunLoop(&testSet, pTestName);
    }
}

//---------------------------------------------------------------------------
//! @brief  初期速度ランダムスケール設定テストを行います。
//---------------------------------------------------------------------------
TEST( HandlingTest, SetRandomVelTest )
{
    {
        const char* pTestName = "SetRandomVelTest";

        HandlingTestSet testSet;

        testSet.SetEffectSystem( g_EffectSystem.GetSystem() );
        testSet.SetEsetHandle( g_EffectSystem.GetHandle()  );

        int captureFrames[] = { 10, 30, 60, -1 };
        testSet.SetCaptureFrames( captureFrames );

        CreateEsetCommand createEsetCommand( 0, 0 );
        testSet.AddCommand( &createEsetCommand );

        SetRandomVelCommand setRandomVelCommand( 30, 3.2f );
        testSet.AddCommand( &setRandomVelCommand );

        RunLoop(&testSet, pTestName);
    }
}

//---------------------------------------------------------------------------
//! @brief  ユーザーデータ設定テストを行います。
//---------------------------------------------------------------------------
TEST( HandlingTest, SetRuntimeUserDataTest )
{
    {
        const char* pTestName = "SetRuntimeUserDataTest";

        HandlingTestSet testSet;

        testSet.SetEffectSystem( g_EffectSystem.GetSystem() );
        testSet.SetEsetHandle( g_EffectSystem.GetHandle()  );

        int captureFrames[] = { 10, 30, 60, -1 };
        testSet.SetCaptureFrames( captureFrames );

        CreateEsetCommand createEsetCommand( 0, 0 );
        testSet.AddCommand( &createEsetCommand );

        SetRuntimeUserDataCommand setRuntimeUserDataCommand( 30, (void*)3939 );
        testSet.AddCommand( &setRuntimeUserDataCommand );

        RunLoop(&testSet, pTestName);
    }
}

//---------------------------------------------------------------------------
//! @brief  開始フレーム設定テストを行います。
//---------------------------------------------------------------------------
TEST( HandlingTest, SetStartFrameTest )
{
    {
        const char* pTestName = "SetStartFrameTest";

        HandlingTestSet testSet;

        testSet.SetEffectSystem( g_EffectSystem.GetSystem() );
        testSet.SetEsetHandle( g_EffectSystem.GetHandle()  );

        int captureFrames[] = { 10, 30, 60, -1 };
        testSet.SetCaptureFrames( captureFrames );

        CreateEsetCommand createEsetCommand( 0, StartFrameTest );
        testSet.AddCommand( &createEsetCommand );

        SetStartFrameCommand setStartFrameCommand( 0, 15 );
        testSet.AddCommand( &setStartFrameCommand );

        RunLoop(&testSet, pTestName);
    }
}

//---------------------------------------------------------------------------
//! @brief  表示/非表示設定テストを行います。
//---------------------------------------------------------------------------
TEST( HandlingTest, SetVisibleTest )
{
    {
        const char* pTestName = "SetVisibleTest";

        HandlingTestSet testSet;

        testSet.SetEffectSystem( g_EffectSystem.GetSystem() );
        testSet.SetEsetHandle( g_EffectSystem.GetHandle() );

        int captureFrames[] = { 10, 30, 60, -1 };
        testSet.SetCaptureFrames( captureFrames );

        CreateEsetCommand createEsetCommand( 0, 0 );
        testSet.AddCommand( &createEsetCommand );

        SetVisibleCommand setVisibleCommand( 30, false );
        testSet.AddCommand( &setVisibleCommand );

        RunLoop(&testSet, pTestName);
    }
}

