﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Windows.h>
#include <vector>
#include <sstream>

#include "PathUtility.h"

#if (defined(EFT_IS_CAFE) && EFT_IS_CAFE)
enum{
    MaxPathLength = 256,
};

    #define sprintf_s sprintf
#else
enum{
    MaxPathLength = 512,
};
#endif

//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
static const char* DirectorySeparatorChar = "\\";
static const std::string CurrentDirectoryString = "." + std::string(DirectorySeparatorChar);
static const std::string ParentDirectoryString = ".." + std::string(DirectorySeparatorChar);
static const std::string RootDirectoryString = std::string(DirectorySeparatorChar);

//------------------------------------------------------------------------------
// 分割
//------------------------------------------------------------------------------
std::vector<std::string> PathUtility::Sprit(const std::string &str, char sep)
{
    std::vector<std::string> v;
    std::stringstream ss(str);
    std::string buffer;
    while (std::getline(ss, buffer, sep))
    {
        v.push_back(buffer);
    }
    return v;
}

//------------------------------------------------------------------------------
// 前後の余分な空白文字を取り除く
//------------------------------------------------------------------------------
std::string PathUtility::Trim(const std::string& string, const char* trimCharacterList = "\t\v\r\n")
{
    std::string result;

    // 左側からトリムする文字以外が見つかる位置を検索します。
    std::string::size_type left = string.find_first_not_of(trimCharacterList);
    if (left != std::string::npos)
    {
        // 左側からトリムする文字以外が見つかった場合は、同じように右側からも検索します。
        std::string::size_type right = string.find_last_not_of(trimCharacterList);

        // 戻り値を決定します。ここでは右側から検索しても、トリムする文字以外が必ず存在するので判定不要です。
        result = string.substr(left, right - left + 1);
    }
    return result;
}

//------------------------------------------------------------------------------
// 相対パスから絶対パスを取得する
//------------------------------------------------------------------------------
std::string PathUtility::GetAbsolutePath(std::string basePath, std::string relativePath)
{
    if (basePath.length() == 0)
    {
        return relativePath;
    }
    if (relativePath.length() == 0)
    {
        return basePath;
    }
    basePath = Trim(basePath, DirectorySeparatorChar);

    if (relativePath.find_first_of(ParentDirectoryString.c_str(), 0, relativePath.length()) == 0)
    {
        //相対パスが"..\"で始まっている時
        //基準パスの最後の"\"から後ろを削除する
        int pos = static_cast<int>(basePath.find_last_of(DirectorySeparatorChar));
        if (pos == static_cast<int>(std::string::npos))
        {
            // "\"..\\\"が多すぎます。
            return relativePath;
        }
        basePath = basePath.erase(pos);

        // 相対パスのはじめの "..\" を削除する
        relativePath = relativePath.substr(ParentDirectoryString.length());
        // あらためて絶対パスを取得する
        return GetAbsolutePath(basePath, relativePath);
    }
    else if (relativePath.find_first_of(CurrentDirectoryString.c_str(), 0, CurrentDirectoryString.length()) == 0)
    {
        // 相対パスが ".\" で始まっているとき
        // 相対パスのはじめの ".\" を削除する
        relativePath = relativePath.substr(CurrentDirectoryString.length());
        // あらためて絶対パスを取得する
        return GetAbsolutePath(basePath, relativePath);
    }
    else if (relativePath.find_first_of(RootDirectoryString.c_str(), 0, RootDirectoryString.length()) == 0)
    {
        // 相対パスが "\" で始まっている時
        // 基準パスのルートパスを取得する
        basePath = std::tr2::sys::path(basePath);
        basePath = Trim(basePath, DirectorySeparatorChar);
        // 相対パスのはじめの "\" を削除する
        relativePath = relativePath.substr(RootDirectoryString.length());
    }
    // パスを連結する
    return basePath + DirectorySeparatorChar + relativePath;
}

//------------------------------------------------------------------------------
// 絶対パスから相対パスを取得します
//------------------------------------------------------------------------------
std::string PathUtility::GetRelativePath(std::string basePath, std::string absolutePath)
{
    if (basePath.length() == 0)
    {
        return absolutePath;
    }
    if (absolutePath.length() == 0)
    {
        return basePath;
    }

    basePath = Trim(basePath, DirectorySeparatorChar);

    // パスを "\" で分割する
    std::vector<std::string> basePathDirs = Sprit(basePath, *DirectorySeparatorChar);
    std::vector<std::string> absolutePathDirs = Sprit(absolutePath, *DirectorySeparatorChar);

    // 基準パスと絶対パスで、先頭から共通する部分を探す
    int commonCount = 0;
    for (int i = 0;
        i < static_cast<int>(basePathDirs.size()) &&
        i < static_cast<int>(absolutePathDirs.size()) &&
        strcmp(basePathDirs[i].c_str(), absolutePathDirs[i].c_str()) == 0; i++)
    {
        // 共通部分の数を覚えておく
        commonCount++;
    }

    // 共通部分が無い時
    if (commonCount == 0)
    {
        return absolutePath;
    }

    // 共通部分以降の基準パスのフォルダの深さを取得する
    int baseOnlyCount = static_cast<int>(basePathDirs.size() - commonCount);
    // その数だけ "..\" をつける
    std::string buf = "";
    for (int i=0; i< baseOnlyCount; i++)
    {
        buf.append(ParentDirectoryString);
    }
    // 共通部分以降の絶対パス部分を追加する
    for(int i = commonCount; i < static_cast<int>(absolutePathDirs.size()); i++)
    {
        // 最初以外は "\" をつける
        if (i != commonCount)
        {
            buf.append(DirectorySeparatorChar);
        }
        buf.append(absolutePathDirs[i]);

    }
    return buf;
}

//---------------------------------------------------------------------------
// カレントディレクトリを基点に相対パスを取得
//---------------------------------------------------------------------------
std::string PathUtility::GetAbsolutePathFromCurrentDirectory(char* path)
{
    char drive[MaxPathLength];
    _splitpath(
       path,
       drive,
       NULL,
       NULL,
       NULL);

    // ドライブが存在する場合は絶対パスとして認識
    if (strlen(drive) > 0)
    {
        return std::string(path);
    }

    TCHAR wcdir[MaxPathLength];
    char cdir[MaxPathLength];
    GetCurrentDirectory(MaxPathLength, wcdir);

    // UTF-16文字列からShift-JISに変換
    WideCharToMultiByte(CP_ACP, 0, wcdir, -1, cdir, MaxPathLength, NULL, NULL);

    std::string basePath = std::string(cdir);
    std::string relativePath = std::string(path);
    std::string absolutePath = GetAbsolutePath(basePath, relativePath);

    return absolutePath;
}

//---------------------------------------------------------------------------
// カレントディレクトリを基点に相対ファイルパスを取得
//---------------------------------------------------------------------------
std::string PathUtility::GetFilePathFromCurrentDirectory(char* path)
{
    char dir[MaxPathLength];
    char fname[MaxPathLength];
    char ext[MaxPathLength];
    char drive[MaxPathLength];
    _splitpath(
       path,
       drive,
       dir,
       fname,
       ext);

    // ドライブが存在する場合は絶対パスとして認識
    if (strlen(drive) > 0)
    {
        return std::string(path);
    }

    return GetAbsolutePathFromCurrentDirectory(dir) + std::string(fname) + std::string(ext);
}
