﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "testGraphics_ProfileDataComparer.h"

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>

namespace nnt{ namespace graphics{

//-------------------------------------------------------------------------------------
bool CompareCpuAverageLoad(const PerformanceProfileData& lhs, const PerformanceProfileData& rhs, float fTolerance)
{
    return CompareValue(lhs.GetAverageValue(PerformanceProfileData::FrameValueType_CpuLoad), rhs.GetAverageValue(PerformanceProfileData::FrameValueType_CpuLoad), fTolerance, "CompareCpuAverageLoad");
}

//-------------------------------------------------------------------------------------
bool CompareGpuAverageLoad(const PerformanceProfileData& lhs, const PerformanceProfileData& rhs, float fTolerance)
{
    return CompareValue(lhs.GetAverageValue(PerformanceProfileData::FrameValueType_GpuLoad), rhs.GetAverageValue(PerformanceProfileData::FrameValueType_GpuLoad), fTolerance, "CompareGpuAverageLoad");
}

//-------------------------------------------------------------------------------------
bool CompareMemoryUsageAverage(const PerformanceProfileData& lhs, const PerformanceProfileData& rhs, float fTolerance)
{
    return CompareValue(lhs.GetAverageValue(PerformanceProfileData::FrameValueType_MemoryUsage), rhs.GetAverageValue(PerformanceProfileData::FrameValueType_MemoryUsage), fTolerance, "CompareMemoryUsageAverage");
}

//-------------------------------------------------------------------------------------
bool CompareInitializationLoad(const PerformanceProfileData& lhs, const PerformanceProfileData& rhs, float fTolerance)
{
    if (lhs.GetInitializationLoadValue() != 0 &&
        rhs.GetInitializationLoadValue() != 0)
    {
        return CompareValue(lhs.GetInitializationLoadValue(), rhs.GetInitializationLoadValue(), fTolerance, "CompareInitializationLoad");
    }

    NN_LOG("Comparing initialization load test is skipped.");

    return true;
}

//-------------------------------------------------------------------------------------
bool CompareCpuLoadFrameValue(const PerformanceProfileData& lhs, const PerformanceProfileData& rhs, float fTolerance)
{
    return CompareFrameValue(lhs.GetFrameValues(PerformanceProfileData::FrameValueType_CpuLoad),
                            lhs.GetFrameValueCount(PerformanceProfileData::FrameValueType_CpuLoad),
                            rhs.GetFrameValues(PerformanceProfileData::FrameValueType_CpuLoad),
                            rhs.GetFrameValueCount(PerformanceProfileData::FrameValueType_CpuLoad),
                fTolerance, "CompareFpuLoadFrameValue");
}

//-------------------------------------------------------------------------------------
bool CompareGpuLoadFrameValue(const PerformanceProfileData& lhs, const PerformanceProfileData& rhs, float fTolerance)
{
    return CompareFrameValue(lhs.GetFrameValues(PerformanceProfileData::FrameValueType_GpuLoad),
                            lhs.GetFrameValueCount(PerformanceProfileData::FrameValueType_GpuLoad),
                            rhs.GetFrameValues(PerformanceProfileData::FrameValueType_GpuLoad),
                            rhs.GetFrameValueCount(PerformanceProfileData::FrameValueType_GpuLoad),
                            fTolerance, "CompareGpuLoadFrameValue");
}

//-------------------------------------------------------------------------------------
bool CompareMemoryUsageFrameValue(const PerformanceProfileData& lhs, const PerformanceProfileData& rhs, float fTolerance)
{
    return CompareFrameValue(lhs.GetFrameValues(PerformanceProfileData::FrameValueType_MemoryUsage),
                            lhs.GetFrameValueCount(PerformanceProfileData::FrameValueType_MemoryUsage),
                            rhs.GetFrameValues(PerformanceProfileData::FrameValueType_MemoryUsage),
                            rhs.GetFrameValueCount(PerformanceProfileData::FrameValueType_MemoryUsage),
                            fTolerance, "CompareMemoryUsageFrameValue");
}
//-------------------------------------------------------------------------------------
bool CompareValue(int64_t lhs, int64_t rhs, float fTolerance, const char *message)
{
    NN_SDK_ASSERT(fTolerance < 1.0f);

    int64_t nTolerance = 0;
    if (fTolerance > 0.0f)
    {
        nTolerance = rhs / static_cast<int64_t>(1.0f / fTolerance);
    }
    int64_t nDiff = rhs - lhs;
    if (nDiff > nTolerance || nDiff < -nTolerance)
    {
        NN_LOG("%s failed: lhs=%llu, rhs=%llu, tolerance=%f%%\n", message, (long long)lhs, (long long)rhs, fTolerance * 100);
        return false;
    }

    return true;
}

//-------------------------------------------------------------------------------------
bool CompareFrameValue(const PerformanceProfileData::FrameValue* pLhs, int lhsCount, const PerformanceProfileData::FrameValue* pRhs, int rhsCount, float fTolerance, const char *msg)
{
    if (lhsCount != rhsCount)
    {
        return false;
    }

    for(int i = 0; i < lhsCount;++i)
    {
        if (pLhs->GetFrame() != pRhs->GetFrame() ||
            !CompareValue(pLhs[i].GetValue(), pRhs[i].GetValue(), fTolerance, msg))
        {
            return false;
        }
    }

    return true;
}


}}

