﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstdint>
#include <iostream>

#define NNT_GRAPHICS_ALLOCATE(size, alignment) ImageIo::AllocateMemory(size, alignment)
#define NNT_GRAPHICS_FREE(ptr) ImageIo::FreeMemory(ptr)

#define NNT_GRAPHICS_NEW(size, alignment) new(ImageIo::AllocateMemory(size, alignment))
#define NNT_GRAPHICS_DELETE(ptr) ImageIo::FreeMemory(ptr)

namespace nnt{ namespace graphics{

    /**
    * @brief PNG入出力
    *
    * @details
    */
    class ImageIo
    {
    public:
        static void* AllocateMemory(size_t size, int alignment);
        static void FreeMemory(void* ptr);

    protected:

        /**
         * @brief 8bitデータをストリームに出力
         *
         * @param[in] pStream 出力ストリーム
         * @param[in] pData   データ
         * @param[in] pCrc    CRC
        */
        static void Output8( std::ostream* pStream, const void* pData, uint32_t* pCrc = NULL );

        /**
         * @brief 32bitデータをストリームに出力
         *
         * @param[in] pStream 出力ストリーム
         * @param[in] pData   データ
         * @param[in] pCrc    CRC
        */
        static void Output32( std::ostream* pStream, const void* pData, uint32_t* pCrc = NULL );


        /**
         * @brief バイト列データをストリームに出力
         *
         * @param[in] pStream 出力ストリーム
         * @param[in] pData   データ
         * @param[in] size    データサイズ
         * @param[in] pCrc    CRC
        */
        static void OutputBytes( std::ostream* pStream, const void* pData, size_t size, uint32_t* pCrc = NULL );


        /**
         * @brief 8bitデータをストリームから入力
         *
         * @param[in] pStream 入力ストリーム
         * @return 入力データ
        */
        static uint8_t Input8( uint8_t** pStream );

        /**
         * @brief 16bitデータをストリームから入力
         *
         * @param[in] pStream 入力ストリーム
         * @return 入力データ
        */
        static uint16_t Input16( uint8_t** pStream );

        /**
         * @brief 16bitデータをストリームから入力
         *
         * @param[in] pStream 入力ストリーム
         * @param[in] isFlip エンディアンスワップ
         * @return 入力データ
        */
        static uint16_t Input16( uint8_t** pStream, bool isFlip );


        /**
         * @brief 32bitデータをストリームから入力
         *
         * @param[in] pStream 入力ストリーム
         * @return 入力データ
        */
        static uint32_t Input32( uint8_t** pStream );

        /**
         * @brief 32bitデータをストリームから入力
         *
         * @param[in] pStream 入力ストリーム
         * @param[in] isFlip エンディアンスワップ
         * @return 入力データ
        */
        static uint32_t Input32( uint8_t** pStream, bool isFlip );

        /**
         * @brief バイト列データをストリームから入力
         *
         * @param[in] pStream 入力ストリーム
         * @param[in] pData   データ
         * @param[in] size    データサイズ
        */
        static void InputBytes( uint8_t** pStream, void* pData, size_t size );
    };
}}
