﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <cstdint>
#include <utility>
#include <algorithm>

#include <nn/fs.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>
#include <nn/util/util_StringUtil.h>
#include <nnt/nntest.h>
#include "testGraphics_ProgramOptions.h"
#include "testGraphics_PngIO.h"
#include "testGraphics_BmpIo.h"
#include "testGraphics_PixelwiseImageComparison.h"
#include "testGraphics_ImageCompareResult.h"
#include "testGraphics_Path.h"
#include "testGraphics_Allocator.hpp"

namespace {

    void* AllocateByAlignedMalloc(size_t size, int alignment, void*)
    {
        return _aligned_malloc(size, static_cast<size_t>(alignment));
    }
    void FreeByAlignedFree(void* ptr, void*)
    {
        _aligned_free(ptr);
    }
    void GetAlignedMallocAllocator(nnt::graphics::Allocator* pOutValue)
    {
        pOutValue->pArgument = NULL;
        pOutValue->allocateFunction = AllocateByAlignedMalloc;
        pOutValue->freeFunction = FreeByAlignedFree;
    }


    // ディレクトリ内の PNG 画像のリストを取得。
    void CreateImagePathList(
        int* pOutPathCount,
        nnt::graphics::Path** pOutPathList,
        const nnt::graphics::Path* dirpath,
        const nnt::graphics::Allocator* allocator)
    {
        nn::Result result;
        nn::fs::DirectoryHandle dirHandle;
        result = nn::fs::OpenDirectory(&dirHandle, dirpath->GetString(), nn::fs::OpenDirectoryMode_File);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        int64_t directoryEntryCount = 0;
        result = nn::fs::GetDirectoryEntryCount(&directoryEntryCount, dirHandle);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        auto directoryEntries = allocator->AllocateTypedArray<nn::fs::DirectoryEntry>(static_cast<int>(directoryEntryCount));
        auto pathList = allocator->AllocateTypedArray<nnt::graphics::Path>(static_cast<int>(directoryEntryCount));
        NN_ASSERT(directoryEntries != nullptr);
        NN_ASSERT(pathList != nullptr);

        result = nn::fs::ReadDirectory(&directoryEntryCount, directoryEntries, dirHandle, directoryEntryCount);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        nn::fs::CloseDirectory(dirHandle);

        // 拡張子の一致するものだけをフィルタリング
        int pathCount = 0;
        for(int64_t i = 0; i < directoryEntryCount; i++)
        {
            nn::fs::DirectoryEntry* pEntry = directoryEntries + i;
            bool setResult = pathList[pathCount].SetString(pEntry->name);
            if(!setResult)
            {
                continue;
            }
            if(pathList[pathCount].IsEqualExtension(".png", true))
            {
                pathCount++;
            }
            else if(pathList[pathCount].IsEqualExtension(".bmp", true))
            {
                pathCount++;
            }
        }

        allocator->FreeTypedArray(directoryEntries);
        directoryEntries = nullptr;

        // ソート
        std::sort(
            pathList,
            pathList + pathCount,
            [](const nnt::graphics::Path& a, const::nnt::graphics::Path& b)
            {
                return nn::util::Strncmp(a.GetString(), b.GetString(), nnt::graphics::Path::Capacity) < 0;
            }
        );

        // ディレクトリパスと連結
        for(int i = 0; i < pathCount; i++)
        {
            nnt::graphics::Path tmpPath;
            bool combine_result;
            combine_result = tmpPath.SetString(dirpath->GetString());
            NN_ASSERT(combine_result == true);
            combine_result = tmpPath.CombineAssign(&pathList[i]);
            NN_ASSERT(combine_result == true);
            pathList[i] = tmpPath;
        }

        *pOutPathCount = pathCount;
        *pOutPathList = pathList;
    }
    void DestroyImagePathList(int* pPathCount, nnt::graphics::Path** pPathList, nnt::graphics::Allocator* allocator)
    {
        allocator->FreeTypedArray(*pPathList);
        *pPathList = nullptr;
        *pPathCount = 0;
    }

    void CreateCapturedReferencePairList(
        int* pOutPairCount,
        std::pair<nnt::graphics::Path, nnt::graphics::Path>** pOutPairList,
        int capturedImagePathCount,
        const nnt::graphics::Path* capturedImagePathList,
        int referenceImagePathCount,
        const nnt::graphics::Path* referenceImagePathList,
        const nnt::graphics::Allocator* allocator)
    {
        NN_ASSERT(pOutPairCount != nullptr);
        std::pair<nnt::graphics::Path, nnt::graphics::Path>* pairList = nullptr;
        int pairCount = 0;

        if(pOutPairList != nullptr)
        {
            NN_ASSERT(allocator != nullptr);
            int numberOfPairs = 0;
            CreateCapturedReferencePairList(
                &numberOfPairs,
                nullptr, /* pOutPairList : nullptr の場合カウントだけ行う */
                capturedImagePathCount,
                capturedImagePathList,
                referenceImagePathCount,
                referenceImagePathList,
                nullptr);
            pairList = allocator->AllocateTypedArray<std::pair<nnt::graphics::Path, nnt::graphics::Path>>(numberOfPairs);
        }

        int iCap = 0;
        int iRef = 0;
        while(iCap < capturedImagePathCount || iRef < referenceImagePathCount)
        {
            // ファイル名を比較
            int compareResult = 0;
            if(iCap == capturedImagePathCount)
            {
                compareResult = 1;
            }
            else if(iRef == referenceImagePathCount)
            {
                compareResult = -1;
            }
            else
            {
                nnt::graphics::Path capFilename;
                nnt::graphics::Path refFilename;
                capturedImagePathList[iCap].GetFilename(&capFilename);
                referenceImagePathList[iRef].GetFilename(&refFilename);
                compareResult = nn::util::Strncmp(
                    capFilename.GetString(),
                    refFilename.GetString(),
                    nnt::graphics::Path::Capacity);
            }
            // ファイル名の小さい方を進める
            if(compareResult == 0)
            {
                if(pairList != nullptr)
                {
                    pairList[pairCount] = std::make_pair(capturedImagePathList[iCap], referenceImagePathList[iRef]);
                }
                ++pairCount;
                ++iCap;
                ++iRef;
            }
            else if(compareResult < 0)
            {
                if(pairList != nullptr)
                {
                    pairList[pairCount] = std::make_pair(capturedImagePathList[iCap], nnt::graphics::Path());
                }
                ++pairCount;
                ++iCap;
            }
            else
            {
                if(pairList != nullptr)
                {
                    pairList[pairCount] = std::make_pair(nnt::graphics::Path(), referenceImagePathList[iRef]);
                }
                ++pairCount;
                ++iRef;
            }
        }
        *pOutPairCount = pairCount;
        if(pOutPairList != nullptr)
        {
            *pOutPairList = pairList;
        }
    }

    void DestroyCapturedReferencePairList(int* pPairCount, std::pair<nnt::graphics::Path, nnt::graphics::Path>** pPairList, const nnt::graphics::Allocator* allocator)
    {
        NN_ASSERT(pPairList != nullptr);
        NN_ASSERT(allocator != nullptr);

        allocator->FreeTypedArray(*pPairList);
        *pPairList = nullptr;
        *pPairCount = 0;
    }

    bool CheckImageFormat(const nnt::graphics::PNG_IHDR* ihdr)
    {
        NN_ASSERT(ihdr != nullptr);
        bool result = true;
        if(ihdr->colorType != nnt::graphics::PngIO::PNG_COLOR_TYPE_RGB
            && ihdr->colorType != nnt::graphics::PngIO::PNG_COLOR_TYPE_RGB_ALPHA)
        {
            NN_LOG("PNG format must be RGB or RGBA.\n");
            result = false;
        }
        if(ihdr->channels != 3 && ihdr->channels != 4)
        {
            NN_LOG("PNG channels must be 3 or4.\n");
            result = false;
        }
        if(ihdr->bitDepth != 8)
        {
            NN_LOG("PNG bitDepth must be 8.\n");
            result = false;
        }
        return result;
    }

    bool CheckImageFormat(const nnt::graphics::BMP_IHDR* ihdr)
    {
        NN_ASSERT(ihdr != nullptr);
        bool result = true;
        if( ( ihdr->biBitCount != 32 ) && ( ihdr->biBitCount != 24 ) )
        {
            NN_LOG("BMP format must be RGB or RGBA.\n");
        }
        if( ( ihdr->biCompression != 0 ) && ( ihdr->biCompression != 3 ) )
        {
            NN_LOG("BMP compression must be BI_RGB or BI_BITFIELDS.\n");
        }
        return result;
    }

    bool SetImageInformationRgb(nnt::graphics::ImageInformation* pOutValue, const nnt::graphics::PNG_IHDR* ihdr, uint8_t* data)
    {
        NN_ASSERT(pOutValue != nullptr);
        int width = static_cast<int>(ihdr->width);
        int height = static_cast<int>(ihdr->height);
        int channels = static_cast<int>(ihdr->channels);
        if(ihdr->colorType == nnt::graphics::PngIO::PNG_COLOR_TYPE_RGB)
        {
            pOutValue->SetDefaultR8_G8_B8(data, width * height * channels, width, height);
        }
        else if(ihdr->colorType == nnt::graphics::PngIO::PNG_COLOR_TYPE_RGB_ALPHA)
        {
            pOutValue->SetDefaultR8_G8_B8_A8(data, width * height * channels, width, height);
        }
        else
        {
            return false;
        }
        return true;
    }

    bool SetImageInformationRgb(nnt::graphics::ImageInformation* pOutValue, const nnt::graphics::BMP_IHDR* ihdr, uint8_t* data)
    {
        NN_ASSERT(pOutValue != nullptr);
        int width = static_cast<int>(ihdr->biWidth);
        int height = static_cast<int>(ihdr->biHeight);
        int channels = static_cast<int>(ihdr->GetPixelDepth() / 8 );
        if( channels == 3 )
        {
            pOutValue->SetDefaultR8_G8_B8(data, width * height * channels, width, height);
        }
        else if( channels == 4 )
        {
            pOutValue->SetDefaultR8_G8_B8_A8(data, width * height * channels, width, height);
        }
        else
        {
            return false;
        }
        return true;
    }
    void SetDifferenceImageIhdr(nnt::graphics::PNG_IHDR* pOutValue, uint32_t width, uint32_t height)
    {
        pOutValue->width = width;
        pOutValue->height = height;
        pOutValue->bitDepth = 8;
        pOutValue->colorType = nnt::graphics::PngIO::PNG_COLOR_TYPE_RGB_ALPHA;
        pOutValue->compressionType = 0;
        pOutValue->filterType = 0;
        pOutValue->interlaceType = 0;
        pOutValue->channels = 4;
    }

    class CompareImageFileResult
    {
    public:
        int numDifferentPixels;
        nnt::graphics::Path differenceImagePath;
        nnt::graphics::Path capturedImagePath;
        nnt::graphics::Path referenceImagePath;
    };

    // ファイルを読み込んで画像を比較
    // 差分画像を保存して差のあったピクセルの数を返す。
    // 完全に一致する場合、0 を返す。
    CompareImageFileResult CompareImageFile(
        const nnt::graphics::Path* differenceImageOutputPath,
        const nnt::graphics::Path* capturedImagePath,
        const nnt::graphics::Path* referenceImagePath,
        int colorTolerance,
        int displacementTolerance,
        bool isResultFilledWithMaximum,
        const nnt::graphics::Allocator* allocator)
    {
        std::uint8_t* capImgData = nullptr;
        std::uint8_t* refImgData = nullptr;
        nnt::graphics::PNG_IHDR difImgIhdr;
        nnt::graphics::ImageInformation capImgInfo;
        nnt::graphics::ImageInformation refImgInfo;

        if( capturedImagePath->IsEqualExtension( ".png", true ) )
        {
            nnt::graphics::PNG_IHDR capImgIhdr;
            nnt::graphics::PNG_IHDR refImgIhdr;

            nnt::graphics::PngIO::ReadPng(&capImgData, capturedImagePath->GetString(), capImgIhdr);
            nnt::graphics::PngIO::ReadPng(&refImgData, referenceImagePath->GetString(), refImgIhdr);

            EXPECT_TRUE(CheckImageFormat(&capImgIhdr));
            EXPECT_TRUE(CheckImageFormat(&refImgIhdr));
            EXPECT_TRUE(capImgIhdr.width == refImgIhdr.width);
            EXPECT_TRUE(capImgIhdr.height == refImgIhdr.height);

            SetDifferenceImageIhdr(&difImgIhdr, capImgIhdr.width, capImgIhdr.height);
            difImgIhdr.channels = capImgIhdr.channels;
            difImgIhdr.colorType = capImgIhdr.colorType;
            SetImageInformationRgb(&capImgInfo, &capImgIhdr, capImgData);
            SetImageInformationRgb(&refImgInfo, &refImgIhdr, refImgData);
        }
        else if( capturedImagePath->IsEqualExtension( ".bmp", true ) )
        {
            nnt::graphics::BMP_IHDR capImgIhdr;
            nnt::graphics::BMP_IHDR refImgIhdr;

            nnt::graphics::BmpIo::ReadBmp(&capImgData, capturedImagePath->GetString(), capImgIhdr);
            nnt::graphics::BmpIo::ReadBmp(&refImgData, referenceImagePath->GetString(), refImgIhdr);

            EXPECT_TRUE(CheckImageFormat(&capImgIhdr));
            EXPECT_TRUE(CheckImageFormat(&refImgIhdr));
            EXPECT_TRUE(capImgIhdr.biWidth == refImgIhdr.biWidth);
            EXPECT_TRUE(capImgIhdr.biHeight == refImgIhdr.biHeight);

            SetDifferenceImageIhdr(&difImgIhdr, capImgIhdr.biWidth, capImgIhdr.biHeight); // Default to 32bpp
            if( capImgIhdr.GetPixelDepth() >> 3 == 3 )
            {
                difImgIhdr.channels = 3;
                difImgIhdr.colorType = nnt::graphics::PngIO::PNG_COLOR_TYPE_RGB;
            }
            SetImageInformationRgb(&capImgInfo, &capImgIhdr, capImgData);
            SetImageInformationRgb(&refImgInfo, &refImgIhdr, refImgData);
        }

        size_t size = static_cast<size_t>(difImgIhdr.width * difImgIhdr.height * difImgIhdr.channels);

        uint8_t* difImgData = allocator->AllocateTypedArray<uint8_t>(static_cast<int>(size));
        std::memset(difImgData, 255, static_cast<size_t>(sizeof(uint8_t) * size));

        nnt::graphics::ImageInformation difImgInfo;

        SetImageInformationRgb(&difImgInfo, &difImgIhdr, difImgData);

        nnt::graphics::PixelwiseImageComparison::Parameter compareParam;
        compareParam.SetDefault();
        compareParam.SetColorTolerance(colorTolerance);
        compareParam.SetDisplacementTolerance(displacementTolerance);
        compareParam.SetResultFilledWithMaximum(isResultFilledWithMaximum);

        int compareResult = nnt::graphics::PixelwiseImageComparison::Compare(&difImgInfo, &capImgInfo, &refImgInfo, &compareParam);

        nnt::graphics::PngIO::WritePng(differenceImageOutputPath->GetString(), difImgData, difImgIhdr);

        allocator->FreeTypedArray(difImgData);
        nnt::graphics::ImageIo::FreeMemory(capImgData);
        nnt::graphics::ImageIo::FreeMemory(refImgData);

        CompareImageFileResult result;
        result.numDifferentPixels = compareResult;
        result.differenceImagePath.SetString(differenceImageOutputPath->GetString());
        result.capturedImagePath.SetString(capturedImagePath->GetString());
        result.referenceImagePath.SetString(referenceImagePath->GetString());

        return result;
    }

    // タイトルを取得。
    // ファイル名中の最後の '_' の直前までの文字列を返す。
    // '_' がない場合、ファイル名全体を返す。
    void GetImageTitle(nnt::graphics::Path* pOutValue, const std::pair<nnt::graphics::Path, nnt::graphics::Path>* pathPair)
    {
        nnt::graphics::Path title;
        if(!pathPair->first.IsEmpty())
        {
            pathPair->first.GetFilename(&title);
        }
        else
        {
            pathPair->second.GetFilename(&title);
        }
        title.RemoveExtension();

        int length = title.GetLength();
        for(int i = length - 1; i >= 0; i--)
        {
            if(title.GetString()[i] == '_')
            {
                title.GetString()[i] = '\0';
                break;
            }
        }

        pOutValue->SetString(title.GetString());
    }

    void GetFrameTitle(nnt::graphics::Path* pOutValue, nnt::graphics::Path* filename)
    {
        int length = filename->GetLength();
        int pos = 0;
        for(pos = length - 1; pos >= 0; pos--)
        {
            if(filename->GetString()[pos] == '_')
            {
                break;
            }
        }
        nnt::graphics::Path title(filename->GetString() + pos + 1);
        title.RemoveExtension();

        pOutValue->SetString(title.GetString());
    }
}

TEST(CompareCapturedImages, ImageDirectory)
{
    const nnt::graphics::ProgramOptions* pOpts = nnt::graphics::ProgramOptions::GetInstancePointer();
    EXPECT_TRUE(!pOpts->capturedImageDirectoryPath.IsEmpty());
    EXPECT_TRUE(!pOpts->referenceImageDirectoryPath.IsEmpty());
    EXPECT_TRUE(!pOpts->resultRootPath.IsEmpty());
    NN_LOG("Options:\n");
    NN_LOG("  --captured-image-directory  %s\n", pOpts->capturedImageDirectoryPath.GetString());
    NN_LOG("  --reference-image-directory %s\n", pOpts->referenceImageDirectoryPath.GetString());
    NN_LOG("  --result-root-path          %s\n", pOpts->resultRootPath.GetString());
    NN_LOG("  --color-tolerance           %d\n", pOpts->colorTolerance);
    NN_LOG("  --displacement-tolerance    %d\n", pOpts->displacementTolerance);
    NN_LOG("  --different-area-tolerance  %d\n", pOpts->differentAreaTolerance);
    NN_LOG("  --ignore-captured-image-not-exists %s\n", pOpts->ignoreCapturedImageNotExits ? "true" : "false");
    NN_LOG("  --ignore-comparison-result %s\n", pOpts->ignoreComparisonResult ? "true" : "false");
    NN_LOG("  --save-actual-difference-image %s\n", pOpts->saveActualImageDifference ? "true" : "false");

    nnt::graphics::Allocator allocator;
    GetAlignedMallocAllocator(&allocator);

    nn::fs::MountHostRoot();

    int imagePathPairCount = 0;
    std::pair<nnt::graphics::Path, nnt::graphics::Path>* imagePathPairList = nullptr;
    {
        int capturedImagePathCount = 0;
        nnt::graphics::Path* capturedImagePathList = nullptr;
        int referenceImagePathCount = 0;
        nnt::graphics::Path* referenceImagePathList = nullptr;
        NN_LOG("Search image files in directory \"%s\" ...\n", pOpts->capturedImageDirectoryPath.GetString());
        CreateImagePathList(&capturedImagePathCount, &capturedImagePathList, &pOpts->capturedImageDirectoryPath, &allocator);
        NN_LOG("Search image files in directory \"%s\" ...\n", pOpts->referenceImageDirectoryPath.GetString());
        CreateImagePathList(&referenceImagePathCount, &referenceImagePathList, &pOpts->referenceImageDirectoryPath, &allocator);

        CreateCapturedReferencePairList(
            &imagePathPairCount,
            &imagePathPairList,
            capturedImagePathCount,
            capturedImagePathList,
            referenceImagePathCount,
            referenceImagePathList,
            &allocator);

        DestroyImagePathList(&capturedImagePathCount, &capturedImagePathList, &allocator);
        DestroyImagePathList(&referenceImagePathCount, &referenceImagePathList, &allocator);
    }

    NN_LOG("Detected Image pairs\n");
    for(int i = 0; i < imagePathPairCount; i++)
    {
        auto pathPair = imagePathPairList[i];
        nnt::graphics::Path filename;
        if(!pathPair.first.IsEmpty())
        {
            pathPair.first.GetFilename(&filename);
        }
        else
        {
            pathPair.second.GetFilename(&filename);
        }

        NN_LOG("  [%d] %s : %s %s\n", i, filename.GetString(),  (pathPair.first.IsEmpty() ? "---" : "CAP"), (pathPair.second.IsEmpty() ? "---" : "REF"));
    }
    // 結果サマリを用意
    nnt::graphics::ImageCompareResult resultSummary;
    resultSummary.Initialize(pOpts->resultRootPath.GetString(), &allocator);

    NN_LOG("Start Comparison\n");
    nnt::graphics::Path title;
    nnt::graphics::Path ngTitle;
    for(int i = 0; i < imagePathPairCount; i++)
    {
        NN_LOG("  Comparing %d/%d ... ", i + 1, imagePathPairCount);
        const auto& imagePathPair = imagePathPairList[i];

        nnt::graphics::Path newTitle;
        GetImageTitle(&newTitle, &imagePathPair);
        if(!title.IsEqual(newTitle))
        {
            title.SetString(newTitle.GetString());
            resultSummary.AddDataHeader(title.GetString(), resultSummary.GetResultHtmlTable());
        }
        CompareImageFileResult result;
        if(!imagePathPair.first.IsEmpty() && !imagePathPair.second.IsEmpty())
        {
            nnt::graphics::Path filename;
            imagePathPair.first.GetFilename(&filename);
            nnt::graphics::Path difPath(resultSummary.GetDifferenceImageDirectoryPath());
            difPath.CombineAssign(filename);
            result = CompareImageFile(
                &difPath,
                &imagePathPair.first,
                &imagePathPair.second,
                pOpts->colorTolerance,
                pOpts->displacementTolerance,
                !pOpts->saveActualImageDifference,
                &allocator);
            nnt::graphics::Path frameTitle;
            GetFrameTitle(&frameTitle, &filename);
            bool isOk = result.numDifferentPixels <= pOpts->differentAreaTolerance;

            if (!isOk)
            {
                if (!ngTitle.IsEqual(newTitle))
                {
                    ngTitle.SetString(newTitle.GetString());
                    resultSummary.AddDataHeader(ngTitle.GetString(), resultSummary.GetNgResultHtmlTable());
                }
            }

            resultSummary.AddCompareResult(
                isOk,
                result.numDifferentPixels,
                frameTitle.GetString(),
                imagePathPair.first.GetString(),
                imagePathPair.second.GetString(),
                difPath.GetString());
            if(isOk)
            {
                NN_LOG("ok (%d pixels)", result.numDifferentPixels);
            }
            else
            {
                NN_LOG("NG (%d pixels)", result.numDifferentPixels);
            }
        }
        else if(!imagePathPair.first.IsEmpty())
        {
            if (!ngTitle.IsEqual(newTitle))
            {
                ngTitle.SetString(newTitle.GetString());
                resultSummary.AddDataHeader(ngTitle.GetString(), resultSummary.GetNgResultHtmlTable());
            }

            nnt::graphics::Path filename;
            imagePathPair.first.GetFilename(&filename);
            nnt::graphics::Path frameTitle;
            GetFrameTitle(&frameTitle, &filename);
            resultSummary.AddCompareResult(
                false,
                -1,
                frameTitle.GetString(),
                imagePathPair.first.GetString(),
                "",
                "");
            NN_LOG("skipped");
        }
        else if(!imagePathPair.second.IsEmpty())
        {
            if(!pOpts->ignoreCapturedImageNotExits)
            {
                if (!ngTitle.IsEqual(newTitle))
                {
                    ngTitle.SetString(newTitle.GetString());
                    resultSummary.AddDataHeader(ngTitle.GetString(), resultSummary.GetNgResultHtmlTable());
                }

                nnt::graphics::Path filename;
                imagePathPair.second.GetFilename(&filename);
                nnt::graphics::Path frameTitle;
                GetFrameTitle(&frameTitle, &filename);
                resultSummary.AddCompareResult(
                    pOpts->ignoreCapturedImageNotExits,
                    -1,
                    frameTitle.GetString(),
                    "",
                    imagePathPair.second.GetString(),
                    "");
            }
            NN_LOG("skipped");
        }


        NN_LOG("\n");
    }

    bool succeed = true;
    if (!pOpts->ignoreComparisonResult)
    {
        succeed &= !resultSummary.HasNgFrame();
    }

    resultSummary.SaveResultHtml();

    resultSummary.SaveNgResultHtml(succeed);

    resultSummary.Finalize();
    DestroyCapturedReferencePairList(&imagePathPairCount, &imagePathPairList, &allocator);
    nn::fs::UnmountHostRoot();

    EXPECT_TRUE(succeed);
}// NOLINT(impl/function_size);
