﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <new>
#include <sstream>
#include <zlib.h>
#include <nn/nn_Assert.h>
#include <nn/fs.h>
#include <nn/nn_Windows.h>
#include "testGraphics_BmpIo.h"

namespace nnt{ namespace graphics{


//------------------------------------------------------------------------------
//  BMPファイル読み込み
//------------------------------------------------------------------------------
bool BmpIo::ReadBmp( uint8_t** pImage, const char* pPath, BMP_IHDR& ihdr )
{
    // ファイルからデータを入力
    uint8_t* pData;
    uint32_t size;
    {
        nn::Result result;
        nn::fs::FileHandle file = {};
        int64_t fileSize = 0;
        result = nn::fs::OpenFile(&file, pPath, nn::fs::OpenMode_Read);
        NN_ASSERT(result.IsSuccess());
        result= nn::fs::GetFileSize(&fileSize, file);
        NN_ASSERT(result.IsSuccess());
        size = static_cast<uint32_t>(fileSize);
        pData = reinterpret_cast<uint8_t*>(NNT_GRAPHICS_ALLOCATE(size, NN_ALIGNOF(int)));
        NN_ASSERT_NOT_NULL(pData);
        result = nn::fs::ReadFile(file, 0, pData, size);
        NN_ASSERT(result.IsSuccess());
        nn::fs::CloseFile(file);
    }

    uint8_t* dataHead = pData;

    // BITMAPFILEHEADER
    InputBytes( &pData, &ihdr.bfType, sizeof( BMP_IHDR().bfType ) );
    NN_ASSERT( ( ihdr.bfType[0] == 'B' && ihdr.bfType[1] == 'M' ), "Invalid BMP format.\n" );
    ihdr.bfSize = Input32( &pData, false );
    ihdr.bfReserved1 = Input16( &pData, false );
    ihdr.bfReserved2 = Input16( &pData, false );
    ihdr.bfOffBits = Input32( &pData, false );

    // BITMAPINFOHEADER
    ihdr.biSize = Input32( &pData, false );
    ihdr.biWidth = Input32( &pData, false );
    ihdr.biHeight = Input32( &pData, false );
    NN_ASSERT( ihdr.biHeight > 0, "Bottom-up DIB BMP is acceptable.\n" );
    ihdr.biPlanes = Input16( &pData, false );
    ihdr.biBitCount = Input16( &pData, false );
    ihdr.biCompression = Input32( &pData, false );

    uint32_t rowBytes;
    rowBytes = ((ihdr.biWidth * static_cast<uint32_t>(ihdr.biBitCount) + 7) >> 3);

    uint32_t imageSize = (rowBytes) * ihdr.biHeight;
    *pImage = static_cast<uint8_t*>(AllocateMemory(imageSize, NN_ALIGNOF(int)) );

    // BGRA to RGBA and flip vertical to top-left origin.
    uint8_t* src = dataHead + ihdr.bfOffBits;
    uint8_t* dst = *pImage + imageSize - rowBytes;
    int chCount = ihdr.biBitCount >> 3;
    int idxLine = ihdr.biHeight;
    while( 0 < idxLine )
    {
        uint8_t* pLine = dst;
        for( int px = ihdr.biWidth - 1; 0 <= px; --px )
        {
            *( pLine++ ) = *( src + 2 ); // src:R
            *( pLine++ ) = *( src + 1 ); // src:G
            *( pLine++  ) = *( src );    // src:B
            if( chCount == 4 )
            {
                *( pLine++ ) = *( src + 3 ); // src:A
            }
            src += chCount;
        }
        --idxLine;
        dst -= rowBytes;
    }

    FreeMemory(dataHead);

    return true;
}// NOLINT(impl/function_size);

//------------------------------------------------------------------------------
//  BMP ファイル読み込み
//------------------------------------------------------------------------------
bool BmpIo::ReadBmp( uint8_t** pImage, const char* pPath)
{
    BMP_IHDR ihdr = {};
    return ReadBmp( pImage, pPath, ihdr );
}

}}
