﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/TargetConfigs/build_Base.h>
#include <nn/nn_Log.h>

#include <nn/os.h>
#include <nn/svc/svc_MemoryMapSelect.h>

#include <nn/gpio/gpio.h>
#include <nn/gpio/gpio_PadAccessorDev.h>

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>

#include "../common/testGpio_Util.h"

namespace
{

// 割り込みテストの時の GPIO パッドの H/L を切り替える時間
const nn::TimeSpan PadValueSwitchTimeForInterrupt = nn::TimeSpan::FromMilliSeconds(100);

// サブスレッドで H/L を切り替える回数
const int LoopNumber = 3;

// 割り込みスレッドのスタック領域
const size_t InterruptThreadStackSize = nn::os::StackRegionAlignment;
NN_ALIGNAS(nn::os::StackRegionAlignment) char s_Stack[InterruptThreadStackSize];


// テスト終了条件に使用するビットフラグのマスク(NumberOfTestGpioPads の数分のビットを立てておく)
const nn::Bit32 EndConditionMask = (1 << nnt::gpio::NumberOfTestPad) - 1;
//const nn::Bit32 EndConditionMask = 0x7F;

// 割り込み待ちのタイムアウト時間
const nn::TimeSpan TimeOut = nn::TimeSpan::FromMilliSeconds(5000);

// テスト用 pinmux ライブラリ
nnt::gpio::Pinmux g_Pinmux;

}

namespace nnt { namespace gpio {

// PadValueSwitchTimeForInterrupt だけ待機し、一度だけ H → L を行うスレッド
void SwitchGpioValueThreadBody(void* arg)
{
    nnt::gpio::PadName pad = *reinterpret_cast<nnt::gpio::PadName*>(arg);

    NN_LOG("%s: Sub Thread Start\n", __FUNCTION__);

    nn::os::SleepThread(PadValueSwitchTimeForInterrupt);

    g_Pinmux.PullDown(pad);
    //NN_LOG("%s: Pulldown(L)\n", __FUNCTION__);

    nn::os::SleepThread(PadValueSwitchTimeForInterrupt);

    g_Pinmux.PullUp(pad);
    //NN_LOG("%s: Pullup(H)\n", __FUNCTION__);
}

// 割り込みテストのための H/L を切り替えるスレッドの実態
void RepeatHAndLThreadBodyForOneGpioPad(void* arg)
{
    nnt::gpio::PadName pad = *reinterpret_cast<nnt::gpio::PadName*>(arg);

    int loopCount = 0;

    NN_LOG("%s: Sub Thread Start\n", __FUNCTION__);

    while(1)
    {
        nn::os::SleepThread(PadValueSwitchTimeForInterrupt);

        // 入力を内部的に pulldown する。
        g_Pinmux.PullDown(pad);
        //NN_LOG("%s: Pulldown(L)\n", __FUNCTION__);

        nn::os::SleepThread(PadValueSwitchTimeForInterrupt);

        // 入力を内部的に pullup する。
        g_Pinmux.PullUp(pad);
        //NN_LOG("%s: Pullup(H)\n", __FUNCTION__);

        loopCount++;
        if(loopCount == LoopNumber)
        {
            break;
        }
    }
}

// 割り込みテストのための H/L を切り替えるスレッドの実態(複数パッド対応版)
void RepeatHAndLThreadBodyForGpioPads(void* arg)
{
    int loopCount = 0;

    NN_LOG("%s: Sub Thread Start\n", __FUNCTION__);

    while(1)
    {
        for(int i =0; i < NumberOfTestPad; i++)
        {
            nn::os::SleepThread(PadValueSwitchTimeForInterrupt);
            // 入力を内部的に pulldown する。
            g_Pinmux.PullDown(PadNameList[i]);
            //NN_LOG("Pad[%d] Pulldown(L)\n", PadNameList[i]);
        }

        for(int i =0; i < NumberOfTestPad; i++)
        {
            nn::os::SleepThread(PadValueSwitchTimeForInterrupt);
            // 入力を内部的に pulldown する。
            g_Pinmux.PullUp(PadNameList[i]);
            //NN_LOG("Pad[%d] PullUp(H)\n", PadNameList[i]);
        }

        loopCount++;
        if(loopCount == LoopNumber)
        {
            break;
        }
    }
}


// Edge トリガーの InterruptTest の終了条件を判定する関数
bool IsEndInterruptTest(nn::gpio::InterruptMode mode, int highCount, int lowCount)
{
    // mode ごとに終了条件を分岐する
    switch(mode)
    {
    case nn::gpio::InterruptMode_RisingEdge:
        // 想定のカウントを超えていた場合エラーとしつつ、終了処理へ飛ばす
        if(highCount > LoopNumber || lowCount > 0)
        {
            EXPECT_LE(highCount, LoopNumber);
            EXPECT_LE(lowCount, 0);
            return true;
        }
        if(highCount == LoopNumber && lowCount == 0)
        {
            return true;
        }
        break;

    case nn::gpio::InterruptMode_FallingEdge:
        // 想定のカウントを超えていた場合エラーとしつつ、終了処理へ飛ばす
        if(highCount > 0 || lowCount > LoopNumber - 1)
        {
            EXPECT_LE(highCount, 0);
            EXPECT_LE(lowCount, LoopNumber - 1);
            return true;
        }
        if(highCount == 0 && lowCount == LoopNumber - 1)
        {
            return true;
        }
        break;

    case nn::gpio::InterruptMode_AnyEdge:
        // 想定のカウントを超えていた場合エラーとしつつ、終了処理へ飛ばす
        if(highCount > LoopNumber || lowCount > LoopNumber - 1)
        {
            EXPECT_LE(highCount, LoopNumber);
            EXPECT_LE(lowCount, LoopNumber - 1);
            return true;
        }
        if(highCount == LoopNumber && lowCount == LoopNumber - 1)
        {
            return true;
        }
        break;

    default:
        // エッジトリガー以外が渡されていることになるので、失敗として記録して、終了条件を満たしたことにする。
        EXPECT_TRUE(false);
        return true;
    }
    return false;
}

// 1パッドごとの基本的な割り込みのテスト(レベルトリガー版)
void TestOneGpioPadLevelInterrupt(nnt::gpio::PadName pad, nn::gpio::InterruptMode mode)
{
    NN_LOG("Start Basic Interrupt Test For GPIO Pad: %d\n", pad);

    // セッション用の構造体
    nn::gpio::GpioPadSession session;

    //GPIO ピンの値が変化した際に通知を受けるイベントオブジェクト
    nn::os::SystemEventType event;

    // H/L を繰り返すスレッド
    nn::os::ThreadType thread;

    // 状態が不定の可能性があるので、入力を内部的に pulldown しておく。
    g_Pinmux.PullDown(pad);
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(50));

#if (defined(NN_GPIO_GEN) && (NN_GPIO_GEN == 2))
    NNT_ASSERT_RESULT_SUCCESS(nn::gpio::OpenSession(&session, pad));
#else // gen1
    nn::gpio::OpenSessionForDev(&session, static_cast<int>(pad));
#endif

    nn::gpio::SetDirection(&session, nn::gpio::Direction_Input);

    nn::gpio::SetInterruptMode(&session, mode);

    // 割り込みを disable にする。
    nn::gpio::SetInterruptEnable(&session, false);

    // 割り込みをクリアする
    nn::gpio::ClearInterruptStatus(&session);

    // 設定を表示
    NN_LOG("-----------------------\n");
    NN_LOG(" GPIO Pad        : %d\n", pad);
    NN_LOG(" Direction       : %d\n", nn::gpio::GetDirection(&session));
    NN_LOG(" InterruptMode   : %d\n", nn::gpio::GetInterruptMode(&session));
    NN_LOG(" InterruptEnable : %d\n", nn::gpio::GetInterruptEnable(&session));
    NN_LOG(" isInterrupted   : %d\n", nn::gpio::GetInterruptStatus(&session));
    NN_LOG("-----------------------\n");

    // Interrupt を bind する。
    nn::gpio::BindInterrupt(&event, &session);

    // テストに入る前にイベントをクリアしておく。
    nn::os::ClearSystemEvent(&event);
    nn::gpio::SetInterruptEnable(&session,true);

    // PadValueSwitchTimeForInterrupt だけ待機し、一度だけ H → L を行うスレッド
    nn::os::CreateThread(&thread, SwitchGpioValueThreadBody, &pad, s_Stack, InterruptThreadStackSize, nn::os::HighestThreadPriority);
    nn::os::StartThread(&thread);

    NN_LOG(" Start Interrupt Test \n");

    // 終了条件に使用する
    int higeCount = 0;
    int lowCount = 0;
    bool checkFlag = false;

    // 割り込みが発生したら、値を取得する。
    while(1)
    {
        if(!nn::os::TimedWaitSystemEvent(&event, TimeOut))
        {
            ADD_FAILURE();
        }

        // 値のチェック
        if(nn::gpio::GetValue(&session) == nn::gpio::GpioValue_High)
        {
            NN_LOG(" Get Interrupt : H\n");
            higeCount++;
        }
        else
        {
            NN_LOG(" Get Interrupt : L\n");
            lowCount++;
        }


        // 1回値を取得し、イベントをクリア、その後もう割り込みが発生して値が取れていることのチェック
        if(!checkFlag)
        {
            checkFlag = true;
        }
        else
        {
            // 対象のCount は 2 になっているはず。
            if(mode == nn::gpio::InterruptMode_LowLevel)
            {
                EXPECT_EQ(2, lowCount);
                EXPECT_EQ(0, higeCount);
            }
            else
            {
                EXPECT_EQ(2, higeCount);
                EXPECT_EQ(0, lowCount);
            }

            nn::gpio::UnbindInterrupt(&session);
            nn::gpio::CloseSession(&session);

            nn::os::SleepThread(PadValueSwitchTimeForInterrupt);
            nn::os::DestroyThread(&thread);
            return;
        }

        // 割り込みステータスをクリアする
        nn::gpio::ClearInterruptStatus(&session);

        // イベントのクリア
        nn::os::ClearSystemEvent(&event);

        // 割り込み許可状態に戻す
        nn::gpio::SetInterruptEnable(&session,true);
    }
}


// 1パッドごとの基本的な割り込みのテスト(Edge トリガー版)
void TestOneGpioPadEdgeInterrupt(nnt::gpio::PadName pad, nn::gpio::InterruptMode mode)
{
    NN_LOG("Start Basic Interrupt Test For GPIO Pad: %d\n", pad);

    // セッション用の構造体
    nn::gpio::GpioPadSession session;

    //GPIO ピンの値が変化した際に通知を受けるイベントオブジェクト
    nn::os::SystemEventType event;

    // H/L を繰り返すスレッド
    nn::os::ThreadType thread;

#if (defined(NN_GPIO_GEN) && (NN_GPIO_GEN == 2))
    NNT_ASSERT_RESULT_SUCCESS(nn::gpio::OpenSession(&session, pad));
#else // gen1
    nn::gpio::OpenSessionForDev(&session, static_cast<int>(pad));
#endif

    nn::gpio::SetDirection(&session, nn::gpio::Direction_Input);

    // 状態が不定の可能性があるので、入力を内部的に pulldown しておく。
    g_Pinmux.PullDown(pad);
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(50));

    nn::gpio::SetInterruptMode(&session, mode);

    // 割り込みをクリアする
    nn::gpio::ClearInterruptStatus(&session);

    // 割り込みを enable にする。
    nn::gpio::SetInterruptEnable(&session, true);

    // 設定を表示
    NN_LOG("-----------------------\n");
    NN_LOG(" GPIO Pad        : %d\n", pad);
    NN_LOG(" Direction       : %d\n", nn::gpio::GetDirection(&session));
    NN_LOG(" InterruptMode   : %d\n", nn::gpio::GetInterruptMode(&session));
    NN_LOG(" InterruptEnable : %d\n", nn::gpio::GetInterruptEnable(&session));
    NN_LOG(" isInterrupted   : %d\n", nn::gpio::GetInterruptStatus(&session));
    NN_LOG("-----------------------\n");

    // Interrupt を bind する。
    nn::gpio::BindInterrupt(&event, &session);

    // GPIO の H/L を変更するスレッド(本来はハードウェアからの信号なので、最優先のスレッドで作成する。)
    nn::os::CreateThread(&thread, RepeatHAndLThreadBodyForOneGpioPad, &pad, s_Stack, InterruptThreadStackSize, nn::os::HighestThreadPriority);
    nn::os::StartThread(&thread);

    NN_LOG(" Start Interrupt Test \n");

    // 終了条件に使用する
    int higeCount =0;
    int lowCount = 0;

    // 割り込みが発生したら、値を取得する。
    while(1)
    {
        if(!nn::os::TimedWaitSystemEvent(&event, TimeOut))
        {
            ADD_FAILURE();
        }

        // 割り込みステータスをクリアする
        nn::gpio::ClearInterruptStatus(&session);

        // イベントのクリア
        nn::os::ClearSystemEvent(&event);

        if(nn::gpio::GetValue(&session) == nn::gpio::GpioValue_High)
        {
            NN_LOG(" Get Interrupt : H\n");
            higeCount++;
        }
        else
        {
            NN_LOG(" Get Interrupt : L\n");
            lowCount++;
        }

        if(IsEndInterruptTest(mode, higeCount, lowCount))
        {
            nn::gpio::UnbindInterrupt(&session);
            nn::gpio::CloseSession(&session);
            nn::os::DestroyThread(&thread);
            return;
        }

        // 割り込み許可状態に戻す
        nn::gpio::SetInterruptEnable(&session,true);
    }
}

// 複数パッドを登録する割り込みのテスト
void TestGpioPadsInterrupt(nn::gpio::InterruptMode mode)
{
    NN_LOG("Start Interrupt Test For GPIO Pads\n");

    // TORIAEZU
    NN_LOG("mask = %x\n", EndConditionMask);
    NN_LOG("Number of Pad = %d\n", NumberOfTestPad);

    // セッション用の構造体
    nn::gpio::GpioPadSession session[NumberOfTestPad];

    //GPIO ピンの値が変化した際に通知を受けるイベントオブジェクト
    nn::os::SystemEventType event[NumberOfTestPad];

    // H/L を繰り返すスレッド
    nn::os::ThreadType thread;

    // 多重待ち管理用
    nn::os::MultiWaitType           waiter;
    nn::os::MultiWaitHolderType     eventHolder[NumberOfTestPad];

    nn::os::InitializeMultiWait(&waiter);

    for(int i = 0; i < NumberOfTestPad; i++)
    {
#if (defined(NN_GPIO_GEN) && (NN_GPIO_GEN == 2))
        NNT_ASSERT_RESULT_SUCCESS(nn::gpio::OpenSession(&session[i], PadNameList[i]));
#else // gen1
        nn::gpio::OpenSessionForDev(&session[i], static_cast<int>(PadNameList[i]));
#endif

        nn::gpio::SetDirection(&session[i], nn::gpio::Direction_Input);

        // 状態が不定の可能性があるので、入力を内部的に pulldown しておく。
        g_Pinmux.PullDown(PadNameList[i]);

        nn::gpio::SetInterruptMode(&session[i], mode);

        nn::gpio::SetInterruptEnable(&session[i], true);

        nn::gpio::ClearInterruptStatus(&session[i]);

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(50));

        // 設定を表示
        NN_LOG("-----------------------\n");
        NN_LOG(" GPIO Pad        : %d\n", PadNameList[i]);
        NN_LOG(" Direction       : %d\n", nn::gpio::GetDirection(&session[i]));
        NN_LOG(" InterruptMode   : %d\n", nn::gpio::GetInterruptMode(&session[i]));
        NN_LOG(" InterruptEnable : %d\n", nn::gpio::GetInterruptEnable(&session[i]));
        NN_LOG(" isInterrupted   : %d\n", nn::gpio::GetInterruptStatus(&session[i]));
        NN_LOG("-----------------------\n");

        // Interrupt を bind する。
        nn::gpio::BindInterrupt(&event[i], &session[i]);

        // 各イベントを多重待ちするために登録
        nn::os::InitializeMultiWaitHolder(&eventHolder[i], &event[i]);
        nn::os::LinkMultiWaitHolder(&waiter, &eventHolder[i]);
        nn::os::SetMultiWaitHolderUserData(&eventHolder[i], static_cast<uintptr_t>(i));
    }

    // GPIO の H/L を定期的にに変更するスレッド(本来はハードウェアからの信号なので、最優先のスレッドで作成する。)
    nn::os::CreateThread(&thread, RepeatHAndLThreadBodyForGpioPads, NULL, s_Stack, InterruptThreadStackSize, nn::os::HighestThreadPriority);
    nn::os::StartThread(&thread);

    // 終了条件に使用する変数
    int higeCount[NumberOfTestPad];
    int lowCount[NumberOfTestPad];
    nn::Bit32 endFlag = 0;

    // カウントを初期化
    for(int i = 0; i < NumberOfTestPad; i++)
    {
        higeCount[i] = 0;
        lowCount[i] = 0;
    }

    NN_LOG(" Start Interrupt Test \n");

    // 割り込みが発生したら、値を取得する。
    while(1)
    {

        nn::os::MultiWaitHolderType* pEventHolder = nn::os::TimedWaitAny(&waiter, TimeOut);

        if(pEventHolder == nullptr)
        {
            ADD_FAILURE();
        }

        auto interruptIdx = nn::os::GetMultiWaitHolderUserData(pEventHolder);

        if(nn::gpio::GetValue(&session[interruptIdx]) == nn::gpio::GpioValue_High)
        {
            NN_LOG(" GPIO %d Get Interrupt : H\n", PadNameList[interruptIdx]);
            higeCount[interruptIdx]++;
        }
        else
        {
            NN_LOG(" GPIO %d Get Interrupt : L\n", PadNameList[interruptIdx]);
            lowCount[interruptIdx]++;
        }

        // イベントのクリア
        nn::os::ClearSystemEvent(&event[interruptIdx]);

        // 割り込みステータスをクリアする
        nn::gpio::ClearInterruptStatus(&session[interruptIdx]);

        // 割り込み許可状態に戻す
        nn::gpio::SetInterruptEnable(&session[interruptIdx],true);

        // 終了条件をチェックする
        if(IsEndInterruptTest(mode, higeCount[interruptIdx], lowCount[interruptIdx]))
        {
            endFlag |= 1 << interruptIdx;

            // 割込みがこないように disable にする
            nn::gpio::SetInterruptEnable(&session[interruptIdx],false);

            // MultiWait 対象から該当イベントをアンリンクする
            nn::os::UnlinkMultiWaitHolder(pEventHolder);

            nn::gpio::UnbindInterrupt(&session[interruptIdx]);
            nn::gpio::CloseSession(&session[interruptIdx]);
        }

        // 全 GPIO パッドが終了しているのであれば、テスト自体を終了する。
        if(endFlag == EndConditionMask)
        {
            nn::os::DestroyThread(&thread);
            return;
        }
    }
}

} } // nnt::gpio


//-----------------------------------------------------------------------------
// 複数パッドをイベントと紐付けて正しく割り込みが発生することを確認するテスト(内部で GPIO パッドを Pullup/PullDown するため、端子が解放されていること)
TEST(testGpio_Interrupt, MultiWaitInterrupt)
{
    nn::gpio::Initialize();

    nnt::gpio::TestGpioPadsInterrupt(nn::gpio::InterruptMode_AnyEdge);

    nn::gpio::Finalize();
}


//-----------------------------------------------------------------------------
// GPIO ライブラリの Interrupt の基本的なテスト レベルトリガ版(内部で GPIO パッドを Pullup/PullDown するため、端子が解放されていること)

TEST(testGpio_Interrupt, InterruptLowLevel)
{
    nn::gpio::Initialize();

    for(int i = 0; i<nnt::gpio::NumberOfTestPad; i++)
    {
        nnt::gpio::TestOneGpioPadLevelInterrupt(nnt::gpio::PadNameList[i], nn::gpio::InterruptMode_LowLevel);
    }

    nn::gpio::Finalize();
}

TEST(testGpio_Interrupt, InterruptHighLevel)
{
    nn::gpio::Initialize();

    for(int i = 0; i<nnt::gpio::NumberOfTestPad; i++)
    {
        nnt::gpio::TestOneGpioPadLevelInterrupt(nnt::gpio::PadNameList[i], nn::gpio::InterruptMode_HighLevel);
    }

    nn::gpio::Finalize();
}

//-----------------------------------------------------------------------------
// GPIO ライブラリの Interrupt の基本的なテスト エッジトリガ版(内部で GPIO パッドを Pullup/PullDown するため、端子が解放されていること)
TEST(testGpio_Interrupt, InterruptRisingEdge)
{
    nn::gpio::Initialize();

    for(int i = 0; i<nnt::gpio::NumberOfTestPad; i++)
    {
        nnt::gpio::TestOneGpioPadEdgeInterrupt(nnt::gpio::PadNameList[i], nn::gpio::InterruptMode_RisingEdge);
    }

    nn::gpio::Finalize();
}

TEST(testGpio_Interrupt, InterruptFallingEdge)
{
    nn::gpio::Initialize();

    for(int i = 0; i<nnt::gpio::NumberOfTestPad; i++)
    {
        nnt::gpio::TestOneGpioPadEdgeInterrupt(nnt::gpio::PadNameList[i], nn::gpio::InterruptMode_FallingEdge);
    }

    nn::gpio::Finalize();
}

TEST(testGpio_Interrupt, InterruptAnyEdge)
{
    nn::gpio::Initialize();

    for(int i = 0; i<nnt::gpio::NumberOfTestPad; i++)
    {
        nnt::gpio::TestOneGpioPadEdgeInterrupt(nnt::gpio::PadNameList[i], nn::gpio::InterruptMode_AnyEdge);
    }

    nn::gpio::Finalize();
}


