﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/nn_Log.h>
#include <nn/hid.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>
#include <nn/hid/system/hid_Npad.h>
#include <nn/hid/hid_Result.h>
#include <nnt/nntest.h>

#include "../Common/testGamePad_Common.h"
#include "../Common/testGamePad_Npad.h"

namespace {

//!< システム向けのデバイス状態取得機能の検証を行う
void TestSystemInfo(const nn::hid::NpadIdType& id)
{
    nn::hid::system::NpadPowerInfo right;
    nn::hid::system::NpadPowerInfo left;
    nn::hid::system::NpadPowerInfo fullKey;

    // 電源状態の読み出し
    fullKey = nn::hid::system::GetNpadPowerInfo(id);
    nn::hid::system::GetNpadPowerInfo(&left, &right, id);

    // UI 状態の読み出し
    auto isSlSr = nn::hid::system::IsSlSrButtonOriented(id);
    auto isAbxy = nn::hid::system::IsAbxyButtonOriented(id);
    bool plusButton;
    bool minusButton;
    nn::hid::system::GetPlusMinusButtonCapability(&plusButton, &minusButton, id);

    auto deviceType = nn::hid::system::GetNpadDeviceType(id);
    auto style = nn::hid::GetNpadStyleSet(id);
    auto holdType = nn::hid::GetNpadJoyHoldType();

    // 電源状態のチェック
    if (deviceType.Test<nn::hid::system::NpadDeviceType::JoyConLeft>() ||
        deviceType.Test<nn::hid::system::NpadDeviceType::HandheldJoyLeft>())
    {
        EXPECT_NE(left.batteryLevel, nn::hid::system::NpadBatteryLevel_None);
        EXPECT_EQ(left.isPowered, true);
    }
    if (deviceType.Test<nn::hid::system::NpadDeviceType::JoyConRight>() ||
        deviceType.Test<nn::hid::system::NpadDeviceType::HandheldJoyRight>())
    {
        EXPECT_NE(right.batteryLevel, nn::hid::system::NpadBatteryLevel_None);
        EXPECT_EQ(right.isPowered, true);
    }
    if (deviceType.Test<nn::hid::system::NpadDeviceType::SwitchProController>())
    {
        // ProConのみ電池残量の確認をする(USB専用コンを除外する)
        if (nnt::gamepad::GetControllerTypeFromNpad(id) == nnt::gamepad::ControllerType::ProCon)
        {
            EXPECT_NE(fullKey.batteryLevel, nn::hid::system::NpadBatteryLevel_None);
            EXPECT_EQ(fullKey.isPowered, true);
        }
    }

    if (style.Test<nn::hid::NpadStyleFullKey>() ||
        style.Test<nn::hid::NpadStyleJoyDual>() ||
        style.Test<nn::hid::NpadStyleHandheld>())
    {
        EXPECT_FALSE(isSlSr);
        EXPECT_TRUE(isAbxy);
        EXPECT_TRUE(plusButton);
        EXPECT_TRUE(minusButton);
    }
    if (style.Test<nn::hid::NpadStyleJoyLeft>())
    {
        if (holdType == nn::hid::NpadJoyHoldType_Horizontal)
        {
            EXPECT_TRUE(isSlSr);
            EXPECT_FALSE(isAbxy);
            EXPECT_FALSE(plusButton);
            EXPECT_TRUE(minusButton);
        }
        else
        {
            EXPECT_FALSE(isSlSr);
            EXPECT_FALSE(isAbxy);
            EXPECT_FALSE(plusButton);
            EXPECT_TRUE(minusButton);
        }
    }
    if (style.Test<nn::hid::NpadStyleJoyRight>())
    {
        if (holdType == nn::hid::NpadJoyHoldType_Horizontal)
        {
            EXPECT_TRUE(isSlSr);
            EXPECT_FALSE(isAbxy);
            EXPECT_TRUE(plusButton);
            EXPECT_FALSE(minusButton);
        }
        else
        {
            EXPECT_FALSE(isSlSr);
            EXPECT_TRUE(isAbxy);
            EXPECT_TRUE(plusButton);
            EXPECT_FALSE(minusButton);
        }
    }
}

}

//!< 本体機能向けの各状態が意図したものになっているかの検証
TEST(NpadSystemInfo, ReadDeviceInfoForDefault)
{
    // テストマシンの構成に応じた Npad の初期化
    nnt::gamepad::Initialize();

    // USB有線通信を無効化
    nnt::gamepad::DisableUsbConnect();

    // コントローラを切断
    nnt::gamepad::DisconnectAll();

    // コントローラを接続
    nnt::gamepad::ConnectAll();

    for (const auto& id : nnt::gamepad::NpadIds)
    {
        TestSystemInfo(id);
    }
}

//!< 本体機能向けの各状態が意図したものになっているかの検証
TEST(NpadSystemInfo, ReadDeviceInfoForSingleVertical)
{
    // テストマシンの構成に応じた Npad の初期化
    nnt::gamepad::Initialize();

    // USB有線通信を無効化
    nnt::gamepad::DisableUsbConnect();

    // コントローラを切断
    nnt::gamepad::DisconnectAll();

    // コントローラを接続
    nnt::gamepad::ConnectAll();

    // 2本持ち接続のものをすべて1本持ちに変更
    for (const auto& id : nnt::gamepad::NpadIds)
    {
        if (nn::hid::GetNpadStyleSet(id).Test<nn::hid::NpadStyleJoyDual>())
        {
            nn::hid::SetNpadJoyAssignmentModeSingle(id);
        }
    }


    for (const auto& id : nnt::gamepad::NpadIds)
    {
        if ((nn::hid::GetNpadStyleSet(id) & (nn::hid::NpadStyleJoyLeft::Mask | nn::hid::NpadStyleJoyRight::Mask)).IsAnyOn())
        {
            TestSystemInfo(id);
        }
    }
}

