﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/hid.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>
#include <nn/hid/system/hid_Npad.h>
#include <nn/hid/hid_Result.h>
#include <nn/hid/hid_ResultController.h>
#include <nnt/nntest.h>

#include "../Common/testGamePad_Common.h"
#include "../Common/testGamePad_Npad.h"

namespace {
nn::hid::NpadStyleSet g_PreviousStyles[nnt::gamepad::NpadIdCountMax];
nn::hid::system::NpadDeviceTypeSet g_PreviousDeviceTypes[nnt::gamepad::NpadIdCountMax];

//!< コントローラー接続のリトライ回数の上限
const int RetryCountMax = 5;

// 現在の Style / DeviceType をダンプする
void DumpCurrentStyleAndDeviceType() NN_NOEXCEPT
{
    for (int i = 0; i < nnt::gamepad::NpadIdCountMax; i++)
    {
        g_PreviousStyles[i] = nn::hid::GetNpadStyleSet(nnt::gamepad::NpadIds[i]);
        g_PreviousDeviceTypes[i] = nn::hid::system::GetNpadDeviceType(nnt::gamepad::NpadIds[i]);
    }
}

void VerifyStyleAndDeviceTypeAfterSwitchToSingle(nn::hid::NpadJoyDeviceType joyDeviceType) NN_NOEXCEPT
{
    // 全ての NpadId に対してデフォルト設定で NpadJoyAssignmentMode_Single にセットする
    for (int i = 0; i < nnt::gamepad::NpadIdCountMax; i++)
    {
        auto currentStyle = nn::hid::GetNpadStyleSet(nnt::gamepad::NpadIds[i]);
        auto currentDeviceType = nn::hid::system::GetNpadDeviceType(nnt::gamepad::NpadIds[i]);
        auto previousStyle = g_PreviousStyles[i];
        auto previousDeviceType = g_PreviousDeviceTypes[i];
        // 以下のスタイルにおいては何も変わらない
        if (previousStyle.Test<nn::hid::NpadStyleFullKey>() ||
            previousStyle.Test<nn::hid::NpadStyleHandheld>() ||
            previousStyle.Test<nn::hid::NpadStyleJoyLeft>() ||
            previousStyle.Test<nn::hid::NpadStyleJoyRight>())
        {
            EXPECT_EQ(currentStyle, previousStyle);
            EXPECT_EQ(currentDeviceType, previousDeviceType);
        }
        // JoyDual の場合は、Single への切り替えが起きる
        if (previousStyle.Test<nn::hid::NpadStyleJoyDual>())
        {
            // もともと 2本持ちだった場合は押し出されたコントローラーが 1本持ちとして接続されちえる
            if (previousDeviceType.Test<nn::hid::system::NpadDeviceType::JoyConLeft>() &&
                previousDeviceType.Test<nn::hid::system::NpadDeviceType::JoyConRight>())
            {
                if (joyDeviceType == nn::hid::NpadJoyDeviceType_Left)
                {
                    EXPECT_TRUE(currentStyle.Test<nn::hid::NpadStyleJoyLeft>());
                    EXPECT_EQ(currentDeviceType, nn::hid::system::NpadDeviceType::JoyConLeft::Mask);
                }
                else
                {
                    EXPECT_TRUE(currentStyle.Test<nn::hid::NpadStyleJoyRight>());
                    EXPECT_EQ(currentDeviceType, nn::hid::system::NpadDeviceType::JoyConRight::Mask);
                }

                for (int j = 0; j < nnt::gamepad::NpadIdCountMax; j++)
                {
                    if (g_PreviousStyles[j].IsAllOff() == true)
                    {
                        if (joyDeviceType == nn::hid::NpadJoyDeviceType_Left)
                        {
                            EXPECT_TRUE(nn::hid::GetNpadStyleSet(nnt::gamepad::NpadIds[j]).Test<nn::hid::NpadStyleJoyRight>());
                            EXPECT_EQ(nn::hid::system::GetNpadDeviceType(nnt::gamepad::NpadIds[j]), nn::hid::system::NpadDeviceType::JoyConRight::Mask);
                        }
                        else
                        {
                            EXPECT_TRUE(nn::hid::GetNpadStyleSet(nnt::gamepad::NpadIds[j]).Test<nn::hid::NpadStyleJoyLeft>());
                            EXPECT_EQ(nn::hid::system::GetNpadDeviceType(nnt::gamepad::NpadIds[j]), nn::hid::system::NpadDeviceType::JoyConLeft::Mask);
                        }
                    }
                    break;
                }
            }
            else if (previousDeviceType.Test<nn::hid::system::NpadDeviceType::JoyConLeft>())
            {
                EXPECT_TRUE(currentStyle.Test<nn::hid::NpadStyleJoyLeft>());
                EXPECT_EQ(currentDeviceType, nn::hid::system::NpadDeviceType::JoyConLeft::Mask);
            }
            else if (previousDeviceType.Test<nn::hid::system::NpadDeviceType::JoyConRight>())
            {
                EXPECT_TRUE(currentStyle.Test<nn::hid::NpadStyleJoyRight>());
                EXPECT_EQ(currentDeviceType, nn::hid::system::NpadDeviceType::JoyConRight::Mask);
            }
        }
    }
}

}

//!< ジョイコンの割り当てモードを1本持ちに変更して 2本持ちに戻す
TEST(NpadJoyAssignmentTest, SwitchToSingleByDefault)
{
    // テストマシンの構成に応じた Npad の初期化
    nnt::gamepad::Initialize();

    // USB有線通信を無効化
    nnt::gamepad::DisableUsbConnect();

    // コントローラを切断
    nnt::gamepad::DisconnectAll();

    // コントローラを接続
    nnt::gamepad::ConnectAll();

    DumpCurrentStyleAndDeviceType();

    // 全ての NpadId に対してデフォルト設定で NpadJoyAssignmentMode_Single にセットする
    for (const auto& id : nnt::gamepad::NpadIds)
    {
        nn::hid::SetNpadJoyAssignmentModeSingle(id);
    }

    // 意図した Style / DeviceType になっているか検証
    VerifyStyleAndDeviceTypeAfterSwitchToSingle(nn::hid::NpadJoyDeviceType_Left);

    // Dump
    DumpCurrentStyleAndDeviceType();

    // 全ての NpadId に対してデフォルト設定で NpadJoyAssignmentMode_Dual にセットする
    for (const auto& id : nnt::gamepad::NpadIds)
    {
        nn::hid::SetNpadJoyAssignmentModeDual(id);
    }


    // Single から Dual に変えても Style は変更されるけど、DeviceTypeに変更はない
    for (int i = 0; i < nnt::gamepad::NpadIdCountMax; i++)
    {
        auto currentStyle = nn::hid::GetNpadStyleSet(nnt::gamepad::NpadIds[i]);
        auto currentDeviceType = nn::hid::system::GetNpadDeviceType(nnt::gamepad::NpadIds[i]);
        auto previousStyle = g_PreviousStyles[i];
        auto previousDeviceType = g_PreviousDeviceTypes[i];
        // 以下のスタイルにおいては何も変わらない
        if (previousStyle.Test<nn::hid::NpadStyleFullKey>() ||
            previousStyle.Test<nn::hid::NpadStyleHandheld>() ||
            previousStyle.Test<nn::hid::NpadStyleJoyDual>())
        {
            EXPECT_EQ(currentStyle, previousStyle);
            EXPECT_EQ(currentDeviceType, previousDeviceType);
        }
        // JoyLeft/JoyRight の場合は、JoyDual への切り替えが起きる
        if (previousStyle.Test<nn::hid::NpadStyleJoyLeft>() ||
            previousStyle.Test<nn::hid::NpadStyleJoyRight>())
        {
            EXPECT_TRUE(currentStyle.Test<nn::hid::NpadStyleJoyDual>());
            EXPECT_EQ(currentDeviceType, previousDeviceType);
        }
    }
}

//!< ジョイコンの割り当てモードを左残しで1本持ちに変更した場合に、溢れたJoy-Con が Dual で接続されることの検証
TEST(NpadJoyAssignmentTest, SwitchToSingleAndGetNewDual)
{
    nn::hid::NpadIdType idSetToSingle;
    bool setToSingleComplete = false;
    int retryCount = 0;

    do
    {
        // テストマシンの構成に応じた Npad の初期化
        nnt::gamepad::Initialize();

        // USB有線通信を無効化
        nnt::gamepad::DisableUsbConnect();

        // コントローラを切断
        nnt::gamepad::DisconnectAll();

        // コントローラを接続
        nnt::gamepad::ConnectAll();

        DumpCurrentStyleAndDeviceType();

        // 全ての NpadId に対してデフォルト設定で NpadJoyAssignmentMode_Single にセットする
        // 2本持ち接続のものをすべて1本持ちに変更
        for (const auto& id : nnt::gamepad::NpadIds)
        {
            if (nn::hid::GetNpadStyleSet(id).Test<nn::hid::NpadStyleJoyDual>() &&
                nn::hid::system::GetNpadDeviceType(id) == (nn::hid::system::NpadDeviceType::JoyConLeft::Mask | nn::hid::system::NpadDeviceType::JoyConRight::Mask))
            {
                idSetToSingle = id;
                nn::hid::SetNpadJoyAssignmentModeSingle(idSetToSingle);
                setToSingleComplete = true;
                break;
            }
        }

        retryCount++;
    } while (setToSingleComplete == false && retryCount < RetryCountMax );

    if (setToSingleComplete == false)
    {
        NN_LOG("Test Skipped\n");
        return;
    }

    EXPECT_TRUE(nn::hid::GetNpadStyleSet(idSetToSingle).Test<nn::hid::NpadStyleJoyLeft>());
    EXPECT_EQ(nn::hid::system::GetNpadDeviceType(idSetToSingle), nn::hid::system::NpadDeviceType::JoyConLeft::Mask);

    for (int i = 0; i < nnt::gamepad::NpadIdCountMax; i++)
    {
        if (g_PreviousStyles[i].IsAllOff() == true)
        {
            EXPECT_TRUE(nn::hid::GetNpadStyleSet(nnt::gamepad::NpadIds[i]).Test<nn::hid::NpadStyleJoyDual>());
            EXPECT_EQ(nn::hid::system::GetNpadDeviceType(nnt::gamepad::NpadIds[i]), nn::hid::system::NpadDeviceType::JoyConRight::Mask);
        }
        break;
    }
}

//!< ジョイコンの割り当てモードを左残しで1本持ちに変更した場合に、溢れたJoy-Con が Dual で接続され、割り当て先も正しくとれることの検証
TEST(NpadJoyAssignmentTest, SwitchToSingleAndGetNewDualWithDestination)
{
    nn::hid::NpadIdType idSetToSingle;
    bool setToSingleComplete = false;
    int retryCount = 0;
    bool isAssigned = false;
    nn::hid::NpadIdType destinationId;

    do
    {
        // テストマシンの構成に応じた Npad の初期化
        nnt::gamepad::Initialize();

        // USB有線通信を無効化
        nnt::gamepad::DisableUsbConnect();

        // コントローラを切断
        nnt::gamepad::DisconnectAll();

        // コントローラを接続
        nnt::gamepad::ConnectAll();

        DumpCurrentStyleAndDeviceType();

        // 全ての NpadId に対してデフォルト設定で NpadJoyAssignmentMode_Single にセットする
        // 2本持ち接続のものをすべて1本持ちに変更
        for (const auto& id : nnt::gamepad::NpadIds)
        {
            if (nn::hid::GetNpadStyleSet(id).Test<nn::hid::NpadStyleJoyDual>() &&
                nn::hid::system::GetNpadDeviceType(id) == (nn::hid::system::NpadDeviceType::JoyConLeft::Mask | nn::hid::system::NpadDeviceType::JoyConRight::Mask))
            {
                idSetToSingle = id;
                isAssigned = nn::hid::SetNpadJoyAssignmentModeSingle(&destinationId, idSetToSingle, nn::hid::NpadJoyDeviceType_Right);
                setToSingleComplete = true;
                EXPECT_TRUE(isAssigned);
                break;
            }
        }

        retryCount++;
    } while (setToSingleComplete == false && retryCount < RetryCountMax );

    if (setToSingleComplete == false)
    {
        NN_LOG("Test Skipped\n");
        return;
    }

    EXPECT_TRUE(nn::hid::GetNpadStyleSet(idSetToSingle).Test<nn::hid::NpadStyleJoyRight>());
    EXPECT_EQ(nn::hid::system::GetNpadDeviceType(idSetToSingle), nn::hid::system::NpadDeviceType::JoyConRight::Mask);

    for (int i = 0; i < nnt::gamepad::NpadIdCountMax; i++)
    {
        if (g_PreviousStyles[i].IsAllOff() == true)
        {
            EXPECT_TRUE(nn::hid::GetNpadStyleSet(nnt::gamepad::NpadIds[i]).Test<nn::hid::NpadStyleJoyDual>());
            EXPECT_EQ(nn::hid::system::GetNpadDeviceType(nnt::gamepad::NpadIds[i]), nn::hid::system::NpadDeviceType::JoyConLeft::Mask);
            EXPECT_EQ(nnt::gamepad::NpadIds[i], destinationId);
        }
        break;
    }
}

//!< ジョイコンの割り当てモードを左残しで1本持ちに変更する
TEST(NpadJoyAssignmentTest, SwitchToSingleWithLeft)
{
    // テストマシンの構成に応じた Npad の初期化
    nnt::gamepad::Initialize();

    // USB有線通信を無効化
    nnt::gamepad::DisableUsbConnect();

    // コントローラを切断
    nnt::gamepad::DisconnectAll();

    // コントローラを接続
    nnt::gamepad::ConnectAll();

    DumpCurrentStyleAndDeviceType();

    // 全ての NpadId に対してデフォルト設定で NpadJoyAssignmentMode_Single にセットする
    // 2本持ち接続のものをすべて1本持ちに変更
    for (const auto& id : nnt::gamepad::NpadIds)
    {
        nn::hid::SetNpadJoyAssignmentModeSingle(id, nn::hid::NpadJoyDeviceType_Left);
    }

    // 意図した Style / DeviceType になっているか検証
    VerifyStyleAndDeviceTypeAfterSwitchToSingle(nn::hid::NpadJoyDeviceType_Left);
}

//!< ジョイコンの割り当てモードを右残しで1本持ちに変更する
TEST(NpadJoyAssignmentTest, SwitchToSingleWithRight)
{
    // テストマシンの構成に応じた Npad の初期化
    nnt::gamepad::Initialize();

    // USB有線通信を無効化
    nnt::gamepad::DisableUsbConnect();

    // コントローラを切断
    nnt::gamepad::DisconnectAll();

    // コントローラを接続
    nnt::gamepad::ConnectAll();

    DumpCurrentStyleAndDeviceType();

    // 全ての NpadId に対してデフォルト設定で NpadJoyAssignmentMode_Single にセットする
    // 2本持ち接続のものをすべて1本持ちに変更
    for (const auto& id : nnt::gamepad::NpadIds)
    {
        nn::hid::SetNpadJoyAssignmentModeSingle(id, nn::hid::NpadJoyDeviceType_Right);
    }

    // 意図した Style / DeviceType になっているか検証
    VerifyStyleAndDeviceTypeAfterSwitchToSingle(nn::hid::NpadJoyDeviceType_Right);
}

//!< ジョイコンのSwap のテスト
TEST(NpadJoyAssignmentTest, Swap)
{
    // テストマシンの構成に応じた Npad の初期化
    nnt::gamepad::Initialize();

    // USB有線通信を無効化
    nnt::gamepad::DisableUsbConnect();

    // コントローラを切断
    nnt::gamepad::DisconnectAll();

    // コントローラを接続
    nnt::gamepad::ConnectAll();

    DumpCurrentStyleAndDeviceType();

    // 入れ替える index
    const int indexFirst = 0;
    const int indexSecond = 1;
    auto npadIdFirst = nnt::gamepad::NpadIds[indexFirst];
    auto npadIdSecond = nnt::gamepad::NpadIds[indexSecond];

    // 入れ替え
    nn::hid::SwapNpadAssignment(npadIdFirst, npadIdSecond);

    // 入れ替え後の検証
    EXPECT_EQ(nn::hid::GetNpadStyleSet(npadIdFirst), g_PreviousStyles[indexSecond]);
    EXPECT_EQ(nn::hid::system::GetNpadDeviceType(npadIdFirst), g_PreviousDeviceTypes[indexSecond]);
    EXPECT_EQ(nn::hid::GetNpadStyleSet(npadIdSecond), g_PreviousStyles[indexFirst]);
    EXPECT_EQ(nn::hid::system::GetNpadDeviceType(npadIdSecond), g_PreviousDeviceTypes[indexFirst]);
}


//!< JoyLeft/JoyRight 状態のジョイコンの Merge のテスト
TEST(NpadJoyAssignmentTest, MergeJoySingle)
{
    bool merged = false;
    int retryCount = 0;

    // 入れ替える index
    nn::hid::NpadIdType npadIdFirst;
    nn::hid::NpadIdType npadIdSecond;

    do
    {
        // テストマシンの構成に応じた Npad の初期化
        nnt::gamepad::Initialize();

        // USB有線通信を無効化
        nnt::gamepad::DisableUsbConnect();

        // コントローラを切断
        nnt::gamepad::DisconnectAll();

        // コントローラを接続
        nnt::gamepad::ConnectAll();

        // 全ての NpadId に対してデフォルト設定で NpadJoyAssignmentMode_Single にセットする
        // 2本持ち接続のものをすべて1本持ちに変更
        for (const auto& id : nnt::gamepad::NpadIds)
        {
            if (nn::hid::GetNpadStyleSet(id).Test<nn::hid::NpadStyleJoyDual>())
            {
                nn::hid::SetNpadJoyAssignmentModeSingle(id);
            }
        }

        for (int i = 0; i < nnt::gamepad::NpadIdCountMax; i++)
        {
            npadIdFirst = nnt::gamepad::NpadIds[i];
            if (nn::hid::GetNpadJoyAssignment(npadIdFirst) == nn::hid::NpadJoyAssignmentMode_Single &&
                nn::hid::system::GetNpadDeviceType(npadIdFirst).Test<nn::hid::system::NpadDeviceType::JoyConLeft>())
            {
                for (int j = 1; j < nnt::gamepad::NpadIdCountMax; j++)
                {
                    npadIdSecond = nnt::gamepad::NpadIds[j];
                    if (nn::hid::GetNpadJoyAssignment(npadIdSecond) == nn::hid::NpadJoyAssignmentMode_Single &&
                        nn::hid::system::GetNpadDeviceType(npadIdSecond).Test<nn::hid::system::NpadDeviceType::JoyConRight>())
                    {
                        merged = true;
                        nn::hid::MergeSingleJoyAsDualJoy(npadIdFirst, npadIdSecond);
                        break;
                    }
                }
                break;
            }
        }

        retryCount++;
    } while (merged == false && retryCount < RetryCountMax);

    if (merged == false)
    {
        NN_LOG("Test Skipped\n");
    }

    // マージの検証
    EXPECT_TRUE(nn::hid::GetNpadStyleSet(npadIdFirst).Test<nn::hid::NpadStyleJoyDual>());
    EXPECT_TRUE(nn::hid::GetNpadStyleSet(npadIdSecond).IsAllOff());
    EXPECT_EQ(nn::hid::system::GetNpadDeviceType(npadIdFirst), nn::hid::system::NpadDeviceType::JoyConLeft::Mask | nn::hid::system::NpadDeviceType::JoyConRight::Mask);
    EXPECT_TRUE(nn::hid::system::GetNpadDeviceType(npadIdSecond).IsAllOff());
}


//!< JoyLeft/JoyRight 状態のジョイコンの Merge のテスト
TEST(NpadJoyAssignmentTest, MergeJoyDual)
{
    bool merged = false;
    int retryCount = 0;

    // 入れ替える index
    nn::hid::NpadIdType npadIdFirst;
    nn::hid::NpadIdType npadIdSecond;

    do
    {
        // テストマシンの構成に応じた Npad の初期化
        nnt::gamepad::Initialize();

        // USB有線通信を無効化
        nnt::gamepad::DisableUsbConnect();

        // コントローラを切断
        nnt::gamepad::DisconnectAll();

        // コントローラを接続
        nnt::gamepad::ConnectAll();

        // 全ての NpadId に対してデフォルト設定で NpadJoyAssignmentMode_Single にセットする
        // 2本持ち接続のものをすべて1本持ちに変更
        for (const auto& id : nnt::gamepad::NpadIds)
        {
            if (nn::hid::GetNpadStyleSet(id).Test<nn::hid::NpadStyleJoyDual>())
            {
                nn::hid::SetNpadJoyAssignmentModeSingle(id);
                // すかさず Dual に戻す
                nn::hid::SetNpadJoyAssignmentModeDual(id);
            }
        }

        for (int i = 0; i < nnt::gamepad::NpadIdCountMax; i++)
        {
            npadIdFirst = nnt::gamepad::NpadIds[i];
            if (nn::hid::GetNpadJoyAssignment(npadIdFirst) == nn::hid::NpadJoyAssignmentMode_Dual &&
                nn::hid::system::GetNpadDeviceType(npadIdFirst) == nn::hid::system::NpadDeviceType::JoyConLeft::Mask)
            {
                for (int j = 1; j < nnt::gamepad::NpadIdCountMax; j++)
                {
                    npadIdSecond = nnt::gamepad::NpadIds[j];
                    if (nn::hid::GetNpadJoyAssignment(npadIdSecond) == nn::hid::NpadJoyAssignmentMode_Dual &&
                        nn::hid::system::GetNpadDeviceType(npadIdSecond) == nn::hid::system::NpadDeviceType::JoyConRight::Mask)
                    {
                        merged = true;
                        nn::hid::MergeSingleJoyAsDualJoy(npadIdFirst, npadIdSecond);
                        break;
                    }
                }
                break;
            }
        }

        retryCount++;
    } while (merged == false && retryCount < RetryCountMax);

    if (merged == false)
    {
        NN_LOG("Test Skipped\n");
    }

    // マージの検証
    EXPECT_TRUE(nn::hid::GetNpadStyleSet(npadIdFirst).Test<nn::hid::NpadStyleJoyDual>());
    EXPECT_TRUE(nn::hid::GetNpadStyleSet(npadIdSecond).IsAllOff());
    EXPECT_EQ(nn::hid::system::GetNpadDeviceType(npadIdFirst), nn::hid::system::NpadDeviceType::JoyConLeft::Mask | nn::hid::system::NpadDeviceType::JoyConRight::Mask);
    EXPECT_TRUE(nn::hid::system::GetNpadDeviceType(npadIdSecond).IsAllOff());
}

