﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include "../../Common/testGamePad_Common.h"
#include "../../Common/testGamePad_IrSensor.h"

namespace nnt { namespace irsensor {
    nn::irsensor::IrCameraHandle IrSensorTest::s_Handles[nnt::gamepad::NpadIdCountMax];
    int IrSensorTest::s_HandleCount;
}} // namaspace nnt::irsensor

namespace
{

const int VerifyStatesTestCountMax = 5;                                         // ステータスのテストを実施する回数
const nn::TimeSpan VerifyStatesTestLimitTime = nn::TimeSpan::FromSeconds(5);    // ステータスが更新されない状態の許容時間
const int NotReadyWaitLoopCountMax = 1000;                                      // ステータス取得の準備が完了するまでに待つ最大のループ回数
const nn::TimeSpan PollingInterval = nn::TimeSpan::FromMilliSeconds(15);        // ステータス取得の準備が完了していない時のポーリング間隔

nn::irsensor::IrCameraLightTarget LightTargets[] =
{
    nn::irsensor::IrCameraLightTarget_AllObjects,
    nn::irsensor::IrCameraLightTarget_FarObjects,
    nn::irsensor::IrCameraLightTarget_NearObjects,
    nn::irsensor::IrCameraLightTarget_None
};

class IrLedProcessorTest : public nnt::irsensor::IrSensorTest
{
protected:
    void DoTestRunningProcessor(const nn::irsensor::IrCameraHandle& handle) NN_NOEXCEPT;
    void DoTestStopImageProcessor(const nn::irsensor::IrCameraHandle& handle) NN_NOEXCEPT;
    void DoTestReadingIrLedStates(const nn::irsensor::IrCameraHandle& handle, nn::irsensor::IrLedProcessorConfig config, const int testCount) NN_NOEXCEPT;
};

//==================================================
// プロセッサの起動までの時間を計測
//==================================================
void IrLedProcessorTest::DoTestRunningProcessor(const nn::irsensor::IrCameraHandle& handle) NN_NOEXCEPT
{
    int counter = 0;
    nn::TimeSpanType startTime = nn::os::GetSystemTick().ToTimeSpan();      // 計測を開始した時間を記憶

    while (NN_STATIC_CONDITION(true))
    {
        auto status = nn::irsensor::GetImageProcessorStatus(handle);

        if (status == nn::irsensor::ImageProcessorStatus_Stopped)
        {
            counter++;
            nn::os::SleepThread(PollingInterval);
            ASSERT_LT(counter, NotReadyWaitLoopCountMax);
        }
        else if (status == nn::irsensor::ImageProcessorStatus_Running)
        {
            NN_LOG("#     ImageProcessorStatus [ Stopped -> Running ] (%lld ms)\n", nn::os::GetSystemTick().ToTimeSpan().GetMilliSeconds() - startTime.GetMilliSeconds());
            break;
        }
    }
    NNT_IRSENSOR_EXIT_SUCCESS;
}

//==================================================
// プロセッサの停止までの時間を計測
//==================================================
void IrLedProcessorTest::DoTestStopImageProcessor(const nn::irsensor::IrCameraHandle& handle) NN_NOEXCEPT
{
    nn::irsensor::StopImageProcessorAsync(handle);

    auto counter = 0;
    nn::TimeSpanType startTime = nn::os::GetSystemTick().ToTimeSpan();

    while (NN_STATIC_CONDITION(true))
    {
        auto status = nn::irsensor::GetImageProcessorStatus(handle);

        if (status == nn::irsensor::ImageProcessorStatus_Running)
        {
            counter++;
            nn::os::SleepThread(PollingInterval);
            ASSERT_LT(counter, NotReadyWaitLoopCountMax);
        }
        else
        {
            NN_LOG("#     StopImageProcessor (%lld ms)\n", nn::os::GetSystemTick().ToTimeSpan().GetMilliSeconds() - startTime.GetMilliSeconds());
            break;
        }
    }
    NNT_IRSENSOR_EXIT_SUCCESS;
}

//==================================================
// ステータスの取得と更新時間の計測
//==================================================
void IrLedProcessorTest::DoTestReadingIrLedStates(const nn::irsensor::IrCameraHandle& handle, nn::irsensor::IrLedProcessorConfig config, const int testCount) NN_NOEXCEPT
{
    NN_LOG("# ************************************************************\n");
    NN_LOG("#     ReadingIrLedStatesTest (Target : %d)\n", config.lightTarget);
    NN_LOG("# ************************************************************\n");

    //-------------------------------------------------------------------------------
    // プロセッサが起動するまでの時間の計測します
    //-------------------------------------------------------------------------------
    nn::irsensor::RunIrLedProcessor(handle, config);
    DoTestRunningProcessor(handle);

    //-------------------------------------------------------------------------------
    // ステートが更新されるまでにかかる時間を計測します
    //-------------------------------------------------------------------------------
    for (int i = 0; i < testCount; ++i)
    {
        auto startTime = nn::os::GetSystemTick().ToTimeSpan();                       // 計測を開始した時間を記憶

        NN_LOG("#     Test : [%2d/%2d]\n", i + 1, testCount);
        NN_LOG("#         CallGetIrLedProcessorStates\n");

        while (NN_STATIC_CONDITION(true))
        {
            nn::Result result = nn::irsensor::GetIrLedProcessorState(handle);
            if (nn::irsensor::ResultIrsensorNotReady::Includes(result) == false)
            {
                if (nn::irsensor::ResultIrsensorUnconnected::Includes(result))
                {
                    NN_LOG("############# ResultIrsensorUnconnected #############\n");
                    return;
                }
                if (nn::irsensor::ResultIrsensorUnsupported::Includes(result))
                {
                    NN_LOG("############# ResultIrsensorUnsupported #############\n");
                    return;
                }
                if (nn::irsensor::ResultIrsensorDeviceError::Includes(result))
                {
                    NN_LOG("############# ResultIrsensorDeviceError #############\n");
                    ADD_FAILURE();
                    return;
                }
                break;
            }
            ASSERT_LT(nn::os::GetSystemTick().ToTimeSpan().GetMilliSeconds() - startTime.GetMilliSeconds(), VerifyStatesTestLimitTime.GetMilliSeconds());
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1));
        }
        NN_LOG("#             RunningIrLedProcessorStates (%lld ms)\n", nn::os::GetSystemTick().ToTimeSpan().GetMilliSeconds() - startTime.GetMilliSeconds());
    }
    NN_LOG("#     SUCCESS\n");
    NNT_IRSENSOR_EXIT_SUCCESS;
}

//==================================================
// IRセンサの入力状態が正しく取得できるか
//==================================================
TEST_F(IrLedProcessorTest, StateReading)
{
    nnt::gamepad::Initialize();

    // ProCon USB 無効
    ::nnt::gamepad::DisableUsbConnect();

    // コントローラの再接続
    nnt::gamepad::DisconnectAll();
    nnt::gamepad::ConnectAll();

    NN_LOG("#\n# Start IrLed Processor Test.\n#\n");
    for (const auto& handle : nnt::irsensor::IrSensorTest::s_Handles)
    {
        // IRカメラが利用できない場合はスキップ
        if (GetIrCameraStatus(handle) != nn::irsensor::IrCameraStatus_Available)
        {
            continue;
        }

        NN_LOG("#-----------------------------------------------\n");
        NN_LOG("# IrCameraHandle : 0x%x\n", handle);
        NN_LOG("#-----------------------------------------------\n");

        nn::irsensor::IrLedProcessorConfig config;
        nn::irsensor::GetIrLedProcessorDefaultConfig(&config);
        NNT_IRSENSOR_EXPECT_EXIT(DoTestReadingIrLedStates(handle, config, VerifyStatesTestCountMax), NNT_IRSENSOR_EXIT_0, "");
        NNT_IRSENSOR_EXPECT_EXIT(DoTestStopImageProcessor(handle), NNT_IRSENSOR_EXIT_0, "");
        for (const auto& target : LightTargets)
        {
            config.lightTarget = target;
            NNT_IRSENSOR_EXPECT_EXIT(DoTestReadingIrLedStates(handle, config, 1), NNT_IRSENSOR_EXIT_0, "");
            NNT_IRSENSOR_EXPECT_EXIT(DoTestStopImageProcessor(handle), NNT_IRSENSOR_EXIT_0, "");
        }
    }
    nnt::gamepad::DisconnectAll();
}

//==================================================
// ProコントローラのUSB有線接続が有効時にIRセンサの入力状態が正しく取得できるか
//==================================================
TEST_F(IrLedProcessorTest, StateReadingForUsb)
{
    nnt::gamepad::Initialize();

    // ProCon USB 有効
    ::nnt::gamepad::EnableUsbConnect();

    // コントローラの再接続
    nnt::gamepad::DisconnectAll();
    nnt::gamepad::ConnectAll();

    NN_LOG("#\n# Start IrLed Processor Test.\n#\n");
    for (const auto& handle : nnt::irsensor::IrSensorTest::s_Handles)
    {
        // IRカメラが利用できない場合はスキップ
        if (GetIrCameraStatus(handle) != nn::irsensor::IrCameraStatus_Available)
        {
            continue;
        }

        NN_LOG("#-----------------------------------------------\n");
        NN_LOG("# IrCameraHandle : 0x%x\n", handle);
        NN_LOG("#-----------------------------------------------\n");

        nn::irsensor::IrLedProcessorConfig config;
        nn::irsensor::GetIrLedProcessorDefaultConfig(&config);
        NNT_IRSENSOR_EXPECT_EXIT(DoTestReadingIrLedStates(handle, config, VerifyStatesTestCountMax), NNT_IRSENSOR_EXIT_0, "");
        NNT_IRSENSOR_EXPECT_EXIT(DoTestStopImageProcessor(handle), NNT_IRSENSOR_EXIT_0, "");
        for (const auto& target : LightTargets)
        {
            config.lightTarget = target;
            NNT_IRSENSOR_EXPECT_EXIT(DoTestReadingIrLedStates(handle, config, 1), NNT_IRSENSOR_EXIT_0, "");
            NNT_IRSENSOR_EXPECT_EXIT(DoTestStopImageProcessor(handle), NNT_IRSENSOR_EXIT_0, "");
        }
    }
    nnt::gamepad::DisconnectAll();
}

} // namespace
