﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include "../../Common/testGamePad_Common.h"
#include "../../Common/testGamePad_IrSensor.h"

namespace nnt { namespace irsensor {
    nn::irsensor::IrCameraHandle IrSensorTest::s_Handles[nnt::gamepad::NpadIdCountMax];
    int IrSensorTest::s_HandleCount;
}} // namaspace nnt::irsensor

namespace
{

nn::irsensor::ImageTransferProcessorFormat ImageFormats[] =
{
    nn::irsensor::ImageTransferProcessorFormat::ImageTransferProcessorFormat_320x240,
    nn::irsensor::ImageTransferProcessorFormat::ImageTransferProcessorFormat_160x120,
    nn::irsensor::ImageTransferProcessorFormat::ImageTransferProcessorFormat_80x60,
    nn::irsensor::ImageTransferProcessorFormat::ImageTransferProcessorFormat_40x30,
    nn::irsensor::ImageTransferProcessorFormat::ImageTransferProcessorFormat_20x15
};

NN_ALIGNAS(0x1000)
uint8_t WorkBuffer[nn::irsensor::ImageTransferProcessorWorkBufferSize320x240];

uint8_t ImageBuffer[nn::irsensor::ImageTransferProcessorImageSize320x240];

const int VerifyStatesTestCountMax = 5;                                         // ステータスのテストを実施する回数
const nn::TimeSpan VerifyStatesTestLimitTime = nn::TimeSpan::FromSeconds(15);   // ステータスが更新されない状態の許容時間
const int NotReadyWaitLoopCountMax = 1000;                                      // ステータス取得の準備が完了するまでに待つ最大のループ回数
const nn::TimeSpan PollingInterval = nn::TimeSpan::FromMilliSeconds(15);        // ステータス取得の準備が完了していない時のポーリング間隔

class ImageTransferProcessorTest : public ::nnt::irsensor::IrSensorTest
{
protected:
    void DoTestRunningProcessor(const nn::irsensor::IrCameraHandle& handle) NN_NOEXCEPT;
    void DoTestStopImageProcessor(const nn::irsensor::IrCameraHandle& handle) NN_NOEXCEPT;
    void DoTestVerifyImageTransferState(const nn::irsensor::ImageTransferProcessorState& state) NN_NOEXCEPT;

    void DoTestReadingImageTransferState(const nn::irsensor::IrCameraHandle& handle, const nn::irsensor::ImageTransferProcessorConfig& config, const int testCount) NN_NOEXCEPT;
    void DoTestReadingImageTransferState(const nn::irsensor::IrCameraHandle& handle, const nn::irsensor::ImageTransferProcessorExConfig& config, const int testCount) NN_NOEXCEPT;

    void DoTestFormatBoundary(const nn::irsensor::IrCameraHandle& handle, nn::irsensor::ImageTransferProcessorFormat format) NN_NOEXCEPT;
    void ReadingStateTest(const nn::irsensor::IrCameraHandle& handle) NN_NOEXCEPT;
};

//==================================================
// プロセッサの起動までの時間を計測
//==================================================
void ImageTransferProcessorTest::DoTestRunningProcessor(const nn::irsensor::IrCameraHandle& handle) NN_NOEXCEPT
{
    int counter = 0;
    nn::TimeSpanType startTime = nn::os::GetSystemTick().ToTimeSpan();      // 計測を開始した時間を記憶

    while (NN_STATIC_CONDITION(true))
    {
        auto status = nn::irsensor::GetImageProcessorStatus(handle);

        if (status == nn::irsensor::ImageProcessorStatus_Stopped)
        {
            counter++;
            nn::os::SleepThread(PollingInterval);
            ASSERT_LT(counter, NotReadyWaitLoopCountMax);
        }
        else if (status == nn::irsensor::ImageProcessorStatus_Running)
        {
            NN_LOG("#     ImageProcessorStatus [ Stopped -> Running ] (%lld ms)\n", nn::os::GetSystemTick().ToTimeSpan().GetMilliSeconds() - startTime.GetMilliSeconds());
            break;
        }
    }
    NNT_IRSENSOR_EXIT_SUCCESS;
}

//==================================================
// プロセッサの停止までの時間を計測
//==================================================
void ImageTransferProcessorTest::DoTestStopImageProcessor(const nn::irsensor::IrCameraHandle& handle) NN_NOEXCEPT
{
    nn::irsensor::StopImageProcessorAsync(handle);

    auto counter = 0;
    nn::TimeSpanType startTime = nn::os::GetSystemTick().ToTimeSpan();

    while (NN_STATIC_CONDITION(true))
    {
        auto status = nn::irsensor::GetImageProcessorStatus(handle);

        if (status == nn::irsensor::ImageProcessorStatus_Running)
        {
            counter++;
            nn::os::SleepThread(PollingInterval);
            ASSERT_LT(counter, NotReadyWaitLoopCountMax);
        }
        else
        {
            NN_LOG("#     StopImageProcessor (%lld ms)\n", nn::os::GetSystemTick().ToTimeSpan().GetMilliSeconds() - startTime.GetMilliSeconds());
            break;
        }
    }
    NNT_IRSENSOR_EXIT_SUCCESS;
}

//==================================================
// ステータスの検証
//==================================================
void ImageTransferProcessorTest::DoTestVerifyImageTransferState(const nn::irsensor::ImageTransferProcessorState& state) NN_NOEXCEPT
{
    ASSERT_GE(state.samplingNumber, 0);

    ASSERT_GE(state.ambientNoiseLevel, nn::irsensor::IrCameraAmbientNoiseLevel_Low);
    ASSERT_LE(state.ambientNoiseLevel, nn::irsensor::IrCameraAmbientNoiseLevel_Unknown);

    NNT_IRSENSOR_EXIT_SUCCESS;
}

//==================================================
// ステータスの取得と更新時間の計測
//==================================================
void ImageTransferProcessorTest::DoTestReadingImageTransferState(const nn::irsensor::IrCameraHandle& handle, const nn::irsensor::ImageTransferProcessorConfig& config, const int testCount) NN_NOEXCEPT
{
    NN_LOG("#     ReadingImageTransferProcessorStateTest\n");
    int64_t prevSamplingNumber = -1;

    //-------------------------------------------------------------------------------
    // ワークバッファのサイズを計算します
    //-------------------------------------------------------------------------------
    int workBufferSize = 0;
    int imageBufferSize = 0;
    switch (config.format)
    {
    case nn::irsensor::ImageTransferProcessorFormat_320x240:
        workBufferSize = nn::irsensor::ImageTransferProcessorWorkBufferSize320x240;
        imageBufferSize = nn::irsensor::ImageTransferProcessorImageSize320x240;
        break;
    case nn::irsensor::ImageTransferProcessorFormat_160x120:
        workBufferSize = nn::irsensor::ImageTransferProcessorWorkBufferSize160x120;
        imageBufferSize = nn::irsensor::ImageTransferProcessorImageSize160x120;
        break;
    case nn::irsensor::ImageTransferProcessorFormat_80x60:
        workBufferSize = nn::irsensor::ImageTransferProcessorWorkBufferSize80x60;
        imageBufferSize = nn::irsensor::ImageTransferProcessorImageSize80x60;
        break;
    case nn::irsensor::ImageTransferProcessorFormat_40x30:
        workBufferSize = nn::irsensor::ImageTransferProcessorWorkBufferSize40x30;
        imageBufferSize = nn::irsensor::ImageTransferProcessorImageSize40x30;
        break;
    case nn::irsensor::ImageTransferProcessorFormat_20x15:
        workBufferSize = nn::irsensor::ImageTransferProcessorWorkBufferSize20x15;
        imageBufferSize = nn::irsensor::ImageTransferProcessorImageSize20x15;
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }

    //-------------------------------------------------------------------------------
    // プロセッサが起動するまでの時間の計測します
    //-------------------------------------------------------------------------------
    nn::irsensor::RunImageTransferProcessor(handle, config, WorkBuffer, workBufferSize);
    DoTestRunningProcessor(handle);

    //-------------------------------------------------------------------------------
    // ステートが更新されるまでにかかる時間を計測します
    //-------------------------------------------------------------------------------
    for (int i = 0; i < testCount; ++i)
    {
        nn::irsensor::ImageTransferProcessorState state;
        auto samplingNumber = prevSamplingNumber;
        auto startTime = nn::os::GetSystemTick().ToTimeSpan();                       // 計測を開始した時間を記憶

        NN_LOG("#     Test : [%2d/%2d]\n", i + 1, VerifyStatesTestCountMax);
        NN_LOG("#         CallGetImageTransferProcessorStates\n");

        while (NN_STATIC_CONDITION(true))
        {
            #if (false)
            nn::Result result = nn::irsensor::GetImageTransferProcessorState(&state, ImageBuffer, imageBufferSize, handle);
            #else
            nn::Result result = nn::irsensor::GetImageTransferProcessorState(&state, ImageBuffer, nn::irsensor::ImageTransferProcessorImageSize320x240, handle);
            #endif
            if (nn::irsensor::ResultIrsensorNotReady::Includes(result) == false)
            {
                if (nn::irsensor::ResultIrsensorDeviceError::Includes(result))
                {
                    NN_LOG("############# ResultIrsensorDeviceError #############\n");
                    ADD_FAILURE();
                    return;
                }
                samplingNumber = state.samplingNumber;
            }
            if (prevSamplingNumber != samplingNumber)
            {
                break;
            }
            ASSERT_LT(nn::os::GetSystemTick().ToTimeSpan().GetMilliSeconds() - startTime.GetMilliSeconds(), VerifyStatesTestLimitTime.GetMilliSeconds());
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1));
        }
        NN_LOG("#             UpdateImageTransferProcessorStates (%lld ms)\n", nn::os::GetSystemTick().ToTimeSpan().GetMilliSeconds() - startTime.GetMilliSeconds());
        NN_LOG("#             SamplingNumber %lld -> %lld\n", prevSamplingNumber, samplingNumber);
        prevSamplingNumber = samplingNumber;
        //-------------------------------------------------------------------------------
        // ステータスの検証を行います
        //-------------------------------------------------------------------------------
        NN_LOG("#         VerifyImageTransferState\n");
        DoTestVerifyImageTransferState(state);
    }
    NN_LOG("#     SUCCESS\n");
    NNT_IRSENSOR_EXIT_SUCCESS;
}

//==================================================
// ステータスの取得と更新時間の計測
//==================================================
void ImageTransferProcessorTest::DoTestReadingImageTransferState(const nn::irsensor::IrCameraHandle& handle, const nn::irsensor::ImageTransferProcessorExConfig& config, const int testCount) NN_NOEXCEPT
{
    NN_LOG("#     [EX] ReadingImageTransferProcessorStateTest\n");
    int64_t prevSamplingNumber = -1;

    //-------------------------------------------------------------------------------
    // ワークバッファのサイズを計算します
    //-------------------------------------------------------------------------------
    int workBufferSize = 0;
    int imageBufferSize = 0;
    switch (config.origFormat)
    {
    case nn::irsensor::ImageTransferProcessorFormat_320x240:
        workBufferSize = nn::irsensor::ImageTransferProcessorWorkBufferSize320x240;
        imageBufferSize = nn::irsensor::ImageTransferProcessorImageSize320x240;
        break;
    case nn::irsensor::ImageTransferProcessorFormat_160x120:
        workBufferSize = nn::irsensor::ImageTransferProcessorWorkBufferSize160x120;
        imageBufferSize = nn::irsensor::ImageTransferProcessorImageSize160x120;
        break;
    case nn::irsensor::ImageTransferProcessorFormat_80x60:
        workBufferSize = nn::irsensor::ImageTransferProcessorWorkBufferSize80x60;
        imageBufferSize = nn::irsensor::ImageTransferProcessorImageSize80x60;
        break;
    case nn::irsensor::ImageTransferProcessorFormat_40x30:
        workBufferSize = nn::irsensor::ImageTransferProcessorWorkBufferSize40x30;
        imageBufferSize = nn::irsensor::ImageTransferProcessorImageSize40x30;
        break;
    case nn::irsensor::ImageTransferProcessorFormat_20x15:
        workBufferSize = nn::irsensor::ImageTransferProcessorWorkBufferSize20x15;
        imageBufferSize = nn::irsensor::ImageTransferProcessorImageSize20x15;
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }

    //-------------------------------------------------------------------------------
    // プロセッサが起動するまでの時間の計測します
    //-------------------------------------------------------------------------------
    nn::irsensor::RunImageTransferProcessor(handle, config, WorkBuffer, workBufferSize);
    DoTestRunningProcessor(handle);

    //-------------------------------------------------------------------------------
    // ステートが更新されるまでにかかる時間を計測します
    //-------------------------------------------------------------------------------
    for (int i = 0; i < testCount; ++i)
    {
        nn::irsensor::ImageTransferProcessorState state;
        auto samplingNumber = prevSamplingNumber;
        auto startTime = nn::os::GetSystemTick().ToTimeSpan();                       // 計測を開始した時間を記憶

        NN_LOG("#     Test : [%2d/%2d]\n", i + 1, VerifyStatesTestCountMax);
        NN_LOG("#         CallGetImageTransferProcessorStates\n");

        while (NN_STATIC_CONDITION(true))
        {
            #if (false)
            nn::Result result = nn::irsensor::GetImageTransferProcessorState(&state, ImageBuffer, imageBufferSize, handle);
            #else
            nn::Result result = nn::irsensor::GetImageTransferProcessorState(&state, ImageBuffer, nn::irsensor::ImageTransferProcessorImageSize320x240, handle);
            #endif
            if (nn::irsensor::ResultIrsensorNotReady::Includes(result) == false)
            {
                if (nn::irsensor::ResultIrsensorDeviceError::Includes(result))
                {
                    NN_LOG("############# ResultIrsensorDeviceError #############\n");
                    ADD_FAILURE();
                    return;
                }
                samplingNumber = state.samplingNumber;
            }
            if (prevSamplingNumber != samplingNumber)
            {
                break;
            }
            ASSERT_LT(nn::os::GetSystemTick().ToTimeSpan().GetMilliSeconds() - startTime.GetMilliSeconds(), VerifyStatesTestLimitTime.GetMilliSeconds());
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1));
        }
        NN_LOG("#             UpdateImageTransferProcessorStates (%lld ms)\n", nn::os::GetSystemTick().ToTimeSpan().GetMilliSeconds() - startTime.GetMilliSeconds());
        NN_LOG("#             SamplingNumber %lld -> %lld\n", prevSamplingNumber, samplingNumber);
        prevSamplingNumber = samplingNumber;
        //-------------------------------------------------------------------------------
        // ステータスの検証を行います
        //-------------------------------------------------------------------------------
        NN_LOG("#         VerifyImageTransferState\n");
        DoTestVerifyImageTransferState(state);
    }
    NN_LOG("#     SUCCESS\n");
    NNT_IRSENSOR_EXIT_SUCCESS;
}

//==================================================
// 処理対象の矩形領域を変更
//==================================================
void ImageTransferProcessorTest::DoTestFormatBoundary(const nn::irsensor::IrCameraHandle& handle, nn::irsensor::ImageTransferProcessorFormat format) NN_NOEXCEPT
{
    NN_LOG("# ************************************************************\n");
    NN_LOG("#     FormatBoundaryTest (format: %d)\n", format);
    NN_LOG("# ************************************************************\n");
    EXPECT_LT(0, s_HandleCount);

    nn::irsensor::ImageTransferProcessorConfig config;
    nn::irsensor::GetImageTransferProcessorDefaultConfig(&config);

    config.format = format;

    DoTestReadingImageTransferState(handle, config, 1);

    DoTestStopImageProcessor(handle);

    NNT_IRSENSOR_EXIT_SUCCESS;
}

//==================================================
// ステータス取得のテスト
//==================================================
void ImageTransferProcessorTest::ReadingStateTest(const nn::irsensor::IrCameraHandle& handle) NN_NOEXCEPT
{
    EXPECT_LT(0, s_HandleCount);

    nn::irsensor::ImageTransferProcessorConfig config;
    nn::irsensor::GetImageTransferProcessorDefaultConfig(&config);

    DoTestReadingImageTransferState(handle, config, VerifyStatesTestCountMax);

    // 画像サイズを切り替えてプロセッサの起動とステータス取得が出来るか
    for (auto format : ImageFormats)
    {
        DoTestFormatBoundary(handle, format);
    }

    nn::irsensor::ImageTransferProcessorExConfig configEx;
    nn::irsensor::GetImageTransferProcessorDefaultConfig(&configEx);

    NN_LOG("# ************************************************************\n");
    NN_LOG("#     ImageTransferProcessorExTest\n");
    NN_LOG("# ************************************************************\n");
    DoTestReadingImageTransferState(handle, configEx, VerifyStatesTestCountMax);

    DoTestStopImageProcessor(handle);

    NNT_IRSENSOR_EXIT_SUCCESS;
}

//==================================================
// ステータス取得のテスト
//==================================================
TEST_F(ImageTransferProcessorTest, StateReading)
{
    nnt::gamepad::Initialize();

    // コントローラの再接続
    nnt::gamepad::DisconnectAll();
    nnt::gamepad::ConnectAll();

    NN_LOG("#\n# Start ImageTransfer Processor Test.\n#\n");
    for (const auto& handle : nnt::irsensor::IrSensorTest::s_Handles)
    {
        // IRカメラが利用できない場合はスキップ
        if (GetIrCameraStatus(handle) != nn::irsensor::IrCameraStatus_Available)
        {
            continue;
        }

        NN_LOG("#-----------------------------------------------\n");
        NN_LOG("# IrCameraHandle : 0x%x\n", handle);
        NN_LOG("#-----------------------------------------------\n");
        ReadingStateTest(handle);
    }
    nnt::gamepad::DisconnectAll();
}

// USB有線接続が有効時にステータスが正常に取得できるか
TEST_F(ImageTransferProcessorTest, StateReadingForUsb)
{
    nnt::gamepad::Initialize();

    // USB 有線接続を有効にする
    ::nnt::gamepad::EnableUsbConnect();

    // コントローラの再接続
    nnt::gamepad::DisconnectAll();
    nnt::gamepad::ConnectAll();

    NN_LOG("#\n# Start ImageTransfer Processor Test.\n#\n");
    for (const auto& handle : nnt::irsensor::IrSensorTest::s_Handles)
    {
        // IRカメラが利用できない場合はスキップ
        if (GetIrCameraStatus(handle) != nn::irsensor::IrCameraStatus_Available)
        {
            continue;
        }

        NN_LOG("#-----------------------------------------------\n");
        NN_LOG("# IrCameraHandle : 0x%x\n", handle);
        NN_LOG("#-----------------------------------------------\n");
        ReadingStateTest(handle);
    }

    nnt::gamepad::DisconnectAll();
}

} // namespace
