﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include "../../Common/testGamePad_Common.h"
#include "../../Common/testGamePad_IrSensor.h"

namespace nnt { namespace irsensor {
    nn::irsensor::IrCameraHandle IrSensorTest::s_Handles[nnt::gamepad::NpadIdCountMax];
    int IrSensorTest::s_HandleCount;
}} // namaspace nnt::irsensor

namespace
{

const int VerifyStatesTestCountMax = 5;                                         // ステータスのテストを実施する回数
const nn::TimeSpan VerifyStatesTestLimitTime = nn::TimeSpan::FromSeconds(5);    // ステータスが更新されない状態の許容時間
const int NotReadyWaitLoopCountMax = 1000;                                      // ステータス取得の準備が完了するまでに待つ最大のループ回数
const nn::TimeSpan PollingInterval = nn::TimeSpan::FromMilliSeconds(15);        // ステータス取得の準備が完了していない時のポーリング間隔

class HandAnalysisProcessorTest : public nnt::irsensor::IrSensorTest
{
protected:

    void DoTestRunningProcessor(const nn::irsensor::IrCameraHandle& handle) NN_NOEXCEPT;

    void DoTestReadingHandAnalysisSilhouetteStates(const nn::irsensor::IrCameraHandle& handle, nn::irsensor::HandAnalysisConfig config) NN_NOEXCEPT;

    void DoTestReadingHandAnalysisImageStates(const nn::irsensor::IrCameraHandle& handle, nn::irsensor::HandAnalysisConfig config) NN_NOEXCEPT;

    void DoTestVerifyHandAnalysisSilhouetteStates(const nn::irsensor::HandAnalysisSilhouetteState& state) NN_NOEXCEPT;

    void DoTestVerifyHandAnalysisImageStates(const nn::irsensor::HandAnalysisImageState& state) NN_NOEXCEPT;

    void DoTestStopImageProcessor(const nn::irsensor::IrCameraHandle& handle) NN_NOEXCEPT;

    void ReadingStateTest(const nn::irsensor::IrCameraHandle& handle, const nn::irsensor::HandAnalysisMode& mode) NN_NOEXCEPT;
};

//==================================================
// プロセッサの起動までの時間を計測
//==================================================

void HandAnalysisProcessorTest::DoTestRunningProcessor(const nn::irsensor::IrCameraHandle& handle) NN_NOEXCEPT
{
    int counter = 0;
    nn::TimeSpanType startTime = nn::os::GetSystemTick().ToTimeSpan();      // 計測を開始した時間を記憶

    while (NN_STATIC_CONDITION(true))
    {
        auto status = nn::irsensor::GetImageProcessorStatus(handle);

        if (status == nn::irsensor::ImageProcessorStatus_Stopped)
        {
            counter++;
            nn::os::SleepThread(PollingInterval);
            ASSERT_LT(counter, NotReadyWaitLoopCountMax);
        }
        else if (status == nn::irsensor::ImageProcessorStatus_Running)
        {
            NN_LOG("#         Image processor started running. (%lld ms)\n", nn::os::GetSystemTick().ToTimeSpan().GetMilliSeconds() - startTime.GetMilliSeconds());
            break;
        }
    }
    NNT_IRSENSOR_EXIT_SUCCESS;
}

//==================================================
// 状態取得
//==================================================

void HandAnalysisProcessorTest::DoTestReadingHandAnalysisSilhouetteStates(const nn::irsensor::IrCameraHandle& handle, nn::irsensor::HandAnalysisConfig config) NN_NOEXCEPT
{
    NN_LOG("#     <GetHandAnalysisSilhouetteState>\n");
    int64_t prevSamplingNumber = -1;

    for (int i = 0; i < VerifyStatesTestCountMax; ++i)
    {
        nn::irsensor::HandAnalysisSilhouetteState state[nn::irsensor::HandAnalysisProcessorStateCountMax];

        NN_LOG("#     Test : [%2d/%2d]\n", i + 1, VerifyStatesTestCountMax);
        //-------------------------------------------------------------------------------
        // プロセッサが起動するまでの時間の計測します
        //-------------------------------------------------------------------------------
        if (i == 0)
        {
            // 初回のみ実行
            RunHandAnalysis(handle, config);
            DoTestRunningProcessor(handle);
        }
        //-------------------------------------------------------------------------------
        // ステートが更新されるまでにかかる時間を計測します
        //-------------------------------------------------------------------------------
        auto samplingNumber = prevSamplingNumber;
        auto stateCount = 0;
        auto startTime = nn::os::GetSystemTick().ToTimeSpan();                       // 計測を開始した時間を記憶

        while(NN_STATIC_CONDITION(true))
        {
            nn::Result result = GetHandAnalysisSilhouetteState(state, &stateCount, nn::irsensor::HandAnalysisProcessorStateCountMax, 0, handle);
            if (nn::irsensor::ResultIrsensorNotReady::Includes(result) == false)
            {
                if (nn::irsensor::ResultIrsensorUnavailable::Includes(result))
                {
                    NN_LOG("#             ResultIrsensorUnavailable\n");
                    NNT_IRSENSOR_EXIT_SUCCESS;
                    return;
                }
                if (nn::irsensor::ResultIrsensorDeviceError::Includes(result))
                {
                    NN_LOG("############# ResultIrsensorDeviceError #############\n");
                    return;
                }
                if (nn::irsensor::ResultHandAnalysisModeIncorrect::Includes(result))
                {
                    NN_LOG("#             ResultHandAnalysisModeIncorrect\n");
                    return;
                }
                samplingNumber = state[0].samplingNumber;
            }
            if (prevSamplingNumber != samplingNumber)
            {
                break;
            }
            ASSERT_LT(nn::os::GetSystemTick().ToTimeSpan().GetMilliSeconds() - startTime.GetMilliSeconds(), VerifyStatesTestLimitTime.GetMilliSeconds());
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1));
        }

        NN_LOG("#             UpdateHandAnalysisState (%lld ms)\n", nn::os::GetSystemTick().ToTimeSpan().GetMilliSeconds() - startTime.GetMilliSeconds());
        NN_LOG("#             SamplingNumber %lld -> %lld\n", prevSamplingNumber, samplingNumber);
        prevSamplingNumber = samplingNumber;
        //-------------------------------------------------------------------------------
        // ステートの妥当性を検証します
        //-------------------------------------------------------------------------------
        NN_LOG("#         VerifyHandAnalysisSilhouetteStates (StateCount : %d)\n", stateCount);
        for (auto j = 0; j < stateCount; ++j)
        {
            DoTestVerifyHandAnalysisSilhouetteStates(state[j]);
        }
    }

    NN_LOG("#     SUCCESS\n");

    NNT_IRSENSOR_EXIT_SUCCESS;
}

void HandAnalysisProcessorTest::DoTestReadingHandAnalysisImageStates(const nn::irsensor::IrCameraHandle& handle, nn::irsensor::HandAnalysisConfig config) NN_NOEXCEPT
{
    NN_LOG("#     <GetHandAnalysisImageState>\n");
    int64_t prevSamplingNumber = -1;

    for (int i = 0; i < VerifyStatesTestCountMax; ++i)
    {
        nn::irsensor::HandAnalysisImageState state[nn::irsensor::HandAnalysisProcessorStateCountMax];

        NN_LOG("#     Test : [%2d/%2d]\n", i + 1, VerifyStatesTestCountMax);
        //-------------------------------------------------------------------------------
        // プロセッサが起動するまでの時間の計測します
        //-------------------------------------------------------------------------------
        if (i == 0)
        {
            // 初回のみ実行
            RunHandAnalysis(handle, config);
            DoTestRunningProcessor(handle);
        }
        //-------------------------------------------------------------------------------
        // ステートが更新されるまでにかかる時間を計測します
        //-------------------------------------------------------------------------------
        auto samplingNumber = prevSamplingNumber;                                   // 最新のサンプリングナンバーを記憶
        auto stateCount = 0;
        auto startTime = nn::os::GetSystemTick().ToTimeSpan();                      // 計測を開始した時間を記憶

        while (NN_STATIC_CONDITION(true))
        {
            nn::Result result = GetHandAnalysisImageState(state, &stateCount, nn::irsensor::HandAnalysisProcessorStateCountMax, 0, handle);
            if (nn::irsensor::ResultIrsensorNotReady::Includes(result) == false)
            {
                if (nn::irsensor::ResultIrsensorUnavailable::Includes(result))
                {
                    NN_LOG("#             ResultIrsensorUnavailable\n");
                    NNT_IRSENSOR_EXIT_SUCCESS;
                    return;
                }
                if (nn::irsensor::ResultIrsensorDeviceError::Includes(result))
                {
                    NN_LOG("############# ResultIrsensorDeviceError #############\n");
                    return;
                }
                samplingNumber = state[0].samplingNumber;
            }
            if (prevSamplingNumber != samplingNumber)
            {
                break;
            }
            ASSERT_LT(nn::os::GetSystemTick().ToTimeSpan().GetMilliSeconds() - startTime.GetMilliSeconds(), VerifyStatesTestLimitTime.GetMilliSeconds());
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1));
        }

        NN_LOG("#             UpdateHandAnalysisState (%lld ms)\n", nn::os::GetSystemTick().ToTimeSpan().GetMilliSeconds() - startTime.GetMilliSeconds());
        NN_LOG("#             SamplingNumber %lld -> %lld\n", prevSamplingNumber, samplingNumber);
        prevSamplingNumber = samplingNumber;
        //-------------------------------------------------------------------------------
        // ステートの妥当性を検証します
        //-------------------------------------------------------------------------------
        NN_LOG("#         VerifyHandAnalysisImageStates (StateCount : %d)\n", stateCount);
        for (auto j = 0; j < stateCount; ++j)
        {
            DoTestVerifyHandAnalysisImageStates(state[j]);
        }
    }

    NN_LOG("#     SUCCESS\n");

    NNT_IRSENSOR_EXIT_SUCCESS;
}

void HandAnalysisProcessorTest::DoTestVerifyHandAnalysisSilhouetteStates(const nn::irsensor::HandAnalysisSilhouetteState& state) NN_NOEXCEPT
{
    ASSERT_GE(state.samplingNumber, 0);

    ASSERT_GE(state.ambientNoiseLevel, static_cast<nn::irsensor::IrCameraAmbientNoiseLevel>(0));
    ASSERT_LT(state.ambientNoiseLevel, static_cast<nn::irsensor::IrCameraAmbientNoiseLevel>(3));

    ASSERT_GE(state.shapeCount, 0);
    ASSERT_LE(state.shapeCount, nn::irsensor::IrHandAnalysisShapeCountMax);

    for (auto i = 0; i < state.shapeCount; i++)
    {
        const auto shape = state.shapes[i];
        ASSERT_GE(shape.pointCount, 0);
        ASSERT_LE(shape.pointCount, nn::irsensor::IrHandAnalysisShapePointCountMax);
    }

    ASSERT_GE(state.handCount, 0);
    ASSERT_LE(state.handCount, nn::irsensor::IrHandAnalysisHandCountMax);

    for (auto i = 0; i < state.handCount; i++)
    {
        const auto hand = state.hands[i];
        ASSERT_GE(hand.protrusionCount, 0);
        ASSERT_LE(hand.protrusionCount, nn::irsensor::IrHandAnalysisProtrusionCountMax);
    }

    NNT_IRSENSOR_EXIT_SUCCESS;
}

void HandAnalysisProcessorTest::DoTestVerifyHandAnalysisImageStates(const nn::irsensor::HandAnalysisImageState& state) NN_NOEXCEPT
{
    ASSERT_GE(state.samplingNumber, 0);

    ASSERT_GE(state.ambientNoiseLevel, static_cast<nn::irsensor::IrCameraAmbientNoiseLevel>(0));
    ASSERT_LT(state.ambientNoiseLevel, static_cast<nn::irsensor::IrCameraAmbientNoiseLevel>(3));

    NNT_IRSENSOR_EXIT_SUCCESS;
}

// 停止処理
void HandAnalysisProcessorTest::DoTestStopImageProcessor(const nn::irsensor::IrCameraHandle& handle) NN_NOEXCEPT
{
    nn::irsensor::StopImageProcessorAsync(handle);

    nn::TimeSpanType startTime = nn::os::GetSystemTick().ToTimeSpan();
    auto counter = 0;
    while (NN_STATIC_CONDITION(true))
    {
        auto status = nn::irsensor::GetImageProcessorStatus(handle);

        if (status == nn::irsensor::ImageProcessorStatus_Running)
        {
            counter++;
            nn::os::SleepThread(PollingInterval);
            ASSERT_LT(counter, NotReadyWaitLoopCountMax);
        }
        else
        {
            NN_LOG("#     StopImageProcessor (%lld ms)\n", nn::os::GetSystemTick().ToTimeSpan().GetMilliSeconds() - startTime.GetMilliSeconds());
            break;
        }
    }
    NNT_IRSENSOR_EXIT_SUCCESS;
}

void HandAnalysisProcessorTest::ReadingStateTest(const nn::irsensor::IrCameraHandle& handle, const nn::irsensor::HandAnalysisMode& mode) NN_NOEXCEPT
{
    EXPECT_LT(0, s_HandleCount);

    nn::irsensor::HandAnalysisConfig config;
    config.mode = mode;

    NN_LOG("# [HandAnalysisMode : %s]\n",
        mode == nn::irsensor::HandAnalysisMode::HandAnalysisMode_None ?                 "None" :
        mode == nn::irsensor::HandAnalysisMode::HandAnalysisMode_Silhouette ?           "Silhouette" :
        mode == nn::irsensor::HandAnalysisMode::HandAnalysisMode_Image ?                "Image" :
        mode == nn::irsensor::HandAnalysisMode::HandAnalysisMode_SilhouetteAndImage ?   "SilhouetteAndImage" :
        mode == nn::irsensor::HandAnalysisMode::HandAnalysisMode_SilhouetteOnly ?       "SilhouetteOnly" :
        "UnKnown"
        );

    if (
        (mode == nn::irsensor::HandAnalysisMode::HandAnalysisMode_Silhouette) ||
        (mode == nn::irsensor::HandAnalysisMode::HandAnalysisMode_SilhouetteAndImage) ||
        (mode == nn::irsensor::HandAnalysisMode::HandAnalysisMode_SilhouetteOnly)
        )
    {
        DoTestReadingHandAnalysisSilhouetteStates(handle, config);
        DoTestStopImageProcessor(handle);
    }

    if (
        (mode == nn::irsensor::HandAnalysisMode::HandAnalysisMode_Image) ||
        (mode == nn::irsensor::HandAnalysisMode::HandAnalysisMode_SilhouetteAndImage)
        )
    {
        DoTestReadingHandAnalysisImageStates(handle, config);
        DoTestStopImageProcessor(handle);
    }

    NNT_IRSENSOR_EXIT_SUCCESS;
}

// IRセンサの入力状態が正しく取得できるか
TEST_F(HandAnalysisProcessorTest, StateReading)
{
    nnt::gamepad::Initialize();

    // ProCon USB 無効
    ::nnt::gamepad::DisableUsbConnect();

    // コントローラの再接続
    nnt::gamepad::DisconnectAll();
    nnt::gamepad::ConnectAll();

    // IR センサの状態取得
    for (auto i = 0; i < s_HandleCount; ++i)
    {
        const auto& handle = s_Handles[i];
        NN_LOG("#-----------------------------------------------\n");
        NN_LOG("# IrCameraHandle : 0x%x\n", handle);
        NN_LOG("#-----------------------------------------------\n");
        ReadingStateTest(handle, nn::irsensor::HandAnalysisMode::HandAnalysisMode_Silhouette);
        ReadingStateTest(handle, nn::irsensor::HandAnalysisMode::HandAnalysisMode_Image);
        ReadingStateTest(handle, nn::irsensor::HandAnalysisMode::HandAnalysisMode_SilhouetteAndImage);
        ReadingStateTest(handle, nn::irsensor::HandAnalysisMode::HandAnalysisMode_SilhouetteOnly);
    }

    nnt::gamepad::DisconnectAll();
}

// ProコントローラのUSB有線接続が有効時にIRセンサの入力状態が正しく取得できるか
TEST_F(HandAnalysisProcessorTest, StateReadingForUsb)
{
    nnt::gamepad::Initialize();

    // ProCon USB 有効
    ::nnt::gamepad::EnableUsbConnect();

    // コントローラの再接続
    nnt::gamepad::DisconnectAll();
    nnt::gamepad::ConnectAll();

    // IR センサの状態取得
    for (auto i = 0; i < s_HandleCount; ++i)
    {
        const auto& handle = s_Handles[i];
        NN_LOG("#-----------------------------------------------\n");
        NN_LOG("# IrCameraHandle : 0x%x\n", handle);
        NN_LOG("#-----------------------------------------------\n");
        ReadingStateTest(handle, nn::irsensor::HandAnalysisMode::HandAnalysisMode_Silhouette);
        ReadingStateTest(handle, nn::irsensor::HandAnalysisMode::HandAnalysisMode_Image);
        ReadingStateTest(handle, nn::irsensor::HandAnalysisMode::HandAnalysisMode_SilhouetteAndImage);
        ReadingStateTest(handle, nn::irsensor::HandAnalysisMode::HandAnalysisMode_SilhouetteOnly);
    }

    nnt::gamepad::DisconnectAll();
}

} // namespace
