﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>

#include <nn/hid/hid_NpadCommon.h>
#include <nn/hid/hid_NpadJoy.h>
#include <nn/hid/debug/hid_UsbPad.h>
#include <nn/hid/system/hid_FirmwareUpdate.h>
#include <nn/hid/system/hid_NpadCommon.h>
#include <nn/hid/system/hid_RegisteredDevice.h>
#include <nn/hid/system/hid_UniquePad.h>
#include <nn/hid/system/hid_UsbFullkey.h>
#include <nn/hid/system/hid_Result.h>
#include <nn/oe/oe_SelfControlApis.h>

#include <nn/os/os_Event.h>
#include <nn/os/os_MultipleWait.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_ThreadApi.h>
#include <nn/os/os_TimerEvent.h>

#include <nnt/nntest.h>

#include "testGamePad_Common.h"

#define WORKAROUND

namespace {

bool IsUniquePadConnected(nn::bluetooth::Address& address)
{
    bool isConnected = false;

    ::nn::hid::system::UniquePadId uniquePadIds[::nn::hid::system::UniquePadIdCountMax];

    auto uniquePadIdCount = ::nn::hid::system::ListUniquePads(uniquePadIds,
                                                              sizeof(uniquePadIds) / sizeof(uniquePadIds[0]));
    for (int i = 0; i < uniquePadIdCount; i++)
    {
        ::nn::bluetooth::Address gotAddress;
        if (::nn::hid::system::GetUniquePadBluetoothAddress(&gotAddress, uniquePadIds[i]).IsSuccess())
        {
            if (gotAddress == address)
            {
                isConnected = true;
            }
        }
        else
        {
            isConnected = false;
        }
    }
    return isConnected;
}

bool Connect(nn::bluetooth::Address& address) NN_NOEXCEPT
{
    // Wlan 優先モードを利用して接続確立を上げる
    ::nn::oe::SetWirelessPriorityMode(::nn::oe::WirelessPriorityMode_OptimizedForWlan);

    ::nn::os::MultiWaitType multiWait;

    ::nn::os::SystemEventType timeoutEvent;
    ::nn::os::MultiWaitHolderType timeoutEventHolder;

    ::nn::os::SystemEventType connectionEvent;
    ::nn::os::MultiWaitHolderType connectionEventHolder;

    ::nn::os::TimerEventType timerEvent;
    ::nn::os::MultiWaitHolderType timerEventHolder;

    ::nn::os::InitializeMultiWait(&multiWait);

    ::nn::hid::system::BindConnectionTriggerTimeoutEvent(&timeoutEvent,
                                                         ::nn::os::EventClearMode_ManualClear);
    ::nn::os::InitializeMultiWaitHolder(&timeoutEventHolder, &timeoutEvent);
    ::nn::os::LinkMultiWaitHolder(&multiWait, &timeoutEventHolder);

    ::nn::hid::system::BindUniquePadConnectionEvent(&connectionEvent,
                                                    ::nn::os::EventClearMode_ManualClear);
    ::nn::os::InitializeMultiWaitHolder(&connectionEventHolder, &connectionEvent);
    ::nn::os::LinkMultiWaitHolder(&multiWait, &connectionEventHolder);

    ::nn::os::InitializeTimerEvent(&timerEvent,
                                   ::nn::os::EventClearMode_ManualClear);
    ::nn::os::InitializeMultiWaitHolder(&timerEventHolder, &timerEvent);
    ::nn::os::LinkMultiWaitHolder(&multiWait, &timerEventHolder);

    bool runs = true;
    bool isConnected = false;

    // 一発目の接続イベントに向けたタイマーをセット
    ::nn::os::StartOneShotTimerEvent(&timerEvent, ::nn::TimeSpan::FromSeconds(1));

#ifdef WORKAROUND
    ::nn::os::TimerEventType workaroundEvent;
    ::nn::os::MultiWaitHolderType workaroundEventHolder;

    ::nn::os::InitializeTimerEvent(&workaroundEvent,
        ::nn::os::EventClearMode_ManualClear);
    ::nn::os::InitializeMultiWaitHolder(&workaroundEventHolder, &workaroundEvent);
    ::nn::os::LinkMultiWaitHolder(&multiWait, &workaroundEventHolder);

    // ワークアラウンドイベントに向けたタイマーをセット
    const auto WorkaroundTimeoutSeconds = 15;
    ::nn::os::StartOneShotTimerEvent(&workaroundEvent, ::nn::TimeSpan::FromSeconds(WorkaroundTimeoutSeconds));
#endif

    while(NN_STATIC_CONDITION(runs))
    {
        auto pHolder = nn::os::WaitAny(&multiWait);

        if(pHolder == &timerEventHolder)
        {
            ::nn::os::ClearTimerEvent(&timerEvent);

            // 接続を試みる
            auto connectionResult = ::nn::hid::system::SendConnectionTrigger(address);
            if (connectionResult.IsFailure())
            {
                NN_LOG("[hid] Send Trigger Failed : %d\n", connectionResult.GetDescription());
                ::nn::os::StartOneShotTimerEvent(&timerEvent, ::nn::TimeSpan::FromSeconds(1));
            }
        }
        if(pHolder == &timeoutEventHolder)
        {
            ::nn::os::ClearSystemEvent(&timeoutEvent);

            NN_LOG("ConnectionTriggerTimeout\n");
            runs = false;
        }
        if(pHolder == &connectionEventHolder)
        {
            ::nn::os::ClearSystemEvent(&connectionEvent);

            if (IsUniquePadConnected(address) == true)
            {
                runs = false;
                isConnected = true;
            }
            else
            {
                // 接続トリガーをかけているデバイスはまだ接続されていない
            }
        }

#ifdef WORKAROUND
        if(pHolder == &workaroundEventHolder)
        {
            ::nn::os::ClearTimerEvent(&workaroundEvent);

            NN_LOG("Workaround Timeout\n");
            runs = false;
        }
#endif

    }
    ::nn::oe::SetWirelessPriorityMode(::nn::oe::WirelessPriorityMode_Default);

    return isConnected;
}

void Disconnect(nn::hid::system::UniquePadId& uniquePadId)
{
    ::nn::hid::system::DisconnectUniquePad(uniquePadId);
}

const char* ToString(const ::nn::hid::system::DeviceTypeSet& deviceType) NN_NOEXCEPT
{
    if (deviceType.Test<::nn::hid::system::DeviceType::FullKeyController>())
    {
        return "FullKeyController";
    }
    else if(deviceType.Test<::nn::hid::system::DeviceType::Handheld>())
    {
        return "Handheld";
    }
    else if (deviceType.Test<::nn::hid::system::DeviceType::HandheldJoyLeft>())
    {
        return "HandheldJoyLeft";
    }
    else if (deviceType.Test<::nn::hid::system::DeviceType::HandheldJoyRight>())
    {
        return "HandheldJoyRight";
    }
    else if (deviceType.Test<::nn::hid::system::DeviceType::JoyConLeft>())
    {
        return "JoyConLeft";
    }
    else if (deviceType.Test<::nn::hid::system::DeviceType::JoyConRight>())
    {
        return "JoyConRight";
    }
    else if (deviceType.Test<::nn::hid::system::DeviceType::SwitchProController>())
    {
        return "SwitchProController";
    }
    else
    {
        return "Unknown";
    }
}

const char* ToString(const ::nn::hid::system::InterfaceType& interfaceType) NN_NOEXCEPT
{
    switch (interfaceType)
    {
    case ::nn::hid::system::InterfaceType_Bluetooth:
        return "Bluetooth";
    case ::nn::hid::system::InterfaceType_Rail:
        return "Rail";
    case ::nn::hid::system::InterfaceType_Usb:
        return "Usb";
    default:
        break;
    }

    return "Unknown";
}

const char* ToString(const ::nn::hid::system::UniquePadInterface& uniquePadInterface) NN_NOEXCEPT
{
    switch (uniquePadInterface)
    {
    case ::nn::hid::system::UniquePadInterface_Rail:
        return "Rail";
    case ::nn::hid::system::UniquePadInterface_Bluetooth:
        return "Bluetooth";
    case ::nn::hid::system::UniquePadInterface_Usb:
        return "Usb";
    case ::nn::hid::system::UniquePadInterface_Embedded:
        return "Embedded";
    default:
        break;
    }

    return "Unknown";
}

const char* ToString(const ::nn::hid::system::UniquePadType& uniquePadType) NN_NOEXCEPT
{
    switch (uniquePadType)
    {
    case ::nn::hid::system::UniquePadType_Embedded:
        return "Embedded";
    case ::nn::hid::system::UniquePadType_FullKeyController:
        return "FullKeyController";
    case ::nn::hid::system::UniquePadType_LeftController:
        return "LeftController";
    case ::nn::hid::system::UniquePadType_RightController:
        return "RightController";
    default:
        break;
    }

    return "Unknown";
}

} // namespace

namespace nnt { namespace gamepad {

void Initialize() NN_NOEXCEPT
{
    nn::hid::InitializeNpad();

    // 使用する操作形態を設定
    nn::hid::SetSupportedNpadStyleSet(NpadStyleSets);

    // Npad を有効にします
    nn::hid::SetSupportedNpadIdType(NpadIds, NpadIdCountMax);

    // AssignmentMode を 全て 2本持ちに設定
    for (const auto& id : NpadIds)
    {
        nn::hid::SetNpadJoyAssignmentModeDual(id);
    }

    // 持ち方を縦持ちに設定
    nn::hid::SetNpadJoyHoldType(nn::hid::NpadJoyHoldType_Vertical);
}

void ConnectAll() NN_NOEXCEPT
{

    //////////////////////////////////////
    // Bluetoothコントローラの接続
    //////////////////////////////////////
    ::nn::hid::system::RegisteredDevice registeredDevices[::nn::hid::system::RegisteredDeviceCountMax] = {};

    const auto RegisteredDeviceCount = ::nn::hid::system::GetRegisteredDevices(registeredDevices,
                                                                               NN_ARRAY_SIZE(registeredDevices));

    // コントローラ接続台数分実施する
    for (int i = 0; i < RegisteredDeviceCount; i++)
    {
        // ワークアラウンド回数の定義
        int retryCount = 0;
        const int RetryCountMax = 5;

        bool runs = true;
        bool isConnected = false;

        // 現在接続されているコントローラの台数を取得します
        ::nn::hid::system::UniquePadId uniquePadIds[::nn::hid::system::UniquePadIdCountMax];
        int connectedDeviceCount = ::nn::hid::system::ListUniquePads(uniquePadIds,
                                                                     NN_ARRAY_SIZE(uniquePadIds));

        auto& device = registeredDevices[i];

        // 既に所望の台数分接続済なので何もしない
        if (connectedDeviceCount == RegisteredDeviceCount)
        {
            NN_LOG("[%s] Do nothing, since all the paired %d devices are connected.\n"
                , __FUNCTION__
                , RegisteredDeviceCount);
            return;
        }

        while (NN_STATIC_CONDITION(runs))
        {
            NN_LOG("BluetoothConnectionDevice = %d/%d : Retry = %d/%d\n"
                , i + 1
                , DevicePairingCountMax
                , retryCount
                , RetryCountMax);

            isConnected = Connect(device.address);

            if (isConnected)
            {
                NN_LOG("Success\n");
                runs = false;
            }
            else if (retryCount >= RetryCountMax)
            {
                NN_LOG("Failed\n");
                runs = false;
            }

            retryCount++;
        }

        NN_LOG("STATS_CONNECTION, %d, %d, %x, %x, %x, %x, \n"
            , connectedDeviceCount
            , retryCount
            , isConnected ? 1 : 0
            , device.deviceType
            , device.subType
            , device.interfaceType
        );
    }

    //////////////////////////////////////
    // USB専用コントローラの接続
    //////////////////////////////////////
    ConnectUsbPads();

    NN_LOG("USBConnectionDevice = %d/%d\n"
        , GetConnectedUsbPadsCount()
        , UsbPadsCountMax);

    ::nnt::gamepad::PrintUniquePads();
}

void DisconnectAll() NN_NOEXCEPT
{
    ::nn::hid::system::UniquePadId uniquePadIds[::nn::hid::system::UniquePadIdCountMax];


    auto uniquePadIdCount = ::nn::hid::system::ListUniquePads(uniquePadIds,
                                                              sizeof(uniquePadIds) / sizeof(uniquePadIds[0]));

    //////////////////////////////////////
    // Bluetoothコントローラの切断
    //////////////////////////////////////
    for (int i = 0; i < uniquePadIdCount; i++)
    {
        Disconnect(uniquePadIds[i]);
    }

    //////////////////////////////////////
    // USB専用コントローラの切断
    //////////////////////////////////////
    DisconnectUsbPads();
}

void EnableUsbConnect() NN_NOEXCEPT
{
    // USB接続の有効化
    ::nn::hid::system::EnableUsbFullKeyController(true);

    ASSERT_EQ(::nn::hid::system::IsUsbFullKeyControllerEnabled(), true);
}

void DisableUsbConnect() NN_NOEXCEPT
{
    // USB接続の無効化
    ::nn::hid::system::EnableUsbFullKeyController(false);

    ASSERT_EQ(::nn::hid::system::IsUsbFullKeyControllerEnabled(), false);
}

void ConnectUsbPads() NN_NOEXCEPT
{
    ::nn::hid::debug::ConnectUsbPadsAsync();

    // TORIAEZU: hid プロセス内で Npad として登録されるまでにラグがあるため
    ::nn::os::SleepThread(::nn::TimeSpan::FromSeconds(1));
}

void DisconnectUsbPads() NN_NOEXCEPT
{
    ::nn::hid::debug::DisconnectUsbPadsAsync();

    // TORIAEZU: hid プロセス内で Npad の登録が解除されるまでにラグがあるため
    ::nn::os::SleepThread(::nn::TimeSpan::FromSeconds(1));
}

nnt::gamepad::ControllerType GetControllerTypeFromNpad(const nn::hid::NpadIdType& npadId)
{
    nnt::gamepad::ControllerType type = Undefined;

    nn::hid::system::UniquePadId pOutValues[::nn::hid::system::UniquePadIdCountMax];
    size_t size;

    size = ::nn::hid::system::GetUniquePadsFromNpad(pOutValues, NN_ARRAY_SIZE(pOutValues) ,npadId);

    if (size == 2)
    {
        ::nn::hid::system::UniquePadInterface uniqueInterface = ::nn::hid::system::UniquePadInterface_Unknown;
        ::nn::hid::system::GetUniquePadInterface(&uniqueInterface, pOutValues[0]);

        if (uniqueInterface == ::nn::hid::system::UniquePadInterface_Rail)
        {
            type = Handheld;
        }
        else if (uniqueInterface == ::nn::hid::system::UniquePadInterface_Bluetooth)
        {
            type = Kuina;
        }
    }
    else if (size == 1)
    {
        ::nn::bluetooth::Address gotAddress;
        if (::nn::hid::system::GetUniquePadBluetoothAddress(&gotAddress, pOutValues[0]).IsSuccess())
        {
            nn::hid::NpadStyleSet NpadStyle = nn::hid::GetNpadStyleSet(npadId);

            if (NpadStyle.Test<::nn::hid::NpadStyleFullKey>())
            {
                type = ProCon;
            }
            else if (NpadStyle.Test<::nn::hid::NpadStyleJoyLeft>())
            {
                type = JoyLeft;
            }
            else if (NpadStyle.Test<::nn::hid::NpadStyleJoyRight>())
            {
                type = JoyRight;
            }
        }
        else if (::nn::hid::system::GetUniquePadBluetoothAddress(&gotAddress, pOutValues[0]) <= ::nn::hid::system::ResultUniquePadNoBluetoothAddress())
        {
            type = UsbCon;
        }
    }
    return type;
}

void PrintRegisteredDevices() NN_NOEXCEPT
{
    ::nn::hid::system::RegisteredDevice registeredDevices[::nn::hid::system::RegisteredDeviceCountMax] = {};

    const auto RegisteredDeviceCount = ::nn::hid::system::GetRegisteredDevices(registeredDevices,
                                                                               NN_ARRAY_SIZE(registeredDevices));

    NN_LOG("# of paired devices : %d\n", RegisteredDeviceCount);

    for (int i = 0; i < RegisteredDeviceCount; i++)
    {
        const auto& device = registeredDevices[i];

        NN_LOG("# %d\n", i);
        NN_LOG("\tDeviceType    : %s\n", ToString(device.deviceType));
        NN_LOG("\tInterfaceType : %s\n", ToString(device.interfaceType));
        NN_LOG("\tBD_ADDR       : %x:%x:%x:%x:%x:%x\n", device.address.address[0]
                                                      , device.address.address[1]
                                                      , device.address.address[2]
                                                      , device.address.address[3]
                                                      , device.address.address[4]
                                                      , device.address.address[5]);
    }

}

void PrintUniquePads() NN_NOEXCEPT
{
    ::nn::hid::system::UniquePadId uniquePadIds[::nn::hid::system::UniquePadIdCountMax];

    auto uniquePadIdCount = ::nn::hid::system::ListUniquePads(uniquePadIds,
                                                              NN_ARRAY_SIZE(uniquePadIds));

    NN_LOG("# of unique pads : %d\n", uniquePadIdCount);

    for (int i = 0; i < uniquePadIdCount; i++)
    {
        const auto& uniquePadId = uniquePadIds[i];

        // ファームウェアバージョン
        ::nn::hid::system::FirmwareVersion firmwareVersion = {};
        auto result = ::nn::hid::system::GetFirmwareVersion(&firmwareVersion,
                                                            uniquePadId);

        // コントローラ番号
        int controllerNumber = -1;
        ::nn::hid::system::GetUniquePadControllerNumber(&controllerNumber,
                                                        uniquePadId);

        // Bluetooth アドレス
        ::nn::bluetooth::Address bd_addr = {};
        ::nn::hid::system::GetUniquePadBluetoothAddress(&bd_addr, uniquePadId);

        // インタフェース
        ::nn::hid::system::UniquePadInterface uniquePadInterface;
        ::nn::hid::system::GetUniquePadInterface(&uniquePadInterface, uniquePadId);

        // デバイスタイプ
        ::nn::hid::system::UniquePadType uniquePadType =
            ::nn::hid::system::GetUniquePadType(uniquePadId);

        NN_LOG("UniquePadId = 0x%x\n", uniquePadId);
        NN_LOG("\tFirmwareVersion  : %s %u.%u.%u.%u %s\n", result.IsSuccess() ? "Success" : "Failure"
                                                         , firmwareVersion.major
                                                         , firmwareVersion.micro
                                                         , firmwareVersion.minor
                                                         , firmwareVersion.revision
                                                         , firmwareVersion.identifier);
        NN_LOG("\tControllerNumber : %d\n", controllerNumber);
        NN_LOG("\tBD_ADDR          : %x:%x:%x:%x:%x:%x\n", bd_addr.address[0]
                                                         , bd_addr.address[1]
                                                         , bd_addr.address[2]
                                                         , bd_addr.address[3]
                                                         , bd_addr.address[4]
                                                         , bd_addr.address[5]);
        NN_LOG("\tInterface        : %s\n", ToString(uniquePadInterface));
        NN_LOG("\tType             : %s\n", ToString(uniquePadType));

    }
}

// 接続済 USB コントローラの台数を取得する関数
int GetConnectedUsbPadsCount() NN_NOEXCEPT
{
    ::nn::hid::system::UniquePadId uniquePadIds[::nn::hid::system::UniquePadIdCountMax];
    auto count = ::nn::hid::system::ListUniquePads(uniquePadIds,
        NN_ARRAY_SIZE(uniquePadIds));

    int usbPadsCount = 0;
    for (int i = 0; i < count; i++)
    {
        ::nn::hid::system::UniquePadInterface uniquePadInterface =
            ::nn::hid::system::UniquePadInterface_Unknown;

        ::nn::hid::system::GetUniquePadInterface(&uniquePadInterface, uniquePadIds[i]);
        if (uniquePadInterface == ::nn::hid::system::UniquePadInterface_Usb)
        {
            ::nn::bluetooth::Address gotAddress;
            if (::nn::hid::system::GetUniquePadBluetoothAddress(&gotAddress, uniquePadIds[i]).IsSuccess())
            {
                // 何もしない
            }
            else
            {
                usbPadsCount++;
            }

        }
    }

    NN_LOG("ConnectedUsbPadsCount = %d\n", usbPadsCount);

    return usbPadsCount;
}

}} // namespace nnt::gamepad
