﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nnd/ftm/ftm.h>
#include <nnd/ftm/ftm_Fab.h>

namespace
{
// Mutual Compensation データの読み出し
nn::Result ReadMutualCompensationData() NN_NOEXCEPT
{
    nnd::ftm::MutualCompensationData compensation;
    NN_RESULT_DO(nnd::ftm::ReadMutualCompensationData(&compensation));

    // FTM SW Guide v1.12 p49 : force チャンネルのノード数が他での結果と異なることの説明
    // Note that since the device uses a multiple TX driving method to achieve a fast report rate
    // during Mutual- sensing, it requires dummy channel based on hardware design.
    // Due to this, Mutual compensation data header not only shows
    // physical channel length used by the panel (18 TX channels) but also a dummy length (2 channels).
    NN_LOG("force channel count = %d\n", compensation.forceChannelCount);
    NN_LOG("sense channel count = %d\n", compensation.senseChannelCount);

    for (int force = 0; force < compensation.forceChannelCount; force++)
    {
        for (int sense = 0; sense < compensation.senseChannelCount; sense++)
        {
            NN_LOG("%02d ", compensation.cx2[force][sense]);
        }
        NN_LOG("\n");
    }

    NN_RESULT_SUCCESS;
}

// Self Compensation データの読み出し
nn::Result ReadSelfCompensationData() NN_NOEXCEPT
{
    nnd::ftm::SelfCompensationData compensation;
    NN_RESULT_DO(nnd::ftm::ReadSelfCompensationData(&compensation));

    NN_LOG("force channel count = %d\n", compensation.forceChannelCount);
    NN_LOG("sense channel count = %d\n", compensation.senseChannelCount);
    NN_LOG("force ix1 = %d\n", compensation.forceIx1);
    NN_LOG("sense ix1 = %d\n", compensation.senseIx1);
    NN_LOG("force cx1 = %d\n", compensation.forceIx1);
    NN_LOG("sense cx1 = %d\n", compensation.senseIx1);
    int totalChannelCount = compensation.forceChannelCount + compensation.senseChannelCount;

    NN_LOG("ix2 : \n");
    for (int byte = 0; byte < totalChannelCount; byte++)
    {
        if (byte == compensation.forceChannelCount)
        {
            NN_LOG("\n");
        }
        NN_LOG("%02d ", compensation.ix2[byte]);
    }
    NN_LOG("\n");

    NN_LOG("cx2 : \n");
    for (int byte = 0; byte < totalChannelCount; byte++)
    {
        if (byte == compensation.forceChannelCount)
        {
            NN_LOG("\n");
        }
        NN_LOG("%02d ", compensation.cx2[byte]);
    }
    NN_LOG("\n");

    NN_RESULT_SUCCESS;
}

// Mutual Raw データの読み出し
nn::Result ReadMutualRawData() NN_NOEXCEPT
{
    nnd::ftm::MutualRawData raw;
    NN_RESULT_DO(nnd::ftm::ReadMutualRawData(&raw));

    NN_LOG("force channel count = %d\n", raw.forceChannelCount);
    NN_LOG("sense channel count = %d\n", raw.senseChannelCount);

    for (int force = 0; force < raw.forceChannelCount; force++)
    {
        for (int sense = 0; sense < raw.senseChannelCount; sense++)
        {
            NN_LOG("%04d ", raw.data[force][sense]);
        }
        NN_LOG("\n");
    }

    NN_RESULT_SUCCESS;
}

// Self Raw データの読み出し
nn::Result ReadSelfRawData() NN_NOEXCEPT
{
    nnd::ftm::SelfRawData raw;
    NN_RESULT_DO(nnd::ftm::ReadSelfRawData(&raw));

    NN_LOG("force channel count = %d\n", raw.forceChannelCount);
    NN_LOG("sense channel count = %d\n", raw.senseChannelCount);

    for (int sense = 0; sense < raw.senseChannelCount; sense++)
    {
        NN_LOG("%04d ", raw.senseData[sense]);
    }
    NN_LOG("\n");

    for (int force = 0; force < raw.forceChannelCount; force++)
    {
        NN_LOG("%04d ", raw.forceData[force]);
    }
    NN_LOG("\n");

    NN_RESULT_SUCCESS;
}

// ITO Test の実行
nn::Result RunItoTest() NN_NOEXCEPT
{
    nnd::ftm::ItoEventReport report;
    NN_RESULT_DO(nnd::ftm::RunItoTest(&report));

    if (report.type == nnd::ftm::ItoErrorType::NoError)
    {
        NN_LOG("No errors\n");
    }
    else
    {
        NN_LOG("number = %d : type = %d\n", report.number, static_cast<int>(report.type));
    }

    NN_RESULT_SUCCESS;
}

} // namespace

extern "C" void nnMain()
{
    NN_LOG("/// FTM Diagnostic Features Checker ///\n");

    nnd::ftm::Initialize();
    nn::Result result = nnd::ftm::ResetDevice();
    NN_ASSERT(result.IsSuccess(), "Failed to reset device");

    result = nnd::ftm::SleepOutDevice();
    NN_ASSERT(result.IsSuccess(), "Failed to sleep out");

    // AutoTune 後にセンシングを有効にすると Self Raw Data の値が変化してしまうので、
    // この処理以降、nnd::ftm::ReadSelfRawData() までに nnd::ftm::ActivateSensing() を実行してはいけない。
    // https://redbtsred.nintendo.co.jp/redmine/issues/3175
    result = nnd::ftm::RunAutoTune();
    NN_ASSERT(result.IsSuccess(), "Failed to Auto Tune\n");

    NN_LOG("[Mutual Compensation Data]\n");
    result = ReadMutualCompensationData();
    NN_ASSERT(result.IsSuccess(), "Failed to read mutual compensation data");

    NN_LOG("[Self Compensation Data]\n");
    result = ReadSelfCompensationData();
    NN_ASSERT(result.IsSuccess(), "Failed to read self compensation data");

    NN_LOG("[Mutual Raw Data]\n");
    result = ReadMutualRawData();
    NN_ASSERT(result.IsSuccess(), "Failed to read self raw data");

    NN_LOG("[Self Raw Data]\n");
    result = ReadSelfRawData();
    NN_ASSERT(result.IsSuccess(), "Failed to read self raw data");

    // ITO Test 時にはセンシング有効にしておく必要がある。
    NN_LOG("[ITO Test]\n");
    result = nnd::ftm::ActivateSensing();
    NN_ASSERT(result.IsSuccess(), "Failed to activate sensing");
    result = RunItoTest();
    NN_ASSERT(result.IsSuccess(), "Failed to read ito test report");

    nnd::ftm::SleepInDevice();
    nnd::ftm::Finalize();

    NN_LOG("[[ PASSED !! ]]\n");
}
