﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <numeric>

#include <nn/nn_Common.h>
#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/fsUtil/testFs_util.h>

struct TestParam
{
    size_t storageSize;
    size_t blockSize;
};

nnt::fs::util::Vector<TestParam> MakeTestParam() NN_NOEXCEPT
{
    nnt::fs::util::Vector<TestParam> params;

    {
        TestParam data;
        data.storageSize = 1024;
        data.blockSize = 256;
        params.push_back(data);
    }

    {
        TestParam data;
        data.storageSize = 16 * 1024;
        data.blockSize = 256;
        params.push_back(data);
    }

    {
        TestParam data;
        data.storageSize = 512;
        data.blockSize = 256;
        params.push_back(data);
    }

    {
        TestParam data;
        data.storageSize = 32 * 1024 + 1;
        data.blockSize = 1024;
        params.push_back(data);
    }

    {
        TestParam data;
        data.storageSize = 16 * 1024 + 900;
        data.blockSize = 8 * 1024;
        params.push_back(data);
    }

    {
        TestParam data;
        data.storageSize = 32 * 1024 + 300;
        data.blockSize = 512;
        params.push_back(data);
    }
    return params;
}

class AlignCheckStorageDeathTest : public ::testing::TestWithParam<TestParam>
{
};

/**
* インスタンスを作成して、初期化せずに Read Write を行うとアサートが発生するのを確認するテスト
*/
TEST_P(AlignCheckStorageDeathTest, NoInitializeTest)
{
#ifndef NN_SDK_BUILD_RELEASE
    const TestParam& param = GetParam();
    size_t storageSize = param.storageSize;
    // インスタンスを作成して、初期化しない
    {
        nnt::fs::util::AlignCheckStorage storage;

        // Read Write を確認
        std::unique_ptr<char[]> buffer(new char[storageSize]);
        std::iota(buffer.get(), buffer.get() + storageSize, '\x1');
        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultNotInitialized,
            storage.Write(0, buffer.get(), storageSize)
        );
        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultNotInitialized,
            storage.Read(0, buffer.get(), storageSize)
        );

        storage.Finalize();
    }
#endif  // NN_SDK_BUILD_RELEASE
}

INSTANTIATE_TEST_CASE_P(
    TestAlignCheckStorage,
    AlignCheckStorageDeathTest,
    ::testing::ValuesIn(MakeTestParam())
);

class AlignCheckStorageTest : public ::testing::TestWithParam<TestParam>
{
public:
    //! デストラクタ
    virtual ~AlignCheckStorageTest() NN_NOEXCEPT NN_OVERRIDE {}

protected:
    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
    {
        const TestParam& param = GetParam();
        m_BaseStorage.Initialize(param.storageSize);
    }

protected:
    nnt::fs::util::SafeMemoryStorage m_BaseStorage;
};

/**
* 初期化し、Read Write ができるかテスト
*/
TEST_P(AlignCheckStorageTest, InitializeTest)
{
    const TestParam& param = GetParam();
    size_t storageSize = param.storageSize;
    size_t blockSize = param.blockSize;

    // 先にインスタンスを作成してから初期化する
    {
        nnt::fs::util::AlignCheckStorage storage;
        storage.Initialize(nn::fs::SubStorage(&m_BaseStorage, 0, storageSize), blockSize);

        // Read Write を確認
        std::unique_ptr<char[]> buffer(new char[storageSize]);
        std::iota(buffer.get(), buffer.get() + storageSize, '\x1');
        storage.Write(0, buffer.get(), storageSize);
        storage.Read(0, buffer.get(), storageSize);
        storage.Finalize();
    }

    // インスタンスの作成と同時に初期化する
    {
        nnt::fs::util::AlignCheckStorage storage(
            nn::fs::SubStorage(&m_BaseStorage, 0, storageSize),
            blockSize
        );
        // Read Write を確認
        std::unique_ptr<char[]> buffer(new char[storageSize]);
        std::iota(buffer.get(), buffer.get() + storageSize, '\x1');
        storage.Write(0, buffer.get(), storageSize);
        storage.Read(0, buffer.get(), storageSize);
        storage.Finalize();
    }
}

/**
* 書き込んだデータが読み取れるかテスト
*/
TEST_P(AlignCheckStorageTest, AlignedReadWriteTest)
{
    const TestParam& param = GetParam();
    size_t storageSize = param.storageSize;
    size_t blockSize = param.blockSize;
    nnt::fs::util::AlignCheckStorage storage;
    storage.Initialize(nn::fs::SubStorage(&m_BaseStorage, 0, storageSize), blockSize);

    std::unique_ptr<char[]> readBuffer(new char[storageSize]);
    std::unique_ptr<char[]> writeBuffer(new char[storageSize]);
    std::iota(writeBuffer.get(), writeBuffer.get() + storageSize, '\x1');


    // 全体書き込み、全体読み取り
    {
        storage.Write(0, writeBuffer.get(), storageSize);
        storage.Read(0, readBuffer.get(), storageSize);
        ASSERT_EQ(0, memcmp(writeBuffer.get(), readBuffer.get(), storageSize));
    }

    // 1 バイト書き込み、1 バイト読み込み
    {
        storage.Write(blockSize, writeBuffer.get(), 1);
        storage.Read(blockSize, readBuffer.get(), 1);
        ASSERT_EQ(0, memcmp(writeBuffer.get(), readBuffer.get(), 1));
    }

    // Read と Write でオフセットが違う
    if( storageSize >= blockSize * 4 )
    {
        storage.Write(blockSize * 2, writeBuffer.get(), blockSize * 2);
        storage.Read(blockSize * 3, readBuffer.get(), blockSize);
        ASSERT_EQ(0, memcmp(writeBuffer.get() + blockSize, readBuffer.get(), blockSize));
    }

    // 終端書き込み、終端読み込み
    if( storageSize >= blockSize * 2 )
    {
        // 最後の 3 ブロックに書き込みを行い、
        // 最後の 2 ブロックから読み込みを行う。
        // ストレージサイズがブロックサイズの倍数なら 2 ブロック書き込み 1ブロック読み込み
        int64_t writeOffset = nn::util::align_down(storageSize - blockSize * 2, blockSize);
        int64_t readOffset = nn::util::align_down(storageSize - blockSize, blockSize);
        storage.Write(writeOffset, writeBuffer.get(), storageSize - static_cast<size_t>(writeOffset));
        storage.Read(readOffset, readBuffer.get(), storageSize - static_cast<size_t>(readOffset));
        ASSERT_EQ(
            0,
            memcmp(writeBuffer.get() + blockSize, readBuffer.get(), storageSize - static_cast<size_t>(readOffset))
        );
    }
}

/**
* アライメントにそっていない Read を行うとエラーリザルトが返ってくることを確認するテスト
*/
TEST_P(AlignCheckStorageTest, NotAlignedReadTest)
{
    const TestParam& param = GetParam();
    size_t storageSize = param.storageSize;
    size_t blockSize = param.blockSize;
    nnt::fs::util::AlignCheckStorage storage;
    storage.Initialize(nn::fs::SubStorage(&m_BaseStorage, 0, storageSize), blockSize);

    std::unique_ptr<char[]> writeBuffer(new char[storageSize]);
    std::iota(writeBuffer.get(), writeBuffer.get() + storageSize, '\x1');

    // まず全体に書き込んでおく
    storage.Write(0, writeBuffer.get(), storageSize);

    // 読み込んだ値と比較用の値を準備
    std::unique_ptr<char[]> readBuffer(new char[storageSize]);
    std::iota(readBuffer.get(), readBuffer.get() + storageSize, '\x2');

    for( size_t writeSize = 0; writeSize < storageSize; ++writeSize )
    {
        if( writeSize % blockSize == 0 )
        {
            continue;
        }
        // writeSize から最後まで書き込む
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultInvalidAlignment,
            storage.Read(writeSize, readBuffer.get(), storageSize - writeSize)
        );
    }

    for( size_t writeSize = 0; writeSize < storageSize; ++writeSize )
    {
        if( writeSize % blockSize == 0 )
        {
            continue;
        }
        // writeSize から 1 バイトだけ書き込む
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultInvalidAlignment,
            storage.Read(writeSize, readBuffer.get(), 1)
        );
    }
}

/**
* アライメントにそっていない Write を行うとエラーリザルトが返ってくることを確認するテスト
*/
TEST_P(AlignCheckStorageTest, NotAlignedWriteTest)
{
    const TestParam& param = GetParam();
    size_t storageSize = param.storageSize;
    size_t blockSize = param.blockSize;
    nnt::fs::util::AlignCheckStorage storage;
    storage.Initialize(nn::fs::SubStorage(&m_BaseStorage, 0, storageSize), blockSize);

    std::unique_ptr<char[]> writeBuffer(new char[storageSize]);
    std::iota(writeBuffer.get(), writeBuffer.get() + storageSize, '\x1');

    for( size_t writeSize = 0; writeSize < storageSize; ++writeSize )
    {
        if( writeSize % blockSize == 0 )
        {
            continue;
        }
        // writeSize から最後まで書き込む
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultInvalidAlignment,
            storage.Write(writeSize, writeBuffer.get(), storageSize - writeSize)
        );
    }

    for( size_t writeSize = 0; writeSize < storageSize; ++writeSize )
    {
        if( writeSize % blockSize == 0 )
        {
            continue;
        }
        // writeSize から 1 バイトだけ書き込む
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultInvalidAlignment,
            storage.Write(writeSize, writeBuffer.get(), 1)
        );
    }
}

// 4 GB を超えるオフセットのテスト
TEST(AlignCheckStorageLargeTest, ReadWriteLargeOffset)
{
    static const size_t BlockSize = 512;
    static const int64_t StorageSize = nnt::fs::util::LargeOffsetMax + BlockSize;

    nnt::fs::util::VirtualMemoryStorage baseStorage(StorageSize);
    nnt::fs::util::AlignCheckStorage storage(
        nn::fs::SubStorage(&baseStorage, 0, StorageSize), BlockSize);

    nnt::fs::util::TestStorageAccessWithLargeOffset(&storage, BlockSize);

    auto buffer = nnt::fs::util::AllocateBuffer(BlockSize);
    nnt::fs::util::FillBufferWith32BitCount(buffer.get(), BlockSize, 0);

    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultInvalidAlignment,
        storage.Read(nnt::fs::util::LargeOffsetMax - 1, buffer.get(), BlockSize));

    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultInvalidAlignment,
        storage.Write(nnt::fs::util::LargeOffsetMax - 1, buffer.get(), BlockSize));
}

INSTANTIATE_TEST_CASE_P(
    TestAlignCheckStorage,
    AlignCheckStorageTest,
    ::testing::ValuesIn(MakeTestParam())
);
