﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>

#include <memory>
#include <nnt/nntest.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>

#include <nnt/result/testResult_Assert.h>
#include <nnt/fsUtil/testFs_util.h>

#include <nn/fssystem/fs_StorageFile.h>
#include <nn/fssystem/fs_HostFileSystem.h>
#include <nn/fssystem/fs_PathOnExecutionDirectory.h>
#include <nn/fs/fs_FileStorage.h>
#include <nn/fs/fs_MemoryStorage.h>
#include <nn/fs/fs_FileSystem.h>

#include "testFs_Unit_StorageLayerTestCase.h"

namespace {

class StorageFileTest : public ::testing::Test
{
protected:

    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
    {
        auto pHostFileSystem = new nn::fssystem::HostFileSystem();
        pHostFileSystem->Initialize(nn::fssystem::PathOnExecutionDirectory("").Get());

        m_pFileSystem.reset(pHostFileSystem);
    }

protected:
    std::unique_ptr<nn::fs::fsa::IFileSystem> m_pFileSystem;
};

TEST(StorageFile, ReadTooBig)
{
    // ファイルサイズ以上を読み込もうとして、読み込まれるデータ長はファイルサイズと一致しているか？

    static const int StorageSize = 64 * 1024;

    // 読み込まれるデータ
    std::unique_ptr<uint8_t[]> storageBuffer(new uint8_t[StorageSize]);
    nnt::fs::util::FillBufferWithRandomValue(storageBuffer.get(), StorageSize);

    std::unique_ptr<nn::fs::IStorage> pMemoryStorage
        (new nn::fs::MemoryStorage(storageBuffer.get(), StorageSize));

    std::unique_ptr<nn::fs::fsa::IFile> pStorageFile
        (new nn::fssystem::StorageFile(pMemoryStorage.get(), nn::fs::OpenMode_Read));

    // 読み込み
    static const int BiggerSize = StorageSize * 2;
    std::unique_ptr<uint8_t[]> readBuffer(new uint8_t[BiggerSize]);

    // 0xCD で初期化
    std::fill(
        readBuffer.get(),
        readBuffer.get() + BiggerSize,
        static_cast<uint8_t>(0xCD)
    );

    // 元のサイズ以上に読もうとする
    size_t readSize = 0;
    NNT_ASSERT_RESULT_SUCCESS(pStorageFile->Read(&readSize, 0, readBuffer.get(), BiggerSize, nn::fs::ReadOption::MakeValue(0)));

    // 読まれた数が元のサイズと同じ
    EXPECT_TRUE(readSize == StorageSize);

    // 正しく読まれている
    NNT_FS_UTIL_EXPECT_MEMCMPEQ(storageBuffer.get(), readBuffer.get(), StorageSize);

    // 実際、読み込みサイズ以上には読まれていない
    EXPECT_TRUE(readBuffer[StorageSize] == 0xCD);

    pMemoryStorage.reset();
    storageBuffer.reset();
}

TEST_F(StorageFileTest, AllowAppend)
{
    // 問題なくファイルの自動伸長が行われるか？

    const char* path = "/test.file";

    // 書き込まれるデータ
    static const size_t SrcDataSize = 8 * 1024;
    std::unique_ptr<uint8_t[]> srcData(new uint8_t[SrcDataSize]);
    nnt::fs::util::FillBufferWithRandomValue(srcData.get(), SrcDataSize);

    // ファイル作成(ファイルサイズ0)
    NNT_ASSERT_RESULT_SUCCESS(m_pFileSystem->CreateFile(path, /* size =*/0, /* option =*/0));

    std::unique_ptr<nn::fs::fsa::IFile> pBaseFile;

    // オリジナルのファイルオープン
    auto openMode = static_cast<nn::fs::OpenMode>(nn::fs::OpenMode_Read |
                                                  nn::fs::OpenMode_Write |
                                                  nn::fs::OpenMode_AllowAppend);
    NNT_ASSERT_RESULT_SUCCESS(m_pFileSystem->OpenFile(&pBaseFile, path, openMode));

    // オリジナルのファイルをストレージにする
    std::unique_ptr<nn::fs::IStorage> pBaseStorage(new nn::fs::FileStorage(pBaseFile.get()));

    // ストレージをファイルにする
    std::unique_ptr<nn::fs::fsa::IFile> pFile(new nn::fssystem::StorageFile(pBaseStorage.get(), openMode));

    // ファイルサイズを越えて書き込む
    nnt::fs::util::ConfirmWriteRead(pFile.get(), srcData.get(), SrcDataSize);

    // 書き込んだサイズとファイルサイズが一致している
    int64_t wroteSize = 0;
    NNT_ASSERT_RESULT_SUCCESS(pFile->GetSize(&wroteSize));
    EXPECT_TRUE(wroteSize == SrcDataSize);

    // ファイル閉じる
    pFile.reset();
    pBaseStorage.reset();
    pBaseFile.reset();

    // ファイル削除
    NNT_ASSERT_RESULT_SUCCESS(m_pFileSystem->DeleteFile(path));
}

TEST(StorageFile, QueryRange)
{
    static const int StorageSize = 64 * 1024;

    // 読み込まれるデータ
    std::unique_ptr<uint8_t[]> storageBuffer(new uint8_t[StorageSize]);
    nnt::fs::util::FillBufferWithRandomValue(storageBuffer.get(), StorageSize);

    std::unique_ptr<nn::fs::IStorage> pMemoryStorage
        (new nn::fs::MemoryStorage(storageBuffer.get(), StorageSize));

    std::unique_ptr<nn::fs::fsa::IFile> pStorageFile
        (new nn::fssystem::StorageFile(pMemoryStorage.get(), nn::fs::OpenMode_Read));

    nn::fs::QueryRangeInfo info;
    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultNullptrArgument, pStorageFile->OperateRange(nullptr, sizeof(info), nn::fs::OperationId::QueryRange, 0, StorageSize, nullptr, 0));
    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultInvalidSize, pStorageFile->OperateRange(&info, 0, nn::fs::OperationId::QueryRange, 0, StorageSize, nullptr, 0));

    NNT_ASSERT_RESULT_SUCCESS(pStorageFile->OperateRange(&info, sizeof(info), nn::fs::OperationId::QueryRange, 0, StorageSize, nullptr, 0));
    EXPECT_EQ(0, info.aesCtrKeyTypeFlag);
    EXPECT_EQ(0, info.speedEmulationTypeFlag);
}

TEST(StorageFileLargeTest, ReadWrite)
{
    static const size_t BufferSize = 256;

    nnt::fs::util::VirtualMemoryStorage baseStorage(static_cast<int64_t>(64) * 1024 * 1024 * 1024 + 256);

    {
        // AllowAppend なしのテスト
        nn::fssystem::StorageFile storageFile(
            &baseStorage,
            static_cast<nn::fs::OpenMode>(nn::fs::OpenMode_Read | nn::fs::OpenMode_Write));
        TestWriteReadStorageWithLargeOffset(&storageFile, BufferSize);
    }
    {
        // AllowAppend ありのテスト
        nn::fssystem::StorageFile storageFile(
            &baseStorage,
            static_cast<nn::fs::OpenMode>(nn::fs::OpenMode_Read | nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend));
        TestWriteReadStorageWithLargeOffsetAllowAppend(&storageFile, BufferSize);
    }
}

//! @brief 書き込みモードで開いたファイルを読み込み、ResultInvalidOperationForOpenMode が返却されることを確認する
TEST(StorageFile, InvalidOpenModeReadFile)
{
    static const int StorageSize = 64 * 1024;
    std::unique_ptr<uint8_t[]> storageBuffer(new uint8_t[StorageSize]);
    nnt::fs::util::FillBufferWithRandomValue(storageBuffer.get(), StorageSize);
    std::unique_ptr<nn::fs::IStorage> pMemoryStorage (new nn::fs::MemoryStorage(storageBuffer.get(), StorageSize));
    std::unique_ptr<nn::fs::fsa::IFile> file(new nn::fssystem::StorageFile(pMemoryStorage.get(), nn::fs::OpenMode_Write));

    char buffer[0x100 + 1] = {0};
    size_t readSize = 0;
    NNT_EXPECT_RESULT_FAILURE(
        nn::fs::ResultInvalidOperationForOpenMode,
        file->Read(&readSize, 0, buffer, sizeof(buffer) - 1, nn::fs::ReadOption()));
}

//! @brief 読み込みモードで開いたファイルに書き込み、ResultInvalidOperationForOpenMode が返却されることを確認する
TEST(StorageFile, InvalidOpenModeWriteFile)
{
    static const int StorageSize = 64 * 1024;
    std::unique_ptr<uint8_t[]> storageBuffer(new uint8_t[StorageSize]);
    nnt::fs::util::FillBufferWithRandomValue(storageBuffer.get(), StorageSize);
    std::unique_ptr<nn::fs::IStorage> pMemoryStorage (new nn::fs::MemoryStorage(storageBuffer.get(), StorageSize));
    std::unique_ptr<nn::fs::fsa::IFile> file(new nn::fssystem::StorageFile(pMemoryStorage.get(), nn::fs::OpenMode_Read));

    char buffer[0x100 + 1] = {0};
    NNT_EXPECT_RESULT_FAILURE(
        nn::fs::ResultInvalidOperationForOpenMode,
        file->Write(0, buffer, sizeof(buffer) - 1, nn::fs::WriteOption()));
}

//! @brief 読み込みモードで開いたファイルを SetSize し、ResultInvalidOperationForOpenMode が返却されることを確認する
TEST(StorageFile, InvalidOpenModeSetFileSize)
{
    static const int StorageSize = 64 * 1024;
    std::unique_ptr<uint8_t[]> storageBuffer(new uint8_t[StorageSize]);
    nnt::fs::util::FillBufferWithRandomValue(storageBuffer.get(), StorageSize);
    std::unique_ptr<nn::fs::IStorage> pMemoryStorage (new nn::fs::MemoryStorage(storageBuffer.get(), StorageSize));
    std::unique_ptr<nn::fs::fsa::IFile> file(new nn::fssystem::StorageFile(pMemoryStorage.get(), nn::fs::OpenMode_Read));

    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultInvalidOperationForOpenMode, file->SetSize(0));
    NNT_EXPECT_RESULT_FAILURE(
        nn::fs::ResultInvalidOperationForOpenMode,
        file->SetSize(0x100));
}

//! @brief 読み込みモードで開いたファイルを Flush し、ResultSuccess が返却されることを確認する
TEST(StorageFile, InvalidOpenModeFlushFile)
{
    static const int StorageSize = 64 * 1024;
    std::unique_ptr<uint8_t[]> storageBuffer(new uint8_t[StorageSize]);
    nnt::fs::util::FillBufferWithRandomValue(storageBuffer.get(), StorageSize);
    std::unique_ptr<nn::fs::IStorage> pMemoryStorage (new nn::fs::MemoryStorage(storageBuffer.get(), StorageSize));
    std::unique_ptr<nn::fs::fsa::IFile> file(new nn::fssystem::StorageFile(pMemoryStorage.get(), nn::fs::OpenMode_Read));

    NNT_EXPECT_RESULT_SUCCESS(file->Flush());
}

}
