﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <cstring>

#include <nn/os.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/nn_StaticAssert.h>
#include <nn/fs.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_Optional.h>

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>
#include <nn/crypto/crypto_Aes128XtsEncryptor.h>
#include <nn/crypto/crypto_Aes128XtsDecryptor.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nn/fs/fs_MemoryStorage.h>
#include "testFs_Unit_StorageLayerTestCase.h"
#include "detail/fssrv_BuiltInStorageService.h"
#include "detail/fssrv_SdmmcStorageService.h"
#include <nn/fssrv/fscreator/fssrv_BuiltInStorageCreator.h>

using namespace nn::fs;
using namespace nn::fssrv::fscreator;

namespace {

const int RootPartitionCount = 3;
nn::fs::IStorage* g_MemoryStorage[RootPartitionCount] = {0};

void SetStorage(nn::fs::IStorage* pStorage, MmcPartition id)
{
    NN_STATIC_ASSERT(static_cast<int>(MmcPartition::UserData)       < RootPartitionCount);
    NN_STATIC_ASSERT(static_cast<int>(MmcPartition::BootPartition1) < RootPartitionCount);
    NN_STATIC_ASSERT(static_cast<int>(MmcPartition::BootPartition2) < RootPartitionCount);

    g_MemoryStorage[static_cast<int>(id)] = pStorage;
}

int g_OpenMmcStorageMockCallCount;

nn::Result OpenMmcStorageMock(nn::fs::IStorage** outValue, MmcPartition id) NN_NOEXCEPT
{
    g_OpenMmcStorageMockCallCount++;
    *outValue = g_MemoryStorage[static_cast<int>(id)];
    NN_RESULT_SUCCESS;
}

} // namespace

#define OpenMmcStorage OpenMmcStorageMock
#include "detail/fssrv_BuiltInStorageService.cpp"
#include "fscreator/fssrv_BuiltInStorageCreator.cpp"



using namespace nn::fssrv::detail;

namespace {

void ConstructGptHeaderInterim(char* buffer, int numberOfPartitionEntries)
{
    auto pGptHeader = reinterpret_cast<GptHeader*>(buffer + SectorSize * 1); // skip MBR
    pGptHeader->numberOfPartitionEntries = numberOfPartitionEntries;
    pGptHeader->partitionEntryLba = 2;
    memcpy(pGptHeader->signature, GptHeaderSignature, sizeof(pGptHeader->signature));
}

void ConstructGptEntryInterim(char* buffer, Guid typeGuid, int index, int64_t offset, int64_t size)
{
    const auto pGptHeader = reinterpret_cast<GptHeader*>(buffer + SectorSize * 1);

    auto bufferOffset = pGptHeader->partitionEntryLba * SectorSize + index * sizeof(GptPartitionEntry);

    auto pGptEntry = reinterpret_cast<GptPartitionEntry*>(buffer + bufferOffset);
    pGptEntry->startingLba = offset / SectorSize;
    pGptEntry->endingLba   = (offset + size) / SectorSize - 1;
    pGptEntry->partitionTypeGuid = typeGuid;
}

void ConstructGptEntries(char* buffer, Guid typeGuid, int startIndex, int entryCount, int64_t offset, int64_t size)
{
    for(auto i = 0; i < entryCount; ++i)
    {
        ConstructGptEntryInterim(buffer, typeGuid, startIndex + i, offset, size);
    }
}

void CheckPartitionExistence(IStorage* pBaseStorage, const Guid* pTargetGuid, bool isExpectedExistence, int64_t expectedSize)
{
    std::unique_ptr<nn::fs::IStorage> storage;

    if( isExpectedExistence )
    {
        NNT_EXPECT_RESULT_SUCCESS(OpenGptPartition(&storage, pBaseStorage, pTargetGuid));

        int64_t storageSize;
        NNT_EXPECT_RESULT_SUCCESS(storage->GetSize(&storageSize));
        EXPECT_EQ(expectedSize, storageSize);
    }
    else
    {
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultPartitionNotFound, OpenGptPartition(&storage, pBaseStorage, pTargetGuid));
    }
}

}


// gpt を構築して、guid に応じて gpt ヘッダの検索が正しく出来ることをテスト
TEST(OpenGptPartitionTest, basic)
{
    const int StorageSize = 16 * 1024 * 1024;
    std::unique_ptr<char[]> storageBuffer(new char[StorageSize]);

    MemoryStorage baseStorage(storageBuffer.get(), StorageSize);

    std::unique_ptr<nn::fs::IStorage> storage;


    const auto offset = 0x1000ULL * SectorSize;
    const auto size = 0x123ULL * SectorSize;

    const Guid* pTargetGuid = &BisGuids[10];
    const Guid* pDummyGuid = &BisGuids[0];

    // Invalid gpt
    {
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultGptHeaderSignatureVerificationFailed, OpenGptPartition(&storage, &baseStorage, pTargetGuid));
    }

    // 0 entry, not found
    {
        SCOPED_TRACE("");
        ConstructGptHeaderInterim(storageBuffer.get(), 1);
        CheckPartitionExistence(&baseStorage, pTargetGuid, false, 0);
    }

    // 1 entry, not found
    {
        SCOPED_TRACE("");
        ConstructGptHeaderInterim(storageBuffer.get(), 1);
        ConstructGptEntryInterim(storageBuffer.get(), *pDummyGuid, 0, 0, 0);
        ConstructGptEntryInterim(storageBuffer.get(), *pTargetGuid, 1, offset, size); // out of range
        CheckPartitionExistence(&baseStorage, pTargetGuid, false, 0);
    }

    // 1 entry
    {
        SCOPED_TRACE("");
        ConstructGptHeaderInterim(storageBuffer.get(), 1);
        ConstructGptEntryInterim(storageBuffer.get(), *pTargetGuid, 0, offset, size);
        CheckPartitionExistence(&baseStorage, pTargetGuid, true, size);
    }

    // 4 entries
    {
        SCOPED_TRACE("");
        ConstructGptHeaderInterim(storageBuffer.get(), 4);
        ConstructGptEntries(storageBuffer.get(), *pDummyGuid, 0, 3, 0, 0); // index 0 -- 2
        ConstructGptEntryInterim(storageBuffer.get(), *pTargetGuid, 3, offset, size);
        CheckPartitionExistence(&baseStorage, pTargetGuid, true, size);
    }

    // 5 entries
    {
        SCOPED_TRACE("");
        ConstructGptHeaderInterim(storageBuffer.get(), 5);
        ConstructGptEntries(storageBuffer.get(), *pDummyGuid, 0, 4, 0, 0); // index 0 -- 3
        ConstructGptEntryInterim(storageBuffer.get(), *pTargetGuid, 4, offset, size);
        CheckPartitionExistence(&baseStorage, pTargetGuid, true, size);
    }

    // 16 entries
    {
        SCOPED_TRACE("");
        ConstructGptHeaderInterim(storageBuffer.get(), 16);
        ConstructGptEntries(storageBuffer.get(), *pDummyGuid, 0, 15, 0, 0); // index 0 -- 14
        ConstructGptEntryInterim(storageBuffer.get(), *pTargetGuid, 15, offset, size);
        CheckPartitionExistence(&baseStorage, pTargetGuid, true, size);
    }


}


namespace nn { namespace fssrv { namespace fscreator {

void GetBisEncryptionKey(BuiltInStorageCreator::Key* pOutValueKey, int keyCount, BisPartitionEncryptionKeyId id, int keyGeneration)
{
    memset(pOutValueKey, 0x01, sizeof(BuiltInStorageCreator::Key) * keyCount);
    pOutValueKey[0].value[0] = static_cast<char>(id);
    pOutValueKey[0].value[1] = static_cast<char>(keyGeneration);
}


// BootPartition1/2Root と、 UserDataRoot 上に構築した gpt partition を、対応する BisPartitionId の値で埋める
class BuiltInStorageServiceTestBase : public ::testing::Test
{
protected:

    static const int RootStorageCount = 3;
    static const int StorageSize = 16 * 1024 * 1024;

    std::unique_ptr<char> storageBuffer[RootStorageCount];
    std::unique_ptr<MemoryStorage> pRootStorage[RootStorageCount];

    std::unique_ptr<BuiltInStorageCreator> pBisCreator;

    virtual void SetUp()
    {

        // root 3 partitions
        for(int i=0; i<RootStorageCount; ++i)
        {
            storageBuffer[i].reset(new char[StorageSize]);
            ASSERT_NE(storageBuffer[i], nullptr);

            const nn::fs::BisPartitionId rootPartitionIds[] =
            {
                nn::fs::BisPartitionId::BootPartition1Root,
                nn::fs::BisPartitionId::BootPartition2Root,
                nn::fs::BisPartitionId::UserDataRoot
            };

            memset(storageBuffer[i].get(), static_cast<char>(rootPartitionIds[i]), StorageSize);

            pRootStorage[i].reset(new MemoryStorage(storageBuffer[i].get(), StorageSize));
        }

        // gpt partitions on UserDataRoot
        ConstructGptHeaderInterim(storageBuffer[2].get(), BisGptPartitionCount);
        for(int i=0; i<BisGptPartitionCount; ++i)
        {
            const nn::fs::BisPartitionId gptPartitionIds[] =
            {
                nn::fs::BisPartitionId::CalibrationBinary,
                nn::fs::BisPartitionId::CalibrationFile,
                nn::fs::BisPartitionId::BootConfigAndPackage2Part1,
                nn::fs::BisPartitionId::BootConfigAndPackage2Part2,
                nn::fs::BisPartitionId::BootConfigAndPackage2Part3,
                nn::fs::BisPartitionId::BootConfigAndPackage2Part4,
                nn::fs::BisPartitionId::BootConfigAndPackage2Part5,
                nn::fs::BisPartitionId::BootConfigAndPackage2Part6,
                nn::fs::BisPartitionId::SafeMode,
                nn::fs::BisPartitionId::User,
                nn::fs::BisPartitionId::System,
                // nn::fs::BisPartitionId::SystemProperEncryption は System と同じ領域なので書き込まない
            };

            const int PartitionSize = 1024 * 1024;

            // 先頭 PartitionSize バイト領域は gpt ヘッダのため飛ばす
            ConstructGptEntryInterim(storageBuffer[2].get(), BisGuids[i], i, PartitionSize * (i + 1), PartitionSize - i * SectorSize);

            memset(storageBuffer[2].get() + PartitionSize * (i + 1), static_cast<char>(gptPartitionIds[i]), PartitionSize);
        }

        SetStorage(pRootStorage[0].get(), MmcPartition::BootPartition1);
        SetStorage(pRootStorage[1].get(), MmcPartition::BootPartition2);
        SetStorage(pRootStorage[2].get(), MmcPartition::UserData);
    }

    Result OpenAndCacheBuiltInStoragePartition(nn::fs::IStorage** outValue, nn::fs::BisPartitionId id) NN_NOEXCEPT
    {
        return pBisCreator->OpenAndCacheBuiltInStoragePartition(outValue, id);
    }

};

class BuiltInStorageServiceTest : public BuiltInStorageServiceTestBase
{
protected:
    virtual void SetUp()
    {
        BuiltInStorageServiceTestBase::SetUp();

        BuiltInStorageCreator::Configuration config = { {0} };
        config.getKeyFunc = GetBisEncryptionKey;
        pBisCreator.reset(new BuiltInStorageCreator(config));
        NN_ASSERT_NOT_NULL(pBisCreator);
    }
};


class BuiltInStorageServiceRedirectTest : public BuiltInStorageServiceTestBase
{
protected:
    virtual void SetUp()
    {
        BuiltInStorageServiceTestBase::SetUp();

        BuiltInStorageCreator::Configuration config = { {0} };
        config.getKeyFunc = GetBisEncryptionKey;
        pBisCreator.reset(new BuiltInStorageCreatorRedirectSafe(config));
        NN_ASSERT_NOT_NULL(pBisCreator);
    }
};


// BisPartitionId に応じて 3 root 領域が取得できること
TEST_F(BuiltInStorageServiceTest, root)
{
    const nn::fs::BisPartitionId ids[] =
    {
        nn::fs::BisPartitionId::BootPartition1Root,
        nn::fs::BisPartitionId::BootPartition2Root,
        nn::fs::BisPartitionId::UserDataRoot,
    };

    for(auto i = 0; i < RootStorageCount; ++i)
    {
        IStorage* partition = nullptr;
        NNT_ASSERT_RESULT_SUCCESS(OpenAndCacheBuiltInStoragePartition(&partition, ids[i]));

        char buffer[1];
        NNT_EXPECT_RESULT_SUCCESS(partition->Read(0, buffer, 1));
        EXPECT_EQ(static_cast<char>(ids[i]), buffer[0]);
    }

}

// 2 回目以降はキャッシュから取得され、再生成されないこと
TEST_F(BuiltInStorageServiceTest, cache)
{
    const nn::fs::BisPartitionId ids[] =
    {
        nn::fs::BisPartitionId::BootPartition1Root,
        nn::fs::BisPartitionId::BootPartition2Root,
        nn::fs::BisPartitionId::UserDataRoot,

        nn::fs::BisPartitionId::CalibrationBinary,
        nn::fs::BisPartitionId::CalibrationFile,
        nn::fs::BisPartitionId::BootConfigAndPackage2Part1,
        nn::fs::BisPartitionId::BootConfigAndPackage2Part2,
        nn::fs::BisPartitionId::BootConfigAndPackage2Part3,
        nn::fs::BisPartitionId::BootConfigAndPackage2Part4,
        nn::fs::BisPartitionId::BootConfigAndPackage2Part5,
        nn::fs::BisPartitionId::BootConfigAndPackage2Part6,
        nn::fs::BisPartitionId::SafeMode,
        nn::fs::BisPartitionId::User,
        nn::fs::BisPartitionId::System,
        nn::fs::BisPartitionId::SystemProperEncryption,
    };

    const int PartitionCount = sizeof(ids) / sizeof(ids[0]);

    g_OpenMmcStorageMockCallCount = 0;

    for(auto i = 0; i < PartitionCount; ++i)
    {
        IStorage* partition = nullptr;
        NNT_ASSERT_RESULT_SUCCESS(OpenAndCacheBuiltInStoragePartition(&partition, ids[i]));
    }

    EXPECT_EQ(RootStorageCount, g_OpenMmcStorageMockCallCount);

    g_OpenMmcStorageMockCallCount = 0;

    for(auto i = 0; i < PartitionCount; ++i)
    {
        IStorage* partition = nullptr;
        NNT_ASSERT_RESULT_SUCCESS(OpenAndCacheBuiltInStoragePartition(&partition, ids[i]));
    }

    EXPECT_EQ(0, g_OpenMmcStorageMockCallCount);

}



// UserDataRoot に gpt を構築して、 BisPartitionId に応じて適切な領域が取得でき、正しく暗号化レイヤが被せられること
TEST_F(BuiltInStorageServiceTest, encryption)
{
    SetStorage(nullptr, MmcPartition::BootPartition1);
    SetStorage(nullptr, MmcPartition::BootPartition2);

    const int BufferSize = 32 * 1024;
    char buffer[BufferSize];

    const nn::fs::BisPartitionId ids[] =
    {
        nn::fs::BisPartitionId::CalibrationBinary,
        nn::fs::BisPartitionId::CalibrationFile,
        nn::fs::BisPartitionId::BootConfigAndPackage2Part1,
        nn::fs::BisPartitionId::BootConfigAndPackage2Part2,
        nn::fs::BisPartitionId::BootConfigAndPackage2Part3,
        nn::fs::BisPartitionId::BootConfigAndPackage2Part4,
        nn::fs::BisPartitionId::BootConfigAndPackage2Part5,
        nn::fs::BisPartitionId::BootConfigAndPackage2Part6,
        nn::fs::BisPartitionId::SafeMode,
        nn::fs::BisPartitionId::User,
        nn::fs::BisPartitionId::System,
        nn::fs::BisPartitionId::SystemProperEncryption,
    };

    for(auto i = 0; i < sizeof(ids) / sizeof(ids[0]); ++i)
    {
        IStorage* partition = nullptr;
        NNT_ASSERT_RESULT_SUCCESS(OpenAndCacheBuiltInStoragePartition(&partition, ids[i]));

        memset(buffer, 0xCD, BufferSize);
        NNT_EXPECT_RESULT_SUCCESS(partition->Read(0, buffer, BufferSize));

        uint64_t expected;
        memset(&expected, static_cast<char>(ids[i]), sizeof(expected));

        if( ids[i] == nn::fs::BisPartitionId::CalibrationBinary ||
            ids[i] == nn::fs::BisPartitionId::CalibrationFile   ||
            ids[i] == nn::fs::BisPartitionId::SafeMode ||
            ids[i] == nn::fs::BisPartitionId::User     ||
            ids[i] == nn::fs::BisPartitionId::System   ||
            ids[i] == nn::fs::BisPartitionId::SystemProperEncryption )
        {
            // encrypted
            EXPECT_NE(expected, *reinterpret_cast<uint64_t*>(buffer));

            // TODO: 暗号化内容のチェック
        }
        else
        {
            EXPECT_EQ(expected, *reinterpret_cast<uint64_t*>(buffer));
        }

    }

}



// System と SystemProperEncryption がそれぞれの鍵で同じ領域を指していること
TEST_F(BuiltInStorageServiceTest, properEncryption)
{
    const size_t BufferSize = 1024;
    char buffer[3][BufferSize];

    IStorage* system       = nullptr;
    IStorage* systemProper = nullptr;
    NNT_ASSERT_RESULT_SUCCESS(OpenAndCacheBuiltInStoragePartition(&system,       nn::fs::BisPartitionId::System));
    NNT_ASSERT_RESULT_SUCCESS(OpenAndCacheBuiltInStoragePartition(&systemProper, nn::fs::BisPartitionId::SystemProperEncryption));

    // system 経由で read
    memset(buffer[0], 0xCD, BufferSize);
    NNT_ASSERT_RESULT_SUCCESS(system->Read(0, buffer[0], BufferSize));

    // proper 経由で write
    nnt::fs::util::FillBufferWith32BitCount(buffer[1], BufferSize, 0);
    NNT_ASSERT_RESULT_SUCCESS(systemProper->Write(0, buffer[1], BufferSize));

    // system 経由で 再 read
    memset(buffer[2], 0xCD, BufferSize);
    NNT_ASSERT_RESULT_SUCCESS(system->Read(0, buffer[2], BufferSize));

    // 変化していること
    EXPECT_TRUE( memcmp(buffer[0], buffer[2], BufferSize) != 0 );

    // 鍵が違うこと
    EXPECT_TRUE( memcmp(buffer[1], buffer[2], BufferSize) != 0 );

}

// 通常時は System で真の System パーティションが取得できること
TEST_F(BuiltInStorageServiceTest, systemPartitionRedirection)
{
    IStorage* redirected = nullptr; // system or safemode
    IStorage* system     = nullptr;

    NNT_ASSERT_RESULT_SUCCESS(OpenAndCacheBuiltInStoragePartition(&redirected, nn::fs::BisPartitionId::System));
    NNT_ASSERT_RESULT_SUCCESS(OpenAndCacheBuiltInStoragePartition(&system,     nn::fs::BisPartitionId::SystemProperEncryption));

    int64_t redirectedSize;
    NNT_EXPECT_RESULT_SUCCESS(redirected->GetSize(&redirectedSize));

    int64_t systemSize;
    NNT_EXPECT_RESULT_SUCCESS(system->GetSize(&systemSize));

    EXPECT_EQ(systemSize, redirectedSize);
}

// リダイレクト時は System で Safe パーティションが取得できること
TEST_F(BuiltInStorageServiceRedirectTest, systemPartitionRedirection)
{
    IStorage* redirected = nullptr; // system or safemode
    IStorage* safe       = nullptr;

    NNT_ASSERT_RESULT_SUCCESS(OpenAndCacheBuiltInStoragePartition(&redirected, nn::fs::BisPartitionId::System));
    NNT_ASSERT_RESULT_SUCCESS(OpenAndCacheBuiltInStoragePartition(&safe,       nn::fs::BisPartitionId::SafeMode));

    int64_t redirectedSize;
    NNT_EXPECT_RESULT_SUCCESS(redirected->GetSize(&redirectedSize));

    int64_t safeModeSize;
    NNT_EXPECT_RESULT_SUCCESS(safe->GetSize(&safeModeSize));

    EXPECT_EQ(safeModeSize, redirectedSize);
}


}}}
