﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <cstring>

#include <nn/os.h>

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/fs.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/fs/fsa/fs_IFile.h>

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/fsUtil/testFs_util.h>

#include <nn/fssystem/fs_AsynchronousAccess.h>
#include <nn/fssystem/fs_ServiceContext.h>

#include "testFs_Unit_StorageLayerTestCase.h"

#include <nn/fs/fs_SubStorage.h>

using namespace nn::fssystem;
using namespace nn::crypto;
using namespace nnt::fs::util;

namespace {

typedef nn::fs::ResultSdCardAccessFailed ResultStorageBroken;

const auto PooledThreadCount = 3;
const auto PooledThreadStackSize = 32 * 1024;
nn::fssystem::PooledThread g_PooledThreads[PooledThreadCount];
NN_OS_ALIGNAS_THREAD_STACK
    char g_PooledThreadStack[PooledThreadCount * PooledThreadStackSize] = {};
nn::fssystem::ThreadPool g_ThreadPool(g_PooledThreads, PooledThreadCount);

// 指定した offset にアクセスしたらエラーを返すストレージ
class BreakableMemoryStorage : public nn::fs::MemoryStorage
{
public:
    BreakableMemoryStorage(void* buffer, int64_t size) NN_NOEXCEPT
        : nn::fs::MemoryStorage(buffer, size)
        , m_BreakOffset(0)
        , m_IsBroken(false)
    {
    }

    virtual ~BreakableMemoryStorage() NN_NOEXCEPT NN_OVERRIDE
    {
    }

    void SetBreak(int64_t offset) NN_NOEXCEPT
    {
        m_IsBroken = true;
        m_BreakOffset = offset;
    }

    void ClearBreak() NN_NOEXCEPT
    {
        m_IsBroken = false;
    }

    virtual nn::Result Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        if( m_IsBroken
            && (offset <= m_BreakOffset)
            && (offset + static_cast<int64_t>(size) > m_BreakOffset))
        {
            NN_RESULT_THROW(ResultStorageBroken());
        }
        NN_RESULT_DO(MemoryStorage::Read(offset, buffer, size));
        NN_RESULT_SUCCESS;
    }

private:
    int64_t m_BreakOffset;
    bool m_IsBroken;
};


class AsynchronousAccessStorageTestBase : public ::testing::Test, public ::testing::WithParamInterface<nn::fs::PriorityRaw>
{
protected:
    static const auto StorageSize = 16 * 1024 * 1024;

protected:
    AsynchronousAccessStorageTestBase() NN_NOEXCEPT
        : m_Memory(),
        m_pBaseStorage(std::make_shared<BreakableMemoryStorage>(m_Memory.get(), StorageSize)),
        m_pStorage()
    {
        if( GetParam() == static_cast<nn::fs::PriorityRaw>(-1) )
        {
            nn::fssystem::UnregisterServiceContext();
        }
        else
        {
            nn::fssystem::RegisterServiceContext(&m_Context);
            m_Context.SetPriority(GetParam());
        }
    }

    virtual ~AsynchronousAccessStorageTestBase() NN_NOEXCEPT NN_OVERRIDE
    {
        nn::fssystem::UnregisterServiceContext();
    }

protected:
    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_ThreadPool.Initialize(
            g_PooledThreadStack, PooledThreadCount * PooledThreadStackSize));
        m_Memory.reset(new char[StorageSize]);
        m_pBaseStorage = std::make_shared<BreakableMemoryStorage>(m_Memory.get(), StorageSize);
        SetUpBaseStorage();
        m_pStorage.reset(new nn::fssystem::AsynchronousAccessStorage(GetBaseStorage(), &g_ThreadPool));
        FillBufferWith32BitCount(m_Memory.get(), StorageSize, 0);
    }

    nn::fs::IStorage* GetStorage() NN_NOEXCEPT
    {
        return m_pStorage.get();
    }

    virtual const std::shared_ptr<nn::fs::IStorage> GetBaseStorage() NN_NOEXCEPT
    {
        return m_pBaseStorage;
    }

    virtual void SetUpBaseStorage() NN_NOEXCEPT = 0;

    void BreakStorage(int64_t offset)
    {
        m_pBaseStorage->SetBreak(offset);
    }

    void ReasetStorageBroken()
    {
        m_pBaseStorage->ClearBreak();
    }

private:
    std::unique_ptr<char[]> m_Memory;
    std::shared_ptr<BreakableMemoryStorage> m_pBaseStorage;
    std::unique_ptr<nn::fssystem::AsynchronousAccessStorage> m_pStorage;
    nn::fssystem::ServiceContext m_Context;
};

class AsynchronousAccessStorageTest : public AsynchronousAccessStorageTestBase
{
protected:
    virtual ~AsynchronousAccessStorageTest() NN_NOEXCEPT NN_OVERRIDE {}

protected:
    virtual void SetUpBaseStorage() NN_NOEXCEPT NN_OVERRIDE {}
};

class AsynchronousAccessStorageRandomAllocationFailureTest : public AsynchronousAccessStorageTestBase
{
protected:
    virtual ~AsynchronousAccessStorageRandomAllocationFailureTest() NN_NOEXCEPT NN_OVERRIDE {}

protected:
    virtual void SetUpBaseStorage() NN_NOEXCEPT NN_OVERRIDE
    {
        m_Mt.seed(nnt::fs::util::GetRandomSeed());
        m_RandomFailureStorage = std::make_shared<nnt::fs::util::RandomAllocationFailureStorage>(
            AsynchronousAccessStorageTestBase::GetBaseStorage().get(), &m_Mt
        );
        m_RandomFailureStorage->SetDivider(1000);
    }

    virtual const std::shared_ptr<nn::fs::IStorage> GetBaseStorage() NN_NOEXCEPT NN_OVERRIDE
    {
        return m_RandomFailureStorage;
    }

    int GetOperationCount() NN_NOEXCEPT
    {
        return m_RandomFailureStorage->GetOperationCount();
    }

private:
    std::mt19937 m_Mt;
    std::shared_ptr<nnt::fs::util::RandomAllocationFailureStorage> m_RandomFailureStorage;
};

class AsynchronousAccessStorageAccessRangeCheckTest : public AsynchronousAccessStorageTestBase
{
protected:
    virtual ~AsynchronousAccessStorageAccessRangeCheckTest() NN_NOEXCEPT NN_OVERRIDE {}

protected:
    virtual void SetUpBaseStorage() NN_NOEXCEPT NN_OVERRIDE
    {
        m_pRecordingAccessRangeStorage = std::make_shared<nnt::fs::util::RecordingAccessRangeStorage>(
            AsynchronousAccessStorageTestBase::GetBaseStorage().get());
    }

    virtual const std::shared_ptr<nn::fs::IStorage> GetBaseStorage() NN_NOEXCEPT NN_OVERRIDE
    {
        return m_pRecordingAccessRangeStorage;
    }

    nnt::fs::util::RecordingAccessRangeStorage* GetRecordingAccessRangeStorage() NN_NOEXCEPT
    {
        return m_pRecordingAccessRangeStorage.get();
    }

private:
    std::shared_ptr<nnt::fs::util::RecordingAccessRangeStorage> m_pRecordingAccessRangeStorage;
};

class AsynchronousAccessStorageLargeSizeTest : public ::testing::Test
{
protected:
    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
    {
        m_pBaseStorage.reset(new nnt::fs::util::VirtualMemoryStorage);
        m_pBaseStorage->Initialize(static_cast<int64_t>(64) * 1024 * 1024 * 1024 + 512 * 1024);
        m_pStorage.reset(new nn::fssystem::AsynchronousAccessStorage(m_pBaseStorage, &g_ThreadPool));
    }

    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
    {
        m_pBaseStorage->Finalize();
    }

    nn::fs::IStorage* GetStorage() NN_NOEXCEPT
    {
        return m_pStorage.get();
    }

private:
    std::shared_ptr<nnt::fs::util::VirtualMemoryStorage> m_pBaseStorage;
    std::unique_ptr<nn::fssystem::AsynchronousAccessStorage> m_pStorage;
};

TEST_P(AsynchronousAccessStorageTest, GetSize)
{
    TestGetSize(GetStorage(), StorageSize);
}

TEST_P(AsynchronousAccessStorageTest, StorageContent)
{
    for( auto bufferSize = 1024 * 1024; bufferSize <= StorageSize; bufferSize <<= 1 )
    {
        TestStorageContent(GetStorage(), bufferSize);
    }
}

TEST_P(AsynchronousAccessStorageTest, FailStorage)
{
    size_t bufferSize = 1024 * 1024;
    std::unique_ptr<char[]> buffer(new char[bufferSize]);

    NN_UTIL_SCOPE_EXIT
    {
        ReasetStorageBroken();
    };
    auto testFailStorage = [&](int64_t offset, size_t size, int64_t errorOffset)
    {
        BreakStorage(errorOffset);
        NNT_EXPECT_RESULT_FAILURE(
            ResultStorageBroken,
            GetStorage()->Read(offset, buffer.get(), size)
        );
    };
    testFailStorage(0, bufferSize, 0);
    testFailStorage(bufferSize, bufferSize, bufferSize / 2 * 3);
    testFailStorage(StorageSize - bufferSize, bufferSize, StorageSize - 1);
}

TEST_P(AsynchronousAccessStorageTest, BoundaryOffsetSize)
{
    TestBoundaryOffsetSize(GetStorage(), 1);
}
TEST_P(AsynchronousAccessStorageTest, WriteRead)
{
    for( auto bufferSize = 1024 * 1024; bufferSize <= StorageSize / 4; bufferSize <<= 1 )
    {
        TestWriteReadStorage(GetStorage(), bufferSize);
    }
}

TEST_P(AsynchronousAccessStorageTest, ConcurrentWriteRead)
{
    TestConcurrentWriteRead(GetStorage());
}

TEST_P(AsynchronousAccessStorageTest, FractionWriteRead)
{
    const auto readSize = 512 * 1024;
    TestFractionWriteRead(GetStorage(), 0, readSize, readSize + 1024);
}

TEST_P(AsynchronousAccessStorageRandomAllocationFailureTest, RandomRangeRead)
{
    static const int TestCount = 1000;
    TestRandomRangeRead(GetStorage(), TestCount);
    NN_LOG("Operation Count %d\n", GetOperationCount());
}

TEST_P(AsynchronousAccessStorageAccessRangeCheckTest, AlignmentAndSplitSizeCheckForReading)
{
    auto priority = GetParam();
    nnt::fs::util::AsynchronousAccessAlignmentAndSplitSizeChecker checker(
        priority,
        GetRecordingAccessRangeStorage()->GetRecorder(),
        GetStorage()
    );
    checker.CheckForReading();
}

TEST_P(AsynchronousAccessStorageAccessRangeCheckTest, AlignmentAndSplitSizeCheckForWriting)
{
    auto priority = GetParam();
    nnt::fs::util::AsynchronousAccessAlignmentAndSplitSizeChecker checker(
        priority,
        GetRecordingAccessRangeStorage()->GetRecorder(),
        GetStorage()
    );
    checker.CheckForWriting();
}

TEST_F(AsynchronousAccessStorageLargeSizeTest, ReadWrite)
{
    TestWriteReadStorageWithLargeOffset(GetStorage(), 512 * 1024);
}

nn::fs::PriorityRaw priorityParam[] = {
    static_cast<nn::fs::PriorityRaw>(-1),
    nn::fs::PriorityRaw_Background,
    nn::fs::PriorityRaw_Low,
    nn::fs::PriorityRaw_Normal,
    nn::fs::PriorityRaw_Realtime
};

INSTANTIATE_TEST_CASE_P(WithPriority,
                        AsynchronousAccessStorageTest,
                        ::testing::ValuesIn(priorityParam));
INSTANTIATE_TEST_CASE_P(WithPriority,
                        AsynchronousAccessStorageAccessRangeCheckTest,
                        ::testing::ValuesIn(priorityParam));


nn::fs::PriorityRaw priorityParamForRandomTest[] = {
    nn::fs::PriorityRaw_Normal,
};

INSTANTIATE_TEST_CASE_P(WithPriority,
                        AsynchronousAccessStorageRandomAllocationFailureTest,
                        ::testing::ValuesIn(priorityParamForRandomTest));

}
