﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fssystem/save/fs_SavePath.h>
#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/result/testResult_Assert.h>

// 文字列の正規化のテストを行います。
TEST(FsPathTest, Normalize)
{
    // 異常系
    {
        char out[256];
        size_t dstLength;

        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultInvalidPathFormat,
            nn::fssystem::PathTool::Normalize(
                out,
                &dstLength,
                "dir",
                sizeof(out),
                false
            )
        );

        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultDirectoryUnobtainable,
            nn::fssystem::PathTool::Normalize(
                out,
                &dstLength,
                "/dir/../..",
                sizeof(out),
                false
            )
        );

        // UNC 指定の場合でホスト名に相当する部分は遡れる
        NNT_EXPECT_RESULT_SUCCESS(
            nn::fssystem::PathTool::Normalize(
                out,
                &dstLength,
                "//net/dir/../..",
                sizeof(out),
                true
            )
        );
    }

    // 正常系 + バッファ不足
    struct PathData
    {
        const char* pIn;
        const char* pOut;
        size_t length;
        int32_t outSize; // 正の値であれば正規化後バッファサイズを制限します。
        bool isBufferEnough;
        bool isUncPreserved;
    };
    static const PathData TestData[] =
    {
        // '/' に関するテスト
        { "/", "/", 1, -1, true, false },
        { "//", "/", 1, -1, true, false },
        { "/file", "/file", 5, -1, true, false },
        { "/dir/", "/dir", 4, -1, true, false },
        { "/dir//" , "/dir", 4, -1, true, false },
        { "/dir/dir2" , "/dir/dir2", 9, -1, true, false },
        { "/dir/dir2/" , "/dir/dir2", 9, -1, true, false },

        // '.' に関するテスト
        { "/./dir/../", "/", 1, -1, true, false },
        { "/./dir//dir2/../", "/dir", 4, -1, true, false },
        { "/./dir//dir2/.././//..", "/", 1, -1, true, false },
        { "/.", "/", 1, -1, true, false },
        { "/dir/.", "/dir", 4, -1, true, false },
        { "/dir/.../...", "/dir/.../...", 12, -1, true, false },

        // outSize 指定に関するテスト
        { "/", "", 0, 0, false, false },
        { "/file", "", 0, 1, false, false },
        { "/file", "/", 1, 2, false, false },
        { "/file", "/fil", 4, 5, false, false },
        { "/file", "/file", 5, 6, true, false },
        { "/./dir//dir2/../", "/d", 2, 3, false, false },
        { "/dir/dir2", "/dir", 4, 5, false, false },

        // UNC に指定に関するテスト
        { "//", "/", 1, -1, true, true },
        { "///net", "///net", 6, -1, true, true },
        { "//net//./shared/", "//net/shared", 12, -1, true, true },

        // UNC + outSize 指定のテスト
        { "////net", "", 3, 4, false, true },
    };
    static const int TestDataLength = sizeof(TestData) / sizeof(TestData[0]);

    for( size_t i = 0; i < TestDataLength; ++i )
    {
        char out[256];
        size_t dstLength;
        nn::Result result = nn::fssystem::PathTool::Normalize(
            out,
            &dstLength,
            TestData[i].pIn,
            (TestData[i].outSize < 0) ? 256 : TestData[i].outSize,
            TestData[i].isUncPreserved
        );
        if( TestData[i].isBufferEnough )
        {
            NNT_ASSERT_RESULT_SUCCESS(result);
            ASSERT_EQ(TestData[i].length, dstLength);
            if( 0 < dstLength )
            {
                ASSERT_EQ(0, std::memcmp(TestData[i].pOut, out, (std::strlen(out) + 1) * sizeof(char)));
            }

            if( !TestData[i].isUncPreserved )
            {
                bool isNormalized;
                NNT_ASSERT_RESULT_SUCCESS(nn::fssystem::PathTool::IsNormalized(&isNormalized, out));
                EXPECT_TRUE(isNormalized) << TestData[i].pIn << " " << out;
            }

            char out2[256];
            NNT_ASSERT_RESULT_SUCCESS(
                nn::fssystem::PathTool::Normalize(
                    out2,
                    &dstLength,
                    out,
                    sizeof(out2),
                    TestData[i].isUncPreserved
                )
            );
            EXPECT_STREQ(out, out2) << TestData[i].pIn << " " << out << " " << out2;
        }
        else
        {
            NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultTooLongPath, result);
        }
    }
} // NOLINT(impl/function_size)

// パスの比較
TEST(FsPathTest, IsEqualPath)
{
    nn::fssystem::save::Path path1a;
    NNT_ASSERT_RESULT_SUCCESS(path1a.Initialize("/abcdefg"));
    nn::fssystem::save::Path path1b;
    NNT_ASSERT_RESULT_SUCCESS(path1b.Initialize("/abcdefg"));
    nn::fssystem::save::Path path1c;
    NNT_ASSERT_RESULT_SUCCESS(path1c.Initialize("/abcdefc"));
    nn::fssystem::save::Path path1d;
    NNT_ASSERT_RESULT_SUCCESS(path1d.Initialize("/abcd"));

    ASSERT_TRUE(nn::fssystem::save::Path::IsEqualPath(path1a, path1b));
    ASSERT_FALSE(nn::fssystem::save::Path::IsEqualPath(path1a, path1c));
    ASSERT_FALSE(nn::fssystem::save::Path::IsEqualPath(path1a, path1d));
}

// パスの抱合
TEST(FsPathTest, IsContainingPath)
{
    nn::fssystem::save::Path path1;
    NNT_ASSERT_RESULT_SUCCESS(path1.Initialize("/"));
    nn::fssystem::save::Path path2;
    NNT_ASSERT_RESULT_SUCCESS(path2.Initialize("/abcdefg"));
    nn::fssystem::save::Path path3;
    NNT_ASSERT_RESULT_SUCCESS(path3.Initialize("/abcdefg/hijk"));
    nn::fssystem::save::Path path4;
    NNT_ASSERT_RESULT_SUCCESS(path4.Initialize("/abcdefg/hijk/lmnop"));
    nn::fssystem::save::Path path5;
    NNT_ASSERT_RESULT_SUCCESS(path5.Initialize("/abcdefg/hijm/lmnop"));

    ASSERT_TRUE(nn::fssystem::save::Path::IsContainingPath(path2, path1));
    ASSERT_TRUE(nn::fssystem::save::Path::IsContainingPath(path3, path2));
    ASSERT_TRUE(nn::fssystem::save::Path::IsContainingPath(path4, path3));
    ASSERT_TRUE(nn::fssystem::save::Path::IsContainingPath(path5, path2));
    ASSERT_FALSE(nn::fssystem::save::Path::IsContainingPath(path5, path3));
    ASSERT_FALSE(nn::fssystem::save::Path::IsContainingPath(path5, path4));
}

// 親子パス
TEST(FsPathTest, IsInsidePath)
{
    nn::fssystem::save::Path path1;
    NNT_ASSERT_RESULT_SUCCESS(path1.Initialize("/"));
    nn::fssystem::save::Path path2;
    NNT_ASSERT_RESULT_SUCCESS(path2.Initialize("/d"));
    nn::fssystem::save::Path path3;
    NNT_ASSERT_RESULT_SUCCESS(path3.Initialize("/director"));
    nn::fssystem::save::Path path4;
    NNT_ASSERT_RESULT_SUCCESS(path4.Initialize("/dir1/dir2/dir3"));
    nn::fssystem::save::Path path5;
    NNT_ASSERT_RESULT_SUCCESS(path5.Initialize("/dir1/dir2/dir3/director"));
    nn::fssystem::save::Path path6;
    NNT_ASSERT_RESULT_SUCCESS(path6.Initialize("/e/c"));
    nn::fssystem::save::Path path7;
    NNT_ASSERT_RESULT_SUCCESS(path7.Initialize("/dd/c"));
    nn::fssystem::save::Path path8;
    NNT_ASSERT_RESULT_SUCCESS(path8.Initialize("/dir1/dir2/dir3/director/y"));

    ASSERT_TRUE(nn::fssystem::save::Path::IsInsidePath(path1, path2));
    ASSERT_TRUE(nn::fssystem::save::Path::IsInsidePath(path1, path3));
    ASSERT_TRUE(nn::fssystem::save::Path::IsInsidePath(path4, path5));

    ASSERT_FALSE(nn::fssystem::save::Path::IsInsidePath(path2, path2));
    ASSERT_FALSE(nn::fssystem::save::Path::IsInsidePath(path2, path6));
    ASSERT_FALSE(nn::fssystem::save::Path::IsInsidePath(path2, path7));
    ASSERT_FALSE(nn::fssystem::save::Path::IsInsidePath(path4, path8));
}

// ルートの親の取得
TEST(FsPathTest, RootParent)
{
    const auto ExpectUnobtainable = [](const char* path) NN_NOEXCEPT
    {
        char buffer[256];
        size_t length = 0;
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultDirectoryUnobtainable,
            nn::fssystem::PathTool::Normalize(buffer, &length, path, strlen(path)))
            << "PathTool::Normalize(\"" << path << "\") returned unexpected result";
    };

    ExpectUnobtainable("/..");
    ExpectUnobtainable("/../");
    ExpectUnobtainable("/../.");
    ExpectUnobtainable("/foo/../..");
    ExpectUnobtainable("/foo/../../");
    ExpectUnobtainable("/foo/./../..");
    ExpectUnobtainable("/foo/.././..");
    ExpectUnobtainable("/foo/../../.");
    ExpectUnobtainable("/foo/../../..");
    ExpectUnobtainable("/foo/../../bar");
    ExpectUnobtainable("/foo/./../../bar");
    ExpectUnobtainable("/foo/.././../bar");
    ExpectUnobtainable("/foo/../.././bar");
}

// 正規化済み判定
TEST(FsPathTest, IsNormalized)
{
    const auto ExpectInvalidFormat = [](const char* path) NN_NOEXCEPT
    {
        auto isNormalized = false;
        const auto result = nn::fssystem::PathTool::IsNormalized(&isNormalized, path);
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultInvalidPathFormat, result)
            << "PathTool::IsNormalized(\"" << path << "\") succeeded";
    };

    const auto ExpectNormalized = [](const char* path, bool expectNormalized) NN_NOEXCEPT
    {
        auto isNormalized = false;
        const auto result = nn::fssystem::PathTool::IsNormalized(&isNormalized, path);
        NNT_ASSERT_RESULT_SUCCESS(result)
            << "PathTool::IsNormalized(\"" << path << "\") failed";
        EXPECT_EQ(expectNormalized, isNormalized)
            << "PathTool::IsNormalized(\"" << path << "\") returned " << isNormalized;
    };

    ExpectInvalidFormat("");
    ExpectInvalidFormat("C");
    ExpectInvalidFormat("C/");
    ExpectInvalidFormat("0:/");
    ExpectInvalidFormat("::/");
    ExpectInvalidFormat(".:/");

    ExpectNormalized("/.", false);
    ExpectNormalized("/..", false);
    ExpectNormalized("//", false);
    ExpectNormalized("/foo/.", false);
    ExpectNormalized("/foo/..", false);
    ExpectNormalized("/foo/", false);
    ExpectNormalized("/foo//", false);
    ExpectNormalized("/./foo", false);
    ExpectNormalized("/../foo", false);
    ExpectNormalized("//foo", false);
    ExpectNormalized("/foo/./bar", false);
    ExpectNormalized("/foo/../bar", false);
    ExpectNormalized("/foo//bar", false);
    ExpectNormalized("/foo//bar/.././", false);

    ExpectNormalized("C:/", true);
    ExpectNormalized("C:/Windows/Temp", true);
    ExpectNormalized("C:/Windows//Temp/.././", true);
    ExpectNormalized("/", true);
    ExpectNormalized("/.txt", true);
    ExpectNormalized("/..txt", true);
    ExpectNormalized("/foo.txt", true);
    ExpectNormalized("/foo/bar.txt", true);
    ExpectNormalized("/.../foo.txt", true);
    ExpectNormalized("/foo./bar", true);
    ExpectNormalized("/foo../bar", true);
}

// サブパス判定
TEST(FsPathTest, IsSubpath)
{
    struct PathData
    {
        const char* pPath1;
        const char* pPath2;
        bool isSubpath;
    };
    static const PathData TestData[] =
    {
        // ルートのテスト
        {"/", "/", false},
        {"/", "/path", true},

        // 非ルートのテスト
        {"/path", "/path", false},
        {"/path", "/path1", false},
        {"/path", "/path/to", true},
        {"/path/to/1", "/path/to/2", false},
        {"/path/to/1/../2", "/path/to/2", false},
        {"/path/to/1/../2/3", "/path/to/2", true},

        // UNC のテスト
        {"//unc", "/", false},
        {"//unc", "//unc", false},
        {"//unc", "//unc1", false},
        {"//unc/path", "/path", false},
        {"//unc/path", "//unc/path", false},
        {"//unc/path", "//unc/path/to", true},
        {"//unc/path ", "//unc/path/to", false},
        {"//unc/path/to/1", "//unc/path/to/2", false},
        {"//unc/path/to/1", "//unc/path/to/ ", false},
    };
    static const int TestDataLength = sizeof(TestData) / sizeof(TestData[0]);

    for( size_t i = 0; i < TestDataLength; ++i )
    {
        nn::Result result;
        char path1[256];
        char path2[256];
        size_t dstLength;

        result = nn::fssystem::PathTool::Normalize(
            path1,
            &dstLength,
            TestData[i].pPath1,
            sizeof(path1),
            true
        );
        NNT_ASSERT_RESULT_SUCCESS(result);
        result = nn::fssystem::PathTool::Normalize(
            path2,
            &dstLength,
            TestData[i].pPath2,
            sizeof(path2),
            true
        );
        NNT_ASSERT_RESULT_SUCCESS(result);

        if( TestData[i].isSubpath )
        {
            EXPECT_TRUE(nn::fssystem::PathTool::IsSubpath(path1, path2))
                << "PathTool::IsSubpath(\"" << path1 << "\", \"" << path2 << "\") returns false.";
            EXPECT_TRUE(nn::fssystem::PathTool::IsSubpath(path2, path1))
                << "PathTool::IsSubpath(\"" << path2 << "\", \"" << path1 << "\") returns false.";
        }
        else
        {
            EXPECT_FALSE(nn::fssystem::PathTool::IsSubpath(path1, path2))
                << "PathTool::IsSubpath(\"" << path1 << "\", \"" << path2 << "\") returns true.";
            EXPECT_FALSE(nn::fssystem::PathTool::IsSubpath(path2, path1))
                << "PathTool::IsSubpath(\"" << path2 << "\", \"" << path1 << "\") returns true.";
        }
    }
}
