﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <random>
#include <nn/fssystem/save/fs_JournalStorage.h>
#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/fsUtil/testFs_util.h>
#include "testFs_util_CommonStorageTests.h"

namespace {

// フォーマットパラメータ
struct FormatTestParam
{
    int64_t offsetMeta;
    int64_t offsetData;
    uint32_t countMapEntry;
    uint32_t countReserved;
    uint32_t countMapEntryExpand;
    uint32_t countReservedExpand;
    int64_t sizeBlock;
};

// フォーマットパラメータの作成
nnt::fs::util::Vector<FormatTestParam> MakeFormatTestParam() NN_NOEXCEPT
{
    nnt::fs::util::Vector<FormatTestParam> result;

    {
        FormatTestParam data;
        data.offsetMeta = 0;
        data.offsetData = 0;
        data.countMapEntry = 1;
        data.countReserved = 1;
        data.countMapEntryExpand = 0;
        data.countReservedExpand = 0;
        data.sizeBlock = 512;
        result.push_back(data);
    }

    {
        FormatTestParam data;
        data.offsetMeta = 0;
        data.offsetData = 0;
        data.countMapEntry = 128;
        data.countReserved = 32;
        data.countMapEntryExpand = 16;
        data.countReservedExpand = 0;
        data.sizeBlock = 4096;
        result.push_back(data);
    }

    {
        FormatTestParam data;
        data.offsetMeta = 32;
        data.offsetData = 32;
        data.countMapEntry = 160;
        data.countReserved = 42;
        data.countMapEntryExpand = 40;
        data.countReservedExpand = 8;
        data.sizeBlock = 16 * 1024;
        result.push_back(data);
    }

    return result;
}

}

// フォーマットテストクラス
class JournalStorageFormatTest : public ::testing::TestWithParam< FormatTestParam >
{
};

// フォーマットテスト
TEST_P(JournalStorageFormatTest, Format)
{
    static const int64_t SizeControlArea = sizeof(nn::fssystem::save::JournalStorage::ControlArea);

    const FormatTestParam& param = GetParam();

    const uint32_t countMapEntry = param.countMapEntry;
    const uint32_t countReserved = param.countReserved;

    const int64_t sizeBlock = param.sizeBlock;
    const int64_t sizeArea = (countMapEntry + countReserved) * sizeBlock;
    const int64_t sizeReserved = countReserved * sizeBlock;
    const size_t offsetMeta = static_cast<size_t>(param.offsetMeta);

    int64_t sizeTable;
    int64_t sizeBitmapUpdatedPhysical;
    int64_t sizeBitmapUpdatedVirtual;
    int64_t sizeBitmapUnassigned;
    nn::fssystem::save::JournalStorage::QueryMappingMetaSize(
        &sizeTable,
        &sizeBitmapUpdatedPhysical,
        &sizeBitmapUpdatedVirtual,
        &sizeBitmapUnassigned,
        sizeArea,
        sizeReserved,
        sizeBlock
    );

    nnt::fs::util::SafeMemoryStorage
        storageControlArea(SizeControlArea + offsetMeta);
    nnt::fs::util::SafeMemoryStorage
        storageTable(sizeTable + offsetMeta);
    nnt::fs::util::SafeMemoryStorage
        storageBitmapUpdatedPhysical(sizeBitmapUpdatedPhysical + offsetMeta);
    nnt::fs::util::SafeMemoryStorage
        storageBitmapUpdatedVirtual(sizeBitmapUpdatedVirtual + offsetMeta);
    nnt::fs::util::SafeMemoryStorage
        storageBitmapUnassigned(sizeBitmapUnassigned + offsetMeta);

    // フォーマットが関係ない範囲に書き込まないかを確認するためのデータを書き込みます。
    if( offsetMeta > 0 )
    {
        storageControlArea.FillBuffer(0, offsetMeta, 0x12);
        storageTable.FillBuffer(0, offsetMeta, 0x12);
        storageBitmapUpdatedPhysical.FillBuffer(0, offsetMeta, 0x12);
        storageBitmapUpdatedVirtual.FillBuffer(0, offsetMeta, 0x12);
        storageBitmapUnassigned.FillBuffer(0, offsetMeta, 0x12);
    }

    nn::fs::SubStorage subStorageControlArea(
        &storageControlArea,
        offsetMeta,
        SizeControlArea
    );
    nn::fs::SubStorage subStorageTable(
        &storageTable,
        offsetMeta,
        sizeTable
    );
    nn::fs::SubStorage subStorageBitmapUpdatedPhysical(
        &storageBitmapUpdatedPhysical,
        offsetMeta,
        sizeBitmapUpdatedPhysical
    );
    nn::fs::SubStorage subStorageBitmapUpdatedVirtual(
        &storageBitmapUpdatedVirtual,
        offsetMeta,
        sizeBitmapUpdatedVirtual
    );
    nn::fs::SubStorage subStorageBitmapUnassigned(
        &storageBitmapUnassigned,
        offsetMeta,
        sizeBitmapUnassigned
    );

    NNT_ASSERT_RESULT_SUCCESS(
        nn::fssystem::save::JournalStorage::Format(
            subStorageControlArea,
            subStorageTable,
            subStorageBitmapUpdatedPhysical,
            subStorageBitmapUpdatedVirtual,
            subStorageBitmapUnassigned,
            sizeArea,
            sizeReserved,
            sizeBlock
        )
    );

    // フォーマットが関係ない範囲に書き込みを行なっていないか確認します。
    if( offsetMeta > 0 )
    {
        ASSERT_EQ(true, storageControlArea.VerifyBuffer(0, offsetMeta, 0x12));
        ASSERT_EQ(true, storageTable.VerifyBuffer(0, offsetMeta, 0x12));
        ASSERT_EQ(true, storageBitmapUpdatedPhysical.VerifyBuffer(0, offsetMeta, 0x12));
        ASSERT_EQ(true, storageBitmapUpdatedVirtual.VerifyBuffer(0, offsetMeta, 0x12));
        ASSERT_EQ(true, storageBitmapUnassigned.VerifyBuffer(0, offsetMeta, 0x12));
    }

    EXPECT_EQ(true, storageControlArea.CheckValid());
    EXPECT_EQ(true, storageTable.CheckValid());
    EXPECT_EQ(true, storageBitmapUpdatedPhysical.CheckValid());
    EXPECT_EQ(true, storageBitmapUpdatedVirtual.CheckValid());
    EXPECT_EQ(true, storageBitmapUnassigned.CheckValid());
}

// フォーマット/マウントテスト
TEST_P(JournalStorageFormatTest, FormatAndInitialize)
{
    static const int64_t SizeCacheBlock = sizeof(nn::fssystem::save::JournalStorage::ControlArea);

    const FormatTestParam& param = GetParam();

    const uint32_t countMapEntry = param.countMapEntry;
    const uint32_t countReserved = param.countReserved;

    const int64_t sizeBlock = param.sizeBlock;
    const int64_t sizeArea = (countMapEntry + countReserved) * sizeBlock;
    const int64_t sizeReserved = countReserved * sizeBlock;
    const size_t offsetMeta = static_cast<size_t>(param.offsetMeta);

    int64_t sizeTable;
    int64_t sizeBitmapUpdatedPhysical;
    int64_t sizeBitmapUpdatedVirtual;
    int64_t sizeBitmapUnassigned;
    nn::fssystem::save::JournalStorage::QueryMappingMetaSize(
        &sizeTable,
        &sizeBitmapUpdatedPhysical,
        &sizeBitmapUpdatedVirtual,
        &sizeBitmapUnassigned,
        sizeArea,
        sizeReserved,
        sizeBlock
    );

    nnt::fs::util::SafeMemoryStorage
        storageControlArea(SizeCacheBlock + offsetMeta);
    nnt::fs::util::SafeMemoryStorage
        storageTable(sizeTable + offsetMeta);
    nnt::fs::util::SafeMemoryStorage
        storageBitmapUpdatedPhysical(sizeBitmapUpdatedPhysical + offsetMeta);
    nnt::fs::util::SafeMemoryStorage
        storageBitmapUpdatedVirtual(sizeBitmapUpdatedVirtual + offsetMeta);
    nnt::fs::util::SafeMemoryStorage
        storageBitmapUnassigned(sizeBitmapUnassigned + offsetMeta);

    const size_t offsetData = static_cast<size_t>(param.offsetData);
    const int64_t sizeData = sizeArea + offsetData;
    nnt::fs::util::SafeMemoryStorage storageData(sizeData);

    // フォーマットが関係ない範囲に書き込まないかを確認するための
    // データを書き込みます。
    if( offsetMeta > 0 )
    {
        storageControlArea.FillBuffer(0, offsetMeta, 0x12);
        storageTable.FillBuffer(0, offsetMeta, 0x12);
        storageBitmapUpdatedPhysical.FillBuffer(0, offsetMeta, 0x12);
        storageBitmapUpdatedVirtual.FillBuffer(0, offsetMeta, 0x12);
        storageBitmapUnassigned.FillBuffer(0, offsetMeta, 0x12);
    }
    if( offsetData > 0 )
    {
        storageData.FillBuffer(0, offsetData, 0x12);
    }

    nn::fs::SubStorage subStorageControlArea(
        &storageControlArea,
        offsetMeta,
        SizeCacheBlock
    );
    nn::fs::SubStorage subStorageTable(
        &storageTable,
        offsetMeta,
        sizeTable
    );
    nn::fs::SubStorage subStorageBitmapUpdatedPhysical(
        &storageBitmapUpdatedPhysical,
        offsetMeta,
        sizeBitmapUpdatedPhysical
    );
    nn::fs::SubStorage subStorageBitmapUpdatedVirtual(
        &storageBitmapUpdatedVirtual,
        offsetMeta,
        sizeBitmapUpdatedVirtual
    );
    nn::fs::SubStorage subStorageBitmapUnassigned(
        &storageBitmapUnassigned,
        offsetMeta,
        sizeBitmapUnassigned
    );
    nn::fs::SubStorage subStorageData(
        &storageData,
        offsetData,
        sizeData
    );

    NNT_ASSERT_RESULT_SUCCESS(
        nn::fssystem::save::JournalStorage::Format(
            subStorageControlArea,
            subStorageTable,
            subStorageBitmapUpdatedPhysical,
            subStorageBitmapUpdatedVirtual,
            subStorageBitmapUnassigned,
            sizeArea,
            sizeReserved,
            sizeBlock
        )
    );

    for( int i = 0; i < 10; ++i )
    {
        nn::fssystem::save::JournalStorage storage;
        NNT_ASSERT_RESULT_SUCCESS(
            storage.Initialize(
                subStorageControlArea,
                subStorageTable,
                subStorageBitmapUpdatedPhysical,
                subStorageBitmapUpdatedVirtual,
                subStorageBitmapUnassigned,
                subStorageData
            )
        );
    }

    // フォーマットが関係ない範囲に書き込みを行なっていないか確認します。
    if( offsetMeta > 0 )
    {
        ASSERT_EQ(true, storageControlArea.VerifyBuffer(0, offsetMeta, 0x12));
        ASSERT_EQ(true, storageTable.VerifyBuffer(0, offsetMeta, 0x12));
        ASSERT_EQ(true, storageBitmapUpdatedPhysical.VerifyBuffer(0, offsetMeta, 0x12));
        ASSERT_EQ(true, storageBitmapUpdatedVirtual.VerifyBuffer(0, offsetMeta, 0x12));
        ASSERT_EQ(true, storageBitmapUnassigned.VerifyBuffer(0, offsetMeta, 0x12));
    }
    if( offsetData > 0 )
    {
        ASSERT_EQ(true, storageData.VerifyBuffer(0, offsetData, 0x12));
    }

    EXPECT_EQ(true, storageControlArea.CheckValid());
    EXPECT_EQ(true, storageTable.CheckValid());
    EXPECT_EQ(true, storageBitmapUpdatedPhysical.CheckValid());
    EXPECT_EQ(true, storageBitmapUpdatedVirtual.CheckValid());
    EXPECT_EQ(true, storageBitmapUnassigned.CheckValid());
    EXPECT_EQ(true, storageData.CheckValid());
} // NOLINT(impl/function_size)

// フォーマット/マウント/拡張テスト
TEST_P(JournalStorageFormatTest, Expand)
{
    static const int64_t SizeCacheBlock = sizeof(nn::fssystem::save::JournalStorage::ControlArea);

    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    const FormatTestParam& param = GetParam();

    uint32_t countMapEntry = param.countMapEntry;
    uint32_t countReserved = param.countReserved;

    const uint32_t countMapEntryExpand = param.countMapEntryExpand;
    const uint32_t countReservedExpand = param.countReservedExpand;
    const uint32_t countMapEntryMax = countMapEntry + countMapEntryExpand;
    const uint32_t countReservedMax = countReserved + countReservedExpand;

    const int64_t sizeBlock = param.sizeBlock;
    const size_t offsetMeta = static_cast<size_t>(param.offsetMeta);
    const int64_t sizeAreaMax = (countMapEntry +
                                 countReserved +
                                 countMapEntryExpand +
                                 countReservedExpand) * sizeBlock;
    const int64_t sizeReservedMax = (countReserved +
                                     countReservedExpand) * sizeBlock;

    int64_t sizeTableMax;
    int64_t sizeBitmapMaxUpdatedPhysical;
    int64_t sizeBitmapMaxUpdatedVirtual;
    int64_t sizeBitmapMaxUnassigned;
    nn::fssystem::save::JournalStorage::QueryMappingMetaSize(
        &sizeTableMax,
        &sizeBitmapMaxUpdatedPhysical,
        &sizeBitmapMaxUpdatedVirtual,
        &sizeBitmapMaxUnassigned,
        sizeAreaMax,
        sizeReservedMax,
        sizeBlock
    );

    nnt::fs::util::SafeMemoryStorage
        storageControlArea(SizeCacheBlock + offsetMeta);
    nnt::fs::util::SafeMemoryStorage
        storageTable(sizeTableMax + offsetMeta);
    nnt::fs::util::SafeMemoryStorage
        storageBitmapUpdatedPhysical(sizeBitmapMaxUpdatedPhysical + offsetMeta);
    nnt::fs::util::SafeMemoryStorage
        storageBitmapUpdatedVirtual(sizeBitmapMaxUpdatedVirtual + offsetMeta);
    nnt::fs::util::SafeMemoryStorage
        storageBitmapUnassigned(sizeBitmapMaxUnassigned + offsetMeta);

    const size_t offsetData = static_cast<size_t>(param.offsetData);
    const int64_t sizeDataMax = sizeAreaMax + offsetData;
    nnt::fs::util::SafeMemoryStorage storageData(sizeDataMax);

    nn::fs::SubStorage subStorageControlArea(
        &storageControlArea,
        offsetMeta,
        SizeCacheBlock
    );

    // フォーマットが関係ない範囲に書き込まないかを確認するためのデータを書き込みます。
    if( offsetMeta > 0 )
    {
        storageControlArea.FillBuffer(0, offsetMeta, 0x12);
        storageTable.FillBuffer(0, offsetMeta, 0x12);
        storageBitmapUpdatedPhysical.FillBuffer(0, offsetMeta, 0x12);
        storageBitmapUpdatedVirtual.FillBuffer(0, offsetMeta, 0x12);
        storageBitmapUnassigned.FillBuffer(0, offsetMeta, 0x12);
    }
    if( offsetData > 0 )
    {
        storageData.FillBuffer(0, offsetData, 0x12);
    }

    // フォーマット
    {
        const int64_t sizeArea = (countMapEntry + countReserved) * sizeBlock;
        const int64_t sizeReserved = countReserved * sizeBlock;
        const size_t sizeData = static_cast<size_t>(sizeArea) + offsetData;

        int64_t sizeTable;
        int64_t sizeBitmapUpdatedPhysical;
        int64_t sizeBitmapUpdatedVirtual;
        int64_t sizeBitmapUnassigned;
        nn::fssystem::save::JournalStorage::QueryMappingMetaSize(
            &sizeTable,
            &sizeBitmapUpdatedPhysical,
            &sizeBitmapUpdatedVirtual,
            &sizeBitmapUnassigned,
            sizeArea,
            sizeReserved,
            sizeBlock
        );

        nn::fs::SubStorage subStorageTable(
            &storageTable,
            offsetMeta,
            sizeTable
        );
        nn::fs::SubStorage subStorageBitmapUpdatedPhysical(
            &storageBitmapUpdatedPhysical,
            offsetMeta,
            sizeBitmapUpdatedPhysical
        );
        nn::fs::SubStorage subStorageBitmapUpdatedVirtual(
            &storageBitmapUpdatedVirtual,
            offsetMeta,
            sizeBitmapUpdatedVirtual
        );
        nn::fs::SubStorage subStorageBitmapUnassigned(
            &storageBitmapUnassigned,
            offsetMeta,
            sizeBitmapUnassigned
        );
        nn::fs::SubStorage subStorageData(
            &storageData,
            offsetData,
            sizeData
        );

        NNT_ASSERT_RESULT_SUCCESS(
            nn::fssystem::save::JournalStorage::Format(
                subStorageControlArea,
                subStorageTable,
                subStorageBitmapUpdatedPhysical,
                subStorageBitmapUpdatedVirtual,
                subStorageBitmapUnassigned,
                sizeArea,
                sizeReserved,
                sizeBlock
            )
        );
    }

    for( int i = 0; i < 10; ++i )
    {
        // マウント
        {
            const int64_t sizeArea = (countMapEntry + countReserved) * sizeBlock;
            const int64_t sizeReserved = countReserved * sizeBlock;
            const int64_t sizeData = sizeArea + offsetData;

            int64_t sizeTable;
            int64_t sizeBitmapUpdatedPhysical;
            int64_t sizeBitmapUpdatedVirtual;
            int64_t sizeBitmapUnassigned;
            nn::fssystem::save::JournalStorage::QueryMappingMetaSize(
                &sizeTable,
                &sizeBitmapUpdatedPhysical,
                &sizeBitmapUpdatedVirtual,
                &sizeBitmapUnassigned,
                sizeArea,
                sizeReserved,
                sizeBlock
            );

            nn::fs::SubStorage subStorageTable(
                &storageTable,
                offsetMeta,
                sizeTable
            );
            nn::fs::SubStorage subStorageBitmapUpdatedPhysical(
                &storageBitmapUpdatedPhysical,
                offsetMeta,
                sizeBitmapUpdatedPhysical
            );
            nn::fs::SubStorage subStorageBitmapUpdatedVirtual(
                &storageBitmapUpdatedVirtual,
                offsetMeta,
                sizeBitmapUpdatedVirtual
            );
            nn::fs::SubStorage subStorageBitmapUnassigned(
                &storageBitmapUnassigned,
                offsetMeta,
                sizeBitmapUnassigned
            );
            nn::fs::SubStorage subStorageData(
                &storageData,
                offsetData,
                sizeData
            );

            nn::fssystem::save::JournalStorage storage;
            NNT_ASSERT_RESULT_SUCCESS(
                storage.Initialize(
                    subStorageControlArea,
                    subStorageTable,
                    subStorageBitmapUpdatedPhysical,
                    subStorageBitmapUpdatedVirtual,
                    subStorageBitmapUnassigned,
                    subStorageData
                )
            );
        }

        // 拡張
        {
            const uint32_t countMapEntryNew = std::min(
                countMapEntryMax,
                countMapEntry + std::uniform_int_distribution<uint32_t>(1, 4)(mt)
            );
            const uint32_t countReservedNew = std::min(
                countReservedMax,
                countReserved + std::uniform_int_distribution<uint32_t>(1, 4)(mt)
            );

            if( (countMapEntry != countMapEntryNew) || (countReserved != countReservedNew) )
            {
                const int64_t sizeAreaNew = (countMapEntryNew + countReservedNew) * sizeBlock;
                const int64_t sizeReservedNew = countReservedNew * sizeBlock;
                const int64_t sizeDataNew = sizeAreaNew + offsetData;

                int64_t sizeTableNew;
                int64_t sizeBitmapNewUpdatedPhysical;
                int64_t sizeBitmapNewUpdatedVirtual;
                int64_t sizeBitmapNewUnassigned;
                nn::fssystem::save::JournalStorage::QueryMappingMetaSize(
                    &sizeTableNew,
                    &sizeBitmapNewUpdatedPhysical,
                    &sizeBitmapNewUpdatedVirtual,
                    &sizeBitmapNewUnassigned,
                    sizeAreaNew,
                    sizeReservedNew,
                    sizeBlock
                );

                nn::fs::SubStorage subStorageTable(
                    &storageTable,
                    offsetMeta,
                    sizeTableNew
                );
                nn::fs::SubStorage subStorageBitmapUpdatedPhysical(
                    &storageBitmapUpdatedPhysical,
                    offsetMeta,
                    sizeBitmapNewUpdatedPhysical
                );
                nn::fs::SubStorage subStorageBitmapUpdatedVirtual(
                    &storageBitmapUpdatedVirtual,
                    offsetMeta,
                    sizeBitmapNewUpdatedVirtual
                );
                nn::fs::SubStorage subStorageBitmapUnassigned(
                    &storageBitmapUnassigned,
                    offsetMeta,
                    sizeBitmapNewUnassigned
                );
                nn::fs::SubStorage subStorageData(
                    &storageData,
                    offsetData,
                    sizeDataNew
                );

                NNT_ASSERT_RESULT_SUCCESS(
                    nn::fssystem::save::JournalStorage::Expand(
                        subStorageControlArea,
                        subStorageTable,
                        subStorageBitmapUpdatedPhysical,
                        subStorageBitmapUpdatedVirtual,
                        subStorageBitmapUnassigned,
                        sizeAreaNew,
                        sizeReservedNew
                    )
                );

                countMapEntry = countMapEntryNew;
                countReserved = countReservedNew;
            }
        }
    }


    // フォーマットが関係ない範囲に書き込みを行なっていないか確認します。
    if( offsetMeta > 0 )
    {
        ASSERT_EQ(true, storageControlArea.VerifyBuffer(0, offsetMeta, 0x12));
        ASSERT_EQ(true, storageTable.VerifyBuffer(0, offsetMeta, 0x12));
        ASSERT_EQ(true, storageBitmapUpdatedPhysical.VerifyBuffer(0, offsetMeta, 0x12));
        ASSERT_EQ(true, storageBitmapUpdatedVirtual.VerifyBuffer(0, offsetMeta, 0x12));
        ASSERT_EQ(true, storageBitmapUnassigned.VerifyBuffer(0, offsetMeta, 0x12));
    }
    if( offsetData > 0 )
    {
        ASSERT_EQ(true, storageData.VerifyBuffer(0, offsetData, 0x12));
    }

    EXPECT_EQ(true, storageControlArea.CheckValid());
    EXPECT_EQ(true, storageTable.CheckValid());
    EXPECT_EQ(true, storageBitmapUpdatedPhysical.CheckValid());
    EXPECT_EQ(true, storageBitmapUpdatedVirtual.CheckValid());
    EXPECT_EQ(true, storageBitmapUnassigned.CheckValid());
    EXPECT_EQ(true, storageData.CheckValid());
} // NOLINT(impl/function_size)

INSTANTIATE_TEST_CASE_P(
    JournalStorageTest,
    JournalStorageFormatTest,
    ::testing::ValuesIn(MakeFormatTestParam())
);

namespace {

// アクセスパラメータ
struct AccessTestParam
{
    int64_t offsetMeta;
    int64_t offsetData;
    uint32_t countMapEntry;
    uint32_t countReserved;
    uint32_t countMapEntryExpand;
    uint32_t countReservedExpand;
    int64_t sizeBlock;
};

// アクセスパラメータの作成
nnt::fs::util::Vector<AccessTestParam> MakeAccessTestParam() NN_NOEXCEPT
{
    nnt::fs::util::Vector<AccessTestParam> result;

    {
        AccessTestParam data;
        data.offsetMeta = 0;
        data.offsetData = 0;
        data.countMapEntry = 128;
        data.countReserved = 32;
        data.countMapEntryExpand = 0;
        data.countReservedExpand = 0;
        data.sizeBlock = 512;
        result.push_back(data);
    }

    {
        AccessTestParam data;
        data.offsetMeta = 32;
        data.offsetData = 32;
        data.countMapEntry = 160;
        data.countReserved = 42;
        data.countMapEntryExpand = 16;
        data.countReservedExpand = 4;
        data.sizeBlock = 512;
        result.push_back(data);
    }

    return result;
}

nnt::fs::util::Vector<AccessTestParam> MakeAccessTestLargeParam() NN_NOEXCEPT
{
    nnt::fs::util::Vector<AccessTestParam> result;

    {
        AccessTestParam data;
        data.offsetMeta = 0;
        data.offsetData = 0;
        data.countMapEntry = 8 * 1024 * 1024;
        data.countReserved = 42;
        data.countMapEntryExpand = 16;
        data.countReservedExpand = 4;
        data.sizeBlock = 16 * 1024;
        result.push_back(data);
    }

    return result;
}

}

// アクセステストクラス
template<typename TStorage>
class JournalStorageAccessTestTemplate : public ::testing::TestWithParam< AccessTestParam >
{
protected:
    typedef TStorage BaseStorage;

protected:
    // ストレージ領域のサイズを取得する
    int64_t GetTotalAreaSize() const NN_NOEXCEPT
    {
        return ((m_CountMapEntry +
                 m_CountMapEntryExpanded +
                 m_CountReserved +
                 m_CountReservedExpanded) * m_SizeBlock);
    }

    // 書き込み可能領域のサイズを取得する
    int64_t GetWritableAreaSize() const NN_NOEXCEPT
    {
        return ((m_CountMapEntry + m_CountMapEntryExpanded) * m_SizeBlock);
    }

    // 予約領域のサイズを取得する
    int64_t GetReservedAreaSize() const NN_NOEXCEPT
    {
        return ((m_CountReserved + m_CountReservedExpanded) * m_SizeBlock);
    }

    // ブロックサイズを取得する
    int64_t GetBlockSize() const NN_NOEXCEPT
    {
        return m_SizeBlock;
    }

    // セットアップ
    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
    {
        static const int64_t SizeCacheBlock =
            sizeof(nn::fssystem::save::JournalStorage::ControlArea);

        const AccessTestParam& param = GetParam();
        m_OffsetMeta = param.offsetMeta;
        m_OffsetData = param.offsetData;
        m_CountMapEntry = param.countMapEntry;
        m_CountReserved = param.countReserved;
        m_CountMapEntryExpandable = param.countMapEntryExpand;
        m_CountReservedExpandable = param.countReservedExpand;
        m_CountMapEntryExpanded = 0;
        m_CountReservedExpanded = 0;
        m_SizeBlock = param.sizeBlock;

        const int64_t sizeAreaMax = (m_CountMapEntry +
                                     m_CountReserved +
                                     m_CountMapEntryExpandable +
                                     m_CountReservedExpandable) * m_SizeBlock;
        const int64_t sizeReservedMax = (m_CountReserved +
                                         m_CountReservedExpandable) * m_SizeBlock;
        const int64_t sizeMaxData = sizeAreaMax;

        int64_t sizeTableMax;
        int64_t sizeBitmapMaxUpdatedPhysical;
        int64_t sizeBitmapMaxUpdatedVirtual;
        int64_t sizeBitmapMaxUnassigned;
        nn::fssystem::save::JournalStorage::QueryMappingMetaSize(
            &sizeTableMax,
            &sizeBitmapMaxUpdatedPhysical,
            &sizeBitmapMaxUpdatedVirtual,
            &sizeBitmapMaxUnassigned,
            sizeAreaMax,
            sizeReservedMax,
            m_SizeBlock
        );

        InitializeStorage(&m_StorageControlArea, m_OffsetMeta, SizeCacheBlock);
        InitializeStorage(&m_StorageTable, m_OffsetMeta, sizeTableMax);
        InitializeStorage(&m_StorageBitmapUpdatedPhysical, m_OffsetMeta, sizeBitmapMaxUpdatedPhysical);
        InitializeStorage(&m_StorageBitmapUpdatedVirtual, m_OffsetMeta, sizeBitmapMaxUpdatedVirtual);
        InitializeStorage(&m_StorageBitmapUnassigned, m_OffsetMeta, sizeBitmapMaxUnassigned);
        InitializeStorage(&m_StorageData, m_OffsetData, sizeMaxData);

        nn::fs::SubStorage subStorageControlArea(
            &m_StorageControlArea,
            m_OffsetMeta,
            SizeCacheBlock
        );

        // ストレージの初期化
        {
            int64_t sizeTable;
            int64_t sizeBitmapUpdatedPhysical;
            int64_t sizeBitmapUpdatedVirtual;
            int64_t sizeBitmapUnassigned;
            nn::fssystem::save::JournalStorage::QueryMappingMetaSize(
                &sizeTable,
                &sizeBitmapUpdatedPhysical,
                &sizeBitmapUpdatedVirtual,
                &sizeBitmapUnassigned,
                GetTotalAreaSize(),
                GetReservedAreaSize(),
                GetBlockSize()
            );

            nn::fs::SubStorage subStorageTable(
                &m_StorageTable,
                m_OffsetMeta,
                sizeTable
            );
            nn::fs::SubStorage subStorageBitmapUpdatedPhysical(
                &m_StorageBitmapUpdatedPhysical,
                m_OffsetMeta,
                sizeBitmapUpdatedPhysical
            );
            nn::fs::SubStorage subStorageBitmapUpdatedVirtual(
                &m_StorageBitmapUpdatedVirtual,
                m_OffsetMeta,
                sizeBitmapUpdatedVirtual
            );
            nn::fs::SubStorage subStorageBitmapUnassigned(
                &m_StorageBitmapUnassigned,
                m_OffsetMeta,
                sizeBitmapUnassigned
            );
            nn::fs::SubStorage subStorageData(
                &m_StorageData,
                m_OffsetData,
                GetTotalAreaSize()
            );

            NNT_ASSERT_RESULT_SUCCESS(
                nn::fssystem::save::JournalStorage::Format(
                    subStorageControlArea,
                    subStorageTable,
                    subStorageBitmapUpdatedPhysical,
                    subStorageBitmapUpdatedVirtual,
                    subStorageBitmapUnassigned,
                    GetTotalAreaSize(),
                    GetReservedAreaSize(),
                    GetBlockSize()
                )
            );

            m_Storage.reset(new nn::fssystem::save::JournalStorage());

            NNT_ASSERT_RESULT_SUCCESS(
                m_Storage->Initialize(
                    subStorageControlArea,
                    subStorageTable,
                    subStorageBitmapUpdatedPhysical,
                    subStorageBitmapUpdatedVirtual,
                    subStorageBitmapUnassigned,
                    subStorageData
                )
            );
        }
    } // NOLINT(impl/function_size)

    // 再マウントを行ないます。
    nn::Result Remount(bool isFormat) NN_NOEXCEPT
    {
        static const int64_t SizeCacheBlock =
            sizeof(nn::fssystem::save::JournalStorage::ControlArea);

        m_Storage.reset(new nn::fssystem::save::JournalStorage());

        int64_t sizeTable;
        int64_t sizeBitmapUpdatedPhysical;
        int64_t sizeBitmapUpdatedVirtual;
        int64_t sizeBitmapUnassigned;
        nn::fssystem::save::JournalStorage::QueryMappingMetaSize(
            &sizeTable,
            &sizeBitmapUpdatedPhysical,
            &sizeBitmapUpdatedVirtual,
            &sizeBitmapUnassigned,
            GetTotalAreaSize(),
            GetReservedAreaSize(),
            GetBlockSize()
        );

        nn::fs::SubStorage subStorageControlArea(
            &m_StorageControlArea,
            m_OffsetMeta,
            SizeCacheBlock
        );
        nn::fs::SubStorage subStorageTable(
            &m_StorageTable,
            m_OffsetMeta,
            sizeTable
        );
        nn::fs::SubStorage subStorageBitmapUpdatedPhysical(
            &m_StorageBitmapUpdatedPhysical,
            m_OffsetMeta,
            sizeBitmapUpdatedPhysical
        );
        nn::fs::SubStorage subStorageBitmapUpdatedVirtual(
            &m_StorageBitmapUpdatedVirtual,
            m_OffsetMeta,
            sizeBitmapUpdatedVirtual
        );
        nn::fs::SubStorage subStorageBitmapUnassigned(
            &m_StorageBitmapUnassigned,
            m_OffsetMeta,
            sizeBitmapUnassigned
        );
        nn::fs::SubStorage subStorageData(
            &m_StorageData,
            m_OffsetData,
            GetTotalAreaSize()
        );

        if( isFormat )
        {
            NN_RESULT_DO(
                nn::fssystem::save::JournalStorage::Format(
                    subStorageControlArea,
                    subStorageTable,
                    subStorageBitmapUpdatedPhysical,
                    subStorageBitmapUpdatedVirtual,
                    subStorageBitmapUnassigned,
                    GetTotalAreaSize(),
                    GetReservedAreaSize(),
                    GetBlockSize()
                )
            );
        }

        NN_RESULT_DO(
            m_Storage->Initialize(
                subStorageControlArea,
                subStorageTable,
                subStorageBitmapUpdatedPhysical,
                subStorageBitmapUpdatedVirtual,
                subStorageBitmapUnassigned,
                subStorageData
            )
        );

        NN_RESULT_SUCCESS;
    }

    // 拡張、再マウントを行ないます。
    // m_CountMapEntryExpanded、m_CountReservedExpanded を呼び出し側で拡張してください。
    nn::Result Expand() NN_NOEXCEPT
    {
        static const int64_t SizeCacheBlock =
            sizeof(nn::fssystem::save::JournalStorage::ControlArea);

        if( (m_CountMapEntryExpanded == 0) && (m_CountReservedExpanded == 0) )
        {
            NN_RESULT_SUCCESS;
        }

        int64_t sizeTable;
        int64_t sizeBitmapUpdatedPhysical;
        int64_t sizeBitmapUpdatedVirtual;
        int64_t sizeBitmapUnassigned;
        nn::fssystem::save::JournalStorage::QueryMappingMetaSize(
            &sizeTable,
            &sizeBitmapUpdatedPhysical,
            &sizeBitmapUpdatedVirtual,
            &sizeBitmapUnassigned,
            (m_CountMapEntry +
             m_CountMapEntryExpanded +
             m_CountReserved +
             m_CountReservedExpanded) * m_SizeBlock,
            (m_CountReserved + m_CountReservedExpanded) * m_SizeBlock,
            GetBlockSize()
        );

        nn::fs::SubStorage subStorageControlArea(
            &m_StorageControlArea,
            m_OffsetMeta,
            SizeCacheBlock
        );
        nn::fs::SubStorage subStorageTable(
            &m_StorageTable,
            m_OffsetMeta,
            sizeTable
        );
        nn::fs::SubStorage subStorageBitmapUpdatedPhysical(
            &m_StorageBitmapUpdatedPhysical,
            m_OffsetMeta,
            sizeBitmapUpdatedPhysical
        );
        nn::fs::SubStorage subStorageBitmapUpdatedVirtual(
            &m_StorageBitmapUpdatedVirtual,
            m_OffsetMeta,
            sizeBitmapUpdatedVirtual
        );
        nn::fs::SubStorage subStorageBitmapUnassigned(
            &m_StorageBitmapUnassigned,
            m_OffsetMeta,
            sizeBitmapUnassigned
        );
        nn::fs::SubStorage subStorageData(
            &m_StorageData,
            m_OffsetData,
            GetTotalAreaSize()
        );

        NN_RESULT_DO(
            nn::fssystem::save::JournalStorage::Expand(
                subStorageControlArea,
                subStorageTable,
                subStorageBitmapUpdatedPhysical,
                subStorageBitmapUpdatedVirtual,
                subStorageBitmapUnassigned,
                GetTotalAreaSize(),
                GetReservedAreaSize()
            )
        );

        m_Storage.reset(new nn::fssystem::save::JournalStorage());

        NN_RESULT_DO(
            m_Storage->Initialize(
                subStorageControlArea,
                subStorageTable,
                subStorageBitmapUpdatedPhysical,
                subStorageBitmapUpdatedVirtual,
                subStorageBitmapUnassigned,
                subStorageData
            )
        );

        NN_RESULT_SUCCESS;
    }

    BaseStorage* GetStorageControlArea() NN_NOEXCEPT
    {
        return &m_StorageControlArea;
    }

    BaseStorage* GetStorageTable() NN_NOEXCEPT
    {
        return &m_StorageTable;
    }

    BaseStorage* GetStorageBitmapUpdatedPhysical() NN_NOEXCEPT
    {
        return &m_StorageBitmapUpdatedPhysical;
    }

    BaseStorage* GetStorageBitmapUpdatedVirtual() NN_NOEXCEPT
    {
        return &m_StorageBitmapUpdatedVirtual;
    }

    BaseStorage* GetStorageBitmapUnassigned() NN_NOEXCEPT
    {
        return &m_StorageBitmapUnassigned;
    }

    BaseStorage* GetStorageData() NN_NOEXCEPT
    {
        return &m_StorageData;
    }

    nn::fs::IStorage* GetStorage() NN_NOEXCEPT
    {
        return m_Storage.get();
    }

    virtual void InitializeStorage(BaseStorage* pStorage, int64_t offset, int64_t size) NN_NOEXCEPT = 0;

protected:
    std::unique_ptr<nn::fssystem::save::JournalStorage> m_Storage;
    uint32_t m_CountMapEntryExpandable;
    uint32_t m_CountReservedExpandable;
    uint32_t m_CountMapEntryExpanded;
    uint32_t m_CountReservedExpanded;

private:
    int64_t m_OffsetMeta;
    int64_t m_OffsetData;
    uint32_t m_CountMapEntry;
    uint32_t m_CountReserved;
    int64_t m_SizeBlock;
    BaseStorage m_StorageControlArea;
    BaseStorage m_StorageTable;
    BaseStorage m_StorageBitmapUpdatedPhysical;
    BaseStorage m_StorageBitmapUpdatedVirtual;
    BaseStorage m_StorageBitmapUnassigned;
    BaseStorage m_StorageData;
};

class JournalStorageAccessTest : public JournalStorageAccessTestTemplate<nnt::fs::util::SafeMemoryStorage>
{
protected:
    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
    {
        const AccessTestParam& param = GetParam();
        const int64_t offsetMeta = param.offsetMeta;
        const int64_t offsetData = param.offsetData;

        // 関係ない範囲に書き込みを行なっていないか確認します。
        if( offsetMeta > 0 )
        {
            ASSERT_EQ(
                true,
                GetStorageControlArea()->VerifyBuffer(0, static_cast<size_t>(offsetMeta), 0x66)
            );
            ASSERT_EQ(
                true,
                GetStorageTable()->VerifyBuffer(0, static_cast<size_t>(offsetMeta), 0x66)
            );
            ASSERT_EQ(
                true,
                GetStorageBitmapUpdatedPhysical()->VerifyBuffer(
                    0, static_cast<size_t>(offsetMeta), 0x66)
            );
            ASSERT_EQ(
                true,
                GetStorageBitmapUpdatedVirtual()->VerifyBuffer(
                    0, static_cast<size_t>(offsetMeta), 0x66)
            );
            ASSERT_EQ(
                true,
                GetStorageBitmapUnassigned()->VerifyBuffer(
                    0, static_cast<size_t>(offsetMeta), 0x66)
            );
        }
        if( offsetData > 0 )
        {
            ASSERT_EQ(
                true, GetStorageData()->VerifyBuffer(0, static_cast<size_t>(offsetMeta), 0x66));
        }

        EXPECT_EQ(true, GetStorageControlArea()->CheckValid());
        EXPECT_EQ(true, GetStorageTable()->CheckValid());
        EXPECT_EQ(true, GetStorageBitmapUpdatedPhysical()->CheckValid());
        EXPECT_EQ(true, GetStorageBitmapUpdatedVirtual()->CheckValid());
        EXPECT_EQ(true, GetStorageBitmapUnassigned()->CheckValid());
        EXPECT_EQ(true, GetStorageData()->CheckValid());
    }

    virtual void InitializeStorage(BaseStorage* pStorage, int64_t offset, int64_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        pStorage->Initialize(size + offset);
        if( offset > 0 )
        {
            // テスト中に関係ない範囲に書き込まないかを確認するためのデータを書き込みます。
            pStorage->FillBuffer(0, static_cast<size_t>(offset), 0x66);
        }
    }
};

class JournalStorageAccessLargeTest : public JournalStorageAccessTestTemplate<nnt::fs::util::VirtualMemoryStorage>
{
protected:
    virtual void InitializeStorage(BaseStorage* pStorage, int64_t offset, int64_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        pStorage->Initialize(size + offset);
    }

};

// 1バイト書き込みテストを行ないます。
TEST_P(JournalStorageAccessTest, TestWrite1Byte)
{
    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    for( int i = 0; i < GetWritableAreaSize() / GetBlockSize(); ++i )
    {
        const size_t sizeAccess = 1;
        const int64_t offset = (GetBlockSize() * i) % GetWritableAreaSize();

        std::unique_ptr<char[]> bufWrite(new char[sizeAccess]);

        for( size_t indexBuf = 0; indexBuf < sizeAccess; ++indexBuf )
        {
            bufWrite[indexBuf] = indexBuf & 0xFF;
        }

        NNT_ASSERT_RESULT_SUCCESS(m_Storage->Write(offset, bufWrite.get(), sizeAccess))
            << "loop [" << i << "] offset: " << offset << ", size: " << sizeAccess;

        // 繰り返しテストするために再フォーマット、マウントを行ないます。
        NNT_ASSERT_RESULT_SUCCESS(Remount(true));
    }
}

// 1バイト読み込みテストを行ないます。
TEST_P(JournalStorageAccessTest, TestRead1Byte)
{
    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    for( int i = 0; i < GetWritableAreaSize() / GetBlockSize(); ++i )
    {
        const size_t sizeAccess = 1;
        const int64_t offset = (GetBlockSize() * i) % GetWritableAreaSize();

        std::unique_ptr<char[]> bufRead(new char[sizeAccess]);

        NNT_ASSERT_RESULT_SUCCESS(m_Storage->Read(offset, bufRead.get(), sizeAccess))
            << "loop [" << i << "] offset: " << offset << ", size: " << sizeAccess;

        // 繰り返しテストするために再フォーマット、マウントを行ないます。
        NNT_ASSERT_RESULT_SUCCESS(Remount(true));
    }
}

// 予約領域未満の書き込み、ベリファイを行ないます。
TEST_P(JournalStorageAccessTest, TestLessThanReservedAreaSize)
{
    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    for( int i = 0; i < 20; ++i )
    {
        const size_t sizeAccess = std::uniform_int_distribution<size_t>(
            1, static_cast<size_t>(GetReservedAreaSize() - GetBlockSize() - 1))(mt);
        const int64_t offset = std::uniform_int_distribution<int64_t>(
            0, GetWritableAreaSize() - sizeAccess - 1)(mt);

        std::unique_ptr<char[]> bufWrite(new char[sizeAccess]);
        std::unique_ptr<char[]> bufRead(new char[sizeAccess]);

        for( size_t indexBuf = 0; indexBuf < sizeAccess; ++indexBuf )
        {
            bufWrite[indexBuf] = indexBuf & 0xFF;
        }

        NNT_ASSERT_RESULT_SUCCESS(m_Storage->Write(offset, bufWrite.get(), sizeAccess))
            << "loop [" << i << "] offset: " << offset << ", size: " << sizeAccess;
        NNT_ASSERT_RESULT_SUCCESS(m_Storage->Read(offset, bufRead.get(), sizeAccess))
            << "loop [" << i << "] offset: " << offset << ", size: " << sizeAccess;

        for( size_t indexBuf = 0; indexBuf < sizeAccess; ++indexBuf )
        {
            ASSERT_EQ(indexBuf & 0xFF, static_cast<uint8_t>(bufRead[indexBuf]));
        }

        // 繰り返しテストするために再フォーマット、マウントを行ないます。
        NNT_ASSERT_RESULT_SUCCESS(Remount(true));
    }
}

// ストレージの終端に対する書き込み、ベリファイを行ないます。
TEST_P(JournalStorageAccessTest, TestTail)
{
    const size_t sizeAccess = 100;
    const int64_t offset = GetWritableAreaSize() - sizeAccess - 1;

    std::unique_ptr<char[]> bufWrite(new char[sizeAccess]);
    std::unique_ptr<char[]> bufRead(new char[sizeAccess]);

    for( size_t indexBuf = 0; indexBuf < sizeAccess; ++indexBuf )
    {
        bufWrite[indexBuf] = indexBuf & 0xFF;
    }

    NNT_ASSERT_RESULT_SUCCESS(m_Storage->Write(offset, bufWrite.get(), sizeAccess))
        << "offset: " << offset << ", size: " << sizeAccess;
    NNT_ASSERT_RESULT_SUCCESS(m_Storage->Read(offset, bufRead.get(), sizeAccess))
        << "offset: " << offset << ", size: " << sizeAccess;

    for( size_t indexBuf = 0; indexBuf < sizeAccess; ++indexBuf )
    {
        ASSERT_EQ(indexBuf & 0xFF, static_cast<uint8_t>(bufRead[indexBuf]));
    }
}

// 1回で予約領域を超える書き込みを行なったときのエラー処理を確認します
TEST_P(JournalStorageAccessTest, TestLargeThanReservedAreaSize1)
{
    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    for( int i = 0; i < 20; ++i )
    {
        const size_t sizeAccess = std::uniform_int_distribution<size_t>(
            static_cast<size_t>(GetReservedAreaSize() + 1),
            static_cast<size_t>(GetReservedAreaSize() + 64))(mt);

        const int64_t offset = std::uniform_int_distribution<int64_t>(
            0, GetWritableAreaSize() - sizeAccess - 1)(mt);

        std::unique_ptr<char[]> bufWrite(new char[sizeAccess]);

        for( size_t indexBuf = 0; indexBuf < sizeAccess; ++indexBuf )
        {
            bufWrite[indexBuf] = static_cast<char>(std::uniform_int_distribution<>(0, 255)(mt));
        }
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultMappingTableFull,
            m_Storage->Write(offset, bufWrite.get(), sizeAccess)
        );

        // 繰り返しテストするために再フォーマット、マウントを行ないます。
        NNT_ASSERT_RESULT_SUCCESS(Remount(true));
    }
}

// 先頭ブロックに対して書き込み、ベリファイ -> Commit の繰り返しエラーが発生しないことを確認します。
TEST_P(JournalStorageAccessTest, TestCommitFirst)
{
    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    for( int i = 0; i < 20; ++i )
    {
        const size_t sizeAccess = (i + 1) * 4;
        NN_ASSERT(sizeAccess < static_cast<size_t>(GetBlockSize()));

        const int64_t offset = 0;

        std::unique_ptr<char[]> bufWrite(new char[sizeAccess]);
        std::unique_ptr<char[]> bufRead(new char[sizeAccess]);

        for( size_t indexBuf = 0; indexBuf < sizeAccess; ++indexBuf )
        {
            bufWrite[indexBuf] = indexBuf & 0xFF;
        }

        NNT_ASSERT_RESULT_SUCCESS(m_Storage->Write(offset, bufWrite.get(), sizeAccess));
        NNT_ASSERT_RESULT_SUCCESS(m_Storage->Read(offset, bufRead.get(), sizeAccess));

        for( size_t indexBuf = 0; indexBuf < sizeAccess; ++indexBuf )
        {
            ASSERT_EQ(indexBuf & 0xFF, static_cast<uint8_t>(bufRead[indexBuf]));
        }
        NNT_EXPECT_RESULT_SUCCESS(m_Storage->Commit());
    }
}

// 1ブロックごとに対して書き込み、ベリファイ -> Commit の繰り返しエラーが発生しないことを確認します。
TEST_P(JournalStorageAccessTest, TestCommitSequence)
{
    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    for( int i = 0; i < 20; ++i )
    {
        const size_t sizeAccess = std::uniform_int_distribution<size_t>(
            1, static_cast<size_t>(GetBlockSize() - 1))(mt);
        NN_ASSERT(sizeAccess < static_cast<size_t>(GetBlockSize()));

        const int64_t offset = GetBlockSize() * i;

        std::unique_ptr<char[]> bufWrite(new char[sizeAccess]);
        std::unique_ptr<char[]> bufRead(new char[sizeAccess]);

        for( size_t indexBuf = 0; indexBuf < sizeAccess; ++indexBuf )
        {
            bufWrite[indexBuf] = indexBuf & 0xFF;
        }

        NNT_ASSERT_RESULT_SUCCESS(m_Storage->Write(offset, bufWrite.get(), sizeAccess));
        NNT_ASSERT_RESULT_SUCCESS(m_Storage->Read(offset, bufRead.get(), sizeAccess));

        for( size_t indexBuf = 0; indexBuf < sizeAccess; ++indexBuf )
        {
            ASSERT_EQ(indexBuf & 0xFF, static_cast<uint8_t>(bufRead[indexBuf]));
        }

        NNT_EXPECT_RESULT_SUCCESS(m_Storage->Commit());
    }
}

// 1ブロックごとに対して書き込み、ベリファイ -> Commit の繰り返し、
// その後領域拡張をして同じ処理を繰り返してもエラーが発生しないことを確認します。
TEST_P(JournalStorageAccessTest, TestCommitSequenceWithExpand)
{
    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    m_CountMapEntryExpanded = 0;

    for( int expand = 0; expand < 4; ++expand )
    {
        // 書き込みテスト
        for( int block = 0; block < GetWritableAreaSize() / GetBlockSize(); ++block )
        {
            const size_t sizeAccess = std::uniform_int_distribution<size_t>(
                1, static_cast<size_t>(GetBlockSize() - 1))(mt);
            NN_ASSERT(sizeAccess < static_cast<size_t>(GetBlockSize()));

            const int64_t offset = GetBlockSize() * block;

            std::unique_ptr<char[]> bufWrite(new char[sizeAccess]);
            std::unique_ptr<char[]> bufRead(new char[sizeAccess]);

            for( size_t indexBuf = 0; indexBuf < sizeAccess; ++indexBuf )
            {
                bufWrite[indexBuf] = indexBuf & 0xFF;
            }

            NNT_ASSERT_RESULT_SUCCESS(m_Storage->Write(offset, bufWrite.get(), sizeAccess))
                << "Failed @ block: " << block;
            NNT_ASSERT_RESULT_SUCCESS(m_Storage->Read(offset, bufRead.get(), sizeAccess));

            for( size_t indexBuf = 0; indexBuf < sizeAccess; ++indexBuf )
            {
                ASSERT_EQ(indexBuf & 0xFF, static_cast<uint8_t>(bufRead[indexBuf]));
            }

            NNT_EXPECT_RESULT_SUCCESS(m_Storage->Commit());
        }

        // 領域を拡張します。
        const uint32_t countMapEntryOld = m_CountMapEntryExpanded;
        if( m_CountMapEntryExpandable > 0 )
        {
            if( m_CountMapEntryExpanded == 0 )
            {
                m_CountMapEntryExpanded = 4;
            }
            else
            {
                m_CountMapEntryExpanded <<= 1;
            }
        }
        m_CountMapEntryExpanded = std::min(m_CountMapEntryExpanded, m_CountMapEntryExpandable);

        if( countMapEntryOld != m_CountMapEntryExpanded )
        {
            NNT_EXPECT_RESULT_SUCCESS(Expand());
        }
    }
}

// 1ブロックごとに対して書き込み、ベリファイ -> Commit の繰り返し、
// その後領域拡張をして同じ処理を繰り返してもエラーが発生しないことを確認します。
TEST_P(JournalStorageAccessTest, TestCommitSequenceWithExpand2)
{
    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    m_CountMapEntryExpanded = 0;

    for( int expand = 0; expand < 10; ++expand )
    {
        // 書き込みテスト
        for( int block = 0; block < GetWritableAreaSize() / GetBlockSize(); ++block )
        {
            const size_t sizeAccess = std::uniform_int_distribution<size_t>(
                1, static_cast<size_t>(GetBlockSize() - 1))(mt);
            NN_ASSERT(sizeAccess < static_cast<size_t>(GetBlockSize()));

            const int64_t offset = GetBlockSize() * block;

            std::unique_ptr<char[]> bufWrite(new char[sizeAccess]);
            std::unique_ptr<char[]> bufRead(new char[sizeAccess]);

            for( size_t indexBuf = 0; indexBuf < sizeAccess; ++indexBuf )
            {
                bufWrite[indexBuf] = indexBuf & 0xFF;
            }

            NNT_ASSERT_RESULT_SUCCESS(m_Storage->Write(offset, bufWrite.get(), sizeAccess))
                << "Failed @ block: " << block;
            NNT_ASSERT_RESULT_SUCCESS(m_Storage->Read(offset, bufRead.get(), sizeAccess));

            for( size_t indexBuf = 0; indexBuf < sizeAccess; ++indexBuf )
            {
                ASSERT_EQ(indexBuf & 0xFF, static_cast<uint8_t>(bufRead[indexBuf]));
            }

            NNT_EXPECT_RESULT_SUCCESS(m_Storage->Commit());
        }

        // 領域を拡張します。
        const uint32_t countMapEntryOld = m_CountMapEntryExpanded;
        const uint32_t countReservedOld = m_CountReservedExpanded;
        if( m_CountMapEntryExpandable > 0 )
        {
            m_CountMapEntryExpanded += std::uniform_int_distribution<uint32_t>(1, 4)(mt);
            m_CountMapEntryExpanded = std::min(m_CountMapEntryExpanded,
                                               m_CountMapEntryExpandable);
        }
        if( m_CountReservedExpandable > 0 )
        {
            m_CountReservedExpanded += std::uniform_int_distribution<uint32_t>(1, 4)(mt);
            m_CountReservedExpanded = std::min(m_CountReservedExpanded,
                                               m_CountReservedExpandable);
        }
        if( (countMapEntryOld != m_CountMapEntryExpanded) ||
            (countReservedOld != m_CountReservedExpanded) )
        {
            NNT_EXPECT_RESULT_SUCCESS(Expand());
        }
    }
}

// QueryRange をテストします。
TEST_P(JournalStorageAccessTest, TestQueryRange)
{
    {
        nn::fs::QueryRangeInfo info;

        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultNullptrArgument,
            m_Storage->OperateRange(
                nullptr,
                sizeof(info),
                nn::fs::OperationId::QueryRange,
                0,
                1,
                nullptr,
                0));

        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultInvalidSize,
            m_Storage->OperateRange(
                &info,
                0,
                nn::fs::OperationId::QueryRange,
                0,
                1,
                nullptr,
                0));
    }

    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    for( int i = 0; i < 20; ++i )
    {
        const size_t sizeAccess = std::uniform_int_distribution<size_t>(
            1, static_cast<size_t>(GetReservedAreaSize() - GetBlockSize() - 1))(mt);
        const int64_t offset = std::uniform_int_distribution<int64_t>(
            0, GetWritableAreaSize() - sizeAccess - 1)(mt);

        nn::fs::QueryRangeInfo info;
        NNT_ASSERT_RESULT_SUCCESS(m_Storage->OperateRange(
            &info,
            sizeof(info),
            nn::fs::OperationId::QueryRange,
            offset,
            sizeAccess,
            nullptr,
            0
        )) << "loop [" << i << "] offset: " << offset << ", size: " << sizeAccess;

        EXPECT_EQ(0, info.aesCtrKeyTypeFlag);
        EXPECT_EQ(0, info.speedEmulationTypeFlag);

        // 繰り返しテストするために再フォーマット、マウントを行ないます。
        NNT_ASSERT_RESULT_SUCCESS(Remount(true));
    }}

TEST_P(JournalStorageAccessLargeTest, TestReadWrite)
{
    TestLargeOffsetAccess(GetStorage(), 1024);
}

INSTANTIATE_TEST_CASE_P(
    JournalStorageTest,
    JournalStorageAccessTest,
    ::testing::ValuesIn(MakeAccessTestParam())
);

INSTANTIATE_TEST_CASE_P(
    JournalStorageLargeTest,
    JournalStorageAccessLargeTest,
    ::testing::ValuesIn(MakeAccessTestLargeParam())
);

namespace {

// エージングパラメータ
struct AgingTestParam
{
    int64_t offsetMeta;
    int64_t offsetData;
    uint32_t countMapEntry;
    uint32_t countReserved;
    uint32_t countMapEntryExpand;
    uint32_t countReservedExpand;
    int64_t sizeBlock;
};

// エージングパラメータの作成
nnt::fs::util::Vector<AgingTestParam> MakeAgingTestParam() NN_NOEXCEPT
{
    nnt::fs::util::Vector<AgingTestParam> result;

    {
        AgingTestParam data;
        data.offsetMeta = 0;
        data.offsetData = 0;
        data.countMapEntry = 128;
        data.countReserved = 32;
        data.countMapEntryExpand = 16;
        data.countReservedExpand = 0;
        data.sizeBlock = 512;
        result.push_back(data);
    }

    {
        AgingTestParam data;
        data.offsetMeta = 32;
        data.offsetData = 32;
        data.countMapEntry = 160;
        data.countReserved = 42;
        data.countMapEntryExpand = 16;
        data.countReservedExpand = 4;
        data.sizeBlock = 512;
        result.push_back(data);
    }

    {
        AgingTestParam data;
        data.offsetMeta = 1442;
        data.offsetData = 21442;
        data.countMapEntry = 192;
        data.countReserved = 48;
        data.countMapEntryExpand = 400;
        data.countReservedExpand = 18;
        data.sizeBlock = 512;
        result.push_back(data);
    }

    return result;
}

}

// エージングテストクラス
class JournalStorageAgingTest : public ::testing::TestWithParam< AgingTestParam >
{
protected:
    // ストレージ領域のサイズを取得する
    int64_t GetTotalAreaSize() const NN_NOEXCEPT
    {
        return (m_CountMapEntry +
                m_CountMapEntryExpanded +
                m_CountReserved +
                m_CountReservedExpanded) * m_SizeBlock;
    }

    // 書き込み可能領域のサイズを取得する
    int64_t GetWritableAreaSize() const NN_NOEXCEPT
    {
        return (m_CountMapEntry + m_CountMapEntryExpanded) * m_SizeBlock;
    }

    // 予約領域のサイズを取得する
    int64_t GetReservedAreaSize() const NN_NOEXCEPT
    {
        return (m_CountReserved + m_CountReservedExpanded) * m_SizeBlock;
    }

    // ブロックサイズを取得する
    int64_t GetBlockSize() const NN_NOEXCEPT
    {
        return m_SizeBlock;
    }

    // セットアップ
    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
    {
        static const int64_t SizeCacheBlock =
            sizeof(nn::fssystem::save::JournalStorage::ControlArea);

        const AgingTestParam& param = GetParam();
        m_OffsetMeta = param.offsetMeta;
        m_OffsetData = param.offsetData;
        m_CountMapEntry = param.countMapEntry;
        m_CountReserved = param.countReserved;
        m_CountMapEntryExpand = param.countMapEntryExpand;
        m_CountReservedExpand = param.countReservedExpand;
        m_CountMapEntryExpanded = 0;
        m_CountReservedExpanded = 0;
        m_SizeBlock = param.sizeBlock;

        const int64_t sizeAreaMax = (m_CountMapEntry +
                                     m_CountReserved +
                                     m_CountMapEntryExpand +
                                     m_CountReservedExpand) * m_SizeBlock;
        const int64_t sizeReservedMax = (m_CountReserved +
                                         m_CountReservedExpand) * m_SizeBlock;
        const int64_t sizeMaxData = sizeAreaMax + sizeReservedMax;

        int64_t sizeTableMax;
        int64_t sizeBitmapMaxUpdatedPhysical;
        int64_t sizeBitmapMaxUpdatedVirtual;
        int64_t sizeBitmapMaxUnassigned;
        nn::fssystem::save::JournalStorage::QueryMappingMetaSize(
            &sizeTableMax,
            &sizeBitmapMaxUpdatedPhysical,
            &sizeBitmapMaxUpdatedVirtual,
            &sizeBitmapMaxUnassigned,
            sizeAreaMax,
            sizeReservedMax,
            m_SizeBlock
        );

        m_StorageControlArea.Initialize(SizeCacheBlock + m_OffsetMeta);
        m_StorageTable.Initialize(sizeTableMax + m_OffsetMeta);
        m_StorageBitmapUpdatedPhysical.Initialize(sizeBitmapMaxUpdatedPhysical + m_OffsetMeta);
        m_StorageBitmapUpdatedVirtual.Initialize(sizeBitmapMaxUpdatedVirtual + m_OffsetMeta);
        m_StorageBitmapUnassigned.Initialize(sizeBitmapMaxUnassigned + m_OffsetMeta);
        m_StorageData.Initialize(sizeMaxData + m_OffsetData);

        // テスト中に関係ない範囲に書き込まないかを確認するためのデータを書き込みます。
        if( m_OffsetMeta > 0 )
        {
            m_StorageControlArea.FillBuffer(0, static_cast<size_t>(m_OffsetMeta), 0x66);
            m_StorageTable.FillBuffer(0, static_cast<size_t>(m_OffsetMeta), 0x66);
            m_StorageBitmapUpdatedPhysical.FillBuffer(0, static_cast<size_t>(m_OffsetMeta), 0x66);
            m_StorageBitmapUpdatedVirtual.FillBuffer(0, static_cast<size_t>(m_OffsetMeta), 0x66);
            m_StorageBitmapUnassigned.FillBuffer(0, static_cast<size_t>(m_OffsetMeta), 0x66);
        }
        if( m_OffsetData > 0 )
        {
            m_StorageData.FillBuffer(0, static_cast<size_t>(m_OffsetData), 0x66);
        }

        // ストレージの初期化
        {
            int64_t sizeTable;
            int64_t sizeBitmapUpdatedPhysical;
            int64_t sizeBitmapUpdatedVirtual;
            int64_t sizeBitmapUnassigned;
            nn::fssystem::save::JournalStorage::QueryMappingMetaSize(
                &sizeTable,
                &sizeBitmapUpdatedPhysical,
                &sizeBitmapUpdatedVirtual,
                &sizeBitmapUnassigned,
                GetTotalAreaSize(),
                GetReservedAreaSize(),
                GetBlockSize()
            );

            nn::fs::SubStorage subStorageControlArea(
                &m_StorageControlArea,
                m_OffsetMeta,
                SizeCacheBlock
            );
            nn::fs::SubStorage subStorageTable(
                &m_StorageTable,
                m_OffsetMeta,
                sizeTable
            );
            nn::fs::SubStorage subStorageBitmapUpdatedPhysical(
                &m_StorageBitmapUpdatedPhysical,
                m_OffsetMeta,
                sizeBitmapUpdatedPhysical
            );
            nn::fs::SubStorage subStorageBitmapUpdatedVirtual(
                &m_StorageBitmapUpdatedVirtual,
                m_OffsetMeta,
                sizeBitmapUpdatedVirtual
            );
            nn::fs::SubStorage subStorageBitmapUnassigned(
                &m_StorageBitmapUnassigned,
                m_OffsetMeta,
                sizeBitmapUnassigned
            );
            nn::fs::SubStorage subStorageData(
                &m_StorageData,
                m_OffsetData,
                GetTotalAreaSize()
            );

            NNT_ASSERT_RESULT_SUCCESS(
                nn::fssystem::save::JournalStorage::Format(
                    subStorageControlArea,
                    subStorageTable,
                    subStorageBitmapUpdatedPhysical,
                    subStorageBitmapUpdatedVirtual,
                    subStorageBitmapUnassigned,
                    GetTotalAreaSize(),
                    GetReservedAreaSize(),
                    GetBlockSize()
                )
            );

            m_Storage.reset(new nn::fssystem::save::JournalStorage());

            NNT_ASSERT_RESULT_SUCCESS(
                m_Storage->Initialize(
                    subStorageControlArea,
                    subStorageTable,
                    subStorageBitmapUpdatedPhysical,
                    subStorageBitmapUpdatedVirtual,
                    subStorageBitmapUnassigned,
                    subStorageData
                )
            );
        }
    } // NOLINT(impl/function_size)

    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
    {
        // 関係ない範囲に書き込みを行なっていないか確認します。
        if( m_OffsetMeta > 0 )
        {
            ASSERT_EQ(
                true,
                m_StorageControlArea.VerifyBuffer(0, static_cast<size_t>(m_OffsetMeta), 0x66)
            );
            ASSERT_EQ(
                true,
                m_StorageTable.VerifyBuffer(0, static_cast<size_t>(m_OffsetMeta), 0x66)
            );
            ASSERT_EQ(
                true,
                m_StorageBitmapUpdatedPhysical.VerifyBuffer(
                    0, static_cast<size_t>(m_OffsetMeta), 0x66)
            );
            ASSERT_EQ(
                true,
                m_StorageBitmapUpdatedVirtual.VerifyBuffer(
                    0, static_cast<size_t>(m_OffsetMeta), 0x66)
            );
            ASSERT_EQ(
                true,
                m_StorageBitmapUnassigned.VerifyBuffer(
                    0, static_cast<size_t>(m_OffsetMeta), 0x66)
            );
        }
        if( m_OffsetData > 0 )
        {
            ASSERT_EQ(
                true, m_StorageData.VerifyBuffer(0, static_cast<size_t>(m_OffsetMeta), 0x66));
        }
        EXPECT_EQ(true, m_StorageControlArea.CheckValid());
        EXPECT_EQ(true, m_StorageTable.CheckValid());
        EXPECT_EQ(true, m_StorageBitmapUpdatedPhysical.CheckValid());
        EXPECT_EQ(true, m_StorageBitmapUpdatedVirtual.CheckValid());
        EXPECT_EQ(true, m_StorageBitmapUnassigned.CheckValid());
        EXPECT_EQ(true, m_StorageData.CheckValid());
    }

    // 再マウントを行ないます。
    nn::Result Remount() NN_NOEXCEPT
    {
        static const int64_t SizeCacheBlock =
            sizeof(nn::fssystem::save::JournalStorage::ControlArea);

        m_Storage.reset(new nn::fssystem::save::JournalStorage());

        int64_t sizeTable;
        int64_t sizeBitmapUpdatedPhysical;
        int64_t sizeBitmapUpdatedVirtual;
        int64_t sizeBitmapUnassigned;
        nn::fssystem::save::JournalStorage::QueryMappingMetaSize(
            &sizeTable,
            &sizeBitmapUpdatedPhysical,
            &sizeBitmapUpdatedVirtual,
            &sizeBitmapUnassigned,
            GetTotalAreaSize(),
            GetReservedAreaSize(),
            GetBlockSize()
        );

        nn::fs::SubStorage subStorageControlArea(
            &m_StorageControlArea,
            m_OffsetMeta,
            SizeCacheBlock
        );
        nn::fs::SubStorage subStorageTable(
            &m_StorageTable,
            m_OffsetMeta,
            sizeTable
        );
        nn::fs::SubStorage subStorageBitmapUpdatedPhysical(
            &m_StorageBitmapUpdatedPhysical,
            m_OffsetMeta,
            sizeBitmapUpdatedPhysical
        );
        nn::fs::SubStorage subStorageBitmapUpdatedVirtual(
            &m_StorageBitmapUpdatedVirtual,
            m_OffsetMeta,
            sizeBitmapUpdatedVirtual
        );
        nn::fs::SubStorage subStorageBitmapUnassigned(
            &m_StorageBitmapUnassigned,
            m_OffsetMeta,
            sizeBitmapUnassigned
        );
        nn::fs::SubStorage subStorageData(
            &m_StorageData,
            m_OffsetData,
            GetTotalAreaSize()
        );

        NN_RESULT_DO(
            m_Storage->Initialize(
                subStorageControlArea,
                subStorageTable,
                subStorageBitmapUpdatedPhysical,
                subStorageBitmapUpdatedVirtual,
                subStorageBitmapUnassigned,
                subStorageData
            )
        );

        NN_RESULT_SUCCESS;
    }

    // 拡張、再マウントを行ないます。
    // m_CountMapEntryExpanded、m_CountReservedExpanded を呼び出し側で拡張してください。
    nn::Result Expand() NN_NOEXCEPT
    {
        static const int64_t SizeCacheBlock =
            sizeof(nn::fssystem::save::JournalStorage::ControlArea);

        if( (m_CountMapEntryExpanded == 0) && (m_CountReservedExpanded == 0) )
        {
            NN_RESULT_SUCCESS;
        }

        int64_t sizeTable;
        int64_t sizeBitmapUpdatedPhysical;
        int64_t sizeBitmapUpdatedVirtual;
        int64_t sizeBitmapUnassigned;
        nn::fssystem::save::JournalStorage::QueryMappingMetaSize(
            &sizeTable,
            &sizeBitmapUpdatedPhysical,
            &sizeBitmapUpdatedVirtual,
            &sizeBitmapUnassigned,
            GetTotalAreaSize(),
            GetReservedAreaSize(),
            GetBlockSize()
        );

        nn::fs::SubStorage subStorageControlArea(
            &m_StorageControlArea,
            m_OffsetMeta,
            SizeCacheBlock
        );
        nn::fs::SubStorage subStorageTable(
            &m_StorageTable,
            m_OffsetMeta,
            sizeTable
        );
        nn::fs::SubStorage subStorageBitmapUpdatedPhysical(
            &m_StorageBitmapUpdatedPhysical,
            m_OffsetMeta,
            sizeBitmapUpdatedPhysical
        );
        nn::fs::SubStorage subStorageBitmapUpdatedVirtual(
            &m_StorageBitmapUpdatedVirtual,
            m_OffsetMeta,
            sizeBitmapUpdatedVirtual
        );
        nn::fs::SubStorage subStorageBitmapUnassigned(
            &m_StorageBitmapUnassigned,
            m_OffsetMeta,
            sizeBitmapUnassigned
        );
        nn::fs::SubStorage subStorageData(
            &m_StorageData,
            m_OffsetData,
            GetTotalAreaSize()
        );

        NN_RESULT_DO(
            nn::fssystem::save::JournalStorage::Expand(
                subStorageControlArea,
                subStorageTable,
                subStorageBitmapUpdatedPhysical,
                subStorageBitmapUpdatedVirtual,
                subStorageBitmapUnassigned,
                GetTotalAreaSize(),
                GetReservedAreaSize()
            )
        );

        m_Storage.reset(new nn::fssystem::save::JournalStorage());

        NN_RESULT_DO(
            m_Storage->Initialize(
                subStorageControlArea,
                subStorageTable,
                subStorageBitmapUpdatedPhysical,
                subStorageBitmapUpdatedVirtual,
                subStorageBitmapUnassigned,
                subStorageData
            )
        );

        NN_RESULT_SUCCESS;
    }

protected:
    std::unique_ptr<nn::fssystem::save::JournalStorage> m_Storage;
    uint32_t m_CountMapEntryExpand;
    uint32_t m_CountReservedExpand;
    uint32_t m_CountMapEntryExpanded;
    uint32_t m_CountReservedExpanded;

private:
    int64_t m_OffsetMeta;
    int64_t m_OffsetData;
    uint32_t m_CountMapEntry;
    uint32_t m_CountReserved;
    int64_t m_SizeBlock;
    nnt::fs::util::SafeMemoryStorage m_StorageControlArea;
    nnt::fs::util::SafeMemoryStorage m_StorageTable;
    nnt::fs::util::SafeMemoryStorage m_StorageBitmapUpdatedPhysical;
    nnt::fs::util::SafeMemoryStorage m_StorageBitmapUpdatedVirtual;
    nnt::fs::util::SafeMemoryStorage m_StorageBitmapUnassigned;
    nnt::fs::util::SafeMemoryStorage m_StorageData;
};

// 1ブロックごとに対して書き込み、ベリファイ -> Commit の繰り返しエラーが発生しないことを確認します。
TEST_P(JournalStorageAgingTest, TestCommitSequence)
{
    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    for( int i = 0; i < 200; ++i )
    {
        const size_t sizeAccess = std::uniform_int_distribution<size_t>(
            1, static_cast<size_t>(GetBlockSize() - 1))(mt);
        NN_ASSERT(sizeAccess < static_cast<size_t>(GetBlockSize()));

        const int64_t offset = (GetBlockSize() * i) % GetWritableAreaSize();

        std::unique_ptr<char[]> bufWrite(new char[sizeAccess]);
        std::unique_ptr<char[]> bufRead(new char[sizeAccess]);

        for( size_t indexBuf = 0; indexBuf < sizeAccess; ++indexBuf )
        {
            bufWrite[indexBuf] = indexBuf & 0xFF;
        }

        NNT_ASSERT_RESULT_SUCCESS(m_Storage->Write(offset, bufWrite.get(), sizeAccess))
            << "loop [" << i << "] offset: " << offset << ", size: " << sizeAccess
            << "/" << GetReservedAreaSize();
        NNT_ASSERT_RESULT_SUCCESS(m_Storage->Read(offset, bufRead.get(), sizeAccess))
            << "loop [" << i << "] offset: " << offset << ", size: " << sizeAccess
            << "/" << GetReservedAreaSize();

        for( size_t indexBuf = 0; indexBuf < sizeAccess; ++indexBuf )
        {
            ASSERT_EQ(indexBuf & 0xFF, static_cast<uint8_t>(bufRead[indexBuf]))
                << "loop [" << i << "] failed @ " << indexBuf;
        }

        NNT_EXPECT_RESULT_SUCCESS(m_Storage->Commit());
    }
}

// ランダム範囲への書き込み、ベリファイ -> Commit の繰り返しで
// エラーが発生しないことを確認します。
TEST_P(JournalStorageAgingTest, TestCommitRandom)
{
    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    for( int i = 0; i < 200; ++i )
    {
        const size_t sizeAccess = std::uniform_int_distribution<size_t>(
            1, static_cast<size_t>(GetReservedAreaSize() / 4 - 1))(mt);
        const int64_t offset = std::uniform_int_distribution<int64_t>(
            0, GetWritableAreaSize() - sizeAccess - 1)(mt);

        std::unique_ptr<char[]> bufWrite(new char[sizeAccess]);
        std::unique_ptr<char[]> bufRead(new char[sizeAccess]);

        for( size_t indexBuf = 0; indexBuf < sizeAccess; ++indexBuf )
        {
            bufWrite[indexBuf] = indexBuf & 0xFF;
        }

        NNT_ASSERT_RESULT_SUCCESS(m_Storage->Write(offset, bufWrite.get(), sizeAccess))
            << "loop [" << i << "] offset: " << offset << ", size: " << sizeAccess
            << "/" << GetReservedAreaSize();
        NNT_ASSERT_RESULT_SUCCESS(m_Storage->Read(offset, bufRead.get(), sizeAccess))
            << "loop [" << i << "] offset: " << offset << ", size: " << sizeAccess
            << "/" << GetReservedAreaSize();

        for( size_t indexBuf = 0; indexBuf < sizeAccess; ++indexBuf )
        {
            ASSERT_EQ(indexBuf & 0xFF, static_cast<uint8_t>(bufRead[indexBuf]))
                << "loop [" << i << "] failed @ " << indexBuf;
        }

        NNT_EXPECT_RESULT_SUCCESS(m_Storage->Commit());
    }
}

// ランダム範囲へのライト、リード、Commit をランダムに繰り返します。
TEST_P(JournalStorageAgingTest, TestAccessRandom)
{
    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    for( int i = 0; i < 2000; ++i )
    {
        // ランダムなタイミングでライトします。
        if( std::uniform_int_distribution<int>(0, 100)(mt) > 50 )
        {
            const size_t sizeAccess = std::uniform_int_distribution<size_t>(
                1, static_cast<size_t>(GetReservedAreaSize() / 2 - 1))(mt);
            const int64_t offset = std::uniform_int_distribution<int64_t>(
                0, GetWritableAreaSize() - sizeAccess - 1)(mt);

            std::unique_ptr<char[]> bufWrite(new char[sizeAccess]);

            for( size_t indexBuf = 0; indexBuf < sizeAccess; ++indexBuf )
            {
                bufWrite[indexBuf] = indexBuf & 0xFF;
            }

            nn::Result result = m_Storage->Write(offset, bufWrite.get(), sizeAccess);
            if( !result.IsSuccess() )
            {
                if( !nn::fs::ResultMappingTableFull::Includes(result) )
                {
                    NNT_ASSERT_RESULT_SUCCESS(result)
                        << "loop [" << i
                        << "] offset: " << offset
                        << ", size: " << sizeAccess
                        << " / writable:" << GetWritableAreaSize()
                        << ", reserved: " << GetReservedAreaSize();
                }
            }
        }

        // ランダムなタイミングでリードします。
        if( std::uniform_int_distribution<int>(0, 100)(mt) > 50 )
        {
            const size_t sizeAccess = std::uniform_int_distribution<size_t>(
                1, static_cast<size_t>(GetReservedAreaSize() / 2 - 1))(mt);
            const int64_t offset = std::uniform_int_distribution<int64_t>(
                0, GetWritableAreaSize() - sizeAccess - 1)(mt);

            std::unique_ptr<char[]> bufRead(new char[sizeAccess]);

            nn::Result result = m_Storage->Read(offset, bufRead.get(), sizeAccess);
            NNT_ASSERT_RESULT_SUCCESS(result)
                << "loop [" << i
                << "] offset: " << offset
                << ", size: " << sizeAccess
                << " / writable:" << GetWritableAreaSize()
                << ", reserved: " << GetReservedAreaSize();
        }

        // ランダムなタイミングでコミットします。
        if( std::uniform_int_distribution<int>(0, 100)(mt) < 10 )
        {
            NNT_EXPECT_RESULT_SUCCESS(m_Storage->Commit());
        }

        // ランダムなタイミングで再マウントします。
        if( std::uniform_int_distribution<int>(0, 100)(mt) < 5 )
        {
            NNT_EXPECT_RESULT_SUCCESS(Remount());
        }

        // ランダムなタイミングで領域を拡張します。
        if( std::uniform_int_distribution<int>(0, 100)(mt) < 5 )
        {
            NNT_EXPECT_RESULT_SUCCESS(m_Storage->Commit());

            uint32_t countMapEntryOld = m_CountMapEntryExpanded;
            uint32_t countReservedOld = m_CountReservedExpanded;
            if( m_CountMapEntryExpand > 0 )
            {
                m_CountMapEntryExpanded += std::uniform_int_distribution<uint32_t>(0, 16)(mt);
                m_CountMapEntryExpanded = std::min(m_CountMapEntryExpanded,
                                                   m_CountMapEntryExpand);
            }
            if( m_CountReservedExpand > 0 )
            {
                m_CountReservedExpanded += std::uniform_int_distribution<uint32_t>(0, 8)(mt);
                m_CountReservedExpanded = std::min(m_CountReservedExpanded,
                                                   m_CountReservedExpand);
            }
            if( (countMapEntryOld != m_CountMapEntryExpanded) ||
                (countReservedOld != m_CountReservedExpanded) )
            {
                NNT_EXPECT_RESULT_SUCCESS(Expand());
            }
        }
    }
}

INSTANTIATE_TEST_CASE_P(
    JournalStorageTest,
    JournalStorageAgingTest,
    ::testing::ValuesIn(MakeAgingTestParam())
);

class JournalStoragePowerInterruptionTest : public ::testing::TestWithParam< AccessTestParam >
{
protected:
    //! ストレージ領域のサイズを取得する
    int64_t GetTotalAreaSize() const NN_NOEXCEPT
    {
        return (m_CountMapEntry + m_CountMapEntryExpanded + m_CountReserved + m_CountReservedExpanded) * m_SizeBlock;
    }

    //! 書き込み可能領域のサイズを取得する
    int64_t GetWritableAreaSize() const NN_NOEXCEPT
    {
        return (m_CountMapEntry + m_CountMapEntryExpanded) * m_SizeBlock;
    }

    //! 予約領域のサイズを取得する
    int64_t GetReservedAreaSize() const NN_NOEXCEPT
    {
        return (m_CountReserved + m_CountReservedExpanded) * m_SizeBlock;
    }

    //! ブロックサイズを取得する
    int64_t GetBlockSize() const NN_NOEXCEPT
    {
        return m_SizeBlock;
    }

protected:
    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
    {
        const AccessTestParam& param = GetParam();
        m_OffsetMeta = param.offsetMeta;
        m_OffsetData = param.offsetData;
        m_CountMapEntry = param.countMapEntry;
        m_CountReserved = param.countReserved;
        m_CountMapEntryExpandable = param.countMapEntryExpand;
        m_CountReservedExpandable = param.countReservedExpand;
        m_CountMapEntryExpanded = 0;
        m_CountReservedExpanded = 0;
        m_SizeBlock = param.sizeBlock;

        const int64_t sizeAreaMax = (m_CountMapEntry + m_CountReserved + m_CountMapEntryExpandable + m_CountReservedExpandable) * m_SizeBlock;
        const int64_t sizeReservedMax = (m_CountReserved + m_CountReservedExpandable) * m_SizeBlock;
        const int64_t sizeMaxData = sizeAreaMax;

        const int64_t sizeControlArea = sizeof(nn::fssystem::save::JournalStorage::ControlArea);
        int64_t sizeTableMax;
        int64_t sizeBitmapMaxUpdatedPhysical;
        int64_t sizeBitmapMaxUpdatedVirtual;
        int64_t sizeBitmapMaxUnassigned;
        nn::fssystem::save::JournalStorage::QueryMappingMetaSize(
            &sizeTableMax,
            &sizeBitmapMaxUpdatedPhysical,
            &sizeBitmapMaxUpdatedVirtual,
            &sizeBitmapMaxUnassigned,
            sizeAreaMax,
            sizeReservedMax,
            m_SizeBlock
        );

        m_StorageControlArea.Initialize(sizeControlArea + m_OffsetMeta);
        m_StorageTable.Initialize(sizeTableMax + m_OffsetMeta);
        m_StorageBitmapUpdatedPhysical.Initialize(sizeBitmapMaxUpdatedPhysical + m_OffsetMeta);
        m_StorageBitmapUpdatedVirtual.Initialize(sizeBitmapMaxUpdatedVirtual + m_OffsetMeta);
        m_StorageBitmapUnassigned.Initialize(sizeBitmapMaxUnassigned + m_OffsetMeta);
        m_StorageBaseData.Initialize(sizeMaxData + m_OffsetData);
        m_StorageData.Initialize(
            nn::fs::SubStorage(&m_StorageBaseData, 0, sizeMaxData + m_OffsetData)
        );

        nn::fs::SubStorage subStorageControlArea(
            &m_StorageControlArea,
            m_OffsetMeta,
            sizeControlArea
        );

        // テスト中に関係ない範囲に書き込まないかを確認するためのデータを書き込みます。
        if( m_OffsetMeta > 0 )
        {
            m_StorageControlArea.FillBuffer(0, static_cast<size_t>(m_OffsetMeta), 0x66);
            m_StorageTable.FillBuffer(0, static_cast<size_t>(m_OffsetMeta), 0x66);
            m_StorageBitmapUpdatedPhysical.FillBuffer(0, static_cast<size_t>(m_OffsetMeta), 0x66);
            m_StorageBitmapUpdatedVirtual.FillBuffer(0, static_cast<size_t>(m_OffsetMeta), 0x66);
            m_StorageBitmapUnassigned.FillBuffer(0, static_cast<size_t>(m_OffsetMeta), 0x66);
        }
        if( m_OffsetData > 0 )
        {
            m_StorageBaseData.FillBuffer(0, static_cast<size_t>(m_OffsetData), 0x66);
        }

        {
            int64_t sizeTable;
            int64_t sizeBitmapUpdatedPhysical;
            int64_t sizeBitmapUpdatedVirtual;
            int64_t sizeBitmapUnassigned;
            nn::fssystem::save::JournalStorage::QueryMappingMetaSize(
                &sizeTable,
                &sizeBitmapUpdatedPhysical,
                &sizeBitmapUpdatedVirtual,
                &sizeBitmapUnassigned,
                GetTotalAreaSize(),
                GetReservedAreaSize(),
                GetBlockSize()
            );

            nn::fs::SubStorage subStorageTable(
                &m_StorageTable,
                m_OffsetMeta,
                sizeTable
            );
            nn::fs::SubStorage subStorageBitmapUpdatedPhysical(
                &m_StorageBitmapUpdatedPhysical,
                m_OffsetMeta,
                sizeBitmapUpdatedPhysical
            );
            nn::fs::SubStorage subStorageBitmapUpdatedVirtual(
                &m_StorageBitmapUpdatedVirtual,
                m_OffsetMeta,
                sizeBitmapUpdatedVirtual
            );
            nn::fs::SubStorage subStorageBitmapUnassigned(
                &m_StorageBitmapUnassigned,
                m_OffsetMeta,
                sizeBitmapUnassigned
            );
            nn::fs::SubStorage subStorageData(
                &m_StorageData,
                m_OffsetData,
                GetTotalAreaSize()
            );

            NNT_ASSERT_RESULT_SUCCESS(
                nn::fssystem::save::JournalStorage::Format(
                    subStorageControlArea,
                    subStorageTable,
                    subStorageBitmapUpdatedPhysical,
                    subStorageBitmapUpdatedVirtual,
                    subStorageBitmapUnassigned,
                    GetTotalAreaSize(),
                    GetReservedAreaSize(),
                    GetBlockSize()
                )
            );

            m_Storage.reset(new nn::fssystem::save::JournalStorage());

            NNT_ASSERT_RESULT_SUCCESS(
                m_Storage->Initialize(
                    subStorageControlArea,
                    subStorageTable,
                    subStorageBitmapUpdatedPhysical,
                    subStorageBitmapUpdatedVirtual,
                    subStorageBitmapUnassigned,
                    subStorageData
                )
            );
        }
    } // NOLINT(impl/function_size)

    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
    {
        EXPECT_TRUE(m_StorageControlArea.CheckValid());
        EXPECT_TRUE(m_StorageTable.CheckValid());
        EXPECT_TRUE(m_StorageBitmapUpdatedPhysical.CheckValid());
        EXPECT_TRUE(m_StorageBitmapUpdatedVirtual.CheckValid());
        EXPECT_TRUE(m_StorageBitmapUnassigned.CheckValid());
        EXPECT_TRUE(m_StorageBaseData.CheckValid());
    }

    // 再マウントを行ないます。
    nn::Result Remount(bool format) NN_NOEXCEPT
    {
        m_StorageData.StopCounting();
        m_Storage.reset(new nn::fssystem::save::JournalStorage());

        const int64_t sizeControlArea = sizeof(nn::fssystem::save::JournalStorage::ControlArea);
        int64_t sizeTable;
        int64_t sizeBitmapUpdatedPhysical;
        int64_t sizeBitmapUpdatedVirtual;
        int64_t sizeBitmapUnassigned;
        nn::fssystem::save::JournalStorage::QueryMappingMetaSize(
            &sizeTable,
            &sizeBitmapUpdatedPhysical,
            &sizeBitmapUpdatedVirtual,
            &sizeBitmapUnassigned,
            GetTotalAreaSize(),
            GetReservedAreaSize(),
            GetBlockSize()
        );

        nn::fs::SubStorage subStorageControlArea(
            &m_StorageControlArea,
            m_OffsetMeta,
            sizeControlArea
        );
        nn::fs::SubStorage subStorageTable(
            &m_StorageTable,
            m_OffsetMeta,
            sizeTable
        );
        nn::fs::SubStorage subStorageBitmapUpdatedPhysical(
            &m_StorageBitmapUpdatedPhysical,
            m_OffsetMeta,
            sizeBitmapUpdatedPhysical
        );
        nn::fs::SubStorage subStorageBitmapUpdatedVirtual(
            &m_StorageBitmapUpdatedVirtual,
            m_OffsetMeta,
            sizeBitmapUpdatedVirtual
        );
        nn::fs::SubStorage subStorageBitmapUnassigned(
            &m_StorageBitmapUnassigned,
            m_OffsetMeta,
            sizeBitmapUnassigned
        );
        nn::fs::SubStorage subStorageData(
            &m_StorageData,
            m_OffsetData,
            GetTotalAreaSize()
        );

        if( format )
        {
            NN_RESULT_DO(
                nn::fssystem::save::JournalStorage::Format(
                    subStorageControlArea,
                    subStorageTable,
                    subStorageBitmapUpdatedPhysical,
                    subStorageBitmapUpdatedVirtual,
                    subStorageBitmapUnassigned,
                    GetTotalAreaSize(),
                    GetReservedAreaSize(),
                    GetBlockSize()
                )
            );
        }

        NN_RESULT_DO(
            m_Storage->Initialize(
                subStorageControlArea,
                subStorageTable,
                subStorageBitmapUpdatedPhysical,
                subStorageBitmapUpdatedVirtual,
                subStorageBitmapUnassigned,
                subStorageData
            )
        );

        NN_RESULT_SUCCESS;
    }

    // マッピングテーブルをバックアップ
    void BuckupMappingTable() NN_NOEXCEPT
    {
        int64_t sizeTable = 0;
        int64_t sizeBitmapUpdatedPhysical = 0;
        int64_t sizeBitmapUpdatedVirtual = 0;
        int64_t sizeBitmapUnassigned = 0;

        m_StorageTable.GetSize(&sizeTable);
        m_StorageBitmapUpdatedPhysical.GetSize(&sizeBitmapUpdatedPhysical);
        m_StorageBitmapUpdatedVirtual.GetSize(&sizeBitmapUpdatedVirtual);
        m_StorageBitmapUnassigned.GetSize(&sizeBitmapUnassigned);

        m_BufferTable.reset(new char[static_cast<size_t>(sizeTable)]);
        m_BufferBitmapUpdatedPhysical.reset(new char[static_cast<size_t>(sizeBitmapUpdatedPhysical)]);
        m_BufferBitmapUpdatedVirtual.reset(new char[static_cast<size_t>(sizeBitmapUpdatedVirtual)]);
        m_BufferBitmapUnassigned.reset(new char[static_cast<size_t>(sizeBitmapUnassigned)]);

        m_StorageTable.Read(
            0,
            m_BufferTable.get(),
            static_cast<size_t>(sizeTable)
        );
        m_StorageBitmapUpdatedPhysical.Read(
            0,
            m_BufferBitmapUpdatedPhysical.get(),
            static_cast<size_t>(sizeBitmapUpdatedPhysical)
        );
        m_StorageBitmapUpdatedVirtual.Read(
            0,
            m_BufferBitmapUpdatedVirtual.get(),
            static_cast<size_t>(sizeBitmapUpdatedVirtual)
        );
        m_StorageBitmapUnassigned.Read(
            0,
            m_BufferBitmapUnassigned.get(),
            static_cast<size_t>(sizeBitmapUnassigned)
        );
    }

    // マッピングテーブルをロールバック
    void RollbackMappingTable() NN_NOEXCEPT
    {
        int64_t sizeTable = 0;
        int64_t sizeBitmapUpdatedPhysical = 0;
        int64_t sizeBitmapUpdatedVirtual = 0;
        int64_t sizeBitmapUnassigned = 0;

        m_StorageTable.GetSize(&sizeTable);
        m_StorageBitmapUpdatedPhysical.GetSize(&sizeBitmapUpdatedPhysical);
        m_StorageBitmapUpdatedVirtual.GetSize(&sizeBitmapUpdatedVirtual);
        m_StorageBitmapUnassigned.GetSize(&sizeBitmapUnassigned);

        m_StorageTable.Write(
            0,
            m_BufferTable.get(),
            static_cast<size_t>(sizeTable)
        );
        m_StorageBitmapUpdatedPhysical.Write(
            0,
            m_BufferBitmapUpdatedPhysical.get(),
            static_cast<size_t>(sizeBitmapUpdatedPhysical)
        );
        m_StorageBitmapUpdatedVirtual.Write(
            0,
            m_BufferBitmapUpdatedVirtual.get(),
            static_cast<size_t>(sizeBitmapUpdatedVirtual)
        );
        m_StorageBitmapUnassigned.Write(0,
            m_BufferBitmapUnassigned.get(),
            static_cast<size_t>(sizeBitmapUnassigned)
        );
    }

    // 拡張、再マウントを行ないます。
    nn::Result Expand() NN_NOEXCEPT
    {
        // これ以上拡張できないなら終了
        if( m_CountMapEntryExpanded == m_CountMapEntryExpandable
            && m_CountReservedExpanded == m_CountReservedExpandable )
        {
            NN_RESULT_SUCCESS;
        }

        // 拡張可能サイズの 1 / 4 ずつ拡張
        m_CountMapEntryExpanded += m_CountMapEntryExpandable / 4;
        m_CountReservedExpanded += m_CountReservedExpandable / 4;

        if( m_CountMapEntryExpanded > m_CountMapEntryExpandable )
        {
            m_CountMapEntryExpanded = m_CountMapEntryExpandable;
        }
        if( m_CountReservedExpanded > m_CountReservedExpandable )
        {
            m_CountReservedExpanded = m_CountReservedExpandable;
        }

        const int64_t sizeControlArea = sizeof(nn::fssystem::save::JournalStorage::ControlArea);
        int64_t sizeTable;
        int64_t sizeBitmapUpdatedPhysical;
        int64_t sizeBitmapUpdatedVirtual;
        int64_t sizeBitmapUnassigned;
        nn::fssystem::save::JournalStorage::QueryMappingMetaSize(
            &sizeTable,
            &sizeBitmapUpdatedPhysical,
            &sizeBitmapUpdatedVirtual,
            &sizeBitmapUnassigned,
            GetTotalAreaSize(),
            GetReservedAreaSize(),
            GetBlockSize()
        );

        nn::fs::SubStorage subStorageControlArea(
            &m_StorageControlArea,
            m_OffsetMeta,
            sizeControlArea
        );
        nn::fs::SubStorage subStorageTable(
            &m_StorageTable,
            m_OffsetMeta,
            sizeTable
        );
        nn::fs::SubStorage subStorageBitmapUpdatedPhysical(
            &m_StorageBitmapUpdatedPhysical,
            m_OffsetMeta,
            sizeBitmapUpdatedPhysical
        );
        nn::fs::SubStorage subStorageBitmapUpdatedVirtual(
            &m_StorageBitmapUpdatedVirtual,
            m_OffsetMeta,
            sizeBitmapUpdatedVirtual
        );
        nn::fs::SubStorage subStorageBitmapUnassigned(
            &m_StorageBitmapUnassigned,
            m_OffsetMeta,
            sizeBitmapUnassigned
        );
        nn::fs::SubStorage subStorageData(
            &m_StorageData,
            m_OffsetData,
            GetTotalAreaSize()
        );

        NN_RESULT_DO(
            nn::fssystem::save::JournalStorage::Expand(
                subStorageControlArea,
                subStorageTable,
                subStorageBitmapUpdatedPhysical,
                subStorageBitmapUpdatedVirtual,
                subStorageBitmapUnassigned,
                GetTotalAreaSize(),
                GetReservedAreaSize()
            )
        );

        m_Storage.reset(new nn::fssystem::save::JournalStorage());

        NN_RESULT_DO(
            m_Storage->Initialize(
                subStorageControlArea,
                subStorageTable,
                subStorageBitmapUpdatedPhysical,
                subStorageBitmapUpdatedVirtual,
                subStorageBitmapUnassigned,
                subStorageData
            )
        );

        NN_RESULT_SUCCESS;
    }

    nn::fssystem::save::JournalStorage* GetJournalStorage() NN_NOEXCEPT
    {
        return m_Storage.get();
    }

    nnt::fs::util::PowerInterruptionStorage* GetPowerInterruptionStorage() NN_NOEXCEPT
    {
        return &m_StorageData;
    }

private:
    std::unique_ptr<nn::fssystem::save::JournalStorage> m_Storage;
    nnt::fs::util::PowerInterruptionStorage m_StorageData;

    uint32_t m_CountMapEntryExpandable;
    uint32_t m_CountReservedExpandable;
    uint32_t m_CountMapEntryExpanded;
    uint32_t m_CountReservedExpanded;

    int64_t m_OffsetMeta;
    int64_t m_OffsetData;
    uint32_t m_CountMapEntry;
    uint32_t m_CountReserved;
    int64_t m_SizeBlock;

    nnt::fs::util::SafeMemoryStorage m_StorageControlArea;
    nnt::fs::util::SafeMemoryStorage m_StorageTable;
    nnt::fs::util::SafeMemoryStorage m_StorageBitmapUpdatedPhysical;
    nnt::fs::util::SafeMemoryStorage m_StorageBitmapUpdatedVirtual;
    nnt::fs::util::SafeMemoryStorage m_StorageBitmapUnassigned;
    nnt::fs::util::SafeMemoryStorage m_StorageBaseData;

    std::unique_ptr<char[]> m_BufferTable;
    std::unique_ptr<char[]> m_BufferBitmapUpdatedPhysical;
    std::unique_ptr<char[]> m_BufferBitmapUpdatedVirtual;
    std::unique_ptr<char[]> m_BufferBitmapUnassigned;
};

// 書き込み可能回数を増やしながら、先頭 1024 バイトに書き込みます
// 書き込みの際に電源断が発生しなければ終了します。
TEST_P(JournalStoragePowerInterruptionTest, TestWriteHead)
{
    const size_t SizeAccess = 1024;
    const int64_t Offset = 0;
    std::unique_ptr<char[]> bufWrite(new char[SizeAccess]);
    nnt::fs::util::FillBufferWithRandomValue(bufWrite.get(), SizeAccess);

    for( int writableCount = 0;; ++writableCount )
    {
        Remount(true);

        // 初期値を書き込みます
        std::unique_ptr<char[]> bufDefault(new char[SizeAccess]);
        nnt::fs::util::FillBufferWithRandomValue(bufDefault.get(), SizeAccess);
        {
            NNT_ASSERT_RESULT_SUCCESS(
                GetJournalStorage()->Write(Offset, bufDefault.get(), SizeAccess)
            );
            NNT_ASSERT_RESULT_SUCCESS(
                GetJournalStorage()->Commit()
            );
        }

        // マッピングテーブルをバックアップ
        BuckupMappingTable();

        // 電源断テスト開始(writableCount 回書き込んだ時点で電源断が発生)
        GetPowerInterruptionStorage()->StartCounting(writableCount);

        // 書き込み
        nn::Result resultWrite;
        nn::Result resultCommit;
        resultWrite = GetJournalStorage()->Write(Offset, bufWrite.get(), SizeAccess);

        if( resultWrite.IsSuccess() )
        {
            resultCommit = GetJournalStorage()->Commit();
        }

        if( !GetPowerInterruptionStorage()->IsPowerInterruptionOccurred() )
        {
            ASSERT_TRUE(resultWrite.IsSuccess());
            ASSERT_TRUE(resultCommit.IsSuccess());

            // 読み込みチェック
            std::unique_ptr<char[]> bufRead(new char[SizeAccess]);
            NNT_ASSERT_RESULT_SUCCESS(
                GetJournalStorage()->Read(Offset, bufRead.get(), SizeAccess)
            );
            ASSERT_EQ(0, memcmp(bufRead.get(), bufWrite.get(), SizeAccess));

            // 電源断が発生しなかったので終了
            break;
        }
        GetPowerInterruptionStorage()->StopCounting();

        // コミットに失敗していた場合はマッピングテーブルを差し戻し
        RollbackMappingTable();

        // 再マウントすると書き込み前の値が読み込めることを確認
        Remount(false);

        {
            if( resultWrite.IsSuccess() )
            {
                ASSERT_TRUE(resultCommit.IsFailure());
            }

            // 読み込みチェック
            std::unique_ptr<char[]> bufRead(new char[SizeAccess]);
            NNT_ASSERT_RESULT_SUCCESS(
                GetJournalStorage()->Read(Offset, bufRead.get(), SizeAccess)
            );

            ASSERT_EQ(0, memcmp(bufRead.get(), bufDefault.get(), SizeAccess));
        }
    };
}

// 書き込み可能回数を増やしながら、終端の 1024 バイトに書き込みます
// 書き込みの際に電源断が発生しなければ終了します。
TEST_P(JournalStoragePowerInterruptionTest, TestWriteTail)
{
    const size_t SizeAccess = 1024;
    const int64_t Offset = GetWritableAreaSize() - SizeAccess;
    std::unique_ptr<char[]> bufWrite(new char[SizeAccess]);
    nnt::fs::util::FillBufferWithRandomValue(bufWrite.get(), SizeAccess);
    for( int writableCount = 0;; ++writableCount )
    {
        Remount(true);

        // 初期値を書き込みます
        std::unique_ptr<char[]> bufDefault(new char[SizeAccess]);
        nnt::fs::util::FillBufferWithRandomValue(bufDefault.get(), SizeAccess);
        {
            NNT_ASSERT_RESULT_SUCCESS(
                GetJournalStorage()->Write(Offset, bufDefault.get(), SizeAccess)
            );
            NNT_ASSERT_RESULT_SUCCESS(
                GetJournalStorage()->Commit()
            );
        }

        // マッピングテーブルをバックアップ
        BuckupMappingTable();

        // 電源断テスト開始(writableCount 回書き込んだ時点で電源断が発生)
        GetPowerInterruptionStorage()->StartCounting(writableCount);

        // 書き込み
        nn::Result resultWrite;
        nn::Result resultCommit;
        resultWrite = GetJournalStorage()->Write(Offset, bufWrite.get(), SizeAccess);
        if( resultWrite.IsSuccess() )
        {
            resultCommit = GetJournalStorage()->Commit();
        }

        if( !GetPowerInterruptionStorage()->IsPowerInterruptionOccurred() )
        {
            ASSERT_TRUE(resultWrite.IsSuccess());
            ASSERT_TRUE(resultCommit.IsSuccess());

            // 読み込みチェック
            std::unique_ptr<char[]> bufRead(new char[SizeAccess]);
            NNT_ASSERT_RESULT_SUCCESS(
                GetJournalStorage()->Read(Offset, bufRead.get(), SizeAccess)
            );
            ASSERT_EQ(0, memcmp(bufRead.get(), bufWrite.get(), SizeAccess));

            // 電源断が発生しなかったので終了
            break;
        }

        GetPowerInterruptionStorage()->StopCounting();

        // コミットに失敗していた場合はマッピングテーブルを差し戻し
        RollbackMappingTable();

        // 再マウントすると書き込み前の値が読み込めることを確認
        Remount(false);

        {
            if( resultWrite.IsSuccess() )
            {
                ASSERT_TRUE(resultCommit.IsFailure());
            }

            // 読み込みチェック
            std::unique_ptr<char[]> bufRead(new char[SizeAccess]);
            NNT_ASSERT_RESULT_SUCCESS(
                GetJournalStorage()->Read(Offset, bufRead.get(), SizeAccess)
            );
            ASSERT_EQ(0, memcmp(bufRead.get(), bufDefault.get(), SizeAccess));
        }
    };
}

// 終端書き込み -> コミット -> 領域拡張 x 4 -> 終端書き込み -> コミット を繰り返し行う
// 電源断が発生した場合、発生した時点でのベリファイを行う
// 電源断の回数を徐々に増やし、最後のコミットまで電源断無しで行えると終了
TEST_P(JournalStoragePowerInterruptionTest, TestExpand)
{
    const size_t SizeAccess = 16 * 1024;
    std::unique_ptr<char[]> bufDefault(new char[SizeAccess]);
    std::unique_ptr<char[]> bufWrite(new char[SizeAccess]);
    nn::Result result;
    for( int writableCount = 0;; ++writableCount )
    {
        // ランダムな値を書き込んでおきます
        nnt::fs::util::FillBufferWithRandomValue(bufDefault.get(), SizeAccess);
        nnt::fs::util::FillBufferWithRandomValue(bufWrite.get(), SizeAccess);

        Remount(true);

        int64_t offset = GetWritableAreaSize() - SizeAccess;

        // 初期値を書き込みます
        {
            NNT_ASSERT_RESULT_SUCCESS(
                GetJournalStorage()->Write(offset, bufDefault.get(), SizeAccess)
            );
            NNT_ASSERT_RESULT_SUCCESS(
                GetJournalStorage()->Commit()
            );
        }

        // マッピングテーブルをバックアップ
        BuckupMappingTable();

        // 電源断テスト開始(writableCount 回書き込んだ時点で電源断が発生)
        GetPowerInterruptionStorage()->StartCounting(writableCount);

        // 書き込み、コミット
        nn::Result writeResult;
        nn::Result commitResult = nn::ResultSuccess();
        writeResult = GetJournalStorage()->Write(offset, bufWrite.get(), SizeAccess);
        if( writeResult.IsSuccess() )
        {
            commitResult = GetJournalStorage()->Commit();
        }

        // 書き込みかコミットで失敗したなら、再マウントしてから中身をチェックし、はじめからやり直す
        if( writeResult.IsFailure() || commitResult.IsFailure() )
        {
            // コミットに失敗していた場合はマッピングテーブルを差し戻し
            RollbackMappingTable();

            Remount(false);
            std::unique_ptr<char[]> bufRead(new char[SizeAccess]);
            NNT_ASSERT_RESULT_SUCCESS(
                GetJournalStorage()->Read(offset, bufRead.get(), SizeAccess)
            );
            ASSERT_EQ(0, memcmp(bufDefault.get(), bufRead.get(), SizeAccess));
            continue;
        }

        // マッピングテーブルをバックアップ
        BuckupMappingTable();

        // 領域を拡張します。
        nn::Result resultExpand;
        for( int i = 0; i < 4; ++i )
        {
            resultExpand = Expand();
            if( resultExpand.IsFailure() )
            {
                break;
            }
        }

        if( resultExpand.IsFailure() )
        {
            // マッピングテーブルを差し戻し
            RollbackMappingTable();

            Remount(false);
            std::unique_ptr<char[]> bufRead(new char[SizeAccess]);
            NNT_ASSERT_RESULT_SUCCESS(
                GetJournalStorage()->Read(offset, bufRead.get(), SizeAccess)
            );
            ASSERT_EQ(0, memcmp(bufWrite.get(), bufRead.get(), SizeAccess));
            continue;
        }

        // マッピングテーブルをバックアップ
        BuckupMappingTable();

        // オフセットを拡張後の部分に書き込むように更新
        offset = GetWritableAreaSize() - SizeAccess;

        // オフセット更新にあわせて bufDefault を更新
        NNT_ASSERT_RESULT_SUCCESS(
            GetJournalStorage()->Read(offset, bufDefault.get(), SizeAccess)
        );

        writeResult = GetJournalStorage()->Write(offset, bufWrite.get(), SizeAccess);
        if( writeResult.IsSuccess() )
        {
            commitResult = GetJournalStorage()->Commit();
        }

        // 書き込みかコミットで失敗したなら、再マウントしてから中身をチェックし、はじめからやり直す
        if( writeResult.IsFailure() || commitResult.IsFailure() )
        {
            // コミットに失敗していた場合はマッピングテーブルを差し戻し
            RollbackMappingTable();

            Remount(false);
            std::unique_ptr<char[]> bufRead(new char[SizeAccess]);
            NNT_ASSERT_RESULT_SUCCESS(
                GetJournalStorage()->Read(offset, bufRead.get(), SizeAccess)
            );
            ASSERT_EQ(0, memcmp(bufDefault.get(), bufRead.get(), SizeAccess));
            continue;
        }
        ASSERT_FALSE(GetPowerInterruptionStorage()->IsPowerInterruptionOccurred());
        break;
    }
}

INSTANTIATE_TEST_CASE_P(
    JournalStorageTest,
    JournalStoragePowerInterruptionTest,
    ::testing::ValuesIn(MakeAccessTestParam())
);
