﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <random>
#include <numeric>
#include <nn/fssystem/save/fs_BlockCacheBufferedStorage.h>
#include <nnt/nntest.h>
#include <nnt/fsUtil/testFs_util.h>
#include "testFs_util_CommonStorageTests.h"

// テストクラス
class FsBlockCacheBufferedStorageTest : public testing::TestWithParam<bool>
{
public:
    // ストレージのラッパーです。
    template< typename TMemory >
    class StorageWrapper
    {
    public:
        // コンストラクタです。
        StorageWrapper(
            int cacheCountMax,
            size_t cacheBufferSize,
            size_t cacheBlockSize,
            nn::fs::StorageType storageType
        ) NN_NOEXCEPT
            : m_CacheBuffer(cacheBufferSize)
            , m_Memory()
            , m_Mutex(true)
            , m_Buffer()
            , m_Storage()
            , m_StorageType(storageType)
        {
            nn::Result result = m_Buffer.Initialize(
                                    cacheCountMax,
                                    reinterpret_cast<uintptr_t>(&m_CacheBuffer[0]),
                                    cacheBufferSize,
                                    cacheBlockSize
                                );
            NN_UNUSED(result);
            NN_SDK_ASSERT(result.IsSuccess());
        }

        // コンストラクタです。
        StorageWrapper(
            int cacheCountMax,
            size_t cacheBufferSize,
            size_t cacheBlockSize,
            int64_t sizeData,
            size_t sizeVerificationBlock,
            nn::fs::StorageType storageType
        ) NN_NOEXCEPT
            : m_CacheBuffer(cacheBufferSize)
            , m_Memory(sizeData, sizeVerificationBlock)
            , m_Mutex(true)
            , m_Buffer()
            , m_Storage()
            , m_StorageType(storageType)
        {
            nn::Result result = m_Buffer.Initialize(
                                    cacheCountMax,
                                    reinterpret_cast<uintptr_t>(&m_CacheBuffer[0]),
                                    cacheBufferSize,
                                    cacheBlockSize
                                );
            NN_UNUSED(result);
            NN_SDK_ASSERT(result.IsSuccess());
        }

        // 初期化します。
        nn::Result Initialize(
                       int64_t sizeData,
                       size_t sizeVerificationBlock,
                       bool isRealDataCache,
                       bool isKeepBurstMode
                   ) NN_NOEXCEPT
        {
            m_Memory.Initialize(sizeData);

            return m_Storage.Initialize(
                &m_Buffer,
                &m_Mutex,
                &m_Memory,
                sizeData,
                sizeVerificationBlock,
                m_Storage.DefaultMaxCacheEntryCount,
                isRealDataCache,
                0,
                isKeepBurstMode,
                m_StorageType
            );
        }

        // ストレージを取得します。
        nn::fssystem::save::BlockCacheBufferedStorage& Get() NN_NOEXCEPT
        {
            return m_Storage;
        }

        // ストレージを取得します。
        const nn::fssystem::save::BlockCacheBufferedStorage& Get() const NN_NOEXCEPT
        {
            return m_Storage;
        }

        // ストレージ用のメモリを取得します。
        TMemory& GetMemory() NN_NOEXCEPT
        {
            return m_Memory;
        }

        // ストレージ用のメモリを取得します。
        const TMemory& GetMemory() const NN_NOEXCEPT
        {
            return m_Memory;
        }

        // バッファの空き容量を取得します。
        const size_t GetBufferAlocatableSize() const NN_NOEXCEPT
        {
            return m_Buffer.GetTotalAllocatableSize();
        }

    private:
        nnt::fs::util::Vector<char> m_CacheBuffer;
        TMemory m_Memory;
        nn::os::Mutex m_Mutex;
        nn::fssystem::FileSystemBufferManager m_Buffer;
        nn::fssystem::save::BlockCacheBufferedStorage m_Storage;
        nn::fs::StorageType m_StorageType;
    };
};

// ストレージの初期化・書き込みを行う簡単なテスト
TEST_P(FsBlockCacheBufferedStorageTest, Simple)
{
    static const size_t TestSize = 512 * 1024;
    static const size_t VerificationBlockSize = 32;
    static const int CacheCountMax = 1024;
    static const size_t CacheBlockSize = 4 * 1024;
    static const size_t CacheBufferSize = 1024 * 1024;

    const bool isRealData = GetParam();
    nnt::fs::util::Vector<char> buf1(TestSize + 1);
    nnt::fs::util::Vector<char> buf2(TestSize + 1);
    char* ptr1 = &buf1[0];
    char* ptr2 = &buf2[0];

    // テストデータ設定
    for( size_t i = 0; i < TestSize; ++i )
    {
        buf1[i] = i & 0xFF;
    }

    for( int burstMode = 0; burstMode < 2; ++burstMode )
    {
        for( size_t storageSize = 32; storageSize <= TestSize; storageSize *= 2 )
        {
            NN_SDK_LOG(".");

            StorageWrapper<nnt::fs::util::SafeMemoryStorage>
                storage(CacheCountMax, CacheBufferSize, CacheBlockSize, nn::fs::StorageType::StorageType_SaveData);

            // ストレージの初期化
            NNT_ASSERT_RESULT_SUCCESS(
                storage.Initialize(
                    storageSize,
                    VerificationBlockSize,
                    isRealData,
                    (burstMode != 0)
                )
            );

            // ストレージへの初期値書き込み
            std::memset(ptr2, 0x55, TestSize + 1);
            NNT_ASSERT_RESULT_SUCCESS(storage.Get().Write(0, ptr2, storageSize));

            // ストレージへの書き込み
            if( 512 <= storageSize )
            {
                const size_t size = storageSize / 16;
                for( size_t offset = 0; offset < storageSize; offset += size )
                {
                    NNT_ASSERT_RESULT_SUCCESS(storage.Get().Write(offset, &buf1[offset], size));
                }
            }
            else
            {
                for( size_t offset = 0; offset < storageSize; ++offset )
                {
                    NNT_ASSERT_RESULT_SUCCESS(storage.Get().Write(offset, &buf1[offset], 1));
                }
            }
            NNT_ASSERT_RESULT_SUCCESS(storage.Get().Write(storageSize, &buf1[storageSize], 1));

            // 範囲外のアクセステスト
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultInvalidOffset,
                storage.Get().Write(storageSize + 1, ptr1, 1)
            );

            NNT_ASSERT_RESULT_SUCCESS(storage.Get().Read(0, ptr2, TestSize));

            // データ比較
            ASSERT_EQ(0, std::memcmp(ptr1, ptr2, storageSize)); // ストレージから読み込んだデータ
            ASSERT_EQ(0x55, buf2[storageSize]); // 初期値のまま
        }
    }
    NN_SDK_LOG("\n");
}

// 4 GB を超えるオフセットの読み書きテスト
TEST_P(FsBlockCacheBufferedStorageTest, LargeOffset)
{
    static const size_t AccessSize = 1024;
    static const int64_t StorageSize = static_cast<int64_t>(64) * 1024 * 1024 * 1024 + AccessSize;
    static const size_t VerificationBlockSize = 32;
    static const int CacheCountMax = 1024;
    static const size_t CacheBlockSize = 4 * 1024;
    static const size_t CacheBufferSize = 1024 * 1024;

    const bool isRealData = GetParam();

    for( int burstMode = 0; burstMode < 2; ++burstMode )
    {
        StorageWrapper<nnt::fs::util::VirtualMemoryStorage>
            storage(CacheCountMax, CacheBufferSize, CacheBlockSize, nn::fs::StorageType::StorageType_SaveData);

        // ストレージの初期化
        NNT_ASSERT_RESULT_SUCCESS(
            storage.Initialize(
                StorageSize,
                VerificationBlockSize,
                isRealData,
                (burstMode != 0)
            )
        );

        TestLargeOffsetAccess(
            &storage.Get(),
            AccessSize,
            [&](int64_t offset, char* readBuffer, const char* writeBuffer, size_t bufferSize) NN_NOEXCEPT
            {
                // BlockCacheBufferedStorage から読み込み
                NNT_ASSERT_RESULT_SUCCESS(storage.Get().Read(offset, readBuffer, bufferSize));
                NNT_FS_UTIL_EXPECT_MEMCMPEQ(writeBuffer, readBuffer, bufferSize);

                // 元ストレージから読み込み
                NNT_ASSERT_RESULT_SUCCESS(storage.GetMemory().Read(offset, readBuffer, bufferSize));
                NNT_FS_UTIL_EXPECT_MEMCMPEQ(writeBuffer, readBuffer, bufferSize);
            }
        );
    }
}

// ストレージへランダムなオフセット・サイズで読み書きするテスト
TEST_P(FsBlockCacheBufferedStorageTest, RandomAccessHeavy)
{
    static const int LoopCount = 1000;
    static const size_t TestSize = 512 * 1024;
    static const size_t VerificationBlockSize = 32;
    static const int CacheCountMax = 1024;
    static const size_t CacheBlockSize = 4096;
    static const size_t CacheBufferSize = 1024 * 1024;

    const bool isReadData = GetParam();
    std::mt19937 mt(nnt::fs::util::GetRandomSeed());
    nnt::fs::util::Vector<char> buf1(TestSize);
    nnt::fs::util::Vector<char> buf2(TestSize);
    nnt::fs::util::Vector<char> buf3(TestSize);
    char* ptr1 = &buf1[0];
    char* ptr2 = &buf2[0];
    char* ptr3 = &buf3[0];

    for( int burstMode = 0; burstMode < 2; ++burstMode )
    {
        for( size_t storageSize = 32; storageSize <= TestSize; storageSize *= 2 )
        {
            NN_SDK_LOG(".");

            StorageWrapper<nnt::fs::util::SafeMemoryStorage>
                storage(CacheCountMax, CacheBufferSize, CacheBlockSize, nn::fs::StorageType::StorageType_SaveData);

            // ストレージの初期化
            NNT_ASSERT_RESULT_SUCCESS(
                storage.Initialize(
                    storageSize,
                    VerificationBlockSize,
                    isReadData,
                    (burstMode != 0)
                )
            );

            // ストレージのクリア
            std::memset(ptr2, 0, storageSize);
            NNT_ASSERT_RESULT_SUCCESS(storage.Get().Write(0, ptr2, storageSize));

            // ランダムアクセステスト
            for( int loop = 1; loop <= LoopCount; ++loop )
            {
                size_t offset = std::uniform_int_distribution<size_t>(0, storageSize - 1)(mt);
                size_t size = std::uniform_int_distribution<size_t>(0, storageSize / 2 - 1)(mt);
                if( storageSize < offset + size )
                {
                    size = storageSize - offset;
                }

                // ストレージに書き込み
                std::memset(ptr1, loop & 0xFF, size);
                NNT_ASSERT_RESULT_SUCCESS(storage.Get().Write(offset, ptr1, size));

                // 書き込んだデータをコピー
                std::memcpy(&buf2[offset], ptr1, size);

                if( (std::uniform_int_distribution<>(0, LoopCount / 10 - 1)(mt) == 0) ||
                    (loop % (LoopCount / 10) == (LoopCount / 10 - 1)) )
                {
                    if( std::uniform_int_distribution<>(0, 1)(mt) == 0 )
                    {
                        // ストレージから読み込み
                        NNT_ASSERT_RESULT_SUCCESS(storage.Get().Read(0, ptr3, storageSize));
                    }
                    else
                    {
                        // ダーティーキャッシュをコミット
                        NNT_ASSERT_RESULT_SUCCESS(storage.Get().Commit());

                        // コミットされたので直接読んでも値が一致するはず
                        NNT_ASSERT_RESULT_SUCCESS(storage.GetMemory().Read(0, ptr3, storageSize));
                    }

                    ASSERT_EQ(0, std::memcmp(ptr2, ptr3, storageSize));
                }
            }
        }
    }

    NN_SDK_LOG("\n");
}

// ストレージのフラッシュテスト
TEST_F(FsBlockCacheBufferedStorageTest, Flush)
{
    static const size_t TestSize = 512 * 1024;
    static const size_t VerificationBlockSize = 32;
    static const int CacheCountMax = 1024;
    static const size_t CacheBlockSize = 4 * 1024;
    static const size_t CacheBufferSize = 1024 * 1024;

    nnt::fs::util::Vector<char> buf1(TestSize);
    char* ptr1 = &buf1[0];

    // テストデータ設定
    for( size_t i = 0; i < TestSize; ++i )
    {
        buf1[i] = i & 0xFF;
    }

    StorageWrapper<nnt::fs::util::AccessCountedMemoryStorage>
        storage(CacheCountMax, CacheBufferSize, CacheBlockSize, nn::fs::StorageType::StorageType_SaveData);

    // ストレージの初期化
    NNT_ASSERT_RESULT_SUCCESS(storage.Initialize(TestSize, VerificationBlockSize, false, false));

    storage.GetMemory().ResetAccessCounter();

    // 範囲内へのデータ書き込み
    NNT_ASSERT_RESULT_SUCCESS(storage.Get().Write(0, ptr1, TestSize / 2));
    NNT_ASSERT_RESULT_SUCCESS(storage.Get().Write(TestSize / 2, ptr1, TestSize / 2));
    NNT_ASSERT_RESULT_SUCCESS(storage.Get().Write(0, ptr1, TestSize));

    ASSERT_EQ(0, storage.GetMemory().GetFlushTimes());

    storage.GetMemory().ResetAccessCounter();

    // バッファ書き出し指定で 0 バイトのデータ書き込み
    NNT_ASSERT_RESULT_SUCCESS(storage.Get().Write(0, ptr1, 0));

    ASSERT_EQ(0, storage.GetMemory().GetFlushTimes());

    storage.GetMemory().ResetAccessCounter();

    // 範囲外へのアクセス
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultInvalidOffset,
        storage.Get().Write(TestSize + 1, ptr1, TestSize)
    );
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultInvalidOffset,
        storage.Get().Write(TestSize * 2, ptr1, 32 * 1024)
    );

    // 範囲外エラー発生時に書き込みは行われない
    ASSERT_EQ(0, storage.GetMemory().GetWriteTimes());
}

// ストレージのキャッシュ無効化テスト
TEST_F(FsBlockCacheBufferedStorageTest, InvalidateRead)
{
    static const auto StorageSize = 512 * 1024;
    static const auto VerificationBlockSize = 32;
    static const auto CacheCountMax = 1024;
    static const auto CacheBlockSize = 4 * 1024;
    static const auto CacheBufferSize = 1024 * 1024;

    // ストレージを準備する
    StorageWrapper<nnt::fs::util::AccessCountedMemoryStorage> storage(
        CacheCountMax,
        CacheBufferSize,
        CacheBlockSize,
        nn::fs::StorageType::StorageType_RomFs);
    NNT_ASSERT_RESULT_SUCCESS(storage.Initialize(
        StorageSize,
        VerificationBlockSize,
        false,
        false));
    storage.GetMemory().ResetAccessCounter();

    std::mt19937 rng(nnt::fs::util::GetRandomSeed());

    for( auto count = 0; count < 100; ++count )
    {
        const auto offset = std::uniform_int_distribution<int64_t>(0, StorageSize - 1)(rng);
        char buffer[1] = {};

        // 下位ストレージからデータを読み込む
        NNT_EXPECT_RESULT_SUCCESS(storage.Get().Read(offset, buffer, 1));

        // 同じ個所を読み込んでもキャッシュされているので下位ストレージへのアクセスは発生しないはず
        storage.GetMemory().ResetAccessCounter();
        NNT_EXPECT_RESULT_SUCCESS(storage.Get().Read(offset, buffer, 1));
        EXPECT_EQ(0, storage.GetMemory().GetReadTimes());

        // キャッシュを破棄してから読むと再び下位ストレージへのアクセスが発生するはず
        storage.GetMemory().ResetAccessCounter();
        NNT_EXPECT_RESULT_SUCCESS(storage.Get().OperateRange(
            nn::fs::OperationId::Invalidate,
            0,
            std::numeric_limits<int64_t>::max()));
        NNT_EXPECT_RESULT_SUCCESS(storage.Get().Read(offset, buffer, 1));
        EXPECT_GT(storage.GetMemory().GetInvalidateTimes(), 0);
        EXPECT_GT(storage.GetMemory().GetReadTimes(), 0);
    }

}

// ストレージのキャッシュ無効化テスト
TEST_F(FsBlockCacheBufferedStorageTest, InvalidateWrite)
{
    static const auto StorageSize = 512 * 1024;
    static const auto VerificationBlockSize = 32;
    static const auto CacheCountMax = 1024;
    static const auto CacheBlockSize = 4 * 1024;
    static const auto CacheBufferSize = 1024 * 1024;

    // ストレージを準備する
    StorageWrapper<nnt::fs::util::AccessCountedMemoryStorage> storage(
        CacheCountMax,
        CacheBufferSize,
        CacheBlockSize,
        nn::fs::StorageType::StorageType_RomFs);
    NNT_ASSERT_RESULT_SUCCESS(storage.Initialize(
        StorageSize,
        VerificationBlockSize,
        false,
        false));

    size_t BufferSizeBeforeTest = storage.GetBufferAlocatableSize();
    std::mt19937 rng(nnt::fs::util::GetRandomSeed());

    for( auto count = 0; count < 100; ++count )
    {
        // ランダムに書き込みを行い、フラッシュを行わない
        const auto offset = std::uniform_int_distribution<int64_t>(0, StorageSize - 1)(rng);
        char buffer[1] = {};
        NNT_EXPECT_RESULT_SUCCESS(storage.Get().Write(offset, buffer, 1));
    }

    // ストレージ全体に無効化処理を行う
    NNT_EXPECT_RESULT_SUCCESS(storage.Get().OperateRange(
        nn::fs::OperationId::Invalidate,
        0,
        std::numeric_limits<int64_t>::max()));

    // バッファがリークしていないことを確かめる
    size_t BufferSizeAfterTest = storage.GetBufferAlocatableSize();
    ASSERT_EQ(BufferSizeBeforeTest, BufferSizeAfterTest);
}

// StorageType_SaveData時のInvalidate失敗テスト
TEST_F(FsBlockCacheBufferedStorageTest, StorageTypeSaveDataInvalidate)
{
    static const auto StorageSize = 512 * 1024;
    static const auto VerificationBlockSize = 32;
    static const auto CacheCountMax = 1024;
    static const auto CacheBlockSize = 4 * 1024;
    static const auto CacheBufferSize = 1024 * 1024;

    // ストレージを準備する
    StorageWrapper<nnt::fs::util::AccessCountedMemoryStorage> storage(
        CacheCountMax,
        CacheBufferSize,
        CacheBlockSize,
        nn::fs::StorageType::StorageType_SaveData);
    NNT_ASSERT_RESULT_SUCCESS(storage.Initialize(
        StorageSize,
        VerificationBlockSize,
        false,
        false));

    // ストレージ全体に無効化処理を行う
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultUnsupportedOperation,
        storage.Get().OperateRange(
        nn::fs::OperationId::Invalidate,
        0,
        std::numeric_limits<int64_t>::max()));
}

// 読み込み時のメモリアクセスのテスト
TEST_P(FsBlockCacheBufferedStorageTest, Read)
{
    static const size_t TestSize = 128 * 1024;
    static const size_t VerificationBlockSize = 512;
    static const int CacheCountMax = 1024;
    static const size_t CacheBlockSize = 4 * 1024;
    static const size_t CacheBufferSize = 64 * 1024;

    const bool isRealData = GetParam();
    nnt::fs::util::Vector<char> buf1(TestSize);
    char* ptr1 = &buf1[0];

    // テストデータ設定
    for( size_t i = 0; i < TestSize; i++ )
    {
        buf1[i] = i & 0xFF;
    }

    StorageWrapper<nnt::fs::util::AccessCountedMemoryStorage>
        storage(CacheCountMax, CacheBufferSize, CacheBlockSize, nn::fs::StorageType::StorageType_SaveData);

    // ストレージの初期化
    NNT_ASSERT_RESULT_SUCCESS(
        storage.Initialize(
            TestSize,
            VerificationBlockSize,
            isRealData,
            true
        )
    );

    NNT_ASSERT_RESULT_SUCCESS(storage.Get().Write(0, ptr1, TestSize));

    storage.GetMemory().ResetAccessCounter();

    // ストレージ全体
    NNT_ASSERT_RESULT_SUCCESS(storage.Get().Read(0, ptr1, TestSize));
    NNT_ASSERT_RESULT_SUCCESS(storage.Get().Read(0, ptr1, TestSize));
    NNT_ASSERT_RESULT_SUCCESS(storage.Get().Read(0, ptr1, TestSize));

    // カウンタの確認
    ASSERT_EQ(3, storage.GetMemory().GetReadTimes());

    storage.GetMemory().ResetAccessCounter();

    // 一部の領域
    NNT_ASSERT_RESULT_SUCCESS(storage.Get().Read(10, ptr1, 10));
    NNT_ASSERT_RESULT_SUCCESS(storage.Get().Read(20, ptr1, 10));
    NNT_ASSERT_RESULT_SUCCESS(storage.Get().Read(30, ptr1, 10));

    // カウンタの確認
    ASSERT_EQ(1, storage.GetMemory().GetReadTimes());

    storage.GetMemory().ResetAccessCounter();

    // 一部足りない領域
    NNT_ASSERT_RESULT_SUCCESS(storage.Get().Read(0, ptr1, TestSize - 1));
    ASSERT_EQ(1, storage.GetMemory().GetReadTimes());
    NNT_ASSERT_RESULT_SUCCESS(storage.Get().Read(0, ptr1, TestSize - 2));
    ASSERT_EQ(2, storage.GetMemory().GetReadTimes());

    storage.GetMemory().ResetAccessCounter();

    // 領域のかぶるデータ
    NNT_ASSERT_RESULT_SUCCESS(storage.Get().Read(1010, ptr1, 20));
    NNT_ASSERT_RESULT_SUCCESS(storage.Get().Read(1000, ptr1, 20));
    NNT_ASSERT_RESULT_SUCCESS(storage.Get().Read(1030, ptr1, 20));

    // カウンタの確認
    ASSERT_GE(2, storage.GetMemory().GetReadTimes());

    storage.GetMemory().ResetAccessCounter();

    // 抱合する領域
    NNT_ASSERT_RESULT_SUCCESS(storage.Get().Read(0, ptr1, TestSize));
    NNT_ASSERT_RESULT_SUCCESS(storage.Get().Read(0, ptr1, TestSize));
    NNT_ASSERT_RESULT_SUCCESS(storage.Get().Read(0, ptr1, TestSize));
    NNT_ASSERT_RESULT_SUCCESS(storage.Get().Read(1010, ptr1, 20));

    // カウンタの確認
    ASSERT_TRUE((3 == storage.GetMemory().GetReadTimes()) ||
                (4 == storage.GetMemory().GetReadTimes()));

    storage.GetMemory().ResetAccessCounter();

    // 抱合する領域
    NNT_ASSERT_RESULT_SUCCESS(storage.Get().Read(0, ptr1, 8 * 1024));
    NNT_ASSERT_RESULT_SUCCESS(storage.Get().Read(1010, ptr1, 20));

    // カウンタの確認
    ASSERT_TRUE((1 == storage.GetMemory().GetReadTimes()) ||
                (2 == storage.GetMemory().GetReadTimes()));
}

// 書き込み時のメモリアクセスのテスト
TEST_P(FsBlockCacheBufferedStorageTest, Write)
{
    static const size_t TestSize = 128 * 1024;
    static const size_t VerificationBlockSize = 512;
    static const int CacheCountMax = 1024;
    static const size_t CacheBlockSize = 4 * 1024;
    static const size_t CacheBufferSize = 8 * TestSize;

    const bool isRealData = GetParam();
    nnt::fs::util::Vector<char> buf1(TestSize);
    char* ptr1 = &buf1[0];

    // テストデータ設定
    for( size_t i = 0; i < TestSize; i++ )
    {
        ptr1[i] = i & 0xFF;
    }

    StorageWrapper<nnt::fs::util::AccessCountedMemoryStorage>
        storage(CacheCountMax, CacheBufferSize, CacheBlockSize, nn::fs::StorageType::StorageType_SaveData);

    // ストレージの初期化
    NNT_ASSERT_RESULT_SUCCESS(
        storage.Initialize(
            TestSize,
            VerificationBlockSize,
            isRealData,
            true
        )
    );

    storage.GetMemory().ResetAccessCounter();

    // ストレージ全体
    NNT_ASSERT_RESULT_SUCCESS(storage.Get().Write(0, ptr1, TestSize));
    ASSERT_EQ(0, storage.GetMemory().GetReadTimes());
    ASSERT_EQ(1, storage.GetMemory().GetWriteTimes());
    ASSERT_EQ(0, storage.GetMemory().GetFlushTimes());

    storage.GetMemory().ResetAccessCounter();

    // ストレージ全体（読み書き）
    NNT_ASSERT_RESULT_SUCCESS(storage.Get().Read(0, ptr1, TestSize));
    NNT_ASSERT_RESULT_SUCCESS(storage.Get().Write(0, ptr1, TestSize));
    ASSERT_EQ(1, storage.GetMemory().GetReadTimes());
    ASSERT_EQ(1, storage.GetMemory().GetWriteTimes());
    ASSERT_EQ(0, storage.GetMemory().GetFlushTimes());

    storage.GetMemory().ResetAccessCounter();

    // ストレージ全体（フラッシュ）
    NNT_ASSERT_RESULT_SUCCESS(storage.Get().Flush());
    ASSERT_EQ(0, storage.GetMemory().GetReadTimes());
    ASSERT_EQ(0, storage.GetMemory().GetWriteTimes());
    ASSERT_EQ(1, storage.GetMemory().GetFlushTimes());

    storage.GetMemory().ResetAccessCounter();

    // 細切れの領域（読み書き）
    NNT_ASSERT_RESULT_SUCCESS(storage.Get().Read(0, ptr1, VerificationBlockSize * 1));
    NNT_ASSERT_RESULT_SUCCESS(storage.Get().Write(0, ptr1, VerificationBlockSize * 1));
    ASSERT_EQ(1, storage.GetMemory().GetReadTimes());
    ASSERT_EQ(0, storage.GetMemory().GetWriteTimes());
    ASSERT_EQ(0, storage.GetMemory().GetFlushTimes());

    // ずれた細切れの領域（読み書き）
    NNT_ASSERT_RESULT_SUCCESS(storage.Get().Read(1, ptr1, VerificationBlockSize * 1));
    NNT_ASSERT_RESULT_SUCCESS(storage.Get().Write(1, ptr1, VerificationBlockSize * 1));
    ASSERT_EQ(2, storage.GetMemory().GetReadTimes());
    ASSERT_EQ(0, storage.GetMemory().GetWriteTimes()); // ライトキャッシュ対象
    ASSERT_EQ(0, storage.GetMemory().GetFlushTimes()); // ライトキャッシュ対象

    // 連続書き込み（キャッシュ範囲になし）
    NNT_ASSERT_RESULT_SUCCESS(
        storage.Get().Write(
            VerificationBlockSize * 2,
            ptr1,
            TestSize - VerificationBlockSize * 2
        )
    );
    ASSERT_EQ(2, storage.GetMemory().GetReadTimes());
    ASSERT_EQ(1, storage.GetMemory().GetWriteTimes());
    ASSERT_EQ(0, storage.GetMemory().GetFlushTimes());

    // 連続書き込み（キャッシュ範囲に１エントリー存在）
    NNT_ASSERT_RESULT_SUCCESS(
        storage.Get().Write(
            VerificationBlockSize * 1,
            ptr1,
            TestSize - VerificationBlockSize * 1
        )
    );
    ASSERT_EQ(2, storage.GetMemory().GetReadTimes());
    ASSERT_EQ(3, storage.GetMemory().GetWriteTimes());
    ASSERT_EQ(0, storage.GetMemory().GetFlushTimes());

    storage.GetMemory().ResetAccessCounter();

    // ずれた細切れの領域
    NNT_ASSERT_RESULT_SUCCESS(storage.Get().Write(1, ptr1, VerificationBlockSize * 1));
    ASSERT_EQ(1, storage.GetMemory().GetReadTimes());
    ASSERT_EQ(0, storage.GetMemory().GetWriteTimes());

    // ストレージ全体
    NNT_ASSERT_RESULT_SUCCESS(
        storage.Get().Write(
            VerificationBlockSize * 0,
            ptr1,
            TestSize - VerificationBlockSize * 0
        )
    );
    ASSERT_EQ(1, storage.GetMemory().GetReadTimes());
    ASSERT_EQ(3, storage.GetMemory().GetWriteTimes());
    ASSERT_EQ(0, storage.GetMemory().GetFlushTimes());

    storage.GetMemory().ResetAccessCounter();

    // 書き込んだまま閉じる
    NNT_ASSERT_RESULT_SUCCESS(storage.Get().Write(0, ptr1, 1));
    ASSERT_EQ(1, storage.GetMemory().GetReadTimes());
    ASSERT_EQ(0, storage.GetMemory().GetWriteTimes());
    ASSERT_EQ(0, storage.GetMemory().GetFlushTimes());
}

// ゼロフィルのテスト
TEST_F(FsBlockCacheBufferedStorageTest, FillZero)
{
    static const size_t TestSize = 512 * 1024;
    static const int CacheCountMax = 1024;
    static const size_t CacheBlockSize = 4 * 1024;
    static const size_t CacheBufferSize = 1024 * 1024;

    bool isRealData = true;
    bool isKeepsBurstMode = true;
    std::mt19937 mt(nnt::fs::util::GetRandomSeed());
    nnt::fs::util::Vector<char> bufRead(TestSize, 0);
    nnt::fs::util::Vector<char> bufWrite(TestSize, -1);

    for( size_t sizeMax = 32; sizeMax <= TestSize; sizeMax *= 2 )
    {
        NN_SDK_LOG(".");

        StorageWrapper<nnt::fs::util::SafeMemoryStorage> storage(
            CacheCountMax, CacheBufferSize, CacheBlockSize, nn::fs::StorageType::StorageType_SaveData);

        NNT_ASSERT_RESULT_SUCCESS(storage.Initialize(sizeMax, 32, isRealData, isKeepsBurstMode));

        nn::Result result;
        // ゼロ埋めしてから読み込めば全てゼロが読み込めるはず
        result = storage.Get().OperateRange(
            nn::fs::OperationId::FillZero, 0, sizeMax);
        NNT_ASSERT_RESULT_SUCCESS(result);
        result = storage.Get().Read(0, bufRead.data(), sizeMax);
        NNT_ASSERT_RESULT_SUCCESS(result);
        for( size_t index = 0; index < sizeMax; ++index )
        {
            EXPECT_EQ(0, bufRead[index]);
        }

        // 書き込んでから読み込めば書いた値が読めるはず
        result = storage.Get().Write(0, bufWrite.data(), sizeMax);
        NNT_ASSERT_RESULT_SUCCESS(result);
        result = storage.Get().Read(0, bufRead.data(), sizeMax);
        NNT_ASSERT_RESULT_SUCCESS(result);
        for( size_t index = 0; index < sizeMax; ++index )
        {
            EXPECT_EQ(static_cast<char>(-1), bufRead[index]);
        }

        // 一部をゼロ埋めすればそこからはゼロが、それ以外からはもともと書かれていた値が読み込めるはず
        {
            const size_t offsetFill =
                std::uniform_int_distribution<size_t>(0, sizeMax - 1)(mt);
            const size_t sizeFill =
                std::uniform_int_distribution<size_t>(0, sizeMax - offsetFill)(mt);

            result = storage.Get().OperateRange(
                nn::fs::OperationId::FillZero, offsetFill, sizeFill);
            NNT_ASSERT_RESULT_SUCCESS(result);
            result = storage.Get().Read(0, bufRead.data(), sizeMax);
            NNT_ASSERT_RESULT_SUCCESS(result);

            for( size_t index = 0; index < sizeMax; ++index )
            {
                if( offsetFill <= index && index < offsetFill + sizeFill )
                {
                    EXPECT_EQ(0, bufRead[index]);
                }
                else
                {
                    EXPECT_EQ(static_cast<char>(-1), bufRead[index]);
                }
            }
        }

        // 完全に範囲外のゼロ埋め要求は失敗するはず
        result = storage.Get().OperateRange(
            nn::fs::OperationId::FillZero, sizeMax + 1, 1);
        NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultInvalidOffset, result);
        result = storage.Get().OperateRange(
            nn::fs::OperationId::FillZero, -1, 1);
        NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultInvalidOffset, result);

        // 部分的に範囲外のゼロ埋め要求は成功するはず
        {
            const size_t offsetFill =
                std::uniform_int_distribution<size_t>(0, sizeMax - 1)(mt);
            const size_t sizeFill =
                sizeMax - offsetFill + std::uniform_int_distribution<size_t>(1, 256)(mt);

            result = storage.Get().Write(0, bufWrite.data(), sizeMax);
            NNT_ASSERT_RESULT_SUCCESS(result);
            result = storage.Get().OperateRange(
                nn::fs::OperationId::FillZero, offsetFill, sizeFill);
            NNT_ASSERT_RESULT_SUCCESS(result);
            result = storage.Get().Read(0, bufRead.data(), sizeMax);
            NNT_ASSERT_RESULT_SUCCESS(result);

            for( size_t index = 0; index < sizeMax; ++index )
            {
                if( offsetFill <= index )
                {
                    EXPECT_EQ(0, bufRead[index]);
                }
                else
                {
                    EXPECT_EQ(static_cast<char>(-1), bufRead[index]);
                }
            }
        }

        isRealData = (isRealData && isKeepsBurstMode) || (!isRealData && !isKeepsBurstMode);
        isKeepsBurstMode = !isKeepsBurstMode;
    }

    NN_SDK_LOG("\n");
}

// メモリ破壊を監視するストレージです。
class DestroyableMemoryStorage : public nnt::fs::util::SafeMemoryStorage
{
public:
    DestroyableMemoryStorage(int64_t sizeStorage, size_t sizeBlock) NN_NOEXCEPT
        : nnt::fs::util::SafeMemoryStorage()
        , m_Destroyed(static_cast<size_t>(sizeStorage / sizeBlock), false)
        , m_SizeBlock(sizeBlock)
    {
        NN_SDK_REQUIRES_ALIGNED(sizeStorage, sizeBlock);
    }

    virtual nn::Result Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_REQUIRES_ALIGNED(offset, m_SizeBlock);
        NN_SDK_REQUIRES_ALIGNED(size, m_SizeBlock);
        for( auto index = offset / m_SizeBlock; index < (offset + size) / m_SizeBlock; ++index )
        {
            if( m_Destroyed[static_cast<size_t>(index)] )
            {
                return nn::fs::ResultIntegrityVerificationStorageCorrupted();
            }
        }
        return nnt::fs::util::SafeMemoryStorage::Read(offset, buffer, size);
    }

    virtual nn::Result Write(
        int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_REQUIRES_ALIGNED(offset, m_SizeBlock);
        NN_SDK_REQUIRES_ALIGNED(size, m_SizeBlock);
        for( auto index = offset / m_SizeBlock; index < (offset + size) / m_SizeBlock; ++index )
        {
            m_Destroyed[static_cast<size_t>(index)] = false;
        }
        return nnt::fs::util::SafeMemoryStorage::Write(offset, buffer, size);
    }

    virtual nn::Result OperateRange(
        void* outBuffer,
        size_t outBufferSize,
        nn::fs::OperationId operationId,
        int64_t offset,
        int64_t size,
        const void* inBuffer,
        size_t inBufferSize) NN_NOEXCEPT NN_OVERRIDE
    {
        switch( operationId )
        {
        case nn::fs::OperationId::DestroySignature:
            {
                NN_SDK_REQUIRES_ALIGNED(offset, m_SizeBlock);
                NN_SDK_REQUIRES_ALIGNED(size, m_SizeBlock);
                auto limit = (offset + size) / m_SizeBlock;
                for( auto index = offset / m_SizeBlock; index < limit; ++index )
                {
                    m_Destroyed[static_cast<size_t>(index)] = true;
                }
                NN_RESULT_SUCCESS;
            }
            break;

        default:
            return nnt::fs::util::SafeMemoryStorage::OperateRange(
                outBuffer, outBufferSize, operationId, offset, size, inBuffer, inBufferSize);
        }
    }

private:
    nnt::fs::util::Vector<bool> m_Destroyed;
    size_t m_SizeBlock;
};

// 署名の破壊テスト
TEST_F(FsBlockCacheBufferedStorageTest, DestroySignature)
{
    static const size_t TestSize = 512 * 1024;
    static const size_t CacheBufferSize = 1024 * 1024;
    static const int MaxCacheCount = 1024;
    static const size_t CacheBlockSize = 4 * 1024;
    static const size_t VerificationBlockSize = 32;

    bool isRealData = true;
    bool isKeepsBurstMode = true;
    std::mt19937 mt(nnt::fs::util::GetRandomSeed());
    nnt::fs::util::Vector<char> bufWrite(TestSize, -1);

    for( size_t sizeMax = 32; sizeMax <= TestSize; sizeMax *= 2 )
    {
        NN_SDK_LOG(".");

        const size_t countBlock = sizeMax / VerificationBlockSize;

        StorageWrapper<DestroyableMemoryStorage> storage(
            MaxCacheCount, CacheBufferSize, CacheBlockSize, sizeMax, VerificationBlockSize, nn::fs::StorageType::StorageType_SaveData);

        NNT_ASSERT_RESULT_SUCCESS(
            storage.Initialize(sizeMax, VerificationBlockSize, isRealData, isKeepsBurstMode));

        nn::Result result;

        // storage からの読み込みが一度でも検証エラーになったかどうか
        bool isSignatureVerificationFailed = false;

        // 署名を破壊した領域の読み込みはエラーになるが、それ以外の領域の読み込みは成功するはず
        // ただし、一度でも破壊した領域を読むとそれ以降はどこを読んでも同じエラーが返ります。
        {
            const size_t offsetDestroy =
                std::uniform_int_distribution<size_t>(0, sizeMax - 1)(mt);
            const size_t sizeDestroy =
                std::uniform_int_distribution<size_t>(0, sizeMax - offsetDestroy)(mt);

            result = storage.Get().Write(0, bufWrite.data(), sizeMax);
            NNT_ASSERT_RESULT_SUCCESS(result);
            result = storage.Get().OperateRange(
                nn::fs::OperationId::DestroySignature,
                offsetDestroy,
                sizeDestroy
            );
            NNT_ASSERT_RESULT_SUCCESS(result);

            char bufRead[VerificationBlockSize] = {};
            for( size_t index = 0; index < countBlock; ++index )
            {
                result = storage.Get().Read(
                    index * VerificationBlockSize,
                    bufRead,
                    VerificationBlockSize
                );
                if ( isSignatureVerificationFailed )
                {
                    // 一度でも破壊した領域を読むとそれ以降はどこを読んでも同じエラーが返ります。
                    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultIntegrityVerificationStorageCorrupted, result);
                }
                else if( index * VerificationBlockSize < offsetDestroy ||
                    offsetDestroy + sizeDestroy < (index + 1) * VerificationBlockSize )
                {
                    NNT_EXPECT_RESULT_SUCCESS(result);
                }
                else
                {
                    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultIntegrityVerificationStorageCorrupted, result);
                    isSignatureVerificationFailed = true;
                }
            }
        }
    }

    for( size_t sizeMax = 32; sizeMax <= TestSize; sizeMax *= 2 )
    {
        NN_SDK_LOG(".");

        const size_t countBlock = sizeMax / VerificationBlockSize;

        StorageWrapper<DestroyableMemoryStorage> storage(
            MaxCacheCount, CacheBufferSize, CacheBlockSize, sizeMax, VerificationBlockSize, nn::fs::StorageType::StorageType_SaveData);

        NNT_ASSERT_RESULT_SUCCESS(
            storage.Initialize(sizeMax, VerificationBlockSize, isRealData, isKeepsBurstMode));

        nn::Result result;

        // storage からの読み込みが一度でも検証エラーになったかどうか
        bool isSignatureVerificationFailed = false;

        // 完全に範囲外の署名破壊要求は失敗するはず
        result = storage.Get().OperateRange(
            nn::fs::OperationId::DestroySignature, sizeMax + 1, 1);
        NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultInvalidOffset, result);
        result = storage.Get().OperateRange(
            nn::fs::OperationId::DestroySignature, -1, 1);
        NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultInvalidOffset, result);

        // 部分的に範囲外の署名破壊要求は成功するはず
        {
            const size_t offsetDestroy =
                std::uniform_int_distribution<size_t>(0, sizeMax - 1)(mt);
            const size_t sizeDestroy =
                sizeMax - offsetDestroy + std::uniform_int_distribution<size_t>(1, 256)(mt);

            result = storage.Get().Write(0, bufWrite.data(), sizeMax);
            NNT_ASSERT_RESULT_SUCCESS(result);
            result = storage.Get().OperateRange(
                nn::fs::OperationId::DestroySignature,
                offsetDestroy,
                sizeDestroy
            );
            NNT_ASSERT_RESULT_SUCCESS(result);

            char bufRead[VerificationBlockSize] = {};
            for( size_t index = 0; index < countBlock; ++index )
            {
                result = storage.Get().Read(
                    index * VerificationBlockSize,
                    bufRead,
                    VerificationBlockSize
                );

                if ( isSignatureVerificationFailed )
                {
                    // 一度でも破壊した領域を読むとそれ以降はどこを読んでも同じエラーが返ります。
                    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultIntegrityVerificationStorageCorrupted, result);
                }
                else if( index * VerificationBlockSize < offsetDestroy )
                {
                    NNT_EXPECT_RESULT_SUCCESS(result);
                }
                else
                {
                    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultIntegrityVerificationStorageCorrupted, result);
                    isSignatureVerificationFailed = true;
                }
            }
        }

        isRealData = (isRealData && isKeepsBurstMode) || (!isRealData && !isKeepsBurstMode);
        isKeepsBurstMode = !isKeepsBurstMode;
    }

    NN_SDK_LOG("\n");
} // NOLINT(impl/function_size)

// 不明な操作のテスト
TEST_F(FsBlockCacheBufferedStorageTest, UnknownOperation)
{
    static const size_t TestSize = 512 * 1024;
    static const size_t CacheBufferSize = 1024 * 1024;
    static const int CacheCountMax = 1024;
    static const size_t CacheBlockSize = 4 * 1024;
    static const size_t VerificationBlockSize = 32;

    bool isRealData = true;
    bool isKeepsBurstMode = true;
    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    for( size_t sizeMax = 32; sizeMax <= TestSize; sizeMax *= 2 )
    {
        NN_SDK_LOG(".");

        StorageWrapper<nnt::fs::util::SafeMemoryStorage> storage(
            CacheCountMax, CacheBufferSize, CacheBlockSize, nn::fs::StorageType::StorageType_SaveData);

        NNT_ASSERT_RESULT_SUCCESS(
            storage.Initialize(sizeMax, VerificationBlockSize, isRealData, isKeepsBurstMode));

        // 不明な操作は失敗するはず
        const size_t offsetOperate =
            std::uniform_int_distribution<size_t>(0, sizeMax - 1)(mt);
        const size_t sizeOperate =
            std::uniform_int_distribution<size_t>(1, sizeMax - offsetOperate)(mt);

        auto result = storage.Get().OperateRange(
            static_cast<nn::fs::OperationId>(-1),
            offsetOperate,
            sizeOperate
        );
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultUnsupportedOperation, result);

        isRealData = (isRealData && isKeepsBurstMode) || (!isRealData && !isKeepsBurstMode);
        isKeepsBurstMode = !isKeepsBurstMode;
    }

    NN_SDK_LOG("\n");
}

// アライメントのずれたシーケンシャルアクセスのテスト
TEST(FsBlockCacheBufferedStorageAccessTest, NotAlignedSequentialAccess)
{
    static const size_t TestSize = 512 * 1024;
    static const size_t CacheBufferSize = 1024 * 1024;
    static const int CacheCountMax = 1024;
    static const size_t CacheBlockSize = 4 * 1024;
    static const size_t VerificationBlockSize = 16 * 1024;

    auto cacheBuffer = nnt::fs::util::AllocateBuffer(CacheBufferSize);

    // BlockCacheBufferedStorage にアライメントの揃っていないオフセットで
    // 16KB 単位のシーケンシャルアクセスをすると
    // SequentialTestStorage に 32KB, 16KB, 16KB, ...
    // というシーケンシャルアクセスが発生することをテスト
    //
    // 具体的には
    // BlockCacheBufferedStorage に offset = 1, size = 16KB のアクセス
    // → SequentialTestStorage に offset = 0, size = 32KB のアクセス
    //
    // BlockCacheBufferedStorage に offset = 16KB + 1, size = 16KB のアクセス
    // → SequentialTestStorage に offset = 32KB, size = 16KB のアクセス
    // (16KB ～ 32KB まではキャッシュされているからアクセスされないはず)
    class SequentialTestStorage : public nn::fs::IStorage
    {
    public:
        SequentialTestStorage()
            : m_PreviousEnd(-1)
        {
        }

        virtual nn::Result Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_ABORT_UNLESS_NOT_NULL(buffer);

            TestSequentialAccess(offset, size);

            memset(buffer, 0, size);
            m_PreviousEnd = offset + size;
            NN_RESULT_SUCCESS;
        }

        void TestSequentialAccess(int64_t offset, size_t size) const NN_NOEXCEPT
        {
            if( offset == 0 )
            {
                // アライメントのずれたアクセスをするので 2 ブロックの読み込みになる
                EXPECT_EQ(VerificationBlockSize * 2, size)
                    << "offset: " << offset << " size: " << size;
            }
            else
            {
                EXPECT_EQ(VerificationBlockSize, size)
                    << "offset: " << offset << " size: " << size;

                // シーケンシャルアクセスになっているか
                EXPECT_EQ(m_PreviousEnd, offset);
            }
        }

        virtual nn::Result Flush() NN_NOEXCEPT NN_OVERRIDE
        {
            NN_RESULT_SUCCESS;
        }

        virtual nn::Result GetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_ABORT_UNLESS_NOT_NULL(outValue);
            *outValue = TestSize;
            NN_RESULT_SUCCESS;
        }

    private:
        int64_t m_PreviousEnd;
    };
    SequentialTestStorage testStorage;

    nn::fssystem::FileSystemBufferManager bufferManager;
    NNT_ASSERT_RESULT_SUCCESS(
        bufferManager.Initialize(
            CacheCountMax,
            reinterpret_cast<uintptr_t>(cacheBuffer.get()),
            CacheBufferSize,
            CacheBlockSize));

    nn::os::Mutex mutex(true);

    nn::fssystem::save::BlockCacheBufferedStorage storage;
    NNT_ASSERT_RESULT_SUCCESS(
        storage.Initialize(
            &bufferManager,
            &mutex,
            &testStorage,
            TestSize,
            VerificationBlockSize,
            storage.DefaultMaxCacheEntryCount,
            true,
            0,
            true,
            nn::fs::StorageType_SaveData));

    const size_t ReadSize = VerificationBlockSize;
    auto readBuffer = nnt::fs::util::AllocateBuffer(ReadSize);

    for( int64_t offset = 1; offset + ReadSize < TestSize; offset += ReadSize )
    {
        NNT_ASSERT_RESULT_SUCCESS(
            storage.Read(offset, readBuffer.get(), ReadSize));
    }
}

// ロールバックのテスト
TEST_F(FsBlockCacheBufferedStorageTest, Rollback)
{
    static const size_t TestSize = 512 * 1024;
    static const size_t VerificationBlockSize = 32;
    static const int CacheCountMax = 1024;
    static const size_t CacheBlockSize = 4 * 1024;
    static const size_t CacheBufferSize = 1024 * 1024;

    std::unique_ptr<char[]> writeBuffer(new char[TestSize]);

    std::memset(writeBuffer.get(), 0xFE, TestSize);

    StorageWrapper<nnt::fs::util::AccessCountedMemoryStorage>
        storage(CacheCountMax, CacheBufferSize, CacheBlockSize, nn::fs::StorageType::StorageType_SaveData);

    // ストレージの初期化
    NNT_ASSERT_RESULT_SUCCESS(storage.Initialize(TestSize, VerificationBlockSize, false, false));

    size_t BufferSizeBeforeTest = storage.GetBufferAlocatableSize();

    // 書き込んだ後にフラッシュを行う
    NNT_ASSERT_RESULT_SUCCESS(storage.Get().Write(0, writeBuffer.get(), TestSize));
    NNT_ASSERT_RESULT_SUCCESS(storage.Get().Flush());

    // 1 バイトずつ書き込んだ後にフラッシュを行わずロールバックを行う
    std::mt19937 rng(nnt::fs::util::GetRandomSeed());
    for( auto count = 0; count < 100; ++count )
    {
        const auto offset = std::uniform_int_distribution<int64_t>(1, TestSize - 1)(rng);
        char data = '\0';
        NNT_ASSERT_RESULT_SUCCESS(storage.Get().Write(offset, &data, 1));
    }
    NNT_ASSERT_RESULT_SUCCESS(storage.Get().OnRollback());

    // バッファがリークしていないことを確かめる
    size_t BufferSizeAfterTest = storage.GetBufferAlocatableSize();
    ASSERT_EQ(BufferSizeBeforeTest, BufferSizeAfterTest);
}

// 一括読み込みのテスト
TEST_P(FsBlockCacheBufferedStorageTest, BulkRead)
{
    static const size_t TestSize = 4 * 1024 * 1024;
    static const size_t VerificationBlockSize = 512;
    static const int CacheCountMax = 1024;
    static const size_t CacheBlockSize = 512;
    static const size_t CacheBufferSize = 64 * 1024;
    static const int BulkReadSizeMax = 2 * 1024 * 1024;

    const bool isRealData = GetParam();
    nnt::fs::util::Vector<char> writeBuffer(TestSize);
    nnt::fs::util::Vector<char> readBuffer(TestSize);
    auto readBufferPtr = &readBuffer[0];

    // テストデータ設定
    for( size_t i = 0; i < TestSize; i++ )
    {
        writeBuffer[i] = i & 0xFF;
    }

    StorageWrapper<nnt::fs::util::AccessCountedMemoryStorage>
        storage(CacheCountMax, CacheBufferSize, CacheBlockSize, nn::fs::StorageType::StorageType_SaveData);

    // ストレージの初期化
    NNT_ASSERT_RESULT_SUCCESS(
        storage.Initialize(
            TestSize,
            VerificationBlockSize,
            isRealData,
            false
        )
    );

    NNT_ASSERT_RESULT_SUCCESS(storage.Get().Write(0, &writeBuffer[0], TestSize));

    storage.GetMemory().ResetAccessCounter();

    // 一括読み込みの最大サイズ
    NNT_ASSERT_RESULT_SUCCESS(storage.Get().Read(1, readBufferPtr, BulkReadSizeMax - 1));
    NNT_FS_UTIL_ASSERT_MEMCMPEQ(&writeBuffer[1], readBufferPtr, BulkReadSizeMax - 1);
    EXPECT_EQ(1, storage.GetMemory().GetReadTimes());

    storage.GetMemory().ResetAccessCounter();

    // 一括読み込みの最大サイズ超
    NNT_ASSERT_RESULT_SUCCESS(storage.Get().Read(1, readBufferPtr, BulkReadSizeMax));
    EXPECT_LT(1, storage.GetMemory().GetReadTimes());

    storage.GetMemory().ResetAccessCounter();

    // アライメントが揃っていれば一括読み込みされない
    NN_STATIC_ASSERT(CacheBlockSize * 1024 <= BulkReadSizeMax);
    NNT_ASSERT_RESULT_SUCCESS(storage.Get().Read(0, readBufferPtr, CacheBlockSize * 1024));
    EXPECT_LT(1, storage.GetMemory().GetReadTimes());

    storage.GetMemory().ResetAccessCounter();

    // キャッシュに載せるために読み込み
    const auto maxCacheSize
        = nn::fssystem::save::BlockCacheBufferedStorage::DefaultMaxCacheEntryCount * CacheBlockSize;
    for( int offset = 0; offset < maxCacheSize; offset += CacheBlockSize )
    {
        NNT_ASSERT_RESULT_SUCCESS(storage.Get().Read(offset, readBufferPtr, CacheBlockSize));
    }

    storage.GetMemory().ResetAccessCounter();

    // キャッシュがあればそれを利用
    NNT_ASSERT_RESULT_SUCCESS(storage.Get().Read(1, readBufferPtr, maxCacheSize - CacheBlockSize));
    NNT_FS_UTIL_ASSERT_MEMCMPEQ(&writeBuffer[1], readBufferPtr, maxCacheSize - CacheBlockSize);
    EXPECT_EQ(0, storage.GetMemory().GetReadTimes());

    storage.GetMemory().ResetAccessCounter();
}

namespace {

const int ThreadStackSize = 16 * 1024;
const int ThreadCount = 4;
NN_OS_ALIGNAS_THREAD_STACK char g_ThreadStack[ThreadStackSize * ThreadCount];

void ThreadFunc(void* pArg) NN_NOEXCEPT
{
    auto pStorage = reinterpret_cast<nn::fs::IStorage*>(pArg);
    auto offset = 1;
    static const int BufferSize = 256;
    char buffer[BufferSize];
    NNT_ASSERT_RESULT_SUCCESS(pStorage->Read(offset, buffer, BufferSize));

    for( int i = 0; i < BufferSize; ++i )
    {
        EXPECT_EQ(static_cast<char>((offset + i) & 0xFF), buffer[i]);
    }
};

// 複数スレッドによる読み込みのテスト
TEST_P(FsBlockCacheBufferedStorageTest, MultiRead)
{
    static const size_t TestSize = 1024;
    static const size_t VerificationBlockSize = 512;
    static const int CacheCountMax = 1024;
    static const size_t CacheBlockSize = 512;
    static const size_t CacheBufferSize = 64 * 1024;

    const bool isRealData = GetParam();

    // テストデータ設定
    nnt::fs::util::Vector<char> writeBuffer(TestSize);
    for( size_t i = 0; i < TestSize; i++ )
    {
        writeBuffer[i] = i & 0xFF;
    }

    const auto LoopMax = 1024;
    for( int loop = 0; loop < LoopMax; ++loop )
    {
        StorageWrapper<nnt::fs::util::AccessCountedMemoryStorage>
            storage(CacheCountMax, CacheBufferSize, CacheBlockSize, nn::fs::StorageType::StorageType_SaveData);

        // ストレージの初期化
        NNT_ASSERT_RESULT_SUCCESS(
            storage.Initialize(
                TestSize,
                VerificationBlockSize,
                isRealData,
                false
            )
        );

        // テストデータ書き込み
        NNT_ASSERT_RESULT_SUCCESS(storage.GetMemory().Write(0, &writeBuffer[0], TestSize));

        nn::os::ThreadType threadArray[ThreadCount];

        for( int i = 0; i < ThreadCount; ++i )
        {
            NNT_ASSERT_RESULT_SUCCESS(
                nn::os::CreateThread(
                    &threadArray[i],
                    ThreadFunc,
                    &storage.Get(),
                    &g_ThreadStack[i * ThreadStackSize],
                    ThreadStackSize,
                    nn::os::DefaultThreadPriority
                )
            );
        }

        for( auto& thread : threadArray )
        {
            nn::os::StartThread(&thread);
        }

        for( auto& thread : threadArray )
        {
            nn::os::WaitThread(&thread);
            nn::os::DestroyThread(&thread);
        }
    }
}

// QueryRange のテスト
TEST_P(FsBlockCacheBufferedStorageTest, QueryRange)
{
    static const size_t TestSize = 4 * 1024 * 1024;
    static const size_t VerificationBlockSize = 512;
    static const int CacheCountMax = 1024;
    static const size_t CacheBlockSize = 512;
    static const size_t CacheBufferSize = 64 * 1024;

    const bool isRealData = GetParam();

    StorageWrapper<nnt::fs::util::AccessCountedMemoryStorage>
        storage(CacheCountMax, CacheBufferSize, CacheBlockSize, nn::fs::StorageType::StorageType_SaveData);

    // ストレージの初期化
    NNT_ASSERT_RESULT_SUCCESS(
        storage.Initialize(
            TestSize,
            VerificationBlockSize,
            isRealData,
            false
        )
    );

    nn::fs::QueryRangeInfo info;

    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultNullptrArgument,
        storage.Get().OperateRange(nullptr, sizeof(info), nn::fs::OperationId::QueryRange, 0, static_cast<int64_t>(TestSize), nullptr, 0));
    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultInvalidSize,
        storage.Get().OperateRange(&info, 0, nn::fs::OperationId::QueryRange, 0, static_cast<int64_t>(TestSize), nullptr, 0));

    NNT_ASSERT_RESULT_SUCCESS(storage.Get().OperateRange(
        &info, sizeof(info), nn::fs::OperationId::QueryRange, 0, static_cast<int64_t>(TestSize), nullptr, 0));
    EXPECT_EQ(0, info.aesCtrKeyTypeFlag);
    EXPECT_EQ(0, info.speedEmulationTypeFlag);
}

}

INSTANTIATE_TEST_CASE_P(
    FsBlockCacheBufferedStorageTest,
    FsBlockCacheBufferedStorageTest,
    ::testing::Values(true, false)
);
