﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <array>
#include <nn/nn_Log.h>
#include <nn/fs/fs_FileStorage.h>
#include <nn/fs/fsa/fs_IFile.h>
#include <nn/fssystem/fs_BucketTreeBuilder.h>
#include <nn/fssystem/fs_HostFileSystem.h>
#include <nn/fssystem/fs_PathOnExecutionDirectory.h>
#include <nn/fssystem/utilTool/fs_RelocatedIndirectStorageUtils.h>
#include <nn/fssystem/utilTool/fs_AesCtrCounterExtendedStorageBuilder.h>
#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/fsUtil/testFs_util.h>


typedef nn::fssystem::utilTool::RelocationTable RelocationTable;
typedef nn::fssystem::AesCtrCounterExtendedStorage AesCtrCounterExtendedStorage;
typedef nn::fssystem::BucketTreeBuilder BucketTreeBuilder;

namespace {

class RelocationTableTest : public ::testing::Test
{
public:
    RelocationTableTest() NN_NOEXCEPT
        : m_pFileSystem()
    {
        auto pHostFileSystem = new nn::fssystem::HostFileSystem();
        pHostFileSystem->Initialize(nn::fssystem::PathOnExecutionDirectory("").Get());

        m_pFileSystem.reset(pHostFileSystem);
    }
    virtual ~RelocationTableTest() NN_NOEXCEPT NN_OVERRIDE
    {
    }

protected:
    std::unique_ptr<nn::fs::fsa::IFileSystem> m_pFileSystem;
};

}

TEST_F(RelocationTableTest, AddressTranslation)
{
    // アドレス変換のテスト

    static const int SrcEntriesSize = 2;
    std::array<RelocationTable::Entry, SrcEntriesSize> srcEntries =
        {
            RelocationTable::Entry(50,  0, 50, true),
            RelocationTable::Entry(0, 50, 50, true),
        };

    // 書き込み
    RelocationTable table;
    {
        // テーブル作成
        for (int i = 0; i < static_cast<int>(srcEntries.size()); ++i)
        {
            table.Add(srcEntries[i]);
        }

        NNT_ASSERT_RESULT_SUCCESS(table.Commit());
    }

    auto iter = table.begin();
    EXPECT_EQ(iter->ToDstOffset(25), 75);
    ++iter;
    EXPECT_EQ(iter->ToDstOffset(75), 25);
    ++iter;
    EXPECT_EQ(iter, table.end());
}

TEST_F(RelocationTableTest, FindEntry)
{
    // 検索のテスト

    static const int SrcEntriesSize = 8;
    std::array<RelocationTable::Entry, SrcEntriesSize> srcEntries =
        {
            RelocationTable::Entry(70,  0, 10, true),
            RelocationTable::Entry(60, 10, 10, true),
            RelocationTable::Entry(50, 20, 10, true),
            RelocationTable::Entry(40, 30, 10, true),
            RelocationTable::Entry(30, 40, 10, false),
            RelocationTable::Entry(20, 50, 10, false),
            RelocationTable::Entry(10, 60, 10, false),
            RelocationTable::Entry( 0, 70, 10, false),
        };

    // RelocationTable 作成
    RelocationTable table;
    {
        // テーブル作成
        for (int i = 0; i < static_cast<int>(srcEntries.size()); ++i)
        {
            table.Add(srcEntries[i]);
        }

        NNT_ASSERT_RESULT_SUCCESS(table.Commit());
    }

    // 各エッジについてチェック
    EXPECT_EQ(table.Find(30)->GetSrcOffset(), 30);
    EXPECT_EQ(table.Find(35)->GetSrcOffset(), 30);
    EXPECT_EQ(table.Find(39)->GetSrcOffset(), 30);
    EXPECT_EQ(table.Find(40)->GetSrcOffset(), 40);

    // 範囲外
    EXPECT_EQ(table.Find(-1), table.end());
    EXPECT_EQ(table.Find(0)->GetSrcOffset(), 0);
    EXPECT_EQ(table.Find(80), table.end());
    EXPECT_EQ(table.Find(79)->GetSrcOffset(), 70);
}

TEST_F(RelocationTableTest, RelocateRange)
{
    // 範囲にRelocationTableを適用するテスト

    static const int SrcEntriesSize = 8;
    std::array<RelocationTable::Entry, SrcEntriesSize> srcEntries =
        {
            RelocationTable::Entry(70,  0, 10, true),
            RelocationTable::Entry(60, 10, 10, true),
            RelocationTable::Entry(50, 20, 10, true),
            RelocationTable::Entry(40, 30, 10, true),
            RelocationTable::Entry(30, 40, 10, false),
            RelocationTable::Entry(20, 50, 10, false),
            RelocationTable::Entry(10, 60, 10, false),
            RelocationTable::Entry( 0, 70, 10, false),
        };

    // RelocationTable 作成
    RelocationTable table;
    {
        // テーブル作成
        for (int i = 0; i < static_cast<int>(srcEntries.size()); ++i)
        {
            table.Add(srcEntries[i]);
        }

        NNT_ASSERT_RESULT_SUCCESS(table.Commit());
    }

    // エントリの中間
    {
        std::pair<int64_t, int64_t> expectedArray[] =
        {
            std::make_pair(75, 80),
            std::make_pair(60, 70),
            std::make_pair(50, 60),
            std::make_pair(40, 50),
            std::make_pair(30, 40),
            std::make_pair(20, 30),
            std::make_pair(10, 20),
            std::make_pair(0, 5),
        };
        std::vector<std::pair<int64_t, int64_t>> expected(
            expectedArray, expectedArray + sizeof(expectedArray) / sizeof(expectedArray[0]));

        std::vector<std::pair<int64_t, int64_t>> relocatedRange;
        NNT_ASSERT_RESULT_SUCCESS(
            table.ApplyTo(&relocatedRange, std::pair<int64_t, int64_t>(5, 75)));

        EXPECT_EQ(expected, relocatedRange);
    }

    // エントリピッタリ
    {
        std::pair<int64_t, int64_t> expectedArray[] =
        {
            std::make_pair(70, 80),
            std::make_pair(60, 70),
            std::make_pair(50, 60),
            std::make_pair(40, 50),
            std::make_pair(30, 40),
            std::make_pair(20, 30),
            std::make_pair(10, 20),
            std::make_pair(0, 10),
        };
        std::vector<std::pair<int64_t, int64_t>> expected(
            expectedArray, expectedArray + sizeof(expectedArray) / sizeof(expectedArray[0]));

        std::vector<std::pair<int64_t, int64_t>> relocatedRange;
        NNT_ASSERT_RESULT_SUCCESS(
            table.ApplyTo(&relocatedRange, std::pair<int64_t, int64_t>(0, 80)));

        EXPECT_EQ(expected, relocatedRange);
    }

    // エントリのエッジ
    {
        std::pair<int64_t, int64_t> expectedArray[] =
        {
            std::make_pair(70, 80),
            std::make_pair(60, 70),
            std::make_pair(50, 60),
            std::make_pair(40, 50),
            std::make_pair(30, 40),
            std::make_pair(20, 30),
            std::make_pair(10, 20),
            std::make_pair(0, 9),
        };
        std::vector<std::pair<int64_t, int64_t>> expected(
            expectedArray, expectedArray + sizeof(expectedArray) / sizeof(expectedArray[0]));

        std::vector<std::pair<int64_t, int64_t>> relocatedRange;
        NNT_ASSERT_RESULT_SUCCESS(
            table.ApplyTo(&relocatedRange, std::pair<int64_t, int64_t>(0, 79)));

        EXPECT_EQ(expected, relocatedRange);
    }

    // エントリの範囲外
    {
        std::vector<std::pair<int64_t, int64_t>> relocatedRange;
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultInvalidArgument,
            table.ApplyTo(&relocatedRange, std::pair<int64_t, int64_t>(0, 81)));
    }
    {
        std::vector<std::pair<int64_t, int64_t>> relocatedRange;
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultInvalidArgument,
            table.ApplyTo(&relocatedRange, std::pair<int64_t, int64_t>(-1, 80)));
    }
}

TEST_F(RelocationTableTest, RelocateDataStorage)
{
    const char* srcPath = "/src.bin";
    const char* dstPath = "/dst.bin";
    static const int FileLen = 256;

    nnt::fs::util::Vector<uint8_t> srcData(FileLen);
    for (int i = 0; i < FileLen; ++i)
    {
        srcData[i] = static_cast<uint8_t>(i);
    }

    // ファイル削除
    m_pFileSystem->DeleteFile(srcPath);
    m_pFileSystem->DeleteFile(dstPath);

    {
        // 元のファイル作成
        NNT_ASSERT_RESULT_SUCCESS(m_pFileSystem->CreateFile(srcPath, /* size =*/0, /* option =*/0));
        std::unique_ptr<nn::fs::fsa::IFile> file;
        NNT_ASSERT_RESULT_SUCCESS(
            m_pFileSystem->OpenFile(
                &file, srcPath,
                static_cast<nn::fs::OpenMode>(nn::fs::OpenMode::OpenMode_Write
                                              | nn::fs::OpenMode::OpenMode_AllowAppend)));

        NNT_ASSERT_RESULT_SUCCESS(
            file->Write(0, &srcData[0], srcData.size(),
                nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));
    }
    {
        // 先のファイル作成
        NNT_ASSERT_RESULT_SUCCESS(m_pFileSystem->CreateFile(dstPath, /* size =*/0, /* option =*/0));
        std::unique_ptr<nn::fs::fsa::IFile> file;
        NNT_ASSERT_RESULT_SUCCESS(
            m_pFileSystem->OpenFile(
                &file, dstPath,
                static_cast<nn::fs::OpenMode>(nn::fs::OpenMode::OpenMode_Write
                                              | nn::fs::OpenMode::OpenMode_AllowAppend)));
        nnt::fs::util::Vector<uint8_t> zeroes(FileLen, 0);
        NNT_ASSERT_RESULT_SUCCESS(
            file->Write(0, &zeroes[0], zeroes.size(),
                nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));
    }

    {
        // ファイルオープン
        std::unique_ptr<nn::fs::fsa::IFile> srcFile;
        NNT_ASSERT_RESULT_SUCCESS(
            m_pFileSystem->OpenFile(
                &srcFile, srcPath,
                static_cast<nn::fs::OpenMode>(nn::fs::OpenMode::OpenMode_Read)));
        std::unique_ptr<nn::fs::fsa::IFile> dstFile;
        NNT_ASSERT_RESULT_SUCCESS(
            m_pFileSystem->OpenFile(
                &dstFile, dstPath,
                static_cast<nn::fs::OpenMode>(nn::fs::OpenMode::OpenMode_Write)));

        // 前後を入れ替えるテーブル
        RelocationTable table;
        table.Add(RelocationTable::Entry(0, FileLen / 2, FileLen / 2, true));
        table.Add(RelocationTable::Entry(FileLen / 2, 0, FileLen / 2, true));
        NNT_ASSERT_RESULT_SUCCESS(table.Commit());

        {
            nnt::fs::util::Vector<uint8_t> buffer(16);

            // テーブルにより再配置を行う
            nn::fs::FileStorage srcStorage(srcFile.get());
            nn::fs::FileStorage dstStorage(dstFile.get());

            NNT_ASSERT_RESULT_SUCCESS(
                table.ApplyTo(
                    &dstStorage, &srcStorage, &buffer[0], buffer.size()));

            NNT_ASSERT_RESULT_SUCCESS(dstStorage.Flush());
        }
    }

    {
        // 前後が入れ替わっていることを確認
        std::unique_ptr<nn::fs::fsa::IFile> dstFile;
        NNT_ASSERT_RESULT_SUCCESS(
            m_pFileSystem->OpenFile(
                &dstFile, dstPath,
                static_cast<nn::fs::OpenMode>(nn::fs::OpenMode::OpenMode_Read)));
        size_t readSize;
        nnt::fs::util::Vector<uint8_t> readBuffer(FileLen);
        NNT_ASSERT_RESULT_SUCCESS(
            dstFile->Read(&readSize, 0, &readBuffer[0], readBuffer.size(),
                          nn::fs::ReadOption::MakeValue(0)));
        EXPECT_EQ(FileLen, readSize);

        NNT_FS_UTIL_EXPECT_MEMCMPEQ(&readBuffer[0], &srcData[FileLen / 2], FileLen / 2);
        NNT_FS_UTIL_EXPECT_MEMCMPEQ(&readBuffer[FileLen / 2], &srcData[0], FileLen / 2);
    }

    // ファイル削除
    NNT_ASSERT_RESULT_SUCCESS(m_pFileSystem->DeleteFile(srcPath));
    NNT_ASSERT_RESULT_SUCCESS(m_pFileSystem->DeleteFile(dstPath));
}

class RelocateAesCtrCounterExtendedTestParam
{
public:
    std::vector<AesCtrCounterExtendedStorage::Entry> expectedEntries;
    std::vector<AesCtrCounterExtendedStorage::Entry> oldEntries;
    std::vector<RelocationTable::Entry> relocationEntries;
    int64_t offsetOld;
    int64_t oldDataStorageSize;
    int64_t offsetNew;
    int64_t newDataStorageSize;
    uint32_t generationForUpdate;

    template<size_t ExpectedEntriesSize, size_t OldEntriesSize, size_t RelocationEntriesSize>
    static const RelocateAesCtrCounterExtendedTestParam Make(
        const AesCtrCounterExtendedStorage::Entry (&expectedEntries)[ExpectedEntriesSize],
        const AesCtrCounterExtendedStorage::Entry (&oldEntries)[OldEntriesSize],
        const RelocationTable::Entry (&relocationEntries)[RelocationEntriesSize],
        int64_t offsetOld,
        int64_t oldDataStorageSize,
        int64_t offsetNew,
        int64_t newDataStorageSize,
        uint32_t generationForUpdate
        ) NN_NOEXCEPT
    {
        RelocateAesCtrCounterExtendedTestParam param;
        param.expectedEntries = std::vector<AesCtrCounterExtendedStorage::Entry>(
            expectedEntries, expectedEntries + ExpectedEntriesSize);
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::fssystem::utilTool::OptimizeAesCtrCounterExtendedEntry(&param.expectedEntries, newDataStorageSize, int(generationForUpdate)));

        param.oldEntries = std::vector<AesCtrCounterExtendedStorage::Entry>(
            oldEntries, oldEntries + OldEntriesSize);
        param.relocationEntries = std::vector<RelocationTable::Entry>(
            relocationEntries, relocationEntries + RelocationEntriesSize);
        param.offsetOld = offsetOld;
        param.oldDataStorageSize = oldDataStorageSize;
        param.offsetNew = offsetNew;
        param.newDataStorageSize = newDataStorageSize;
        param.generationForUpdate = generationForUpdate;
        return param;
    }
};

nnt::fs::util::Vector<RelocateAesCtrCounterExtendedTestParam> MakeRelocateAesCtrCounterExtendedParam() NN_NOEXCEPT
{
    auto makeEntry = [](int64_t offset, uint32_t generation) NN_NOEXCEPT
        -> const AesCtrCounterExtendedStorage::Entry
        {
            AesCtrCounterExtendedStorage::Entry entry;
            entry.SetOffset(offset);
            entry.reserved = 0;
            entry.generation = generation;
            return entry;
        };


    // 共通パラメータ

    // 前バージョンのパッチのエントリー
    static const int64_t oldDataStorageSize = 300;
    const AesCtrCounterExtendedStorage::Entry oldEntries[] =
        {
            makeEntry(  0, 10),
            makeEntry( 96, 20),
            makeEntry(192, 30),
        };

    // RelocationTable
    static const int64_t newDataStorageSize = 300;
    const RelocationTable::Entry relocationEntries[] =
        {
            RelocationTable::Entry(  0,   0,  96,  true),
            RelocationTable::Entry( 96,  96,  96, false),
            RelocationTable::Entry(192, 192, 108,  true),
        };

    static const uint32_t GenerationForUpdate = 9999;

    // パラメータ定義
    nnt::fs::util::Vector<RelocateAesCtrCounterExtendedTestParam> params;

    // 前バージョンのパッチと新バージョンのパッチでオフセットのずれ無し
    {
        const int64_t offsetOld = 0;
        const int64_t offsetNew = 0;
        const AesCtrCounterExtendedStorage::Entry expectedArray[] =
            {
                makeEntry(  0, 10),
                makeEntry( 96, GenerationForUpdate),
                makeEntry(192, 30),
            };
        params.push_back(
            RelocateAesCtrCounterExtendedTestParam::Make(
                expectedArray, oldEntries, relocationEntries,
                offsetOld, oldDataStorageSize, offsetNew, newDataStorageSize,
                GenerationForUpdate));
    }

    // 前バージョンのパッチの方が後ろにあるパターン
    {
        const int64_t offsetOld = 16;
        const int64_t offsetNew = 0;
        const AesCtrCounterExtendedStorage::Entry expectedArray[] =
            {
                makeEntry(  0, GenerationForUpdate),
                makeEntry( 16, 10),
                makeEntry( 96, GenerationForUpdate),
                makeEntry(192, 20),
                makeEntry(208, 30),
            };
        params.push_back(
            RelocateAesCtrCounterExtendedTestParam::Make(
                expectedArray, oldEntries, relocationEntries,
                offsetOld, oldDataStorageSize, offsetNew, newDataStorageSize,
                GenerationForUpdate));
    }

    // 新バージョンのパッチの方が後ろにあるパターン
    {
        const int64_t offsetOld = 0;
        const int64_t offsetNew = 16;
        const AesCtrCounterExtendedStorage::Entry expectedArray[] =
            {
                makeEntry(  0, 10),
                makeEntry( 80, 20),
                makeEntry( 96, GenerationForUpdate),
                makeEntry(192, 30),
                makeEntry(272, GenerationForUpdate),
            };
        params.push_back(
            RelocateAesCtrCounterExtendedTestParam::Make(
                expectedArray, oldEntries, relocationEntries,
                offsetOld, oldDataStorageSize, offsetNew, newDataStorageSize,
                GenerationForUpdate));
    }

    return params;
}

class RelocateAesCtrCounterExtendedTest :
    public ::testing::TestWithParam<RelocateAesCtrCounterExtendedTestParam>
{
public:
    RelocateAesCtrCounterExtendedTest() NN_NOEXCEPT
        : m_pFileSystem()
    {
        auto pHostFileSystem = new nn::fssystem::HostFileSystem();
        pHostFileSystem->Initialize(nn::fssystem::PathOnExecutionDirectory("").Get());

        m_pFileSystem.reset(pHostFileSystem);
    }
    virtual ~RelocateAesCtrCounterExtendedTest() NN_NOEXCEPT NN_OVERRIDE
    {
    }

protected:
    std::unique_ptr<nn::fs::fsa::IFileSystem> m_pFileSystem;
};

#if !defined(NN_SDK_BUILD_RELEASE)
TEST(RelocateAesCtrCounterExtendedDeathTest, Precondition)
{
    EXPECT_DEATH_IF_SUPPORTED(
        nn::fssystem::utilTool::OptimizeAesCtrCounterExtendedEntry(nullptr, 0, 0), "");

    std::vector<nn::fssystem::AesCtrCounterExtendedStorage::Entry> entries;

    EXPECT_DEATH_IF_SUPPORTED(
        nn::fssystem::utilTool::OptimizeAesCtrCounterExtendedEntry(&entries, -1, 0), "");

    EXPECT_DEATH_IF_SUPPORTED(
        nn::fssystem::utilTool::OptimizeAesCtrCounterExtendedEntry(&entries, 0, -1), "");
}
#endif

TEST_P(RelocateAesCtrCounterExtendedTest, RelocateAesCtrCounterExtendedTable)
{
    const char* oldPatchPath = "/old.patch";
    const char* newPatchPath = "/new.patch";

    m_pFileSystem->DeleteFile(oldPatchPath);
    NNT_ASSERT_RESULT_SUCCESS(m_pFileSystem->CreateFile(oldPatchPath, 160 * 1024, 0));

    m_pFileSystem->DeleteFile(newPatchPath);
    NNT_ASSERT_RESULT_SUCCESS(m_pFileSystem->CreateFile(newPatchPath, 160 * 1024, 0));

    const auto expected = GetParam().expectedEntries;
    const auto oldEntries = GetParam().oldEntries;
    const auto oldDataStorageSize = GetParam().oldDataStorageSize;
    const auto newDataStorageSize = GetParam().newDataStorageSize;

    // 前バージョンのパッチのデータを準備する
    {
        std::unique_ptr<nn::fs::fsa::IFile> file;
        NNT_ASSERT_RESULT_SUCCESS(m_pFileSystem->OpenFile(
                                      &file, oldPatchPath,
                                      static_cast<nn::fs::OpenMode>(nn::fs::OpenMode::OpenMode_Write)
                                      ));
        ASSERT_TRUE(file != nullptr);
        {
            nn::fs::FileStorage storage(file.get());

            const int entryCount = static_cast<int>(oldEntries.size());
            const auto headerStorageSize = BucketTreeBuilder::QueryHeaderStorageSize();
            const auto nodeStorageSize =
                BucketTreeBuilder::QueryNodeStorageSize(
                    AesCtrCounterExtendedStorage::NodeSize,
                    sizeof(AesCtrCounterExtendedStorage::Entry), entryCount);
            const auto entryStorageSize =
                BucketTreeBuilder::QueryEntryStorageSize(
                    AesCtrCounterExtendedStorage::NodeSize,
                    sizeof(AesCtrCounterExtendedStorage::Entry), entryCount);
            const auto headerStorageOffset = 0;
            const auto nodeStorageOffset = headerStorageSize;
            const auto entryStorageOffset = headerStorageSize + nodeStorageSize;

            // テーブルデータを作成
            BucketTreeBuilder builder;
            NNT_ASSERT_RESULT_SUCCESS(
                builder.Initialize(
                    nnt::fs::util::GetTestLibraryAllocator(),
                    nn::fs::SubStorage(&storage, headerStorageOffset, headerStorageSize),
                    nn::fs::SubStorage(&storage, nodeStorageOffset, nodeStorageSize),
                    nn::fs::SubStorage(&storage, entryStorageOffset, entryStorageSize),
                    AesCtrCounterExtendedStorage::NodeSize,
                    sizeof(AesCtrCounterExtendedStorage::Entry),
                    entryCount
                    )
                );

            NNT_ASSERT_RESULT_SUCCESS(
                builder.Write(&oldEntries[0], static_cast<int>(oldEntries.size())));
            NNT_ASSERT_RESULT_SUCCESS(builder.Commit(oldDataStorageSize));
        }
    }

    // RelocationTable を作成
    int64_t relocatedNewDataSize = 0;
    RelocationTable relocationTable;
    RelocationTable invertedRelocationTable;
    {
        // テーブル作成
        for (const auto& entry : GetParam().relocationEntries)
        {
            relocationTable.Add(entry);
        }

        NNT_ASSERT_RESULT_SUCCESS(relocationTable.Commit());

        // 逆テーブル作成
        NNT_ASSERT_RESULT_SUCCESS(
            relocationTable.Invert(&invertedRelocationTable, &relocatedNewDataSize, 0, 0));
    }
    ASSERT_GE(relocatedNewDataSize, newDataStorageSize);

    // 新バージョンのパッチのデータを作成する
    {
        std::unique_ptr<nn::fs::fsa::IFile> oldFile;
        NNT_ASSERT_RESULT_SUCCESS(m_pFileSystem->OpenFile(
                                      &oldFile, oldPatchPath,
                                      static_cast<nn::fs::OpenMode>(nn::fs::OpenMode::OpenMode_Read)
                                      ));
        ASSERT_TRUE(oldFile != nullptr);

        std::unique_ptr<nn::fs::fsa::IFile> newFile;
        NNT_ASSERT_RESULT_SUCCESS(
            m_pFileSystem->OpenFile(
                &newFile, newPatchPath,
                static_cast<nn::fs::OpenMode>(nn::fs::OpenMode::OpenMode_Write)
                ));
        ASSERT_TRUE(newFile != nullptr);

        {
            nn::fs::FileStorage oldStorage(oldFile.get());
            nn::fs::FileStorage newStorage(newFile.get());

            nn::fs::SubStorage srcNodeStorage;
            nn::fs::SubStorage srcEntryStorage;
            {
                const int entryCount = static_cast<int>(oldEntries.size());
                const auto headerStorageSize = BucketTreeBuilder::QueryHeaderStorageSize();
                const auto nodeStorageSize =
                    BucketTreeBuilder::QueryNodeStorageSize(
                        AesCtrCounterExtendedStorage::NodeSize,
                        sizeof(AesCtrCounterExtendedStorage::Entry), entryCount);
                const auto entryStorageSize =
                    BucketTreeBuilder::QueryEntryStorageSize(
                        AesCtrCounterExtendedStorage::NodeSize,
                        sizeof(AesCtrCounterExtendedStorage::Entry), entryCount);
                const auto nodeStorageOffset = headerStorageSize;
                const auto entryStorageOffset = headerStorageSize + nodeStorageSize;

                srcNodeStorage =
                    nn::fs::SubStorage(&oldStorage, nodeStorageOffset, nodeStorageSize);
                srcEntryStorage =
                    nn::fs::SubStorage(&oldStorage, entryStorageOffset, entryStorageSize);
            }

            nn::fssystem::utilTool::AesCtrCounterExtendedStorageBuilder::TableInfo tableInfo(
                static_cast<int>(oldEntries.size()), srcNodeStorage, srcEntryStorage);

            nn::fssystem::utilTool::AesCtrCounterExtendedStorageBuilder builder;
            builder.Initialize(
                tableInfo,
                GetParam().offsetOld,
                oldDataStorageSize,
                GetParam().offsetNew,
                relocatedNewDataSize
            );

            // RelocationTable を元に AesCtrCounterExtendedTable を作成
            NNT_ASSERT_RESULT_SUCCESS(builder.Build(
                nnt::fs::util::GetTestLibraryAllocator(),
                invertedRelocationTable,
                relocatedNewDataSize,
                0,
                GetParam().generationForUpdate
            ));

            // テーブルの書き込み
            {
                const auto headerStorageOffset = 0;
                const auto headerStorageSize = BucketTreeBuilder::QueryHeaderStorageSize();
                const auto nodeStorageOffset = headerStorageOffset + headerStorageSize;
                const auto nodeStorageSize = builder.QueryTableNodeStorageSize();
                const auto entryStorageOffset = nodeStorageOffset + nodeStorageSize;
                const auto entryStorageSize = builder.QueryTableEntryStorageSize();

                NNT_EXPECT_RESULT_SUCCESS(builder.WriteTable(
                    nnt::fs::util::GetTestLibraryAllocator(),
                    nn::fs::SubStorage(&newStorage, headerStorageOffset, headerStorageSize),
                    nn::fs::SubStorage(&newStorage, nodeStorageOffset, nodeStorageSize),
                    nn::fs::SubStorage(&newStorage, entryStorageOffset, entryStorageSize)
                ));
            }
        }
    }

    // AesCtrCounterExtendedTable の検証
    {
        std::unique_ptr<nn::fs::fsa::IFile> newFile;
        NNT_ASSERT_RESULT_SUCCESS(m_pFileSystem->OpenFile(
                                      &newFile, newPatchPath,
                                      static_cast<nn::fs::OpenMode>(nn::fs::OpenMode::OpenMode_Read)
                                      ));
        ASSERT_TRUE(newFile != nullptr);
        {
            nn::fs::FileStorage newStorage(newFile.get());

            nn::fs::SubStorage nodeStorage;
            nn::fs::SubStorage entryStorage;
            int entryCount = 0;
            {
                nn::fssystem::BucketTree::Header header;
                NNT_ASSERT_RESULT_SUCCESS(newStorage.Read(0, &header, sizeof(header)));
                entryCount = header.entryCount;

                const auto headerStorageSize = BucketTreeBuilder::QueryHeaderStorageSize();
                const auto nodeStorageSize =
                    BucketTreeBuilder::QueryNodeStorageSize(
                        AesCtrCounterExtendedStorage::NodeSize,
                        sizeof(AesCtrCounterExtendedStorage::Entry), entryCount);
                const auto entryStorageSize =
                    BucketTreeBuilder::QueryEntryStorageSize(
                        AesCtrCounterExtendedStorage::NodeSize,
                        sizeof(AesCtrCounterExtendedStorage::Entry), entryCount);
                const auto nodeStorageOffset = headerStorageSize;
                const auto entryStorageOffset = headerStorageSize + nodeStorageSize;

                nodeStorage =
                    nn::fs::SubStorage(&newStorage, nodeStorageOffset, nodeStorageSize);
                entryStorage =
                    nn::fs::SubStorage(&newStorage, entryStorageOffset, entryStorageSize);
            }

            // BucketTree を構成する
            std::unique_ptr<nn::fssystem::BucketTree> pBucketTree(new nn::fssystem::BucketTree());
            ASSERT_NE(pBucketTree, nullptr);
            NNT_ASSERT_RESULT_SUCCESS(
                pBucketTree->Initialize(
                    nnt::fs::util::GetTestLibraryAllocator(), nodeStorage, entryStorage,
                    AesCtrCounterExtendedStorage::NodeSize,
                    sizeof(AesCtrCounterExtendedStorage::Entry), entryCount));

            // 全エントリーを取得
            nnt::fs::util::Vector<AesCtrCounterExtendedStorage::Entry> allEntries;

            nn::fssystem::BucketTree::Visitor visitor;
            NNT_ASSERT_RESULT_SUCCESS(pBucketTree->Find(&visitor, 0));
            while (NN_STATIC_CONDITION(true))
            {
                const AesCtrCounterExtendedStorage::Entry* pEntry =
                    visitor.Get<AesCtrCounterExtendedStorage::Entry>();
                ASSERT_NE(pEntry, nullptr);
                allEntries.push_back(*pEntry);
                if (visitor.CanMoveNext())
                {
                    visitor.MoveNext();
                }
                else
                {
                    break;
                }
            }

            // 比較
            ASSERT_EQ(expected.size(), allEntries.size());
            {
                auto expectedIter = expected.begin();
                auto allEntriesIter = allEntries.begin();
                for ( ; expectedIter != expected.end(); ++expectedIter, ++allEntriesIter)
                {
                    EXPECT_EQ(expectedIter->GetOffset(), allEntriesIter->GetOffset());
                    EXPECT_EQ(expectedIter->generation, allEntriesIter->generation);
                }
            }
        }
    }

    NNT_ASSERT_RESULT_SUCCESS(m_pFileSystem->DeleteFile(oldPatchPath));
    NNT_ASSERT_RESULT_SUCCESS(m_pFileSystem->DeleteFile(newPatchPath));
} // NOLINT(impl/function_size)

INSTANTIATE_TEST_CASE_P(
    RelocateAesCtrCounterExtendedTable,
    RelocateAesCtrCounterExtendedTest,
    ::testing::ValuesIn(MakeRelocateAesCtrCounterExtendedParam())
);

TEST(RelocateAesCtrCounterExtendedTest, MergeSameGenerationEntries)
{
    auto makeEntry = [](int64_t offset, uint32_t generation) NN_NOEXCEPT
        -> const AesCtrCounterExtendedStorage::Entry
        {
            AesCtrCounterExtendedStorage::Entry entry;
            entry.SetOffset(offset);
            entry.reserved = 0;
            entry.generation = generation;
            return entry;
        };

    auto isSame = [](const std::vector<AesCtrCounterExtendedStorage::Entry>& lhs,
                     const std::vector<AesCtrCounterExtendedStorage::Entry>& rhs) NN_NOEXCEPT
        -> bool
    {
        if (lhs.size() != rhs.size())
        {
            return false;
        }

        auto lhsIter = lhs.begin();
        auto rhsIter = rhs.begin();
        for ( ; lhsIter != lhs.end(); ++lhsIter, ++rhsIter)
        {
            if (lhsIter->GetOffset() != rhsIter->GetOffset()
                || lhsIter->generation != rhsIter->generation )
            {
                return false;
            }
        }
        return true;
    };

    {
        const AesCtrCounterExtendedStorage::Entry inputArray[] =
            {
                makeEntry(  0, 10),
                makeEntry( 16, 10),
                makeEntry( 96, 20),
                makeEntry(112, 20),
                makeEntry(192, 30),
                makeEntry(208, 30),
                makeEntry(288, 10),
            };
        const AesCtrCounterExtendedStorage::Entry expectedArray[] =
            {
                makeEntry(  0, 10),
                makeEntry( 96, 20),
                makeEntry(192, 30),
                makeEntry(288, 10),
            };

        std::vector<AesCtrCounterExtendedStorage::Entry> expected(
            expectedArray, expectedArray + sizeof(expectedArray) / sizeof(expectedArray[0]));

        std::vector<AesCtrCounterExtendedStorage::Entry> merged(
            inputArray, inputArray + sizeof(inputArray) / sizeof(inputArray[0]));

        NNT_ASSERT_RESULT_SUCCESS(nn::fssystem::utilTool::OptimizeAesCtrCounterExtendedEntry(&merged, 300, 30));
        EXPECT_TRUE(isSame(expected, merged));
    }

    {
        const AesCtrCounterExtendedStorage::Entry inputArray[] =
            {
                makeEntry(  0, 10),
                makeEntry( 96, 20),
                makeEntry(192, 30),
                makeEntry(288, 10),
            };
        const AesCtrCounterExtendedStorage::Entry expectedArray[] =
            {
                makeEntry(  0, 10),
                makeEntry( 96, 20),
                makeEntry(192, 30),
                makeEntry(288, 10),
            };

        std::vector<AesCtrCounterExtendedStorage::Entry> expected(
            expectedArray, expectedArray + sizeof(expectedArray) / sizeof(expectedArray[0]));

        std::vector<AesCtrCounterExtendedStorage::Entry> merged(
            inputArray, inputArray + sizeof(inputArray) / sizeof(inputArray[0]));

        NNT_ASSERT_RESULT_SUCCESS(nn::fssystem::utilTool::OptimizeAesCtrCounterExtendedEntry(&merged, 300, 30));
        EXPECT_TRUE(isSame(expected, merged));
    }
}

