﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <iterator>
#include <nnt/nntest.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nn/fssystem/utilTool/fs_RelocatedBinaryMatch.h>

namespace {

typedef nn::fssystem::utilTool::BinaryMatchResult MatchResult;
typedef nn::fssystem::utilTool::RelocationShiftRange ShiftRange;
typedef nn::fssystem::utilTool::RelocationShiftRangeContainer ShiftRangeConteiner;

const auto InvalidOffset = MatchResult::InvalidOffset;

template< size_t N >
void Compare(
         const MatchResult (&expected)[N],
         const std::vector<MatchResult>& actual
     ) NN_NOEXCEPT
{
    ASSERT_EQ(N, actual.size());
    for( size_t i = 0; i < N; ++i )
    {
        EXPECT_EQ(expected[i].oldOffset, actual[i].oldOffset);
        EXPECT_EQ(expected[i].newOffset, actual[i].newOffset);
        EXPECT_EQ(expected[i].size, actual[i].size);
        EXPECT_EQ(expected[i].storageIndex, actual[i].storageIndex);
    }
}

template< size_t N, size_t M >
void ShiftBinaryMatchResultByAnchor(
         const MatchResult (&before)[N],
         const MatchResult (&after)[M],
         int64_t limitNewOffset,
         int64_t limitOldOffset
     ) NN_NOEXCEPT
{
    std::vector<MatchResult> relocatedData;
    std::copy(before, before + N, std::back_inserter(relocatedData));
    {
        ShiftRangeConteiner shiftRanges(&relocatedData, limitOldOffset);
        if( shiftRanges.Build(0) )
        {
            shiftRanges.ShiftByAnchor(limitNewOffset);
        }
    }
    Compare(after, relocatedData);
}

template< size_t N, size_t M >
void ShiftBinaryMatchResultBySmallFit(
         const MatchResult (&before)[N],
         const MatchResult (&after)[M],
         int64_t limitOffset
     ) NN_NOEXCEPT
{
    std::vector<MatchResult> relocatedData;
    std::copy(before, before + N, std::back_inserter(relocatedData));
    {
        ShiftRangeConteiner shiftRanges(&relocatedData, limitOffset);
        if( shiftRanges.Build(0) )
        {
            shiftRanges.ShiftBySmallFit();
        }
    }
    Compare(after, relocatedData);
}

template< size_t N, size_t M >
void ShiftBinaryMatchResultByLargeSplit(
         const MatchResult (&before)[N],
         const MatchResult (&after)[M],
         int64_t limitOffset,
         int64_t expandedSize
     ) NN_NOEXCEPT
{
    std::vector<MatchResult> relocatedData;
    std::copy(before, before + N, std::back_inserter(relocatedData));
    {
        ShiftRangeConteiner shiftRanges(&relocatedData, limitOffset);
        if( shiftRanges.Build(expandedSize) )
        {
            shiftRanges.ShiftByLargeSplit();
        }
    }
    Compare(after, relocatedData);
}

template< size_t N, size_t M >
void ShiftBinaryMatchResultByLeftJustify(
         const MatchResult (&before)[N],
         const MatchResult (&after)[M],
         int64_t limitOffset
     ) NN_NOEXCEPT
{
    std::vector<MatchResult> relocatedData;
    std::copy(before, before + N, std::back_inserter(relocatedData));
    {
        ShiftRangeConteiner shiftRanges(&relocatedData, limitOffset);
        if( shiftRanges.Build(0) )
        {
            shiftRanges.ShiftByLeftJustify();
        }
    }
    Compare(after, relocatedData);
}

template< size_t N, size_t M >
void ShiftBinaryMatchResultBySoftSplit(
         const MatchResult (&before)[N],
         const MatchResult (&after)[M],
         int64_t limitOffset,
         int64_t expandedSize
     ) NN_NOEXCEPT
{
    std::vector<MatchResult> relocatedData;
    std::copy(before, before + N, std::back_inserter(relocatedData));
    {
        ShiftRangeConteiner shiftRanges(&relocatedData, limitOffset);
        if( shiftRanges.Build(expandedSize) )
        {
            shiftRanges.ShiftBySoftSplit(expandedSize);
        }
    }
    Compare(after, relocatedData);
}

template< size_t N, size_t M >
void ShiftBinaryMatchResultByRightJustify(
         const MatchResult (&before)[N],
         const MatchResult (&after)[M],
         int64_t limitOffset,
         int64_t expandedSize
     ) NN_NOEXCEPT
{
    std::vector<MatchResult> relocatedData;
    std::copy(before, before + N, std::back_inserter(relocatedData));
    {
        ShiftRangeConteiner shiftRanges(&relocatedData, limitOffset);
        if( shiftRanges.Build(expandedSize) )
        {
            shiftRanges.ShiftByRightJustify();
        }
    }
    Compare(after, relocatedData);
}

}

TEST(RelocatedBinaryMatchTest, ShiftBinaryMatchResultByAnchor)
{
    static const int KB = 1024;
    static const int MB = 1024 * 1024;
    {
        // ShiftBinaryMatchResultByAnchor() 前
        const MatchResult Before[] =
        {
            {   0 * KB,   0 * KB,        2 * KB, 1 },
            {   2 * KB, InvalidOffset,   5 * KB, 1 },
            {   7 * KB,   7 * KB,       17 * KB, 1 },
            {  24 * KB, InvalidOffset,   4 * KB, 1 },
            {  28 * KB,  28 * KB,       22 * KB, 1 },
            {  50 * KB, InvalidOffset,   5 * KB, 1 },
            {  55 * KB,  55 * KB,       35 * KB, 1 },
            {  90 * KB, InvalidOffset,   4 * KB, 1 },
            {  94 * KB,  94 * KB,       34 * KB, 1 },
            // 2KB ～ 7KB に空きあり（5KB）
            { 128 * KB,  24 * KB,        4 * KB, 0 },
            { 132 * KB,  50 * KB,        1 * KB, 0 },
            { 133 * KB,  90 * KB,        4 * KB, 0 },
            { 137 * KB, 137 * KB,        1 * KB, 0 },
        };

        // ShiftBinaryMatchResultByAnchor() 後
        const MatchResult After[] =
        {
            {   0 * KB,   0 * KB,        2 * KB, 1 },
            {   2 * KB,   2 * KB,        5 * KB, 1 },
            {   7 * KB,   7 * KB,       17 * KB, 1 },
            {  24 * KB, InvalidOffset,   4 * KB, 1 },
            {  28 * KB,  28 * KB,       22 * KB, 1 },
            {  50 * KB, InvalidOffset,   5 * KB, 1 },
            {  55 * KB,  55 * KB,       35 * KB, 1 },
            {  90 * KB, InvalidOffset,   4 * KB, 1 },
            {  94 * KB,  94 * KB,       34 * KB, 1 },
            { 128 * KB,  24 * KB,        4 * KB, 0 },
            { 132 * KB,  50 * KB,        1 * KB, 0 },
            { 133 * KB,  90 * KB,        4 * KB, 0 },
            { 137 * KB, 137 * KB,        1 * KB, 0 },
        };

        ShiftBinaryMatchResultByAnchor(Before, After, 138 * KB, 128 * KB);
    }

    // limitNewOffset < limitOldOffset
    {
        // ShiftBinaryMatchResultByAnchor() 前
        const MatchResult Before[] =
        {
            { 0 * KB, 16 * KB,        2 * KB, 0 },
            { 2 * KB, InvalidOffset, 30 * KB, 1 },
        };

        // ShiftBinaryMatchResultByAnchor() 後
        const MatchResult After[] =
        {
            { 0 * KB, 16 * KB,  2 * KB, 0 },
            { 2 * KB, 18 * KB, 30 * KB, 1 },
        };

        ShiftBinaryMatchResultByAnchor(Before, After, 32 * KB, 48 * KB);
    }

    // limitNewOffset > limitOldOffset
    {
        // ShiftBinaryMatchResultByAnchor() 前
        const MatchResult Before[] =
        {
            { 0 * KB, 16 * KB,         2 * KB, 0 },
            { 2 * KB, InvalidOffset, 126 * KB, 1 },
        };

        // ShiftBinaryMatchResultByAnchor() 後
        const MatchResult After[] =
        {
            { 0 * KB, 16 * KB,   2 * KB, 0 },
            { 2 * KB, 18 * KB, 126 * KB, 1 },
        };

        ShiftBinaryMatchResultByAnchor(Before, After, 128 * KB, 32 * KB);
    }

    // limitNewOffset + 1M < limitOldOffset
    {
        // ShiftBinaryMatchResultByAnchor() 前
        const MatchResult Before[] =
        {
            { 0 * MB, 16 * MB,        2 * MB, 0 },
            { 2 * MB, InvalidOffset, 30 * MB, 1 },
        };

        // ShiftBinaryMatchResultByAnchor() 後
        const MatchResult After[] =
        {
            {  0 * MB, 16 * MB,        2 * MB, 0 },
            {  2 * MB, 18 * MB,       15 * MB, 1 },
            { 17 * MB, InvalidOffset, 15 * MB, 1 },
        };

        ShiftBinaryMatchResultByAnchor(Before, After, 32 * MB, 48 * MB);
    }

    // limitNewOffset + 1M > limitOldOffset
    {
        // ShiftBinaryMatchResultByAnchor() 前
        const MatchResult Before[] =
        {
            { 0 * MB, 16 * MB,         2 * MB, 0 },
            { 2 * MB, InvalidOffset, 126 * MB, 1 },
        };

        // ShiftBinaryMatchResultByAnchor() 後
        const MatchResult After[] =
        {
            {   0 * MB, 16 * MB,         2 * MB, 0 },
            {   2 * MB, 18 * MB,       111 * MB, 1 },
            { 113 * MB, InvalidOffset,  15 * MB, 1 },
        };

        ShiftBinaryMatchResultByAnchor(Before, After, 128 * MB, 32 * MB);
    }
}

TEST(RelocatedBinaryMatchTest, ShiftBinaryMatchResultBySmallFit)
{
    static const int KB = 1024;
    {
        // ShiftBinaryMatchResultBySmallFit() 前
        const MatchResult Before[] =
        {
            {   0 * KB,   0 * KB,        2 * KB, 1 },
            {   2 * KB, InvalidOffset,   3 * KB, 1 },
            {   5 * KB,   5 * KB,       19 * KB, 1 },
            {  24 * KB, InvalidOffset,   8 * KB, 1 },
            {  32 * KB,  32 * KB,       18 * KB, 1 },
            {  50 * KB, InvalidOffset,  15 * KB, 1 },
            {  65 * KB,  65 * KB,       25 * KB, 1 },
            {  90 * KB, InvalidOffset,   5 * KB, 1 },
            {  95 * KB,  95 * KB,       33 * KB, 1 },
            { 128 * KB,   2 * KB,        3 * KB, 0 },
            { 131 * KB,  24 * KB,        8 * KB, 0 },
            { 139 * KB,  50 * KB,       16 * KB, 0 },
            { 155 * KB,  90 * KB,        5 * KB, 0 },
            // 128KB ～ 140KB に空きあり（12KB）
            { 160 * KB, 140 * KB,      110 * KB, 0 },
            { 270 * KB, 250 * KB,        1 * KB, 1 },
        };

        // ShiftBinaryMatchResultBySmallFit() 後
        const MatchResult After[] =
        {
            {   0 * KB,   0 * KB,        2 * KB, 1 },
            {   2 * KB, 128 * KB,        3 * KB, 1 },
            {   5 * KB,   5 * KB,       19 * KB, 1 },
            {  24 * KB, 131 * KB,        8 * KB, 1 },
            {  32 * KB,  32 * KB,       18 * KB, 1 },
            {  50 * KB, InvalidOffset,  15 * KB, 1 },
            {  65 * KB,  65 * KB,       25 * KB, 1 },
            {  90 * KB, InvalidOffset,   5 * KB, 1 },
            {  95 * KB,  95 * KB,       33 * KB, 1 },
            { 128 * KB,   2 * KB,        3 * KB, 0 },
            { 131 * KB,  24 * KB,        8 * KB, 0 },
            { 139 * KB,  50 * KB,       16 * KB, 0 },
            { 155 * KB,  90 * KB,        5 * KB, 0 },
            { 160 * KB, 140 * KB,      110 * KB, 0 },
            { 270 * KB, 250 * KB,        1 * KB, 1 },
        };

        ShiftBinaryMatchResultBySmallFit(Before, After, 251 * KB);
    }

    // 空き領域が大きすぎる（16KB）場合
    {
        const MatchResult Data[] =
        {
            {   0 * KB,   0 * KB,        2 * KB, 1 },
            {   2 * KB, InvalidOffset,   3 * KB, 1 },
            {   5 * KB,   5 * KB,       19 * KB, 1 },
            {  24 * KB, InvalidOffset,   8 * KB, 1 },
            {  32 * KB,  32 * KB,       18 * KB, 1 },
            {  50 * KB, InvalidOffset,  15 * KB, 1 },
            {  65 * KB,  65 * KB,       25 * KB, 1 },
            {  90 * KB, InvalidOffset,   5 * KB, 1 },
            {  95 * KB,  95 * KB,       33 * KB, 1 },
            { 128 * KB,   2 * KB,        3 * KB, 0 },
            { 131 * KB,  24 * KB,        8 * KB, 0 },
            { 139 * KB,  50 * KB,       16 * KB, 0 },
            { 155 * KB,  90 * KB,        5 * KB, 0 },
            // 128KB ～ 144KB に空きあり（16KB）
            { 160 * KB, 144 * KB,      110 * KB, 0 },
            { 270 * KB, 254 * KB,        1 * KB, 1 },
        };

        ShiftBinaryMatchResultBySmallFit(Data, Data, 255 * KB);
    }

    // 境界値（16KB）の確認
    {
        // ShiftBinaryMatchResultBySmallFit() 前
        const MatchResult Before[] =
        {
            {   0 * KB, InvalidOffset, 16 * KB, 1 },
            {  16 * KB, 16 * KB,        1 * KB, 1 },
            {  17 * KB, InvalidOffset, 15 * KB, 1 },
            {  32 * KB, 32 * KB,        1 * KB, 0 },
        };

        // ShiftBinaryMatchResultBySmallFit() 後
        const MatchResult After[] =
        {
            {   0 * KB, InvalidOffset, 16 * KB, 1 },
            {  16 * KB, 16 * KB,        1 * KB, 1 },
            {  17 * KB, 17 * KB,       15 * KB, 1 },
            {  32 * KB, 32 * KB,        1 * KB, 0 },
        };

        ShiftBinaryMatchResultBySmallFit(Before, After, 33 * KB);
    }
}

TEST(RelocatedBinaryMatchTest, ShiftBinaryMatchResultByLargeSplit)
{
    static const int KB = 1024;
    static const int64_t MB = 1024 * 1024;
    {
        // ShiftBinaryMatchResultByLargeSplit() 前
        const MatchResult Before[] =
        {
            {    0 * MB,    0 * MB,       20 * MB, 1 },
            {   20 * MB, InvalidOffset,   30 * MB, 1 },
            {   50 * MB,   50 * MB,      190 * MB, 1 },
            {  240 * MB, InvalidOffset,   80 * MB, 1 },
            {  320 * MB,  320 * MB,      180 * MB, 1 },
            {  500 * MB, InvalidOffset,  160 * MB, 1 },
            {  660 * MB,  660 * MB,      240 * MB, 1 },
            {  900 * MB, InvalidOffset,   50 * MB, 1 },
            {  950 * MB,  950 * MB,      330 * MB, 1 },
            { 1280 * MB,   20 * MB,       30 * MB, 0 },
            { 1310 * MB,  240 * MB,       80 * MB, 0 },
            { 1390 * MB,  500 * MB,      160 * MB, 0 },
            { 1550 * MB,  900 * MB,       50 * MB, 0 },
            // 1280MB ～ 1480MB に空きあり（200MB）
            { 1600 * MB, 1480 * MB,     1100 * MB, 0 },
            // 2580MB ～ 2700MB に空きあり（120MB）
            { 2700 * MB, 2700 * MB,       10 * MB, 1 },
        };

        // ShiftBinaryMatchResultByLargeSplit() 後
        const MatchResult After[] =
        {
            {    0 * MB,    0 * MB,   20 * MB, 1 },
            {   20 * MB, 2670 * MB,   30 * MB, 1 },
            {   50 * MB,   50 * MB,  190 * MB, 1 },
            {  240 * MB, 2580 * MB,   80 * MB, 1 },
            {  320 * MB,  320 * MB,  180 * MB, 1 },
            {  500 * MB, 1280 * MB,  160 * MB, 1 },
            {  660 * MB,  660 * MB,  240 * MB, 1 },
            {  900 * MB, 1440 * MB,   40 * MB, 1 },
            {  940 * MB, 2660 * MB,   10 * MB, 1 },
            {  950 * MB,  950 * MB,  330 * MB, 1 },
            { 1280 * MB,   20 * MB,   30 * MB, 0 },
            { 1310 * MB,  240 * MB,   80 * MB, 0 },
            { 1390 * MB,  500 * MB,  160 * MB, 0 },
            { 1550 * MB,  900 * MB,   50 * MB, 0 },
            { 1600 * MB, 1480 * MB, 1100 * MB, 0 },
            { 2700 * MB, 2700 * MB,   10 * MB, 1 },
        };

        ShiftBinaryMatchResultByLargeSplit(Before, After, 2710 * MB, 0);
    }

    // 最大分割数（10）を超える場合
    {
        // ShiftBinaryMatchResultByLargeSplit() 前
        const MatchResult Before[] =
        {
            {   0 * MB, InvalidOffset, 400 * MB, 1 },
            { 400 * MB, 500 * MB,      100 * MB, 1 },
            { 500 * MB,   0 * MB,       20 * MB, 0 },
            { 540 * MB,  40 * MB,       20 * MB, 0 },
            { 580 * MB,  80 * MB,       20 * MB, 0 },
            { 620 * MB, 120 * MB,       20 * MB, 0 },
            { 660 * MB, 160 * MB,       20 * MB, 0 },
            { 700 * MB, 200 * MB,       20 * MB, 0 },
            { 740 * MB, 240 * MB,       20 * MB, 0 },
            { 780 * MB, 280 * MB,       20 * MB, 0 },
            { 820 * MB, 320 * MB,       20 * MB, 0 },
            { 860 * MB, 360 * MB,       20 * MB, 0 },
            { 900 * MB, 400 * MB,       20 * MB, 0 },
            { 940 * MB, 440 * MB,       20 * MB, 0 },
        };

        // ShiftBinaryMatchResultByLargeSplit() 後
        const MatchResult After[] =
        {
            {   0 * MB, 140 * MB,  20 * MB, 1 },
            {  20 * MB, 180 * MB,  20 * MB, 1 },
            {  40 * MB, 220 * MB,  20 * MB, 1 },
            {  60 * MB, 260 * MB,  20 * MB, 1 },
            {  80 * MB, 300 * MB,  20 * MB, 1 },
            { 100 * MB, 340 * MB,  20 * MB, 1 },
            { 120 * MB, 380 * MB,  20 * MB, 1 },
            { 140 * MB, 420 * MB,  20 * MB, 1 },
            { 160 * MB, 460 * MB,  40 * MB, 1 },
            { 200 * MB, 600 * MB, 200 * MB, 1 },
            { 400 * MB, 500 * MB, 100 * MB, 1 },
            { 500 * MB,   0 * MB,  20 * MB, 0 },
            { 540 * MB,  40 * MB,  20 * MB, 0 },
            { 580 * MB,  80 * MB,  20 * MB, 0 },
            { 620 * MB, 120 * MB,  20 * MB, 0 },
            { 660 * MB, 160 * MB,  20 * MB, 0 },
            { 700 * MB, 200 * MB,  20 * MB, 0 },
            { 740 * MB, 240 * MB,  20 * MB, 0 },
            { 780 * MB, 280 * MB,  20 * MB, 0 },
            { 820 * MB, 320 * MB,  20 * MB, 0 },
            { 860 * MB, 360 * MB,  20 * MB, 0 },
            { 900 * MB, 400 * MB,  20 * MB, 0 },
            { 940 * MB, 440 * MB,  20 * MB, 0 },
        };

        ShiftBinaryMatchResultByLargeSplit(Before, After, 1000 * MB, 400 * MB);
    }

    // 境界値（256KB）の確認
    {
        // ShiftBinaryMatchResultByLargeSplit() 前
        const MatchResult Before[] =
        {
            {    0 * KB, InvalidOffset, 255 * KB, 1 },
            {  255 * KB, 255 * KB,        1 * KB, 1 },
            {  256 * KB, InvalidOffset, 256 * KB, 1 },
            {  512 * KB,   0 * KB,       32 * KB, 0 },
            {  544 * KB, 256 * KB,       32 * KB, 0 },
        };

        // ShiftBinaryMatchResultByLargeSplit() 後
        const MatchResult After[] =
        {
            {    0 * KB, InvalidOffset, 255 * KB, 1 },
            {  255 * KB, 255 * KB,        1 * KB, 1 },
            {  256 * KB,  32 * KB,      223 * KB, 1 },
            {  479 * KB, 288 * KB,       33 * KB, 1 },
            {  512 * KB,   0 * KB,       32 * KB, 0 },
            {  544 * KB, 256 * KB,       32 * KB, 0 },
        };

        ShiftBinaryMatchResultByLargeSplit(Before, After, 330 * KB, 0);
    }
} // NOLINT(impl/function_size)

TEST(RelocatedBinaryMatchTest, ShiftBinaryMatchResultByLeftJustify)
{
    static const int MB = 1024 * 1024;
    {
        // ShiftBinaryMatchResultByLeftJustify() 前
        const MatchResult Before[] =
        {
            {   0 * MB,   0 * MB,        2 * MB, 1 },
            {   2 * MB, InvalidOffset,   3 * MB, 1 },
            {   5 * MB,   5 * MB,       19 * MB, 1 },
            {  24 * MB, InvalidOffset,   8 * MB, 1 },
            {  32 * MB,  32 * MB,       18 * MB, 1 },
            {  50 * MB, InvalidOffset,  16 * MB, 1 },
            {  66 * MB,  66 * MB,       24 * MB, 1 },
            {  90 * MB, InvalidOffset,   5 * MB, 1 },
            {  95 * MB,  95 * MB,       33 * MB, 1 },
            { 128 * MB,   2 * MB,        3 * MB, 0 },
            { 131 * MB,  24 * MB,        8 * MB, 0 },
            { 139 * MB,  50 * MB,       16 * MB, 0 },
            { 155 * MB,  90 * MB,        5 * MB, 0 },
            // 128KB ～ 140KB に空きあり（12KB）
            { 160 * MB, 140 * MB,      110 * MB, 0 },
            // 250KB ～ 270KB に空きあり（20KB）
            { 270 * MB, 270 * MB,        1 * MB, 1 },
        };

        // ShiftBinaryMatchResultByLeftJustify() 後
        const MatchResult After[] =
        {
            {   0 * MB,   0 * MB,        2 * MB, 1 },
            {   2 * MB, 128 * MB,        3 * MB, 1 },
            {   5 * MB,   5 * MB,       19 * MB, 1 },
            {  24 * MB, 131 * MB,        8 * MB, 1 },
            {  32 * MB,  32 * MB,       18 * MB, 1 },
            {  50 * MB, 250 * MB,       16 * MB, 1 },
            {  66 * MB,  66 * MB,       24 * MB, 1 },
            {  90 * MB, InvalidOffset,   5 * MB, 1 },
            {  95 * MB,  95 * MB,       33 * MB, 1 },
            { 128 * MB,   2 * MB,        3 * MB, 0 },
            { 131 * MB,  24 * MB,        8 * MB, 0 },
            { 139 * MB,  50 * MB,       16 * MB, 0 },
            { 155 * MB,  90 * MB,        5 * MB, 0 },
            { 160 * MB, 140 * MB,      110 * MB, 0 },
            { 270 * MB, 270 * MB,        1 * MB, 1 },
        };

        ShiftBinaryMatchResultByLeftJustify(Before, After, 271 * MB);
    }

    // 左詰めできない領域を挟む場合
    {
        // ShiftBinaryMatchResultByLeftJustify() 前
        const MatchResult Before[] =
        {
            {   0 * MB,   0 * MB,        2 * MB, 1 },
            {   2 * MB, InvalidOffset,   3 * MB, 1 },
            {   5 * MB,   5 * MB,       19 * MB, 1 },
            {  24 * MB, InvalidOffset,  16 * MB, 1 },
            {  40 * MB,  40 * MB,       18 * MB, 1 },
            {  58 * MB, InvalidOffset,   8 * MB, 1 },
            {  66 * MB,  66 * MB,       24 * MB, 1 },
            {  90 * MB, InvalidOffset,   5 * MB, 1 },
            {  95 * MB,  95 * MB,       33 * MB, 1 },
            { 128 * MB,   2 * MB,        3 * MB, 0 },
            { 131 * MB,  24 * MB,       16 * MB, 0 },
            { 139 * MB,  58 * MB,        8 * MB, 0 },
            { 155 * MB,  90 * MB,        5 * MB, 0 },
            // 128KB ～ 140KB に空きあり（12KB）
            { 160 * MB, 140 * MB,      110 * MB, 0 },
            // 250KB ～ 270KB に空きあり（20KB）
            { 270 * MB, 270 * MB,        1 * MB, 1 },
        };

        // ShiftBinaryMatchResultByLeftJustify() 後
        const MatchResult After[] =
        {
            { 0,          0,             2 * MB, 1 },
            {   2 * MB, 128 * MB,        3 * MB, 1 },
            {   5 * MB,   5 * MB,       19 * MB, 1 },
            {  24 * MB, 250 * MB,       16 * MB, 1 },
            {  40 * MB,  40 * MB,       18 * MB, 1 },
            {  58 * MB, 131 * MB,        8 * MB, 1 },
            {  66 * MB,  66 * MB,       24 * MB, 1 },
            {  90 * MB, InvalidOffset,   5 * MB, 1 },
            {  95 * MB,  95 * MB,       33 * MB, 1 },
            { 128 * MB,   2 * MB,        3 * MB, 0 },
            { 131 * MB,  24 * MB,       16 * MB, 0 },
            { 139 * MB,  58 * MB,        8 * MB, 0 },
            { 155 * MB,  90 * MB,        5 * MB, 0 },
            { 160 * MB, 140 * MB,      110 * MB, 0 },
            { 270 * MB, 270 * MB,        1 * MB, 1 },
        };

        ShiftBinaryMatchResultByLeftJustify(Before, After, 271 * MB);
    }
}

TEST(RelocatedBinaryMatchTest, ShiftBinaryMatchResultBySoftSplit)
{
    static const int MB = 1024 * 1024;
    {
        // ShiftBinaryMatchResultBySoftSplit() 前
        const MatchResult Before[] =
        {
            {   0 * MB,   0 * MB,        2 * MB, 1 },
            {   2 * MB, InvalidOffset,   3 * MB, 1 },
            {   5 * MB,   5 * MB,       19 * MB, 1 },
            {  24 * MB, InvalidOffset,   8 * MB, 1 },
            {  32 * MB,  32 * MB,       18 * MB, 1 },
            {  50 * MB, InvalidOffset,  16 * MB, 1 },
            {  66 * MB,  66 * MB,       24 * MB, 1 },
            {  90 * MB, InvalidOffset,   5 * MB, 1 },
            {  95 * MB,  95 * MB,       33 * MB, 1 },
            { 128 * MB,   2 * MB,        3 * MB, 0 },
            { 131 * MB,  24 * MB,        8 * MB, 0 },
            { 139 * MB,  50 * MB,       16 * MB, 0 },
            { 155 * MB,  90 * MB,        5 * MB, 0 },
            // 128KB ～ 140KB に空きあり（12KB）
            { 160 * MB, 140 * MB,      110 * MB, 0 },
            // 250KB ～ 270KB に空きあり（20KB）
            { 270 * MB, 270 * MB,        1 * MB, 1 },
        };

        // ShiftBinaryMatchResultBySoftSplit() 後
        const MatchResult After[] =
        {
            {   0 * MB,   0 * MB,   2 * MB, 1 },
            {   2 * MB, 128 * MB,   3 * MB, 1 },
            {   5 * MB,   5 * MB,  19 * MB, 1 },
            {  24 * MB, 131 * MB,   8 * MB, 1 },
            {  32 * MB,  32 * MB,  18 * MB, 1 },
            {  50 * MB, 139 * MB,   1 * MB, 1 },
            {  51 * MB, 250 * MB,  15 * MB, 1 },
            {  66 * MB,  66 * MB,  24 * MB, 1 },
            {  90 * MB, 265 * MB,   5 * MB, 1 },
            {  95 * MB,  95 * MB,  33 * MB, 1 },
            { 128 * MB,   2 * MB,   3 * MB, 0 },
            { 131 * MB,  24 * MB,   8 * MB, 0 },
            { 139 * MB,  50 * MB,  16 * MB, 0 },
            { 155 * MB,  90 * MB,   5 * MB, 0 },
            { 160 * MB, 140 * MB, 110 * MB, 0 },
            { 270 * MB, 270 * MB,   1 * MB, 1 },
        };

        ShiftBinaryMatchResultBySoftSplit(Before, After, 271 * MB, 0);
    }

    // 空き領域がない場合
    {
        const MatchResult Data[] =
        {
            {   0 * MB,   0 * MB,        2 * MB, 1 },
            {   2 * MB, InvalidOffset,   3 * MB, 1 },
            {   5 * MB,   5 * MB,       19 * MB, 1 },
            {  24 * MB, InvalidOffset,   8 * MB, 1 },
            {  32 * MB,  32 * MB,       18 * MB, 1 },
            {  50 * MB, InvalidOffset,  16 * MB, 1 },
            {  66 * MB,  66 * MB,       24 * MB, 1 },
            {  90 * MB, InvalidOffset,   5 * MB, 1 },
            {  95 * MB,  95 * MB,       33 * MB, 1 },
            { 128 * MB,   2 * MB,        3 * MB, 0 },
            { 131 * MB,  24 * MB,        8 * MB, 0 },
            { 139 * MB,  50 * MB,       16 * MB, 0 },
            { 155 * MB,  90 * MB,        5 * MB, 0 },
            { 160 * MB, 128 * MB,      139 * MB, 0 },
        };

        ShiftBinaryMatchResultBySoftSplit(Data, Data, 139 * MB, 0);
    }

    // 最大分割数を超える場合
    {
        const MatchResult Data[] =
        {
            {  0 * MB,  0 * MB,        1 * MB, 0 },
            {  1 * MB,  2 * MB,        1 * MB, 0 },
            {  2 * MB,  4 * MB,        1 * MB, 0 },
            {  3 * MB,  6 * MB,        1 * MB, 0 },
            {  4 * MB,  8 * MB,        1 * MB, 0 },
            {  5 * MB, 10 * MB,        1 * MB, 0 },
            {  6 * MB, 12 * MB,        1 * MB, 0 },
            {  7 * MB, 14 * MB,        1 * MB, 0 },
            {  8 * MB, 16 * MB,        1 * MB, 0 },
            {  9 * MB, 18 * MB,        1 * MB, 0 },
            { 10 * MB, 20 * MB,        1 * MB, 0 },
            { 11 * MB, 22 * MB,        1 * MB, 0 },
            { 12 * MB, InvalidOffset, 16 * MB, 1 },
        };

        ShiftBinaryMatchResultBySoftSplit(Data, Data, 24 * MB, 0);
    }
}

TEST(RelocatedBinaryMatchTest, ShiftBinaryMatchResultByRightJustify)
{
    // ShiftBinaryMatchResultByRightJustify() 前
    const MatchResult Before[] =
    {
        { 0,          0,                            2 * 1024, 1 },
        {   2 * 1024, MatchResult::InvalidOffset,   3 * 1024, 1 },
        {   5 * 1024,   5 * 1024,                  19 * 1024, 1 },
        {  24 * 1024, MatchResult::InvalidOffset,   8 * 1024, 1 },
        {  32 * 1024,  32 * 1024,                  18 * 1024, 1 },
        {  50 * 1024, MatchResult::InvalidOffset,  16 * 1024, 1 },
        {  66 * 1024,  66 * 1024,                  24 * 1024, 1 },
        {  90 * 1024, MatchResult::InvalidOffset,   5 * 1024, 1 },
        {  95 * 1024,  95 * 1024,                  33 * 1024, 1 },
        { 128 * 1024,   2 * 1024,                   3 * 1024, 0 },
        { 131 * 1024,  24 * 1024,                   8 * 1024, 0 },
        { 139 * 1024,  50 * 1024,                  16 * 1024, 0 },
        { 155 * 1024,  90 * 1024,                   5 * 1024, 0 },
        { 160 * 1024, 128 * 1024,                 110 * 1024, 0 },
        { 270 * 1024, 239 * 1024,                   1 * 1024, 1 },
    };

    // ShiftBinaryMatchResultByRightJustify() 後
    const MatchResult After[] =
    {
        { 0,          0,            2 * 1024, 1 },
        {   2 * 1024, 240 * 1024,   3 * 1024, 1 },
        {   5 * 1024,   5 * 1024,  19 * 1024, 1 },
        {  24 * 1024, 243 * 1024,   8 * 1024, 1 },
        {  32 * 1024,  32 * 1024,  18 * 1024, 1 },
        {  50 * 1024, 251 * 1024,  16 * 1024, 1 },
        {  66 * 1024,  66 * 1024,  24 * 1024, 1 },
        {  90 * 1024, 267 * 1024,   5 * 1024, 1 },
        {  95 * 1024,  95 * 1024,  33 * 1024, 1 },
        { 128 * 1024,   2 * 1024,   3 * 1024, 0 },
        { 131 * 1024,  24 * 1024,   8 * 1024, 0 },
        { 139 * 1024,  50 * 1024,  16 * 1024, 0 },
        { 155 * 1024,  90 * 1024,   5 * 1024, 0 },
        { 160 * 1024, 128 * 1024, 110 * 1024, 0 },
        { 270 * 1024, 239 * 1024,   1 * 1024, 1 },
    };

    ShiftBinaryMatchResultByRightJustify(Before, After, 270 * 1024, 30 * 1024);
}

TEST(RelocatedBinaryMatchTest, ShiftBinaryMatchResultByAnchorLarge)
{
    static const int64_t GB = 1024 * 1024 * 1024;
    // ShiftBinaryMatchResultByAnchor() 前
    const MatchResult Before[] =
    {
        {   0 * GB,   0 * GB,        2 * GB, 1 },
        {   2 * GB, InvalidOffset,   5 * GB, 1 },
        {   7 * GB,   7 * GB,       17 * GB, 1 },
        {  24 * GB, InvalidOffset,   4 * GB, 1 },
        {  28 * GB,  28 * GB,       22 * GB, 1 },
        {  50 * GB, InvalidOffset,   5 * GB, 1 },
        {  55 * GB,  55 * GB,       35 * GB, 1 },
        {  90 * GB, InvalidOffset,   4 * GB, 1 },
        {  94 * GB,  94 * GB,       34 * GB, 1 },
        // 2GB ～ 7GB に空きあり（5GB）
        { 128 * GB,  24 * GB,        4 * GB, 0 },
        { 132 * GB,  50 * GB,        1 * GB, 0 },
        { 133 * GB,  90 * GB,        4 * GB, 0 },
        { 137 * GB, 137 * GB,        1 * GB, 0 },
    };

    // ShiftBinaryMatchResultByAnchor() 後
    const MatchResult After[] =
    {
        {   0 * GB,   0 * GB,        2 * GB, 1 },
        {   2 * GB,   2 * GB,        5 * GB, 1 },
        {   7 * GB,   7 * GB,       17 * GB, 1 },
        {  24 * GB, InvalidOffset,   4 * GB, 1 },
        {  28 * GB,  28 * GB,       22 * GB, 1 },
        {  50 * GB, InvalidOffset,   5 * GB, 1 },
        {  55 * GB,  55 * GB,       35 * GB, 1 },
        {  90 * GB, InvalidOffset,   4 * GB, 1 },
        {  94 * GB,  94 * GB,       34 * GB, 1 },
        { 128 * GB,  24 * GB,        4 * GB, 0 },
        { 132 * GB,  50 * GB,        1 * GB, 0 },
        { 133 * GB,  90 * GB,        4 * GB, 0 },
        { 137 * GB, 137 * GB,        1 * GB, 0 },
    };

    ShiftBinaryMatchResultByAnchor(Before, After, 138 * GB, 128 * GB);
}

TEST(RelocatedBinaryMatchTest, ShiftBinaryMatchResultBySmallFitLarge)
{
    static const int64_t GB = 1024 * 1024 * 1024;

    // ShiftBinaryMatchResultBySmallFit() 前
    const MatchResult Before[] =
    {
        {   0 * GB,   0 * GB,        2 * GB, 1 },
        {   2 * GB, InvalidOffset,   3 * GB, 1 },
        {   5 * GB,   5 * GB,       19 * GB, 1 },
        {  24 * GB, InvalidOffset,   8 * GB, 1 },
        {  32 * GB,  32 * GB,       18 * GB, 1 },
        {  50 * GB, InvalidOffset,  15 * GB, 1 },
        {  65 * GB,  65 * GB,       25 * GB, 1 },
        {  90 * GB, InvalidOffset,   5 * GB, 1 },
        {  95 * GB,  95 * GB,       33 * GB, 1 },
        { 128 * GB,   2 * GB,        3 * GB, 0 },
        { 131 * GB,  24 * GB,        8 * GB, 0 },
        { 139 * GB,  50 * GB,       16 * GB, 0 },
        { 155 * GB,  90 * GB,        5 * GB, 0 },
        // 128GB ～ 140GB に空きあり（12GB）
        { 160 * GB, 140 * GB,      110 * GB, 0 },
        { 270 * GB, 250 * GB,        1 * GB, 1 },
    };

    // ShiftBinaryMatchResultBySmallFit() 後
    const MatchResult After[] =
    {
        {   0 * GB,   0 * GB,        2 * GB, 1 },
        {   2 * GB, InvalidOffset,   3 * GB, 1 },
        {   5 * GB,   5 * GB,       19 * GB, 1 },
        {  24 * GB, InvalidOffset,   8 * GB, 1 },
        {  32 * GB,  32 * GB,       18 * GB, 1 },
        {  50 * GB, InvalidOffset,  15 * GB, 1 },
        {  65 * GB,  65 * GB,       25 * GB, 1 },
        {  90 * GB, InvalidOffset,   5 * GB, 1 },
        {  95 * GB,  95 * GB,       33 * GB, 1 },
        { 128 * GB,   2 * GB,        3 * GB, 0 },
        { 131 * GB,  24 * GB,        8 * GB, 0 },
        { 139 * GB,  50 * GB,       16 * GB, 0 },
        { 155 * GB,  90 * GB,        5 * GB, 0 },
        { 160 * GB, 140 * GB,      110 * GB, 0 },
        { 270 * GB, 250 * GB,        1 * GB, 1 },
    };

    ShiftBinaryMatchResultBySmallFit(Before, After, 251 * GB);
}

TEST(RelocatedBinaryMatchTest, ShiftBinaryMatchResultByLargeSplitLarge)
{
    static const int64_t GB = 1024 * 1024 * 1024;

    // ShiftBinaryMatchResultByLargeSplit() 前
    const MatchResult Before[] =
    {
        {    0 * GB,    0 * GB,       20 * GB, 1 },
        {   20 * GB, InvalidOffset,   30 * GB, 1 },
        {   50 * GB,   50 * GB,      190 * GB, 1 },
        {  240 * GB, InvalidOffset,   80 * GB, 1 },
        {  320 * GB,  320 * GB,      180 * GB, 1 },
        {  500 * GB, InvalidOffset,  160 * GB, 1 },
        {  660 * GB,  660 * GB,      240 * GB, 1 },
        {  900 * GB, InvalidOffset,   50 * GB, 1 },
        {  950 * GB,  950 * GB,      330 * GB, 1 },
        { 1280 * GB,   20 * GB,       30 * GB, 0 },
        { 1310 * GB,  240 * GB,       80 * GB, 0 },
        { 1390 * GB,  500 * GB,      160 * GB, 0 },
        { 1550 * GB,  900 * GB,       50 * GB, 0 },
        // 1280GB ～ 1480GB に空きあり（200GB）
        { 1600 * GB, 1480 * GB,     1100 * GB, 0 },
        // 2580GB ～ 2700GB に空きあり（120GB）
        { 2700 * GB, 2700 * GB,       10 * GB, 1 },
    };

    // ShiftBinaryMatchResultByLargeSplit() 後
    const MatchResult After[] =
    {
        {    0 * GB,    0 * GB,   20 * GB, 1 },
        {   20 * GB, 2670 * GB,   30 * GB, 1 },
        {   50 * GB,   50 * GB,  190 * GB, 1 },
        {  240 * GB, 2580 * GB,   80 * GB, 1 },
        {  320 * GB,  320 * GB,  180 * GB, 1 },
        {  500 * GB, 1280 * GB,  160 * GB, 1 },
        {  660 * GB,  660 * GB,  240 * GB, 1 },
        {  900 * GB, 1440 * GB,   40 * GB, 1 },
        {  940 * GB, 2660 * GB,   10 * GB, 1 },
        {  950 * GB,  950 * GB,  330 * GB, 1 },
        { 1280 * GB,   20 * GB,   30 * GB, 0 },
        { 1310 * GB,  240 * GB,   80 * GB, 0 },
        { 1390 * GB,  500 * GB,  160 * GB, 0 },
        { 1550 * GB,  900 * GB,   50 * GB, 0 },
        { 1600 * GB, 1480 * GB, 1100 * GB, 0 },
        { 2700 * GB, 2700 * GB,   10 * GB, 1 },
    };

    ShiftBinaryMatchResultByLargeSplit(Before, After, 2710 * GB, 0);
}

TEST(RelocatedBinaryMatchTest, ShiftBinaryMatchResultByLeftJustifyLarge)
{
    static const int64_t GB = 1024 * 1024 * 1024;

    // ShiftBinaryMatchResultByLeftJustify() 前
    const MatchResult Before[] =
    {
        {   0 * GB,   0 * GB,        2 * GB, 1 },
        {   2 * GB, InvalidOffset,   3 * GB, 1 },
        {   5 * GB,   5 * GB,       19 * GB, 1 },
        {  24 * GB, InvalidOffset,   8 * GB, 1 },
        {  32 * GB,  32 * GB,       18 * GB, 1 },
        {  50 * GB, InvalidOffset,  16 * GB, 1 },
        {  66 * GB,  66 * GB,       24 * GB, 1 },
        {  90 * GB, InvalidOffset,   5 * GB, 1 },
        {  95 * GB,  95 * GB,       33 * GB, 1 },
        { 128 * GB,   2 * GB,        3 * GB, 0 },
        { 131 * GB,  24 * GB,        8 * GB, 0 },
        { 139 * GB,  50 * GB,       16 * GB, 0 },
        { 155 * GB,  90 * GB,        5 * GB, 0 },
        // 128KB ～ 140KB に空きあり（12KB）
        { 160 * GB, 140 * GB,      110 * GB, 0 },
        // 250KB ～ 270KB に空きあり（20KB）
        { 270 * GB, 270 * GB,        1 * GB, 1 },
    };

    // ShiftBinaryMatchResultByLeftJustify() 後
    const MatchResult After[] =
    {
        {   0 * GB,   0 * GB,        2 * GB, 1 },
        {   2 * GB, 128 * GB,        3 * GB, 1 },
        {   5 * GB,   5 * GB,       19 * GB, 1 },
        {  24 * GB, 131 * GB,        8 * GB, 1 },
        {  32 * GB,  32 * GB,       18 * GB, 1 },
        {  50 * GB, 250 * GB,       16 * GB, 1 },
        {  66 * GB,  66 * GB,       24 * GB, 1 },
        {  90 * GB, InvalidOffset,   5 * GB, 1 },
        {  95 * GB,  95 * GB,       33 * GB, 1 },
        { 128 * GB,   2 * GB,        3 * GB, 0 },
        { 131 * GB,  24 * GB,        8 * GB, 0 },
        { 139 * GB,  50 * GB,       16 * GB, 0 },
        { 155 * GB,  90 * GB,        5 * GB, 0 },
        { 160 * GB, 140 * GB,      110 * GB, 0 },
        { 270 * GB, 270 * GB,        1 * GB, 1 },
    };

    ShiftBinaryMatchResultByLeftJustify(Before, After, 271 * GB);
}

TEST(RelocatedBinaryMatchTest, ShiftBinaryMatchResultBySoftSplitLarge)
{
    static const int64_t GB = 1024 * 1024 * 1024;

    // ShiftBinaryMatchResultBySoftSplit() 前
    const MatchResult Before[] =
    {
        {   0 * GB,   0 * GB,        2 * GB, 1 },
        {   2 * GB, InvalidOffset,   3 * GB, 1 },
        {   5 * GB,   5 * GB,       19 * GB, 1 },
        {  24 * GB, InvalidOffset,   8 * GB, 1 },
        {  32 * GB,  32 * GB,       18 * GB, 1 },
        {  50 * GB, InvalidOffset,  16 * GB, 1 },
        {  66 * GB,  66 * GB,       24 * GB, 1 },
        {  90 * GB, InvalidOffset,   5 * GB, 1 },
        {  95 * GB,  95 * GB,       33 * GB, 1 },
        { 128 * GB,   2 * GB,        3 * GB, 0 },
        { 131 * GB,  24 * GB,        8 * GB, 0 },
        { 139 * GB,  50 * GB,       16 * GB, 0 },
        { 155 * GB,  90 * GB,        5 * GB, 0 },
        // 128KB ～ 140KB に空きあり（12KB）
        { 160 * GB, 140 * GB,      110 * GB, 0 },
        // 250KB ～ 270KB に空きあり（20KB）
        { 270 * GB, 270 * GB,        1 * GB, 1 },
    };

    // ShiftBinaryMatchResultBySoftSplit() 後
    const MatchResult After[] =
    {
        {   0 * GB,   0 * GB,   2 * GB, 1 },
        {   2 * GB, 128 * GB,   3 * GB, 1 },
        {   5 * GB,   5 * GB,  19 * GB, 1 },
        {  24 * GB, 131 * GB,   8 * GB, 1 },
        {  32 * GB,  32 * GB,  18 * GB, 1 },
        {  50 * GB, 139 * GB,   1 * GB, 1 },
        {  51 * GB, 250 * GB,  15 * GB, 1 },
        {  66 * GB,  66 * GB,  24 * GB, 1 },
        {  90 * GB, 265 * GB,   5 * GB, 1 },
        {  95 * GB,  95 * GB,  33 * GB, 1 },
        { 128 * GB,   2 * GB,   3 * GB, 0 },
        { 131 * GB,  24 * GB,   8 * GB, 0 },
        { 139 * GB,  50 * GB,  16 * GB, 0 },
        { 155 * GB,  90 * GB,   5 * GB, 0 },
        { 160 * GB, 140 * GB, 110 * GB, 0 },
        { 270 * GB, 270 * GB,   1 * GB, 1 },
    };

    ShiftBinaryMatchResultBySoftSplit(Before, After, 271 * GB, 0);
}

TEST(RelocatedBinaryMatchTest, ShiftBinaryMatchResultByRightJustifyLarge)
{
    static const int64_t GB = 1024 * 1024 * 1024;

    // ShiftBinaryMatchResultByRightJustify() 前
    const MatchResult Before[] =
    {
        { 0,        0,                           2 * GB, 1 },
        {   2 * GB, MatchResult::InvalidOffset,  3 * GB, 1 },
        {   5 * GB,   5 * GB,                   19 * GB, 1 },
        {  24 * GB, MatchResult::InvalidOffset,  8 * GB, 1 },
        {  32 * GB,  32 * GB,                   18 * GB, 1 },
        {  50 * GB, MatchResult::InvalidOffset, 16 * GB, 1 },
        {  66 * GB,  66 * GB,                   24 * GB, 1 },
        {  90 * GB, MatchResult::InvalidOffset,  5 * GB, 1 },
        {  95 * GB,  95 * GB,                   33 * GB, 1 },
        { 128 * GB,   2 * GB,                    3 * GB, 0 },
        { 131 * GB,  24 * GB,                    8 * GB, 0 },
        { 139 * GB,  50 * GB,                   16 * GB, 0 },
        { 155 * GB,  90 * GB,                    5 * GB, 0 },
        { 160 * GB, 128 * GB,                  110 * GB, 0 },
        { 270 * GB, 239 * GB,                    1 * GB, 1 },
    };

    // ShiftBinaryMatchResultByRightJustify() 後
    const MatchResult After[] =
    {
        { 0,        0,          2 * GB, 1 },
        {   2 * GB, 240 * GB,   3 * GB, 1 },
        {   5 * GB,   5 * GB,  19 * GB, 1 },
        {  24 * GB, 243 * GB,   8 * GB, 1 },
        {  32 * GB,  32 * GB,  18 * GB, 1 },
        {  50 * GB, 251 * GB,  16 * GB, 1 },
        {  66 * GB,  66 * GB,  24 * GB, 1 },
        {  90 * GB, 267 * GB,   5 * GB, 1 },
        {  95 * GB,  95 * GB,  33 * GB, 1 },
        { 128 * GB,   2 * GB,   3 * GB, 0 },
        { 131 * GB,  24 * GB,   8 * GB, 0 },
        { 139 * GB,  50 * GB,  16 * GB, 0 },
        { 155 * GB,  90 * GB,   5 * GB, 0 },
        { 160 * GB, 128 * GB, 110 * GB, 0 },
        { 270 * GB, 239 * GB,   1 * GB, 1 },
    };

    ShiftBinaryMatchResultByRightJustify(Before, After, 270 * GB, 30 * GB);
}
